/* valatype.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valatypesymbol.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/map.h>
#include <gee/readonlycollection.h>
#include <vala/valacodenode.h>
#include <vala/valasourcereference.h>
#include <vala/valasourcefile.h>




struct _ValaTypesymbolPrivate {
	GeeList* cheader_filenames;
	GeeMap* array_types;
};
#define VALA_TYPESYMBOL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_TYPESYMBOL, ValaTypesymbolPrivate))
enum  {
	VALA_TYPESYMBOL_DUMMY_PROPERTY
};
static gboolean vala_typesymbol_real_is_reference_type (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_dup_function (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_free_function (ValaTypesymbol* self);
static gboolean vala_typesymbol_real_is_reference_counting (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_ref_function (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_unref_function (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_type_id (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_marshaller_type_name (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_get_value_function (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_set_value_function (ValaTypesymbol* self);
static char* vala_typesymbol_real_get_upper_case_cname (ValaTypesymbol* self, const char* infix);
static char* vala_typesymbol_real_get_default_value (ValaTypesymbol* self);
static GeeCollection* vala_typesymbol_real_get_cheader_filenames (ValaSymbol* base);
static gboolean vala_typesymbol_real_is_subtype_of (ValaTypesymbol* self, ValaTypesymbol* t);
static gint vala_typesymbol_real_get_type_parameter_index (ValaTypesymbol* self, const char* name);
static gpointer vala_typesymbol_parent_class = NULL;
static void vala_typesymbol_dispose (GObject * obj);



/**
 * Returns the name of this data type as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_typesymbol_get_cname (ValaTypesymbol* self, gboolean const_type) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_cname (self, const_type);
}


/**
 * Checks whether this data type has value or reference type semantics.
 *
 * @return true if this data type has reference type semantics
 */
static gboolean vala_typesymbol_real_is_reference_type (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), FALSE);
	return FALSE;
}


gboolean vala_typesymbol_is_reference_type (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->is_reference_type (self);
}


/**
 * Returns the C function name that duplicates instances of this data
 * type. The specified C function must accept one argument referencing
 * the instance of this data type and return a reference to the
 * duplicate.
 *
 * @return the name of the C function if supported or null otherwise
 */
static char* vala_typesymbol_real_get_dup_function (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_dup_function (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_dup_function (self);
}


/**
 * Returns the C function name that frees instances of this data type.
 * This is only valid for data types with reference type semantics that
 * do not support reference counting. The specified C function must
 * accept one argument pointing to the instance to be freed.
 *
 * @return the name of the C function or null if this data type is not a
 *         reference type or if it supports reference counting
 */
static char* vala_typesymbol_real_get_free_function (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_free_function (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_free_function (self);
}


/**
 * Checks whether this data type supports reference counting. This is
 * only valid for reference types.
 *
 * @return true if this data type supports reference counting
 */
static gboolean vala_typesymbol_real_is_reference_counting (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), FALSE);
	return FALSE;
}


gboolean vala_typesymbol_is_reference_counting (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->is_reference_counting (self);
}


/**
 * Returns the C function name that increments the reference count of
 * instances of this data type. This is only valid for data types
 * supporting reference counting. The specified C function must accept
 * one argument referencing the instance of this data type and return
 * the reference.
 *
 * @return the name of the C function or null if this data type does not
 *         support reference counting
 */
static char* vala_typesymbol_real_get_ref_function (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_ref_function (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_ref_function (self);
}


/**
 * Returns the C function name that decrements the reference count of
 * instances of this data type. This is only valid for data types
 * supporting reference counting. The specified C function must accept
 * one argument referencing the instance of this data type.
 *
 * @return the name of the C function or null if this data type does not
 *         support reference counting
 */
static char* vala_typesymbol_real_get_unref_function (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_unref_function (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_unref_function (self);
}


/**
 * Returns the C symbol representing the runtime type id for this data
 * type. The specified symbol must express a registered GType.
 *
 * @return the name of the GType name in C code or null if this data
 *         type is not registered with GType
 */
static char* vala_typesymbol_real_get_type_id (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_type_id (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_type_id (self);
}


/**
 * Returns the name of this data type as used in C code marshallers
 *
 * @return type name for marshallers
 */
static char* vala_typesymbol_real_get_marshaller_type_name (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_marshaller_type_name (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_marshaller_type_name (self);
}


/**
 * Returns the cname of the GValue getter function,
 */
static char* vala_typesymbol_real_get_get_value_function (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_get_value_function (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_get_value_function (self);
}


/**
 * Returns the cname of the GValue setter function,
 */
static char* vala_typesymbol_real_get_set_value_function (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_set_value_function (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_set_value_function (self);
}


/**
 * Returns the C name of this data type in upper case. Words are
 * separated by underscores. The upper case C name of the namespace is
 * prefix of the result.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the upper case name to be used in C code
 */
static char* vala_typesymbol_real_get_upper_case_cname (ValaTypesymbol* self, const char* infix) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_upper_case_cname (ValaTypesymbol* self, const char* infix) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_upper_case_cname (self, infix);
}


/**
 * Returns the default value for the given type. Returning null means
 * there is no default value (i.e. not that the default name is NULL).
 *
 * @returnthe name of the default value
 */
static char* vala_typesymbol_real_get_default_value (ValaTypesymbol* self) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), NULL);
	return NULL;
}


char* vala_typesymbol_get_default_value (ValaTypesymbol* self) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_default_value (self);
}


static GeeCollection* vala_typesymbol_real_get_cheader_filenames (ValaSymbol* base) {
	ValaTypesymbol * self;
	self = VALA_TYPESYMBOL (base);
	if (gee_collection_get_size (GEE_COLLECTION (self->priv->cheader_filenames)) == 0) {
		{
			GeeCollection* filename_collection;
			GeeIterator* filename_it;
			filename_collection = vala_symbol_get_cheader_filenames (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)));
			filename_it = gee_iterable_iterator (GEE_ITERABLE (filename_collection));
			while (gee_iterator_next (filename_it)) {
				char* filename;
				filename = ((char*) gee_iterator_get (filename_it));
				{
					/* default to header filenames of the namespace */
					vala_typesymbol_add_cheader_filename (self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			(filename_collection == NULL ? NULL : (filename_collection = (g_object_unref (filename_collection), NULL)));
			(filename_it == NULL ? NULL : (filename_it = (g_object_unref (filename_it), NULL)));
		}
		if (gee_collection_get_size (GEE_COLLECTION (self->priv->cheader_filenames)) == 0 && vala_code_node_get_source_reference (VALA_CODE_NODE (self)) != NULL && !vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (self))))) {
			char* _tmp0;
			/* don't add default include directives for VAPI files*/
			_tmp0 = NULL;
			gee_collection_add (GEE_COLLECTION (self->priv->cheader_filenames), (_tmp0 = vala_source_file_get_cinclude_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (self))))));
			_tmp0 = (g_free (_tmp0), NULL);
		}
	}
	return GEE_COLLECTION (gee_read_only_collection_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, GEE_COLLECTION (self->priv->cheader_filenames)));
}


/**
 * Adds a filename to the list of C header filenames users of this data
 * type must include.
 *
 * @param filename a C header filename
 */
void vala_typesymbol_add_cheader_filename (ValaTypesymbol* self, const char* filename) {
	g_return_if_fail (VALA_IS_TYPESYMBOL (self));
	gee_collection_add (GEE_COLLECTION (self->priv->cheader_filenames), filename);
}


/**
 * Checks whether this data type is equal to or a subtype of the
 * specified data type.
 *
 * @param t a data type
 * @return  true if t is a supertype of this data type, false otherwise
 */
static gboolean vala_typesymbol_real_is_subtype_of (ValaTypesymbol* self, ValaTypesymbol* t) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), FALSE);
	g_return_val_if_fail (t == NULL || VALA_IS_TYPESYMBOL (t), FALSE);
	return (self == t);
}


gboolean vala_typesymbol_is_subtype_of (ValaTypesymbol* self, ValaTypesymbol* t) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->is_subtype_of (self, t);
}


/**
 * Return the index of the specified type parameter name.
 */
static gint vala_typesymbol_real_get_type_parameter_index (ValaTypesymbol* self, const char* name) {
	g_return_val_if_fail (VALA_IS_TYPESYMBOL (self), 0);
	return -1;
}


gint vala_typesymbol_get_type_parameter_index (ValaTypesymbol* self, const char* name) {
	return VALA_TYPESYMBOL_GET_CLASS (self)->get_type_parameter_index (self, name);
}


static void vala_typesymbol_class_init (ValaTypesymbolClass * klass) {
	vala_typesymbol_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaTypesymbolPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_typesymbol_dispose;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_typesymbol_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_dup_function = vala_typesymbol_real_get_dup_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_free_function = vala_typesymbol_real_get_free_function;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_counting = vala_typesymbol_real_is_reference_counting;
	VALA_TYPESYMBOL_CLASS (klass)->get_ref_function = vala_typesymbol_real_get_ref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_unref_function = vala_typesymbol_real_get_unref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_typesymbol_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_typesymbol_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_typesymbol_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_typesymbol_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_typesymbol_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_default_value = vala_typesymbol_real_get_default_value;
	VALA_SYMBOL_CLASS (klass)->get_cheader_filenames = vala_typesymbol_real_get_cheader_filenames;
	VALA_TYPESYMBOL_CLASS (klass)->is_subtype_of = vala_typesymbol_real_is_subtype_of;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_parameter_index = vala_typesymbol_real_get_type_parameter_index;
}


static void vala_typesymbol_init (ValaTypesymbol * self) {
	self->priv = VALA_TYPESYMBOL_GET_PRIVATE (self);
	self->priv->cheader_filenames = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_equal));
}


static void vala_typesymbol_dispose (GObject * obj) {
	ValaTypesymbol * self;
	self = VALA_TYPESYMBOL (obj);
	(self->priv->cheader_filenames == NULL ? NULL : (self->priv->cheader_filenames = (g_object_unref (self->priv->cheader_filenames), NULL)));
	(self->priv->array_types == NULL ? NULL : (self->priv->array_types = (g_object_unref (self->priv->array_types), NULL)));
	G_OBJECT_CLASS (vala_typesymbol_parent_class)->dispose (obj);
}


GType vala_typesymbol_get_type (void) {
	static GType vala_typesymbol_type_id = 0;
	if (G_UNLIKELY (vala_typesymbol_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaTypesymbolClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_typesymbol_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaTypesymbol), 0, (GInstanceInitFunc) vala_typesymbol_init };
		vala_typesymbol_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaTypesymbol", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_typesymbol_type_id;
}




