/* valasemanticanalyzer.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valasemanticanalyzer.h>
#include <gee/collection.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valatypesymbol.h>
#include <vala/valavaluetype.h>
#include <vala/valascope.h>
#include <vala/valaclasstype.h>
#include <vala/valainterfacetype.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valaexpression.h>
#include <vala/valanamespace.h>
#include <vala/valaattribute.h>
#include <vala/valastatement.h>
#include <vala/valavoidtype.h>
#include <vala/valadelegatetype.h>
#include <vala/valapointertype.h>
#include <vala/valaarraytype.h>
#include <vala/valaswitchlabel.h>
#include <vala/valaerrortype.h>
#include <vala/valaerrordomain.h>
#include <vala/valalockable.h>
#include <vala/valaliteral.h>
#include <vala/valaintegertype.h>
#include <vala/valanulltype.h>
#include <vala/valamethodtype.h>
#include <vala/valasignaltype.h>
#include <vala/valanamespacereference.h>
#include <vala/valareferencetype.h>
#include <vala/valaerrorcode.h>
#include <vala/valamemberinitializer.h>




struct _ValaSemanticAnalyzerPrivate {
	ValaCodeContext* context;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaSourceFile* current_source_file;
	ValaDataType* current_return_type;
	ValaClass* current_class;
	ValaStruct* current_struct;
	GeeCollection* current_using_directives;
	ValaDataType* bool_type;
	ValaDataType* string_type;
	ValaDataType* int_type;
	ValaDataType* uint_type;
	ValaDataType* ulong_type;
	ValaDataType* size_t_type;
	ValaDataType* unichar_type;
	ValaDataType* type_type;
	ValaTypesymbol* pointer_type;
	ValaClass* object_type;
	ValaTypesymbol* initially_unowned_type;
	ValaDataType* glist_type;
	ValaDataType* gslist_type;
	ValaClass* gerror_type;
	ValaDataType* iterable_type;
	ValaInterface* iterator_type;
	ValaInterface* list_type;
	ValaInterface* map_type;
	gint next_lambda_id;
	GeeCollection* binding_providers;
};
#define VALA_SEMANTIC_ANALYZER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerPrivate))
enum  {
	VALA_SEMANTIC_ANALYZER_DUMMY_PROPERTY
};
static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface, GeeCollection* list);
static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaTypesymbol* t);
static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_semantic_analyzer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_semantic_analyzer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev);
static void vala_semantic_analyzer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* d);
static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static gboolean vala_semantic_analyzer_is_type_accessible (ValaSemanticAnalyzer* self, ValaSymbol* sym, ValaDataType* type);
static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_semantic_analyzer_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_semantic_analyzer_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_semantic_analyzer_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section);
static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_semantic_analyzer_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt);
static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_semantic_analyzer_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_semantic_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_semantic_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_semantic_analyzer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static void vala_semantic_analyzer_real_visit_delete_statement (ValaCodeVisitor* base, ValaDeleteStatement* stmt);
static gint vala_semantic_analyzer_create_sizes_from_initializer_list (ValaSemanticAnalyzer* self, ValaInitializerList* il, gint rank, GeeList* sl);
static void vala_semantic_analyzer_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr);
static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr);
static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr);
static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr);
static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr);
static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr);
static ValaDataType* vala_semantic_analyzer_get_static_type_for_symbol (ValaSemanticAnalyzer* self, ValaSymbol* sym);
static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static void vala_semantic_analyzer_real_visit_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaDataType* mtype, GeeCollection* params, GeeCollection* args);
static ValaDataType* vala_semantic_analyzer_get_instance_base_type (ValaDataType* instance_type, ValaDataType* base_type, ValaCodeNode* node_reference);
static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr);
static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr);
static void vala_semantic_analyzer_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr);
static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr);
static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaDataType* type);
static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaDataType* type);
static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr);
static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr);
static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr);
static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr);
static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static ValaDataType* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaDataType* left_type, ValaDataType* right_type);
static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr);
static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr);
static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self);
static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self);
static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self);
static void vala_semantic_analyzer_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_semantic_analyzer_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_semantic_analyzer_parent_class = NULL;
static void vala_semantic_analyzer_dispose (GObject * obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaSemanticAnalyzer* vala_semantic_analyzer_new (void) {
	ValaSemanticAnalyzer * self;
	self = g_object_newv (VALA_TYPE_SEMANTIC_ANALYZER, 0, NULL);
	return self;
}


void vala_semantic_analyzer_add_binding_provider (ValaSemanticAnalyzer* self, ValaBindingProvider* binding_provider) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (binding_provider == NULL || VALA_IS_BINDING_PROVIDER (binding_provider));
	gee_collection_add (self->priv->binding_providers, binding_provider);
}


/**
 * Analyze and check code in the specified context.
 *
 * @param context a code context
 */
void vala_semantic_analyzer_analyze (ValaSemanticAnalyzer* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	ValaSymbol* _tmp3;
	ValaNamespace* _tmp2;
	ValaDataType* _tmp5;
	ValaTypesymbol* _tmp4;
	ValaDataType* _tmp7;
	ValaClass* _tmp6;
	ValaTypesymbol* _tmp8;
	ValaDataType* _tmp10;
	ValaTypesymbol* _tmp9;
	ValaDataType* _tmp12;
	ValaTypesymbol* _tmp11;
	ValaDataType* _tmp14;
	ValaTypesymbol* _tmp13;
	ValaDataType* _tmp16;
	ValaTypesymbol* _tmp15;
	ValaDataType* _tmp18;
	ValaTypesymbol* _tmp17;
	ValaSymbol* glib_ns;
	ValaSymbol* gee_ns;
	ValaSymbol* _tmp34;
	ValaSymbol* _tmp33;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (context == NULL || VALA_IS_CODE_CONTEXT (context));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->root_symbol = (_tmp3 = VALA_SYMBOL ((_tmp2 = vala_code_context_get_root (context), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)))), (self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL))), _tmp3);
	_tmp5 = NULL;
	_tmp4 = NULL;
	self->priv->bool_type = (_tmp5 = VALA_DATA_TYPE (vala_value_type_new ((_tmp4 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "bool"))))), (self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL))), _tmp5);
	(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
	_tmp7 = NULL;
	_tmp6 = NULL;
	self->priv->string_type = (_tmp7 = VALA_DATA_TYPE (vala_class_type_new ((_tmp6 = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "string"))))), (self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL))), _tmp7);
	(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
	_tmp8 = NULL;
	self->priv->pointer_type = (_tmp8 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "pointer")), (self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL))), _tmp8);
	_tmp10 = NULL;
	_tmp9 = NULL;
	self->priv->int_type = (_tmp10 = VALA_DATA_TYPE (vala_value_type_new ((_tmp9 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "int"))))), (self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL))), _tmp10);
	(_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL)));
	_tmp12 = NULL;
	_tmp11 = NULL;
	self->priv->uint_type = (_tmp12 = VALA_DATA_TYPE (vala_value_type_new ((_tmp11 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "uint"))))), (self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL))), _tmp12);
	(_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL)));
	_tmp14 = NULL;
	_tmp13 = NULL;
	self->priv->ulong_type = (_tmp14 = VALA_DATA_TYPE (vala_value_type_new ((_tmp13 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "ulong"))))), (self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL))), _tmp14);
	(_tmp13 == NULL ? NULL : (_tmp13 = (g_object_unref (_tmp13), NULL)));
	_tmp16 = NULL;
	_tmp15 = NULL;
	self->priv->size_t_type = (_tmp16 = VALA_DATA_TYPE (vala_value_type_new ((_tmp15 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "size_t"))))), (self->priv->size_t_type == NULL ? NULL : (self->priv->size_t_type = (g_object_unref (self->priv->size_t_type), NULL))), _tmp16);
	(_tmp15 == NULL ? NULL : (_tmp15 = (g_object_unref (_tmp15), NULL)));
	_tmp18 = NULL;
	_tmp17 = NULL;
	self->priv->unichar_type = (_tmp18 = VALA_DATA_TYPE (vala_value_type_new ((_tmp17 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "unichar"))))), (self->priv->unichar_type == NULL ? NULL : (self->priv->unichar_type = (g_object_unref (self->priv->unichar_type), NULL))), _tmp18);
	(_tmp17 == NULL ? NULL : (_tmp17 = (g_object_unref (_tmp17), NULL)));
	/* TODO: don't require GLib namespace in semantic analyzer*/
	glib_ns = vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "GLib");
	if (glib_ns != NULL) {
		ValaClass* _tmp19;
		ValaTypesymbol* _tmp20;
		ValaDataType* _tmp22;
		ValaTypesymbol* _tmp21;
		ValaDataType* _tmp24;
		ValaClass* _tmp23;
		ValaDataType* _tmp26;
		ValaClass* _tmp25;
		ValaClass* _tmp27;
		_tmp19 = NULL;
		self->priv->object_type = (_tmp19 = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Object")), (self->priv->object_type == NULL ? NULL : (self->priv->object_type = (g_object_unref (self->priv->object_type), NULL))), _tmp19);
		_tmp20 = NULL;
		self->priv->initially_unowned_type = (_tmp20 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "InitiallyUnowned")), (self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL))), _tmp20);
		_tmp22 = NULL;
		_tmp21 = NULL;
		self->priv->type_type = (_tmp22 = VALA_DATA_TYPE (vala_value_type_new ((_tmp21 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Type"))))), (self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL))), _tmp22);
		(_tmp21 == NULL ? NULL : (_tmp21 = (g_object_unref (_tmp21), NULL)));
		_tmp24 = NULL;
		_tmp23 = NULL;
		self->priv->glist_type = (_tmp24 = VALA_DATA_TYPE (vala_class_type_new ((_tmp23 = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "List"))))), (self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL))), _tmp24);
		(_tmp23 == NULL ? NULL : (_tmp23 = (g_object_unref (_tmp23), NULL)));
		_tmp26 = NULL;
		_tmp25 = NULL;
		self->priv->gslist_type = (_tmp26 = VALA_DATA_TYPE (vala_class_type_new ((_tmp25 = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "SList"))))), (self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL))), _tmp26);
		(_tmp25 == NULL ? NULL : (_tmp25 = (g_object_unref (_tmp25), NULL)));
		_tmp27 = NULL;
		self->priv->gerror_type = (_tmp27 = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Error")), (self->priv->gerror_type == NULL ? NULL : (self->priv->gerror_type = (g_object_unref (self->priv->gerror_type), NULL))), _tmp27);
	}
	gee_ns = vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "Gee");
	if (gee_ns != NULL) {
		ValaDataType* _tmp29;
		ValaInterface* _tmp28;
		ValaInterface* _tmp30;
		ValaInterface* _tmp31;
		ValaInterface* _tmp32;
		_tmp29 = NULL;
		_tmp28 = NULL;
		self->priv->iterable_type = (_tmp29 = VALA_DATA_TYPE (vala_interface_type_new ((_tmp28 = VALA_INTERFACE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Iterable"))))), (self->priv->iterable_type == NULL ? NULL : (self->priv->iterable_type = (g_object_unref (self->priv->iterable_type), NULL))), _tmp29);
		(_tmp28 == NULL ? NULL : (_tmp28 = (g_object_unref (_tmp28), NULL)));
		_tmp30 = NULL;
		self->priv->iterator_type = (_tmp30 = VALA_INTERFACE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Iterator")), (self->priv->iterator_type == NULL ? NULL : (self->priv->iterator_type = (g_object_unref (self->priv->iterator_type), NULL))), _tmp30);
		_tmp31 = NULL;
		self->priv->list_type = (_tmp31 = VALA_INTERFACE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "List")), (self->priv->list_type == NULL ? NULL : (self->priv->list_type = (g_object_unref (self->priv->list_type), NULL))), _tmp31);
		_tmp32 = NULL;
		self->priv->map_type = (_tmp32 = VALA_INTERFACE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Map")), (self->priv->map_type == NULL ? NULL : (self->priv->map_type = (g_object_unref (self->priv->map_type), NULL))), _tmp32);
	}
	_tmp34 = NULL;
	_tmp33 = NULL;
	self->priv->current_symbol = (_tmp34 = (_tmp33 = self->priv->root_symbol, (_tmp33 == NULL ? NULL : g_object_ref (_tmp33))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp34);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	(glib_ns == NULL ? NULL : (glib_ns = (g_object_unref (glib_ns), NULL)));
	(gee_ns == NULL ? NULL : (gee_ns = (g_object_unref (gee_ns), NULL)));
}


static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file) {
	ValaSemanticAnalyzer * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* _tmp0;
	GeeCollection* _tmp2;
	GeeCollection* _tmp3;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (file == NULL || VALA_IS_SOURCE_FILE (file));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_source_file = (_tmp1 = (_tmp0 = file, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), _tmp1);
	_tmp2 = NULL;
	self->priv->current_using_directives = (_tmp2 = vala_source_file_get_using_directives (file), (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL))), _tmp2);
	self->priv->next_lambda_id = 0;
	vala_source_file_accept_children (file, VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	self->priv->current_using_directives = (_tmp3 = NULL, (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL))), _tmp3);
}


static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp1;
	ValaClass* _tmp0;
	ValaClass* _tmp3;
	ValaClass* _tmp2;
	GeeList* prerequisites;
	GeeList* missing_prereqs;
	ValaSymbol* _tmp22;
	ValaSymbol* _tmp21;
	ValaClass* _tmp23;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (cl == NULL || VALA_IS_CLASS (cl));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL ((_tmp0 = cl, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_class = (_tmp3 = (_tmp2 = cl, (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), _tmp3);
	{
		GeeCollection* base_type_reference_collection;
		GeeIterator* base_type_reference_it;
		base_type_reference_collection = vala_class_get_base_types (cl);
		base_type_reference_it = gee_iterable_iterator (GEE_ITERABLE (base_type_reference_collection));
		while (gee_iterator_next (base_type_reference_it)) {
			ValaDataType* base_type_reference;
			base_type_reference = ((ValaDataType*) gee_iterator_get (base_type_reference_it));
			{
				vala_source_file_add_type_dependency (self->priv->current_source_file, base_type_reference, VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
				(base_type_reference == NULL ? NULL : (base_type_reference = (g_object_unref (base_type_reference), NULL)));
			}
		}
		(base_type_reference_collection == NULL ? NULL : (base_type_reference_collection = (g_object_unref (base_type_reference_collection), NULL)));
		(base_type_reference_it == NULL ? NULL : (base_type_reference_it = (g_object_unref (base_type_reference_it), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	/* gather all prerequisites */
	prerequisites = GEE_LIST (gee_array_list_new (VALA_TYPE_TYPESYMBOL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	{
		GeeCollection* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = vala_class_get_base_types (cl);
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaDataType* base_type;
			base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
			{
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (base_type))) {
					vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (vala_data_type_get_data_type (base_type)), GEE_COLLECTION (prerequisites));
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	/* check whether all prerequisites are met */
	missing_prereqs = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_equal));
	{
		GeeList* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = prerequisites;
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaTypesymbol* prereq;
			prereq = ((ValaTypesymbol*) gee_iterator_get (prereq_it));
			{
				if (!vala_semantic_analyzer_class_is_a (self, cl, prereq)) {
					char* _tmp4;
					_tmp4 = NULL;
					gee_list_insert (missing_prereqs, 0, (_tmp4 = vala_symbol_get_full_name (VALA_SYMBOL (prereq))));
					_tmp4 = (g_free (_tmp4), NULL);
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
			}
		}
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
	/* report any missing prerequisites */
	if (gee_collection_get_size (GEE_COLLECTION (missing_prereqs)) > 0) {
		char* _tmp5;
		char* _tmp6;
		char* error_string;
		gboolean first;
		char* _tmp9;
		vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
		_tmp5 = NULL;
		_tmp6 = NULL;
		error_string = (_tmp6 = g_strdup_printf ("%s: some prerequisites (", (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (cl)))), (_tmp5 = (g_free (_tmp5), NULL)), _tmp6);
		first = TRUE;
		{
			GeeList* s_collection;
			GeeIterator* s_it;
			s_collection = missing_prereqs;
			s_it = gee_iterable_iterator (GEE_ITERABLE (s_collection));
			while (gee_iterator_next (s_it)) {
				char* s;
				s = ((char*) gee_iterator_get (s_it));
				{
					if (first) {
						char* _tmp7;
						_tmp7 = NULL;
						error_string = (_tmp7 = g_strdup_printf ("%s`%s'", error_string, s), (error_string = (g_free (error_string), NULL)), _tmp7);
						first = FALSE;
					} else {
						char* _tmp8;
						_tmp8 = NULL;
						error_string = (_tmp8 = g_strdup_printf ("%s, `%s'", error_string, s), (error_string = (g_free (error_string), NULL)), _tmp8);
					}
					s = (g_free (s), NULL);
				}
			}
			(s_it == NULL ? NULL : (s_it = (g_object_unref (s_it), NULL)));
		}
		_tmp9 = NULL;
		error_string = (_tmp9 = g_strconcat (error_string, (") are not met"), NULL), (error_string = (g_free (error_string), NULL)), _tmp9);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), error_string);
		error_string = (g_free (error_string), NULL);
	}
	/* VAPI classes don't have to specify overridden methods */
	if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl))))) {
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = vala_class_get_base_types (cl);
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
				{
					/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
					if (VALA_IS_INTERFACE (vala_data_type_get_data_type (base_type))) {
						ValaInterface* _tmp10;
						ValaInterface* iface;
						_tmp10 = NULL;
						iface = (_tmp10 = VALA_INTERFACE (vala_data_type_get_data_type (base_type)), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
						{
							GeeCollection* m_collection;
							GeeIterator* m_it;
							m_collection = vala_interface_get_methods (iface);
							m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
							while (gee_iterator_next (m_it)) {
								ValaMethod* m;
								m = ((ValaMethod*) gee_iterator_get (m_it));
								{
									/* We do not need to do expensive equality checking here since this is done
									 * already. We only need to guarantee the symbols are present.
									 
									 check methods */
									if (vala_method_get_is_abstract (m)) {
										ValaSymbol* sym;
										sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (cl)), vala_symbol_get_name (VALA_SYMBOL (m)));
										if (!(VALA_IS_METHOD (sym))) {
											char* _tmp13;
											char* _tmp12;
											char* _tmp11;
											vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
											_tmp13 = NULL;
											_tmp12 = NULL;
											_tmp11 = NULL;
											vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (_tmp13 = g_strdup_printf ("`%s' does not implement interface method `%s'", (_tmp11 = vala_symbol_get_full_name (VALA_SYMBOL (cl))), (_tmp12 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
											_tmp13 = (g_free (_tmp13), NULL);
											_tmp12 = (g_free (_tmp12), NULL);
											_tmp11 = (g_free (_tmp11), NULL);
										}
										(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
									}
									(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								}
							}
							(m_collection == NULL ? NULL : (m_collection = (g_object_unref (m_collection), NULL)));
							(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
						}
						(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		/* all abstract symbols defined in base classes have to be implemented in non-abstract classes */
		if (!vala_class_get_is_abstract (cl)) {
			ValaClass* _tmp14;
			ValaClass* base_class;
			_tmp14 = NULL;
			base_class = (_tmp14 = vala_class_get_base_class (cl), (_tmp14 == NULL ? NULL : g_object_ref (_tmp14)));
			while (base_class != NULL && vala_class_get_is_abstract (base_class)) {
				ValaClass* _tmp20;
				ValaClass* _tmp19;
				{
					GeeCollection* base_method_collection;
					GeeIterator* base_method_it;
					base_method_collection = vala_class_get_methods (base_class);
					base_method_it = gee_iterable_iterator (GEE_ITERABLE (base_method_collection));
					while (gee_iterator_next (base_method_it)) {
						ValaMethod* base_method;
						base_method = ((ValaMethod*) gee_iterator_get (base_method_it));
						{
							if (vala_method_get_is_abstract (base_method)) {
								ValaSymbol* _tmp15;
								ValaMethod* override_method;
								_tmp15 = NULL;
								override_method = (_tmp15 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (cl), vala_symbol_get_name (VALA_SYMBOL (base_method))), (VALA_IS_METHOD (_tmp15) ? ((ValaMethod*) _tmp15) : NULL));
								if (override_method == NULL || !vala_method_get_overrides (override_method)) {
									char* _tmp18;
									char* _tmp17;
									char* _tmp16;
									vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
									_tmp18 = NULL;
									_tmp17 = NULL;
									_tmp16 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (_tmp18 = g_strdup_printf ("`%s' does not implement abstract method `%s'", (_tmp16 = vala_symbol_get_full_name (VALA_SYMBOL (cl))), (_tmp17 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))))));
									_tmp18 = (g_free (_tmp18), NULL);
									_tmp17 = (g_free (_tmp17), NULL);
									_tmp16 = (g_free (_tmp16), NULL);
								}
								(override_method == NULL ? NULL : (override_method = (g_object_unref (override_method), NULL)));
							}
							(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
						}
					}
					(base_method_collection == NULL ? NULL : (base_method_collection = (g_object_unref (base_method_collection), NULL)));
					(base_method_it == NULL ? NULL : (base_method_it = (g_object_unref (base_method_it), NULL)));
				}
				_tmp20 = NULL;
				_tmp19 = NULL;
				base_class = (_tmp20 = (_tmp19 = vala_class_get_base_class (base_class), (_tmp19 == NULL ? NULL : g_object_ref (_tmp19))), (base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL))), _tmp20);
			}
			(base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL)));
		}
	}
	_tmp22 = NULL;
	_tmp21 = NULL;
	self->priv->current_symbol = (_tmp22 = (_tmp21 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp21 == NULL ? NULL : g_object_ref (_tmp21))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp22);
	_tmp23 = NULL;
	self->priv->current_class = (_tmp23 = NULL, (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), _tmp23);
	(prerequisites == NULL ? NULL : (prerequisites = (g_object_unref (prerequisites), NULL)));
	(missing_prereqs == NULL ? NULL : (missing_prereqs = (g_object_unref (missing_prereqs), NULL)));
}


static void vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface, GeeCollection* list) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (iface == NULL || VALA_IS_INTERFACE (iface));
	g_return_if_fail (list == NULL || GEE_IS_COLLECTION (list));
	{
		GeeCollection* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = vala_interface_get_prerequisites (iface);
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaDataType* prereq;
			prereq = ((ValaDataType*) gee_iterator_get (prereq_it));
			{
				ValaTypesymbol* _tmp0;
				ValaTypesymbol* type;
				_tmp0 = NULL;
				type = (_tmp0 = vala_data_type_get_data_type (prereq), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
				/* skip on previous errors */
				if (type == NULL) {
					(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					continue;
				}
				gee_collection_add (list, type);
				if (VALA_IS_INTERFACE (type)) {
					vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (type), list);
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
}


static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaTypesymbol* t) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (cl == NULL || VALA_IS_CLASS (cl), FALSE);
	g_return_val_if_fail (t == NULL || VALA_IS_TYPESYMBOL (t), FALSE);
	if (VALA_TYPESYMBOL (cl) == t) {
		return TRUE;
	}
	{
		GeeCollection* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = vala_class_get_base_types (cl);
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaDataType* base_type;
			base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
			{
				if (VALA_IS_CLASS (vala_data_type_get_data_type (base_type))) {
					if (vala_semantic_analyzer_class_is_a (self, VALA_CLASS (vala_data_type_get_data_type (base_type)), t)) {
						gboolean _tmp1;
						return (_tmp1 = TRUE, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), _tmp1);
					}
				} else {
					if (vala_data_type_get_data_type (base_type) == t) {
						gboolean _tmp2;
						return (_tmp2 = TRUE, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), _tmp2);
					}
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	return FALSE;
}


static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp1;
	ValaStruct* _tmp0;
	ValaStruct* _tmp3;
	ValaStruct* _tmp2;
	ValaSymbol* _tmp5;
	ValaSymbol* _tmp4;
	ValaStruct* _tmp6;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (st == NULL || VALA_IS_STRUCT (st));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL ((_tmp0 = st, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_struct = (_tmp3 = (_tmp2 = st, (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), _tmp3);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	_tmp5 = NULL;
	_tmp4 = NULL;
	self->priv->current_symbol = (_tmp5 = (_tmp4 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp5);
	_tmp6 = NULL;
	self->priv->current_struct = (_tmp6 = NULL, (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), _tmp6);
}


static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp1;
	ValaInterface* _tmp0;
	ValaClass* prereq_class;
	ValaSymbol* _tmp10;
	ValaSymbol* _tmp9;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (iface == NULL || VALA_IS_INTERFACE (iface));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL ((_tmp0 = iface, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	{
		GeeCollection* prerequisite_reference_collection;
		GeeIterator* prerequisite_reference_it;
		prerequisite_reference_collection = vala_interface_get_prerequisites (iface);
		prerequisite_reference_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_reference_collection));
		while (gee_iterator_next (prerequisite_reference_it)) {
			ValaDataType* prerequisite_reference;
			prerequisite_reference = ((ValaDataType*) gee_iterator_get (prerequisite_reference_it));
			{
				vala_source_file_add_type_dependency (self->priv->current_source_file, prerequisite_reference, VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
				(prerequisite_reference == NULL ? NULL : (prerequisite_reference = (g_object_unref (prerequisite_reference), NULL)));
			}
		}
		(prerequisite_reference_collection == NULL ? NULL : (prerequisite_reference_collection = (g_object_unref (prerequisite_reference_collection), NULL)));
		(prerequisite_reference_it == NULL ? NULL : (prerequisite_reference_it = (g_object_unref (prerequisite_reference_it), NULL)));
	}
	/* check prerequisites */
	prereq_class = NULL;
	{
		GeeCollection* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = vala_interface_get_prerequisites (iface);
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaDataType* prereq;
			prereq = ((ValaDataType*) gee_iterator_get (prereq_it));
			{
				ValaTypesymbol* _tmp2;
				ValaTypesymbol* class_or_interface;
				_tmp2 = NULL;
				class_or_interface = (_tmp2 = vala_data_type_get_data_type (prereq), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
				/* skip on previous errors */
				if (class_or_interface == NULL) {
					vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE);
					(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
					(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
					continue;
				}
				/* interfaces are not allowed to have multiple instantiable prerequisites */
				if (VALA_IS_CLASS (class_or_interface)) {
					ValaClass* _tmp8;
					ValaClass* _tmp7;
					if (prereq_class != NULL) {
						char* _tmp6;
						char* _tmp5;
						char* _tmp4;
						char* _tmp3;
						vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE);
						_tmp6 = NULL;
						_tmp5 = NULL;
						_tmp4 = NULL;
						_tmp3 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (iface)), (_tmp6 = g_strdup_printf ("%s: Interfaces cannot have multiple instantiable prerequisites (`%s' and `%s')", (_tmp3 = vala_symbol_get_full_name (VALA_SYMBOL (iface))), (_tmp4 = vala_symbol_get_full_name (VALA_SYMBOL (class_or_interface))), (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (prereq_class))))));
						_tmp6 = (g_free (_tmp6), NULL);
						_tmp5 = (g_free (_tmp5), NULL);
						_tmp4 = (g_free (_tmp4), NULL);
						_tmp3 = (g_free (_tmp3), NULL);
						(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
						(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
						(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
						(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
						(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
						return;
					}
					_tmp8 = NULL;
					_tmp7 = NULL;
					prereq_class = (_tmp8 = (_tmp7 = VALA_CLASS (class_or_interface), (_tmp7 == NULL ? NULL : g_object_ref (_tmp7))), (prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL))), _tmp8);
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
				(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
			}
		}
		(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	_tmp10 = NULL;
	_tmp9 = NULL;
	self->priv->current_symbol = (_tmp10 = (_tmp9 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp10);
	(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
}


static void vala_semantic_analyzer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (en == NULL || VALA_IS_ENUM (en));
	vala_code_node_accept_children (VALA_CODE_NODE (en), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (ev == NULL || VALA_IS_ENUM_VALUE (ev));
	vala_code_node_accept_children (VALA_CODE_NODE (ev), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* d) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (d == NULL || VALA_IS_DELEGATE (d));
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (c == NULL || VALA_IS_CONSTANT (c));
	vala_code_node_accept (VALA_CODE_NODE (vala_constant_get_type_reference (c)), VALA_CODE_VISITOR (self));
	if (!vala_source_file_get_pkg (self->priv->current_source_file)) {
		if (vala_constant_get_initializer (c) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (c), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (c)), "A const field requires a initializer to be provided");
		} else {
			vala_expression_set_expected_type (vala_constant_get_initializer (c), vala_constant_get_type_reference (c));
			vala_code_node_accept (VALA_CODE_NODE (vala_constant_get_initializer (c)), VALA_CODE_VISITOR (self));
		}
	}
}


static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (f == NULL || VALA_IS_FIELD (f));
	vala_code_node_accept_children (VALA_CODE_NODE (f), VALA_CODE_VISITOR (self));
	if (vala_field_get_instance (f) && VALA_IS_INTERFACE (vala_symbol_get_parent_symbol (VALA_SYMBOL (f)))) {
		vala_code_node_set_error (VALA_CODE_NODE (f), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (f)), "Interfaces may not have instance fields");
		return;
	}
	if (!vala_symbol_is_internal_symbol (VALA_SYMBOL (f))) {
		vala_source_file_add_type_dependency (self->priv->current_source_file, vala_field_get_type_reference (f), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
	} else {
		if (VALA_IS_NAMESPACE (vala_symbol_get_parent_symbol (VALA_SYMBOL (f)))) {
			vala_code_node_set_error (VALA_CODE_NODE (f), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (f)), "Namespaces may not have private members");
			return;
		}
		vala_source_file_add_type_dependency (self->priv->current_source_file, vala_field_get_type_reference (f), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
}


static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp1;
	ValaSymbol* old_symbol;
	ValaDataType* _tmp2;
	ValaDataType* old_return_type;
	ValaSymbol* _tmp4;
	ValaMethod* _tmp3;
	ValaDataType* _tmp6;
	ValaDataType* _tmp5;
	ValaAttribute* init_attr;
	ValaSymbol* _tmp8;
	ValaSymbol* _tmp7;
	ValaDataType* _tmp10;
	ValaDataType* _tmp9;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (m == NULL || VALA_IS_METHOD (m));
	if (vala_method_get_is_abstract (m)) {
		if (VALA_IS_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (m)))) {
			ValaClass* _tmp0;
			ValaClass* cl;
			_tmp0 = NULL;
			cl = (_tmp0 = VALA_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (m))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			if (!vala_class_get_is_abstract (cl)) {
				vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), "Abstract methods may not be declared in non-abstract classes");
				(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
				return;
			}
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		} else {
			if (!(VALA_IS_INTERFACE (vala_symbol_get_parent_symbol (VALA_SYMBOL (m))))) {
				vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), "Abstract methods may not be declared outside of classes and interfaces");
				return;
			}
		}
	} else {
		if (vala_method_get_is_virtual (m)) {
			if (!(VALA_IS_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (m))))) {
				vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), "Virtual methods may not be declared outside of classes");
				return;
			}
		} else {
			if (vala_method_get_overrides (m)) {
				if (!(VALA_IS_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (m))))) {
					vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), "Methods may not be overridden outside of classes");
					return;
				}
			}
		}
	}
	_tmp1 = NULL;
	old_symbol = (_tmp1 = self->priv->current_symbol, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
	_tmp2 = NULL;
	old_return_type = (_tmp2 = self->priv->current_return_type, (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_symbol = (_tmp4 = VALA_SYMBOL ((_tmp3 = m, (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp4);
	_tmp6 = NULL;
	_tmp5 = NULL;
	self->priv->current_return_type = (_tmp6 = (_tmp5 = vala_method_get_return_type (m), (_tmp5 == NULL ? NULL : g_object_ref (_tmp5))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp6);
	init_attr = vala_code_node_get_attribute (VALA_CODE_NODE (m), "ModuleInit");
	if (init_attr != NULL) {
		vala_code_context_set_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m)))), m);
	}
	if (!vala_symbol_is_internal_symbol (VALA_SYMBOL (m))) {
		vala_source_file_add_type_dependency (self->priv->current_source_file, vala_method_get_return_type (m), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
	}
	vala_source_file_add_type_dependency (self->priv->current_source_file, vala_method_get_return_type (m), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	_tmp8 = NULL;
	_tmp7 = NULL;
	self->priv->current_symbol = (_tmp8 = (_tmp7 = old_symbol, (_tmp7 == NULL ? NULL : g_object_ref (_tmp7))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp8);
	_tmp10 = NULL;
	_tmp9 = NULL;
	self->priv->current_return_type = (_tmp10 = (_tmp9 = old_return_type, (_tmp9 == NULL ? NULL : g_object_ref (_tmp9))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp10);
	if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol))) {
		ValaMethod* _tmp11;
		ValaMethod* up_method;
		ValaDataType* _tmp13;
		ValaDataType* _tmp12;
		/* lambda expressions produce nested methods */
		_tmp11 = NULL;
		up_method = (_tmp11 = VALA_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol)), (_tmp11 == NULL ? NULL : g_object_ref (_tmp11)));
		_tmp13 = NULL;
		_tmp12 = NULL;
		self->priv->current_return_type = (_tmp13 = (_tmp12 = vala_method_get_return_type (up_method), (_tmp12 == NULL ? NULL : g_object_ref (_tmp12))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp13);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (VALA_IS_CLASS (self->priv->current_symbol)) {
		/* VAPI classes don't specify overridden methods */
		if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (self->priv->current_symbol))))) {
			if (!(VALA_IS_CREATION_METHOD (m))) {
				vala_semantic_analyzer_find_base_interface_method (self, m, VALA_CLASS (self->priv->current_symbol));
				if (vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
					vala_semantic_analyzer_find_base_class_method (self, m, VALA_CLASS (self->priv->current_symbol));
					if (vala_method_get_base_method (m) == NULL) {
						char* _tmp15;
						char* _tmp14;
						_tmp15 = NULL;
						_tmp14 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp15 = g_strdup_printf ("%s: no suitable method found to override", (_tmp14 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
						_tmp15 = (g_free (_tmp15), NULL);
						_tmp14 = (g_free (_tmp14), NULL);
					}
				}
			}
		} else {
			if (vala_method_get_is_virtual (m) || vala_method_get_is_abstract (m)) {
				vala_method_set_base_method (m, m);
			}
		}
	} else {
		if (VALA_IS_STRUCT (self->priv->current_symbol)) {
			if (vala_method_get_is_abstract (m) || vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				char* _tmp17;
				char* _tmp16;
				_tmp17 = NULL;
				_tmp16 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp17 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", (_tmp16 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
				_tmp17 = (g_free (_tmp17), NULL);
				_tmp16 = (g_free (_tmp16), NULL);
				(old_symbol == NULL ? NULL : (old_symbol = (g_object_unref (old_symbol), NULL)));
				(old_return_type == NULL ? NULL : (old_return_type = (g_object_unref (old_return_type), NULL)));
				(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
				return;
			}
		}
	}
	/* check whether return type is at least as accessible as the method*/
	if (!vala_semantic_analyzer_is_type_accessible (self, VALA_SYMBOL (m), vala_method_get_return_type (m))) {
		char* _tmp20;
		char* _tmp19;
		char* _tmp18;
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		_tmp20 = NULL;
		_tmp19 = NULL;
		_tmp18 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp20 = g_strdup_printf ("return type `%s` is less accessible than method `%s`", (_tmp18 = vala_code_node_to_string (VALA_CODE_NODE (vala_method_get_return_type (m)))), (_tmp19 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
		_tmp20 = (g_free (_tmp20), NULL);
		_tmp19 = (g_free (_tmp19), NULL);
		_tmp18 = (g_free (_tmp18), NULL);
		(old_symbol == NULL ? NULL : (old_symbol = (g_object_unref (old_symbol), NULL)));
		(old_return_type == NULL ? NULL : (old_return_type = (g_object_unref (old_return_type), NULL)));
		(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
		return;
	}
	{
		GeeCollection* precondition_collection;
		GeeIterator* precondition_it;
		precondition_collection = vala_method_get_preconditions (m);
		precondition_it = gee_iterable_iterator (GEE_ITERABLE (precondition_collection));
		while (gee_iterator_next (precondition_it)) {
			ValaExpression* precondition;
			precondition = ((ValaExpression*) gee_iterator_get (precondition_it));
			{
				if (vala_code_node_get_error (VALA_CODE_NODE (precondition))) {
					/* if there was an error in the precondition, skip this check*/
					vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
					(precondition == NULL ? NULL : (precondition = (g_object_unref (precondition), NULL)));
					(precondition_collection == NULL ? NULL : (precondition_collection = (g_object_unref (precondition_collection), NULL)));
					(precondition_it == NULL ? NULL : (precondition_it = (g_object_unref (precondition_it), NULL)));
					(old_symbol == NULL ? NULL : (old_symbol = (g_object_unref (old_symbol), NULL)));
					(old_return_type == NULL ? NULL : (old_return_type = (g_object_unref (old_return_type), NULL)));
					(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
					return;
				}
				if (!vala_data_type_compatible (vala_expression_get_static_type (precondition), self->priv->bool_type, TRUE)) {
					vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (precondition)), "Precondition must be boolean");
					(precondition == NULL ? NULL : (precondition = (g_object_unref (precondition), NULL)));
					(precondition_collection == NULL ? NULL : (precondition_collection = (g_object_unref (precondition_collection), NULL)));
					(precondition_it == NULL ? NULL : (precondition_it = (g_object_unref (precondition_it), NULL)));
					(old_symbol == NULL ? NULL : (old_symbol = (g_object_unref (old_symbol), NULL)));
					(old_return_type == NULL ? NULL : (old_return_type = (g_object_unref (old_return_type), NULL)));
					(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
					return;
				}
				(precondition == NULL ? NULL : (precondition = (g_object_unref (precondition), NULL)));
			}
		}
		(precondition_collection == NULL ? NULL : (precondition_collection = (g_object_unref (precondition_collection), NULL)));
		(precondition_it == NULL ? NULL : (precondition_it = (g_object_unref (precondition_it), NULL)));
	}
	{
		GeeCollection* postcondition_collection;
		GeeIterator* postcondition_it;
		postcondition_collection = vala_method_get_postconditions (m);
		postcondition_it = gee_iterable_iterator (GEE_ITERABLE (postcondition_collection));
		while (gee_iterator_next (postcondition_it)) {
			ValaExpression* postcondition;
			postcondition = ((ValaExpression*) gee_iterator_get (postcondition_it));
			{
				if (vala_code_node_get_error (VALA_CODE_NODE (postcondition))) {
					/* if there was an error in the postcondition, skip this check*/
					vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
					(postcondition == NULL ? NULL : (postcondition = (g_object_unref (postcondition), NULL)));
					(postcondition_collection == NULL ? NULL : (postcondition_collection = (g_object_unref (postcondition_collection), NULL)));
					(postcondition_it == NULL ? NULL : (postcondition_it = (g_object_unref (postcondition_it), NULL)));
					(old_symbol == NULL ? NULL : (old_symbol = (g_object_unref (old_symbol), NULL)));
					(old_return_type == NULL ? NULL : (old_return_type = (g_object_unref (old_return_type), NULL)));
					(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
					return;
				}
				if (!vala_data_type_compatible (vala_expression_get_static_type (postcondition), self->priv->bool_type, TRUE)) {
					vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (postcondition)), "Postcondition must be boolean");
					(postcondition == NULL ? NULL : (postcondition = (g_object_unref (postcondition), NULL)));
					(postcondition_collection == NULL ? NULL : (postcondition_collection = (g_object_unref (postcondition_collection), NULL)));
					(postcondition_it == NULL ? NULL : (postcondition_it = (g_object_unref (postcondition_it), NULL)));
					(old_symbol == NULL ? NULL : (old_symbol = (g_object_unref (old_symbol), NULL)));
					(old_return_type == NULL ? NULL : (old_return_type = (g_object_unref (old_return_type), NULL)));
					(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
					return;
				}
				(postcondition == NULL ? NULL : (postcondition = (g_object_unref (postcondition), NULL)));
			}
		}
		(postcondition_collection == NULL ? NULL : (postcondition_collection = (g_object_unref (postcondition_collection), NULL)));
		(postcondition_it == NULL ? NULL : (postcondition_it = (g_object_unref (postcondition_it), NULL)));
	}
	(old_symbol == NULL ? NULL : (old_symbol = (g_object_unref (old_symbol), NULL)));
	(old_return_type == NULL ? NULL : (old_return_type = (g_object_unref (old_return_type), NULL)));
	(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
}


static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl) {
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (m == NULL || VALA_IS_METHOD (m));
	g_return_if_fail (cl == NULL || VALA_IS_CLASS (cl));
	sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (cl)), vala_symbol_get_name (VALA_SYMBOL (m)));
	if (VALA_IS_METHOD (sym)) {
		ValaMethod* _tmp0;
		ValaMethod* base_method;
		_tmp0 = NULL;
		base_method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (vala_method_get_is_abstract (base_method) || vala_method_get_is_virtual (base_method)) {
			if (!vala_method_equals (m, base_method)) {
				char* _tmp3;
				char* _tmp2;
				char* _tmp1;
				vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				_tmp1 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp3 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))))));
				_tmp3 = (g_free (_tmp3), NULL);
				_tmp2 = (g_free (_tmp2), NULL);
				_tmp1 = (g_free (_tmp1), NULL);
				(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			vala_method_set_base_method (m, base_method);
			(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_method (self, m, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (m == NULL || VALA_IS_METHOD (m));
	g_return_if_fail (cl == NULL || VALA_IS_CLASS (cl));
	{
		GeeCollection* type_collection;
		GeeIterator* type_it;
		type_collection = vala_class_get_base_types (cl);
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaDataType* type;
			type = ((ValaDataType*) gee_iterator_get (type_it));
			{
				/* FIXME report error if multiple possible base methods are found*/
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_data_type_get_data_type (type))), vala_symbol_get_name (VALA_SYMBOL (m)));
					if (VALA_IS_METHOD (sym)) {
						ValaMethod* _tmp0;
						ValaMethod* base_method;
						_tmp0 = NULL;
						base_method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
						if (vala_method_get_is_abstract (base_method)) {
							if (!vala_method_equals (m, base_method)) {
								char* _tmp3;
								char* _tmp2;
								char* _tmp1;
								vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
								_tmp3 = NULL;
								_tmp2 = NULL;
								_tmp1 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp3 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))))));
								_tmp3 = (g_free (_tmp3), NULL);
								_tmp2 = (g_free (_tmp2), NULL);
								_tmp1 = (g_free (_tmp1), NULL);
								(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
								(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
								return;
							}
							vala_method_set_base_interface_method (m, base_method);
							(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
							(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
							return;
						}
						(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp3;
	ValaCreationMethod* _tmp2;
	ValaDataType* _tmp5;
	ValaDataType* _tmp4;
	ValaSymbol* _tmp7;
	ValaSymbol* _tmp6;
	ValaDataType* _tmp8;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (m == NULL || VALA_IS_CREATION_METHOD (m));
	if (vala_creation_method_get_type_name (m) != NULL && _vala_strcmp0 (vala_creation_method_get_type_name (m), vala_symbol_get_name (self->priv->current_symbol)) != 0) {
		char* _tmp1;
		char* _tmp0;
		/* type_name is null for constructors generated by GIdlParser*/
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp1 = g_strdup_printf ("missing return type in method `%s.%s´", (_tmp0 = vala_symbol_get_full_name (self->priv->current_symbol)), vala_creation_method_get_type_name (m))));
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp0 = (g_free (_tmp0), NULL);
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_symbol = (_tmp3 = VALA_SYMBOL ((_tmp2 = m, (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp3);
	_tmp5 = NULL;
	_tmp4 = NULL;
	self->priv->current_return_type = (_tmp5 = (_tmp4 = vala_method_get_return_type (VALA_METHOD (m)), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp5);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	_tmp7 = NULL;
	_tmp6 = NULL;
	self->priv->current_symbol = (_tmp7 = (_tmp6 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp7);
	_tmp8 = NULL;
	self->priv->current_return_type = (_tmp8 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp8);
	if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol))) {
		ValaMethod* _tmp9;
		ValaMethod* up_method;
		ValaDataType* _tmp11;
		ValaDataType* _tmp10;
		/* lambda expressions produce nested methods */
		_tmp9 = NULL;
		up_method = (_tmp9 = VALA_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol)), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9)));
		_tmp11 = NULL;
		_tmp10 = NULL;
		self->priv->current_return_type = (_tmp11 = (_tmp10 = vala_method_get_return_type (up_method), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp11);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (vala_method_get_is_abstract (VALA_METHOD (m)) || vala_method_get_is_virtual (VALA_METHOD (m)) || vala_method_get_overrides (VALA_METHOD (m))) {
		char* _tmp13;
		char* _tmp12;
		_tmp13 = NULL;
		_tmp12 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp13 = g_strdup_printf ("The creation method `%s' cannot be marked as override, virtual, or abstract", (_tmp12 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
		_tmp13 = (g_free (_tmp13), NULL);
		_tmp12 = (g_free (_tmp12), NULL);
		return;
	}
}


static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (p == NULL || VALA_IS_FORMAL_PARAMETER (p));
	vala_code_node_accept_children (VALA_CODE_NODE (p), VALA_CODE_VISITOR (self));
	if (!vala_formal_parameter_get_ellipsis (p)) {
		if (!vala_symbol_is_internal_symbol (VALA_SYMBOL (p))) {
			vala_source_file_add_type_dependency (self->priv->current_source_file, vala_formal_parameter_get_type_reference (p), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
		vala_source_file_add_type_dependency (self->priv->current_source_file, vala_formal_parameter_get_type_reference (p), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		/* check whether parameter type is at least as accessible as the method*/
		if (!vala_semantic_analyzer_is_type_accessible (self, VALA_SYMBOL (p), vala_formal_parameter_get_type_reference (p))) {
			char* _tmp2;
			char* _tmp1;
			char* _tmp0;
			vala_code_node_set_error (VALA_CODE_NODE (p), TRUE);
			_tmp2 = NULL;
			_tmp1 = NULL;
			_tmp0 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), (_tmp2 = g_strdup_printf ("parameter type `%s` is less accessible than method `%s`", (_tmp0 = vala_code_node_to_string (VALA_CODE_NODE (vala_formal_parameter_get_type_reference (p)))), (_tmp1 = vala_symbol_get_full_name (vala_symbol_get_parent_symbol (VALA_SYMBOL (p)))))));
			_tmp2 = (g_free (_tmp2), NULL);
			_tmp1 = (g_free (_tmp1), NULL);
			_tmp0 = (g_free (_tmp0), NULL);
			return;
		}
	}
	/* special treatment for construct formal parameters used in creation methods */
	if (vala_formal_parameter_get_construct_parameter (p)) {
		ValaBlock* _tmp3;
		ValaBlock* method_body;
		ValaMemberAccess* _tmp4;
		ValaMemberAccess* _tmp5;
		ValaMemberAccess* left;
		ValaMemberAccess* right;
		ValaExpressionStatement* _tmp7;
		ValaAssignment* _tmp6;
		if (!(VALA_IS_CREATION_METHOD (vala_symbol_get_parent_symbol (VALA_SYMBOL (p))))) {
			vala_code_node_set_error (VALA_CODE_NODE (p), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), "construct parameters are only allowed in type creation methods");
			return;
		}
		_tmp3 = NULL;
		method_body = (_tmp3 = vala_method_get_body (VALA_METHOD ((VALA_CREATION_METHOD (vala_symbol_get_parent_symbol (VALA_SYMBOL (p)))))), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
		_tmp4 = NULL;
		_tmp5 = NULL;
		left = (_tmp5 = vala_member_access_new (VALA_EXPRESSION ((_tmp4 = vala_member_access_new_simple ("this", NULL))), vala_symbol_get_name (VALA_SYMBOL (p)), NULL), (_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL))), _tmp5);
		right = vala_member_access_new_simple (vala_symbol_get_name (VALA_SYMBOL (p)), NULL);
		_tmp7 = NULL;
		_tmp6 = NULL;
		vala_block_add_statement (method_body, VALA_STATEMENT ((_tmp7 = vala_expression_statement_new (VALA_EXPRESSION ((_tmp6 = vala_code_context_create_assignment (self->priv->context, VALA_EXPRESSION (left), VALA_EXPRESSION (right), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL))), vala_code_node_get_source_reference (VALA_CODE_NODE (p))))));
		(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
		(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
		(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
		(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
		(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
	}
}


/* check whether type is at least as accessible as the specified symbol*/
static gboolean vala_semantic_analyzer_is_type_accessible (ValaSemanticAnalyzer* self, ValaSymbol* sym, ValaDataType* type) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (sym == NULL || VALA_IS_SYMBOL (sym), FALSE);
	g_return_val_if_fail (type == NULL || VALA_IS_DATA_TYPE (type), FALSE);
	{
		GeeCollection* type_symbol_collection;
		GeeIterator* type_symbol_it;
		type_symbol_collection = vala_data_type_get_symbols (type);
		type_symbol_it = gee_iterable_iterator (GEE_ITERABLE (type_symbol_collection));
		while (gee_iterator_next (type_symbol_it)) {
			ValaSymbol* type_symbol;
			type_symbol = ((ValaSymbol*) gee_iterator_get (type_symbol_it));
			{
				ValaScope* method_scope;
				ValaScope* type_scope;
				method_scope = vala_symbol_get_top_accessible_scope (sym);
				type_scope = vala_symbol_get_top_accessible_scope (type_symbol);
				if ((method_scope == NULL && type_scope != NULL) || (method_scope != NULL && !vala_scope_is_subscope_of (method_scope, type_scope))) {
					gboolean _tmp0;
					return (_tmp0 = FALSE, (type_symbol == NULL ? NULL : (type_symbol = (g_object_unref (type_symbol), NULL))), (method_scope == NULL ? NULL : (method_scope = (g_object_unref (method_scope), NULL))), (type_scope == NULL ? NULL : (type_scope = (g_object_unref (type_scope), NULL))), (type_symbol_collection == NULL ? NULL : (type_symbol_collection = (g_object_unref (type_symbol_collection), NULL))), (type_symbol_it == NULL ? NULL : (type_symbol_it = (g_object_unref (type_symbol_it), NULL))), _tmp0);
				}
				(type_symbol == NULL ? NULL : (type_symbol = (g_object_unref (type_symbol), NULL)));
				(method_scope == NULL ? NULL : (method_scope = (g_object_unref (method_scope), NULL)));
				(type_scope == NULL ? NULL : (type_scope = (g_object_unref (type_scope), NULL)));
			}
		}
		(type_symbol_collection == NULL ? NULL : (type_symbol_collection = (g_object_unref (type_symbol_collection), NULL)));
		(type_symbol_it == NULL ? NULL : (type_symbol_it = (g_object_unref (type_symbol_it), NULL)));
	}
	return TRUE;
}


static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl) {
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (prop == NULL || VALA_IS_PROPERTY (prop));
	g_return_if_fail (cl == NULL || VALA_IS_CLASS (cl));
	sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (cl)), vala_symbol_get_name (VALA_SYMBOL (prop)));
	if (VALA_IS_PROPERTY (sym)) {
		ValaProperty* _tmp0;
		ValaProperty* base_property;
		_tmp0 = NULL;
		base_property = (_tmp0 = VALA_PROPERTY (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (vala_property_get_is_abstract (base_property) || vala_property_get_is_virtual (base_property)) {
			if (!vala_property_equals (prop, base_property)) {
				char* _tmp3;
				char* _tmp2;
				char* _tmp1;
				vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				_tmp1 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp3 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (prop))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (base_property))))));
				_tmp3 = (g_free (_tmp3), NULL);
				_tmp2 = (g_free (_tmp2), NULL);
				_tmp1 = (g_free (_tmp1), NULL);
				(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			vala_property_set_base_property (prop, base_property);
			(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_property (self, prop, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (prop == NULL || VALA_IS_PROPERTY (prop));
	g_return_if_fail (cl == NULL || VALA_IS_CLASS (cl));
	{
		GeeCollection* type_collection;
		GeeIterator* type_it;
		type_collection = vala_class_get_base_types (cl);
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaDataType* type;
			type = ((ValaDataType*) gee_iterator_get (type_it));
			{
				/* FIXME report error if multiple possible base properties are found*/
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_data_type_get_data_type (type))), vala_symbol_get_name (VALA_SYMBOL (prop)));
					if (VALA_IS_PROPERTY (sym)) {
						ValaProperty* _tmp0;
						ValaProperty* base_property;
						_tmp0 = NULL;
						base_property = (_tmp0 = VALA_PROPERTY (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
						if (vala_property_get_is_abstract (base_property)) {
							if (!vala_property_equals (prop, base_property)) {
								char* _tmp3;
								char* _tmp2;
								char* _tmp1;
								vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
								_tmp3 = NULL;
								_tmp2 = NULL;
								_tmp1 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp3 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (prop))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (base_property))))));
								_tmp3 = (g_free (_tmp3), NULL);
								_tmp2 = (g_free (_tmp2), NULL);
								_tmp1 = (g_free (_tmp1), NULL);
								(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
								(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
								return;
							}
							vala_property_set_base_interface_property (prop, base_property);
							(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
							(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
							return;
						}
						(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp1;
	ValaProperty* _tmp0;
	ValaSymbol* _tmp9;
	ValaSymbol* _tmp8;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (prop == NULL || VALA_IS_PROPERTY (prop));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL ((_tmp0 = prop, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	if (!vala_property_get_instance (prop)) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), "static properties are not yet supported");
		vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
		return;
	}
	vala_code_node_accept_children (VALA_CODE_NODE (prop), VALA_CODE_VISITOR (self));
	/* check whether property type is at least as accessible as the property*/
	if (!vala_semantic_analyzer_is_type_accessible (self, VALA_SYMBOL (prop), vala_property_get_type_reference (prop))) {
		char* _tmp4;
		char* _tmp3;
		char* _tmp2;
		vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
		_tmp4 = NULL;
		_tmp3 = NULL;
		_tmp2 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp4 = g_strdup_printf ("property type `%s` is less accessible than property `%s`", (_tmp2 = vala_code_node_to_string (VALA_CODE_NODE (vala_property_get_type_reference (prop)))), (_tmp3 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
		_tmp4 = (g_free (_tmp4), NULL);
		_tmp3 = (g_free (_tmp3), NULL);
		_tmp2 = (g_free (_tmp2), NULL);
		return;
	}
	/* abstract/virtual properties using reference types without
	 * reference counting need to transfer ownership of their
	 * return values because of limitations in the GObject property
	 * system (g_object_get always returns strong references).
	 * Reference counting types can simulate to return a weak
	 * reference */
	if ((vala_property_get_is_abstract (prop) || vala_property_get_is_virtual (prop)) && vala_data_type_get_data_type (vala_property_get_type_reference (prop)) != NULL && vala_typesymbol_is_reference_type (vala_data_type_get_data_type (vala_property_get_type_reference (prop))) && !vala_typesymbol_is_reference_counting (vala_data_type_get_data_type (vala_property_get_type_reference (prop))) && !vala_data_type_get_transfers_ownership (vala_property_get_type_reference (prop))) {
		char* _tmp7;
		char* _tmp6;
		char* _tmp5;
		_tmp7 = NULL;
		_tmp6 = NULL;
		_tmp5 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp7 = g_strdup_printf ("%s: abstract or virtual properties using reference types not supporting reference counting, like `%s', have to mark their return value to transfer ownership.", (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (prop))), (_tmp6 = vala_symbol_get_full_name (VALA_SYMBOL (vala_data_type_get_data_type (vala_property_get_type_reference (prop))))))));
		_tmp7 = (g_free (_tmp7), NULL);
		_tmp6 = (g_free (_tmp6), NULL);
		_tmp5 = (g_free (_tmp5), NULL);
		vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
	}
	_tmp9 = NULL;
	_tmp8 = NULL;
	self->priv->current_symbol = (_tmp9 = (_tmp8 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp9);
	if (!vala_symbol_is_internal_symbol (VALA_SYMBOL (prop))) {
		vala_source_file_add_type_dependency (self->priv->current_source_file, vala_property_get_type_reference (prop), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
	}
	vala_source_file_add_type_dependency (self->priv->current_source_file, vala_property_get_type_reference (prop), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (prop)))) {
		ValaClass* _tmp10;
		ValaClass* cl;
		_tmp10 = NULL;
		cl = (_tmp10 = VALA_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (prop))), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
		/* VAPI classes don't specify overridden properties */
		if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl))))) {
			vala_semantic_analyzer_find_base_interface_property (self, prop, cl);
			if (vala_property_get_is_virtual (prop) || vala_property_get_overrides (prop)) {
				vala_semantic_analyzer_find_base_class_property (self, prop, cl);
				if (vala_property_get_base_property (prop) == NULL) {
					char* _tmp12;
					char* _tmp11;
					vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
					_tmp12 = NULL;
					_tmp11 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp12 = g_strdup_printf ("%s: no suitable property found to override", (_tmp11 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
					_tmp12 = (g_free (_tmp12), NULL);
					_tmp11 = (g_free (_tmp11), NULL);
				}
			}
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
	/* construct properties must be public */
	if (vala_property_get_set_accessor (prop) != NULL && vala_property_accessor_get_construction (vala_property_get_set_accessor (prop))) {
		if (vala_symbol_get_access (VALA_SYMBOL (prop)) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
			char* _tmp14;
			char* _tmp13;
			vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
			_tmp14 = NULL;
			_tmp13 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp14 = g_strdup_printf ("%s: construct properties must be public", (_tmp13 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
			_tmp14 = (g_free (_tmp14), NULL);
			_tmp13 = (g_free (_tmp13), NULL);
		}
	}
}


static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc) {
	ValaSemanticAnalyzer * self;
	ValaDataType* _tmp13;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (acc == NULL || VALA_IS_PROPERTY_ACCESSOR (acc));
	vala_property_accessor_set_prop (acc, VALA_PROPERTY (self->priv->current_symbol));
	if (vala_property_accessor_get_readable (acc)) {
		ValaDataType* _tmp1;
		ValaDataType* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->current_return_type = (_tmp1 = (_tmp0 = vala_property_get_type_reference (vala_property_accessor_get_prop (acc)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp1);
	} else {
		ValaDataType* _tmp2;
		/* void*/
		_tmp2 = NULL;
		self->priv->current_return_type = (_tmp2 = VALA_DATA_TYPE (vala_void_type_new ()), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp2);
	}
	if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (acc))))) {
		if (vala_property_accessor_get_body (acc) == NULL && !vala_property_get_interface_only (vala_property_accessor_get_prop (acc)) && !vala_property_get_is_abstract (vala_property_accessor_get_prop (acc))) {
			ValaBlock* _tmp3;
			/* no accessor body specified, insert default body */
			if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol (VALA_SYMBOL (vala_property_accessor_get_prop (acc))))) {
				vala_code_node_set_error (VALA_CODE_NODE (acc), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (acc)), "Automatic properties can't be used in interfaces");
				return;
			}
			vala_property_accessor_set_automatic_body (acc, TRUE);
			_tmp3 = NULL;
			vala_property_accessor_set_body (acc, (_tmp3 = vala_block_new (NULL)));
			(_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)));
			if (vala_property_accessor_get_readable (acc)) {
				ValaReturnStatement* _tmp6;
				ValaMemberAccess* _tmp5;
				char* _tmp4;
				_tmp6 = NULL;
				_tmp5 = NULL;
				_tmp4 = NULL;
				vala_block_add_statement (vala_property_accessor_get_body (acc), VALA_STATEMENT ((_tmp6 = vala_return_statement_new (VALA_EXPRESSION ((_tmp5 = vala_member_access_new_simple ((_tmp4 = g_strdup_printf ("_%s", vala_symbol_get_name (VALA_SYMBOL (vala_property_accessor_get_prop (acc))))), NULL))), vala_code_node_get_source_reference (VALA_CODE_NODE (acc))))));
				(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
				(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
				_tmp4 = (g_free (_tmp4), NULL);
			} else {
				ValaExpressionStatement* _tmp11;
				ValaAssignment* _tmp10;
				ValaMemberAccess* _tmp9;
				ValaMemberAccess* _tmp8;
				char* _tmp7;
				_tmp11 = NULL;
				_tmp10 = NULL;
				_tmp9 = NULL;
				_tmp8 = NULL;
				_tmp7 = NULL;
				vala_block_add_statement (vala_property_accessor_get_body (acc), VALA_STATEMENT ((_tmp11 = vala_expression_statement_new (VALA_EXPRESSION ((_tmp10 = vala_code_context_create_assignment (self->priv->context, VALA_EXPRESSION ((_tmp8 = vala_member_access_new_simple ((_tmp7 = g_strdup_printf ("_%s", vala_symbol_get_name (VALA_SYMBOL (vala_property_accessor_get_prop (acc))))), NULL))), VALA_EXPRESSION ((_tmp9 = vala_member_access_new_simple ("value", NULL))), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL))), vala_code_node_get_source_reference (VALA_CODE_NODE (acc))))));
				(_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL)));
				(_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL)));
				(_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL)));
				(_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL)));
				_tmp7 = (g_free (_tmp7), NULL);
			}
		}
		if (vala_property_accessor_get_body (acc) != NULL && (vala_property_accessor_get_writable (acc) || vala_property_accessor_get_construction (acc))) {
			ValaFormalParameter* _tmp12;
			_tmp12 = NULL;
			vala_property_accessor_set_value_parameter (acc, (_tmp12 = vala_formal_parameter_new ("value", vala_property_get_type_reference (vala_property_accessor_get_prop (acc)), vala_code_node_get_source_reference (VALA_CODE_NODE (acc)))));
			(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
			vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (vala_property_accessor_get_body (acc))), vala_symbol_get_name (VALA_SYMBOL (vala_property_accessor_get_value_parameter (acc))), VALA_SYMBOL (vala_property_accessor_get_value_parameter (acc)));
		}
	}
	vala_code_node_accept_children (VALA_CODE_NODE (acc), VALA_CODE_VISITOR (self));
	_tmp13 = NULL;
	self->priv->current_return_type = (_tmp13 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp13);
}


static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (sig == NULL || VALA_IS_SIGNAL (sig));
	vala_code_node_accept_children (VALA_CODE_NODE (sig), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c) {
	ValaSemanticAnalyzer * self;
	ValaFormalParameter* _tmp1;
	ValaClassType* _tmp0;
	ValaSymbol* _tmp3;
	ValaConstructor* _tmp2;
	ValaSymbol* _tmp5;
	ValaSymbol* _tmp4;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (c == NULL || VALA_IS_CONSTRUCTOR (c));
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_constructor_set_this_parameter (c, (_tmp1 = vala_formal_parameter_new ("this", VALA_DATA_TYPE ((_tmp0 = vala_class_type_new (self->priv->current_class))), NULL)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (c)), vala_symbol_get_name (VALA_SYMBOL (vala_constructor_get_this_parameter (c))), VALA_SYMBOL (vala_constructor_get_this_parameter (c)));
	vala_symbol_set_owner (VALA_SYMBOL (c), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_symbol = (_tmp3 = VALA_SYMBOL ((_tmp2 = c, (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp3);
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	_tmp5 = NULL;
	_tmp4 = NULL;
	self->priv->current_symbol = (_tmp5 = (_tmp4 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp5);
}


static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp1;
	ValaDestructor* _tmp0;
	ValaSymbol* _tmp3;
	ValaSymbol* _tmp2;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (d == NULL || VALA_IS_DESTRUCTOR (d));
	vala_symbol_set_owner (VALA_SYMBOL (d), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL ((_tmp0 = d, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_symbol = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp3);
}


static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (n == NULL || VALA_IS_NAMED_ARGUMENT (n));
}


static void vala_semantic_analyzer_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp1;
	ValaBlock* _tmp0;
	ValaSymbol* _tmp3;
	ValaSymbol* _tmp2;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (b == NULL || VALA_IS_BLOCK (b));
	vala_symbol_set_owner (VALA_SYMBOL (b), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL ((_tmp0 = b, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (b), VALA_CODE_VISITOR (self));
	{
		GeeCollection* decl_collection;
		GeeIterator* decl_it;
		decl_collection = vala_block_get_local_variables (b);
		decl_it = gee_iterable_iterator (GEE_ITERABLE (decl_collection));
		while (gee_iterator_next (decl_it)) {
			ValaVariableDeclarator* decl;
			decl = ((ValaVariableDeclarator*) gee_iterator_get (decl_it));
			{
				vala_symbol_set_active (VALA_SYMBOL (decl), FALSE);
				(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
			}
		}
		(decl_collection == NULL ? NULL : (decl_collection = (g_object_unref (decl_collection), NULL)));
		(decl_it == NULL ? NULL : (decl_it = (g_object_unref (decl_it), NULL)));
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_symbol = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp3);
}


static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl) {
	ValaSemanticAnalyzer * self;
	ValaBlock* _tmp10;
	ValaBlock* block;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (decl == NULL || VALA_IS_VARIABLE_DECLARATOR (decl));
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		vala_expression_set_expected_type (vala_variable_declarator_get_initializer (decl), vala_variable_declarator_get_type_reference (decl));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (decl), VALA_CODE_VISITOR (self));
	if (vala_variable_declarator_get_type_reference (decl) == NULL) {
		ValaDataType* _tmp0;
		/* var type */
		if (vala_variable_declarator_get_initializer (decl) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed without initializer");
			return;
		}
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed with non-typed initializer");
			return;
		}
		_tmp0 = NULL;
		vala_variable_declarator_set_type_reference (decl, (_tmp0 = vala_data_type_copy (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
		vala_data_type_set_takes_ownership (vala_variable_declarator_get_type_reference (decl), (vala_data_type_get_data_type (vala_variable_declarator_get_type_reference (decl)) == NULL || vala_typesymbol_is_reference_type (vala_data_type_get_data_type (vala_variable_declarator_get_type_reference (decl)))));
		vala_data_type_set_transfers_ownership (vala_variable_declarator_get_type_reference (decl), FALSE);
	}
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			if (!(VALA_IS_MEMBER_ACCESS (vala_variable_declarator_get_initializer (decl))) && !(VALA_IS_LAMBDA_EXPRESSION (vala_variable_declarator_get_initializer (decl)))) {
				vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
			if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl))) && VALA_IS_DELEGATE_TYPE (vala_variable_declarator_get_type_reference (decl))) {
				ValaMethod* _tmp1;
				ValaMethod* m;
				ValaDelegateType* _tmp2;
				ValaDelegateType* dt;
				ValaDelegate* _tmp3;
				ValaDelegate* cb;
				_tmp1 = NULL;
				m = (_tmp1 = VALA_METHOD (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl))), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
				_tmp2 = NULL;
				dt = (_tmp2 = VALA_DELEGATE_TYPE (vala_variable_declarator_get_type_reference (decl)), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
				_tmp3 = NULL;
				cb = (_tmp3 = vala_delegate_type_get_delegate_symbol (dt), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
				/* check whether method matches callback type */
				if (!vala_delegate_matches_method (cb, m)) {
					char* _tmp6;
					char* _tmp5;
					char* _tmp4;
					vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
					_tmp6 = NULL;
					_tmp5 = NULL;
					_tmp4 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), (_tmp6 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (_tmp4 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
					_tmp6 = (g_free (_tmp6), NULL);
					_tmp5 = (g_free (_tmp5), NULL);
					_tmp4 = (g_free (_tmp4), NULL);
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					return;
				}
				vala_expression_set_static_type (vala_variable_declarator_get_initializer (decl), vala_variable_declarator_get_type_reference (decl));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			} else {
				vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
		}
		if (!vala_data_type_compatible (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)), vala_variable_declarator_get_type_reference (decl), TRUE)) {
			char* _tmp9;
			char* _tmp8;
			char* _tmp7;
			vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
			_tmp9 = NULL;
			_tmp8 = NULL;
			_tmp7 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), (_tmp9 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (_tmp7 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl))))), (_tmp8 = vala_code_node_to_string (VALA_CODE_NODE (vala_variable_declarator_get_type_reference (decl)))))));
			_tmp9 = (g_free (_tmp9), NULL);
			_tmp8 = (g_free (_tmp8), NULL);
			_tmp7 = (g_free (_tmp7), NULL);
			return;
		}
		if (vala_data_type_get_transfers_ownership (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))) {
			/* rhs transfers ownership of the expression */
			if (!(VALA_IS_POINTER_TYPE (vala_variable_declarator_get_type_reference (decl))) && !vala_data_type_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
				/* lhs doesn't own the value */
				vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "Invalid assignment from owned expression to unowned variable");
				return;
			}
		}
	}
	vala_source_file_add_type_dependency (self->priv->current_source_file, vala_variable_declarator_get_type_reference (decl), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	vala_scope_add (vala_symbol_get_scope (self->priv->current_symbol), vala_symbol_get_name (VALA_SYMBOL (decl)), VALA_SYMBOL (decl));
	_tmp10 = NULL;
	block = (_tmp10 = VALA_BLOCK (self->priv->current_symbol), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
	vala_block_add_local_variable (block, decl);
	vala_symbol_set_active (VALA_SYMBOL (decl), TRUE);
	(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list) {
	ValaSemanticAnalyzer * self;
	gboolean error;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (list == NULL || VALA_IS_INITIALIZER_LIST (list));
	if (VALA_IS_ARRAY_TYPE (vala_expression_get_expected_type (VALA_EXPRESSION (list)))) {
		ValaArrayType* _tmp0;
		ValaArrayType* array_type;
		/* initializer is used as array initializer */
		_tmp0 = NULL;
		array_type = (_tmp0 = VALA_ARRAY_TYPE (vala_expression_get_expected_type (VALA_EXPRESSION (list))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		{
			GeeCollection* e_collection;
			GeeIterator* e_it;
			e_collection = vala_initializer_list_get_initializers (list);
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = ((ValaExpression*) gee_iterator_get (e_it));
				{
					ValaDataType* _tmp1;
					_tmp1 = NULL;
					vala_expression_set_expected_type (e, (_tmp1 = vala_data_type_copy (vala_array_type_get_element_type (array_type))));
					(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
		(array_type == NULL ? NULL : (array_type = (g_object_unref (array_type), NULL)));
	} else {
		if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_STRUCT (vala_data_type_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
			ValaStruct* _tmp2;
			ValaStruct* st;
			GeeCollection* _tmp3;
			GeeIterator* _tmp4;
			GeeIterator* field_it;
			/* initializer is used as struct initializer */
			_tmp2 = NULL;
			st = (_tmp2 = VALA_STRUCT (vala_data_type_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
			_tmp3 = NULL;
			_tmp4 = NULL;
			field_it = (_tmp4 = gee_iterable_iterator (GEE_ITERABLE ((_tmp3 = vala_struct_get_fields (st)))), (_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL))), _tmp4);
			{
				GeeCollection* e_collection;
				GeeIterator* e_it;
				e_collection = vala_initializer_list_get_initializers (list);
				e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
				while (gee_iterator_next (e_it)) {
					ValaExpression* e;
					e = ((ValaExpression*) gee_iterator_get (e_it));
					{
						ValaField* field;
						ValaDataType* _tmp9;
						field = NULL;
						while (field == NULL) {
							ValaField* _tmp7;
							if (!gee_iterator_next (field_it)) {
								char* _tmp6;
								char* _tmp5;
								vala_code_node_set_error (VALA_CODE_NODE (list), TRUE);
								_tmp6 = NULL;
								_tmp5 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (_tmp6 = g_strdup_printf ("too many expressions in initializer list for `%s'", (_tmp5 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_expected_type (VALA_EXPRESSION (list))))))));
								_tmp6 = (g_free (_tmp6), NULL);
								_tmp5 = (g_free (_tmp5), NULL);
								(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
								(field == NULL ? NULL : (field = (g_object_unref (field), NULL)));
								(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
								(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
								(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
								(field_it == NULL ? NULL : (field_it = (g_object_unref (field_it), NULL)));
								return;
							}
							_tmp7 = NULL;
							field = (_tmp7 = ((ValaField*) gee_iterator_get (field_it)), (field == NULL ? NULL : (field = (g_object_unref (field), NULL))), _tmp7);
							if (!vala_field_get_instance (field)) {
								ValaField* _tmp8;
								/* we only initialize instance fields*/
								_tmp8 = NULL;
								field = (_tmp8 = NULL, (field == NULL ? NULL : (field = (g_object_unref (field), NULL))), _tmp8);
							}
						}
						_tmp9 = NULL;
						vala_expression_set_expected_type (e, (_tmp9 = vala_data_type_copy (vala_field_get_type_reference (field))));
						(_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL)));
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
						(field == NULL ? NULL : (field = (g_object_unref (field), NULL)));
					}
				}
				(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
				(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			(field_it == NULL ? NULL : (field_it = (g_object_unref (field_it), NULL)));
		} else {
			if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) == NULL) {
				vala_code_node_set_error (VALA_CODE_NODE (list), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (list)), "initializer list used for unknown type");
				return;
			} else {
				char* _tmp11;
				char* _tmp10;
				vala_code_node_set_error (VALA_CODE_NODE (list), TRUE);
				_tmp11 = NULL;
				_tmp10 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (list)), (_tmp11 = g_strdup_printf ("initializer list used for `%s', which is neither array nor struct", (_tmp10 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_expected_type (VALA_EXPRESSION (list))))))));
				_tmp11 = (g_free (_tmp11), NULL);
				_tmp10 = (g_free (_tmp10), NULL);
				return;
			}
		}
	}
	vala_code_node_accept_children (VALA_CODE_NODE (list), VALA_CODE_VISITOR (self));
	error = FALSE;
	{
		GeeCollection* e_collection;
		GeeIterator* e_it;
		e_collection = vala_initializer_list_get_initializers (list);
		e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
		while (gee_iterator_next (e_it)) {
			ValaExpression* e;
			e = ((ValaExpression*) gee_iterator_get (e_it));
			{
				ValaUnaryExpression* _tmp13;
				ValaExpression* _tmp12;
				ValaUnaryExpression* unary;
				if (vala_expression_get_static_type (e) == NULL) {
					error = TRUE;
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
					continue;
				}
				_tmp13 = NULL;
				_tmp12 = NULL;
				unary = (_tmp13 = (_tmp12 = e, (VALA_IS_UNARY_EXPRESSION (_tmp12) ? ((ValaUnaryExpression*) _tmp12) : NULL)), (_tmp13 == NULL ? NULL : g_object_ref (_tmp13)));
				if (unary != NULL && (vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_REF || vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_OUT)) {
				} else {
					/* TODO check type for ref and out expressions*/
					if (!vala_data_type_compatible (vala_expression_get_static_type (e), vala_expression_get_expected_type (e), TRUE)) {
						if (!vala_data_type_compatible (vala_expression_get_static_type (e), vala_expression_get_expected_type (e), FALSE)) {
							char* _tmp16;
							char* _tmp15;
							char* _tmp14;
							error = TRUE;
							vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
							_tmp16 = NULL;
							_tmp15 = NULL;
							_tmp14 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (_tmp16 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", (_tmp14 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_expected_type (e)))), (_tmp15 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (e)))))));
							_tmp16 = (g_free (_tmp16), NULL);
							_tmp15 = (g_free (_tmp15), NULL);
							_tmp14 = (g_free (_tmp14), NULL);
						} else {
							if (vala_code_context_is_non_null_enabled ()) {
								char* _tmp19;
								char* _tmp18;
								char* _tmp17;
								_tmp19 = NULL;
								_tmp18 = NULL;
								_tmp17 = NULL;
								vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (_tmp19 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", (_tmp17 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_expected_type (e)))), (_tmp18 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (e)))))));
								_tmp19 = (g_free (_tmp19), NULL);
								_tmp18 = (g_free (_tmp18), NULL);
								_tmp17 = (g_free (_tmp17), NULL);
							}
						}
					}
				}
				(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				(unary == NULL ? NULL : (unary = (g_object_unref (unary), NULL)));
			}
		}
		(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
		(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
	}
	if (!error) {
		/* everything seems to be correct */
		vala_expression_set_static_type (VALA_EXPRESSION (list), vala_expression_get_expected_type (VALA_EXPRESSION (list)));
	}
}


static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_EXPRESSION_STATEMENT (stmt));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_expression_statement_get_expression (stmt)))) {
		/* ignore inner error*/
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	vala_code_node_set_tree_can_fail (VALA_CODE_NODE (stmt), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_expression_statement_get_expression (stmt))));
}


static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_IF_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_if_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (!vala_data_type_compatible (vala_expression_get_static_type (vala_if_statement_get_condition (stmt)), self->priv->bool_type, TRUE)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_if_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp1;
	ValaSwitchSection* _tmp0;
	ValaSymbol* _tmp3;
	ValaSymbol* _tmp2;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (section == NULL || VALA_IS_SWITCH_SECTION (section));
	{
		GeeCollection* label_collection;
		GeeIterator* label_it;
		label_collection = vala_switch_section_get_labels (section);
		label_it = gee_iterable_iterator (GEE_ITERABLE (label_collection));
		while (gee_iterator_next (label_it)) {
			ValaSwitchLabel* label;
			label = ((ValaSwitchLabel*) gee_iterator_get (label_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (label), VALA_CODE_VISITOR (self));
				(label == NULL ? NULL : (label = (g_object_unref (label), NULL)));
			}
		}
		(label_collection == NULL ? NULL : (label_collection = (g_object_unref (label_collection), NULL)));
		(label_it == NULL ? NULL : (label_it = (g_object_unref (label_it), NULL)));
	}
	vala_symbol_set_owner (VALA_SYMBOL (section), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL ((_tmp0 = section, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	{
		GeeCollection* st_collection;
		GeeIterator* st_it;
		st_collection = vala_switch_section_get_statements (section);
		st_it = gee_iterable_iterator (GEE_ITERABLE (st_collection));
		while (gee_iterator_next (st_it)) {
			ValaStatement* st;
			st = ((ValaStatement*) gee_iterator_get (st_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			}
		}
		(st_collection == NULL ? NULL : (st_collection = (g_object_unref (st_collection), NULL)));
		(st_it == NULL ? NULL : (st_it = (g_object_unref (st_it), NULL)));
	}
	{
		GeeCollection* decl_collection;
		GeeIterator* decl_it;
		decl_collection = vala_block_get_local_variables (VALA_BLOCK (section));
		decl_it = gee_iterable_iterator (GEE_ITERABLE (decl_collection));
		while (gee_iterator_next (decl_it)) {
			ValaVariableDeclarator* decl;
			decl = ((ValaVariableDeclarator*) gee_iterator_get (decl_it));
			{
				vala_symbol_set_active (VALA_SYMBOL (decl), FALSE);
				(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
			}
		}
		(decl_collection == NULL ? NULL : (decl_collection = (g_object_unref (decl_collection), NULL)));
		(decl_it == NULL ? NULL : (decl_it = (g_object_unref (decl_it), NULL)));
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_symbol = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp3);
}


static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_WHILE_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_while_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (!vala_data_type_compatible (vala_expression_get_static_type (vala_while_statement_get_condition (stmt)), self->priv->bool_type, TRUE)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_while_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_DO_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_do_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (!vala_data_type_compatible (vala_expression_get_static_type (vala_do_statement_get_condition (stmt)), self->priv->bool_type, TRUE)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_do_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_FOR_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	if (vala_for_statement_get_condition (stmt) != NULL && vala_code_node_get_error (VALA_CODE_NODE (vala_for_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (vala_for_statement_get_condition (stmt) != NULL && !vala_data_type_compatible (vala_expression_get_static_type (vala_for_statement_get_condition (stmt)), self->priv->bool_type, TRUE)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_for_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* _tmp0;
	ValaSymbol* _tmp2;
	ValaForeachStatement* _tmp1;
	ValaSymbol* _tmp4;
	ValaSymbol* _tmp3;
	ValaVariableDeclarator* _tmp6;
	char* _tmp5;
	ValaDataType* _tmp7;
	ValaDataType* _tmp8;
	ValaDataType* collection_type;
	ValaDataType* element_data_type;
	gboolean need_type_check;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_FOREACH_STATEMENT (stmt));
	vala_source_file_add_type_dependency (self->priv->current_source_file, vala_foreach_statement_get_type_reference (stmt), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	_tmp0 = NULL;
	vala_foreach_statement_set_variable_declarator (stmt, (_tmp0 = vala_variable_declarator_new (vala_foreach_statement_get_variable_name (stmt), NULL, NULL)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	vala_variable_declarator_set_type_reference (vala_foreach_statement_get_variable_declarator (stmt), vala_foreach_statement_get_type_reference (stmt));
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (vala_foreach_statement_get_body (stmt))), vala_foreach_statement_get_variable_name (stmt), VALA_SYMBOL (vala_foreach_statement_get_variable_declarator (stmt)));
	vala_block_add_local_variable (vala_foreach_statement_get_body (stmt), vala_foreach_statement_get_variable_declarator (stmt));
	vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_variable_declarator (stmt)), TRUE);
	vala_symbol_set_owner (VALA_SYMBOL (stmt), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_symbol = (_tmp2 = VALA_SYMBOL ((_tmp1 = stmt, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp2);
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	{
		GeeCollection* decl_collection;
		GeeIterator* decl_it;
		decl_collection = vala_block_get_local_variables (VALA_BLOCK (stmt));
		decl_it = gee_iterable_iterator (GEE_ITERABLE (decl_collection));
		while (gee_iterator_next (decl_it)) {
			ValaVariableDeclarator* decl;
			decl = ((ValaVariableDeclarator*) gee_iterator_get (decl_it));
			{
				vala_symbol_set_active (VALA_SYMBOL (decl), FALSE);
				(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
			}
		}
		(decl_collection == NULL ? NULL : (decl_collection = (g_object_unref (decl_collection), NULL)));
		(decl_it == NULL ? NULL : (decl_it = (g_object_unref (decl_it), NULL)));
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_symbol = (_tmp4 = (_tmp3 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp4);
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_foreach_statement_get_collection (stmt)))) {
		/* ignore inner error*/
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	} else {
		if (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt)) == NULL) {
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_foreach_statement_get_collection (stmt))), "invalid collection expression");
			vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
			return;
		}
	}
	_tmp6 = NULL;
	_tmp5 = NULL;
	vala_foreach_statement_set_collection_variable_declarator (stmt, (_tmp6 = vala_variable_declarator_new ((_tmp5 = g_strdup_printf ("%s_collection", vala_foreach_statement_get_variable_name (stmt))), NULL, NULL)));
	(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
	_tmp5 = (g_free (_tmp5), NULL);
	_tmp7 = NULL;
	vala_variable_declarator_set_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt), (_tmp7 = vala_data_type_copy (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt)))));
	(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
	vala_data_type_set_transfers_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt)), FALSE);
	vala_data_type_set_takes_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt)), vala_data_type_get_transfers_ownership (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt))));
	vala_block_add_local_variable (VALA_BLOCK (stmt), vala_foreach_statement_get_collection_variable_declarator (stmt));
	vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_collection_variable_declarator (stmt)), TRUE);
	_tmp8 = NULL;
	collection_type = (_tmp8 = vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt)), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8)));
	element_data_type = NULL;
	need_type_check = FALSE;
	if (vala_data_type_is_array (collection_type)) {
		ValaArrayType* _tmp9;
		ValaArrayType* array_type;
		ValaDataType* _tmp11;
		ValaDataType* _tmp10;
		_tmp9 = NULL;
		array_type = (_tmp9 = VALA_ARRAY_TYPE (collection_type), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9)));
		_tmp11 = NULL;
		_tmp10 = NULL;
		element_data_type = (_tmp11 = (_tmp10 = vala_array_type_get_element_type (array_type), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10))), (element_data_type == NULL ? NULL : (element_data_type = (g_object_unref (element_data_type), NULL))), _tmp11);
		need_type_check = TRUE;
		(array_type == NULL ? NULL : (array_type = (g_object_unref (array_type), NULL)));
	} else {
		if (vala_data_type_compatible (collection_type, self->priv->glist_type, TRUE) || vala_data_type_compatible (collection_type, self->priv->gslist_type, TRUE)) {
			GeeList* _tmp12;
			gboolean _tmp13;
			_tmp12 = NULL;
			if ((_tmp13 = gee_collection_get_size (GEE_COLLECTION ((_tmp12 = vala_data_type_get_type_arguments (collection_type)))) > 0, (_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL))), _tmp13)) {
				ValaDataType* _tmp15;
				GeeList* _tmp14;
				_tmp15 = NULL;
				_tmp14 = NULL;
				element_data_type = (_tmp15 = VALA_DATA_TYPE (((ValaDataType*) gee_list_get ((_tmp14 = vala_data_type_get_type_arguments (collection_type)), 0))), (element_data_type == NULL ? NULL : (element_data_type = (g_object_unref (element_data_type), NULL))), _tmp15);
				(_tmp14 == NULL ? NULL : (_tmp14 = (g_object_unref (_tmp14), NULL)));
				need_type_check = TRUE;
			}
		} else {
			if (self->priv->iterable_type != NULL && vala_data_type_compatible (collection_type, self->priv->iterable_type, TRUE)) {
				ValaVariableDeclarator* _tmp17;
				char* _tmp16;
				ValaInterfaceType* _tmp18;
				ValaMethod* it_method;
				GeeList* _tmp19;
				gboolean _tmp20;
				_tmp17 = NULL;
				_tmp16 = NULL;
				vala_foreach_statement_set_iterator_variable_declarator (stmt, (_tmp17 = vala_variable_declarator_new ((_tmp16 = g_strdup_printf ("%s_it", vala_foreach_statement_get_variable_name (stmt))), NULL, NULL)));
				(_tmp17 == NULL ? NULL : (_tmp17 = (g_object_unref (_tmp17), NULL)));
				_tmp16 = (g_free (_tmp16), NULL);
				_tmp18 = NULL;
				vala_variable_declarator_set_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt), VALA_DATA_TYPE ((_tmp18 = vala_interface_type_new (self->priv->iterator_type))));
				(_tmp18 == NULL ? NULL : (_tmp18 = (g_object_unref (_tmp18), NULL)));
				vala_data_type_set_takes_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt)), TRUE);
				vala_data_type_add_type_argument (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt)), vala_foreach_statement_get_type_reference (stmt));
				vala_block_add_local_variable (VALA_BLOCK (stmt), vala_foreach_statement_get_iterator_variable_declarator (stmt));
				vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_iterator_variable_declarator (stmt)), TRUE);
				it_method = VALA_METHOD (vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_data_type_get_data_type (self->priv->iterable_type))), "iterator"));
				_tmp19 = NULL;
				if ((_tmp20 = gee_collection_get_size (GEE_COLLECTION ((_tmp19 = vala_data_type_get_type_arguments (vala_method_get_return_type (it_method))))) > 0, (_tmp19 == NULL ? NULL : (_tmp19 = (g_object_unref (_tmp19), NULL))), _tmp20)) {
					GeeList* _tmp21;
					ValaDataType* _tmp22;
					ValaDataType* type_arg;
					_tmp21 = NULL;
					_tmp22 = NULL;
					type_arg = (_tmp22 = ((ValaDataType*) gee_list_get ((_tmp21 = vala_data_type_get_type_arguments (vala_method_get_return_type (it_method))), 0)), (_tmp21 == NULL ? NULL : (_tmp21 = (g_object_unref (_tmp21), NULL))), _tmp22);
					if (vala_data_type_get_type_parameter (type_arg) != NULL) {
						ValaDataType* _tmp23;
						_tmp23 = NULL;
						element_data_type = (_tmp23 = vala_semantic_analyzer_get_actual_type (collection_type, VALA_SYMBOL (it_method), type_arg, VALA_CODE_NODE (stmt)), (element_data_type == NULL ? NULL : (element_data_type = (g_object_unref (element_data_type), NULL))), _tmp23);
					} else {
						ValaDataType* _tmp25;
						ValaDataType* _tmp24;
						_tmp25 = NULL;
						_tmp24 = NULL;
						element_data_type = (_tmp25 = (_tmp24 = type_arg, (_tmp24 == NULL ? NULL : g_object_ref (_tmp24))), (element_data_type == NULL ? NULL : (element_data_type = (g_object_unref (element_data_type), NULL))), _tmp25);
					}
					need_type_check = TRUE;
					(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
				}
				(it_method == NULL ? NULL : (it_method = (g_object_unref (it_method), NULL)));
			} else {
				vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Collection not iterable");
				(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
				(element_data_type == NULL ? NULL : (element_data_type = (g_object_unref (element_data_type), NULL)));
				return;
			}
		}
	}
	if (need_type_check && element_data_type != NULL) {
		/* allow implicit upcasts ? */
		if (!vala_data_type_compatible (element_data_type, vala_foreach_statement_get_type_reference (stmt), TRUE)) {
			char* _tmp28;
			char* _tmp27;
			char* _tmp26;
			vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
			_tmp28 = NULL;
			_tmp27 = NULL;
			_tmp26 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), (_tmp28 = g_strdup_printf ("Foreach: Cannot convert from `%s' to `%s'", (_tmp26 = vala_code_node_to_string (VALA_CODE_NODE (element_data_type))), (_tmp27 = vala_code_node_to_string (VALA_CODE_NODE (vala_foreach_statement_get_type_reference (stmt)))))));
			_tmp28 = (g_free (_tmp28), NULL);
			_tmp27 = (g_free (_tmp27), NULL);
			_tmp26 = (g_free (_tmp26), NULL);
			(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
			(element_data_type == NULL ? NULL : (element_data_type = (g_object_unref (element_data_type), NULL)));
			return;
		}
	}
	vala_code_node_set_tree_can_fail (VALA_CODE_NODE (stmt), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_foreach_statement_get_collection (stmt))) || vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_foreach_statement_get_body (stmt))));
	(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
	(element_data_type == NULL ? NULL : (element_data_type = (g_object_unref (element_data_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_RETURN_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_code_node_get_error (VALA_CODE_NODE (vala_return_statement_get_return_expression (stmt)))) {
		/* ignore inner error*/
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (self->priv->current_return_type == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return not allowed in this context");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) == NULL && !(VALA_IS_VOID_TYPE (self->priv->current_return_type))) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return without value in non-void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && VALA_IS_VOID_TYPE (self->priv->current_return_type)) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return with value in void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && !vala_data_type_compatible (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)), self->priv->current_return_type, TRUE)) {
		if (!vala_data_type_compatible (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)), self->priv->current_return_type, FALSE)) {
			char* _tmp2;
			char* _tmp1;
			char* _tmp0;
			_tmp2 = NULL;
			_tmp1 = NULL;
			_tmp0 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), (_tmp2 = g_strdup_printf ("Return: Cannot convert from `%s' to `%s'", (_tmp0 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))))), (_tmp1 = vala_code_node_to_string (VALA_CODE_NODE (self->priv->current_return_type))))));
			_tmp2 = (g_free (_tmp2), NULL);
			_tmp1 = (g_free (_tmp1), NULL);
			_tmp0 = (g_free (_tmp0), NULL);
			return;
		} else {
			if (vala_code_context_is_non_null_enabled ()) {
				vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return value may not be null");
			}
		}
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_data_type_get_transfers_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_data_type_get_transfers_ownership (self->priv->current_return_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return value transfers ownership but method return type hasn't been declared to transfer ownership");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && VALA_IS_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt))) && vala_data_type_get_takes_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_data_type_get_transfers_ownership (self->priv->current_return_type)) {
		vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Local variable with strong reference used as return value and method return type hasn't been declared to transfer ownership");
	}
}


static void vala_semantic_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_THROW_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_TRY_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (clause == NULL || VALA_IS_CATCH_CLAUSE (clause));
	if (vala_catch_clause_get_type_reference (clause) != NULL) {
		ValaVariableDeclarator* _tmp0;
		ValaClassType* _tmp1;
		vala_source_file_add_type_dependency (self->priv->current_source_file, vala_catch_clause_get_type_reference (clause), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		_tmp0 = NULL;
		vala_catch_clause_set_variable_declarator (clause, (_tmp0 = vala_variable_declarator_new (vala_catch_clause_get_variable_name (clause), NULL, NULL)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
		_tmp1 = NULL;
		vala_variable_declarator_set_type_reference (vala_catch_clause_get_variable_declarator (clause), VALA_DATA_TYPE ((_tmp1 = vala_class_type_new (self->priv->gerror_type))));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (vala_catch_clause_get_body (clause))), vala_catch_clause_get_variable_name (clause), VALA_SYMBOL (vala_catch_clause_get_variable_declarator (clause)));
	} else {
		ValaErrorType* _tmp2;
		_tmp2 = NULL;
		vala_catch_clause_set_type_reference (clause, VALA_DATA_TYPE ((_tmp2 = vala_error_type_new (NULL, vala_code_node_get_source_reference (VALA_CODE_NODE (clause))))));
		(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (clause), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_LOCK_STATEMENT (stmt));
	/* resource must be a member access and denote a Lockable */
	if (!(VALA_IS_MEMBER_ACCESS (vala_lock_statement_get_resource (stmt)) && VALA_IS_LOCKABLE (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))))) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Expression is either not a member access or does not denote a lockable member");
		return;
	}
	/* parent symbol must be the current class */
	if (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))) != VALA_SYMBOL (self->priv->current_class)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Only members of the current class are lockable");
	}
	vala_lockable_set_lock_used ((VALA_LOCKABLE (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))), TRUE);
}


static void vala_semantic_analyzer_real_visit_delete_statement (ValaCodeVisitor* base, ValaDeleteStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_DELETE_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_delete_statement_get_expression (stmt)))) {
		return;
	}
	/* if there was an error in the inner expression, skip this check*/
	if (!(VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_delete_statement_get_expression (stmt))))) {
		char* _tmp1;
		char* _tmp0;
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), (_tmp1 = g_strdup_printf ("delete operator not supported for `%s'", (_tmp0 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_delete_statement_get_expression (stmt))))))));
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp0 = (g_free (_tmp0), NULL);
	}
}


static gint vala_semantic_analyzer_create_sizes_from_initializer_list (ValaSemanticAnalyzer* self, ValaInitializerList* il, gint rank, GeeList* sl) {
	char* _tmp0;
	ValaIntegerLiteral* _tmp1;
	ValaIntegerLiteral* init;
	gint subsize;
	gint _tmp7;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), 0);
	g_return_val_if_fail (il == NULL || VALA_IS_INITIALIZER_LIST (il), 0);
	g_return_val_if_fail (sl == NULL || GEE_IS_LIST (sl), 0);
	_tmp0 = NULL;
	_tmp1 = NULL;
	init = (_tmp1 = vala_integer_literal_new ((_tmp0 = g_strdup_printf ("%i", vala_initializer_list_get_size (il))), vala_code_node_get_source_reference (VALA_CODE_NODE (il))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	vala_code_node_accept (VALA_CODE_NODE (init), VALA_CODE_VISITOR (self));
	gee_collection_add (GEE_COLLECTION (sl), init);
	subsize = -1;
	{
		GeeCollection* e_collection;
		GeeIterator* e_it;
		e_collection = vala_initializer_list_get_initializers (il);
		e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
		while (gee_iterator_next (e_it)) {
			ValaExpression* e;
			e = ((ValaExpression*) gee_iterator_get (e_it));
			{
				if (VALA_IS_INITIALIZER_LIST (e)) {
					gint size;
					if (rank == 1) {
						gint _tmp2;
						vala_code_node_set_error (VALA_CODE_NODE (il), TRUE);
						vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expected array element, got array initializer list");
						return (_tmp2 = -1, (e == NULL ? NULL : (e = (g_object_unref (e), NULL))), (e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL))), (e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL))), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp2);
					}
					size = vala_semantic_analyzer_create_sizes_from_initializer_list (self, VALA_INITIALIZER_LIST (e), rank - 1, sl);
					if (size == -1) {
						gint _tmp3;
						return (_tmp3 = -1, (e == NULL ? NULL : (e = (g_object_unref (e), NULL))), (e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL))), (e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL))), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp3);
					}
					if (subsize >= 0 && subsize != size) {
						char* _tmp4;
						gint _tmp5;
						vala_code_node_set_error (VALA_CODE_NODE (il), TRUE);
						_tmp4 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (il)), (_tmp4 = g_strdup_printf ("Expected initializer list of size %d, got size %d", subsize, size)));
						_tmp4 = (g_free (_tmp4), NULL);
						return (_tmp5 = -1, (e == NULL ? NULL : (e = (g_object_unref (e), NULL))), (e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL))), (e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL))), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp5);
					} else {
						subsize = size;
					}
				} else {
					if (rank != 1) {
						gint _tmp6;
						vala_code_node_set_error (VALA_CODE_NODE (il), TRUE);
						vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expected array initializer list, got array element");
						return (_tmp6 = -1, (e == NULL ? NULL : (e = (g_object_unref (e), NULL))), (e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL))), (e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL))), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp6);
					}
				}
				(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
			}
		}
		(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
		(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
	}
	return (_tmp7 = vala_initializer_list_get_size (il), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp7);
}


/**
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_semantic_analyzer_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr) {
	ValaSemanticAnalyzer * self;
	GeeCollection* size;
	ValaInitializerList* _tmp0;
	ValaInitializerList* initlist;
	GeeArrayList* calc_sizes;
	ValaArrayType* _tmp2;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	size = GEE_COLLECTION (vala_array_creation_expression_get_sizes (expr));
	_tmp0 = NULL;
	initlist = (_tmp0 = vala_array_creation_expression_get_initializer_list (expr), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	if (vala_array_creation_expression_get_element_type (expr) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_array_creation_expression_get_element_type (expr)), VALA_CODE_VISITOR (self));
	}
	{
		GeeCollection* e_collection;
		GeeIterator* e_it;
		e_collection = size;
		e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
		while (gee_iterator_next (e_it)) {
			ValaExpression* e;
			e = ((ValaExpression*) gee_iterator_get (e_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (e), VALA_CODE_VISITOR (self));
				(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
			}
		}
		(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
	}
	calc_sizes = gee_array_list_new (VALA_TYPE_LITERAL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal);
	if (initlist != NULL) {
		ValaArrayType* _tmp1;
		gint ret;
		_tmp1 = NULL;
		vala_expression_set_expected_type (VALA_EXPRESSION (initlist), VALA_DATA_TYPE ((_tmp1 = vala_array_type_new (vala_array_creation_expression_get_element_type (expr), vala_array_creation_expression_get_rank (expr), vala_code_node_get_source_reference (VALA_CODE_NODE (expr))))));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		vala_data_type_add_type_argument (vala_expression_get_expected_type (VALA_EXPRESSION (initlist)), vala_array_creation_expression_get_element_type (expr));
		vala_code_node_accept (VALA_CODE_NODE (initlist), VALA_CODE_VISITOR (self));
		ret = vala_semantic_analyzer_create_sizes_from_initializer_list (self, initlist, vala_array_creation_expression_get_rank (expr), GEE_LIST (calc_sizes));
		if (ret == -1) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		}
	}
	if (gee_collection_get_size (size) > 0) {
		{
			GeeCollection* e_collection;
			GeeIterator* e_it;
			e_collection = size;
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = ((ValaExpression*) gee_iterator_get (e_it));
				{
					/* check for errors in the size list */
					if (vala_expression_get_static_type (e) == NULL) {
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
						(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
						(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
						(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
						(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
						return;
					} else {
						/* return on previous error */
						if (!(VALA_IS_STRUCT (vala_data_type_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_data_type_get_data_type (vala_expression_get_static_type (e)))))) {
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
						}
					}
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
	} else {
		if (initlist == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			/* this is an internal error because it is already handeld by the parser */
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: initializer list expected");
		} else {
			{
				GeeArrayList* size_collection;
				GeeIterator* size_it;
				size_collection = calc_sizes;
				size_it = gee_iterable_iterator (GEE_ITERABLE (size_collection));
				while (gee_iterator_next (size_it)) {
					ValaExpression* size;
					size = VALA_EXPRESSION (((ValaExpression*) gee_iterator_get (size_it)));
					{
						vala_array_creation_expression_append_size (expr, size);
						(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
					}
				}
				(size_it == NULL ? NULL : (size_it = (g_object_unref (size_it), NULL)));
			}
		}
	}
	if (vala_code_node_get_error (VALA_CODE_NODE (expr))) {
		(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
		(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
		(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
		return;
	}
	/* check for wrong elements inside the initializer */
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL && vala_expression_get_static_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))) == NULL) {
		(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
		(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
		(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
		return;
	}
	/* try to construct the type of the array */
	if (vala_array_creation_expression_get_element_type (expr) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Cannot determine the element type of the created array");
		(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
		(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
		(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
		return;
	}
	/* arrays of struct type elements do not take ownership since they are copied into the array */
	if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_array_creation_expression_get_element_type (expr)))) {
		vala_data_type_set_takes_ownership (vala_array_creation_expression_get_element_type (expr), FALSE);
	} else {
		vala_data_type_set_takes_ownership (vala_array_creation_expression_get_element_type (expr), TRUE);
	}
	_tmp2 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp2 = vala_array_type_new (vala_array_creation_expression_get_element_type (expr), vala_array_creation_expression_get_rank (expr), vala_code_node_get_source_reference (VALA_CODE_NODE (expr))))));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	vala_data_type_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
	vala_data_type_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
	vala_data_type_add_type_argument (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_array_creation_expression_get_element_type (expr));
	(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
	(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
	(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
}


static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_BOOLEAN_LITERAL (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
}


static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaValueType* _tmp1;
	ValaTypesymbol* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_CHARACTER_LITERAL (expr));
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp1 = vala_value_type_new ((_tmp0 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "char")))))));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
}


static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypesymbol* _tmp1;
	char* _tmp0;
	ValaIntegerType* _tmp2;
	ValaIntegerType* int_type;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_INTEGER_LITERAL (expr));
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	int_type = (_tmp2 = vala_integer_type_new ((_tmp1 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), (_tmp0 = vala_integer_literal_get_type_name (expr)))))), (_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
	vala_integer_type_set_literal (int_type, expr);
	vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE (int_type));
	(int_type == NULL ? NULL : (int_type = (g_object_unref (int_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaValueType* _tmp2;
	ValaTypesymbol* _tmp1;
	char* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_REAL_LITERAL (expr));
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp2 = vala_value_type_new ((_tmp1 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), (_tmp0 = vala_real_literal_get_type_name (expr)))))))));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaDataType* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_STRING_LITERAL (expr));
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp0 = vala_data_type_copy (self->priv->string_type)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
}


static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaNullType* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_NULL_LITERAL (expr));
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp0 = vala_null_type_new (vala_code_node_get_source_reference (VALA_CODE_NODE (expr))))));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
}


static ValaDataType* vala_semantic_analyzer_get_static_type_for_symbol (ValaSemanticAnalyzer* self, ValaSymbol* sym) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (sym == NULL || VALA_IS_SYMBOL (sym), NULL);
	if (VALA_IS_FIELD (sym)) {
		ValaField* _tmp0;
		ValaField* f;
		ValaDataType* _tmp1;
		ValaDataType* _tmp2;
		_tmp0 = NULL;
		f = (_tmp0 = VALA_FIELD (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = vala_field_get_type_reference (f), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (f == NULL ? NULL : (f = (g_object_unref (f), NULL))), _tmp2);
	} else {
		if (VALA_IS_CONSTANT (sym)) {
			ValaConstant* _tmp3;
			ValaConstant* c;
			ValaDataType* _tmp4;
			ValaDataType* _tmp5;
			_tmp3 = NULL;
			c = (_tmp3 = VALA_CONSTANT (sym), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
			_tmp4 = NULL;
			_tmp5 = NULL;
			return (_tmp5 = (_tmp4 = vala_constant_get_type_reference (c), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4))), (c == NULL ? NULL : (c = (g_object_unref (c), NULL))), _tmp5);
		} else {
			if (VALA_IS_PROPERTY (sym)) {
				ValaProperty* _tmp6;
				ValaProperty* prop;
				ValaDataType* type;
				ValaDataType* _tmp7;
				_tmp6 = NULL;
				prop = (_tmp6 = VALA_PROPERTY (sym), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
				type = vala_data_type_copy (vala_property_get_type_reference (prop));
				vala_data_type_set_takes_ownership (type, FALSE);
				_tmp7 = NULL;
				return (_tmp7 = type, (prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL))), _tmp7);
			} else {
				if (VALA_IS_FORMAL_PARAMETER (sym)) {
					ValaFormalParameter* _tmp8;
					ValaFormalParameter* p;
					ValaDataType* type;
					ValaDataType* _tmp9;
					_tmp8 = NULL;
					p = (_tmp8 = VALA_FORMAL_PARAMETER (sym), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8)));
					type = vala_data_type_copy (vala_formal_parameter_get_type_reference (p));
					vala_data_type_set_transfers_ownership (type, FALSE);
					_tmp9 = NULL;
					return (_tmp9 = type, (p == NULL ? NULL : (p = (g_object_unref (p), NULL))), _tmp9);
				} else {
					if (VALA_IS_DATA_TYPE (sym)) {
						ValaDataType* _tmp10;
						_tmp10 = NULL;
						return (_tmp10 = VALA_DATA_TYPE (sym), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
					} else {
						if (VALA_IS_VARIABLE_DECLARATOR (sym)) {
							ValaVariableDeclarator* _tmp12;
							ValaVariableDeclarator* decl;
							ValaDataType* _tmp13;
							ValaDataType* _tmp14;
							_tmp12 = NULL;
							decl = (_tmp12 = VALA_VARIABLE_DECLARATOR (sym), (_tmp12 == NULL ? NULL : g_object_ref (_tmp12)));
							_tmp13 = NULL;
							_tmp14 = NULL;
							return (_tmp14 = (_tmp13 = vala_variable_declarator_get_type_reference (decl), (_tmp13 == NULL ? NULL : g_object_ref (_tmp13))), (decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL))), _tmp14);
						} else {
							if (VALA_IS_ENUM_VALUE (sym)) {
								return VALA_DATA_TYPE (vala_value_type_new (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol (sym))));
							} else {
								if (VALA_IS_METHOD (sym)) {
									return VALA_DATA_TYPE (vala_method_type_new (VALA_METHOD (sym)));
								} else {
									if (VALA_IS_SIGNAL (sym)) {
										return VALA_DATA_TYPE (vala_signal_type_new (VALA_SIGNAL (sym)));
									}
								}
							}
						}
					}
				}
			}
		}
	}
	return NULL;
}


ValaSymbol* vala_semantic_analyzer_symbol_lookup_inherited (ValaSymbol* sym, const char* name) {
	ValaSymbol* result;
	gpointer _tmp13;
	g_return_val_if_fail (sym == NULL || VALA_IS_SYMBOL (sym), NULL);
	result = vala_scope_lookup (vala_symbol_get_scope (sym), name);
	if (result != NULL) {
		return result;
	}
	if (VALA_IS_CLASS (sym)) {
		ValaClass* _tmp1;
		ValaClass* cl;
		_tmp1 = NULL;
		cl = (_tmp1 = VALA_CLASS (sym), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = vala_class_get_base_types (cl);
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
				{
					/* first check interfaces without prerequisites
					 (prerequisites can be assumed to be met already)*/
					if (VALA_IS_INTERFACE (vala_data_type_get_data_type (base_type))) {
						ValaSymbol* _tmp2;
						_tmp2 = NULL;
						result = (_tmp2 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_data_type_get_data_type (base_type))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp2);
						if (result != NULL) {
							ValaSymbol* _tmp3;
							_tmp3 = NULL;
							return (_tmp3 = result, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp3);
						}
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		/* then check base class recursively*/
		if (vala_class_get_base_class (cl) != NULL) {
			ValaSymbol* _tmp4;
			_tmp4 = NULL;
			return (_tmp4 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_class_get_base_class (cl)), name), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp4);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (sym)) {
			ValaStruct* _tmp5;
			ValaStruct* st;
			_tmp5 = NULL;
			st = (_tmp5 = VALA_STRUCT (sym), (_tmp5 == NULL ? NULL : g_object_ref (_tmp5)));
			{
				GeeCollection* base_type_collection;
				GeeIterator* base_type_it;
				base_type_collection = vala_struct_get_base_types (st);
				base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
				while (gee_iterator_next (base_type_it)) {
					ValaDataType* base_type;
					base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
					{
						ValaSymbol* _tmp6;
						_tmp6 = NULL;
						result = (_tmp6 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_data_type_get_data_type (base_type)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp6);
						if (result != NULL) {
							ValaSymbol* _tmp7;
							_tmp7 = NULL;
							return (_tmp7 = result, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp7);
						}
						(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
					}
				}
				(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
				(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		} else {
			if (VALA_IS_INTERFACE (sym)) {
				ValaInterface* _tmp8;
				ValaInterface* iface;
				_tmp8 = NULL;
				iface = (_tmp8 = VALA_INTERFACE (sym), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8)));
				{
					GeeCollection* prerequisite_collection;
					GeeIterator* prerequisite_it;
					prerequisite_collection = vala_interface_get_prerequisites (iface);
					prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
					while (gee_iterator_next (prerequisite_it)) {
						ValaDataType* prerequisite;
						prerequisite = ((ValaDataType*) gee_iterator_get (prerequisite_it));
						{
							/* first check interface prerequisites recursively*/
							if (VALA_IS_INTERFACE (vala_data_type_get_data_type (prerequisite))) {
								ValaSymbol* _tmp9;
								_tmp9 = NULL;
								result = (_tmp9 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_data_type_get_data_type (prerequisite)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp9);
								if (result != NULL) {
									ValaSymbol* _tmp10;
									_tmp10 = NULL;
									return (_tmp10 = result, (prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL))), (prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL))), (prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL))), (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), _tmp10);
								}
							}
							(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
						}
					}
					(prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL)));
					(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
				}
				{
					GeeCollection* prerequisite_collection;
					GeeIterator* prerequisite_it;
					prerequisite_collection = vala_interface_get_prerequisites (iface);
					prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
					while (gee_iterator_next (prerequisite_it)) {
						ValaDataType* prerequisite;
						prerequisite = ((ValaDataType*) gee_iterator_get (prerequisite_it));
						{
							/* then check class prerequisite recursively*/
							if (VALA_IS_CLASS (vala_data_type_get_data_type (prerequisite))) {
								ValaSymbol* _tmp11;
								_tmp11 = NULL;
								result = (_tmp11 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_data_type_get_data_type (prerequisite)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp11);
								if (result != NULL) {
									ValaSymbol* _tmp12;
									_tmp12 = NULL;
									return (_tmp12 = result, (prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL))), (prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL))), (prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL))), (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), _tmp12);
								}
							}
							(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
						}
					}
					(prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL)));
					(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
				}
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	return (_tmp13 = NULL, (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp13);
}


static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaDataType* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_PARENTHESIZED_EXPRESSION (expr));
	vala_expression_set_expected_type (vala_parenthesized_expression_get_inner (expr), vala_expression_get_expected_type (VALA_EXPRESSION (expr)));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_parenthesized_expression_get_inner (expr)))) {
		/* ignore inner error*/
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_expression_get_static_type (vala_parenthesized_expression_get_inner (expr)) == NULL) {
		/* static type may be null for method references*/
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_parenthesized_expression_get_inner (expr))), "Invalid expression type");
		return;
	}
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp0 = vala_data_type_copy (vala_expression_get_static_type (vala_parenthesized_expression_get_inner (expr)))));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	/* don't call g_object_ref_sink on inner and outer expression*/
	vala_data_type_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE);
}


static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* base_symbol;
	gboolean may_access_instance_members;
	ValaSymbol* _tmp26;
	ValaSymbol* member;
	ValaSymbolAccessibility access;
	gboolean instance;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_MEMBER_ACCESS (expr));
	base_symbol = NULL;
	may_access_instance_members = FALSE;
	vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), NULL);
	if (vala_member_access_get_inner (expr) == NULL) {
		ValaSymbol* _tmp1;
		ValaSymbol* _tmp0;
		ValaSymbol* _tmp2;
		ValaSymbol* sym;
		_tmp1 = NULL;
		_tmp0 = NULL;
		base_symbol = (_tmp1 = (_tmp0 = self->priv->current_symbol, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), _tmp1);
		_tmp2 = NULL;
		sym = (_tmp2 = self->priv->current_symbol, (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
		while (sym != NULL && vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			ValaSymbol* _tmp4;
			ValaSymbol* _tmp6;
			ValaSymbol* _tmp5;
			if (VALA_IS_CREATION_METHOD (sym) || VALA_IS_PROPERTY (sym) || VALA_IS_CONSTRUCTOR (sym) || VALA_IS_DESTRUCTOR (sym)) {
				may_access_instance_members = TRUE;
			} else {
				if (VALA_IS_METHOD (sym)) {
					ValaMethod* _tmp3;
					ValaMethod* m;
					_tmp3 = NULL;
					m = (_tmp3 = VALA_METHOD (sym), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
					may_access_instance_members = vala_method_get_instance (m);
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				}
			}
			_tmp4 = NULL;
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp4 = vala_semantic_analyzer_symbol_lookup_inherited (sym, vala_member_access_get_member_name (expr))));
			(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
			_tmp6 = NULL;
			_tmp5 = NULL;
			sym = (_tmp6 = (_tmp5 = vala_symbol_get_parent_symbol (sym), (_tmp5 == NULL ? NULL : g_object_ref (_tmp5))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp6);
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			{
				GeeCollection* ns_collection;
				GeeIterator* ns_it;
				ns_collection = self->priv->current_using_directives;
				ns_it = gee_iterable_iterator (GEE_ITERABLE (ns_collection));
				while (gee_iterator_next (ns_it)) {
					ValaNamespaceReference* ns;
					ns = ((ValaNamespaceReference*) gee_iterator_get (ns_it));
					{
						ValaSymbol* local_sym;
						local_sym = vala_scope_lookup (vala_symbol_get_scope (vala_namespace_reference_get_namespace_symbol (ns)), vala_member_access_get_member_name (expr));
						if (local_sym != NULL) {
							if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
								char* _tmp9;
								char* _tmp8;
								char* _tmp7;
								vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
								_tmp9 = NULL;
								_tmp8 = NULL;
								_tmp7 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp9 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", vala_member_access_get_member_name (expr), (_tmp7 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))), (_tmp8 = vala_symbol_get_full_name (local_sym)))));
								_tmp9 = (g_free (_tmp9), NULL);
								_tmp8 = (g_free (_tmp8), NULL);
								_tmp7 = (g_free (_tmp7), NULL);
								(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
								(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
								(ns_it == NULL ? NULL : (ns_it = (g_object_unref (ns_it), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
								return;
							}
							vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), local_sym);
						}
						(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
						(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
					}
				}
				(ns_it == NULL ? NULL : (ns_it = (g_object_unref (ns_it), NULL)));
			}
		}
		(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
	} else {
		if (vala_code_node_get_error (VALA_CODE_NODE (vala_member_access_get_inner (expr)))) {
			/* if there was an error in the inner expression, skip this check */
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			return;
		}
		if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (expr))) {
			ValaMemberAccess* _tmp10;
			ValaMemberAccess* ma;
			_tmp10 = NULL;
			ma = (_tmp10 = VALA_MEMBER_ACCESS (vala_member_access_get_inner (expr)), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
			if (vala_member_access_get_prototype_access (ma)) {
				char* _tmp12;
				char* _tmp11;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp12 = NULL;
				_tmp11 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp12 = g_strdup_printf ("Access to instance member `%s' denied", (_tmp11 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (vala_member_access_get_inner (expr)))))));
				_tmp12 = (g_free (_tmp12), NULL);
				_tmp11 = (g_free (_tmp11), NULL);
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
				return;
			}
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
		}
		if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (expr)) || VALA_IS_BASE_ACCESS (vala_member_access_get_inner (expr))) {
			ValaSymbol* _tmp14;
			ValaSymbol* _tmp13;
			_tmp14 = NULL;
			_tmp13 = NULL;
			base_symbol = (_tmp14 = (_tmp13 = vala_expression_get_symbol_reference (vala_member_access_get_inner (expr)), (_tmp13 == NULL ? NULL : g_object_ref (_tmp13))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), _tmp14);
			if (vala_member_access_get_creation_member (expr) && VALA_IS_TYPESYMBOL (base_symbol)) {
				ValaSymbol* _tmp16;
				char* _tmp15;
				/* check for named creation method*/
				_tmp16 = NULL;
				_tmp15 = NULL;
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp16 = vala_scope_lookup (vala_symbol_get_scope (base_symbol), (_tmp15 = g_strconcat (".new.", vala_member_access_get_member_name (expr), NULL)))));
				(_tmp16 == NULL ? NULL : (_tmp16 = (g_object_unref (_tmp16), NULL)));
				_tmp15 = (g_free (_tmp15), NULL);
			}
			if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && (VALA_IS_NAMESPACE (base_symbol) || VALA_IS_TYPESYMBOL (base_symbol))) {
				ValaSymbol* _tmp17;
				_tmp17 = NULL;
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp17 = vala_scope_lookup (vala_symbol_get_scope (base_symbol), vala_member_access_get_member_name (expr))));
				(_tmp17 == NULL ? NULL : (_tmp17 = (g_object_unref (_tmp17), NULL)));
				if (VALA_IS_BASE_ACCESS (vala_member_access_get_inner (expr))) {
					/* inner expression is base access
					 access to instance members of the base type possible*/
					may_access_instance_members = TRUE;
				}
			}
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_expression_get_static_type (vala_member_access_get_inner (expr)) != NULL) {
			if (vala_member_access_get_pointer_member_access (expr)) {
				ValaSymbol* _tmp18;
				_tmp18 = NULL;
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp18 = vala_data_type_get_pointer_member (vala_expression_get_static_type (vala_member_access_get_inner (expr)), vala_member_access_get_member_name (expr))));
				(_tmp18 == NULL ? NULL : (_tmp18 = (g_object_unref (_tmp18), NULL)));
			} else {
				ValaSymbol* _tmp21;
				if (vala_data_type_get_data_type (vala_expression_get_static_type (vala_member_access_get_inner (expr))) != NULL) {
					ValaSymbol* _tmp20;
					ValaTypesymbol* _tmp19;
					_tmp20 = NULL;
					_tmp19 = NULL;
					base_symbol = (_tmp20 = VALA_SYMBOL ((_tmp19 = vala_data_type_get_data_type (vala_expression_get_static_type (vala_member_access_get_inner (expr))), (_tmp19 == NULL ? NULL : g_object_ref (_tmp19)))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), _tmp20);
				}
				_tmp21 = NULL;
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp21 = vala_data_type_get_member (vala_expression_get_static_type (vala_member_access_get_inner (expr)), vala_member_access_get_member_name (expr))));
				(_tmp21 == NULL ? NULL : (_tmp21 = (g_object_unref (_tmp21), NULL)));
			}
			if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
				/* inner expression is variable, field, or parameter
				 access to instance members of the corresponding type possible*/
				may_access_instance_members = TRUE;
			}
		}
	}
	if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
		{
			GeeCollection* binding_provider_collection;
			GeeIterator* binding_provider_it;
			binding_provider_collection = self->priv->binding_providers;
			binding_provider_it = gee_iterable_iterator (GEE_ITERABLE (binding_provider_collection));
			while (gee_iterator_next (binding_provider_it)) {
				ValaBindingProvider* binding_provider;
				binding_provider = ((ValaBindingProvider*) gee_iterator_get (binding_provider_it));
				{
					ValaSymbol* _tmp22;
					/* allow plug-ins to provide custom member bindings */
					_tmp22 = NULL;
					vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp22 = vala_binding_provider_get_binding (binding_provider, expr)));
					(_tmp22 == NULL ? NULL : (_tmp22 = (g_object_unref (_tmp22), NULL)));
					if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
						may_access_instance_members = TRUE;
						(binding_provider == NULL ? NULL : (binding_provider = (g_object_unref (binding_provider), NULL)));
						break;
					}
					(binding_provider == NULL ? NULL : (binding_provider = (g_object_unref (binding_provider), NULL)));
				}
			}
			(binding_provider_it == NULL ? NULL : (binding_provider_it = (g_object_unref (binding_provider_it), NULL)));
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			char* base_type_name;
			char* _tmp25;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			base_type_name = g_strdup ("(null)");
			if (vala_member_access_get_inner (expr) != NULL && vala_expression_get_static_type (vala_member_access_get_inner (expr)) != NULL) {
				char* _tmp23;
				_tmp23 = NULL;
				base_type_name = (_tmp23 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_member_access_get_inner (expr)))), (base_type_name = (g_free (base_type_name), NULL)), _tmp23);
			} else {
				if (base_symbol != NULL) {
					char* _tmp24;
					_tmp24 = NULL;
					base_type_name = (_tmp24 = vala_symbol_get_full_name (base_symbol), (base_type_name = (g_free (base_type_name), NULL)), _tmp24);
				}
			}
			_tmp25 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp25 = g_strdup_printf ("The name `%s' does not exist in the context of `%s'", vala_member_access_get_member_name (expr), base_type_name)));
			_tmp25 = (g_free (_tmp25), NULL);
			base_type_name = (g_free (base_type_name), NULL);
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			return;
		}
	}
	_tmp26 = NULL;
	member = (_tmp26 = vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)), (_tmp26 == NULL ? NULL : g_object_ref (_tmp26)));
	access = VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
	instance = FALSE;
	if (VALA_IS_FIELD (member)) {
		ValaField* _tmp27;
		ValaField* f;
		_tmp27 = NULL;
		f = (_tmp27 = VALA_FIELD (member), (_tmp27 == NULL ? NULL : g_object_ref (_tmp27)));
		access = vala_symbol_get_access (VALA_SYMBOL (f));
		instance = vala_field_get_instance (f);
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	} else {
		if (VALA_IS_METHOD (member)) {
			ValaMethod* _tmp28;
			ValaMethod* m;
			_tmp28 = NULL;
			m = (_tmp28 = VALA_METHOD (member), (_tmp28 == NULL ? NULL : g_object_ref (_tmp28)));
			access = vala_symbol_get_access (VALA_SYMBOL (m));
			instance = vala_method_get_instance (m);
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			if (VALA_IS_PROPERTY (member)) {
				ValaProperty* _tmp29;
				ValaProperty* prop;
				_tmp29 = NULL;
				prop = (_tmp29 = VALA_PROPERTY (member), (_tmp29 == NULL ? NULL : g_object_ref (_tmp29)));
				access = vala_symbol_get_access (VALA_SYMBOL (prop));
				if (vala_expression_get_lvalue (VALA_EXPRESSION (expr))) {
					if (vala_property_get_set_accessor (prop) == NULL) {
						char* _tmp31;
						char* _tmp30;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp31 = NULL;
						_tmp30 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp31 = g_strdup_printf ("Property `%s' is read-only", (_tmp30 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
						_tmp31 = (g_free (_tmp31), NULL);
						_tmp30 = (g_free (_tmp30), NULL);
						(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
						(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
						(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
						return;
					}
					if (vala_symbol_get_access (VALA_SYMBOL (prop)) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
						access = vala_property_accessor_get_access (vala_property_get_set_accessor (prop));
					} else {
						if (vala_symbol_get_access (VALA_SYMBOL (prop)) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED && vala_property_accessor_get_access (vala_property_get_set_accessor (prop)) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
							access = vala_property_accessor_get_access (vala_property_get_set_accessor (prop));
						}
					}
				} else {
					if (vala_property_get_get_accessor (prop) == NULL) {
						char* _tmp33;
						char* _tmp32;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp33 = NULL;
						_tmp32 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp33 = g_strdup_printf ("Property `%s' is write-only", (_tmp32 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
						_tmp33 = (g_free (_tmp33), NULL);
						_tmp32 = (g_free (_tmp32), NULL);
						(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
						(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
						(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
						return;
					}
					if (vala_symbol_get_access (VALA_SYMBOL (prop)) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
						access = vala_property_accessor_get_access (vala_property_get_get_accessor (prop));
					} else {
						if (vala_symbol_get_access (VALA_SYMBOL (prop)) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED && vala_property_accessor_get_access (vala_property_get_get_accessor (prop)) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
							access = vala_property_accessor_get_access (vala_property_get_get_accessor (prop));
						}
					}
				}
				instance = vala_property_get_instance (prop);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			} else {
				if (VALA_IS_SIGNAL (member)) {
					instance = TRUE;
				}
			}
		}
	}
	if (access == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		ValaSymbol* _tmp34;
		ValaSymbol* target_type;
		gboolean in_target_type;
		_tmp34 = NULL;
		target_type = (_tmp34 = vala_symbol_get_parent_symbol (member), (_tmp34 == NULL ? NULL : g_object_ref (_tmp34)));
		in_target_type = FALSE;
		{
			ValaSymbol* this_symbol;
			ValaSymbol* _tmp38;
			ValaSymbol* _tmp37;
			ValaSymbol* _tmp36;
			ValaSymbol* _tmp35;
			this_symbol = NULL;
			_tmp38 = NULL;
			_tmp37 = NULL;
			_tmp36 = NULL;
			_tmp35 = NULL;
			for (this_symbol = (_tmp36 = (_tmp35 = self->priv->current_symbol, (_tmp35 == NULL ? NULL : g_object_ref (_tmp35))), (this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL))), _tmp36); this_symbol != NULL; this_symbol = (_tmp38 = (_tmp37 = vala_symbol_get_parent_symbol (this_symbol), (_tmp37 == NULL ? NULL : g_object_ref (_tmp37))), (this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL))), _tmp38)) {
				if (target_type == this_symbol) {
					in_target_type = TRUE;
					break;
				}
			}
			(this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL)));
		}
		if (!in_target_type) {
			char* _tmp40;
			char* _tmp39;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp40 = NULL;
			_tmp39 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp40 = g_strdup_printf ("Access to private member `%s' denied", (_tmp39 = vala_symbol_get_full_name (member)))));
			_tmp40 = (g_free (_tmp40), NULL);
			_tmp39 = (g_free (_tmp39), NULL);
			(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
			return;
		}
		(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
	}
	if (instance && !may_access_instance_members) {
		ValaDataType* _tmp41;
		vala_member_access_set_prototype_access (expr, TRUE);
		/* also set static type for prototype access
		 required when using instance methods as delegates in constants*/
		_tmp41 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp41 = vala_semantic_analyzer_get_static_type_for_symbol (self, vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))));
		(_tmp41 == NULL ? NULL : (_tmp41 = (g_object_unref (_tmp41), NULL)));
	} else {
		ValaDataType* _tmp42;
		_tmp42 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp42 = vala_semantic_analyzer_get_static_type_for_symbol (self, vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))));
		(_tmp42 == NULL ? NULL : (_tmp42 = (g_object_unref (_tmp42), NULL)));
		/* resolve generic return values*/
		if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) != NULL && vala_data_type_get_type_parameter (vala_expression_get_static_type (VALA_EXPRESSION (expr))) != NULL) {
			if (vala_member_access_get_inner (expr) != NULL) {
				ValaDataType* _tmp43;
				_tmp43 = NULL;
				vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp43 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_member_access_get_inner (expr)), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)), vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_CODE_NODE (expr))));
				(_tmp43 == NULL ? NULL : (_tmp43 = (g_object_unref (_tmp43), NULL)));
				if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
					(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
					(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
					return;
				}
			}
		}
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
	(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
}


static void vala_semantic_analyzer_real_visit_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaDataType* _tmp3;
	ValaDataType* mtype;
	GeeCollection* params;
	GeeCollection* args;
	GeeIterator* arg_it;
	ValaDataType* ret_type;
	ValaDataType* _tmp5;
	GeeCollection* _tmp6;
	ValaMemberAccess* _tmp8;
	ValaExpression* _tmp7;
	ValaMemberAccess* ma;
	GeeList* resolved_type_args;
	ValaDataType* _tmp11;
	GeeCollection* _tmp14;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_INVOCATION_EXPRESSION (expr));
	vala_code_node_accept (VALA_CODE_NODE (vala_invocation_expression_get_call (expr)), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_invocation_expression_get_call (expr)))) {
		/* if method resolving didn't succeed, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr))) {
		ValaMemberAccess* _tmp0;
		ValaMemberAccess* ma;
		_tmp0 = NULL;
		ma = (_tmp0 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp2;
			char* _tmp1;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp2 = NULL;
			_tmp1 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp2 = g_strdup_printf ("Access to instance member `%s' denied", (_tmp1 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)))))));
			_tmp2 = (g_free (_tmp2), NULL);
			_tmp1 = (g_free (_tmp1), NULL);
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	_tmp3 = NULL;
	mtype = (_tmp3 = vala_expression_get_static_type (vala_invocation_expression_get_call (expr)), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
	params = NULL;
	if (mtype != NULL && vala_data_type_is_invokable (mtype)) {
		GeeCollection* _tmp4;
		_tmp4 = NULL;
		params = (_tmp4 = vala_data_type_get_parameters (mtype), (params == NULL ? NULL : (params = (g_object_unref (params), NULL))), _tmp4);
	} else {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
		(mtype == NULL ? NULL : (mtype = (g_object_unref (mtype), NULL)));
		(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
		return;
	}
	args = vala_invocation_expression_get_argument_list (expr);
	arg_it = gee_iterable_iterator (GEE_ITERABLE (args));
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) gee_iterator_get (param_it));
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					break;
				}
				if (gee_iterator_next (arg_it)) {
					ValaExpression* arg;
					arg = ((ValaExpression*) gee_iterator_get (arg_it));
					/* store expected type for callback parameters */
					vala_expression_set_expected_type (arg, vala_formal_parameter_get_type_reference (param));
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	{
		GeeCollection* arg_collection;
		GeeIterator* arg_it;
		arg_collection = vala_invocation_expression_get_argument_list (expr);
		arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = ((ValaExpression*) gee_iterator_get (arg_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (arg), VALA_CODE_VISITOR (self));
				(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
			}
		}
		(arg_collection == NULL ? NULL : (arg_collection = (g_object_unref (arg_collection), NULL)));
		(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	}
	ret_type = NULL;
	_tmp5 = NULL;
	ret_type = (_tmp5 = vala_data_type_get_return_type (mtype), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), _tmp5);
	_tmp6 = NULL;
	params = (_tmp6 = vala_data_type_get_parameters (mtype), (params == NULL ? NULL : (params = (g_object_unref (params), NULL))), _tmp6);
	if (VALA_IS_VOID_TYPE (ret_type)) {
		/* void return type*/
		if (!(VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (VALA_CODE_NODE (expr))))) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation of void method not allowed as expression");
			(mtype == NULL ? NULL : (mtype = (g_object_unref (mtype), NULL)));
			(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
			(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
			(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
			(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
			return;
		}
	}
	/* resolve generic return values*/
	_tmp8 = NULL;
	_tmp7 = NULL;
	ma = (_tmp8 = (_tmp7 = vala_invocation_expression_get_call (expr), (VALA_IS_MEMBER_ACCESS (_tmp7) ? ((ValaMemberAccess*) _tmp7) : NULL)), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8)));
	if (vala_data_type_get_type_parameter (ret_type) != NULL) {
		if (ma != NULL && vala_member_access_get_inner (ma) != NULL) {
			ValaDataType* _tmp9;
			_tmp9 = NULL;
			ret_type = (_tmp9 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_member_access_get_inner (ma)), vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)), ret_type, VALA_CODE_NODE (expr)), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), _tmp9);
			if (ret_type == NULL) {
				(mtype == NULL ? NULL : (mtype = (g_object_unref (mtype), NULL)));
				(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
				(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
				(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
		}
	}
	resolved_type_args = GEE_LIST (gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	{
		GeeList* type_arg_collection;
		GeeIterator* type_arg_it;
		type_arg_collection = vala_data_type_get_type_arguments (ret_type);
		type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
		while (gee_iterator_next (type_arg_it)) {
			ValaDataType* type_arg;
			type_arg = ((ValaDataType*) gee_iterator_get (type_arg_it));
			{
				if (vala_data_type_get_type_parameter (type_arg) != NULL && ma != NULL && vala_member_access_get_inner (ma) != NULL) {
					ValaDataType* _tmp10;
					_tmp10 = NULL;
					gee_collection_add (GEE_COLLECTION (resolved_type_args), (_tmp10 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_member_access_get_inner (ma)), vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)), type_arg, VALA_CODE_NODE (expr))));
					(_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL)));
				} else {
					gee_collection_add (GEE_COLLECTION (resolved_type_args), type_arg);
				}
				(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
			}
		}
		(type_arg_collection == NULL ? NULL : (type_arg_collection = (g_object_unref (type_arg_collection), NULL)));
		(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
	}
	_tmp11 = NULL;
	ret_type = (_tmp11 = vala_data_type_copy (ret_type), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), _tmp11);
	vala_data_type_remove_all_type_arguments (ret_type);
	{
		GeeList* resolved_type_arg_collection;
		GeeIterator* resolved_type_arg_it;
		resolved_type_arg_collection = resolved_type_args;
		resolved_type_arg_it = gee_iterable_iterator (GEE_ITERABLE (resolved_type_arg_collection));
		while (gee_iterator_next (resolved_type_arg_it)) {
			ValaDataType* resolved_type_arg;
			resolved_type_arg = ((ValaDataType*) gee_iterator_get (resolved_type_arg_it));
			{
				vala_data_type_add_type_argument (ret_type, resolved_type_arg);
				(resolved_type_arg == NULL ? NULL : (resolved_type_arg = (g_object_unref (resolved_type_arg), NULL)));
			}
		}
		(resolved_type_arg_it == NULL ? NULL : (resolved_type_arg_it = (g_object_unref (resolved_type_arg_it), NULL)));
	}
	if (VALA_IS_METHOD_TYPE (mtype)) {
		ValaMethod* _tmp12;
		ValaMethod* m;
		GeeCollection* _tmp13;
		_tmp12 = NULL;
		m = (_tmp12 = vala_method_type_get_method_symbol ((VALA_METHOD_TYPE (mtype))), (_tmp12 == NULL ? NULL : g_object_ref (_tmp12)));
		_tmp13 = NULL;
		vala_code_node_set_tree_can_fail (VALA_CODE_NODE (expr), (vala_expression_set_can_fail (VALA_EXPRESSION (expr), (gee_collection_get_size ((_tmp13 = vala_method_get_error_domains (m))) > 0)), vala_expression_get_can_fail (VALA_EXPRESSION (expr))));
		(_tmp13 == NULL ? NULL : (_tmp13 = (g_object_unref (_tmp13), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	vala_expression_set_static_type (VALA_EXPRESSION (expr), ret_type);
	_tmp14 = NULL;
	vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), mtype, params, (_tmp14 = vala_invocation_expression_get_argument_list (expr)));
	(_tmp14 == NULL ? NULL : (_tmp14 = (g_object_unref (_tmp14), NULL)));
	(mtype == NULL ? NULL : (mtype = (g_object_unref (mtype), NULL)));
	(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
	(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
	(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
	(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	(resolved_type_args == NULL ? NULL : (resolved_type_args = (g_object_unref (resolved_type_args), NULL)));
}


static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaDataType* mtype, GeeCollection* params, GeeCollection* args) {
	ValaExpression* prev_arg;
	GeeIterator* arg_it;
	ValaAttribute* _tmp0;
	gboolean _tmp1;
	gboolean diag;
	gboolean ellipsis;
	gint i;
	gboolean _tmp34;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (expr == NULL || VALA_IS_EXPRESSION (expr), FALSE);
	g_return_val_if_fail (mtype == NULL || VALA_IS_DATA_TYPE (mtype), FALSE);
	g_return_val_if_fail (params == NULL || GEE_IS_COLLECTION (params), FALSE);
	g_return_val_if_fail (args == NULL || GEE_IS_COLLECTION (args), FALSE);
	prev_arg = NULL;
	arg_it = gee_iterable_iterator (GEE_ITERABLE (args));
	_tmp0 = NULL;
	diag = (_tmp1 = (VALA_IS_METHOD_TYPE (mtype) && (_tmp0 = vala_code_node_get_attribute (VALA_CODE_NODE (vala_method_type_get_method_symbol ((VALA_METHOD_TYPE (mtype)))), "Diagnostics")) != NULL), (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1);
	ellipsis = FALSE;
	i = 0;
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) gee_iterator_get (param_it));
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					ellipsis = TRUE;
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					break;
				}
				/* header file necessary if we need to cast argument */
				vala_source_file_add_type_dependency (self->priv->current_source_file, vala_formal_parameter_get_type_reference (param), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
				if (!gee_iterator_next (arg_it)) {
					if (vala_formal_parameter_get_default_expression (param) == NULL) {
						char* _tmp3;
						char* _tmp2;
						gboolean _tmp4;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp3 = NULL;
						_tmp2 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp3 = g_strdup_printf ("Too few arguments, method `%s' does not take %d arguments", (_tmp2 = vala_code_node_to_string (VALA_CODE_NODE (mtype))), gee_collection_get_size (args))));
						_tmp3 = (g_free (_tmp3), NULL);
						_tmp2 = (g_free (_tmp2), NULL);
						return (_tmp4 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp4);
					}
				} else {
					ValaExpression* arg;
					ValaExpression* _tmp23;
					ValaExpression* _tmp22;
					arg = ((ValaExpression*) gee_iterator_get (arg_it));
					if (vala_code_node_get_error (VALA_CODE_NODE (arg))) {
						gboolean _tmp5;
						/* ignore inner error*/
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						return (_tmp5 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp5);
					} else {
						if (vala_expression_get_static_type (arg) == NULL) {
							/* disallow untyped arguments except for type inference of callbacks*/
							if (!(VALA_IS_DELEGATE_TYPE (vala_formal_parameter_get_type_reference (param))) || !(VALA_IS_METHOD (vala_expression_get_symbol_reference (arg)))) {
								char* _tmp6;
								gboolean _tmp7;
								vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
								_tmp6 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp6 = g_strdup_printf ("Invalid type for argument %d", i + 1)));
								_tmp6 = (g_free (_tmp6), NULL);
								return (_tmp7 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp7);
							}
						} else {
							if (!vala_data_type_compatible (vala_expression_get_static_type (arg), vala_formal_parameter_get_type_reference (param), TRUE)) {
								if (!vala_data_type_compatible (vala_expression_get_static_type (arg), vala_formal_parameter_get_type_reference (param), FALSE)) {
									char* _tmp10;
									char* _tmp9;
									char* _tmp8;
									gboolean _tmp11;
									vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
									_tmp10 = NULL;
									_tmp9 = NULL;
									_tmp8 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp10 = g_strdup_printf ("Argument %d: Cannot convert from `%s' to `%s'", i + 1, (_tmp8 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (arg)))), (_tmp9 = vala_code_node_to_string (VALA_CODE_NODE (vala_formal_parameter_get_type_reference (param)))))));
									_tmp10 = (g_free (_tmp10), NULL);
									_tmp9 = (g_free (_tmp9), NULL);
									_tmp8 = (g_free (_tmp8), NULL);
									return (_tmp11 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp11);
								} else {
									if (vala_code_context_is_non_null_enabled ()) {
										char* _tmp12;
										_tmp12 = NULL;
										vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp12 = g_strdup_printf ("Argument %d: Argument may not be null", i + 1)));
										_tmp12 = (g_free (_tmp12), NULL);
									}
								}
							} else {
								gint arg_type;
								/* 0 => null, 1 => in, 2 => ref, 3 => out*/
								arg_type = 1;
								if (VALA_IS_NULL_TYPE (vala_expression_get_static_type (arg))) {
									arg_type = 0;
								} else {
									if (VALA_IS_UNARY_EXPRESSION (arg)) {
										ValaUnaryExpression* _tmp13;
										ValaUnaryExpression* unary;
										_tmp13 = NULL;
										unary = (_tmp13 = VALA_UNARY_EXPRESSION (arg), (_tmp13 == NULL ? NULL : g_object_ref (_tmp13)));
										if (vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_REF) {
											arg_type = 2;
										} else {
											if (vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_OUT) {
												arg_type = 3;
											}
										}
										(unary == NULL ? NULL : (unary = (g_object_unref (unary), NULL)));
									}
								}
								if (arg_type == 0) {
									if (vala_data_type_get_is_ref (vala_formal_parameter_get_type_reference (param))) {
										char* _tmp14;
										gboolean _tmp15;
										vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
										_tmp14 = NULL;
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp14 = g_strdup_printf ("Argument %d: Cannot pass null to reference parameter", i + 1)));
										_tmp14 = (g_free (_tmp14), NULL);
										return (_tmp15 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp15);
									}
								} else {
									if (arg_type == 1) {
										if (vala_data_type_get_is_ref (vala_formal_parameter_get_type_reference (param)) || vala_data_type_get_is_out (vala_formal_parameter_get_type_reference (param))) {
											char* _tmp16;
											gboolean _tmp17;
											vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
											_tmp16 = NULL;
											vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp16 = g_strdup_printf ("Argument %d: Cannot pass value to reference or output parameter", i + 1)));
											_tmp16 = (g_free (_tmp16), NULL);
											return (_tmp17 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp17);
										}
									} else {
										if (arg_type == 2) {
											if (!vala_data_type_get_is_ref (vala_formal_parameter_get_type_reference (param))) {
												char* _tmp18;
												gboolean _tmp19;
												vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
												_tmp18 = NULL;
												vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp18 = g_strdup_printf ("Argument %d: Cannot pass ref argument to non-reference parameter", i + 1)));
												_tmp18 = (g_free (_tmp18), NULL);
												return (_tmp19 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp19);
											}
										} else {
											if (arg_type == 3) {
												if (!vala_data_type_get_is_out (vala_formal_parameter_get_type_reference (param))) {
													char* _tmp20;
													gboolean _tmp21;
													vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
													_tmp20 = NULL;
													vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp20 = g_strdup_printf ("Argument %d: Cannot pass out argument to non-output parameter", i + 1)));
													_tmp20 = (g_free (_tmp20), NULL);
													return (_tmp21 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp21);
												}
											}
										}
									}
								}
							}
						}
					}
					_tmp23 = NULL;
					_tmp22 = NULL;
					prev_arg = (_tmp23 = (_tmp22 = arg, (_tmp22 == NULL ? NULL : g_object_ref (_tmp22))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), _tmp23);
					i++;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	if (ellipsis) {
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = ((ValaExpression*) gee_iterator_get (arg_it));
			if (vala_code_node_get_error (VALA_CODE_NODE (arg))) {
				gboolean _tmp24;
				/* ignore inner error*/
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				return (_tmp24 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp24);
			} else {
				if (vala_expression_get_static_type (arg) == NULL) {
					/* disallow untyped arguments except for type inference of callbacks*/
					if (!(VALA_IS_METHOD (vala_expression_get_symbol_reference (arg)))) {
						char* _tmp25;
						gboolean _tmp26;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp25 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp25 = g_strdup_printf ("Invalid type for argument %d", i + 1)));
						_tmp25 = (g_free (_tmp25), NULL);
						return (_tmp26 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp26);
					}
				}
			}
			i++;
			(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
		}
	} else {
		if (!ellipsis && gee_iterator_next (arg_it)) {
			char* _tmp28;
			char* _tmp27;
			gboolean _tmp29;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp28 = NULL;
			_tmp27 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp28 = g_strdup_printf ("Too many arguments, method `%s' does not take %d arguments", (_tmp27 = vala_code_node_to_string (VALA_CODE_NODE (mtype))), gee_collection_get_size (args))));
			_tmp28 = (g_free (_tmp28), NULL);
			_tmp27 = (g_free (_tmp27), NULL);
			return (_tmp29 = FALSE, (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp29);
		}
	}
	if (diag && prev_arg != NULL) {
		ValaStringLiteral* _tmp31;
		ValaExpression* _tmp30;
		ValaStringLiteral* format_arg;
		_tmp31 = NULL;
		_tmp30 = NULL;
		format_arg = (_tmp31 = (_tmp30 = prev_arg, (VALA_IS_STRING_LITERAL (_tmp30) ? ((ValaStringLiteral*) _tmp30) : NULL)), (_tmp31 == NULL ? NULL : g_object_ref (_tmp31)));
		if (format_arg != NULL) {
			char* _tmp33;
			char* _tmp32;
			_tmp33 = NULL;
			_tmp32 = NULL;
			vala_string_literal_set_value (format_arg, (_tmp33 = g_strdup_printf ("\"%s:%d: %s", (_tmp32 = g_path_get_basename (vala_source_file_get_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)))))), vala_source_reference_get_first_line (vala_code_node_get_source_reference (VALA_CODE_NODE (expr))), g_utf8_offset_to_pointer (vala_string_literal_get_value (format_arg), ((glong) 1)))));
			_tmp33 = (g_free (_tmp33), NULL);
			_tmp32 = (g_free (_tmp32), NULL);
		}
		(format_arg == NULL ? NULL : (format_arg = (g_object_unref (format_arg), NULL)));
	}
	return (_tmp34 = TRUE, (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp34);
}


static ValaDataType* vala_semantic_analyzer_get_instance_base_type (ValaDataType* instance_type, ValaDataType* base_type, ValaCodeNode* node_reference) {
	ValaReferenceType* instance_base_type;
	g_return_val_if_fail (instance_type == NULL || VALA_IS_DATA_TYPE (instance_type), NULL);
	g_return_val_if_fail (base_type == NULL || VALA_IS_DATA_TYPE (base_type), NULL);
	g_return_val_if_fail (node_reference == NULL || VALA_IS_CODE_NODE (node_reference), NULL);
	/* construct a new type reference for the base type with correctly linked type arguments*/
	instance_base_type = NULL;
	if (VALA_IS_CLASS (vala_data_type_get_data_type (base_type))) {
		ValaReferenceType* _tmp0;
		_tmp0 = NULL;
		instance_base_type = (_tmp0 = VALA_REFERENCE_TYPE (vala_class_type_new (VALA_CLASS (vala_data_type_get_data_type (base_type)))), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), _tmp0);
	} else {
		ValaReferenceType* _tmp1;
		_tmp1 = NULL;
		instance_base_type = (_tmp1 = VALA_REFERENCE_TYPE (vala_interface_type_new (VALA_INTERFACE (vala_data_type_get_data_type (base_type)))), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), _tmp1);
	}
	{
		GeeList* type_arg_collection;
		GeeIterator* type_arg_it;
		type_arg_collection = vala_data_type_get_type_arguments (base_type);
		type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
		while (gee_iterator_next (type_arg_it)) {
			ValaDataType* type_arg;
			type_arg = ((ValaDataType*) gee_iterator_get (type_arg_it));
			{
				if (vala_data_type_get_type_parameter (type_arg) != NULL) {
					gint param_index;
					ValaDataType* _tmp5;
					GeeList* _tmp4;
					/* link to type argument of derived type*/
					param_index = vala_typesymbol_get_type_parameter_index (vala_data_type_get_data_type (instance_type), vala_symbol_get_name (VALA_SYMBOL (vala_data_type_get_type_parameter (type_arg))));
					if (param_index == -1) {
						char* _tmp2;
						gpointer _tmp3;
						_tmp2 = NULL;
						vala_report_error (vala_code_node_get_source_reference (node_reference), (_tmp2 = g_strdup_printf ("internal error: unknown type parameter %s", vala_symbol_get_name (VALA_SYMBOL (vala_data_type_get_type_parameter (type_arg))))));
						_tmp2 = (g_free (_tmp2), NULL);
						vala_code_node_set_error (node_reference, TRUE);
						return (_tmp3 = NULL, (type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL))), (type_arg_collection == NULL ? NULL : (type_arg_collection = (g_object_unref (type_arg_collection), NULL))), (type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL))), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), _tmp3);
					}
					_tmp5 = NULL;
					_tmp4 = NULL;
					type_arg = (_tmp5 = ((ValaDataType*) gee_list_get ((_tmp4 = vala_data_type_get_type_arguments (instance_type)), param_index)), (type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL))), _tmp5);
					(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
				}
				vala_data_type_add_type_argument (VALA_DATA_TYPE (instance_base_type), type_arg);
				(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
			}
		}
		(type_arg_collection == NULL ? NULL : (type_arg_collection = (g_object_unref (type_arg_collection), NULL)));
		(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
	}
	return VALA_DATA_TYPE (instance_base_type);
}


ValaDataType* vala_semantic_analyzer_get_actual_type (ValaDataType* derived_instance_type, ValaSymbol* generic_member, ValaDataType* generic_type, ValaCodeNode* node_reference) {
	ValaDataType* _tmp0;
	ValaDataType* instance_type;
	gint param_index;
	ValaDataType* actual_type;
	GeeList* _tmp33;
	gboolean _tmp34;
	ValaDataType* _tmp39;
	ValaDataType* _tmp40;
	g_return_val_if_fail (derived_instance_type == NULL || VALA_IS_DATA_TYPE (derived_instance_type), NULL);
	g_return_val_if_fail (generic_member == NULL || VALA_IS_SYMBOL (generic_member), NULL);
	g_return_val_if_fail (generic_type == NULL || VALA_IS_DATA_TYPE (generic_type), NULL);
	g_return_val_if_fail (node_reference == NULL || VALA_IS_CODE_NODE (node_reference), NULL);
	_tmp0 = NULL;
	instance_type = (_tmp0 = derived_instance_type, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	while (VALA_IS_POINTER_TYPE (instance_type)) {
		ValaPointerType* _tmp1;
		ValaPointerType* instance_pointer_type;
		ValaDataType* _tmp3;
		ValaDataType* _tmp2;
		_tmp1 = NULL;
		instance_pointer_type = (_tmp1 = VALA_POINTER_TYPE (instance_type), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
		_tmp3 = NULL;
		_tmp2 = NULL;
		instance_type = (_tmp3 = (_tmp2 = vala_pointer_type_get_base_type (instance_pointer_type), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp3);
		(instance_pointer_type == NULL ? NULL : (instance_pointer_type = (g_object_unref (instance_pointer_type), NULL)));
	}
	while (VALA_SYMBOL (vala_data_type_get_data_type (instance_type)) != vala_symbol_get_parent_symbol (generic_member)) {
		ValaDataType* instance_base_type;
		ValaDataType* _tmp26;
		ValaDataType* _tmp25;
		/* trace type arguments back to the datatype where the method has been declared*/
		instance_base_type = NULL;
		/* use same algorithm as symbol_lookup_inherited*/
		if (VALA_IS_CLASS (vala_data_type_get_data_type (instance_type))) {
			ValaClass* _tmp4;
			ValaClass* cl;
			_tmp4 = NULL;
			cl = (_tmp4 = VALA_CLASS (vala_data_type_get_data_type (instance_type)), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
			{
				GeeCollection* base_type_collection;
				GeeIterator* base_type_it;
				base_type_collection = vala_class_get_base_types (cl);
				base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
				while (gee_iterator_next (base_type_it)) {
					ValaDataType* base_type;
					base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
					{
						/* first check interfaces without prerequisites
						 (prerequisites can be assumed to be met already)*/
						if (VALA_IS_INTERFACE (vala_data_type_get_data_type (base_type))) {
							ValaSymbol* _tmp5;
							gboolean _tmp6;
							_tmp5 = NULL;
							if ((_tmp6 = (_tmp5 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_data_type_get_data_type (base_type))), vala_symbol_get_name (generic_member))) == generic_member, (_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL))), _tmp6)) {
								ValaDataType* _tmp7;
								_tmp7 = NULL;
								instance_base_type = (_tmp7 = vala_semantic_analyzer_get_instance_base_type (instance_type, base_type, node_reference), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), _tmp7);
								(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
								break;
							}
						}
						(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
					}
				}
				(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
				(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
			}
			/* then check base class recursively*/
			if (instance_base_type == NULL) {
				{
					GeeCollection* base_type_collection;
					GeeIterator* base_type_it;
					base_type_collection = vala_class_get_base_types (cl);
					base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
					while (gee_iterator_next (base_type_it)) {
						ValaDataType* base_type;
						base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
						{
							if (VALA_IS_CLASS (vala_data_type_get_data_type (base_type))) {
								ValaSymbol* _tmp8;
								gboolean _tmp9;
								_tmp8 = NULL;
								if ((_tmp9 = (_tmp8 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_class_get_base_class (cl)), vala_symbol_get_name (generic_member))) == generic_member, (_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL))), _tmp9)) {
									ValaDataType* _tmp10;
									_tmp10 = NULL;
									instance_base_type = (_tmp10 = vala_semantic_analyzer_get_instance_base_type (instance_type, base_type, node_reference), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), _tmp10);
									(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
									break;
								}
							}
							(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
						}
					}
					(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
					(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
				}
			}
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		} else {
			if (VALA_IS_STRUCT (vala_data_type_get_data_type (instance_type))) {
				ValaStruct* _tmp11;
				ValaStruct* st;
				_tmp11 = NULL;
				st = (_tmp11 = VALA_STRUCT (vala_data_type_get_data_type (instance_type)), (_tmp11 == NULL ? NULL : g_object_ref (_tmp11)));
				{
					GeeCollection* base_type_collection;
					GeeIterator* base_type_it;
					base_type_collection = vala_struct_get_base_types (st);
					base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
					while (gee_iterator_next (base_type_it)) {
						ValaDataType* base_type;
						base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
						{
							ValaSymbol* _tmp12;
							gboolean _tmp13;
							_tmp12 = NULL;
							if ((_tmp13 = (_tmp12 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_data_type_get_data_type (base_type))), vala_symbol_get_name (generic_member))) == generic_member, (_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL))), _tmp13)) {
								ValaDataType* _tmp14;
								_tmp14 = NULL;
								instance_base_type = (_tmp14 = vala_semantic_analyzer_get_instance_base_type (instance_type, base_type, node_reference), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), _tmp14);
								(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
								break;
							}
							(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
						}
					}
					(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
					(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
				}
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			} else {
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (instance_type))) {
					ValaInterface* _tmp15;
					ValaInterface* iface;
					_tmp15 = NULL;
					iface = (_tmp15 = VALA_INTERFACE (vala_data_type_get_data_type (instance_type)), (_tmp15 == NULL ? NULL : g_object_ref (_tmp15)));
					{
						GeeCollection* prerequisite_collection;
						GeeIterator* prerequisite_it;
						prerequisite_collection = vala_interface_get_prerequisites (iface);
						prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
						while (gee_iterator_next (prerequisite_it)) {
							ValaDataType* prerequisite;
							prerequisite = ((ValaDataType*) gee_iterator_get (prerequisite_it));
							{
								/* first check interface prerequisites recursively*/
								if (VALA_IS_INTERFACE (vala_data_type_get_data_type (prerequisite))) {
									ValaSymbol* _tmp16;
									gboolean _tmp17;
									_tmp16 = NULL;
									if ((_tmp17 = (_tmp16 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_data_type_get_data_type (prerequisite)), vala_symbol_get_name (generic_member))) == generic_member, (_tmp16 == NULL ? NULL : (_tmp16 = (g_object_unref (_tmp16), NULL))), _tmp17)) {
										ValaDataType* _tmp18;
										_tmp18 = NULL;
										instance_base_type = (_tmp18 = vala_semantic_analyzer_get_instance_base_type (instance_type, prerequisite, node_reference), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), _tmp18);
										(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
										break;
									}
								}
								(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
							}
						}
						(prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL)));
						(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
					}
					if (instance_base_type == NULL) {
						{
							GeeCollection* prerequisite_collection;
							GeeIterator* prerequisite_it;
							prerequisite_collection = vala_interface_get_prerequisites (iface);
							prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
							while (gee_iterator_next (prerequisite_it)) {
								ValaDataType* prerequisite;
								prerequisite = ((ValaDataType*) gee_iterator_get (prerequisite_it));
								{
									/* then check class prerequisite recursively*/
									if (VALA_IS_CLASS (vala_data_type_get_data_type (prerequisite))) {
										ValaSymbol* _tmp19;
										gboolean _tmp20;
										_tmp19 = NULL;
										if ((_tmp20 = (_tmp19 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_data_type_get_data_type (prerequisite)), vala_symbol_get_name (generic_member))) == generic_member, (_tmp19 == NULL ? NULL : (_tmp19 = (g_object_unref (_tmp19), NULL))), _tmp20)) {
											ValaDataType* _tmp21;
											_tmp21 = NULL;
											instance_base_type = (_tmp21 = vala_semantic_analyzer_get_instance_base_type (instance_type, prerequisite, node_reference), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), _tmp21);
											(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
											break;
										}
									}
									(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
								}
							}
							(prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL)));
							(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
						}
					}
					(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
				} else {
					gpointer _tmp22;
					vala_report_error (vala_code_node_get_source_reference (node_reference), "internal error: unsupported generic type");
					vala_code_node_set_error (node_reference, TRUE);
					return (_tmp22 = NULL, (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp22);
				}
			}
		}
		if (instance_base_type == NULL) {
			char* _tmp23;
			gpointer _tmp24;
			_tmp23 = NULL;
			vala_report_error (vala_code_node_get_source_reference (node_reference), (_tmp23 = g_strdup_printf ("internal error: unable to find generic member `%s'", vala_symbol_get_name (generic_member))));
			_tmp23 = (g_free (_tmp23), NULL);
			vala_code_node_set_error (node_reference, TRUE);
			return (_tmp24 = NULL, (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp24);
		}
		_tmp26 = NULL;
		_tmp25 = NULL;
		instance_type = (_tmp26 = (_tmp25 = instance_base_type, (_tmp25 == NULL ? NULL : g_object_ref (_tmp25))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp26);
		while (VALA_IS_POINTER_TYPE (instance_type)) {
			ValaPointerType* _tmp27;
			ValaPointerType* instance_pointer_type;
			ValaDataType* _tmp29;
			ValaDataType* _tmp28;
			_tmp27 = NULL;
			instance_pointer_type = (_tmp27 = VALA_POINTER_TYPE (instance_type), (_tmp27 == NULL ? NULL : g_object_ref (_tmp27)));
			_tmp29 = NULL;
			_tmp28 = NULL;
			instance_type = (_tmp29 = (_tmp28 = vala_pointer_type_get_base_type (instance_pointer_type), (_tmp28 == NULL ? NULL : g_object_ref (_tmp28))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp29);
			(instance_pointer_type == NULL ? NULL : (instance_pointer_type = (g_object_unref (instance_pointer_type), NULL)));
		}
		(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
	}
	if (VALA_SYMBOL (vala_data_type_get_data_type (instance_type)) != vala_symbol_get_parent_symbol (generic_member)) {
		gpointer _tmp30;
		vala_report_error (vala_code_node_get_source_reference (node_reference), "internal error: generic type parameter tracing not supported yet");
		vala_code_node_set_error (node_reference, TRUE);
		return (_tmp30 = NULL, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp30);
	}
	param_index = vala_typesymbol_get_type_parameter_index (vala_data_type_get_data_type (instance_type), vala_symbol_get_name (VALA_SYMBOL (vala_data_type_get_type_parameter (generic_type))));
	if (param_index == -1) {
		char* _tmp31;
		gpointer _tmp32;
		_tmp31 = NULL;
		vala_report_error (vala_code_node_get_source_reference (node_reference), (_tmp31 = g_strdup_printf ("internal error: unknown type parameter %s", vala_symbol_get_name (VALA_SYMBOL (vala_data_type_get_type_parameter (generic_type))))));
		_tmp31 = (g_free (_tmp31), NULL);
		vala_code_node_set_error (node_reference, TRUE);
		return (_tmp32 = NULL, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp32);
	}
	actual_type = NULL;
	_tmp33 = NULL;
	if ((_tmp34 = param_index < gee_collection_get_size (GEE_COLLECTION ((_tmp33 = vala_data_type_get_type_arguments (instance_type)))), (_tmp33 == NULL ? NULL : (_tmp33 = (g_object_unref (_tmp33), NULL))), _tmp34)) {
		ValaDataType* _tmp36;
		GeeList* _tmp35;
		_tmp36 = NULL;
		_tmp35 = NULL;
		actual_type = (_tmp36 = VALA_DATA_TYPE (((ValaDataType*) gee_list_get ((_tmp35 = vala_data_type_get_type_arguments (instance_type)), param_index))), (actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL))), _tmp36);
		(_tmp35 == NULL ? NULL : (_tmp35 = (g_object_unref (_tmp35), NULL)));
	}
	if (actual_type == NULL) {
		ValaDataType* _tmp37;
		ValaDataType* _tmp38;
		_tmp37 = NULL;
		_tmp38 = NULL;
		return (_tmp38 = (_tmp37 = generic_type, (_tmp37 == NULL ? NULL : g_object_ref (_tmp37))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), (actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL))), _tmp38);
	}
	/* no actual type available*/
	_tmp39 = NULL;
	actual_type = (_tmp39 = vala_data_type_copy (actual_type), (actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL))), _tmp39);
	vala_data_type_set_transfers_ownership (actual_type, vala_data_type_get_takes_ownership (actual_type) && vala_data_type_get_transfers_ownership (generic_type));
	vala_data_type_set_takes_ownership (actual_type, vala_data_type_get_takes_ownership (actual_type) && vala_data_type_get_takes_ownership (generic_type));
	_tmp40 = NULL;
	return (_tmp40 = actual_type, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp40);
}


static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypesymbol* _tmp0;
	ValaTypesymbol* container_type;
	gboolean index_int_type_check;
	ValaPointerType* _tmp2;
	ValaDataType* _tmp1;
	ValaPointerType* pointer_type;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_ELEMENT_ACCESS (expr));
	if (vala_expression_get_static_type (vala_element_access_get_container (expr)) == NULL) {
		/* don't proceed if a child expression failed */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	_tmp0 = NULL;
	container_type = (_tmp0 = vala_data_type_get_data_type (vala_expression_get_static_type (vala_element_access_get_container (expr))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	index_int_type_check = TRUE;
	_tmp2 = NULL;
	_tmp1 = NULL;
	pointer_type = (_tmp2 = (_tmp1 = vala_expression_get_static_type (vala_element_access_get_container (expr)), (VALA_IS_POINTER_TYPE (_tmp1) ? ((ValaPointerType*) _tmp1) : NULL)), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
	/* assign a static_type when possible */
	if (VALA_IS_ARRAY_TYPE (vala_expression_get_static_type (vala_element_access_get_container (expr)))) {
		GeeList* args;
		ValaDataType* _tmp4;
		args = vala_data_type_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (expr)));
		if (gee_collection_get_size (GEE_COLLECTION (args)) != 1) {
			char* _tmp3;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp3 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp3 = g_strdup_printf ("internal error: array reference with %d type arguments, expected 1", gee_collection_get_size (GEE_COLLECTION (args)))));
			_tmp3 = (g_free (_tmp3), NULL);
			(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
			(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
			(pointer_type == NULL ? NULL : (pointer_type = (g_object_unref (pointer_type), NULL)));
			return;
		}
		_tmp4 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp4 = ((ValaDataType*) gee_list_get (args, 0))));
		(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
		(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
	} else {
		if (pointer_type != NULL && !vala_data_type_is_reference_type_or_type_parameter (vala_pointer_type_get_base_type (pointer_type))) {
			ValaDataType* _tmp5;
			_tmp5 = NULL;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp5 = vala_data_type_copy (vala_pointer_type_get_base_type (pointer_type))));
			(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
		} else {
			if (container_type == vala_data_type_get_data_type (self->priv->string_type)) {
				GeeList* _tmp6;
				gboolean _tmp7;
				_tmp6 = NULL;
				if ((_tmp7 = gee_collection_get_size (GEE_COLLECTION ((_tmp6 = vala_element_access_get_indices (expr)))) != 1, (_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL))), _tmp7)) {
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Element access with more than one dimension is not supported for strings");
					(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
					(pointer_type == NULL ? NULL : (pointer_type = (g_object_unref (pointer_type), NULL)));
					return;
				}
				vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->unichar_type);
			} else {
				if (container_type != NULL && self->priv->list_type != NULL && self->priv->map_type != NULL && (vala_typesymbol_is_subtype_of (container_type, VALA_TYPESYMBOL (self->priv->list_type)) || vala_typesymbol_is_subtype_of (container_type, VALA_TYPESYMBOL (self->priv->map_type)))) {
					GeeCollection* indices;
					GeeIterator* indices_it;
					ValaExpression* index;
					ValaSymbol* get_sym;
					ValaMethod* _tmp8;
					ValaMethod* get_method;
					GeeCollection* get_params;
					GeeIterator* get_params_it;
					ValaFormalParameter* get_param;
					ValaDataType* _tmp9;
					ValaDataType* index_type;
					ValaDataType* _tmp15;
					ValaDataType* _tmp14;
					indices = GEE_COLLECTION (vala_element_access_get_indices (expr));
					if (gee_collection_get_size (indices) != 1) {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Element access with more than one dimension is not supported for the specified type");
						(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
						(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
						(pointer_type == NULL ? NULL : (pointer_type = (g_object_unref (pointer_type), NULL)));
						return;
					}
					indices_it = gee_iterable_iterator (GEE_ITERABLE (indices));
					gee_iterator_next (indices_it);
					index = ((ValaExpression*) gee_iterator_get (indices_it));
					index_int_type_check = FALSE;
					get_sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (container_type)), "get");
					if (!(VALA_IS_METHOD (get_sym))) {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invalid get method in specified collection type");
						(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
						(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
						(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
						(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
						(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
						(pointer_type == NULL ? NULL : (pointer_type = (g_object_unref (pointer_type), NULL)));
						return;
					}
					_tmp8 = NULL;
					get_method = (_tmp8 = VALA_METHOD (get_sym), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8)));
					get_params = vala_method_get_parameters (get_method);
					get_params_it = gee_iterable_iterator (GEE_ITERABLE (get_params));
					gee_iterator_next (get_params_it);
					get_param = ((ValaFormalParameter*) gee_iterator_get (get_params_it));
					_tmp9 = NULL;
					index_type = (_tmp9 = vala_formal_parameter_get_type_reference (get_param), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9)));
					if (vala_data_type_get_type_parameter (index_type) != NULL) {
						ValaDataType* _tmp10;
						_tmp10 = NULL;
						index_type = (_tmp10 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_element_access_get_container (expr)), VALA_SYMBOL (get_method), vala_formal_parameter_get_type_reference (get_param), VALA_CODE_NODE (expr)), (index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL))), _tmp10);
					}
					if (!vala_data_type_compatible (vala_expression_get_static_type (index), index_type, TRUE)) {
						char* _tmp13;
						char* _tmp12;
						char* _tmp11;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp13 = NULL;
						_tmp12 = NULL;
						_tmp11 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp13 = g_strdup_printf ("index expression: Cannot convert from `%s' to `%s'", (_tmp11 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (index)))), (_tmp12 = vala_code_node_to_string (VALA_CODE_NODE (index_type))))));
						_tmp13 = (g_free (_tmp13), NULL);
						_tmp12 = (g_free (_tmp12), NULL);
						_tmp11 = (g_free (_tmp11), NULL);
						(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
						(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
						(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
						(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
						(get_method == NULL ? NULL : (get_method = (g_object_unref (get_method), NULL)));
						(get_params == NULL ? NULL : (get_params = (g_object_unref (get_params), NULL)));
						(get_params_it == NULL ? NULL : (get_params_it = (g_object_unref (get_params_it), NULL)));
						(get_param == NULL ? NULL : (get_param = (g_object_unref (get_param), NULL)));
						(index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL)));
						(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
						(pointer_type == NULL ? NULL : (pointer_type = (g_object_unref (pointer_type), NULL)));
						return;
					}
					_tmp15 = NULL;
					_tmp14 = NULL;
					vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp15 = vala_data_type_copy ((_tmp14 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_element_access_get_container (expr)), VALA_SYMBOL (get_method), vala_method_get_return_type (get_method), VALA_CODE_NODE (expr))))));
					(_tmp15 == NULL ? NULL : (_tmp15 = (g_object_unref (_tmp15), NULL)));
					(_tmp14 == NULL ? NULL : (_tmp14 = (g_object_unref (_tmp14), NULL)));
					vala_data_type_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE);
					(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
					(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
					(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
					(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
					(get_method == NULL ? NULL : (get_method = (g_object_unref (get_method), NULL)));
					(get_params == NULL ? NULL : (get_params = (g_object_unref (get_params), NULL)));
					(get_params_it == NULL ? NULL : (get_params_it = (g_object_unref (get_params_it), NULL)));
					(get_param == NULL ? NULL : (get_param = (g_object_unref (get_param), NULL)));
					(index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL)));
				} else {
					char* _tmp17;
					char* _tmp16;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp17 = NULL;
					_tmp16 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp17 = g_strdup_printf ("The expression `%s' does not denote an Array", (_tmp16 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_element_access_get_container (expr))))))));
					_tmp17 = (g_free (_tmp17), NULL);
					_tmp16 = (g_free (_tmp16), NULL);
				}
			}
		}
	}
	if (index_int_type_check) {
		{
			GeeList* e_collection;
			GeeIterator* e_it;
			e_collection = vala_element_access_get_indices (expr);
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = ((ValaExpression*) gee_iterator_get (e_it));
				{
					/* check if the index is of type integer 
					 don't proceed if a child expression failed */
					if (vala_expression_get_static_type (e) == NULL) {
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
						(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
						(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
						(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
						(pointer_type == NULL ? NULL : (pointer_type = (g_object_unref (pointer_type), NULL)));
						return;
					}
					/* check if the index is of type integer */
					if (!(VALA_IS_STRUCT (vala_data_type_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_data_type_get_data_type (vala_expression_get_static_type (e)))))) {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
					}
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
	}
	(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
	(pointer_type == NULL ? NULL : (pointer_type = (g_object_unref (pointer_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_BASE_ACCESS (expr));
	if (self->priv->current_class == NULL) {
		GeeCollection* _tmp4;
		GeeIterator* _tmp5;
		GeeIterator* base_type_it;
		ValaDataType* _tmp6;
		if (self->priv->current_struct == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Base access invalid outside of class and struct");
			return;
		} else {
			GeeCollection* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = gee_collection_get_size ((_tmp0 = vala_struct_get_base_types (self->priv->current_struct))) != 1, (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1)) {
				char* _tmp3;
				GeeCollection* _tmp2;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp3 = g_strdup_printf ("Base access invalid without base type %d", gee_collection_get_size ((_tmp2 = vala_struct_get_base_types (self->priv->current_struct))))));
				_tmp3 = (g_free (_tmp3), NULL);
				(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
				return;
			}
		}
		_tmp4 = NULL;
		_tmp5 = NULL;
		base_type_it = (_tmp5 = gee_iterable_iterator (GEE_ITERABLE ((_tmp4 = vala_struct_get_base_types (self->priv->current_struct)))), (_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL))), _tmp5);
		gee_iterator_next (base_type_it);
		_tmp6 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp6 = ((ValaDataType*) gee_iterator_get (base_type_it))));
		(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	} else {
		ValaClassType* _tmp7;
		_tmp7 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp7 = vala_class_type_new (vala_class_get_base_class (self->priv->current_class)))));
		(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
	}
	vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_data_type_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)))));
}


static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_POSTFIX_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_postfix_expression_get_inner (expr)));
}


static void vala_semantic_analyzer_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypesymbol* type;
	ValaDataType* _tmp21;
	GeeCollection* _tmp28;
	gboolean _tmp29;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
	if (vala_object_creation_expression_get_member_name (expr) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_object_creation_expression_get_member_name (expr)), VALA_CODE_VISITOR (self));
	}
	type = NULL;
	if (vala_object_creation_expression_get_type_reference (expr) == NULL) {
		ValaSymbol* _tmp0;
		ValaSymbol* constructor_sym;
		ValaSymbol* _tmp1;
		ValaSymbol* type_sym;
		GeeCollection* type_args;
		if (vala_object_creation_expression_get_member_name (expr) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Incomplete object creation expression");
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		_tmp0 = NULL;
		constructor_sym = (_tmp0 = vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		_tmp1 = NULL;
		type_sym = (_tmp1 = vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
		type_args = vala_member_access_get_type_arguments (vala_object_creation_expression_get_member_name (expr));
		if (VALA_IS_METHOD (constructor_sym)) {
			ValaSymbol* _tmp3;
			ValaSymbol* _tmp2;
			ValaMethod* _tmp4;
			ValaMethod* constructor;
			GeeCollection* _tmp7;
			_tmp3 = NULL;
			_tmp2 = NULL;
			type_sym = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (constructor_sym), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL))), _tmp3);
			_tmp4 = NULL;
			constructor = (_tmp4 = VALA_METHOD (constructor_sym), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
			if (!(VALA_IS_CREATION_METHOD (constructor_sym))) {
				char* _tmp6;
				char* _tmp5;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp6 = NULL;
				_tmp5 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp6 = g_strdup_printf ("`%s' is not a creation method", (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (constructor))))));
				_tmp6 = (g_free (_tmp6), NULL);
				_tmp5 = (g_free (_tmp5), NULL);
				(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
				(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
				(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (constructor));
			_tmp7 = NULL;
			type_args = (_tmp7 = vala_member_access_get_type_arguments ((VALA_MEMBER_ACCESS (vala_member_access_get_inner (vala_object_creation_expression_get_member_name (expr))))), (type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL))), _tmp7);
			(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
		} else {
			if (VALA_IS_ERROR_CODE (constructor_sym)) {
				ValaSymbol* _tmp9;
				ValaSymbol* _tmp8;
				_tmp9 = NULL;
				_tmp8 = NULL;
				type_sym = (_tmp9 = (_tmp8 = vala_symbol_get_parent_symbol (constructor_sym), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8))), (type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL))), _tmp9);
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), constructor_sym);
			}
		}
		if (VALA_IS_CLASS (type_sym)) {
			ValaTypesymbol* _tmp11;
			ValaTypesymbol* _tmp10;
			ValaClassType* _tmp12;
			_tmp11 = NULL;
			_tmp10 = NULL;
			type = (_tmp11 = (_tmp10 = VALA_TYPESYMBOL (type_sym), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp11);
			_tmp12 = NULL;
			vala_object_creation_expression_set_type_reference (expr, VALA_DATA_TYPE ((_tmp12 = vala_class_type_new (VALA_CLASS (type)))));
			(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
		} else {
			if (VALA_IS_STRUCT (type_sym)) {
				ValaTypesymbol* _tmp14;
				ValaTypesymbol* _tmp13;
				ValaValueType* _tmp15;
				_tmp14 = NULL;
				_tmp13 = NULL;
				type = (_tmp14 = (_tmp13 = VALA_TYPESYMBOL (type_sym), (_tmp13 == NULL ? NULL : g_object_ref (_tmp13))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp14);
				_tmp15 = NULL;
				vala_object_creation_expression_set_type_reference (expr, VALA_DATA_TYPE ((_tmp15 = vala_value_type_new (type))));
				(_tmp15 == NULL ? NULL : (_tmp15 = (g_object_unref (_tmp15), NULL)));
			} else {
				if (VALA_IS_ERROR_DOMAIN (type_sym)) {
					ValaErrorType* _tmp16;
					_tmp16 = NULL;
					vala_object_creation_expression_set_type_reference (expr, VALA_DATA_TYPE ((_tmp16 = vala_error_type_new (VALA_ERROR_DOMAIN (type_sym), vala_code_node_get_source_reference (VALA_CODE_NODE (expr))))));
					(_tmp16 == NULL ? NULL : (_tmp16 = (g_object_unref (_tmp16), NULL)));
				} else {
					char* _tmp18;
					char* _tmp17;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp18 = NULL;
					_tmp17 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp18 = g_strdup_printf ("`%s' is not a class, struct, or error domain", (_tmp17 = vala_symbol_get_full_name (type_sym)))));
					_tmp18 = (g_free (_tmp18), NULL);
					_tmp17 = (g_free (_tmp17), NULL);
					(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
					(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
					(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					return;
				}
			}
		}
		{
			GeeCollection* type_arg_collection;
			GeeIterator* type_arg_it;
			type_arg_collection = type_args;
			type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
			while (gee_iterator_next (type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = ((ValaDataType*) gee_iterator_get (type_arg_it));
				{
					vala_data_type_add_type_argument (vala_object_creation_expression_get_type_reference (expr), type_arg);
					vala_source_file_add_type_dependency (self->priv->current_source_file, type_arg, VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
					(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
				}
			}
			(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
		}
		(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
		(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
		(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
	} else {
		ValaTypesymbol* _tmp20;
		ValaTypesymbol* _tmp19;
		_tmp20 = NULL;
		_tmp19 = NULL;
		type = (_tmp20 = (_tmp19 = vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr)), (_tmp19 == NULL ? NULL : g_object_ref (_tmp19))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp20);
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (type), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	_tmp21 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp21 = vala_data_type_copy (vala_object_creation_expression_get_type_reference (expr))));
	(_tmp21 == NULL ? NULL : (_tmp21 = (g_object_unref (_tmp21), NULL)));
	vala_data_type_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
	if (VALA_IS_CLASS (type)) {
		ValaClass* _tmp22;
		ValaClass* cl;
		_tmp22 = NULL;
		cl = (_tmp22 = VALA_CLASS (type), (_tmp22 == NULL ? NULL : g_object_ref (_tmp22)));
		if (vala_class_get_is_abstract (cl)) {
			char* _tmp24;
			char* _tmp23;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL);
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp24 = NULL;
			_tmp23 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp24 = g_strdup_printf ("Can't create instance of abstract class `%s'", (_tmp23 = vala_symbol_get_full_name (VALA_SYMBOL (cl))))));
			_tmp24 = (g_free (_tmp24), NULL);
			_tmp23 = (g_free (_tmp23), NULL);
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_class_get_default_construction_method (cl)));
		}
		while (cl != NULL) {
			ValaClass* _tmp26;
			ValaClass* _tmp25;
			if (VALA_TYPESYMBOL (cl) == self->priv->initially_unowned_type) {
				vala_data_type_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
				break;
			}
			_tmp26 = NULL;
			_tmp25 = NULL;
			cl = (_tmp26 = (_tmp25 = vala_class_get_base_class (cl), (_tmp25 == NULL ? NULL : g_object_ref (_tmp25))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp26);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (type)) {
			ValaStruct* _tmp27;
			ValaStruct* st;
			_tmp27 = NULL;
			st = (_tmp27 = VALA_STRUCT (type), (_tmp27 == NULL ? NULL : g_object_ref (_tmp27)));
			vala_data_type_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE);
			if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_struct_get_default_construction_method (st)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		}
	}
	_tmp28 = NULL;
	if ((_tmp29 = vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && gee_collection_get_size ((_tmp28 = vala_object_creation_expression_get_argument_list (expr))) != 0, (_tmp28 == NULL ? NULL : (_tmp28 = (g_object_unref (_tmp28), NULL))), _tmp29)) {
		char* _tmp31;
		char* _tmp30;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL);
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		_tmp31 = NULL;
		_tmp30 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp31 = g_strdup_printf ("No arguments allowed when constructing type `%s'", (_tmp30 = vala_symbol_get_full_name (VALA_SYMBOL (type))))));
		_tmp31 = (g_free (_tmp31), NULL);
		_tmp30 = (g_free (_tmp30), NULL);
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	if (VALA_IS_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))) {
		ValaMethod* _tmp32;
		ValaMethod* m;
		GeeCollection* args;
		GeeIterator* arg_it;
		GeeCollection* _tmp34;
		ValaMethodType* _tmp33;
		GeeCollection* _tmp35;
		_tmp32 = NULL;
		m = (_tmp32 = VALA_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr))), (_tmp32 == NULL ? NULL : g_object_ref (_tmp32)));
		args = vala_object_creation_expression_get_argument_list (expr);
		arg_it = gee_iterable_iterator (GEE_ITERABLE (args));
		{
			GeeCollection* param_collection;
			GeeIterator* param_it;
			param_collection = vala_method_get_parameters (m);
			param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
			while (gee_iterator_next (param_it)) {
				ValaFormalParameter* param;
				param = ((ValaFormalParameter*) gee_iterator_get (param_it));
				{
					if (vala_formal_parameter_get_ellipsis (param)) {
						(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
						break;
					}
					if (gee_iterator_next (arg_it)) {
						ValaExpression* arg;
						arg = ((ValaExpression*) gee_iterator_get (arg_it));
						/* store expected type for callback parameters */
						vala_expression_set_expected_type (arg, vala_formal_parameter_get_type_reference (param));
						(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
					}
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				}
			}
			(param_collection == NULL ? NULL : (param_collection = (g_object_unref (param_collection), NULL)));
			(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
		}
		{
			GeeCollection* arg_collection;
			GeeIterator* arg_it;
			arg_collection = args;
			arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
			while (gee_iterator_next (arg_it)) {
				ValaExpression* arg;
				arg = ((ValaExpression*) gee_iterator_get (arg_it));
				{
					vala_code_node_accept (VALA_CODE_NODE (arg), VALA_CODE_VISITOR (self));
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
			}
			(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
		}
		_tmp34 = NULL;
		_tmp33 = NULL;
		vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp33 = vala_method_type_new (m))), (_tmp34 = vala_method_get_parameters (m)), args);
		(_tmp34 == NULL ? NULL : (_tmp34 = (g_object_unref (_tmp34), NULL)));
		(_tmp33 == NULL ? NULL : (_tmp33 = (g_object_unref (_tmp33), NULL)));
		_tmp35 = NULL;
		vala_code_node_set_tree_can_fail (VALA_CODE_NODE (expr), (vala_expression_set_can_fail (VALA_EXPRESSION (expr), (gee_collection_get_size ((_tmp35 = vala_method_get_error_domains (m))) > 0)), vala_expression_get_can_fail (VALA_EXPRESSION (expr))));
		(_tmp35 == NULL ? NULL : (_tmp35 = (g_object_unref (_tmp35), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
		(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	} else {
		if (VALA_IS_ERROR_TYPE (vala_object_creation_expression_get_type_reference (expr))) {
			GeeCollection* _tmp36;
			gboolean _tmp37;
			ValaVoidType* _tmp40;
			vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
			_tmp36 = NULL;
			if ((_tmp37 = gee_collection_get_size ((_tmp36 = vala_object_creation_expression_get_argument_list (expr))) == 0, (_tmp36 == NULL ? NULL : (_tmp36 = (g_object_unref (_tmp36), NULL))), _tmp37)) {
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Too few arguments, errors need at least 1 argument");
			} else {
				GeeCollection* _tmp38;
				GeeIterator* _tmp39;
				GeeIterator* arg_it;
				ValaExpression* ex;
				_tmp38 = NULL;
				_tmp39 = NULL;
				arg_it = (_tmp39 = gee_iterable_iterator (GEE_ITERABLE ((_tmp38 = vala_object_creation_expression_get_argument_list (expr)))), (_tmp38 == NULL ? NULL : (_tmp38 = (g_object_unref (_tmp38), NULL))), _tmp39);
				gee_iterator_next (arg_it);
				ex = ((ValaExpression*) gee_iterator_get (arg_it));
				if (vala_expression_get_static_type (ex) == NULL || !vala_data_type_compatible (vala_expression_get_static_type (ex), self->priv->string_type, TRUE)) {
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Invalid type for argument 1");
				}
				(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
				(ex == NULL ? NULL : (ex = (g_object_unref (ex), NULL)));
			}
			_tmp40 = NULL;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp40 = vala_void_type_new ())));
			(_tmp40 == NULL ? NULL : (_tmp40 = (g_object_unref (_tmp40), NULL)));
		}
	}
	{
		GeeCollection* init_collection;
		GeeIterator* init_it;
		init_collection = vala_object_creation_expression_get_object_initializer (expr);
		init_it = gee_iterable_iterator (GEE_ITERABLE (init_collection));
		while (gee_iterator_next (init_it)) {
			ValaMemberInitializer* init;
			init = ((ValaMemberInitializer*) gee_iterator_get (init_it));
			{
				ValaSymbol* _tmp41;
				ValaDataType* member_type;
				vala_code_node_accept (VALA_CODE_NODE (init), VALA_CODE_VISITOR (self));
				_tmp41 = NULL;
				vala_member_initializer_set_symbol_reference (init, (_tmp41 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr))), vala_member_initializer_get_name (init))));
				(_tmp41 == NULL ? NULL : (_tmp41 = (g_object_unref (_tmp41), NULL)));
				if (!(VALA_IS_FIELD (vala_member_initializer_get_symbol_reference (init)) || VALA_IS_PROPERTY (vala_member_initializer_get_symbol_reference (init)))) {
					char* _tmp43;
					char* _tmp42;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp43 = NULL;
					_tmp42 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp43 = g_strdup_printf ("Invalid member `%s' in `%s'", vala_member_initializer_get_name (init), (_tmp42 = vala_symbol_get_full_name (VALA_SYMBOL (vala_data_type_get_data_type (vala_object_creation_expression_get_type_reference (expr))))))));
					_tmp43 = (g_free (_tmp43), NULL);
					_tmp42 = (g_free (_tmp42), NULL);
					(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
					(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
					(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					return;
				}
				if (vala_symbol_get_access (vala_member_initializer_get_symbol_reference (init)) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
					char* _tmp45;
					char* _tmp44;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp45 = NULL;
					_tmp44 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp45 = g_strdup_printf ("Access to private member `%s' denied", (_tmp44 = vala_symbol_get_full_name (vala_member_initializer_get_symbol_reference (init))))));
					_tmp45 = (g_free (_tmp45), NULL);
					_tmp44 = (g_free (_tmp44), NULL);
					(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
					(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
					(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					return;
				}
				member_type = NULL;
				if (VALA_IS_FIELD (vala_member_initializer_get_symbol_reference (init))) {
					ValaField* _tmp46;
					ValaField* f;
					ValaDataType* _tmp48;
					ValaDataType* _tmp47;
					_tmp46 = NULL;
					f = (_tmp46 = VALA_FIELD (vala_member_initializer_get_symbol_reference (init)), (_tmp46 == NULL ? NULL : g_object_ref (_tmp46)));
					_tmp48 = NULL;
					_tmp47 = NULL;
					member_type = (_tmp48 = (_tmp47 = vala_field_get_type_reference (f), (_tmp47 == NULL ? NULL : g_object_ref (_tmp47))), (member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL))), _tmp48);
					(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
				} else {
					if (VALA_IS_PROPERTY (vala_member_initializer_get_symbol_reference (init))) {
						ValaProperty* _tmp49;
						ValaProperty* prop;
						ValaDataType* _tmp51;
						ValaDataType* _tmp50;
						_tmp49 = NULL;
						prop = (_tmp49 = VALA_PROPERTY (vala_member_initializer_get_symbol_reference (init)), (_tmp49 == NULL ? NULL : g_object_ref (_tmp49)));
						_tmp51 = NULL;
						_tmp50 = NULL;
						member_type = (_tmp51 = (_tmp50 = vala_property_get_type_reference (prop), (_tmp50 == NULL ? NULL : g_object_ref (_tmp50))), (member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL))), _tmp51);
						if (vala_property_get_set_accessor (prop) == NULL || !vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
							char* _tmp53;
							char* _tmp52;
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							_tmp53 = NULL;
							_tmp52 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp53 = g_strdup_printf ("Property `%s' is read-only", (_tmp52 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
							_tmp53 = (g_free (_tmp53), NULL);
							_tmp52 = (g_free (_tmp52), NULL);
							(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
							(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
							(member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL)));
							(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
							(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							return;
						}
						(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					}
				}
				if (vala_expression_get_static_type (vala_member_initializer_get_initializer (init)) == NULL || !vala_data_type_compatible (vala_expression_get_static_type (vala_member_initializer_get_initializer (init)), member_type, TRUE)) {
					char* _tmp54;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp54 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (init)), (_tmp54 = g_strdup_printf ("Invalid type for member `%s'", vala_member_initializer_get_name (init))));
					_tmp54 = (g_free (_tmp54), NULL);
					(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
					(member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL)));
					(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
					(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					return;
				}
				(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
				(member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL)));
			}
		}
		(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
		(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
	}
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_SIZEOF_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->ulong_type);
}


static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_TYPEOF_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->type_type);
}


static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaDataType* type) {
	ValaStruct* _tmp1;
	ValaStruct* st;
	gboolean _tmp2;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (type == NULL || VALA_IS_DATA_TYPE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_data_type_get_data_type (type)))) {
		return FALSE;
	}
	_tmp1 = NULL;
	st = (_tmp1 = VALA_STRUCT (vala_data_type_get_data_type (type)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
	return (_tmp2 = vala_struct_is_integer_type (st) || vala_struct_is_floating_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp2);
}


static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaDataType* type) {
	ValaStruct* _tmp1;
	ValaStruct* st;
	gboolean _tmp2;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (type == NULL || VALA_IS_DATA_TYPE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_data_type_get_data_type (type)))) {
		return FALSE;
	}
	_tmp1 = NULL;
	st = (_tmp1 = VALA_STRUCT (vala_data_type_get_data_type (type)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
	return (_tmp2 = vala_struct_is_integer_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp2);
}


static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_UNARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_unary_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_PLUS || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_MINUS) {
		/* integer or floating point type*/
		if (!vala_semantic_analyzer_is_numeric_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
			char* _tmp1;
			char* _tmp0;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp1 = NULL;
			_tmp0 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp1 = g_strdup_printf ("Operator not supported for `%s'", (_tmp0 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))))))));
			_tmp1 = (g_free (_tmp1), NULL);
			_tmp0 = (g_free (_tmp0), NULL);
			return;
		}
		vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
	} else {
		if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_LOGICAL_NEGATION) {
			/* boolean type*/
			if (!vala_data_type_compatible (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)), self->priv->bool_type, TRUE)) {
				char* _tmp3;
				char* _tmp2;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp3 = g_strdup_printf ("Operator not supported for `%s'", (_tmp2 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))))))));
				_tmp3 = (g_free (_tmp3), NULL);
				_tmp2 = (g_free (_tmp2), NULL);
				return;
			}
			vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
		} else {
			if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT) {
				/* integer type*/
				if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
					char* _tmp5;
					char* _tmp4;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp5 = NULL;
					_tmp4 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp5 = g_strdup_printf ("Operator not supported for `%s'", (_tmp4 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))))))));
					_tmp5 = (g_free (_tmp5), NULL);
					_tmp4 = (g_free (_tmp4), NULL);
					return;
				}
				vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
			} else {
				if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_DECREMENT) {
					ValaMemberAccess* ma;
					ValaMemberAccess* old_value;
					ValaIntegerLiteral* _tmp8;
					ValaBinaryExpression* _tmp9;
					ValaBinaryExpression* bin;
					ValaAssignment* assignment;
					ValaParenthesizedExpression* parenthexp;
					/* integer type*/
					if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
						char* _tmp7;
						char* _tmp6;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp7 = NULL;
						_tmp6 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp7 = g_strdup_printf ("Operator not supported for `%s'", (_tmp6 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))))))));
						_tmp7 = (g_free (_tmp7), NULL);
						_tmp6 = (g_free (_tmp6), NULL);
						return;
					}
					ma = vala_semantic_analyzer_find_member_access (self, vala_unary_expression_get_inner (expr));
					if (ma == NULL) {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Prefix operators not supported for this expression");
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), vala_code_node_get_source_reference (VALA_CODE_NODE (vala_unary_expression_get_inner (expr))));
					_tmp8 = NULL;
					_tmp9 = NULL;
					bin = (_tmp9 = vala_binary_expression_new ((vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT ? VALA_BINARY_OPERATOR_PLUS : VALA_BINARY_OPERATOR_MINUS), VALA_EXPRESSION (old_value), VALA_EXPRESSION ((_tmp8 = vala_integer_literal_new ("1", NULL))), vala_code_node_get_source_reference (VALA_CODE_NODE (expr))), (_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL))), _tmp9);
					assignment = vala_code_context_create_assignment (self->priv->context, VALA_EXPRESSION (ma), VALA_EXPRESSION (bin), VALA_ASSIGNMENT_OPERATOR_SIMPLE, vala_code_node_get_source_reference (VALA_CODE_NODE (expr)));
					parenthexp = vala_parenthesized_expression_new (VALA_EXPRESSION (assignment), vala_code_node_get_source_reference (VALA_CODE_NODE (expr)));
					vala_code_node_replace_expression (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_EXPRESSION (expr), VALA_EXPRESSION (parenthexp));
					vala_code_node_accept (VALA_CODE_NODE (parenthexp), VALA_CODE_VISITOR (self));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
					(parenthexp == NULL ? NULL : (parenthexp = (g_object_unref (parenthexp), NULL)));
					return;
				} else {
					if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_REF || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_OUT) {
						if (VALA_IS_FIELD (vala_expression_get_symbol_reference (vala_unary_expression_get_inner (expr))) || VALA_IS_FORMAL_PARAMETER (vala_expression_get_symbol_reference (vala_unary_expression_get_inner (expr))) || VALA_IS_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (vala_unary_expression_get_inner (expr)))) {
							/* ref and out can only be used with fields, parameters, and local variables*/
							vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
						} else {
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "ref and out method arguments can only be used with fields, parameters, and local variables");
							return;
						}
					} else {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported unary operator");
						return;
					}
				}
			}
		}
	}
}


static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (expr == NULL || VALA_IS_EXPRESSION (expr), NULL);
	if (VALA_IS_PARENTHESIZED_EXPRESSION (expr)) {
		ValaParenthesizedExpression* _tmp0;
		ValaParenthesizedExpression* pe;
		ValaMemberAccess* _tmp1;
		_tmp0 = NULL;
		pe = (_tmp0 = VALA_PARENTHESIZED_EXPRESSION (expr), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		_tmp1 = NULL;
		return (_tmp1 = vala_semantic_analyzer_find_member_access (self, vala_parenthesized_expression_get_inner (pe)), (pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL))), _tmp1);
	}
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaMemberAccess* _tmp2;
		_tmp2 = NULL;
		return (_tmp2 = VALA_MEMBER_ACCESS (expr), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
	}
	return NULL;
}


static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_CAST_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_cast_expression_get_inner (expr)))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	/* FIXME: check whether cast is allowed*/
	vala_source_file_add_type_dependency (self->priv->current_source_file, vala_cast_expression_get_type_reference (expr), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_cast_expression_get_type_reference (expr));
	vala_data_type_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_data_type_get_transfers_ownership (vala_expression_get_static_type (vala_cast_expression_get_inner (expr))));
}


static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_POINTER_INDIRECTION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_pointer_indirection_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))) {
		ValaPointerType* _tmp0;
		ValaPointerType* pointer_type;
		_tmp0 = NULL;
		pointer_type = (_tmp0 = VALA_POINTER_TYPE (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_pointer_type_get_base_type (pointer_type));
		(pointer_type == NULL ? NULL : (pointer_type = (g_object_unref (pointer_type), NULL)));
	} else {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Pointer indirection not supported for this expression");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_ADDRESSOF_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_addressof_expression_get_inner (expr)))) {
		return;
	}
	if (!(VALA_IS_VALUE_TYPE (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))) || VALA_IS_CLASS_TYPE (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))) || VALA_IS_INTERFACE_TYPE (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))) || VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Address-of operator not supported for this expression");
		return;
	}
	if (vala_data_type_is_reference_type_or_type_parameter (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)))) {
		ValaPointerType* _tmp1;
		ValaPointerType* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp1 = vala_pointer_type_new (VALA_DATA_TYPE ((_tmp0 = vala_pointer_type_new (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)))))))));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	} else {
		ValaPointerType* _tmp2;
		_tmp2 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_DATA_TYPE ((_tmp2 = vala_pointer_type_new (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))))));
		(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaDataType* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_REFERENCE_TRANSFER_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_reference_transfer_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (!(VALA_IS_MEMBER_ACCESS (vala_reference_transfer_expression_get_inner (expr)) || VALA_IS_ELEMENT_ACCESS (vala_reference_transfer_expression_get_inner (expr)))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Reference transfer not supported for this expression");
		return;
	}
	if (!vala_data_type_get_takes_ownership (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr))) && !(VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr))))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "No reference to be transferred");
		return;
	}
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp0 = vala_data_type_copy (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr)))));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	vala_data_type_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
	vala_data_type_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE);
}


static ValaDataType* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaDataType* left_type, ValaDataType* right_type) {
	ValaStruct* _tmp1;
	ValaStruct* left;
	ValaStruct* _tmp2;
	ValaStruct* right;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (left_type == NULL || VALA_IS_DATA_TYPE (left_type), NULL);
	g_return_val_if_fail (right_type == NULL || VALA_IS_DATA_TYPE (right_type), NULL);
	if (!(VALA_IS_STRUCT (vala_data_type_get_data_type (left_type))) || !(VALA_IS_STRUCT (vala_data_type_get_data_type (right_type)))) {
		return NULL;
	}
	/* at least one operand not struct*/
	_tmp1 = NULL;
	left = (_tmp1 = VALA_STRUCT (vala_data_type_get_data_type (left_type)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
	_tmp2 = NULL;
	right = (_tmp2 = VALA_STRUCT (vala_data_type_get_data_type (right_type)), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
	if ((!vala_struct_is_floating_type (left) && !vala_struct_is_integer_type (left)) || (!vala_struct_is_floating_type (right) && !vala_struct_is_integer_type (right))) {
		gpointer _tmp3;
		return (_tmp3 = NULL, (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp3);
	}
	/* at least one operand not numeric*/
	if (vala_struct_is_floating_type (left) == vala_struct_is_floating_type (right)) {
		/* both operands integer or floating type*/
		if (vala_struct_get_rank (left) >= vala_struct_get_rank (right)) {
			ValaDataType* _tmp4;
			ValaDataType* _tmp5;
			_tmp4 = NULL;
			_tmp5 = NULL;
			return (_tmp5 = (_tmp4 = left_type, (_tmp4 == NULL ? NULL : g_object_ref (_tmp4))), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp5);
		} else {
			ValaDataType* _tmp6;
			ValaDataType* _tmp7;
			_tmp6 = NULL;
			_tmp7 = NULL;
			return (_tmp7 = (_tmp6 = right_type, (_tmp6 == NULL ? NULL : g_object_ref (_tmp6))), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp7);
		}
	} else {
		/* one integer and one floating type operand*/
		if (vala_struct_is_floating_type (left)) {
			ValaDataType* _tmp8;
			ValaDataType* _tmp9;
			_tmp8 = NULL;
			_tmp9 = NULL;
			return (_tmp9 = (_tmp8 = left_type, (_tmp8 == NULL ? NULL : g_object_ref (_tmp8))), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp9);
		} else {
			ValaDataType* _tmp10;
			ValaDataType* _tmp11;
			_tmp10 = NULL;
			_tmp11 = NULL;
			return (_tmp11 = (_tmp10 = right_type, (_tmp10 == NULL ? NULL : g_object_ref (_tmp10))), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp11);
		}
	}
	(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
	(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
}


static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_BINARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_left (expr))) || vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_right (expr)))) {
		/* if there were any errors in inner expressions, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_expression_get_static_type (vala_binary_expression_get_left (expr)) == NULL) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_binary_expression_get_left (expr))), "invalid left operand");
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_binary_expression_get_operator (expr) != VALA_BINARY_OPERATOR_IN && vala_expression_get_static_type (vala_binary_expression_get_right (expr)) == NULL) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_binary_expression_get_right (expr))), "invalid right operand");
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_data_type_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_data_type_get_data_type (self->priv->string_type) && vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS) {
		ValaMemberAccess* _tmp0;
		ValaInvocationExpression* _tmp1;
		ValaInvocationExpression* concat_call;
		if (vala_expression_get_static_type (vala_binary_expression_get_right (expr)) == NULL || vala_data_type_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_data_type_get_data_type (self->priv->string_type)) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be strings");
			return;
		}
		/* string concatenation: convert to a.concat (b) */
		_tmp0 = NULL;
		_tmp1 = NULL;
		concat_call = (_tmp1 = vala_invocation_expression_new (VALA_EXPRESSION ((_tmp0 = vala_member_access_new (vala_binary_expression_get_left (expr), "concat", NULL))), NULL), (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1);
		vala_invocation_expression_add_argument (concat_call, vala_binary_expression_get_right (expr));
		vala_code_node_replace_expression (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_EXPRESSION (expr), VALA_EXPRESSION (concat_call));
		vala_code_node_accept (VALA_CODE_NODE (concat_call), VALA_CODE_VISITOR (self));
		(concat_call == NULL ? NULL : (concat_call = (g_object_unref (concat_call), NULL)));
	} else {
		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MINUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MUL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_DIV) {
			/* check for pointer arithmetic*/
			if (VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))) {
				ValaStruct* _tmp3;
				ValaTypesymbol* _tmp2;
				ValaStruct* offset_type;
				_tmp3 = NULL;
				_tmp2 = NULL;
				offset_type = (_tmp3 = (_tmp2 = vala_data_type_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))), (VALA_IS_STRUCT (_tmp2) ? ((ValaStruct*) _tmp2) : NULL)), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
				if (offset_type != NULL && vala_struct_is_integer_type (offset_type)) {
					if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MINUS) {
						ValaDataType* _tmp4;
						/* pointer arithmetic: pointer +/- offset*/
						_tmp4 = NULL;
						vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp4 = vala_data_type_copy (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))));
						(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
					}
				} else {
					if (VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))) {
						/* pointer arithmetic: pointer - pointer*/
						vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->size_t_type);
					}
				}
				(offset_type == NULL ? NULL : (offset_type = (g_object_unref (offset_type), NULL)));
			}
			if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
				ValaDataType* _tmp5;
				_tmp5 = NULL;
				vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp5 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))));
				(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
			}
			if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
				char* _tmp8;
				char* _tmp7;
				char* _tmp6;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp8 = NULL;
				_tmp7 = NULL;
				_tmp6 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp8 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (_tmp6 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_left (expr))))), (_tmp7 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_right (expr))))))));
				_tmp8 = (g_free (_tmp8), NULL);
				_tmp7 = (g_free (_tmp7), NULL);
				_tmp6 = (g_free (_tmp6), NULL);
				return;
			}
		} else {
			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MOD || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_LEFT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_RIGHT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_XOR) {
				ValaDataType* _tmp9;
				_tmp9 = NULL;
				vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp9 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))));
				(_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL)));
				if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
					char* _tmp12;
					char* _tmp11;
					char* _tmp10;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp12 = NULL;
					_tmp11 = NULL;
					_tmp10 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp12 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (_tmp10 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_left (expr))))), (_tmp11 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_right (expr))))))));
					_tmp12 = (g_free (_tmp12), NULL);
					_tmp11 = (g_free (_tmp11), NULL);
					_tmp10 = (g_free (_tmp10), NULL);
					return;
				}
			} else {
				if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL) {
					if (vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_left (expr)), self->priv->string_type, TRUE) && vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_right (expr)), self->priv->string_type, TRUE)) {
					} else {
						/* string comparison*/
						if (VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) && VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))) {
						} else {
							ValaDataType* resulting_type;
							/* pointer arithmetic*/
							resulting_type = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)));
							if (resulting_type == NULL) {
								char* _tmp15;
								char* _tmp14;
								char* _tmp13;
								vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
								_tmp15 = NULL;
								_tmp14 = NULL;
								_tmp13 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp15 = g_strdup_printf ("Relational operation not supported for types `%s' and `%s'", (_tmp13 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_left (expr))))), (_tmp14 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_right (expr))))))));
								_tmp15 = (g_free (_tmp15), NULL);
								_tmp14 = (g_free (_tmp14), NULL);
								_tmp13 = (g_free (_tmp13), NULL);
								(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
								return;
							}
							(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
						}
					}
					vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
				} else {
					if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY) {
						/* relational operation */
						if (!vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_right (expr)), vala_expression_get_static_type (vala_binary_expression_get_left (expr)), TRUE) && !vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)), TRUE)) {
							if (!vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_right (expr)), vala_expression_get_static_type (vala_binary_expression_get_left (expr)), FALSE) && !vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)), FALSE)) {
								char* _tmp18;
								char* _tmp17;
								char* _tmp16;
								_tmp18 = NULL;
								_tmp17 = NULL;
								_tmp16 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp18 = g_strdup_printf ("Equality operation: `%s' and `%s' are incompatible", (_tmp16 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_right (expr))))), (_tmp17 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_left (expr))))))));
								_tmp18 = (g_free (_tmp18), NULL);
								_tmp17 = (g_free (_tmp17), NULL);
								_tmp16 = (g_free (_tmp16), NULL);
								vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
								return;
							} else {
								if (vala_code_context_is_non_null_enabled ()) {
									char* _tmp21;
									char* _tmp20;
									char* _tmp19;
									/* warn about incompatibility between null and non-null types*/
									_tmp21 = NULL;
									_tmp20 = NULL;
									_tmp19 = NULL;
									vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp21 = g_strdup_printf ("Equality operation: `%s' and `%s' are incompatible", (_tmp19 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_right (expr))))), (_tmp20 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_binary_expression_get_left (expr))))))));
									_tmp21 = (g_free (_tmp21), NULL);
									_tmp20 = (g_free (_tmp20), NULL);
									_tmp19 = (g_free (_tmp19), NULL);
								}
							}
						}
						if (vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_left (expr)), self->priv->string_type, TRUE) && vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_right (expr)), self->priv->string_type, TRUE)) {
						}
						/* string comparison*/
						vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
					} else {
						if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_OR) {
							/* integer type or flags type*/
							vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_binary_expression_get_left (expr)));
						} else {
							if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_OR) {
								if (!vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_left (expr)), self->priv->bool_type, TRUE) || !vala_data_type_compatible (vala_expression_get_static_type (vala_binary_expression_get_right (expr)), self->priv->bool_type, TRUE)) {
									vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be boolean");
								}
								vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
							} else {
								if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_IN) {
									/* integer type or flags type*/
									vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
								} else {
									g_assert_not_reached ();
								}
							}
						}
					}
				}
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_TYPECHECK (expr));
	if (vala_data_type_get_data_type (vala_typecheck_get_type_reference (expr)) == NULL) {
		/* if type resolving didn't succeed, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	vala_source_file_add_type_dependency (self->priv->current_source_file, vala_typecheck_get_type_reference (expr), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
}


static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (expr == NULL || VALA_IS_CONDITIONAL_EXPRESSION (expr));
	if (!vala_data_type_compatible (vala_expression_get_static_type (vala_conditional_expression_get_condition (expr)), self->priv->bool_type, TRUE)) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_conditional_expression_get_condition (expr))), "Condition must be boolean");
		return;
	}
	/* FIXME: support memory management */
	if (vala_data_type_compatible (vala_expression_get_static_type (vala_conditional_expression_get_false_expression (expr)), vala_expression_get_static_type (vala_conditional_expression_get_true_expression (expr)), TRUE)) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp0 = vala_data_type_copy (vala_expression_get_static_type (vala_conditional_expression_get_true_expression (expr)))));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	} else {
		if (vala_data_type_compatible (vala_expression_get_static_type (vala_conditional_expression_get_true_expression (expr)), vala_expression_get_static_type (vala_conditional_expression_get_false_expression (expr)), TRUE)) {
			ValaDataType* _tmp1;
			_tmp1 = NULL;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp1 = vala_data_type_copy (vala_expression_get_static_type (vala_conditional_expression_get_false_expression (expr)))));
			(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		} else {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_conditional_expression_get_condition (expr))), "Incompatible expressions");
			return;
		}
	}
}


static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self) {
	char* result;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	result = g_strdup_printf ("__lambda%d", self->priv->next_lambda_id);
	self->priv->next_lambda_id++;
	return result;
}


static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self) {
	ValaSymbol* _tmp0;
	ValaSymbol* sym;
	gpointer _tmp5;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	_tmp0 = NULL;
	sym = (_tmp0 = self->priv->current_symbol, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	while (sym != NULL) {
		ValaSymbol* _tmp4;
		ValaSymbol* _tmp3;
		if (VALA_IS_METHOD (sym)) {
			ValaMethod* _tmp1;
			ValaMethod* _tmp2;
			_tmp1 = NULL;
			_tmp2 = NULL;
			return (_tmp2 = (_tmp1 = VALA_METHOD (sym), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp2);
		}
		_tmp4 = NULL;
		_tmp3 = NULL;
		sym = (_tmp4 = (_tmp3 = vala_symbol_get_parent_symbol (sym), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp4);
	}
	return (_tmp5 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp5);
}


static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self) {
	ValaSymbol* _tmp0;
	ValaSymbol* sym;
	gboolean _tmp4;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	_tmp0 = NULL;
	sym = (_tmp0 = self->priv->current_symbol, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	while (sym != NULL) {
		ValaSymbol* _tmp3;
		ValaSymbol* _tmp2;
		if (VALA_IS_CONSTRUCTOR (sym)) {
			gboolean _tmp1;
			return (_tmp1 = TRUE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp1);
		}
		_tmp3 = NULL;
		_tmp2 = NULL;
		sym = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (sym), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp3);
	}
	return (_tmp4 = FALSE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp4);
}


static void vala_semantic_analyzer_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l) {
	ValaSemanticAnalyzer * self;
	gboolean in_instance_method;
	ValaMethod* current_method;
	ValaDelegate* _tmp0;
	ValaDelegate* cb;
	ValaMethod* _tmp2;
	char* _tmp1;
	GeeCollection* lambda_params;
	GeeIterator* lambda_param_it;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (l == NULL || VALA_IS_LAMBDA_EXPRESSION (l));
	if (!(VALA_IS_DELEGATE_TYPE (vala_expression_get_expected_type (VALA_EXPRESSION (l))))) {
		vala_code_node_set_error (VALA_CODE_NODE (l), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression not allowed in this context");
		return;
	}
	in_instance_method = FALSE;
	current_method = vala_semantic_analyzer_find_current_method (self);
	if (current_method != NULL) {
		in_instance_method = vala_method_get_instance (current_method);
	} else {
		in_instance_method = vala_semantic_analyzer_is_in_constructor (self);
	}
	_tmp0 = NULL;
	cb = (_tmp0 = VALA_DELEGATE (vala_delegate_type_get_delegate_symbol ((VALA_DELEGATE_TYPE (vala_expression_get_expected_type (VALA_EXPRESSION (l)))))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	_tmp2 = NULL;
	_tmp1 = NULL;
	vala_lambda_expression_set_method (l, (_tmp2 = vala_method_new ((_tmp1 = vala_semantic_analyzer_get_lambda_name (self)), vala_delegate_get_return_type (cb), NULL)));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	_tmp1 = (g_free (_tmp1), NULL);
	vala_method_set_instance (vala_lambda_expression_get_method (l), vala_delegate_get_instance (cb) && in_instance_method);
	vala_symbol_set_owner (VALA_SYMBOL (vala_lambda_expression_get_method (l)), vala_symbol_get_scope (self->priv->current_symbol));
	lambda_params = vala_lambda_expression_get_parameters (l);
	lambda_param_it = gee_iterable_iterator (GEE_ITERABLE (lambda_params));
	{
		GeeList* cb_param_collection;
		GeeIterator* cb_param_it;
		cb_param_collection = vala_delegate_get_parameters (cb);
		cb_param_it = gee_iterable_iterator (GEE_ITERABLE (cb_param_collection));
		while (gee_iterator_next (cb_param_it)) {
			ValaFormalParameter* cb_param;
			cb_param = ((ValaFormalParameter*) gee_iterator_get (cb_param_it));
			{
				char* lambda_param;
				ValaFormalParameter* param;
				if (!gee_iterator_next (lambda_param_it)) {
					(cb_param == NULL ? NULL : (cb_param = (g_object_unref (cb_param), NULL)));
					break;
				}
				/* lambda expressions are allowed to have less parameters */
				lambda_param = ((char*) gee_iterator_get (lambda_param_it));
				param = vala_formal_parameter_new (lambda_param, vala_formal_parameter_get_type_reference (cb_param), NULL);
				vala_method_add_parameter (vala_lambda_expression_get_method (l), param);
				(cb_param == NULL ? NULL : (cb_param = (g_object_unref (cb_param), NULL)));
				lambda_param = (g_free (lambda_param), NULL);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(cb_param_collection == NULL ? NULL : (cb_param_collection = (g_object_unref (cb_param_collection), NULL)));
		(cb_param_it == NULL ? NULL : (cb_param_it = (g_object_unref (cb_param_it), NULL)));
	}
	if (gee_iterator_next (lambda_param_it)) {
		/* lambda expressions may not expect more parameters */
		vala_code_node_set_error (VALA_CODE_NODE (l), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression: too many parameters");
		(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
		(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
		(lambda_params == NULL ? NULL : (lambda_params = (g_object_unref (lambda_params), NULL)));
		(lambda_param_it == NULL ? NULL : (lambda_param_it = (g_object_unref (lambda_param_it), NULL)));
		return;
	}
	if (vala_lambda_expression_get_expression_body (l) != NULL) {
		ValaBlock* block;
		block = vala_block_new (NULL);
		vala_scope_set_parent_scope (vala_symbol_get_scope (VALA_SYMBOL (block)), vala_symbol_get_scope (VALA_SYMBOL (vala_lambda_expression_get_method (l))));
		if (vala_data_type_get_data_type (vala_method_get_return_type (vala_lambda_expression_get_method (l))) != NULL) {
			ValaReturnStatement* _tmp3;
			_tmp3 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((_tmp3 = vala_return_statement_new (vala_lambda_expression_get_expression_body (l), vala_code_node_get_source_reference (VALA_CODE_NODE (l))))));
			(_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)));
		} else {
			ValaExpressionStatement* _tmp4;
			_tmp4 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((_tmp4 = vala_expression_statement_new (vala_lambda_expression_get_expression_body (l), vala_code_node_get_source_reference (VALA_CODE_NODE (l))))));
			(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
		}
		vala_method_set_body (vala_lambda_expression_get_method (l), block);
		(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
	} else {
		vala_method_set_body (vala_lambda_expression_get_method (l), vala_lambda_expression_get_statement_body (l));
	}
	vala_symbol_set_owner (VALA_SYMBOL (vala_method_get_body (vala_lambda_expression_get_method (l))), vala_symbol_get_scope (VALA_SYMBOL (vala_lambda_expression_get_method (l))));
	/* lambda expressions should be usable like MemberAccess of a method */
	vala_expression_set_symbol_reference (VALA_EXPRESSION (l), VALA_SYMBOL (vala_lambda_expression_get_method (l)));
	vala_code_node_accept_children (VALA_CODE_NODE (l), VALA_CODE_VISITOR (self));
	(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
	(lambda_params == NULL ? NULL : (lambda_params = (g_object_unref (lambda_params), NULL)));
	(lambda_param_it == NULL ? NULL : (lambda_param_it = (g_object_unref (lambda_param_it), NULL)));
}


static void vala_semantic_analyzer_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (a == NULL || VALA_IS_ASSIGNMENT (a));
	vala_expression_set_lvalue (vala_assignment_get_left (a), TRUE);
	vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_left (a)), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_left (a)))) {
		/* skip on error in inner expression*/
		vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
		return;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* _tmp0;
		ValaMemberAccess* ma;
		_tmp0 = NULL;
		ma = (_tmp0 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (!(VALA_IS_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) && vala_expression_get_static_type (VALA_EXPRESSION (ma)) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "unsupported lvalue in assignment");
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp2;
			char* _tmp1;
			vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
			_tmp2 = NULL;
			_tmp1 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp2 = g_strdup_printf ("Access to instance member `%s' denied", (_tmp1 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))))));
			_tmp2 = (g_free (_tmp2), NULL);
			_tmp1 = (g_free (_tmp1), NULL);
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		if (vala_code_node_get_error (VALA_CODE_NODE (ma)) || vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		/* if no symbol found, skip this check */
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) {
			ValaSignal* _tmp3;
			ValaSignal* sig;
			ValaDelegateType* _tmp5;
			ValaDelegate* _tmp4;
			_tmp3 = NULL;
			sig = (_tmp3 = VALA_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
			_tmp5 = NULL;
			_tmp4 = NULL;
			vala_expression_set_expected_type (vala_assignment_get_right (a), VALA_DATA_TYPE ((_tmp5 = vala_delegate_type_new ((_tmp4 = vala_signal_get_delegate (sig))))));
			(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
			(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
		} else {
			vala_expression_set_expected_type (vala_assignment_get_right (a), vala_expression_get_static_type (VALA_EXPRESSION (ma)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
		} else {
			/* do nothing*/
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a))) {
			} else {
				/* do nothing*/
				vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "unsupported lvalue in assignment");
				return;
			}
		}
	}
	vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (a)), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_right (a)))) {
		/* skip on error in inner expression*/
		vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
		return;
	}
	if (vala_assignment_get_operator (a) != VALA_ASSIGNMENT_OPERATOR_SIMPLE && VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* _tmp6;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		_tmp6 = NULL;
		ma = (_tmp6 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
		if (!(VALA_IS_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
			ValaMemberAccess* old_value;
			ValaParenthesizedExpression* _tmp7;
			ValaBinaryExpression* _tmp8;
			ValaBinaryExpression* bin;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			_tmp7 = NULL;
			_tmp8 = NULL;
			bin = (_tmp8 = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, VALA_EXPRESSION (old_value), VALA_EXPRESSION ((_tmp7 = vala_parenthesized_expression_new (vala_assignment_get_right (a), vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a)))))), NULL), (_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL))), _tmp8);
			if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR);
			} else {
				if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND);
				} else {
					if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR);
					} else {
						if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_ADD) {
							vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS);
						} else {
							if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SUB) {
								vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS);
							} else {
								if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_MUL) {
									vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL);
								} else {
									if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_DIV) {
										vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV);
									} else {
										if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD);
										} else {
											if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT);
											} else {
												if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			vala_assignment_set_right (a, VALA_EXPRESSION (bin));
			vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (a)), VALA_CODE_VISITOR (self));
			vala_assignment_set_operator (a, VALA_ASSIGNMENT_OPERATOR_SIMPLE);
			(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
			(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* _tmp9;
		ValaMemberAccess* ma;
		_tmp9 = NULL;
		ma = (_tmp9 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9)));
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) {
			ValaSignal* _tmp10;
			ValaSignal* sig;
			_tmp10 = NULL;
			sig = (_tmp10 = VALA_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
			if (vala_expression_get_symbol_reference (vala_assignment_get_right (a)) == NULL) {
				vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "unsupported expression for signal handler");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
		} else {
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) {
				ValaProperty* _tmp11;
				ValaProperty* prop;
				_tmp11 = NULL;
				prop = (_tmp11 = VALA_PROPERTY (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp11 == NULL ? NULL : g_object_ref (_tmp11)));
				if (vala_property_get_set_accessor (prop) == NULL) {
					char* _tmp13;
					char* _tmp12;
					vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE);
					_tmp13 = NULL;
					_tmp12 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ma)), (_tmp13 = g_strdup_printf ("Property `%s' is read-only", (_tmp12 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
					_tmp13 = (g_free (_tmp13), NULL);
					_tmp12 = (g_free (_tmp12), NULL);
					(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					return;
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			} else {
				if (VALA_IS_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
					ValaVariableDeclarator* _tmp14;
					ValaVariableDeclarator* decl;
					_tmp14 = NULL;
					decl = (_tmp14 = VALA_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp14 == NULL ? NULL : g_object_ref (_tmp14)));
					if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (a))) && VALA_IS_DELEGATE_TYPE (vala_variable_declarator_get_type_reference (decl))) {
						ValaMethod* _tmp15;
						ValaMethod* m;
						ValaDelegateType* _tmp16;
						ValaDelegateType* dt;
						ValaDelegate* _tmp17;
						ValaDelegate* cb;
						_tmp15 = NULL;
						m = (_tmp15 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (a))), (_tmp15 == NULL ? NULL : g_object_ref (_tmp15)));
						_tmp16 = NULL;
						dt = (_tmp16 = VALA_DELEGATE_TYPE (vala_variable_declarator_get_type_reference (decl)), (_tmp16 == NULL ? NULL : g_object_ref (_tmp16)));
						_tmp17 = NULL;
						cb = (_tmp17 = vala_delegate_type_get_delegate_symbol (dt), (_tmp17 == NULL ? NULL : g_object_ref (_tmp17)));
						/* check whether method matches callback type */
						if (!vala_delegate_matches_method (cb, m)) {
							char* _tmp20;
							char* _tmp19;
							char* _tmp18;
							vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
							_tmp20 = NULL;
							_tmp19 = NULL;
							_tmp18 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp20 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (_tmp18 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp19 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
							_tmp20 = (g_free (_tmp20), NULL);
							_tmp19 = (g_free (_tmp19), NULL);
							_tmp18 = (g_free (_tmp18), NULL);
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						vala_expression_set_static_type (vala_assignment_get_right (a), vala_variable_declarator_get_type_reference (decl));
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
						(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					} else {
						vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
						(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
				} else {
					if (VALA_IS_FIELD (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
						ValaField* _tmp21;
						ValaField* f;
						_tmp21 = NULL;
						f = (_tmp21 = VALA_FIELD (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp21 == NULL ? NULL : g_object_ref (_tmp21)));
						if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (a))) && VALA_IS_DELEGATE_TYPE (vala_field_get_type_reference (f))) {
							ValaMethod* _tmp22;
							ValaMethod* m;
							ValaDelegateType* _tmp23;
							ValaDelegateType* dt;
							ValaDelegate* _tmp24;
							ValaDelegate* cb;
							_tmp22 = NULL;
							m = (_tmp22 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (a))), (_tmp22 == NULL ? NULL : g_object_ref (_tmp22)));
							_tmp23 = NULL;
							dt = (_tmp23 = VALA_DELEGATE_TYPE (vala_field_get_type_reference (f)), (_tmp23 == NULL ? NULL : g_object_ref (_tmp23)));
							_tmp24 = NULL;
							cb = (_tmp24 = vala_delegate_type_get_delegate_symbol (dt), (_tmp24 == NULL ? NULL : g_object_ref (_tmp24)));
							/* check whether method matches callback type */
							if (!vala_delegate_matches_method (cb, m)) {
								char* _tmp27;
								char* _tmp26;
								char* _tmp25;
								vala_code_node_set_error (VALA_CODE_NODE (f), TRUE);
								_tmp27 = NULL;
								_tmp26 = NULL;
								_tmp25 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp27 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (_tmp25 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp26 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
								_tmp27 = (g_free (_tmp27), NULL);
								_tmp26 = (g_free (_tmp26), NULL);
								_tmp25 = (g_free (_tmp25), NULL);
								(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
								(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
								(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							vala_expression_set_static_type (vala_assignment_get_right (a), vala_field_get_type_reference (f));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						} else {
							vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					} else {
						if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL && vala_expression_get_static_type (vala_assignment_get_right (a)) != NULL) {
							/* if there was an error on either side,
							 * i.e. a.{left|right}.static_type == null, skip type check */
							if (!vala_data_type_compatible (vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)), TRUE)) {
								if (!vala_data_type_compatible (vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)), FALSE)) {
									char* _tmp30;
									char* _tmp29;
									char* _tmp28;
									vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
									_tmp30 = NULL;
									_tmp29 = NULL;
									_tmp28 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp30 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (_tmp28 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_assignment_get_right (a))))), (_tmp29 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_assignment_get_left (a))))))));
									_tmp30 = (g_free (_tmp30), NULL);
									_tmp29 = (g_free (_tmp29), NULL);
									_tmp28 = (g_free (_tmp28), NULL);
									(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
									return;
								} else {
									if (vala_code_context_is_non_null_enabled ()) {
										char* _tmp33;
										char* _tmp32;
										char* _tmp31;
										/* warn about incompatibility between null and non-null types*/
										_tmp33 = NULL;
										_tmp32 = NULL;
										_tmp31 = NULL;
										vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp33 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (_tmp31 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_assignment_get_right (a))))), (_tmp32 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_assignment_get_left (a))))))));
										_tmp33 = (g_free (_tmp33), NULL);
										_tmp32 = (g_free (_tmp32), NULL);
										_tmp31 = (g_free (_tmp31), NULL);
									}
								}
							}
							if (vala_data_type_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
								/* rhs transfers ownership of the expression */
								if (!(VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_assignment_get_left (a)))) && !vala_data_type_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
									/* lhs doesn't own the value */
									vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
								}
							} else {
								if (vala_data_type_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
								}
							}
						}
					}
				}
			}
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		/* lhs wants to own the value
		 * rhs doesn't transfer the ownership
		 * code generator needs to add reference
		 * increment calls */
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
			ValaElementAccess* _tmp34;
			ValaElementAccess* ea;
			_tmp34 = NULL;
			ea = (_tmp34 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (a)), (_tmp34 == NULL ? NULL : g_object_ref (_tmp34)));
			if (!vala_data_type_compatible (vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)), TRUE)) {
				if (!vala_data_type_compatible (vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)), FALSE)) {
					char* _tmp37;
					char* _tmp36;
					char* _tmp35;
					vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
					_tmp37 = NULL;
					_tmp36 = NULL;
					_tmp35 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp37 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (_tmp35 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_assignment_get_right (a))))), (_tmp36 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_assignment_get_left (a))))))));
					_tmp37 = (g_free (_tmp37), NULL);
					_tmp36 = (g_free (_tmp36), NULL);
					_tmp35 = (g_free (_tmp35), NULL);
					(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
					return;
				} else {
					if (vala_code_context_is_non_null_enabled ()) {
						char* _tmp40;
						char* _tmp39;
						char* _tmp38;
						/* warn about incompatibility between null and non-null types*/
						_tmp40 = NULL;
						_tmp39 = NULL;
						_tmp38 = NULL;
						vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp40 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (_tmp38 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_assignment_get_right (a))))), (_tmp39 = vala_code_node_to_string (VALA_CODE_NODE (vala_expression_get_static_type (vala_assignment_get_left (a))))))));
						_tmp40 = (g_free (_tmp40), NULL);
						_tmp39 = (g_free (_tmp39), NULL);
						_tmp38 = (g_free (_tmp38), NULL);
					}
				}
			}
			if (vala_data_type_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
				GeeList* args;
				ValaDataType* element_type;
				/* rhs transfers ownership of the expression */
				args = vala_data_type_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (ea)));
				if (gee_collection_get_size (GEE_COLLECTION (args)) != 1) {
					char* _tmp41;
					vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
					_tmp41 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ea)), (_tmp41 = g_strdup_printf ("internal error: array reference with %d type arguments", gee_collection_get_size (GEE_COLLECTION (args)))));
					_tmp41 = (g_free (_tmp41), NULL);
					(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
					(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
					return;
				}
				element_type = ((ValaDataType*) gee_list_get (args, 0));
				if (!(VALA_IS_POINTER_TYPE (element_type)) && !vala_data_type_get_takes_ownership (element_type)) {
					/* lhs doesn't own the value */
					vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
					(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
					(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
					(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
					return;
				}
				(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
				(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
			} else {
				if (vala_data_type_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
				}
			}
			(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
		} else {
			return;
		}
	}
	/* lhs wants to own the value
	 * rhs doesn't transfer the ownership
	 * code generator needs to add reference
	 * increment calls */
	if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL) {
		ValaDataType* _tmp42;
		_tmp42 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (a), (_tmp42 = vala_data_type_copy (vala_expression_get_static_type (vala_assignment_get_left (a)))));
		(_tmp42 == NULL ? NULL : (_tmp42 = (g_object_unref (_tmp42), NULL)));
		if (VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (VALA_CODE_NODE (a)))) {
			/* Gee.List.get () transfers ownership but void function Gee.List.set () doesn't*/
			vala_data_type_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (a)), FALSE);
		}
	} else {
		vala_expression_set_static_type (VALA_EXPRESSION (a), NULL);
	}
	vala_code_node_set_tree_can_fail (VALA_CODE_NODE (a), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_assignment_get_left (a))) || vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_assignment_get_right (a))));
}


static void vala_semantic_analyzer_class_init (ValaSemanticAnalyzerClass * klass) {
	vala_semantic_analyzer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSemanticAnalyzerPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_semantic_analyzer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_semantic_analyzer_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_semantic_analyzer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_semantic_analyzer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_semantic_analyzer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_semantic_analyzer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_semantic_analyzer_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_semantic_analyzer_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_semantic_analyzer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_semantic_analyzer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_semantic_analyzer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_semantic_analyzer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_semantic_analyzer_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_semantic_analyzer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_semantic_analyzer_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_semantic_analyzer_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_semantic_analyzer_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_semantic_analyzer_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_semantic_analyzer_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_semantic_analyzer_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_semantic_analyzer_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_initializer_list = vala_semantic_analyzer_real_visit_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_semantic_analyzer_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_semantic_analyzer_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_semantic_analyzer_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_semantic_analyzer_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_semantic_analyzer_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_semantic_analyzer_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_semantic_analyzer_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_semantic_analyzer_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_semantic_analyzer_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_semantic_analyzer_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_semantic_analyzer_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_semantic_analyzer_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delete_statement = vala_semantic_analyzer_real_visit_delete_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_array_creation_expression = vala_semantic_analyzer_real_visit_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_semantic_analyzer_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_semantic_analyzer_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_semantic_analyzer_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_semantic_analyzer_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_semantic_analyzer_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_semantic_analyzer_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_semantic_analyzer_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_semantic_analyzer_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_invocation_expression = vala_semantic_analyzer_real_visit_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_semantic_analyzer_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_semantic_analyzer_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_semantic_analyzer_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_object_creation_expression = vala_semantic_analyzer_real_visit_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_sizeof_expression = vala_semantic_analyzer_real_visit_sizeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_semantic_analyzer_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_semantic_analyzer_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_semantic_analyzer_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_semantic_analyzer_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_semantic_analyzer_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_semantic_analyzer_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_semantic_analyzer_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_semantic_analyzer_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_semantic_analyzer_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lambda_expression = vala_semantic_analyzer_real_visit_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_assignment = vala_semantic_analyzer_real_visit_assignment;
}


static void vala_semantic_analyzer_init (ValaSemanticAnalyzer * self) {
	self->priv = VALA_SEMANTIC_ANALYZER_GET_PRIVATE (self);
	self->priv->next_lambda_id = 0;
	self->priv->binding_providers = GEE_COLLECTION (gee_array_list_new (VALA_TYPE_BINDING_PROVIDER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_semantic_analyzer_dispose (GObject * obj) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL)));
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL)));
	(self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL)));
	(self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL)));
	(self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL)));
	(self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL)));
	(self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL)));
	(self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL)));
	(self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL)));
	(self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL)));
	(self->priv->size_t_type == NULL ? NULL : (self->priv->size_t_type = (g_object_unref (self->priv->size_t_type), NULL)));
	(self->priv->unichar_type == NULL ? NULL : (self->priv->unichar_type = (g_object_unref (self->priv->unichar_type), NULL)));
	(self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL)));
	(self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL)));
	(self->priv->object_type == NULL ? NULL : (self->priv->object_type = (g_object_unref (self->priv->object_type), NULL)));
	(self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL)));
	(self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL)));
	(self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL)));
	(self->priv->gerror_type == NULL ? NULL : (self->priv->gerror_type = (g_object_unref (self->priv->gerror_type), NULL)));
	(self->priv->iterable_type == NULL ? NULL : (self->priv->iterable_type = (g_object_unref (self->priv->iterable_type), NULL)));
	(self->priv->iterator_type == NULL ? NULL : (self->priv->iterator_type = (g_object_unref (self->priv->iterator_type), NULL)));
	(self->priv->list_type == NULL ? NULL : (self->priv->list_type = (g_object_unref (self->priv->list_type), NULL)));
	(self->priv->map_type == NULL ? NULL : (self->priv->map_type = (g_object_unref (self->priv->map_type), NULL)));
	(self->priv->binding_providers == NULL ? NULL : (self->priv->binding_providers = (g_object_unref (self->priv->binding_providers), NULL)));
	G_OBJECT_CLASS (vala_semantic_analyzer_parent_class)->dispose (obj);
}


GType vala_semantic_analyzer_get_type (void) {
	static GType vala_semantic_analyzer_type_id = 0;
	if (G_UNLIKELY (vala_semantic_analyzer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSemanticAnalyzerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_semantic_analyzer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSemanticAnalyzer), 0, (GInstanceInitFunc) vala_semantic_analyzer_init };
		vala_semantic_analyzer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaSemanticAnalyzer", &g_define_type_info, 0);
	}
	return vala_semantic_analyzer_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




