/* valamemorymanager.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valamemorymanager.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valasymbol.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valanulltype.h>
#include <vala/valapointertype.h>
#include <vala/valaformalparameter.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valacodenode.h>
#include <vala/valapointerindirection.h>
#include <vala/valasignal.h>




struct _ValaMemoryManagerPrivate {
	ValaSymbol* current_symbol;
};
#define VALA_MEMORY_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MEMORY_MANAGER, ValaMemoryManagerPrivate))
enum  {
	VALA_MEMORY_MANAGER_DUMMY_PROPERTY
};
static void vala_memory_manager_visit_possibly_leaked_expression (ValaMemoryManager* self, ValaExpression* expr);
static void vala_memory_manager_visit_possibly_missing_copy_expression (ValaMemoryManager* self, ValaExpression* expr);
static void vala_memory_manager_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_memory_manager_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_memory_manager_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_memory_manager_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_memory_manager_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_memory_manager_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_memory_manager_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_memory_manager_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_memory_manager_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_memory_manager_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_memory_manager_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_memory_manager_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_memory_manager_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_memory_manager_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_memory_manager_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_memory_manager_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_memory_manager_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_memory_manager_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section);
static void vala_memory_manager_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_memory_manager_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt);
static void vala_memory_manager_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_memory_manager_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_memory_manager_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_memory_manager_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_memory_manager_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_memory_manager_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_memory_manager_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* e);
static void vala_memory_manager_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static void vala_memory_manager_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static void vala_memory_manager_real_visit_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static void vala_memory_manager_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_memory_manager_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_memory_manager_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_memory_manager_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_memory_manager_parent_class = NULL;
static void vala_memory_manager_dispose (GObject * obj);



/**
 * Analyze memory usage in the specified code context.
 *
 * @param context a code context
 */
void vala_memory_manager_analyze (ValaMemoryManager* self, ValaCodeContext* context) {
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (context == NULL || VALA_IS_CODE_CONTEXT (context));
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_visit_possibly_leaked_expression (ValaMemoryManager* self, ValaExpression* expr) {
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (expr == NULL || VALA_IS_EXPRESSION (expr));
	if (vala_expression_get_static_type (expr) != NULL && vala_data_type_get_transfers_ownership (vala_expression_get_static_type (expr))) {
		/* mark reference as leaked */
		vala_expression_set_ref_leaked (expr, TRUE);
	}
}


static void vala_memory_manager_visit_possibly_missing_copy_expression (ValaMemoryManager* self, ValaExpression* expr) {
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (expr == NULL || VALA_IS_EXPRESSION (expr));
	if (vala_expression_get_static_type (expr) != NULL && !vala_data_type_get_transfers_ownership (vala_expression_get_static_type (expr)) && !(VALA_IS_NULL_TYPE (vala_expression_get_static_type (expr)))) {
		/* mark reference as missing */
		vala_expression_set_ref_missing (expr, TRUE);
	}
}


static void vala_memory_manager_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (source_file == NULL || VALA_IS_SOURCE_FILE (source_file));
	if (!vala_source_file_get_pkg (source_file)) {
		vala_source_file_accept_children (source_file, VALA_CODE_VISITOR (self));
	}
}


static void vala_memory_manager_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (cl == NULL || VALA_IS_CLASS (cl));
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (st == NULL || VALA_IS_STRUCT (st));
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (iface == NULL || VALA_IS_INTERFACE (iface));
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (f == NULL || VALA_IS_FIELD (f));
	if (vala_field_get_initializer (f) != NULL) {
		if (!(VALA_IS_POINTER_TYPE (vala_field_get_type_reference (f)))) {
			if (vala_data_type_get_takes_ownership (vala_field_get_type_reference (f))) {
				vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_field_get_initializer (f));
			} else {
				vala_memory_manager_visit_possibly_leaked_expression (self, vala_field_get_initializer (f));
			}
		}
	}
}


static void vala_memory_manager_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaMemoryManager * self;
	ValaSymbol* _tmp0;
	ValaSymbol* old_symbol;
	ValaSymbol* _tmp2;
	ValaMethod* _tmp1;
	ValaSymbol* _tmp4;
	ValaSymbol* _tmp3;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (m == NULL || VALA_IS_METHOD (m));
	_tmp0 = NULL;
	old_symbol = (_tmp0 = self->priv->current_symbol, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_symbol = (_tmp2 = VALA_SYMBOL ((_tmp1 = m, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp2);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_symbol = (_tmp4 = (_tmp3 = old_symbol, (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp4);
	(old_symbol == NULL ? NULL : (old_symbol = (g_object_unref (old_symbol), NULL)));
}


static void vala_memory_manager_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (m == NULL || VALA_IS_CREATION_METHOD (m));
	vala_code_visitor_visit_method (VALA_CODE_VISITOR (self), VALA_METHOD (m));
}


static void vala_memory_manager_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaMemoryManager * self;
	ValaSymbol* _tmp1;
	ValaProperty* _tmp0;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (prop == NULL || VALA_IS_PROPERTY (prop));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL ((_tmp0 = prop, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (prop), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (acc == NULL || VALA_IS_PROPERTY_ACCESSOR (acc));
	vala_code_node_accept_children (VALA_CODE_NODE (acc), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (c == NULL || VALA_IS_CONSTRUCTOR (c));
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (d == NULL || VALA_IS_DESTRUCTOR (d));
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (n == NULL || VALA_IS_NAMED_ARGUMENT (n));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_named_argument_get_argument (n));
}


static void vala_memory_manager_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (b == NULL || VALA_IS_BLOCK (b));
	vala_code_node_accept_children (VALA_CODE_NODE (b), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (decl == NULL || VALA_IS_VARIABLE_DECLARATOR (decl));
	vala_code_node_accept_children (VALA_CODE_NODE (decl), VALA_CODE_VISITOR (self));
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (!(VALA_IS_POINTER_TYPE (vala_variable_declarator_get_type_reference (decl)))) {
			if (vala_data_type_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
				vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_variable_declarator_get_initializer (decl));
			} else {
				vala_memory_manager_visit_possibly_leaked_expression (self, vala_variable_declarator_get_initializer (decl));
			}
		}
	}
}


static void vala_memory_manager_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (list == NULL || VALA_IS_INITIALIZER_LIST (list));
	vala_code_node_accept_children (VALA_CODE_NODE (list), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_EXPRESSION_STATEMENT (stmt));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_expression_statement_get_expression (stmt));
}


static void vala_memory_manager_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_IF_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (section == NULL || VALA_IS_SWITCH_SECTION (section));
	vala_code_node_accept_children (VALA_CODE_NODE (section), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_WHILE_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_DO_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_FOR_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_FOREACH_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_RETURN_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	if (vala_return_statement_get_return_expression (stmt) != NULL) {
		if (VALA_IS_METHOD (self->priv->current_symbol)) {
			ValaMethod* _tmp0;
			ValaMethod* m;
			_tmp0 = NULL;
			m = (_tmp0 = VALA_METHOD (self->priv->current_symbol), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			if (!(VALA_IS_POINTER_TYPE (vala_method_get_return_type (m)))) {
				if (vala_data_type_get_transfers_ownership (vala_method_get_return_type (m))) {
					vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_return_statement_get_return_expression (stmt));
				} else {
					vala_memory_manager_visit_possibly_leaked_expression (self, vala_return_statement_get_return_expression (stmt));
				}
			}
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			if (VALA_IS_PROPERTY (self->priv->current_symbol)) {
				ValaProperty* _tmp1;
				ValaProperty* prop;
				/* property get accessor */
				_tmp1 = NULL;
				prop = (_tmp1 = VALA_PROPERTY (self->priv->current_symbol), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
				if (vala_data_type_get_transfers_ownership (vala_property_get_type_reference (prop))) {
					vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_return_statement_get_return_expression (stmt));
				} else {
					vala_memory_manager_visit_possibly_leaked_expression (self, vala_return_statement_get_return_expression (stmt));
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			} else {
				g_assert_not_reached ();
			}
		}
	}
}


static void vala_memory_manager_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_THROW_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (stmt == NULL || VALA_IS_TRY_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (clause == NULL || VALA_IS_CATCH_CLAUSE (clause));
	vala_code_node_accept_children (VALA_CODE_NODE (clause), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* e) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (e == NULL || VALA_IS_ARRAY_CREATION_EXPRESSION (e));
	if (vala_array_creation_expression_get_initializer_list (e) != NULL) {
		{
			GeeCollection* init_collection;
			GeeIterator* init_it;
			init_collection = vala_initializer_list_get_initializers (vala_array_creation_expression_get_initializer_list (e));
			init_it = gee_iterable_iterator (GEE_ITERABLE (init_collection));
			while (gee_iterator_next (init_it)) {
				ValaExpression* init;
				init = ((ValaExpression*) gee_iterator_get (init_it));
				{
					if (vala_data_type_is_reference_type_or_type_parameter (vala_expression_get_static_type (init))) {
						vala_memory_manager_visit_possibly_missing_copy_expression (self, init);
					} else {
						vala_memory_manager_visit_possibly_leaked_expression (self, init);
					}
					(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
				}
			}
			(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
			(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
		}
	}
}


static void vala_memory_manager_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (expr == NULL || VALA_IS_PARENTHESIZED_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (expr == NULL || VALA_IS_MEMBER_ACCESS (expr));
	if (vala_member_access_get_inner (expr) != NULL) {
		vala_memory_manager_visit_possibly_leaked_expression (self, vala_member_access_get_inner (expr));
	}
}


static void vala_memory_manager_real_visit_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr) {
	ValaMemoryManager * self;
	ValaDataType* _tmp0;
	ValaDataType* mtype;
	GeeCollection* params;
	GeeIterator* params_it;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (expr == NULL || VALA_IS_INVOCATION_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
	_tmp0 = NULL;
	mtype = (_tmp0 = vala_expression_get_static_type (vala_invocation_expression_get_call (expr)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	params = vala_data_type_get_parameters (mtype);
	params_it = gee_iterable_iterator (GEE_ITERABLE (params));
	{
		GeeCollection* arg_collection;
		GeeIterator* arg_it;
		arg_collection = vala_invocation_expression_get_argument_list (expr);
		arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = ((ValaExpression*) gee_iterator_get (arg_it));
			{
				if (gee_iterator_next (params_it)) {
					ValaFormalParameter* param;
					param = ((ValaFormalParameter*) gee_iterator_get (params_it));
					if (!vala_formal_parameter_get_ellipsis (param) && vala_data_type_is_reference_type_or_type_parameter (vala_formal_parameter_get_type_reference (param))) {
						gboolean is_ref;
						is_ref = vala_data_type_get_transfers_ownership (vala_formal_parameter_get_type_reference (param));
						if (is_ref && vala_data_type_get_type_parameter (vala_formal_parameter_get_type_reference (param)) != NULL) {
							if (VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr))) {
								ValaMemberAccess* _tmp1;
								ValaMemberAccess* ma;
								ValaDataType* param_type;
								_tmp1 = NULL;
								ma = (_tmp1 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
								param_type = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_member_access_get_inner (ma)), vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)), vala_formal_parameter_get_type_reference (param), VALA_CODE_NODE (expr));
								if (param_type != NULL) {
									is_ref = vala_data_type_get_takes_ownership (param_type);
								}
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								(param_type == NULL ? NULL : (param_type = (g_object_unref (param_type), NULL)));
							}
						}
						if (is_ref) {
							vala_memory_manager_visit_possibly_missing_copy_expression (self, arg);
						} else {
							vala_memory_manager_visit_possibly_leaked_expression (self, arg);
						}
					} else {
						vala_memory_manager_visit_possibly_leaked_expression (self, arg);
					}
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				} else {
					vala_memory_manager_visit_possibly_leaked_expression (self, arg);
				}
				(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
			}
		}
		(arg_collection == NULL ? NULL : (arg_collection = (g_object_unref (arg_collection), NULL)));
		(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	}
	(mtype == NULL ? NULL : (mtype = (g_object_unref (mtype), NULL)));
	(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
	(params_it == NULL ? NULL : (params_it = (g_object_unref (params_it), NULL)));
}


static void vala_memory_manager_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr) {
	ValaMemoryManager * self;
	ValaMethod* _tmp0;
	ValaMethod* msym;
	GeeCollection* params;
	GeeIterator* params_it;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (expr == NULL || VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
	if (!(VALA_IS_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr))))) {
		return;
	}
	_tmp0 = NULL;
	msym = (_tmp0 = VALA_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	params = vala_method_get_parameters (msym);
	params_it = gee_iterable_iterator (GEE_ITERABLE (params));
	{
		GeeCollection* arg_collection;
		GeeIterator* arg_it;
		arg_collection = vala_object_creation_expression_get_argument_list (expr);
		arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = ((ValaExpression*) gee_iterator_get (arg_it));
			{
				if (gee_iterator_next (params_it)) {
					ValaFormalParameter* param;
					param = ((ValaFormalParameter*) gee_iterator_get (params_it));
					if (!vala_formal_parameter_get_ellipsis (param) && vala_data_type_is_reference_type_or_type_parameter (vala_formal_parameter_get_type_reference (param))) {
						gboolean is_ref;
						is_ref = vala_data_type_get_transfers_ownership (vala_formal_parameter_get_type_reference (param));
						if (is_ref && vala_data_type_get_type_parameter (vala_formal_parameter_get_type_reference (param)) != NULL) {
							ValaDataType* param_type;
							param_type = vala_semantic_analyzer_get_actual_type (vala_object_creation_expression_get_type_reference (expr), VALA_SYMBOL (msym), vala_formal_parameter_get_type_reference (param), VALA_CODE_NODE (expr));
							if (param_type != NULL) {
								is_ref = vala_data_type_get_takes_ownership (param_type);
							}
							(param_type == NULL ? NULL : (param_type = (g_object_unref (param_type), NULL)));
						}
						if (is_ref) {
							vala_memory_manager_visit_possibly_missing_copy_expression (self, arg);
						} else {
							vala_memory_manager_visit_possibly_leaked_expression (self, arg);
						}
					} else {
						vala_memory_manager_visit_possibly_leaked_expression (self, arg);
					}
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				} else {
					vala_memory_manager_visit_possibly_leaked_expression (self, arg);
				}
				(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
			}
		}
		(arg_collection == NULL ? NULL : (arg_collection = (g_object_unref (arg_collection), NULL)));
		(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	}
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
	(params_it == NULL ? NULL : (params_it = (g_object_unref (params_it), NULL)));
}


static void vala_memory_manager_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (expr == NULL || VALA_IS_BINARY_EXPRESSION (expr));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_binary_expression_get_left (expr));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_binary_expression_get_right (expr));
}


static void vala_memory_manager_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (l == NULL || VALA_IS_LAMBDA_EXPRESSION (l));
	vala_code_node_accept_children (VALA_CODE_NODE (l), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (a == NULL || VALA_IS_ASSIGNMENT (a));
	vala_code_node_accept_children (VALA_CODE_NODE (a), VALA_CODE_VISITOR (self));
	if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a)) || VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (a)))) {
	} else {
		if (!(VALA_IS_POINTER_TYPE (vala_expression_get_static_type (vala_assignment_get_left (a))))) {
			if (vala_data_type_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
				vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_assignment_get_right (a));
			} else {
				vala_memory_manager_visit_possibly_leaked_expression (self, vala_assignment_get_right (a));
			}
		}
	}
}


/**
 * Code visitor analyzing memory usage. The memory manager finds leaked and
 * copied references.
 */
ValaMemoryManager* vala_memory_manager_new (void) {
	ValaMemoryManager * self;
	self = g_object_newv (VALA_TYPE_MEMORY_MANAGER, 0, NULL);
	return self;
}


static void vala_memory_manager_class_init (ValaMemoryManagerClass * klass) {
	vala_memory_manager_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMemoryManagerPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_memory_manager_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_memory_manager_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_memory_manager_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_memory_manager_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_memory_manager_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_memory_manager_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_memory_manager_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_memory_manager_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_memory_manager_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_memory_manager_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_memory_manager_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_memory_manager_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_memory_manager_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_memory_manager_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_memory_manager_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_initializer_list = vala_memory_manager_real_visit_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_memory_manager_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_memory_manager_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_memory_manager_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_memory_manager_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_memory_manager_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_memory_manager_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_memory_manager_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_memory_manager_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_memory_manager_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_memory_manager_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_memory_manager_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_array_creation_expression = vala_memory_manager_real_visit_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_memory_manager_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_memory_manager_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_invocation_expression = vala_memory_manager_real_visit_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_object_creation_expression = vala_memory_manager_real_visit_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_memory_manager_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lambda_expression = vala_memory_manager_real_visit_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_assignment = vala_memory_manager_real_visit_assignment;
}


static void vala_memory_manager_init (ValaMemoryManager * self) {
	self->priv = VALA_MEMORY_MANAGER_GET_PRIVATE (self);
}


static void vala_memory_manager_dispose (GObject * obj) {
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (obj);
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	G_OBJECT_CLASS (vala_memory_manager_parent_class)->dispose (obj);
}


GType vala_memory_manager_get_type (void) {
	static GType vala_memory_manager_type_id = 0;
	if (G_UNLIKELY (vala_memory_manager_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMemoryManagerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_memory_manager_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMemoryManager), 0, (GInstanceInitFunc) vala_memory_manager_init };
		vala_memory_manager_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaMemoryManager", &g_define_type_info, 0);
	}
	return vala_memory_manager_type_id;
}




