/* valagidlparser.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valagidlparser.h"
#include <vala/valasourcereference.h>
#include <vala/valadatatype.h>
#include <glib/gstdio.h>
#include <gidlparser.h>
#include <stdio.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaformalparameter.h>
#include <vala/valaenumvalue.h>
#include <gmetadata.h>
#include <vala/valacreationmethod.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valastatement.h>
#include <vala/valaexpression.h>
#include "valagidlparser.h"

struct _ValaGIdlParserPrivate {
	ValaSourceReference* current_source_reference;
	ValaDataType* current_data_type;
	GHashTable* codenode_attributes_map;
	GHashTable* current_type_symbol_map;
};
#define VALA_GIDL_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIDL_PARSER, ValaGIdlParserPrivate))
enum  {
	VALA_GIDL_PARSER_DUMMY_PROPERTY
};
static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file);
static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, GIdlModule* module);
static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, GIdlModule* module);
static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module);
static ValaCallback* vala_gidl_parser_parse_callback (ValaGIdlParser* self, GIdlNodeFunction* f_node);
static ValaStruct* vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node);
static ValaStruct* vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node);
static ValaEnum* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node);
static ValaClass* vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node);
static ValaInterface* vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node);
static ValaTypeReference* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node);
static gboolean vala_gidl_parser_is_value_type (ValaGIdlParser* self, const char* type_name);
static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaTypeReference* type, const char* n);
static ValaTypeReference* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param);
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_virtual);
static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name);
static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node);
static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node);
static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node);
static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode);
static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s);
static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node);
static gpointer vala_gidl_parser_parent_class = NULL;
static void vala_gidl_parser_dispose (GObject * obj);


/**
 * Parse all source files in the specified code context and build a
 * code tree.
 *
 * @param context a code context
 */
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context)
{
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file)
{
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gidl")) {
		vala_gidl_parser_parse_file (self, source_file);
	}
}


static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file)
{
	GError* error;
	char* __temp0;
	char* __temp1;
	char* metadata_filename;
	GHashTable* __temp2;
	GList* modules;
	ValaSourceReference* __temp7;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	error = NULL;
	__temp0 = NULL;
	__temp1 = NULL;
	metadata_filename = (__temp1 = g_strdup_printf ("%s.metadata", (__temp0 = g_strndup (vala_source_file_get_filename (source_file), strlen (vala_source_file_get_filename (source_file)) - strlen (".gidl")))), (__temp0 = (g_free (__temp0), NULL)), __temp1);
	__temp2 = NULL;
	self->priv->codenode_attributes_map = (__temp2 = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free), (self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_hash_table_unref (self->priv->codenode_attributes_map), NULL))), __temp2);
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		char* metadata;
		glong metadata_len;
		metadata = NULL;
		metadata_len = 0L;
		g_file_get_contents (metadata_filename, &metadata, &metadata_len, &error);
		{
			char** __temp6;
			char** line_it;
			__temp6 = NULL;
			__temp6 = g_strsplit (metadata, "\n", 0);
			for (line_it = __temp6; *line_it != NULL; line_it = line_it + 1) {
				char* line;
				line = *line_it;
				{
					char** __temp3;
					gint line_parts_length1;
					char** line_parts;
					const char* __temp5;
					const char* __temp4;
					__temp3 = NULL;
					line_parts = (__temp3 = g_strsplit (line, " ", 2), line_parts_length1 = -1, __temp3);
					if (line_parts[0] == NULL) {
						continue;
					}
					__temp5 = NULL;
					__temp4 = NULL;
					g_hash_table_insert (self->priv->codenode_attributes_map, (__temp4 = line_parts[0], (__temp4 == NULL ? NULL : g_strdup (__temp4))), (__temp5 = line_parts[1], (__temp5 == NULL ? NULL : g_strdup (__temp5))));
					(line_parts = (g_free (line_parts), NULL));
				}
			}
			(__temp6 = (g_free (__temp6), NULL));
		}
		(metadata = (g_free (metadata), NULL));
	}
	modules = g_idl_parse_file (vala_source_file_get_filename (source_file), &error);
	if (error != NULL) {
		fprintf (stdout, "error parsing GIDL file: %s\n", error->message);
	}
	__temp7 = NULL;
	self->priv->current_source_reference = (__temp7 = vala_source_reference_new (source_file, 0, 0, 0, 0), (self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL))), __temp7);
	{
		GList* __temp8;
		GList* module_it;
		__temp8 = NULL;
		__temp8 = modules;
		for (module_it = __temp8; module_it != NULL; module_it = module_it->next) {
			GIdlModule* module;
			module = module_it->data;
			{
				ValaNamespace* ns;
				ns = vala_gidl_parser_parse_module (self, module);
				vala_source_file_add_namespace (source_file, ns);
				(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
			}
		}
	}
	(error == NULL ? NULL : (error = (g_error_free (error), NULL)));
	(metadata_filename = (g_free (metadata_filename), NULL));
	(modules == NULL ? NULL : (modules = (g_list_foreach (modules, (GFunc) g_free, NULL), g_list_free (modules), NULL)));
}


static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, GIdlModule* module)
{
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	g_return_val_if_fail (module != NULL, NULL);
	if (g_str_has_prefix (type_name, module->name)) {
		const char* __temp9;
		__temp9 = NULL;
		return (__temp9 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (module->name, -1)), (__temp9 == NULL ? NULL : g_strdup (__temp9)));
	} else {
		if (g_utf8_collate (module->name, "GLib") == 0 && g_str_has_prefix (type_name, "G")) {
			const char* __temp11;
			__temp11 = NULL;
			return (__temp11 = g_utf8_offset_to_pointer (type_name, 1), (__temp11 == NULL ? NULL : g_strdup (__temp11)));
		}
	}
	return g_strdup (type_name);
}


static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, GIdlModule* module)
{
	char* __temp15;
	char* __temp14;
	gboolean __temp16;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (const_name != NULL, NULL);
	g_return_val_if_fail (module != NULL, NULL);
	__temp15 = NULL;
	__temp14 = NULL;
	if ((__temp16 = g_str_has_prefix (const_name, (__temp15 = g_strconcat ((__temp14 = g_utf8_strup (module->name, -1)), "_", NULL))), (__temp15 = (g_free (__temp15), NULL)), (__temp14 = (g_free (__temp14), NULL)), __temp16)) {
		const char* __temp17;
		__temp17 = NULL;
		return (__temp17 = g_utf8_offset_to_pointer (const_name, g_utf8_strlen (module->name, -1) + 1), (__temp17 == NULL ? NULL : g_strdup (__temp17)));
	} else {
		if (g_utf8_collate (module->name, "GLib") == 0 && g_str_has_prefix (const_name, "G_")) {
			const char* __temp19;
			__temp19 = NULL;
			return (__temp19 = g_utf8_offset_to_pointer (const_name, 2), (__temp19 == NULL ? NULL : g_strdup (__temp19)));
		}
	}
	return g_strdup (const_name);
}


static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module)
{
	ValaNamespace* ns;
	char** __temp22;
	gint attributes_length1;
	char** attributes;
	ValaNamespace* __temp37;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (module != NULL, NULL);
	ns = vala_namespace_new (module->name, self->priv->current_source_reference);
	__temp22 = NULL;
	attributes = (__temp22 = vala_gidl_parser_get_attributes (self, vala_namespace_get_name (ns)), attributes_length1 = -1, __temp22);
	if (attributes != NULL) {
		{
			char** __temp27;
			int attr_it;
			__temp27 = NULL;
			__temp27 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp27[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp27[attr_it];
				{
					char** __temp23;
					gint nv_length1;
					char** nv;
					__temp23 = NULL;
					nv = (__temp23 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp23);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp24;
						__temp24 = NULL;
						vala_namespace_set_cheader_filename (ns, (__temp24 = vala_gidl_parser_eval (self, nv[1])));
						(__temp24 = (g_free (__temp24), NULL));
					} else {
						if (g_utf8_collate (nv[0], "cprefix") == 0) {
							char* __temp25;
							__temp25 = NULL;
							vala_namespace_set_cprefix (ns, (__temp25 = vala_gidl_parser_eval (self, nv[1])));
							(__temp25 = (g_free (__temp25), NULL));
						} else {
							if (g_utf8_collate (nv[0], "lower_case_cprefix") == 0) {
								char* __temp26;
								__temp26 = NULL;
								vala_namespace_set_lower_case_cprefix (ns, (__temp26 = vala_gidl_parser_eval (self, nv[1])));
								(__temp26 = (g_free (__temp26), NULL));
							}
						}
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	{
		GList* __temp36;
		GList* node_it;
		__temp36 = NULL;
		__temp36 = module->entries;
		for (node_it = __temp36; node_it != NULL; node_it = node_it->next) {
			GIdlNode* node;
			node = node_it->data;
			{
				if (node->type == G_IDL_NODE_CALLBACK) {
					ValaCallback* cb;
					char* __temp28;
					cb = vala_gidl_parser_parse_callback (self, ((GIdlNodeFunction*) node));
					if (cb == NULL) {
						continue;
					}
					__temp28 = NULL;
					(vala_data_type_set_name (VALA_DATA_TYPE (cb), (__temp28 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (cb)), module))), vala_data_type_get_name (VALA_DATA_TYPE (cb)));
					(__temp28 = (g_free (__temp28), NULL));
					vala_namespace_add_callback (ns, cb);
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
				} else {
					if (node->type == G_IDL_NODE_STRUCT) {
						ValaStruct* st;
						char* __temp29;
						st = vala_gidl_parser_parse_struct (self, ((GIdlNodeStruct*) node));
						if (st == NULL) {
							continue;
						}
						__temp29 = NULL;
						(vala_data_type_set_name (VALA_DATA_TYPE (st), (__temp29 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (st)), module))), vala_data_type_get_name (VALA_DATA_TYPE (st)));
						(__temp29 = (g_free (__temp29), NULL));
						vala_namespace_add_struct (ns, st);
						(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
					} else {
						if (node->type == G_IDL_NODE_BOXED) {
							ValaStruct* st;
							char* __temp30;
							char* __temp31;
							st = vala_gidl_parser_parse_boxed (self, ((GIdlNodeBoxed*) node));
							__temp30 = NULL;
							(vala_data_type_set_name (VALA_DATA_TYPE (st), (__temp30 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (st)), module))), vala_data_type_get_name (VALA_DATA_TYPE (st)));
							(__temp30 = (g_free (__temp30), NULL));
							vala_namespace_add_struct (ns, st);
							__temp31 = NULL;
							vala_struct_set_type_id (st, (__temp31 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (st), "TYPE_")));
							(__temp31 = (g_free (__temp31), NULL));
							(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
						} else {
							if (node->type == G_IDL_NODE_ENUM) {
								ValaEnum* en;
								char* __temp32;
								en = vala_gidl_parser_parse_enum (self, ((GIdlNodeEnum*) node));
								__temp32 = NULL;
								(vala_data_type_set_name (VALA_DATA_TYPE (en), (__temp32 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (en)), module))), vala_data_type_get_name (VALA_DATA_TYPE (en)));
								(__temp32 = (g_free (__temp32), NULL));
								vala_namespace_add_enum (ns, en);
								(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
							} else {
								if (node->type == G_IDL_NODE_OBJECT) {
									ValaClass* cl;
									char* __temp33;
									cl = vala_gidl_parser_parse_object (self, ((GIdlNodeInterface*) node));
									__temp33 = NULL;
									(vala_data_type_set_name (VALA_DATA_TYPE (cl), (__temp33 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (cl)), module))), vala_data_type_get_name (VALA_DATA_TYPE (cl)));
									(__temp33 = (g_free (__temp33), NULL));
									vala_namespace_add_class (ns, cl);
									(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
								} else {
									if (node->type == G_IDL_NODE_INTERFACE) {
										ValaInterface* iface;
										char* __temp34;
										iface = vala_gidl_parser_parse_interface (self, ((GIdlNodeInterface*) node));
										__temp34 = NULL;
										(vala_data_type_set_name (VALA_DATA_TYPE (iface), (__temp34 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (iface)), module))), vala_data_type_get_name (VALA_DATA_TYPE (iface)));
										(__temp34 = (g_free (__temp34), NULL));
										vala_namespace_add_interface (ns, iface);
										(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
									} else {
										if (node->type == G_IDL_NODE_CONSTANT) {
											ValaConstant* c;
											char* __temp35;
											c = vala_gidl_parser_parse_constant (self, ((GIdlNodeConstant*) node));
											__temp35 = NULL;
											(vala_constant_set_name (c, (__temp35 = vala_gidl_parser_fix_const_name (self, vala_constant_get_name (c), module))), vala_constant_get_name (c));
											(__temp35 = (g_free (__temp35), NULL));
											vala_namespace_add_constant (ns, c);
											(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
										} else {
											if (node->type == G_IDL_NODE_FUNCTION) {
												ValaMethod* m;
												m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) node), FALSE);
												if (m != NULL) {
													(vala_method_set_instance (m, FALSE), vala_method_get_instance (m));
													vala_namespace_add_method (ns, m);
												}
												(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	__temp37 = NULL;
	return (__temp37 = ns, (attributes = (g_free (attributes), NULL)), __temp37);
	(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
	(attributes = (g_free (attributes), NULL));
}


static ValaCallback* vala_gidl_parser_parse_callback (ValaGIdlParser* self, GIdlNodeFunction* f_node)
{
	GIdlNode* node;
	char** __temp38;
	gint attributes_length1;
	char** attributes;
	ValaCallback* cb;
	ValaCallback* __temp45;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (f_node != NULL, NULL);
	node = ((GIdlNode*) f_node);
	__temp38 = NULL;
	attributes = (__temp38 = vala_gidl_parser_get_attributes (self, node->name), attributes_length1 = -1, __temp38);
	if (attributes != NULL) {
		{
			char** __temp43;
			int attr_it;
			__temp43 = NULL;
			__temp43 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp43[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp43[attr_it];
				{
					char** __temp39;
					gint nv_length1;
					char** nv;
					__temp39 = NULL;
					nv = (__temp39 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp39);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* __temp40;
						gboolean __temp41;
						__temp40 = NULL;
						if ((__temp41 = g_utf8_collate ((__temp40 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp40 = (g_free (__temp40), NULL)), __temp41)) {
							gpointer __temp42;
							return VALA_CALLBACK ((__temp42 = NULL, (nv = (g_free (nv), NULL)), __temp42));
						}
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	cb = vala_callback_new (node->name, vala_gidl_parser_parse_param (self, f_node->result), self->priv->current_source_reference);
	VALA_DATA_TYPE (cb)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	{
		GList* __temp44;
		GList* param_it;
		__temp44 = NULL;
		__temp44 = f_node->parameters;
		for (param_it = __temp44; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaFormalParameter* p;
				param_node = ((GIdlNode*) param);
				p = vala_formal_parameter_new (param_node->name, vala_gidl_parser_parse_param (self, param), NULL);
				vala_callback_add_parameter (cb, p);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	__temp45 = NULL;
	return (__temp45 = cb, (attributes = (g_free (attributes), NULL)), __temp45);
	(attributes = (g_free (attributes), NULL));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
}


static ValaStruct* vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node)
{
	GIdlNode* node;
	ValaStruct* st;
	char** __temp47;
	gint st_attributes_length1;
	char** st_attributes;
	ValaDataType* __temp53;
	ValaStruct* __temp52;
	ValaDataType* __temp55;
	ValaStruct* __temp56;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (st_node != NULL, NULL);
	node = ((GIdlNode*) st_node);
	if (st_node->deprecated) {
		return VALA_STRUCT (NULL);
	}
	st = vala_struct_new (node->name, self->priv->current_source_reference);
	VALA_DATA_TYPE (st)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	vala_struct_set_is_reference_type (st, TRUE);
	__temp47 = NULL;
	st_attributes = (__temp47 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, __temp47);
	if (st_attributes != NULL) {
		{
			char** __temp51;
			int attr_it;
			__temp51 = NULL;
			__temp51 = st_attributes;
			for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && __temp51[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp51[attr_it];
				{
					char** __temp48;
					gint nv_length1;
					char** nv;
					char* __temp49;
					gboolean __temp50;
					__temp48 = NULL;
					nv = (__temp48 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp48);
					__temp49 = NULL;
					if ((__temp50 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp49 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp49 = (g_free (__temp49), NULL)), __temp50)) {
						vala_struct_set_is_reference_type (st, FALSE);
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	__temp53 = NULL;
	__temp52 = NULL;
	self->priv->current_data_type = (__temp53 = VALA_DATA_TYPE ((__temp52 = st, (__temp52 == NULL ? NULL : g_object_ref (__temp52)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp53);
	{
		GList* __temp54;
		GList* member_it;
		__temp54 = NULL;
		__temp54 = st_node->members;
		for (member_it = __temp54; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					ValaMethod* m;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
					if (m != NULL) {
						vala_struct_add_method (st, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_FIELD) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_struct_add_field (st, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	__temp55 = NULL;
	self->priv->current_data_type = (__temp55 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp55);
	__temp56 = NULL;
	return (__temp56 = st, (st_attributes = (g_free (st_attributes), NULL)), __temp56);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	(st_attributes = (g_free (st_attributes), NULL));
}


static ValaStruct* vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node)
{
	GIdlNode* node;
	ValaStruct* st;
	char** __temp57;
	gint st_attributes_length1;
	char** st_attributes;
	ValaDataType* __temp63;
	ValaStruct* __temp62;
	ValaDataType* __temp65;
	ValaStruct* __temp66;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (boxed_node != NULL, NULL);
	node = ((GIdlNode*) boxed_node);
	st = vala_struct_new (node->name, self->priv->current_source_reference);
	VALA_DATA_TYPE (st)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	vala_struct_set_is_reference_type (st, TRUE);
	__temp57 = NULL;
	st_attributes = (__temp57 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, __temp57);
	if (st_attributes != NULL) {
		{
			char** __temp61;
			int attr_it;
			__temp61 = NULL;
			__temp61 = st_attributes;
			for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && __temp61[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp61[attr_it];
				{
					char** __temp58;
					gint nv_length1;
					char** nv;
					char* __temp59;
					gboolean __temp60;
					__temp58 = NULL;
					nv = (__temp58 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp58);
					__temp59 = NULL;
					if ((__temp60 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp59 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp59 = (g_free (__temp59), NULL)), __temp60)) {
						vala_struct_set_is_reference_type (st, FALSE);
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	__temp63 = NULL;
	__temp62 = NULL;
	self->priv->current_data_type = (__temp63 = VALA_DATA_TYPE ((__temp62 = st, (__temp62 == NULL ? NULL : g_object_ref (__temp62)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp63);
	{
		GList* __temp64;
		GList* member_it;
		__temp64 = NULL;
		__temp64 = boxed_node->members;
		for (member_it = __temp64; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					ValaMethod* m;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
					if (m != NULL) {
						vala_struct_add_method (st, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_FIELD) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_struct_add_field (st, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	__temp65 = NULL;
	self->priv->current_data_type = (__temp65 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp65);
	__temp66 = NULL;
	return (__temp66 = st, (st_attributes = (g_free (st_attributes), NULL)), __temp66);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	(st_attributes = (g_free (st_attributes), NULL));
}


static ValaEnum* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node)
{
	GIdlNode* node;
	ValaEnum* en;
	char* common_prefix;
	ValaEnum* __temp73;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (en_node != NULL, NULL);
	node = ((GIdlNode*) en_node);
	en = vala_enum_new (node->name, self->priv->current_source_reference);
	VALA_DATA_TYPE (en)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	common_prefix = NULL;
	{
		GList* __temp71;
		GList* value_it;
		__temp71 = NULL;
		__temp71 = en_node->values;
		for (value_it = __temp71; value_it != NULL; value_it = value_it->next) {
			GIdlNode* value;
			value = value_it->data;
			{
				if (common_prefix == NULL) {
					char* __temp68;
					const char* __temp67;
					__temp68 = NULL;
					__temp67 = NULL;
					common_prefix = (__temp68 = (__temp67 = value->name, (__temp67 == NULL ? NULL : g_strdup (__temp67))), (common_prefix = (g_free (common_prefix), NULL)), __temp68);
					while (g_utf8_strlen (common_prefix, -1) > 0 && !g_str_has_suffix (common_prefix, "_")) {
						char* __temp69;
						/* FIXME: could easily be made faster*/
						__temp69 = NULL;
						common_prefix = (__temp69 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), __temp69);
					}
				} else {
					while (!g_str_has_prefix (value->name, common_prefix)) {
						char* __temp70;
						__temp70 = NULL;
						common_prefix = (__temp70 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), __temp70);
					}
				}
			}
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	{
		GList* __temp72;
		GList* value2_it;
		__temp72 = NULL;
		__temp72 = en_node->values;
		for (value2_it = __temp72; value2_it != NULL; value2_it = value2_it->next) {
			GIdlNode* value2;
			value2 = value2_it->data;
			{
				ValaEnumValue* ev;
				ev = vala_enum_value_new (g_utf8_offset_to_pointer (value2->name, g_utf8_strlen (common_prefix, -1)));
				vala_enum_add_value (en, ev);
				(ev == NULL ? NULL : (ev = (g_object_unref (ev), NULL)));
			}
		}
	}
	__temp73 = NULL;
	return (__temp73 = en, (common_prefix = (g_free (common_prefix), NULL)), __temp73);
	(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
	(common_prefix = (g_free (common_prefix), NULL));
}


static ValaClass* vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node)
{
	ValaClass* cl;
	char** __temp74;
	gint attributes_length1;
	char** attributes;
	ValaDataType* __temp80;
	ValaClass* __temp79;
	GHashTable* __temp81;
	GHashTable* current_type_vfunc_map;
	ValaDataType* __temp87;
	GHashTable* __temp88;
	ValaClass* __temp89;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (node != NULL, NULL);
	cl = vala_class_new (node->gtype_name, self->priv->current_source_reference);
	VALA_DATA_TYPE (cl)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp74 = NULL;
	attributes = (__temp74 = vala_gidl_parser_get_attributes (self, vala_data_type_get_name (VALA_DATA_TYPE (cl))), attributes_length1 = -1, __temp74);
	if (attributes != NULL) {
		{
			char** __temp77;
			int attr_it;
			__temp77 = NULL;
			__temp77 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp77[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp77[attr_it];
				{
					char** __temp75;
					gint nv_length1;
					char** nv;
					__temp75 = NULL;
					nv = (__temp75 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp75);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp76;
						__temp76 = NULL;
						vala_data_type_add_cheader_filename (VALA_DATA_TYPE (cl), (__temp76 = vala_gidl_parser_eval (self, nv[1])));
						(__temp76 = (g_free (__temp76), NULL));
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	if (node->parent != NULL) {
		ValaTypeReference* parent;
		parent = vala_type_reference_new ();
		vala_gidl_parser_parse_type_string (self, parent, node->parent);
		vala_class_add_base_type (cl, parent);
		(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
	}
	{
		GList* __temp78;
		GList* iface_name_it;
		__temp78 = NULL;
		__temp78 = node->interfaces;
		for (iface_name_it = __temp78; iface_name_it != NULL; iface_name_it = iface_name_it->next) {
			char* iface_name;
			iface_name = iface_name_it->data;
			{
				ValaTypeReference* iface;
				iface = vala_type_reference_new ();
				vala_gidl_parser_parse_type_string (self, iface, iface_name);
				vala_class_add_base_type (cl, iface);
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	__temp80 = NULL;
	__temp79 = NULL;
	self->priv->current_data_type = (__temp80 = VALA_DATA_TYPE ((__temp79 = cl, (__temp79 == NULL ? NULL : g_object_ref (__temp79)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp80);
	__temp81 = NULL;
	self->priv->current_type_symbol_map = (__temp81 = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free), (self->priv->current_type_symbol_map == NULL ? NULL : (self->priv->current_type_symbol_map = (g_hash_table_unref (self->priv->current_type_symbol_map), NULL))), __temp81);
	current_type_vfunc_map = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	{
		GList* __temp83;
		GList* member_it;
		__temp83 = NULL;
		__temp83 = node->members;
		for (member_it = __temp83; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_VFUNC) {
					const char* __temp82;
					__temp82 = NULL;
					g_hash_table_insert (current_type_vfunc_map, (__temp82 = member->name, (__temp82 == NULL ? NULL : g_strdup (__temp82))), g_strdup ("1"));
				}
			}
		}
	}
	{
		GList* __temp84;
		GList* member_it;
		__temp84 = NULL;
		__temp84 = node->members;
		for (member_it = __temp84; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gboolean is_virtual;
					ValaMethod* m;
					is_virtual = g_hash_table_lookup (current_type_vfunc_map, member->name) != NULL;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), is_virtual);
					if (m != NULL) {
						vala_class_add_method (cl, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_PROPERTY) {
						ValaProperty* prop;
						prop = vala_gidl_parser_parse_property (self, ((GIdlNodeProperty*) member));
						if (prop != NULL) {
							vala_class_add_property (cl, prop, FALSE);
						}
						(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					} else {
						if (member->type == G_IDL_NODE_SIGNAL) {
							ValaSignal* sig;
							sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
							if (sig != NULL) {
								vala_class_add_signal (cl, sig);
							}
							(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
						}
					}
				}
			}
		}
	}
	{
		GList* __temp85;
		GList* member_it;
		__temp85 = NULL;
		__temp85 = node->members;
		for (member_it = __temp85; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FIELD) {
					if (g_hash_table_lookup (self->priv->current_type_symbol_map, member->name) == NULL) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_class_add_field (cl, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	{
		GList* __temp86;
		GList* prop_it;
		__temp86 = NULL;
		__temp86 = vala_class_get_properties (cl);
		for (prop_it = __temp86; prop_it != NULL; prop_it = prop_it->next) {
			ValaProperty* prop;
			prop = prop_it->data;
			{
				char* getter;
				char* setter;
				getter = g_strdup_printf ("get_%s", vala_property_get_name (prop));
				if (vala_property_get_get_accessor (prop) != NULL && g_hash_table_lookup (self->priv->current_type_symbol_map, getter) == NULL) {
					(vala_property_set_no_accessor_method (prop, TRUE), vala_property_get_no_accessor_method (prop));
				}
				setter = g_strdup_printf ("set_%s", vala_property_get_name (prop));
				if (vala_property_get_set_accessor (prop) != NULL && g_hash_table_lookup (self->priv->current_type_symbol_map, setter) == NULL) {
					(vala_property_set_no_accessor_method (prop, TRUE), vala_property_get_no_accessor_method (prop));
				}
				(getter = (g_free (getter), NULL));
				(setter = (g_free (setter), NULL));
			}
		}
		(__temp86 == NULL ? NULL : (__temp86 = (g_list_free (__temp86), NULL)));
	}
	__temp87 = NULL;
	self->priv->current_data_type = (__temp87 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp87);
	__temp88 = NULL;
	self->priv->current_type_symbol_map = (__temp88 = NULL, (self->priv->current_type_symbol_map == NULL ? NULL : (self->priv->current_type_symbol_map = (g_hash_table_unref (self->priv->current_type_symbol_map), NULL))), __temp88);
	__temp89 = NULL;
	return (__temp89 = cl, (attributes = (g_free (attributes), NULL)), (current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_hash_table_unref (current_type_vfunc_map), NULL))), __temp89);
	(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	(attributes = (g_free (attributes), NULL));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_hash_table_unref (current_type_vfunc_map), NULL)));
}


static ValaInterface* vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node)
{
	ValaInterface* iface;
	char** __temp90;
	gint attributes_length1;
	char** attributes;
	ValaDataType* __temp96;
	ValaInterface* __temp95;
	GHashTable* current_type_vfunc_map;
	ValaDataType* __temp100;
	ValaInterface* __temp101;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (node != NULL, NULL);
	iface = vala_interface_new (node->gtype_name, self->priv->current_source_reference);
	VALA_DATA_TYPE (iface)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp90 = NULL;
	attributes = (__temp90 = vala_gidl_parser_get_attributes (self, vala_data_type_get_name (VALA_DATA_TYPE (iface))), attributes_length1 = -1, __temp90);
	if (attributes != NULL) {
		{
			char** __temp93;
			int attr_it;
			__temp93 = NULL;
			__temp93 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp93[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp93[attr_it];
				{
					char** __temp91;
					gint nv_length1;
					char** nv;
					__temp91 = NULL;
					nv = (__temp91 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp91);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp92;
						__temp92 = NULL;
						vala_data_type_add_cheader_filename (VALA_DATA_TYPE (iface), (__temp92 = vala_gidl_parser_eval (self, nv[1])));
						(__temp92 = (g_free (__temp92), NULL));
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	{
		GList* __temp94;
		GList* prereq_name_it;
		__temp94 = NULL;
		__temp94 = node->prerequisites;
		for (prereq_name_it = __temp94; prereq_name_it != NULL; prereq_name_it = prereq_name_it->next) {
			char* prereq_name;
			prereq_name = prereq_name_it->data;
			{
				ValaTypeReference* prereq;
				prereq = vala_type_reference_new ();
				vala_gidl_parser_parse_type_string (self, prereq, prereq_name);
				vala_interface_add_prerequisite (iface, prereq);
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
			}
		}
	}
	__temp96 = NULL;
	__temp95 = NULL;
	self->priv->current_data_type = (__temp96 = VALA_DATA_TYPE ((__temp95 = iface, (__temp95 == NULL ? NULL : g_object_ref (__temp95)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp96);
	current_type_vfunc_map = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	{
		GList* __temp98;
		GList* member_it;
		__temp98 = NULL;
		__temp98 = node->members;
		for (member_it = __temp98; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_VFUNC) {
					const char* __temp97;
					__temp97 = NULL;
					g_hash_table_insert (current_type_vfunc_map, (__temp97 = member->name, (__temp97 == NULL ? NULL : g_strdup (__temp97))), g_strdup ("1"));
				}
			}
		}
	}
	{
		GList* __temp99;
		GList* member_it;
		__temp99 = NULL;
		__temp99 = node->members;
		for (member_it = __temp99; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gboolean is_virtual;
					ValaMethod* m;
					is_virtual = g_hash_table_lookup (current_type_vfunc_map, member->name) != NULL;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), is_virtual);
					if (m != NULL) {
						vala_interface_add_method (iface, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_SIGNAL) {
						ValaSignal* sig;
						sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
						if (sig != NULL) {
							vala_interface_add_signal (iface, sig);
						}
						(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
					}
				}
			}
		}
	}
	__temp100 = NULL;
	self->priv->current_data_type = (__temp100 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp100);
	__temp101 = NULL;
	return (__temp101 = iface, (attributes = (g_free (attributes), NULL)), (current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_hash_table_unref (current_type_vfunc_map), NULL))), __temp101);
	(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
	(attributes = (g_free (attributes), NULL));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_hash_table_unref (current_type_vfunc_map), NULL)));
}


static ValaTypeReference* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node)
{
	ValaTypeReference* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (type_node != NULL, NULL);
	type = vala_type_reference_new ();
	if (type_node->tag == TYPE_TAG_VOID) {
		if (type_node->is_pointer) {
			(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
		} else {
			(vala_type_reference_set_type_name (type, "void"), vala_type_reference_get_type_name (type));
		}
	} else {
		if (type_node->tag == TYPE_TAG_BOOLEAN) {
			(vala_type_reference_set_type_name (type, "bool"), vala_type_reference_get_type_name (type));
		} else {
			if (type_node->tag == TYPE_TAG_INT8) {
				(vala_type_reference_set_type_name (type, "char"), vala_type_reference_get_type_name (type));
			} else {
				if (type_node->tag == TYPE_TAG_UINT8) {
					(vala_type_reference_set_type_name (type, "uchar"), vala_type_reference_get_type_name (type));
				} else {
					if (type_node->tag == TYPE_TAG_INT16) {
						(vala_type_reference_set_type_name (type, "short"), vala_type_reference_get_type_name (type));
					} else {
						if (type_node->tag == TYPE_TAG_UINT16) {
							(vala_type_reference_set_type_name (type, "ushort"), vala_type_reference_get_type_name (type));
						} else {
							if (type_node->tag == TYPE_TAG_INT32) {
								(vala_type_reference_set_type_name (type, "int"), vala_type_reference_get_type_name (type));
							} else {
								if (type_node->tag == TYPE_TAG_UINT32) {
									(vala_type_reference_set_type_name (type, "uint"), vala_type_reference_get_type_name (type));
								} else {
									if (type_node->tag == TYPE_TAG_INT64) {
										(vala_type_reference_set_type_name (type, "int64"), vala_type_reference_get_type_name (type));
									} else {
										if (type_node->tag == TYPE_TAG_UINT64) {
											(vala_type_reference_set_type_name (type, "uint64"), vala_type_reference_get_type_name (type));
										} else {
											if (type_node->tag == TYPE_TAG_INT) {
												(vala_type_reference_set_type_name (type, "int"), vala_type_reference_get_type_name (type));
											} else {
												if (type_node->tag == TYPE_TAG_UINT) {
													(vala_type_reference_set_type_name (type, "uint"), vala_type_reference_get_type_name (type));
												} else {
													if (type_node->tag == TYPE_TAG_LONG) {
														(vala_type_reference_set_type_name (type, "long"), vala_type_reference_get_type_name (type));
													} else {
														if (type_node->tag == TYPE_TAG_ULONG) {
															(vala_type_reference_set_type_name (type, "ulong"), vala_type_reference_get_type_name (type));
														} else {
															if (type_node->tag == TYPE_TAG_SSIZE) {
																(vala_type_reference_set_type_name (type, "long"), vala_type_reference_get_type_name (type));
															} else {
																if (type_node->tag == TYPE_TAG_SIZE) {
																	(vala_type_reference_set_type_name (type, "ulong"), vala_type_reference_get_type_name (type));
																} else {
																	if (type_node->tag == TYPE_TAG_FLOAT) {
																		(vala_type_reference_set_type_name (type, "float"), vala_type_reference_get_type_name (type));
																	} else {
																		if (type_node->tag == TYPE_TAG_DOUBLE) {
																			(vala_type_reference_set_type_name (type, "double"), vala_type_reference_get_type_name (type));
																		} else {
																			if (type_node->tag == TYPE_TAG_UTF8) {
																				(vala_type_reference_set_type_name (type, "string"), vala_type_reference_get_type_name (type));
																			} else {
																				if (type_node->tag == TYPE_TAG_FILENAME) {
																					(vala_type_reference_set_type_name (type, "string"), vala_type_reference_get_type_name (type));
																				} else {
																					if (type_node->tag == TYPE_TAG_ARRAY) {
																						ValaTypeReference* __temp102;
																						__temp102 = NULL;
																						type = (__temp102 = vala_gidl_parser_parse_type (self, type_node->parameter_type1), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp102);
																						(vala_type_reference_set_array_rank (type, 1), vala_type_reference_get_array_rank (type));
																					} else {
																						if (type_node->tag == TYPE_TAG_LIST) {
																							(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																							(vala_type_reference_set_type_name (type, "List"), vala_type_reference_get_type_name (type));
																						} else {
																							if (type_node->tag == TYPE_TAG_SLIST) {
																								(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																								(vala_type_reference_set_type_name (type, "SList"), vala_type_reference_get_type_name (type));
																							} else {
																								if (type_node->tag == TYPE_TAG_HASH) {
																									(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																									(vala_type_reference_set_type_name (type, "HashTable"), vala_type_reference_get_type_name (type));
																								} else {
																									if (type_node->tag == TYPE_TAG_ERROR) {
																										(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																										(vala_type_reference_set_type_name (type, "Error"), vala_type_reference_get_type_name (type));
																									} else {
																										if (type_node->is_interface) {
																											const char* __temp103;
																											char* n;
																											__temp103 = NULL;
																											n = (__temp103 = type_node->interface, (__temp103 == NULL ? NULL : g_strdup (__temp103)));
																											if (g_utf8_collate (n, "") == 0) {
																												gpointer __temp104;
																												return VALA_TYPE_REFERENCE ((__temp104 = NULL, (n = (g_free (n), NULL)), __temp104));
																											}
																											if (g_str_has_prefix (n, "const-")) {
																												char* __temp106;
																												const char* __temp105;
																												__temp106 = NULL;
																												__temp105 = NULL;
																												n = (__temp106 = (__temp105 = g_utf8_offset_to_pointer (n, g_utf8_strlen ("const-", -1)), (__temp105 == NULL ? NULL : g_strdup (__temp105))), (n = (g_free (n), NULL)), __temp106);
																											}
																											if (type_node->is_pointer && (g_utf8_collate (n, "gchar") == 0 || g_utf8_collate (n, "char") == 0)) {
																												(vala_type_reference_set_type_name (type, "string"), vala_type_reference_get_type_name (type));
																											} else {
																												if (g_utf8_collate (n, "gunichar") == 0) {
																													(vala_type_reference_set_type_name (type, "unichar"), vala_type_reference_get_type_name (type));
																												} else {
																													if (g_utf8_collate (n, "gchar") == 0) {
																														(vala_type_reference_set_type_name (type, "char"), vala_type_reference_get_type_name (type));
																													} else {
																														if (g_utf8_collate (n, "guchar") == 0 || g_utf8_collate (n, "guint8") == 0) {
																															(vala_type_reference_set_type_name (type, "uchar"), vala_type_reference_get_type_name (type));
																															if (type_node->is_pointer) {
																																(vala_type_reference_set_array_rank (type, 1), vala_type_reference_get_array_rank (type));
																															}
																														} else {
																															if (g_utf8_collate (n, "gushort") == 0) {
																																(vala_type_reference_set_type_name (type, "ushort"), vala_type_reference_get_type_name (type));
																															} else {
																																if (g_utf8_collate (n, "gshort") == 0) {
																																	(vala_type_reference_set_type_name (type, "short"), vala_type_reference_get_type_name (type));
																																} else {
																																	if (g_utf8_collate (n, "gconstpointer") == 0 || g_utf8_collate (n, "void") == 0) {
																																		(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
																																	} else {
																																		if (g_utf8_collate (n, "goffset") == 0) {
																																			(vala_type_reference_set_type_name (type, "int64"), vala_type_reference_get_type_name (type));
																																		} else {
																																			if (g_utf8_collate (n, "value_array") == 0) {
																																				(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																																				(vala_type_reference_set_type_name (type, "ValueArray"), vala_type_reference_get_type_name (type));
																																			} else {
																																				if (g_utf8_collate (n, "time_t") == 0) {
																																					(vala_type_reference_set_type_name (type, "ulong"), vala_type_reference_get_type_name (type));
																																				} else {
																																					if (g_utf8_collate (n, "pid_t") == 0) {
																																						(vala_type_reference_set_type_name (type, "int"), vala_type_reference_get_type_name (type));
																																					} else {
																																						if (g_utf8_collate (n, "FILE") == 0) {
																																							(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																																							(vala_type_reference_set_type_name (type, "FileStream"), vala_type_reference_get_type_name (type));
																																						} else {
																																							vala_gidl_parser_parse_type_string (self, type, n);
																																							if (type_node->is_pointer && vala_gidl_parser_is_value_type (self, n)) {
																																								(vala_type_reference_set_is_out (type, TRUE), vala_type_reference_get_is_out (type));
																																							}
																																						}
																																					}
																																				}
																																			}
																																		}
																																	}
																																}
																															}
																														}
																													}
																												}
																											}
																											(n = (g_free (n), NULL));
																										} else {
																											fprintf (stdout, "%d\n", type_node->tag);
																										}
																									}
																								}
																							}
																						}
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	return type;
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static gboolean vala_gidl_parser_is_value_type (ValaGIdlParser* self, const char* type_name)
{
	char** __temp108;
	gint type_attributes_length1;
	char** type_attributes;
	gboolean __temp114;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), FALSE);
	g_return_val_if_fail (type_name != NULL, FALSE);
	/* FIXME only works if both types are in current package, e.g. doesn't work when Gtk uses GdkRectangle*/
	__temp108 = NULL;
	type_attributes = (__temp108 = vala_gidl_parser_get_attributes (self, type_name), type_attributes_length1 = -1, __temp108);
	if (type_attributes != NULL) {
		{
			char** __temp113;
			int attr_it;
			__temp113 = NULL;
			__temp113 = type_attributes;
			for (attr_it = 0; (type_attributes_length1 != -1 && attr_it < type_attributes_length1) || (type_attributes_length1 == -1 && __temp113[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp113[attr_it];
				{
					char** __temp109;
					gint nv_length1;
					char** nv;
					char* __temp110;
					gboolean __temp111;
					__temp109 = NULL;
					nv = (__temp109 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp109);
					__temp110 = NULL;
					if ((__temp111 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp110 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp110 = (g_free (__temp110), NULL)), __temp111)) {
						gboolean __temp112;
						return (__temp112 = TRUE, (nv = (g_free (nv), NULL)), __temp112);
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	return (__temp114 = FALSE, (type_attributes = (g_free (type_attributes), NULL)), __temp114);
	(type_attributes = (g_free (type_attributes), NULL));
}


static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaTypeReference* type, const char* n)
{
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	g_return_if_fail (n != NULL);
	/* Generated GIDL misses explicit namespace specifier,
	 so try to guess namespace*/
	if (g_str_has_prefix (n, "H") || g_str_has_suffix (n, "Class") || g_utf8_collate (n, "va_list") == 0 || g_str_has_prefix (n, "LOGFONT")) {
		/* unsupported*/
		(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
	} else {
		if (g_str_has_prefix (n, "cairo")) {
			(vala_type_reference_set_namespace_name (type, "Cairo"), vala_type_reference_get_namespace_name (type));
			if (g_utf8_collate (n, "cairo_t") == 0) {
				(vala_type_reference_set_type_name (type, "Context"), vala_type_reference_get_type_name (type));
			} else {
				if (g_utf8_collate (n, "cairo_surface_t") == 0) {
					(vala_type_reference_set_type_name (type, "Surface"), vala_type_reference_get_type_name (type));
				} else {
					(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
					(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
				}
			}
		} else {
			if (g_str_has_prefix (n, "Atk")) {
				(vala_type_reference_set_namespace_name (type, "Atk"), vala_type_reference_get_namespace_name (type));
				(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Atk", -1))), vala_type_reference_get_type_name (type));
				if (g_utf8_collate (vala_type_reference_get_type_name (type), "AttributeSet") == 0) {
					(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
					(vala_type_reference_set_type_name (type, "SList"), vala_type_reference_get_type_name (type));
				}
			} else {
				if (g_str_has_prefix (n, "Gtk")) {
					(vala_type_reference_set_namespace_name (type, "Gtk"), vala_type_reference_get_namespace_name (type));
					(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Gtk", -1))), vala_type_reference_get_type_name (type));
					if (g_utf8_collate (vala_type_reference_get_type_name (type), "TextLayout") == 0) {
						(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
						(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
					}
				} else {
					if (g_str_has_prefix (n, "Gdk")) {
						(vala_type_reference_set_namespace_name (type, "Gdk"), vala_type_reference_get_namespace_name (type));
						(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Gdk", -1))), vala_type_reference_get_type_name (type));
						if (g_utf8_collate (vala_type_reference_get_type_name (type), "NativeWindow") == 0 || g_utf8_collate (vala_type_reference_get_type_name (type), "XEvent") == 0) {
							(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
							(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
						}
					} else {
						if (g_str_has_prefix (n, "Pango")) {
							(vala_type_reference_set_namespace_name (type, "Pango"), vala_type_reference_get_namespace_name (type));
							(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Pango", -1))), vala_type_reference_get_type_name (type));
							if (g_utf8_collate (vala_type_reference_get_type_name (type), "Glyph") == 0) {
								(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
								(vala_type_reference_set_type_name (type, "uint"), vala_type_reference_get_type_name (type));
							} else {
								if (g_utf8_collate (vala_type_reference_get_type_name (type), "GlyphUnit") == 0) {
									(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
									(vala_type_reference_set_type_name (type, "int"), vala_type_reference_get_type_name (type));
								}
							}
						} else {
							if (g_str_has_prefix (n, "Vte")) {
								(vala_type_reference_set_namespace_name (type, "Vte"), vala_type_reference_get_namespace_name (type));
								(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Vte", -1))), vala_type_reference_get_type_name (type));
							} else {
								if (g_str_has_prefix (n, "Goo")) {
									(vala_type_reference_set_namespace_name (type, "Goo"), vala_type_reference_get_namespace_name (type));
									(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Goo", -1))), vala_type_reference_get_type_name (type));
								} else {
									if (g_str_has_prefix (n, "G")) {
										(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
										(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("G", -1))), vala_type_reference_get_type_name (type));
										if (g_utf8_collate (vala_type_reference_get_type_name (type), "Strv") == 0) {
											(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
											(vala_type_reference_set_type_name (type, "string"), vala_type_reference_get_type_name (type));
											(vala_type_reference_set_array_rank (type, 1), vala_type_reference_get_array_rank (type));
										}
									} else {
										char** __temp115;
										gint name_parts_length1;
										char** name_parts;
										__temp115 = NULL;
										name_parts = (__temp115 = g_strsplit (n, ".", 2), name_parts_length1 = -1, __temp115);
										if (name_parts[1] == NULL) {
											(vala_type_reference_set_type_name (type, name_parts[0]), vala_type_reference_get_type_name (type));
										} else {
											(vala_type_reference_set_namespace_name (type, name_parts[0]), vala_type_reference_get_namespace_name (type));
											(vala_type_reference_set_type_name (type, name_parts[1]), vala_type_reference_get_type_name (type));
										}
										(name_parts = (g_free (name_parts), NULL));
									}
								}
							}
						}
					}
				}
			}
		}
	}
}


static ValaTypeReference* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param)
{
	ValaTypeReference* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (param != NULL, NULL);
	type = vala_gidl_parser_parse_type (self, param->type);
	return type;
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


/* disable for now as null_ok not yet correctly set
 type.non_null = !param.null_ok;*/
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_virtual)
{
	GIdlNode* node;
	ValaTypeReference* return_type;
	ValaMethod* m;
	gboolean add_ellipsis;
	char** __temp130;
	gint attributes_length1;
	char** attributes;
	gboolean first;
	ValaFormalParameter* last_param;
	ValaMethod* __temp144;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (f != NULL, NULL);
	node = ((GIdlNode*) f);
	if (f->deprecated) {
		return VALA_METHOD (NULL);
	}
	return_type = NULL;
	if (f->result != NULL) {
		ValaTypeReference* __temp118;
		__temp118 = NULL;
		return_type = (__temp118 = vala_gidl_parser_parse_param (self, f->result), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), __temp118);
	}
	m = NULL;
	if (f->is_constructor || g_str_has_prefix (node->name, "new")) {
		ValaMethod* __temp119;
		__temp119 = NULL;
		m = (__temp119 = VALA_METHOD (vala_creation_method_new (node->name, self->priv->current_source_reference)), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), __temp119);
		if (g_utf8_collate (vala_method_get_name (m), "new") == 0) {
			(vala_method_set_name (m, NULL), vala_method_get_name (m));
		} else {
			if (g_str_has_prefix (vala_method_get_name (m), "new_")) {
				(vala_method_set_name (m, g_utf8_offset_to_pointer (vala_method_get_name (m), g_utf8_strlen ("new_", -1))), vala_method_get_name (m));
			}
		}
	} else {
		ValaMethod* __temp120;
		__temp120 = NULL;
		m = (__temp120 = vala_method_new (node->name, return_type, self->priv->current_source_reference), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), __temp120);
	}
	m->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	(vala_method_set_is_virtual (m, is_virtual), vala_method_get_is_virtual (m));
	/* GIDL generator can't provide array parameter information yet*/
	(vala_method_set_no_array_length (m, TRUE), vala_method_get_no_array_length (m));
	if (self->priv->current_type_symbol_map != NULL) {
		const char* __temp121;
		__temp121 = NULL;
		g_hash_table_insert (self->priv->current_type_symbol_map, (__temp121 = node->name, (__temp121 == NULL ? NULL : g_strdup (__temp121))), g_strdup ("1"));
	}
	if (self->priv->current_data_type != NULL) {
		char** __temp124;
		gint sig_attributes_length1;
		char* __temp122;
		char** __temp123;
		char** sig_attributes;
		__temp124 = NULL;
		__temp122 = NULL;
		__temp123 = NULL;
		sig_attributes = (__temp124 = (__temp123 = vala_gidl_parser_get_attributes (self, (__temp122 = g_strdup_printf ("%s::%s", vala_data_type_get_name (self->priv->current_data_type), node->name))), (__temp122 = (g_free (__temp122), NULL)), __temp123), sig_attributes_length1 = -1, __temp124);
		if (sig_attributes != NULL) {
			{
				char** __temp129;
				int attr_it;
				__temp129 = NULL;
				__temp129 = sig_attributes;
				for (attr_it = 0; (sig_attributes_length1 != -1 && attr_it < sig_attributes_length1) || (sig_attributes_length1 == -1 && __temp129[attr_it] != NULL); attr_it = attr_it + 1) {
					char* attr;
					attr = __temp129[attr_it];
					{
						char** __temp125;
						gint nv_length1;
						char** nv;
						char* __temp126;
						gboolean __temp127;
						__temp125 = NULL;
						nv = (__temp125 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp125);
						__temp126 = NULL;
						if ((__temp127 = g_utf8_collate (nv[0], "has_emitter") == 0 && g_utf8_collate ((__temp126 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp126 = (g_free (__temp126), NULL)), __temp127)) {
							gpointer __temp128;
							return VALA_METHOD ((__temp128 = NULL, (nv = (g_free (nv), NULL)), __temp128));
						}
						(nv = (g_free (nv), NULL));
					}
				}
			}
		}
		(sig_attributes = (g_free (sig_attributes), NULL));
	}
	add_ellipsis = FALSE;
	__temp130 = NULL;
	attributes = (__temp130 = vala_gidl_parser_get_attributes (self, f->symbol), attributes_length1 = -1, __temp130);
	if (attributes != NULL) {
		{
			char** __temp137;
			int attr_it;
			__temp137 = NULL;
			__temp137 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp137[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp137[attr_it];
				{
					char** __temp131;
					gint nv_length1;
					char** nv;
					__temp131 = NULL;
					nv = (__temp131 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp131);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* __temp132;
						gboolean __temp133;
						__temp132 = NULL;
						if ((__temp133 = g_utf8_collate ((__temp132 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp132 = (g_free (__temp132), NULL)), __temp133)) {
							gpointer __temp134;
							return VALA_METHOD ((__temp134 = NULL, (nv = (g_free (nv), NULL)), __temp134));
						}
					} else {
						if (g_utf8_collate (nv[0], "ellipsis") == 0) {
							char* __temp135;
							gboolean __temp136;
							__temp135 = NULL;
							if ((__temp136 = g_utf8_collate ((__temp135 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp135 = (g_free (__temp135), NULL)), __temp136)) {
								add_ellipsis = TRUE;
							}
						}
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	vala_method_set_cname (m, f->symbol);
	first = TRUE;
	last_param = NULL;
	{
		GList* __temp142;
		GList* param_it;
		__temp142 = NULL;
		__temp142 = f->parameters;
		for (param_it = __temp142; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaFormalParameter* p;
				char* __temp138;
				gboolean __temp139;
				ValaFormalParameter* __temp141;
				ValaFormalParameter* __temp140;
				param_node = ((GIdlNode*) param);
				if (first) {
					first = FALSE;
					if (self->priv->current_data_type != NULL && param->type->is_interface && (g_utf8_collate (param_node->name, "self") == 0 || g_str_has_suffix (param->type->interface, vala_data_type_get_name (self->priv->current_data_type)))) {
						/* instance method*/
						if (!vala_data_type_is_reference_type (self->priv->current_data_type) && param->type->is_pointer) {
							(vala_method_set_instance_by_reference (m, TRUE), vala_method_get_instance_by_reference (m));
						}
						continue;
					} else {
						/* static method*/
						(vala_method_set_instance (m, FALSE), vala_method_get_instance (m));
					}
				}
				p = vala_formal_parameter_new (param_node->name, vala_gidl_parser_parse_param (self, param), NULL);
				vala_method_add_parameter (m, p);
				__temp138 = NULL;
				if ((__temp139 = last_param != NULL && g_utf8_collate (vala_formal_parameter_get_name (p), (__temp138 = g_strconcat ("n_", vala_formal_parameter_get_name (last_param), NULL))) == 0, (__temp138 = (g_free (__temp138), NULL)), __temp139)) {
					/* last_param is array, p is array length*/
					(vala_type_reference_set_array_rank (vala_formal_parameter_get_type_reference (last_param), 1), vala_type_reference_get_array_rank (vala_formal_parameter_get_type_reference (last_param)));
					(vala_type_reference_set_is_out (vala_formal_parameter_get_type_reference (last_param), FALSE), vala_type_reference_get_is_out (vala_formal_parameter_get_type_reference (last_param)));
				}
				__temp141 = NULL;
				__temp140 = NULL;
				last_param = (__temp141 = (__temp140 = p, (__temp140 == NULL ? NULL : g_object_ref (__temp140))), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), __temp141);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	if (first) {
		/* no parameters => static method*/
		(vala_method_set_instance (m, FALSE), vala_method_get_instance (m));
	}
	if (add_ellipsis) {
		ValaFormalParameter* __temp143;
		__temp143 = NULL;
		vala_method_add_parameter (m, (__temp143 = vala_formal_parameter_new_with_ellipsis (NULL)));
		(__temp143 == NULL ? NULL : (__temp143 = (g_object_unref (__temp143), NULL)));
	}
	__temp144 = NULL;
	return (__temp144 = m, (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (attributes = (g_free (attributes), NULL)), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), __temp144);
	(return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL)));
	(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	(attributes = (g_free (attributes), NULL));
	(last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL)));
}


static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name)
{
	GString* str;
	const char* __temp145;
	char* i;
	const char* __temp148;
	char* __temp149;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	str = g_string_new ("");
	__temp145 = NULL;
	i = (__temp145 = name, (__temp145 == NULL ? NULL : g_strdup (__temp145)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* __temp147;
		const char* __temp146;
		c = g_utf8_get_char (i);
		if (c == '-') {
			g_string_append_c (str, '_');
		} else {
			g_string_append_unichar (str, c);
		}
		__temp147 = NULL;
		__temp146 = NULL;
		i = (__temp147 = (__temp146 = g_utf8_next_char (i), (__temp146 == NULL ? NULL : g_strdup (__temp146))), (i = (g_free (i), NULL)), __temp147);
	}
	__temp148 = NULL;
	__temp149 = NULL;
	return (__temp149 = (__temp148 = str->str, (__temp148 == NULL ? NULL : g_strdup (__temp148))), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), __temp149);
	(str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL)));
	(i = (g_free (i), NULL));
}


static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node)
{
	GIdlNode* node;
	ValaPropertyAccessor* get_acc;
	ValaPropertyAccessor* set_acc;
	ValaProperty* prop;
	ValaProperty* __temp154;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (prop_node != NULL, NULL);
	node = ((GIdlNode*) prop_node);
	if (prop_node->deprecated) {
		return VALA_PROPERTY (NULL);
	}
	if (!prop_node->readable && !prop_node->writable) {
		/* buggy GIDL definition*/
		prop_node->readable = TRUE;
		prop_node->writable = TRUE;
	}
	get_acc = NULL;
	set_acc = NULL;
	if (prop_node->readable) {
		ValaPropertyAccessor* __temp151;
		__temp151 = NULL;
		get_acc = (__temp151 = vala_property_accessor_new (TRUE, FALSE, FALSE, NULL, NULL), (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), __temp151);
	}
	if (prop_node->writable) {
		ValaPropertyAccessor* __temp152;
		__temp152 = NULL;
		set_acc = (__temp152 = vala_property_accessor_new (FALSE, FALSE, FALSE, NULL, NULL), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), __temp152);
		if (prop_node->construct_only) {
			(vala_property_accessor_set_construction (set_acc, TRUE), vala_property_accessor_get_construction (set_acc));
		} else {
			(vala_property_accessor_set_writable (set_acc, TRUE), vala_property_accessor_get_writable (set_acc));
			(vala_property_accessor_set_construction (set_acc, prop_node->construct), vala_property_accessor_get_construction (set_acc));
		}
	}
	prop = vala_property_new (vala_gidl_parser_fix_prop_name (self, node->name), vala_gidl_parser_parse_type (self, prop_node->type), get_acc, set_acc, self->priv->current_source_reference);
	(vala_property_set_access (prop, VALA_MEMBER_ACCESSIBILITY_PUBLIC), vala_property_get_access (prop));
	(vala_property_set_interface_only (prop, TRUE), vala_property_get_interface_only (prop));
	if (self->priv->current_type_symbol_map != NULL) {
		const char* __temp153;
		__temp153 = NULL;
		g_hash_table_insert (self->priv->current_type_symbol_map, (__temp153 = vala_property_get_name (prop), (__temp153 == NULL ? NULL : g_strdup (__temp153))), g_strdup ("1"));
	}
	__temp154 = NULL;
	return (__temp154 = prop, (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), __temp154);
	(get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL)));
	(set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL)));
	(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
}


static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node)
{
	GIdlNode* node;
	ValaTypeReference* type;
	ValaConstant* c;
	ValaConstant* __temp156;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (const_node != NULL, NULL);
	node = ((GIdlNode*) const_node);
	type = vala_gidl_parser_parse_type (self, const_node->type);
	if (type == NULL) {
		gpointer __temp155;
		return VALA_CONSTANT ((__temp155 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp155));
	}
	c = vala_constant_new (node->name, type, NULL, self->priv->current_source_reference);
	__temp156 = NULL;
	return (__temp156 = c, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp156);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
}


static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node)
{
	GIdlNode* node;
	ValaTypeReference* type;
	ValaField* field;
	ValaField* __temp160;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (field_node != NULL, NULL);
	node = ((GIdlNode*) field_node);
	type = vala_gidl_parser_parse_type (self, field_node->type);
	if (type == NULL) {
		gpointer __temp157;
		return VALA_FIELD ((__temp157 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp157));
	}
	if (!field_node->readable) {
		gpointer __temp158;
		return VALA_FIELD ((__temp158 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp158));
	}
	if (self->priv->current_type_symbol_map != NULL) {
		const char* __temp159;
		__temp159 = NULL;
		g_hash_table_insert (self->priv->current_type_symbol_map, (__temp159 = node->name, (__temp159 == NULL ? NULL : g_strdup (__temp159))), g_strdup ("1"));
	}
	field = vala_field_new (node->name, type, NULL, self->priv->current_source_reference);
	field->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp160 = NULL;
	return (__temp160 = field, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp160);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	(field == NULL ? NULL : (field = (g_object_unref (field), NULL)));
}


static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode)
{
	const char* __temp161;
	char* attributes;
	char** __temp163;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (codenode != NULL, NULL);
	__temp161 = NULL;
	attributes = (__temp161 = g_hash_table_lookup (self->priv->codenode_attributes_map, codenode), (__temp161 == NULL ? NULL : g_strdup (__temp161)));
	if (attributes == NULL) {
		gpointer __temp162;
		return (__temp162 = NULL, (attributes = (g_free (attributes), NULL)), __temp162);
	}
	__temp163 = NULL;
	return (__temp163 = g_strsplit (attributes, " ", 0), (attributes = (g_free (attributes), NULL)), __temp163);
	(attributes = (g_free (attributes), NULL));
}


static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s)
{
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (s != NULL, NULL);
	return g_strndup (g_utf8_offset_to_pointer (s, 1), strlen (s) - 2);
}


static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node)
{
	GIdlNode* node;
	ValaSignal* sig;
	char** __temp168;
	gint attributes_length1;
	char* __temp166;
	char** __temp167;
	char** attributes;
	gboolean first;
	ValaSignal* __temp178;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (sig_node != NULL, NULL);
	node = ((GIdlNode*) sig_node);
	if (sig_node->deprecated || sig_node->result == NULL) {
		return VALA_SIGNAL (NULL);
	}
	sig = vala_signal_new (vala_gidl_parser_fix_prop_name (self, node->name), vala_gidl_parser_parse_param (self, sig_node->result), self->priv->current_source_reference);
	(vala_signal_set_access (sig, VALA_MEMBER_ACCESSIBILITY_PUBLIC), vala_signal_get_access (sig));
	__temp168 = NULL;
	__temp166 = NULL;
	__temp167 = NULL;
	attributes = (__temp168 = (__temp167 = vala_gidl_parser_get_attributes (self, (__temp166 = g_strdup_printf ("%s::%s", vala_data_type_get_name (self->priv->current_data_type), vala_signal_get_name (sig)))), (__temp166 = (g_free (__temp166), NULL)), __temp167), attributes_length1 = -1, __temp168);
	if (attributes != NULL) {
		{
			char** __temp176;
			int attr_it;
			__temp176 = NULL;
			__temp176 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp176[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp176[attr_it];
				{
					char** __temp169;
					gint nv_length1;
					char** nv;
					__temp169 = NULL;
					nv = (__temp169 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp169);
					if (g_utf8_collate (nv[0], "name") == 0) {
						char* __temp170;
						vala_signal_set_cname (sig, vala_signal_get_name (sig));
						__temp170 = NULL;
						(vala_signal_set_name (sig, (__temp170 = vala_gidl_parser_eval (self, nv[1]))), vala_signal_get_name (sig));
						(__temp170 = (g_free (__temp170), NULL));
					} else {
						char* __temp171;
						gboolean __temp172;
						__temp171 = NULL;
						if ((__temp172 = g_utf8_collate (nv[0], "has_emitter") == 0 && g_utf8_collate ((__temp171 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp171 = (g_free (__temp171), NULL)), __temp172)) {
							(vala_signal_set_has_emitter (sig, TRUE), vala_signal_get_has_emitter (sig));
						} else {
							if (g_utf8_collate (nv[0], "hidden") == 0) {
								char* __temp173;
								gboolean __temp174;
								__temp173 = NULL;
								if ((__temp174 = g_utf8_collate ((__temp173 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp173 = (g_free (__temp173), NULL)), __temp174)) {
									gpointer __temp175;
									return VALA_SIGNAL ((__temp175 = NULL, (nv = (g_free (nv), NULL)), __temp175));
								}
							}
						}
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	first = TRUE;
	{
		GList* __temp177;
		GList* param_it;
		__temp177 = NULL;
		__temp177 = sig_node->parameters;
		for (param_it = __temp177; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaFormalParameter* p;
				if (first) {
					/* ignore implicit first signal parameter (sender)*/
					first = FALSE;
					continue;
				}
				param_node = ((GIdlNode*) param);
				p = vala_formal_parameter_new (param_node->name, vala_gidl_parser_parse_param (self, param), NULL);
				vala_signal_add_parameter (sig, p);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	__temp178 = NULL;
	return (__temp178 = sig, (attributes = (g_free (attributes), NULL)), __temp178);
	(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
	(attributes = (g_free (attributes), NULL));
}


static void vala_gidl_parser_class_init (ValaGIdlParserClass * klass)
{
	vala_gidl_parser_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaGIdlParserPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_gidl_parser_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gidl_parser_real_visit_source_file;
}


static void vala_gidl_parser_init (ValaGIdlParser * self)
{
	self->priv = VALA_GIDL_PARSER_GET_PRIVATE (self);
}


static void vala_gidl_parser_dispose (GObject * obj)
{
	ValaGIdlParser * self;
	ValaGIdlParserClass * klass;
	GObjectClass * parent_class;
	self = VALA_GIDL_PARSER (obj);
	(self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL)));
	(self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL)));
	(self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_hash_table_unref (self->priv->codenode_attributes_map), NULL)));
	(self->priv->current_type_symbol_map == NULL ? NULL : (self->priv->current_type_symbol_map = (g_hash_table_unref (self->priv->current_type_symbol_map), NULL)));
	klass = VALA_GIDL_PARSER_CLASS (g_type_class_peek (VALA_TYPE_GIDL_PARSER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_gidl_parser_get_type ()
{
	static GType vala_gidl_parser_type_id = 0;
	if (G_UNLIKELY (vala_gidl_parser_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIdlParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gidl_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIdlParser), 0, (GInstanceInitFunc) vala_gidl_parser_init };
		vala_gidl_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIdlParser", &g_define_type_info, 0);
	}
	return vala_gidl_parser_type_id;
}




