/* valasemanticanalyzer.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valasemanticanalyzer.h"
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaattribute.h>
#include <vala/valastatement.h>
#include <vala/valaarray.h>
#include <vala/valalockable.h>
#include <vala/valatypeparameter.h>
#include <vala/valaliteral.h>
#include <vala/valaenumvalue.h>
#include <vala/valaflagsvalue.h>
#include <vala/valanamespacereference.h>
#include <vala/valapointer.h>
#include <vala/valaenum.h>
#include <vala/valainvokable.h>

struct _ValaSemanticAnalyzerPrivate {
	gboolean _memory_management;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaSourceFile* current_source_file;
	ValaTypeReference* current_return_type;
	ValaClass* current_class;
	ValaStruct* current_struct;
	GList* current_using_directives;
	ValaTypeReference* bool_type;
	ValaTypeReference* string_type;
	ValaTypeReference* int_type;
	ValaTypeReference* uint_type;
	ValaTypeReference* ulong_type;
	ValaTypeReference* type_type;
	ValaDataType* pointer_type;
	ValaDataType* initially_unowned_type;
	ValaDataType* glist_type;
	ValaDataType* gslist_type;
	gint next_lambda_id;
};
#define VALA_SEMANTIC_ANALYZER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerPrivate))
enum  {
	VALA_SEMANTIC_ANALYZER_DUMMY_PROPERTY,
	VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT
};
static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_semantic_analyzer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static GList* vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface);
static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t);
static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_semantic_analyzer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb);
static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_semantic_analyzer_real_visit_begin_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_end_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_semantic_analyzer_real_visit_begin_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_end_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_semantic_analyzer_real_visit_begin_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_end_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static void vala_semantic_analyzer_real_visit_begin_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_end_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr);
static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr);
static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr);
static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr);
static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr);
static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr);
static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_node (ValaSemanticAnalyzer* self, ValaCodeNode* node);
static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static ValaDataType* vala_semantic_analyzer_find_parent_type (ValaSemanticAnalyzer* self, ValaSymbol* sym);
static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type);
static void vala_semantic_analyzer_real_visit_begin_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GList* params, GList* args);
static void vala_semantic_analyzer_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr);
static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr);
static void vala_semantic_analyzer_real_visit_end_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr);
static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr);
static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr);
static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr);
static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr);
static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr);
static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type);
static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr);
static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GList* types);
static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr);
static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self);
static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self);
static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self);
static void vala_semantic_analyzer_real_visit_begin_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_semantic_analyzer_real_visit_begin_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static void vala_semantic_analyzer_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_semantic_analyzer_parent_class = NULL;
static void vala_semantic_analyzer_dispose (GObject * obj);


ValaSemanticAnalyzer* vala_semantic_analyzer_new (gboolean manage_memory)
{
	GParameter * __params;
	GParameter * __params_it;
	ValaSemanticAnalyzer * self;
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	(__params_it->name = "memory-management", g_value_init (&__params_it->value, G_TYPE_BOOLEAN), g_value_set_boolean (&__params_it->value, manage_memory), __params_it++);
	self = g_object_newv (VALA_TYPE_SEMANTIC_ANALYZER, __params_it - __params, __params);
	return self;
}


/**
 * Analyze and check code in the specified context.
 *
 * @param context a code context
 */
void vala_semantic_analyzer_analyze (ValaSemanticAnalyzer* self, ValaCodeContext* context)
{
	ValaSymbol* __temp0;
	ValaTypeReference* __temp1;
	ValaSymbol* __temp2;
	ValaTypeReference* __temp3;
	ValaSymbol* __temp4;
	ValaDataType* __temp7;
	ValaDataType* __temp6;
	ValaSymbol* __temp5;
	ValaTypeReference* __temp8;
	ValaSymbol* __temp9;
	ValaTypeReference* __temp10;
	ValaSymbol* __temp11;
	ValaTypeReference* __temp12;
	ValaSymbol* __temp13;
	ValaSymbol* glib_ns;
	ValaSymbol* __temp26;
	ValaSymbol* __temp25;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	__temp0 = NULL;
	self->priv->root_symbol = (__temp0 = vala_code_context_get_root (context), (self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL))), __temp0);
	__temp1 = NULL;
	self->priv->bool_type = (__temp1 = vala_type_reference_new (), (self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL))), __temp1);
	__temp2 = NULL;
	(vala_type_reference_set_data_type (self->priv->bool_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp2 = vala_symbol_lookup (self->priv->root_symbol, "bool"))))), vala_type_reference_get_data_type (self->priv->bool_type));
	(__temp2 == NULL ? NULL : (__temp2 = (g_object_unref (__temp2), NULL)));
	__temp3 = NULL;
	self->priv->string_type = (__temp3 = vala_type_reference_new (), (self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL))), __temp3);
	__temp4 = NULL;
	(vala_type_reference_set_data_type (self->priv->string_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp4 = vala_symbol_lookup (self->priv->root_symbol, "string"))))), vala_type_reference_get_data_type (self->priv->string_type));
	(__temp4 == NULL ? NULL : (__temp4 = (g_object_unref (__temp4), NULL)));
	__temp7 = NULL;
	__temp6 = NULL;
	__temp5 = NULL;
	self->priv->pointer_type = (__temp7 = (__temp6 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp5 = vala_symbol_lookup (self->priv->root_symbol, "pointer")))), (__temp6 == NULL ? NULL : g_object_ref (__temp6))), (self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL))), __temp7);
	(__temp5 == NULL ? NULL : (__temp5 = (g_object_unref (__temp5), NULL)));
	__temp8 = NULL;
	self->priv->int_type = (__temp8 = vala_type_reference_new (), (self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL))), __temp8);
	__temp9 = NULL;
	(vala_type_reference_set_data_type (self->priv->int_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp9 = vala_symbol_lookup (self->priv->root_symbol, "int"))))), vala_type_reference_get_data_type (self->priv->int_type));
	(__temp9 == NULL ? NULL : (__temp9 = (g_object_unref (__temp9), NULL)));
	__temp10 = NULL;
	self->priv->uint_type = (__temp10 = vala_type_reference_new (), (self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL))), __temp10);
	__temp11 = NULL;
	(vala_type_reference_set_data_type (self->priv->uint_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp11 = vala_symbol_lookup (self->priv->root_symbol, "uint"))))), vala_type_reference_get_data_type (self->priv->uint_type));
	(__temp11 == NULL ? NULL : (__temp11 = (g_object_unref (__temp11), NULL)));
	__temp12 = NULL;
	self->priv->ulong_type = (__temp12 = vala_type_reference_new (), (self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL))), __temp12);
	__temp13 = NULL;
	(vala_type_reference_set_data_type (self->priv->ulong_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp13 = vala_symbol_lookup (self->priv->root_symbol, "ulong"))))), vala_type_reference_get_data_type (self->priv->ulong_type));
	(__temp13 == NULL ? NULL : (__temp13 = (g_object_unref (__temp13), NULL)));
	/* TODO: don't require GLib namespace in semantic analyzer*/
	glib_ns = vala_symbol_lookup (self->priv->root_symbol, "GLib");
	if (glib_ns != NULL) {
		ValaDataType* __temp16;
		ValaDataType* __temp15;
		ValaSymbol* __temp14;
		ValaTypeReference* __temp17;
		ValaSymbol* __temp18;
		ValaDataType* __temp21;
		ValaDataType* __temp20;
		ValaSymbol* __temp19;
		ValaDataType* __temp24;
		ValaDataType* __temp23;
		ValaSymbol* __temp22;
		__temp16 = NULL;
		__temp15 = NULL;
		__temp14 = NULL;
		self->priv->initially_unowned_type = (__temp16 = (__temp15 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp14 = vala_symbol_lookup (glib_ns, "InitiallyUnowned")))), (__temp15 == NULL ? NULL : g_object_ref (__temp15))), (self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL))), __temp16);
		(__temp14 == NULL ? NULL : (__temp14 = (g_object_unref (__temp14), NULL)));
		__temp17 = NULL;
		self->priv->type_type = (__temp17 = vala_type_reference_new (), (self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL))), __temp17);
		__temp18 = NULL;
		(vala_type_reference_set_data_type (self->priv->type_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp18 = vala_symbol_lookup (glib_ns, "Type"))))), vala_type_reference_get_data_type (self->priv->type_type));
		(__temp18 == NULL ? NULL : (__temp18 = (g_object_unref (__temp18), NULL)));
		__temp21 = NULL;
		__temp20 = NULL;
		__temp19 = NULL;
		self->priv->glist_type = (__temp21 = (__temp20 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp19 = vala_symbol_lookup (glib_ns, "List")))), (__temp20 == NULL ? NULL : g_object_ref (__temp20))), (self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL))), __temp21);
		(__temp19 == NULL ? NULL : (__temp19 = (g_object_unref (__temp19), NULL)));
		__temp24 = NULL;
		__temp23 = NULL;
		__temp22 = NULL;
		self->priv->gslist_type = (__temp24 = (__temp23 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp22 = vala_symbol_lookup (glib_ns, "SList")))), (__temp23 == NULL ? NULL : g_object_ref (__temp23))), (self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL))), __temp24);
		(__temp22 == NULL ? NULL : (__temp22 = (g_object_unref (__temp22), NULL)));
	}
	__temp26 = NULL;
	__temp25 = NULL;
	self->priv->current_symbol = (__temp26 = (__temp25 = self->priv->root_symbol, (__temp25 == NULL ? NULL : g_object_ref (__temp25))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp26);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	(glib_ns == NULL ? NULL : (glib_ns = (g_object_unref (glib_ns), NULL)));
}


static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file)
{
	ValaSemanticAnalyzer * self;
	ValaSourceFile* __temp27;
	GList* __temp28;
	GList* __temp29;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	__temp27 = NULL;
	self->priv->current_source_file = (__temp27 = g_object_ref (file), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), __temp27);
	__temp28 = NULL;
	self->priv->current_using_directives = (__temp28 = vala_source_file_get_using_directives (file), (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL))), __temp28);
	self->priv->next_lambda_id = 0;
	vala_source_file_accept_children (file, VALA_CODE_VISITOR (self));
	__temp29 = NULL;
	self->priv->current_using_directives = (__temp29 = NULL, (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL))), __temp29);
}


static void vala_semantic_analyzer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp31;
	ValaSymbol* __temp30;
	ValaSymbol* __temp33;
	ValaSymbol* __temp32;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	__temp31 = NULL;
	__temp30 = NULL;
	self->priv->current_symbol = (__temp31 = (__temp30 = vala_code_node_get_symbol (VALA_CODE_NODE (ns)), (__temp30 == NULL ? NULL : g_object_ref (__temp30))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp31);
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
	__temp33 = NULL;
	__temp32 = NULL;
	self->priv->current_symbol = (__temp33 = (__temp32 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp32 == NULL ? NULL : g_object_ref (__temp32))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp33);
}


static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp35;
	ValaSymbol* __temp34;
	ValaClass* __temp36;
	GList* prerequisites;
	GList* missing_prereqs;
	ValaSymbol* __temp60;
	ValaSymbol* __temp59;
	ValaClass* __temp61;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	__temp35 = NULL;
	__temp34 = NULL;
	self->priv->current_symbol = (__temp35 = (__temp34 = vala_code_node_get_symbol (VALA_CODE_NODE (cl)), (__temp34 == NULL ? NULL : g_object_ref (__temp34))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp35);
	__temp36 = NULL;
	self->priv->current_class = (__temp36 = g_object_ref (cl), (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), __temp36);
	if (vala_class_get_base_class (cl) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_class_get_base_class (cl))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
	}
	{
		GList* __temp37;
		GList* base_type_reference_it;
		__temp37 = NULL;
		__temp37 = vala_class_get_base_types (cl);
		for (base_type_reference_it = __temp37; base_type_reference_it != NULL; base_type_reference_it = base_type_reference_it->next) {
			ValaTypeReference* base_type_reference;
			base_type_reference = base_type_reference_it->data;
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type_reference))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			}
		}
		(__temp37 == NULL ? NULL : (__temp37 = (g_list_free (__temp37), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	/* gather all prerequisites */
	prerequisites = NULL;
	{
		GList* __temp38;
		GList* base_type_it;
		__temp38 = NULL;
		__temp38 = vala_class_get_base_types (cl);
		for (base_type_it = __temp38; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type;
			base_type = base_type_it->data;
			{
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					prerequisites = g_list_concat (prerequisites, vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (vala_type_reference_get_data_type (base_type))));
				}
			}
		}
		(__temp38 == NULL ? NULL : (__temp38 = (g_list_free (__temp38), NULL)));
	}
	/* check whether all prerequisites are met */
	missing_prereqs = NULL;
	{
		GList* __temp39;
		GList* prereq_it;
		__temp39 = NULL;
		__temp39 = prerequisites;
		for (prereq_it = __temp39; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaDataType* prereq;
			prereq = prereq_it->data;
			{
				if (!vala_semantic_analyzer_class_is_a (self, cl, prereq)) {
					missing_prereqs = g_list_prepend (missing_prereqs, vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prereq))));
				}
			}
		}
	}
	/* report any missing prerequisites */
	if (missing_prereqs != NULL) {
		char* __temp40;
		char* __temp41;
		char* error_string;
		gboolean first;
		char* __temp45;
		(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
		__temp40 = NULL;
		__temp41 = NULL;
		error_string = (__temp41 = g_strdup_printf ("%s: some prerequisites (", (__temp40 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl))))), (__temp40 = (g_free (__temp40), NULL)), __temp41);
		first = TRUE;
		{
			GList* __temp44;
			GList* s_it;
			__temp44 = NULL;
			__temp44 = missing_prereqs;
			for (s_it = __temp44; s_it != NULL; s_it = s_it->next) {
				char* s;
				s = s_it->data;
				{
					if (first) {
						char* __temp42;
						__temp42 = NULL;
						error_string = (__temp42 = g_strdup_printf ("%s`%s'", error_string, s), (error_string = (g_free (error_string), NULL)), __temp42);
						first = FALSE;
					} else {
						char* __temp43;
						__temp43 = NULL;
						error_string = (__temp43 = g_strdup_printf ("%s, `%s'", error_string, s), (error_string = (g_free (error_string), NULL)), __temp43);
					}
				}
			}
		}
		__temp45 = NULL;
		error_string = (__temp45 = g_strconcat (error_string, (") are not met"), NULL), (error_string = (g_free (error_string), NULL)), __temp45);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), error_string);
		(error_string = (g_free (error_string), NULL));
	}
	{
		GList* __temp51;
		GList* base_type_it;
		__temp51 = NULL;
		__temp51 = vala_class_get_base_types (cl);
		for (base_type_it = __temp51; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type;
			base_type = base_type_it->data;
			{
				/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					ValaInterface* __temp46;
					ValaInterface* iface;
					__temp46 = NULL;
					iface = (__temp46 = VALA_INTERFACE (vala_type_reference_get_data_type (base_type)), (__temp46 == NULL ? NULL : g_object_ref (__temp46)));
					{
						GList* __temp50;
						GList* m_it;
						__temp50 = NULL;
						__temp50 = vala_interface_get_methods (iface);
						for (m_it = __temp50; m_it != NULL; m_it = m_it->next) {
							ValaMethod* m;
							m = m_it->data;
							{
								/* We do not need to do expensive equality checking here since this is done
								 * already. We only need to guarantee the symbols are present.
								 
								 check methods */
								if (vala_method_get_is_abstract (m)) {
									ValaSymbol* sym;
									sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
									if (sym == NULL || !(VALA_IS_METHOD (vala_symbol_get_node (sym))) || vala_method_get_base_interface_method ((VALA_METHOD (vala_symbol_get_node (sym)))) != m) {
										char* __temp49;
										char* __temp48;
										char* __temp47;
										(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
										__temp49 = NULL;
										__temp48 = NULL;
										__temp47 = NULL;
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (__temp49 = g_strdup_printf ("`%s' does not implement interface method `%s'", (__temp47 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))), (__temp48 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
										(__temp49 = (g_free (__temp49), NULL));
										(__temp48 = (g_free (__temp48), NULL));
										(__temp47 = (g_free (__temp47), NULL));
									}
									(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								}
							}
						}
						(__temp50 == NULL ? NULL : (__temp50 = (g_list_free (__temp50), NULL)));
					}
					(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
				}
			}
		}
		(__temp51 == NULL ? NULL : (__temp51 = (g_list_free (__temp51), NULL)));
	}
	/* all abstract symbols defined in base classes have to be implemented in non-abstract classes
	 * VAPI classes don't have to specify overridden methods
	 */
	if (!vala_class_get_is_abstract (cl) && !vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl))))) {
		ValaClass* __temp52;
		ValaClass* base_class;
		__temp52 = NULL;
		base_class = (__temp52 = vala_class_get_base_class (cl), (__temp52 == NULL ? NULL : g_object_ref (__temp52)));
		while (base_class != NULL && vala_class_get_is_abstract (base_class)) {
			ValaClass* __temp58;
			ValaClass* __temp57;
			{
				GList* __temp56;
				GList* m_it;
				__temp56 = NULL;
				__temp56 = vala_class_get_methods (base_class);
				for (m_it = __temp56; m_it != NULL; m_it = m_it->next) {
					ValaMethod* m;
					m = m_it->data;
					{
						if (vala_method_get_is_abstract (m)) {
							ValaSymbol* sym;
							sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
							if (sym == NULL || !(VALA_IS_METHOD (vala_symbol_get_node (sym))) || vala_method_get_base_method ((VALA_METHOD (vala_symbol_get_node (sym)))) != m) {
								char* __temp55;
								char* __temp54;
								char* __temp53;
								(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
								__temp55 = NULL;
								__temp54 = NULL;
								__temp53 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (__temp55 = g_strdup_printf ("`%s' does not implement abstract method `%s'", (__temp53 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))), (__temp54 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
								(__temp55 = (g_free (__temp55), NULL));
								(__temp54 = (g_free (__temp54), NULL));
								(__temp53 = (g_free (__temp53), NULL));
							}
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
						}
					}
				}
				(__temp56 == NULL ? NULL : (__temp56 = (g_list_free (__temp56), NULL)));
			}
			__temp58 = NULL;
			__temp57 = NULL;
			base_class = (__temp58 = (__temp57 = vala_class_get_base_class (base_class), (__temp57 == NULL ? NULL : g_object_ref (__temp57))), (base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL))), __temp58);
		}
		(base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL)));
	}
	__temp60 = NULL;
	__temp59 = NULL;
	self->priv->current_symbol = (__temp60 = (__temp59 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp59 == NULL ? NULL : g_object_ref (__temp59))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp60);
	__temp61 = NULL;
	self->priv->current_class = (__temp61 = NULL, (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), __temp61);
	(prerequisites == NULL ? NULL : (prerequisites = (g_list_foreach (prerequisites, (GFunc) g_object_unref, NULL), g_list_free (prerequisites), NULL)));
	(missing_prereqs == NULL ? NULL : (missing_prereqs = (g_list_foreach (missing_prereqs, (GFunc) g_free, NULL), g_list_free (missing_prereqs), NULL)));
}


static GList* vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface)
{
	GList* ret;
	GList* __temp65;
	GList* __temp66;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_INTERFACE (iface), NULL);
	ret = NULL;
	{
		GList* __temp64;
		GList* prereq_it;
		__temp64 = NULL;
		__temp64 = vala_interface_get_prerequisites (iface);
		for (prereq_it = __temp64; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaTypeReference* prereq;
			prereq = prereq_it->data;
			{
				ValaDataType* __temp62;
				ValaDataType* type;
				ValaDataType* __temp63;
				__temp62 = NULL;
				type = (__temp62 = vala_type_reference_get_data_type (prereq), (__temp62 == NULL ? NULL : g_object_ref (__temp62)));
				/* skip on previous errors */
				if (type == NULL) {
					continue;
				}
				__temp63 = NULL;
				ret = g_list_prepend (ret, (__temp63 = type, (__temp63 == NULL ? NULL : g_object_ref (__temp63))));
				if (VALA_IS_INTERFACE (type)) {
					ret = g_list_concat (ret, vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (type)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(__temp64 == NULL ? NULL : (__temp64 = (g_list_free (__temp64), NULL)));
	}
	ret = g_list_reverse (ret);
	__temp65 = NULL;
	__temp66 = NULL;
	return (__temp66 = (__temp65 = ret, ret = NULL, __temp65), (ret == NULL ? NULL : (ret = (g_list_foreach (ret, (GFunc) g_object_unref, NULL), g_list_free (ret), NULL))), __temp66);
	(ret == NULL ? NULL : (ret = (g_list_foreach (ret, (GFunc) g_object_unref, NULL), g_list_free (ret), NULL)));
}


static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_CLASS (cl), FALSE);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	if (VALA_DATA_TYPE (cl) == t) {
		return TRUE;
	}
	{
		GList* __temp70;
		GList* base_type_it;
		__temp70 = NULL;
		__temp70 = vala_class_get_base_types (cl);
		for (base_type_it = __temp70; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type;
			base_type = base_type_it->data;
			{
				if (VALA_IS_CLASS (vala_type_reference_get_data_type (base_type))) {
					if (vala_semantic_analyzer_class_is_a (self, VALA_CLASS (vala_type_reference_get_data_type (base_type)), t)) {
						return TRUE;
					}
				} else {
					if (vala_type_reference_get_data_type (base_type) == t) {
						return TRUE;
					}
				}
			}
		}
		(__temp70 == NULL ? NULL : (__temp70 = (g_list_free (__temp70), NULL)));
	}
	return FALSE;
}


static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp73;
	ValaSymbol* __temp72;
	ValaStruct* __temp74;
	ValaSymbol* __temp76;
	ValaSymbol* __temp75;
	ValaStruct* __temp77;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	__temp73 = NULL;
	__temp72 = NULL;
	self->priv->current_symbol = (__temp73 = (__temp72 = vala_code_node_get_symbol (VALA_CODE_NODE (st)), (__temp72 == NULL ? NULL : g_object_ref (__temp72))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp73);
	__temp74 = NULL;
	self->priv->current_struct = (__temp74 = g_object_ref (st), (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), __temp74);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	__temp76 = NULL;
	__temp75 = NULL;
	self->priv->current_symbol = (__temp76 = (__temp75 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp75 == NULL ? NULL : g_object_ref (__temp75))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp76);
	__temp77 = NULL;
	self->priv->current_struct = (__temp77 = NULL, (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), __temp77);
}


static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp79;
	ValaSymbol* __temp78;
	ValaClass* prereq_class;
	ValaSymbol* __temp90;
	ValaSymbol* __temp89;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	__temp79 = NULL;
	__temp78 = NULL;
	self->priv->current_symbol = (__temp79 = (__temp78 = vala_code_node_get_symbol (VALA_CODE_NODE (iface)), (__temp78 == NULL ? NULL : g_object_ref (__temp78))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp79);
	{
		GList* __temp80;
		GList* prerequisite_reference_it;
		__temp80 = NULL;
		__temp80 = vala_interface_get_prerequisites (iface);
		for (prerequisite_reference_it = __temp80; prerequisite_reference_it != NULL; prerequisite_reference_it = prerequisite_reference_it->next) {
			ValaTypeReference* prerequisite_reference;
			prerequisite_reference = prerequisite_reference_it->data;
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (prerequisite_reference))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			}
		}
		(__temp80 == NULL ? NULL : (__temp80 = (g_list_free (__temp80), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	/* check prerequisites */
	prereq_class = NULL;
	{
		GList* __temp88;
		GList* prereq_it;
		__temp88 = NULL;
		__temp88 = vala_interface_get_prerequisites (iface);
		for (prereq_it = __temp88; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaTypeReference* prereq;
			prereq = prereq_it->data;
			{
				ValaDataType* __temp81;
				ValaDataType* class_or_interface;
				__temp81 = NULL;
				class_or_interface = (__temp81 = vala_type_reference_get_data_type (prereq), (__temp81 == NULL ? NULL : g_object_ref (__temp81)));
				/* skip on previous errors */
				if (class_or_interface == NULL) {
					(vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE), vala_code_node_get_error (VALA_CODE_NODE (iface)));
					continue;
				}
				/* interfaces are not allowed to have multiple instantiable prerequisites */
				if (VALA_IS_CLASS (class_or_interface)) {
					ValaClass* __temp87;
					ValaClass* __temp86;
					if (prereq_class != NULL) {
						char* __temp85;
						char* __temp84;
						char* __temp83;
						char* __temp82;
						(vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE), vala_code_node_get_error (VALA_CODE_NODE (iface)));
						__temp85 = NULL;
						__temp84 = NULL;
						__temp83 = NULL;
						__temp82 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (iface)), (__temp85 = g_strdup_printf ("%s: Interfaces cannot have multiple instantiable prerequisites (`%s' and `%s')", (__temp82 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (iface)))), (__temp83 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (class_or_interface)))), (__temp84 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prereq_class)))))));
						(__temp85 = (g_free (__temp85), NULL));
						(__temp84 = (g_free (__temp84), NULL));
						(__temp83 = (g_free (__temp83), NULL));
						(__temp82 = (g_free (__temp82), NULL));
						(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
						(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
						return;
					}
					__temp87 = NULL;
					__temp86 = NULL;
					prereq_class = (__temp87 = (__temp86 = VALA_CLASS (class_or_interface), (__temp86 == NULL ? NULL : g_object_ref (__temp86))), (prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL))), __temp87);
				}
				(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
			}
		}
		(__temp88 == NULL ? NULL : (__temp88 = (g_list_free (__temp88), NULL)));
	}
	__temp90 = NULL;
	__temp89 = NULL;
	self->priv->current_symbol = (__temp90 = (__temp89 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp89 == NULL ? NULL : g_object_ref (__temp89))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp90);
	(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
}


static void vala_semantic_analyzer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	vala_code_node_accept_children (VALA_CODE_NODE (cb), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	if (!vala_source_file_get_pkg (self->priv->current_source_file)) {
		if (vala_constant_get_initializer (c) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (c), TRUE), vala_code_node_get_error (VALA_CODE_NODE (c)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (c)), "A const field requires a initializer to be provided");
		}
	}
}


static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	vala_code_node_accept_children (VALA_CODE_NODE (f), VALA_CODE_VISITOR (self));
	if (f->access != VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
	} else {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
}


static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp92;
	ValaSymbol* __temp91;
	ValaTypeReference* __temp94;
	ValaTypeReference* __temp93;
	ValaAttribute* init_attr;
	ValaSymbol* __temp96;
	ValaSymbol* __temp95;
	ValaTypeReference* __temp97;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	__temp92 = NULL;
	__temp91 = NULL;
	self->priv->current_symbol = (__temp92 = (__temp91 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp91 == NULL ? NULL : g_object_ref (__temp91))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp92);
	__temp94 = NULL;
	__temp93 = NULL;
	self->priv->current_return_type = (__temp94 = (__temp93 = vala_method_get_return_type (m), (__temp93 == NULL ? NULL : g_object_ref (__temp93))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp94);
	init_attr = vala_code_node_get_attribute (VALA_CODE_NODE (m), "ModuleInit");
	if (init_attr != NULL) {
		(vala_code_context_set_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m)))), m), vala_code_context_get_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m))))));
	}
	if (vala_type_reference_get_data_type (vala_method_get_return_type (m)) != NULL) {
		/* is null if it is void or a reference to a type parameter */
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_method_get_return_type (m)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
	}
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	__temp96 = NULL;
	__temp95 = NULL;
	self->priv->current_symbol = (__temp96 = (__temp95 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp95 == NULL ? NULL : g_object_ref (__temp95))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp96);
	__temp97 = NULL;
	self->priv->current_return_type = (__temp97 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp97);
	if (vala_symbol_get_parent_symbol (self->priv->current_symbol) != NULL && VALA_IS_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol)))) {
		ValaMethod* __temp98;
		ValaMethod* up_method;
		ValaTypeReference* __temp100;
		ValaTypeReference* __temp99;
		/* lambda expressions produce nested methods */
		__temp98 = NULL;
		up_method = (__temp98 = VALA_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol))), (__temp98 == NULL ? NULL : g_object_ref (__temp98)));
		__temp100 = NULL;
		__temp99 = NULL;
		self->priv->current_return_type = (__temp100 = (__temp99 = vala_method_get_return_type (up_method), (__temp99 == NULL ? NULL : g_object_ref (__temp99))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp100);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (self->priv->current_symbol))) {
		if (!(VALA_IS_CREATION_METHOD (m))) {
			vala_semantic_analyzer_find_base_interface_method (self, m, VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
			if (vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				vala_semantic_analyzer_find_base_class_method (self, m, VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
				if (vala_method_get_base_method (m) == NULL) {
					char* __temp102;
					char* __temp101;
					__temp102 = NULL;
					__temp101 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp102 = g_strdup_printf ("%s: no suitable method found to override", (__temp101 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
					(__temp102 = (g_free (__temp102), NULL));
					(__temp101 = (g_free (__temp101), NULL));
				}
			}
		}
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_node (self->priv->current_symbol))) {
			if (vala_method_get_is_abstract (m) || vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				char* __temp104;
				char* __temp103;
				__temp104 = NULL;
				__temp103 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp104 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", (__temp103 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
				(__temp104 = (g_free (__temp104), NULL));
				(__temp103 = (g_free (__temp103), NULL));
				(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
				return;
			}
		}
	}
	(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
}


static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl)
{
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
	if (sym != NULL && VALA_IS_METHOD (vala_symbol_get_node (sym))) {
		ValaMethod* __temp105;
		ValaMethod* base_method;
		__temp105 = NULL;
		base_method = (__temp105 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp105 == NULL ? NULL : g_object_ref (__temp105)));
		if (vala_method_get_is_abstract (base_method) || vala_method_get_is_virtual (base_method)) {
			if (!vala_method_equals (m, base_method)) {
				char* __temp108;
				char* __temp107;
				char* __temp106;
				(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
				__temp108 = NULL;
				__temp107 = NULL;
				__temp106 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp108 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (__temp106 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp107 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_method)))))));
				(__temp108 = (g_free (__temp108), NULL));
				(__temp107 = (g_free (__temp107), NULL));
				(__temp106 = (g_free (__temp106), NULL));
				(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			(vala_method_set_base_method (m, base_method), vala_method_get_base_method (m));
			(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_method (self, m, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GList* __temp113;
		GList* type_it;
		__temp113 = NULL;
		__temp113 = vala_class_get_base_types (cl);
		for (type_it = __temp113; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type;
			type = type_it->data;
			{
				/* FIXME report error if multiple possible base methods are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type))), vala_method_get_name (m));
					if (sym != NULL && VALA_IS_METHOD (vala_symbol_get_node (sym))) {
						ValaMethod* __temp109;
						ValaMethod* base_method;
						__temp109 = NULL;
						base_method = (__temp109 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp109 == NULL ? NULL : g_object_ref (__temp109)));
						if (vala_method_get_is_abstract (base_method)) {
							if (!vala_method_equals (m, base_method)) {
								char* __temp112;
								char* __temp111;
								char* __temp110;
								(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
								__temp112 = NULL;
								__temp111 = NULL;
								__temp110 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp112 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (__temp110 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp111 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_method)))))));
								(__temp112 = (g_free (__temp112), NULL));
								(__temp111 = (g_free (__temp111), NULL));
								(__temp110 = (g_free (__temp110), NULL));
								(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								return;
							}
							(vala_method_set_base_interface_method (m, base_method), vala_method_get_base_interface_method (m));
							(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							return;
						}
						(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
			}
		}
		(__temp113 == NULL ? NULL : (__temp113 = (g_list_free (__temp113), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp114;
	ValaSymbol* __temp119;
	ValaSymbol* __temp118;
	ValaTypeReference* __temp121;
	ValaTypeReference* __temp120;
	ValaSymbol* __temp123;
	ValaSymbol* __temp122;
	ValaTypeReference* __temp124;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	__temp114 = NULL;
	(vala_method_set_return_type (VALA_METHOD (m), (__temp114 = vala_type_reference_new ())), vala_method_get_return_type (VALA_METHOD (m)));
	(__temp114 == NULL ? NULL : (__temp114 = (g_object_unref (__temp114), NULL)));
	(vala_type_reference_set_data_type (vala_method_get_return_type (VALA_METHOD (m)), VALA_DATA_TYPE (vala_symbol_get_node (self->priv->current_symbol))), vala_type_reference_get_data_type (vala_method_get_return_type (VALA_METHOD (m))));
	(vala_type_reference_set_transfers_ownership (vala_method_get_return_type (VALA_METHOD (m)), TRUE), vala_type_reference_get_transfers_ownership (vala_method_get_return_type (VALA_METHOD (m))));
	if (VALA_IS_CLASS (vala_symbol_get_node (self->priv->current_symbol))) {
		ValaClass* __temp115;
		ValaClass* cl;
		/* check for floating reference*/
		__temp115 = NULL;
		cl = (__temp115 = VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)), (__temp115 == NULL ? NULL : g_object_ref (__temp115)));
		while (cl != NULL) {
			ValaClass* __temp117;
			ValaClass* __temp116;
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				(vala_type_reference_set_floating_reference (vala_method_get_return_type (VALA_METHOD (m)), TRUE), vala_type_reference_get_floating_reference (vala_method_get_return_type (VALA_METHOD (m))));
				break;
			}
			__temp117 = NULL;
			__temp116 = NULL;
			cl = (__temp117 = (__temp116 = vala_class_get_base_class (cl), (__temp116 == NULL ? NULL : g_object_ref (__temp116))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp117);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
	__temp119 = NULL;
	__temp118 = NULL;
	self->priv->current_symbol = (__temp119 = (__temp118 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp118 == NULL ? NULL : g_object_ref (__temp118))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp119);
	__temp121 = NULL;
	__temp120 = NULL;
	self->priv->current_return_type = (__temp121 = (__temp120 = vala_method_get_return_type (VALA_METHOD (m)), (__temp120 == NULL ? NULL : g_object_ref (__temp120))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp121);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	__temp123 = NULL;
	__temp122 = NULL;
	self->priv->current_symbol = (__temp123 = (__temp122 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp122 == NULL ? NULL : g_object_ref (__temp122))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp123);
	__temp124 = NULL;
	self->priv->current_return_type = (__temp124 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp124);
	if (vala_symbol_get_parent_symbol (self->priv->current_symbol) != NULL && VALA_IS_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol)))) {
		ValaMethod* __temp125;
		ValaMethod* up_method;
		ValaTypeReference* __temp127;
		ValaTypeReference* __temp126;
		/* lambda expressions produce nested methods */
		__temp125 = NULL;
		up_method = (__temp125 = VALA_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol))), (__temp125 == NULL ? NULL : g_object_ref (__temp125)));
		__temp127 = NULL;
		__temp126 = NULL;
		self->priv->current_return_type = (__temp127 = (__temp126 = vala_method_get_return_type (up_method), (__temp126 == NULL ? NULL : g_object_ref (__temp126))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp127);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (self->priv->current_symbol))) {
		if (!(VALA_IS_CREATION_METHOD (m))) {
			vala_semantic_analyzer_find_base_interface_method (self, VALA_METHOD (m), VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
			if (vala_method_get_is_virtual (VALA_METHOD (m)) || vala_method_get_overrides (VALA_METHOD (m))) {
				vala_semantic_analyzer_find_base_class_method (self, VALA_METHOD (m), VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
				if (vala_method_get_base_method (VALA_METHOD (m)) == NULL) {
					char* __temp129;
					char* __temp128;
					__temp129 = NULL;
					__temp128 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp129 = g_strdup_printf ("%s: no suitable method found to override", (__temp128 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
					(__temp129 = (g_free (__temp129), NULL));
					(__temp128 = (g_free (__temp128), NULL));
				}
			}
		}
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_node (self->priv->current_symbol))) {
			if (vala_method_get_is_abstract (VALA_METHOD (m)) || vala_method_get_is_virtual (VALA_METHOD (m)) || vala_method_get_overrides (VALA_METHOD (m))) {
				char* __temp131;
				char* __temp130;
				__temp131 = NULL;
				__temp130 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp131 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", (__temp130 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
				(__temp131 = (g_free (__temp131), NULL));
				(__temp130 = (g_free (__temp130), NULL));
				return;
			}
		}
	}
	if (vala_method_get_body (VALA_METHOD (m)) != NULL && self->priv->current_class != NULL) {
		gint n_params;
		n_params = 0;
		{
			GList* __temp132;
			GList* stmt_it;
			__temp132 = NULL;
			__temp132 = vala_block_get_statements (vala_method_get_body (VALA_METHOD (m)));
			for (stmt_it = __temp132; stmt_it != NULL; stmt_it = stmt_it->next) {
				ValaStatement* stmt;
				stmt = stmt_it->data;
				{
					gint params;
					params = vala_statement_get_number_of_set_construction_parameters (stmt);
					if (params == -1) {
						(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "class creation methods only allow property assignment statements");
						return;
					}
					n_params = n_params + (params);
				}
			}
			(__temp132 == NULL ? NULL : (__temp132 = (g_list_free (__temp132), NULL)));
		}
		(vala_creation_method_set_n_construction_params (m, n_params), vala_creation_method_get_n_construction_params (m));
	}
}


static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (p));
	vala_code_node_accept_children (VALA_CODE_NODE (p), VALA_CODE_VISITOR (self));
	if (!vala_formal_parameter_get_ellipsis (p)) {
		if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
	/* special treatment for construct formal parameters used in creation methods */
	if (vala_formal_parameter_get_construct_parameter (p)) {
		ValaBlock* __temp133;
		ValaBlock* method_body;
		ValaMemberAccess* left;
		ValaMemberAccess* right;
		ValaSymbol* prop_sym;
		ValaExpressionStatement* __temp136;
		if (!(VALA_IS_CREATION_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (p))))))) {
			(vala_code_node_set_error (VALA_CODE_NODE (p), TRUE), vala_code_node_get_error (VALA_CODE_NODE (p)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), "construct parameters are only allowed in type creation methods");
			return;
		}
		__temp133 = NULL;
		method_body = (__temp133 = vala_method_get_body (VALA_METHOD ((VALA_CREATION_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (p)))))))), (__temp133 == NULL ? NULL : g_object_ref (__temp133)));
		left = vala_member_access_new_simple (vala_formal_parameter_get_name (p), NULL);
		right = vala_member_access_new_simple (vala_formal_parameter_get_name (p), NULL);
		/* try to lookup the requested property */
		prop_sym = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (self->priv->current_class)), vala_formal_parameter_get_name (p));
		if (!(VALA_IS_PROPERTY (vala_symbol_get_node (prop_sym)))) {
			char* __temp135;
			char* __temp134;
			(vala_code_node_set_error (VALA_CODE_NODE (p), TRUE), vala_code_node_get_error (VALA_CODE_NODE (p)));
			__temp135 = NULL;
			__temp134 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), (__temp135 = g_strdup_printf ("class `%s' does not contain a property named `%s'", (__temp134 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self->priv->current_class)))), vala_formal_parameter_get_name (p))));
			(__temp135 = (g_free (__temp135), NULL));
			(__temp134 = (g_free (__temp134), NULL));
			(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
			(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
			(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
			(prop_sym == NULL ? NULL : (prop_sym = (g_object_unref (prop_sym), NULL)));
			return;
		}
		(vala_expression_set_symbol_reference (VALA_EXPRESSION (left), prop_sym), vala_expression_get_symbol_reference (VALA_EXPRESSION (left)));
		(vala_expression_set_symbol_reference (VALA_EXPRESSION (right), vala_code_node_get_symbol (VALA_CODE_NODE (p))), vala_expression_get_symbol_reference (VALA_EXPRESSION (right)));
		__temp136 = NULL;
		vala_block_add_statement (method_body, VALA_STATEMENT ((__temp136 = vala_expression_statement_new (VALA_EXPRESSION (vala_assignment_new (VALA_EXPRESSION (left), VALA_EXPRESSION (right), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL)), NULL))));
		(__temp136 == NULL ? NULL : (__temp136 = (g_object_unref (__temp136), NULL)));
		(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
		(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
		(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
		(prop_sym == NULL ? NULL : (prop_sym = (g_object_unref (prop_sym), NULL)));
	}
}


static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl)
{
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_property_get_name (prop));
	if (sym != NULL && VALA_IS_PROPERTY (vala_symbol_get_node (sym))) {
		ValaProperty* __temp137;
		ValaProperty* base_property;
		__temp137 = NULL;
		base_property = (__temp137 = VALA_PROPERTY (vala_symbol_get_node (sym)), (__temp137 == NULL ? NULL : g_object_ref (__temp137)));
		if (vala_property_get_is_abstract (base_property) || vala_property_get_is_virtual (base_property)) {
			if (!vala_property_equals (prop, base_property)) {
				char* __temp140;
				char* __temp139;
				char* __temp138;
				(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
				__temp140 = NULL;
				__temp139 = NULL;
				__temp138 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp140 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (__temp138 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))), (__temp139 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_property)))))));
				(__temp140 = (g_free (__temp140), NULL));
				(__temp139 = (g_free (__temp139), NULL));
				(__temp138 = (g_free (__temp138), NULL));
				(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			(vala_property_set_base_property (prop, base_property), vala_property_get_base_property (prop));
			(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_property (self, prop, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GList* __temp145;
		GList* type_it;
		__temp145 = NULL;
		__temp145 = vala_class_get_base_types (cl);
		for (type_it = __temp145; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type;
			type = type_it->data;
			{
				/* FIXME report error if multiple possible base properties are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type))), vala_property_get_name (prop));
					if (sym != NULL && VALA_IS_PROPERTY (vala_symbol_get_node (sym))) {
						ValaProperty* __temp141;
						ValaProperty* base_property;
						__temp141 = NULL;
						base_property = (__temp141 = VALA_PROPERTY (vala_symbol_get_node (sym)), (__temp141 == NULL ? NULL : g_object_ref (__temp141)));
						if (vala_property_get_is_abstract (base_property)) {
							if (!vala_property_equals (prop, base_property)) {
								char* __temp144;
								char* __temp143;
								char* __temp142;
								(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
								__temp144 = NULL;
								__temp143 = NULL;
								__temp142 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp144 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (__temp142 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))), (__temp143 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_property)))))));
								(__temp144 = (g_free (__temp144), NULL));
								(__temp143 = (g_free (__temp143), NULL));
								(__temp142 = (g_free (__temp142), NULL));
								(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								return;
							}
							(vala_property_set_base_interface_property (prop, base_property), vala_property_get_base_interface_property (prop));
							(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							return;
						}
						(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
			}
		}
		(__temp145 == NULL ? NULL : (__temp145 = (g_list_free (__temp145), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	vala_code_node_accept_children (VALA_CODE_NODE (prop), VALA_CODE_VISITOR (self));
	if (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)) != NULL) {
		/* is null if it references a type parameter */
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))) {
		ValaClass* __temp146;
		ValaClass* cl;
		__temp146 = NULL;
		cl = (__temp146 = VALA_CLASS (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (prop))))), (__temp146 == NULL ? NULL : g_object_ref (__temp146)));
		vala_semantic_analyzer_find_base_interface_property (self, prop, cl);
		if (vala_property_get_is_virtual (prop) || vala_property_get_overrides (prop)) {
			vala_semantic_analyzer_find_base_class_property (self, prop, cl);
			if (vala_property_get_base_property (prop) == NULL) {
				char* __temp148;
				char* __temp147;
				(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
				__temp148 = NULL;
				__temp147 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp148 = g_strdup_printf ("%s: no suitable property found to override", (__temp147 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))));
				(__temp148 = (g_free (__temp148), NULL));
				(__temp147 = (g_free (__temp147), NULL));
			}
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc)
{
	ValaSemanticAnalyzer * self;
	ValaProperty* __temp149;
	ValaProperty* prop;
	ValaTypeReference* __temp152;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
	__temp149 = NULL;
	prop = (__temp149 = VALA_PROPERTY (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (acc))))), (__temp149 == NULL ? NULL : g_object_ref (__temp149)));
	if (vala_property_accessor_get_readable (acc)) {
		ValaTypeReference* __temp150;
		__temp150 = NULL;
		self->priv->current_return_type = (__temp150 = g_object_ref (vala_property_get_type_reference (prop)), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp150);
	} else {
		ValaTypeReference* __temp151;
		/* void*/
		__temp151 = NULL;
		self->priv->current_return_type = (__temp151 = vala_type_reference_new (), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp151);
	}
	vala_code_node_accept_children (VALA_CODE_NODE (acc), VALA_CODE_VISITOR (self));
	__temp152 = NULL;
	self->priv->current_return_type = (__temp152 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp152);
	(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
}


static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	vala_code_node_accept_children (VALA_CODE_NODE (sig), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp154;
	ValaSymbol* __temp153;
	ValaSymbol* __temp156;
	ValaSymbol* __temp155;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
	__temp154 = NULL;
	__temp153 = NULL;
	self->priv->current_symbol = (__temp154 = (__temp153 = vala_code_node_get_symbol (VALA_CODE_NODE (c)), (__temp153 == NULL ? NULL : g_object_ref (__temp153))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp154);
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	__temp156 = NULL;
	__temp155 = NULL;
	self->priv->current_symbol = (__temp156 = (__temp155 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp155 == NULL ? NULL : g_object_ref (__temp155))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp156);
}


static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp158;
	ValaSymbol* __temp157;
	ValaSymbol* __temp160;
	ValaSymbol* __temp159;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
	__temp158 = NULL;
	__temp157 = NULL;
	self->priv->current_symbol = (__temp158 = (__temp157 = vala_code_node_get_symbol (VALA_CODE_NODE (d)), (__temp157 == NULL ? NULL : g_object_ref (__temp157))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp158);
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
	__temp160 = NULL;
	__temp159 = NULL;
	self->priv->current_symbol = (__temp160 = (__temp159 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp159 == NULL ? NULL : g_object_ref (__temp159))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp160);
}


static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (n));
}


static void vala_semantic_analyzer_real_visit_begin_block (ValaCodeVisitor* base, ValaBlock* b)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp162;
	ValaSymbol* __temp161;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	__temp162 = NULL;
	__temp161 = NULL;
	self->priv->current_symbol = (__temp162 = (__temp161 = vala_code_node_get_symbol (VALA_CODE_NODE (b)), (__temp161 == NULL ? NULL : g_object_ref (__temp161))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp162);
}


static void vala_semantic_analyzer_real_visit_end_block (ValaCodeVisitor* base, ValaBlock* b)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp165;
	ValaSymbol* __temp164;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	{
		GList* __temp163;
		GList* decl_it;
		__temp163 = NULL;
		__temp163 = vala_block_get_local_variables (b);
		for (decl_it = __temp163; decl_it != NULL; decl_it = decl_it->next) {
			ValaVariableDeclarator* decl;
			decl = decl_it->data;
			{
				(vala_symbol_set_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl)), FALSE), vala_symbol_get_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl))));
			}
		}
		(__temp163 == NULL ? NULL : (__temp163 = (g_list_free (__temp163), NULL)));
	}
	__temp165 = NULL;
	__temp164 = NULL;
	self->priv->current_symbol = (__temp165 = (__temp164 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp164 == NULL ? NULL : g_object_ref (__temp164))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp165);
}


static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp172;
	ValaBlock* __temp173;
	ValaBlock* block;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
	if (vala_variable_declarator_get_type_reference (decl) == NULL) {
		ValaTypeReference* __temp166;
		/* var type */
		if (vala_variable_declarator_get_initializer (decl) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed without initializer");
			return;
		}
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed with non-typed initializer");
			return;
		}
		__temp166 = NULL;
		(vala_variable_declarator_set_type_reference (decl, (__temp166 = vala_type_reference_copy (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl))))), vala_variable_declarator_get_type_reference (decl));
		(__temp166 = (g_object_unref (__temp166), NULL));
		(vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (decl), (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) == NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))))), vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl)));
		(vala_type_reference_set_transfers_ownership (vala_variable_declarator_get_type_reference (decl), FALSE), vala_type_reference_get_transfers_ownership (vala_variable_declarator_get_type_reference (decl)));
	}
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			if (!(VALA_IS_MEMBER_ACCESS (vala_variable_declarator_get_initializer (decl)))) {
				(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
			if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
				ValaMethod* __temp167;
				ValaMethod* m;
				ValaCallback* __temp168;
				ValaCallback* cb;
				__temp167 = NULL;
				m = (__temp167 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl)))), (__temp167 == NULL ? NULL : g_object_ref (__temp167)));
				__temp168 = NULL;
				cb = (__temp168 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp168 == NULL ? NULL : g_object_ref (__temp168)));
				/* check whether method matches callback type */
				if (!vala_callback_matches_method (cb, m)) {
					char* __temp171;
					char* __temp170;
					char* __temp169;
					(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
					__temp171 = NULL;
					__temp170 = NULL;
					__temp169 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), (__temp171 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp169 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp170 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
					(__temp171 = (g_free (__temp171), NULL));
					(__temp170 = (g_free (__temp170), NULL));
					(__temp169 = (g_free (__temp169), NULL));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					return;
				}
				(vala_expression_set_static_type (vala_variable_declarator_get_initializer (decl), vala_variable_declarator_get_type_reference (decl)), vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			} else {
				(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
		}
		if (vala_semantic_analyzer_get_memory_management (self)) {
			if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))) {
				/* rhs transfers ownership of the expression */
				if (!vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
					/* lhs doesn't own the value */
					(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "Invalid assignment from owned expression to unowned variable");
					return;
				}
			}
		}
	}
	if (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	__temp172 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (decl), (__temp172 = vala_symbol_new (VALA_CODE_NODE (decl)))), vala_code_node_get_symbol (VALA_CODE_NODE (decl)));
	(__temp172 == NULL ? NULL : (__temp172 = (g_object_unref (__temp172), NULL)));
	vala_symbol_add (self->priv->current_symbol, vala_variable_declarator_get_name (decl), vala_code_node_get_symbol (VALA_CODE_NODE (decl)));
	__temp173 = NULL;
	block = (__temp173 = VALA_BLOCK (vala_symbol_get_node (self->priv->current_symbol)), (__temp173 == NULL ? NULL : g_object_ref (__temp173)));
	vala_block_add_local_variable (block, decl);
	(vala_symbol_set_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl)), TRUE), vala_symbol_get_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl))));
	(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_begin_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
		ValaArray* __temp174;
		ValaArray* edt;
		GList* inits;
		gint rank;
		ValaTypeReference* child_type;
		/* initializer is used as array initializer */
		__temp174 = NULL;
		edt = (__temp174 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (__temp174 == NULL ? NULL : g_object_ref (__temp174)));
		inits = vala_initializer_list_get_initializers (list);
		rank = vala_array_get_rank ((VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))));
		child_type = vala_type_reference_copy (vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		if (rank > 1) {
			ValaArray* __temp175;
			__temp175 = NULL;
			(vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((__temp175 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1)))), vala_type_reference_get_data_type (child_type));
			(__temp175 = (g_object_unref (__temp175), NULL));
		} else {
			(vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt)), vala_type_reference_get_data_type (child_type));
		}
		{
			GList* __temp177;
			GList* e_it;
			__temp177 = NULL;
			__temp177 = inits;
			for (e_it = __temp177; e_it != NULL; e_it = e_it->next) {
				ValaExpression* e;
				e = e_it->data;
				{
					ValaTypeReference* __temp176;
					__temp176 = NULL;
					(vala_expression_set_expected_type (e, (__temp176 = vala_type_reference_copy (child_type))), vala_expression_get_expected_type (e));
					(__temp176 = (g_object_unref (__temp176), NULL));
				}
			}
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_list_free (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_end_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
		ValaArray* __temp178;
		ValaArray* edt;
		GList* inits;
		gint rank;
		ValaTypeReference* child_type;
		gboolean error;
		__temp178 = NULL;
		edt = (__temp178 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (__temp178 == NULL ? NULL : g_object_ref (__temp178)));
		inits = vala_initializer_list_get_initializers (list);
		rank = vala_array_get_rank (edt);
		child_type = vala_type_reference_copy (vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		error = FALSE;
		if (rank > 1) {
			ValaArray* __temp179;
			__temp179 = NULL;
			(vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((__temp179 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1)))), vala_type_reference_get_data_type (child_type));
			(__temp179 = (g_object_unref (__temp179), NULL));
			{
				GList* __temp181;
				GList* e_it;
				__temp181 = NULL;
				__temp181 = inits;
				for (e_it = __temp181; e_it != NULL; e_it = e_it->next) {
					ValaExpression* e;
					e = e_it->data;
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							continue;
						}
						if (!(VALA_IS_INITIALIZER_LIST (e))) {
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Initializer list expected");
							continue;
						}
						if (!vala_type_reference_equals (vala_expression_get_static_type (e), child_type)) {
							char* __temp180;
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							__temp180 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (__temp180 = g_strdup_printf ("Expected initializer list of type `%s' but got `%s'", vala_data_type_get_name (vala_type_reference_get_data_type (child_type)), vala_data_type_get_name (vala_type_reference_get_data_type (vala_expression_get_static_type (e))))));
							(__temp180 = (g_free (__temp180), NULL));
						}
					}
				}
			}
		} else {
			(vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt)), vala_type_reference_get_data_type (child_type));
			{
				GList* __temp183;
				GList* e_it;
				__temp183 = NULL;
				__temp183 = inits;
				for (e_it = __temp183; e_it != NULL; e_it = e_it->next) {
					ValaExpression* e;
					e = e_it->data;
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							continue;
						}
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (e), child_type)) {
							char* __temp182;
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							__temp182 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (__temp182 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", vala_data_type_get_name (vala_type_reference_get_data_type (child_type)), vala_data_type_get_name (vala_type_reference_get_data_type (vala_expression_get_static_type (e))))));
							(__temp182 = (g_free (__temp182), NULL));
						}
					}
				}
			}
		}
		if (!error) {
			/* everything seems to be correct */
			(vala_expression_set_static_type (VALA_EXPRESSION (list), vala_expression_get_expected_type (VALA_EXPRESSION (list))), vala_expression_get_static_type (VALA_EXPRESSION (list)));
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_list_free (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_EXPRESSION_STATEMENT (stmt));
	if (vala_expression_get_static_type (vala_expression_statement_get_expression (stmt)) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_expression_statement_get_expression (stmt)))) {
		vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Short-living reference");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_IF_STATEMENT (stmt));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_if_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_if_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_if_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_WHILE_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_while_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_while_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOR_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_for_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_for_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_begin_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* __temp184;
	ValaSymbol* __temp185;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	__temp184 = NULL;
	(vala_foreach_statement_set_variable_declarator (stmt, (__temp184 = vala_variable_declarator_new (vala_foreach_statement_get_variable_name (stmt), NULL, NULL))), vala_foreach_statement_get_variable_declarator (stmt));
	(__temp184 == NULL ? NULL : (__temp184 = (g_object_unref (__temp184), NULL)));
	(vala_variable_declarator_set_type_reference (vala_foreach_statement_get_variable_declarator (stmt), vala_foreach_statement_get_type_reference (stmt)), vala_variable_declarator_get_type_reference (vala_foreach_statement_get_variable_declarator (stmt)));
	__temp185 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt)), (__temp185 = vala_symbol_new (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))));
	(__temp185 == NULL ? NULL : (__temp185 = (g_object_unref (__temp185), NULL)));
	vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_body (stmt))), vala_foreach_statement_get_variable_name (stmt), vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))));
}


static void vala_semantic_analyzer_real_visit_end_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	ValaDataType* __temp186;
	ValaDataType* collection_type;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	__temp186 = NULL;
	collection_type = (__temp186 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt))), (__temp186 == NULL ? NULL : g_object_ref (__temp186)));
	if (!(VALA_IS_ARRAY (collection_type) || collection_type == self->priv->glist_type || collection_type == self->priv->gslist_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Collection not iterable");
		(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
		return;
	}
	(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
	if (self->priv->current_return_type == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return not allowed in this context");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) == NULL && vala_type_reference_get_data_type (self->priv->current_return_type) != NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return without value in non-void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_data_type (self->priv->current_return_type) == NULL && vala_type_reference_get_type_parameter (self->priv->current_return_type) == NULL) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return with value in void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)), self->priv->current_return_type)) {
		char* __temp189;
		char* __temp188;
		char* __temp187;
		__temp189 = NULL;
		__temp188 = NULL;
		__temp187 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), (__temp189 = g_strdup_printf ("Return: Cannot convert from `%s' to `%s'", (__temp187 = vala_type_reference_to_string (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)))), (__temp188 = vala_type_reference_to_string (self->priv->current_return_type)))));
		(__temp189 = (g_free (__temp189), NULL));
		(__temp188 = (g_free (__temp188), NULL));
		(__temp187 = (g_free (__temp187), NULL));
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return value transfers ownership but method return type hasn't been declared to transfer ownership");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt)) != NULL && VALA_IS_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt)))) && vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Local variable with strong reference used as return value and method return type hasn't been declared to transfer ownership");
	}
}


/**
 * Visit operation called for lock statements.
 *
 * @param stmt a lock statement
 */
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LOCK_STATEMENT (stmt));
	/* resource must be a member access and denote a Lockable */
	if (!(VALA_IS_MEMBER_ACCESS (vala_lock_statement_get_resource (stmt)) && VALA_IS_LOCKABLE (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		(vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE), vala_code_node_get_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Expression is either not a member access or does not denote a lockable member");
		return;
	}
	/* parent symbol must be the current class */
	if (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))) != VALA_CODE_NODE (self->priv->current_class)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		(vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE), vala_code_node_get_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Only members of the current class are lockable");
	}
	vala_lockable_set_lock_used ((VALA_LOCKABLE (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))))), TRUE);
}


static void vala_semantic_analyzer_real_visit_begin_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL) {
		ValaTypeReference* __temp190;
		ValaArray* __temp191;
		__temp190 = NULL;
		(vala_expression_set_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)), (__temp190 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr)))), vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))));
		(__temp190 = (g_object_unref (__temp190), NULL));
		__temp191 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))), VALA_DATA_TYPE ((__temp191 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)))), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)))));
		(__temp191 = (g_object_unref (__temp191), NULL));
	}
}


/* FIXME: add element type to type_argument
*
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_semantic_analyzer_real_visit_end_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	gint i;
	GList* size;
	ValaTypeReference* __temp193;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	i = 0;
	size = vala_array_creation_expression_get_sizes (expr);
	/* check for errors in the size list */
	if (size != NULL) {
		{
			GList* __temp192;
			GList* e_it;
			__temp192 = NULL;
			__temp192 = size;
			for (e_it = __temp192; e_it != NULL; e_it = e_it->next) {
				ValaExpression* e;
				e = e_it->data;
				{
					if (vala_expression_get_static_type (e) == NULL) {
						(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
						return;
					} else {
						/* return on previous error */
						if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
						}
					}
				}
			}
		}
		if (vala_code_node_get_error (VALA_CODE_NODE (expr))) {
			(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
			return;
		}
	}
	/* check for wrong elements inside the initializer */
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL && vala_expression_get_static_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))) == NULL) {
		(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
		return;
	}
	/* try to construct the type of the array */
	if (vala_array_creation_expression_get_element_type (expr) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Cannot determine the element type of the created array");
		(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
		return;
	}
	__temp193 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp193 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr)))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp193 = (g_object_unref (__temp193), NULL));
	if (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)) != NULL) {
		ValaArray* __temp194;
		__temp194 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp194 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
		(__temp194 = (g_object_unref (__temp194), NULL));
	} else {
		ValaArray* __temp195;
		__temp195 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp195 = vala_type_parameter_get_array (vala_type_reference_get_type_parameter (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
		(__temp195 = (g_object_unref (__temp195), NULL));
	}
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	vala_type_reference_add_type_argument (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_array_creation_expression_get_element_type (expr));
	(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
}


static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BOOLEAN_LITERAL (expr));
	(vala_literal_set_static_type (VALA_LITERAL (expr), self->priv->bool_type), vala_literal_get_static_type (VALA_LITERAL (expr)));
}


static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp196;
	ValaSymbol* __temp197;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CHARACTER_LITERAL (expr));
	__temp196 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp196 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp196 == NULL ? NULL : (__temp196 = (g_object_unref (__temp196), NULL)));
	__temp197 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp197 = vala_symbol_lookup (self->priv->root_symbol, "char"))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp197 == NULL ? NULL : (__temp197 = (g_object_unref (__temp197), NULL)));
}


static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp198;
	ValaSymbol* __temp200;
	char* __temp199;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTEGER_LITERAL (expr));
	__temp198 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp198 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp198 == NULL ? NULL : (__temp198 = (g_object_unref (__temp198), NULL)));
	__temp200 = NULL;
	__temp199 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp200 = vala_symbol_lookup (self->priv->root_symbol, (__temp199 = vala_integer_literal_get_type_name (expr))))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp200 == NULL ? NULL : (__temp200 = (g_object_unref (__temp200), NULL)));
	(__temp199 = (g_free (__temp199), NULL));
}


static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp201;
	ValaSymbol* __temp203;
	char* __temp202;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_REAL_LITERAL (expr));
	__temp201 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp201 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp201 == NULL ? NULL : (__temp201 = (g_object_unref (__temp201), NULL)));
	__temp203 = NULL;
	__temp202 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp203 = vala_symbol_lookup (self->priv->root_symbol, (__temp202 = vala_real_literal_get_type_name (expr))))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp203 == NULL ? NULL : (__temp203 = (g_object_unref (__temp203), NULL)));
	(__temp202 = (g_free (__temp202), NULL));
}


static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp204;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRING_LITERAL (expr));
	__temp204 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp204 = vala_type_reference_copy (self->priv->string_type))), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp204 = (g_object_unref (__temp204), NULL));
	(vala_type_reference_set_non_null (vala_literal_get_static_type (VALA_LITERAL (expr)), TRUE), vala_type_reference_get_non_null (vala_literal_get_static_type (VALA_LITERAL (expr))));
}


static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp205;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NULL_LITERAL (expr));
	/* empty TypeReference represents null */
	__temp205 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp205 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp205 == NULL ? NULL : (__temp205 = (g_object_unref (__temp205), NULL)));
}


static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LITERAL_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_literal_get_static_type (vala_literal_expression_get_literal (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_node (ValaSemanticAnalyzer* self, ValaCodeNode* node)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_CODE_NODE (node), NULL);
	if (VALA_IS_FIELD (node)) {
		ValaField* __temp206;
		ValaField* f;
		ValaTypeReference* __temp207;
		__temp206 = NULL;
		f = (__temp206 = VALA_FIELD (node), (__temp206 == NULL ? NULL : g_object_ref (__temp206)));
		__temp207 = NULL;
		return (__temp207 = g_object_ref (vala_field_get_type_reference (f)), (f == NULL ? NULL : (f = (g_object_unref (f), NULL))), __temp207);
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	} else {
		if (VALA_IS_CONSTANT (node)) {
			ValaConstant* __temp208;
			ValaConstant* c;
			ValaTypeReference* __temp209;
			__temp208 = NULL;
			c = (__temp208 = VALA_CONSTANT (node), (__temp208 == NULL ? NULL : g_object_ref (__temp208)));
			__temp209 = NULL;
			return (__temp209 = g_object_ref (vala_constant_get_type_reference (c)), (c == NULL ? NULL : (c = (g_object_unref (c), NULL))), __temp209);
			(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
		} else {
			if (VALA_IS_PROPERTY (node)) {
				ValaProperty* __temp210;
				ValaProperty* prop;
				ValaTypeReference* type;
				ValaTypeReference* __temp211;
				__temp210 = NULL;
				prop = (__temp210 = VALA_PROPERTY (node), (__temp210 == NULL ? NULL : g_object_ref (__temp210)));
				type = vala_type_reference_copy (vala_property_get_type_reference (prop));
				(vala_type_reference_set_takes_ownership (type, FALSE), vala_type_reference_get_takes_ownership (type));
				__temp211 = NULL;
				return (__temp211 = type, (prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL))), __temp211);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
				(type = (g_object_unref (type), NULL));
			} else {
				if (VALA_IS_FORMAL_PARAMETER (node)) {
					ValaFormalParameter* __temp212;
					ValaFormalParameter* p;
					ValaTypeReference* __temp213;
					ValaTypeReference* __temp214;
					__temp212 = NULL;
					p = (__temp212 = VALA_FORMAL_PARAMETER (node), (__temp212 == NULL ? NULL : g_object_ref (__temp212)));
					__temp213 = NULL;
					__temp214 = NULL;
					return (__temp214 = (__temp213 = vala_formal_parameter_get_type_reference (p), (__temp213 == NULL ? NULL : g_object_ref (__temp213))), (p == NULL ? NULL : (p = (g_object_unref (p), NULL))), __temp214);
					(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				} else {
					if (VALA_IS_TYPE_REFERENCE (node)) {
						ValaTypeReference* __temp215;
						__temp215 = NULL;
						return (__temp215 = VALA_TYPE_REFERENCE (node), (__temp215 == NULL ? NULL : g_object_ref (__temp215)));
					} else {
						if (VALA_IS_VARIABLE_DECLARATOR (node)) {
							ValaVariableDeclarator* __temp217;
							ValaVariableDeclarator* decl;
							ValaTypeReference* __temp218;
							ValaTypeReference* __temp219;
							__temp217 = NULL;
							decl = (__temp217 = VALA_VARIABLE_DECLARATOR (node), (__temp217 == NULL ? NULL : g_object_ref (__temp217)));
							__temp218 = NULL;
							__temp219 = NULL;
							return (__temp219 = (__temp218 = vala_variable_declarator_get_type_reference (decl), (__temp218 == NULL ? NULL : g_object_ref (__temp218))), (decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL))), __temp219);
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						} else {
							if (VALA_IS_ENUM_VALUE (node) || VALA_IS_FLAGS_VALUE (node)) {
								ValaTypeReference* type;
								type = vala_type_reference_new ();
								(vala_type_reference_set_data_type (type, VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (node))))), vala_type_reference_get_data_type (type));
								return type;
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							}
						}
					}
				}
			}
		}
	}
	return VALA_TYPE_REFERENCE (NULL);
}


ValaSymbol* vala_semantic_analyzer_symbol_lookup_inherited (ValaSymbol* sym, const char* name)
{
	ValaSymbol* result;
	gpointer __temp235;
	g_return_val_if_fail (VALA_IS_SYMBOL (sym), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	result = vala_symbol_lookup (sym, name);
	if (result != NULL) {
		return result;
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (sym))) {
		ValaClass* __temp223;
		ValaClass* cl;
		__temp223 = NULL;
		cl = (__temp223 = VALA_CLASS (vala_symbol_get_node (sym)), (__temp223 == NULL ? NULL : g_object_ref (__temp223)));
		{
			GList* __temp226;
			GList* base_type_it;
			__temp226 = NULL;
			__temp226 = vala_class_get_base_types (cl);
			for (base_type_it = __temp226; base_type_it != NULL; base_type_it = base_type_it->next) {
				ValaTypeReference* base_type;
				base_type = base_type_it->data;
				{
					ValaSymbol* __temp224;
					__temp224 = NULL;
					result = (__temp224 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp224);
					if (result != NULL) {
						ValaSymbol* __temp225;
						__temp225 = NULL;
						return (__temp225 = result, (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp225);
					}
				}
			}
			(__temp226 == NULL ? NULL : (__temp226 = (g_list_free (__temp226), NULL)));
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_node (sym))) {
			ValaStruct* __temp227;
			ValaStruct* st;
			__temp227 = NULL;
			st = (__temp227 = VALA_STRUCT (vala_symbol_get_node (sym)), (__temp227 == NULL ? NULL : g_object_ref (__temp227)));
			{
				GList* __temp230;
				GList* base_type_it;
				__temp230 = NULL;
				__temp230 = vala_struct_get_base_types (st);
				for (base_type_it = __temp230; base_type_it != NULL; base_type_it = base_type_it->next) {
					ValaTypeReference* base_type;
					base_type = base_type_it->data;
					{
						ValaSymbol* __temp228;
						__temp228 = NULL;
						result = (__temp228 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp228);
						if (result != NULL) {
							ValaSymbol* __temp229;
							__temp229 = NULL;
							return (__temp229 = result, (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp229);
						}
					}
				}
				(__temp230 == NULL ? NULL : (__temp230 = (g_list_free (__temp230), NULL)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		} else {
			if (VALA_IS_INTERFACE (vala_symbol_get_node (sym))) {
				ValaInterface* __temp231;
				ValaInterface* iface;
				__temp231 = NULL;
				iface = (__temp231 = VALA_INTERFACE (vala_symbol_get_node (sym)), (__temp231 == NULL ? NULL : g_object_ref (__temp231)));
				{
					GList* __temp234;
					GList* prerequisite_it;
					__temp234 = NULL;
					__temp234 = vala_interface_get_prerequisites (iface);
					for (prerequisite_it = __temp234; prerequisite_it != NULL; prerequisite_it = prerequisite_it->next) {
						ValaTypeReference* prerequisite;
						prerequisite = prerequisite_it->data;
						{
							ValaSymbol* __temp232;
							__temp232 = NULL;
							result = (__temp232 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (prerequisite))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp232);
							if (result != NULL) {
								ValaSymbol* __temp233;
								__temp233 = NULL;
								return (__temp233 = result, (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), __temp233);
							}
						}
					}
					(__temp234 == NULL ? NULL : (__temp234 = (g_list_free (__temp234), NULL)));
				}
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	return VALA_SYMBOL ((__temp235 = NULL, (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp235));
	(result == NULL ? NULL : (result = (g_object_unref (result), NULL)));
}


static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp236;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PARENTHESIZED_EXPRESSION (expr));
	__temp236 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp236 = vala_type_reference_copy (vala_expression_get_static_type (vala_parenthesized_expression_get_inner (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp236 = (g_object_unref (__temp236), NULL));
	/* don't call g_object_ref_sink on inner and outer expression*/
	(vala_type_reference_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE), vala_type_reference_get_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static ValaDataType* vala_semantic_analyzer_find_parent_type (ValaSemanticAnalyzer* self, ValaSymbol* sym)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (sym == NULL || VALA_IS_SYMBOL (sym), NULL);
	while (sym != NULL) {
		if (VALA_IS_DATA_TYPE (vala_symbol_get_node (sym))) {
			ValaDataType* __temp237;
			__temp237 = NULL;
			return (__temp237 = VALA_DATA_TYPE (vala_symbol_get_node (sym)), (__temp237 == NULL ? NULL : g_object_ref (__temp237)));
		}
		sym = vala_symbol_get_parent_symbol (sym);
	}
	return VALA_DATA_TYPE (NULL);
}


static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* base_symbol;
	ValaCodeNode* __temp258;
	ValaCodeNode* member;
	ValaMemberAccessibility access;
	ValaTypeReference* __temp262;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (expr));
	base_symbol = NULL;
	if (vala_member_access_get_inner (expr) == NULL) {
		ValaSymbol* __temp241;
		ValaSymbol* __temp240;
		ValaSymbol* __temp242;
		ValaSymbol* sym;
		__temp241 = NULL;
		__temp240 = NULL;
		base_symbol = (__temp241 = (__temp240 = self->priv->current_symbol, (__temp240 == NULL ? NULL : g_object_ref (__temp240))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp241);
		__temp242 = NULL;
		sym = (__temp242 = self->priv->current_symbol, (__temp242 == NULL ? NULL : g_object_ref (__temp242)));
		while (sym != NULL && vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			ValaSymbol* __temp243;
			ValaSymbol* __temp245;
			ValaSymbol* __temp244;
			__temp243 = NULL;
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp243 = vala_semantic_analyzer_symbol_lookup_inherited (sym, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			(__temp243 == NULL ? NULL : (__temp243 = (g_object_unref (__temp243), NULL)));
			__temp245 = NULL;
			__temp244 = NULL;
			sym = (__temp245 = (__temp244 = vala_symbol_get_parent_symbol (sym), (__temp244 == NULL ? NULL : g_object_ref (__temp244))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp245);
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			{
				GList* __temp249;
				GList* ns_it;
				__temp249 = NULL;
				__temp249 = self->priv->current_using_directives;
				for (ns_it = __temp249; ns_it != NULL; ns_it = ns_it->next) {
					ValaNamespaceReference* ns;
					ns = ns_it->data;
					{
						ValaSymbol* local_sym;
						local_sym = vala_symbol_lookup (vala_namespace_reference_get_namespace_symbol (ns), vala_member_access_get_member_name (expr));
						if (local_sym != NULL) {
							if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
								char* __temp248;
								char* __temp247;
								char* __temp246;
								(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
								__temp248 = NULL;
								__temp247 = NULL;
								__temp246 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp248 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", vala_member_access_get_member_name (expr), (__temp246 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))), (__temp247 = vala_symbol_get_full_name (local_sym)))));
								(__temp248 = (g_free (__temp248), NULL));
								(__temp247 = (g_free (__temp247), NULL));
								(__temp246 = (g_free (__temp246), NULL));
								(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
								return;
							}
							(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), local_sym), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
						}
						(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
					}
				}
			}
		}
		(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
	} else {
		if (vala_code_node_get_error (VALA_CODE_NODE (vala_member_access_get_inner (expr)))) {
			/* if there was an error in the inner expression, skip this check */
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			return;
		}
		if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (expr)) || VALA_IS_BASE_ACCESS (vala_member_access_get_inner (expr))) {
			ValaSymbol* __temp251;
			ValaSymbol* __temp250;
			__temp251 = NULL;
			__temp250 = NULL;
			base_symbol = (__temp251 = (__temp250 = vala_expression_get_symbol_reference (vala_member_access_get_inner (expr)), (__temp250 == NULL ? NULL : g_object_ref (__temp250))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp251);
			if (VALA_IS_NAMESPACE (vala_symbol_get_node (base_symbol)) || VALA_IS_DATA_TYPE (vala_symbol_get_node (base_symbol))) {
				ValaSymbol* __temp252;
				__temp252 = NULL;
				(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp252 = vala_symbol_lookup (base_symbol, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
				(__temp252 == NULL ? NULL : (__temp252 = (g_object_unref (__temp252), NULL)));
			}
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_expression_get_static_type (vala_member_access_get_inner (expr)) != NULL) {
			ValaSymbol* __temp254;
			ValaSymbol* __temp253;
			ValaSymbol* __temp255;
			__temp254 = NULL;
			__temp253 = NULL;
			base_symbol = (__temp254 = (__temp253 = vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_member_access_get_inner (expr))))), (__temp253 == NULL ? NULL : g_object_ref (__temp253))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp254);
			__temp255 = NULL;
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp255 = vala_semantic_analyzer_symbol_lookup_inherited (base_symbol, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			(__temp255 == NULL ? NULL : (__temp255 = (g_object_unref (__temp255), NULL)));
		}
	}
	if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
		char* __temp257;
		char* __temp256;
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp257 = NULL;
		__temp256 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp257 = g_strdup_printf ("The name `%s' does not exist in the context of `%s'", vala_member_access_get_member_name (expr), (__temp256 = vala_symbol_get_full_name (base_symbol)))));
		(__temp257 = (g_free (__temp257), NULL));
		(__temp256 = (g_free (__temp256), NULL));
		(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
		return;
	}
	__temp258 = NULL;
	member = (__temp258 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr))), (__temp258 == NULL ? NULL : g_object_ref (__temp258)));
	access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	if (VALA_IS_FIELD (member)) {
		access = (VALA_FIELD (member))->access;
	} else {
		if (VALA_IS_METHOD (member)) {
			access = (VALA_METHOD (member))->access;
		}
	}
	if (access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		ValaDataType* __temp259;
		ValaDataType* target_type;
		ValaDataType* this_type;
		__temp259 = NULL;
		target_type = (__temp259 = VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (member)))), (__temp259 == NULL ? NULL : g_object_ref (__temp259)));
		this_type = vala_semantic_analyzer_find_parent_type (self, self->priv->current_symbol);
		if (target_type != this_type) {
			char* __temp261;
			char* __temp260;
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp261 = NULL;
			__temp260 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp261 = g_strdup_printf ("Access to private member `%s' denied", (__temp260 = vala_symbol_get_full_name (vala_code_node_get_symbol (member))))));
			(__temp261 = (g_free (__temp261), NULL));
			(__temp260 = (g_free (__temp260), NULL));
			(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
			(this_type == NULL ? NULL : (this_type = (g_object_unref (this_type), NULL)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
			return;
		}
		(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
		(this_type == NULL ? NULL : (this_type = (g_object_unref (this_type), NULL)));
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	__temp262 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp262 = vala_semantic_analyzer_get_static_type_for_node (self, vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp262 == NULL ? NULL : (__temp262 = (g_object_unref (__temp262), NULL)));
	(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
	(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
}


static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expression_type), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expected_type), FALSE);
	/* only null is compatible to null */
	if (vala_type_reference_get_data_type (expected_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL) {
		return (vala_type_reference_get_data_type (expression_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL);
	}
	if (vala_type_reference_get_data_type (expression_type) == NULL) {
		/* null can be cast to any reference or array type or pointer type */
		if (vala_type_reference_get_type_parameter (expected_type) != NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_is_out (expected_type) || VALA_IS_POINTER (vala_type_reference_get_data_type (expected_type)) || VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type)) || VALA_IS_CALLBACK (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_data_type (expected_type) == self->priv->pointer_type) {
			return TRUE;
		}
		return FALSE;
	}
	/* null is not compatible with any other type (i.e. value types) 
	 temporarily ignore type parameters */
	if (vala_type_reference_get_type_parameter (expected_type) != NULL) {
		return TRUE;
	}
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (expression_type)) != VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type))) {
		return FALSE;
	}
	if (VALA_IS_ENUM (vala_type_reference_get_data_type (expression_type)) && vala_type_reference_get_data_type (expected_type) == vala_type_reference_get_data_type (self->priv->int_type)) {
		return TRUE;
	}
	if (vala_type_reference_get_data_type (expression_type) == vala_type_reference_get_data_type (expected_type)) {
		return TRUE;
	}
	if (VALA_IS_STRUCT (vala_type_reference_get_data_type (expression_type)) && VALA_IS_STRUCT (vala_type_reference_get_data_type (expected_type))) {
		ValaStruct* __temp270;
		ValaStruct* expr_struct;
		ValaStruct* __temp271;
		ValaStruct* expect_struct;
		__temp270 = NULL;
		expr_struct = (__temp270 = VALA_STRUCT (vala_type_reference_get_data_type (expression_type)), (__temp270 == NULL ? NULL : g_object_ref (__temp270)));
		__temp271 = NULL;
		expect_struct = (__temp271 = VALA_STRUCT (vala_type_reference_get_data_type (expected_type)), (__temp271 == NULL ? NULL : g_object_ref (__temp271)));
		/* integer types may be implicitly cast to floating point types */
		if (vala_struct_is_integer_type (expr_struct) && vala_struct_is_floating_type (expect_struct)) {
			gboolean __temp272;
			return (__temp272 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), __temp272);
		}
		if ((vala_struct_is_integer_type (expr_struct) && vala_struct_is_integer_type (expect_struct)) || (vala_struct_is_floating_type (expr_struct) && vala_struct_is_floating_type (expect_struct))) {
			if (vala_struct_get_rank (expr_struct) <= vala_struct_get_rank (expect_struct)) {
				gboolean __temp273;
				return (__temp273 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), __temp273);
			}
		}
		(expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL)));
		(expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL)));
	}
	return vala_data_type_is_subtype_of (vala_type_reference_get_data_type (expression_type), vala_type_reference_get_data_type (expected_type));
}


static void vala_semantic_analyzer_real_visit_begin_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp275;
	ValaSymbol* msym;
	GList* params;
	GList* args;
	GList* arg_it;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_invocation_expression_get_call (expr)))) {
		/* if method resolving didn't succeed, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	__temp275 = NULL;
	msym = (__temp275 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp275 == NULL ? NULL : g_object_ref (__temp275)));
	if (msym == NULL) {
		/* if no symbol found, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		return;
	}
	params = NULL;
	if (VALA_IS_INVOKABLE (vala_symbol_get_node (msym))) {
		ValaInvokable* __temp276;
		ValaInvokable* m;
		__temp276 = NULL;
		m = (__temp276 = VALA_INVOKABLE (vala_symbol_get_node (msym)), (__temp276 == NULL ? NULL : g_object_ref (__temp276)));
		if (vala_invokable_is_invokable (m)) {
			GList* __temp277;
			__temp277 = NULL;
			params = (__temp277 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp277);
		} else {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
			(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
			return;
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	} else {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
		return;
	}
	args = vala_invocation_expression_get_argument_list (expr);
	arg_it = args;
	{
		GList* __temp279;
		GList* param_it;
		__temp279 = NULL;
		__temp279 = params;
		for (param_it = __temp279; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param;
			param = param_it->data;
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					break;
				}
				if (arg_it != NULL) {
					ValaExpression* __temp278;
					ValaExpression* arg;
					__temp278 = NULL;
					arg = (__temp278 = VALA_EXPRESSION (arg_it->data), (__temp278 == NULL ? NULL : g_object_ref (__temp278)));
					/* store expected type for callback parameters */
					(vala_expression_set_expected_type (arg, vala_formal_parameter_get_type_reference (param)), vala_expression_get_expected_type (arg));
					arg_it = arg_it->next;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
			}
		}
	}
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
	(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
}


static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GList* params, GList* args)
{
	GList* prev_arg_it;
	GList* arg_it;
	ValaAttribute* __temp280;
	gboolean __temp281;
	gboolean diag;
	gboolean ellipsis;
	gint i;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), FALSE);
	g_return_val_if_fail (VALA_IS_SYMBOL (msym), FALSE);
	prev_arg_it = NULL;
	arg_it = args;
	__temp280 = NULL;
	diag = (__temp281 = ((__temp280 = vala_code_node_get_attribute (vala_symbol_get_node (msym), "Diagnostics")) != NULL), (__temp280 == NULL ? NULL : (__temp280 = (g_object_unref (__temp280), NULL))), __temp281);
	ellipsis = FALSE;
	i = 0;
	{
		GList* __temp292;
		GList* param_it;
		__temp292 = NULL;
		__temp292 = params;
		for (param_it = __temp292; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param;
			param = param_it->data;
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					ellipsis = TRUE;
					break;
				}
				/* header file necessary if we need to cast argument */
				if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)) != NULL) {
					vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
				}
				if (arg_it == NULL) {
					if (vala_formal_parameter_get_default_expression (param) == NULL) {
						char* __temp283;
						char* __temp282;
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						__temp283 = NULL;
						__temp282 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp283 = g_strdup_printf ("Too few arguments, method `%s' does not take %d arguments", (__temp282 = vala_symbol_get_full_name (msym)), g_list_length (args))));
						(__temp283 = (g_free (__temp283), NULL));
						(__temp282 = (g_free (__temp282), NULL));
						return FALSE;
					}
				} else {
					ValaExpression* __temp285;
					ValaExpression* arg;
					__temp285 = NULL;
					arg = (__temp285 = VALA_EXPRESSION (arg_it->data), (__temp285 == NULL ? NULL : g_object_ref (__temp285)));
					if (vala_expression_get_static_type (arg) == NULL) {
						/* disallow untyped arguments except for type inference of callbacks*/
						if (!(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param))))) {
							char* __temp286;
							gboolean __temp287;
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							__temp286 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp286 = g_strdup_printf ("Invalid type for argument %d", i + 1)));
							(__temp286 = (g_free (__temp286), NULL));
							return (__temp287 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), __temp287);
						}
					} else {
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (arg), vala_formal_parameter_get_type_reference (param))) {
							char* __temp290;
							char* __temp289;
							char* __temp288;
							gboolean __temp291;
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							__temp290 = NULL;
							__temp289 = NULL;
							__temp288 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp290 = g_strdup_printf ("Argument %d: Cannot convert from `%s' to `%s'", i + 1, (__temp288 = vala_type_reference_to_string (vala_expression_get_static_type (arg))), (__temp289 = vala_type_reference_to_string (vala_formal_parameter_get_type_reference (param))))));
							(__temp290 = (g_free (__temp290), NULL));
							(__temp289 = (g_free (__temp289), NULL));
							(__temp288 = (g_free (__temp288), NULL));
							return (__temp291 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), __temp291);
						}
					}
					prev_arg_it = arg_it;
					arg_it = arg_it->next;
					i++;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
			}
		}
	}
	if (!ellipsis && arg_it != NULL) {
		char* __temp294;
		char* __temp293;
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp294 = NULL;
		__temp293 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp294 = g_strdup_printf ("Too many arguments, method `%s' does not take %d arguments", (__temp293 = vala_symbol_get_full_name (msym)), g_list_length (args))));
		(__temp294 = (g_free (__temp294), NULL));
		(__temp293 = (g_free (__temp293), NULL));
		return FALSE;
	}
	if (diag && prev_arg_it != NULL) {
		ValaExpression* __temp296;
		ValaExpression* format_arg;
		__temp296 = NULL;
		format_arg = (__temp296 = VALA_EXPRESSION (prev_arg_it->data), (__temp296 == NULL ? NULL : g_object_ref (__temp296)));
		if (VALA_IS_LITERAL_EXPRESSION (format_arg)) {
			ValaStringLiteral* __temp297;
			ValaStringLiteral* format_lit;
			char* __temp298;
			__temp297 = NULL;
			format_lit = (__temp297 = VALA_STRING_LITERAL (vala_literal_expression_get_literal ((VALA_LITERAL_EXPRESSION (format_arg)))), (__temp297 == NULL ? NULL : g_object_ref (__temp297)));
			__temp298 = NULL;
			(vala_string_literal_set_value (format_lit, (__temp298 = g_strdup_printf ("\"%s:%d: %s", vala_source_file_get_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)))), vala_source_reference_get_first_line (vala_code_node_get_source_reference (VALA_CODE_NODE (expr))), g_utf8_offset_to_pointer (vala_string_literal_get_value (format_lit), 1)))), vala_string_literal_get_value (format_lit));
			(__temp298 = (g_free (__temp298), NULL));
			(format_lit == NULL ? NULL : (format_lit = (g_object_unref (format_lit), NULL)));
		}
		(format_arg == NULL ? NULL : (format_arg = (g_object_unref (format_arg), NULL)));
	}
	return TRUE;
}


static void vala_semantic_analyzer_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp300;
	ValaSymbol* msym;
	ValaTypeReference* ret_type;
	GList* params;
	GList* __temp323;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (expr))) {
		return;
	}
	__temp300 = NULL;
	msym = (__temp300 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp300 == NULL ? NULL : g_object_ref (__temp300)));
	ret_type = NULL;
	params = NULL;
	if (VALA_IS_INVOKABLE (vala_symbol_get_node (msym))) {
		ValaInvokable* __temp301;
		ValaInvokable* m;
		ValaTypeReference* __temp302;
		GList* __temp303;
		__temp301 = NULL;
		m = (__temp301 = VALA_INVOKABLE (vala_symbol_get_node (msym)), (__temp301 == NULL ? NULL : g_object_ref (__temp301)));
		__temp302 = NULL;
		ret_type = (__temp302 = vala_invokable_get_return_type (m), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), __temp302);
		__temp303 = NULL;
		params = (__temp303 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp303);
		if (vala_type_reference_get_data_type (ret_type) == NULL && vala_type_reference_get_type_parameter (ret_type) == NULL) {
			/* void return type*/
			if (!(VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (VALA_CODE_NODE (expr))))) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation of void method not allowed as expression");
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			}
		}
		/* resolve generic return values*/
		if (vala_type_reference_get_type_parameter (ret_type) != NULL) {
			ValaMemberAccess* __temp304;
			ValaMemberAccess* ma;
			if (!(VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)))) {
				vala_report_error (vala_code_node_get_source_reference ((VALA_CODE_NODE (m))), "internal error: unsupported generic return value");
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			}
			__temp304 = NULL;
			ma = (__temp304 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (__temp304 == NULL ? NULL : g_object_ref (__temp304)));
			if (vala_member_access_get_inner (ma) == NULL) {
				/* TODO resolve generic return values within the type hierarchy if possible*/
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: resolving generic return values within type hierarchy not supported yet");
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			} else {
				ValaTypeReference* __temp305;
				ValaTypeReference* instance_type;
				gint param_index;
				ValaTypeReference* __temp321;
				ValaTypeReference* __temp320;
				GList* __temp319;
				__temp305 = NULL;
				instance_type = (__temp305 = vala_expression_get_static_type (vala_member_access_get_inner (ma)), (__temp305 == NULL ? NULL : g_object_ref (__temp305)));
				while (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
					GList* base_types;
					/* trace type arguments back to the datatype where the method has been declared*/
					base_types = NULL;
					if (VALA_IS_CLASS (vala_type_reference_get_data_type (instance_type))) {
						ValaClass* __temp306;
						ValaClass* cl;
						GList* __temp307;
						__temp306 = NULL;
						cl = (__temp306 = VALA_CLASS (vala_type_reference_get_data_type (instance_type)), (__temp306 == NULL ? NULL : g_object_ref (__temp306)));
						__temp307 = NULL;
						base_types = (__temp307 = vala_class_get_base_types (cl), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp307);
						(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
					} else {
						if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (instance_type))) {
							ValaInterface* __temp308;
							ValaInterface* iface;
							GList* __temp309;
							__temp308 = NULL;
							iface = (__temp308 = VALA_INTERFACE (vala_type_reference_get_data_type (instance_type)), (__temp308 == NULL ? NULL : g_object_ref (__temp308)));
							__temp309 = NULL;
							base_types = (__temp309 = vala_interface_get_prerequisites (iface), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp309);
							(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
						} else {
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported generic type");
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
							(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
							(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
							(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
							return;
						}
					}
					{
						GList* __temp317;
						GList* base_type_it;
						__temp317 = NULL;
						__temp317 = base_types;
						for (base_type_it = __temp317; base_type_it != NULL; base_type_it = base_type_it->next) {
							ValaTypeReference* base_type;
							base_type = base_type_it->data;
							{
								ValaSymbol* __temp310;
								gboolean __temp311;
								__temp310 = NULL;
								if ((__temp311 = (__temp310 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), vala_symbol_get_name (msym))) != NULL, (__temp310 == NULL ? NULL : (__temp310 = (g_object_unref (__temp310), NULL))), __temp311)) {
									ValaTypeReference* instance_base_type;
									ValaTypeReference* __temp316;
									ValaTypeReference* __temp315;
									/* construct a new type reference for the base type with correctly linked type arguments*/
									instance_base_type = vala_type_reference_new ();
									(vala_type_reference_set_data_type (instance_base_type, vala_type_reference_get_data_type (base_type)), vala_type_reference_get_data_type (instance_base_type));
									{
										GList* __temp314;
										GList* type_arg_it;
										__temp314 = NULL;
										__temp314 = vala_type_reference_get_type_arguments (base_type);
										for (type_arg_it = __temp314; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
											ValaTypeReference* type_arg;
											type_arg = type_arg_it->data;
											{
												if (vala_type_reference_get_type_parameter (type_arg) != NULL) {
													gint param_index;
													GList* __temp313;
													/* link to type argument of derived type*/
													param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)));
													if (param_index == -1) {
														char* __temp312;
														__temp312 = NULL;
														vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp312 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)))));
														(__temp312 = (g_free (__temp312), NULL));
														(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
														(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
														(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
														(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
														(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
														(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
														(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
														(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
														(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
														return;
													}
													__temp313 = NULL;
													type_arg = g_list_nth_data ((__temp313 = vala_type_reference_get_type_arguments (instance_type)), param_index);
													(__temp313 == NULL ? NULL : (__temp313 = (g_list_free (__temp313), NULL)));
												}
												vala_type_reference_add_type_argument (instance_base_type, type_arg);
											}
										}
										(__temp314 == NULL ? NULL : (__temp314 = (g_list_free (__temp314), NULL)));
									}
									__temp316 = NULL;
									__temp315 = NULL;
									instance_type = (__temp316 = (__temp315 = instance_base_type, (__temp315 == NULL ? NULL : g_object_ref (__temp315))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp316);
									(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
								}
							}
						}
					}
					(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
				}
				if (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: generic type parameter tracing not supported yet");
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)));
				if (param_index == -1) {
					char* __temp318;
					__temp318 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp318 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)))));
					(__temp318 = (g_free (__temp318), NULL));
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				__temp321 = NULL;
				__temp320 = NULL;
				__temp319 = NULL;
				ret_type = (__temp321 = (__temp320 = VALA_TYPE_REFERENCE (g_list_nth_data ((__temp319 = vala_type_reference_get_type_arguments (instance_type)), param_index)), (__temp320 == NULL ? NULL : g_object_ref (__temp320))), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), __temp321);
				(__temp319 == NULL ? NULL : (__temp319 = (g_list_free (__temp319), NULL)));
				if (ret_type == NULL) {
					char* __temp322;
					__temp322 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp322 = g_strdup_printf ("internal error: no actual argument found for type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)))));
					(__temp322 = (g_free (__temp322), NULL));
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
			}
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), ret_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	__temp323 = NULL;
	vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), msym, params, (__temp323 = vala_invocation_expression_get_argument_list (expr)));
	(__temp323 == NULL ? NULL : (__temp323 = (g_list_free (__temp323), NULL)));
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
	(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
}


static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ELEMENT_ACCESS (expr));
	if (vala_expression_get_static_type (vala_element_access_get_container (expr)) == NULL) {
		/* don't proceed if a child expression failed */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	/* assign a static_type when possible */
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_element_access_get_container (expr))))) {
		GList* args;
		args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (expr)));
		if (g_list_length (args) != 1) {
			char* __temp324;
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp324 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp324 = g_strdup_printf ("internal error: array reference with %d type arguments, expected 1", g_list_length (args))));
			(__temp324 = (g_free (__temp324), NULL));
			(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
			return;
		}
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_TYPE_REFERENCE (args->data)), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
	} else {
		char* __temp326;
		char* __temp325;
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp326 = NULL;
		__temp325 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp326 = g_strdup_printf ("The expression `%s' does not denote an Array", (__temp325 = vala_type_reference_to_string (vala_expression_get_static_type (vala_element_access_get_container (expr)))))));
		(__temp326 = (g_free (__temp326), NULL));
		(__temp325 = (g_free (__temp325), NULL));
	}
	{
		GList* __temp327;
		GList* e_it;
		__temp327 = NULL;
		__temp327 = vala_element_access_get_indices (expr);
		for (e_it = __temp327; e_it != NULL; e_it = e_it->next) {
			ValaExpression* e;
			e = e_it->data;
			{
				/* check if the index is of type integer 
				 don't proceed if a child expression failed */
				if (vala_expression_get_static_type (e) == NULL) {
					return;
				}
				/* check if the index is of type integer */
				if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
				}
			}
		}
		(__temp327 == NULL ? NULL : (__temp327 = (g_list_free (__temp327), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BASE_ACCESS (expr));
	if (self->priv->current_class == NULL) {
		GList* __temp332;
		if (self->priv->current_struct == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Base access invalid outside of class and struct");
			return;
		} else {
			GList* __temp328;
			gboolean __temp329;
			__temp328 = NULL;
			if ((__temp329 = g_list_length ((__temp328 = vala_struct_get_base_types (self->priv->current_struct))) != 1, (__temp328 == NULL ? NULL : (__temp328 = (g_list_free (__temp328), NULL))), __temp329)) {
				char* __temp331;
				GList* __temp330;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp331 = NULL;
				__temp330 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp331 = g_strdup_printf ("Base access invalid without base type %d", g_list_length ((__temp330 = vala_struct_get_base_types (self->priv->current_struct))))));
				(__temp331 = (g_free (__temp331), NULL));
				(__temp330 == NULL ? NULL : (__temp330 = (g_list_free (__temp330), NULL)));
				return;
			}
		}
		__temp332 = NULL;
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), g_list_first ((__temp332 = vala_struct_get_base_types (self->priv->current_struct)))->data), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(__temp332 == NULL ? NULL : (__temp332 = (g_list_free (__temp332), NULL)));
	} else {
		ValaTypeReference* __temp333;
		__temp333 = NULL;
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp333 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(__temp333 == NULL ? NULL : (__temp333 = (g_object_unref (__temp333), NULL)));
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE (vala_class_get_base_class (self->priv->current_class))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	}
	(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)))))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_POSTFIX_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_postfix_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_end_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaDataType* type;
	ValaTypeReference* __temp352;
	GList* __temp359;
	gboolean __temp360;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
	type = NULL;
	if (vala_object_creation_expression_get_type_reference (expr) == NULL) {
		ValaCodeNode* __temp334;
		ValaCodeNode* constructor_node;
		ValaCodeNode* __temp335;
		ValaCodeNode* type_node;
		GList* type_args;
		ValaTypeReference* __temp346;
		if (vala_object_creation_expression_get_member_name (expr) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Incomplete object creation expression");
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		__temp334 = NULL;
		constructor_node = (__temp334 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr)))), (__temp334 == NULL ? NULL : g_object_ref (__temp334)));
		__temp335 = NULL;
		type_node = (__temp335 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr)))), (__temp335 == NULL ? NULL : g_object_ref (__temp335)));
		type_args = vala_member_access_get_type_arguments (vala_object_creation_expression_get_member_name (expr));
		if (VALA_IS_METHOD (constructor_node)) {
			ValaCodeNode* __temp337;
			ValaCodeNode* __temp336;
			ValaMethod* __temp338;
			ValaMethod* constructor;
			GList* __temp341;
			__temp337 = NULL;
			__temp336 = NULL;
			type_node = (__temp337 = (__temp336 = vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (constructor_node))), (__temp336 == NULL ? NULL : g_object_ref (__temp336))), (type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL))), __temp337);
			__temp338 = NULL;
			constructor = (__temp338 = VALA_METHOD (constructor_node), (__temp338 == NULL ? NULL : g_object_ref (__temp338)));
			if (!(VALA_IS_CREATION_METHOD (constructor_node))) {
				char* __temp340;
				char* __temp339;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp340 = NULL;
				__temp339 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp340 = g_strdup_printf ("`%s' is not a creation method", (__temp339 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (constructor)))))));
				(__temp340 = (g_free (__temp340), NULL));
				(__temp339 = (g_free (__temp339), NULL));
				(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
				(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
				(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (constructor))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			__temp341 = NULL;
			type_args = (__temp341 = vala_member_access_get_type_arguments ((VALA_MEMBER_ACCESS (vala_member_access_get_inner (vala_object_creation_expression_get_member_name (expr))))), (type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL))), __temp341);
			(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
		}
		if (VALA_IS_CLASS (type_node) || VALA_IS_STRUCT (type_node)) {
			ValaDataType* __temp343;
			ValaDataType* __temp342;
			__temp343 = NULL;
			__temp342 = NULL;
			type = (__temp343 = (__temp342 = VALA_DATA_TYPE (type_node), (__temp342 == NULL ? NULL : g_object_ref (__temp342))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp343);
		} else {
			char* __temp345;
			char* __temp344;
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp345 = NULL;
			__temp344 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp345 = g_strdup_printf ("`%s' is not a class or struct", (__temp344 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (type)))))));
			(__temp345 = (g_free (__temp345), NULL));
			(__temp344 = (g_free (__temp344), NULL));
			(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
			(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
			(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		__temp346 = NULL;
		(vala_object_creation_expression_set_type_reference (expr, (__temp346 = vala_type_reference_new ())), vala_object_creation_expression_get_type_reference (expr));
		(__temp346 == NULL ? NULL : (__temp346 = (g_object_unref (__temp346), NULL)));
		(vala_type_reference_set_data_type (vala_object_creation_expression_get_type_reference (expr), type), vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr)));
		{
			GList* __temp347;
			GList* type_arg_it;
			__temp347 = NULL;
			__temp347 = type_args;
			for (type_arg_it = __temp347; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
				ValaTypeReference* type_arg;
				type_arg = type_arg_it->data;
				{
					vala_type_reference_add_type_argument (vala_object_creation_expression_get_type_reference (expr), type_arg);
				}
			}
		}
		(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
		(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
		(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
	} else {
		ValaDataType* __temp349;
		ValaDataType* __temp348;
		__temp349 = NULL;
		__temp348 = NULL;
		type = (__temp349 = (__temp348 = vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr)), (__temp348 == NULL ? NULL : g_object_ref (__temp348))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp349);
	}
	if (!vala_data_type_is_reference_type (type)) {
		char* __temp351;
		char* __temp350;
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp351 = NULL;
		__temp350 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp351 = g_strdup_printf ("Can't create instance of value type `%s'", (__temp350 = vala_type_reference_to_string (vala_object_creation_expression_get_type_reference (expr))))));
		(__temp351 = (g_free (__temp351), NULL));
		(__temp350 = (g_free (__temp350), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (type)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	__temp352 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp352 = vala_type_reference_copy (vala_object_creation_expression_get_type_reference (expr)))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp352 = (g_object_unref (__temp352), NULL));
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	if (VALA_IS_CLASS (type)) {
		ValaClass* __temp353;
		ValaClass* cl;
		__temp353 = NULL;
		cl = (__temp353 = VALA_CLASS (type), (__temp353 == NULL ? NULL : g_object_ref (__temp353)));
		if (vala_class_get_is_abstract (cl)) {
			char* __temp355;
			char* __temp354;
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp355 = NULL;
			__temp354 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp355 = g_strdup_printf ("Can't create instance of abstract class `%s'", (__temp354 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))))));
			(__temp355 = (g_free (__temp355), NULL));
			(__temp354 = (g_free (__temp354), NULL));
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_class_get_default_construction_method (cl) != NULL) {
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_class_get_default_construction_method (cl)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
		}
		while (cl != NULL) {
			ValaClass* __temp357;
			ValaClass* __temp356;
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				(vala_type_reference_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
				break;
			}
			__temp357 = NULL;
			__temp356 = NULL;
			cl = (__temp357 = (__temp356 = vala_class_get_base_class (cl), (__temp356 == NULL ? NULL : g_object_ref (__temp356))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp357);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (type)) {
			ValaStruct* __temp358;
			ValaStruct* st;
			__temp358 = NULL;
			st = (__temp358 = VALA_STRUCT (type), (__temp358 == NULL ? NULL : g_object_ref (__temp358)));
			if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_struct_get_default_construction_method (st) != NULL) {
				(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_struct_get_default_construction_method (st)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		}
	}
	__temp359 = NULL;
	if ((__temp360 = vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && g_list_length ((__temp359 = vala_object_creation_expression_get_argument_list (expr))) != 0, (__temp359 == NULL ? NULL : (__temp359 = (g_list_free (__temp359), NULL))), __temp360)) {
		char* __temp362;
		char* __temp361;
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp362 = NULL;
		__temp361 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp362 = g_strdup_printf ("No arguments allowed when constructing type `%s'", (__temp361 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (type)))))));
		(__temp362 = (g_free (__temp362), NULL));
		(__temp361 = (g_free (__temp361), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
		ValaMethod* __temp363;
		ValaMethod* m;
		GList* __temp365;
		GList* __temp364;
		__temp363 = NULL;
		m = (__temp363 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))), (__temp363 == NULL ? NULL : g_object_ref (__temp363)));
		__temp365 = NULL;
		__temp364 = NULL;
		vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp364 = vala_invokable_get_parameters (VALA_INVOKABLE (m))), (__temp365 = vala_object_creation_expression_get_argument_list (expr)));
		(__temp365 == NULL ? NULL : (__temp365 = (g_list_free (__temp365), NULL)));
		(__temp364 == NULL ? NULL : (__temp364 = (g_list_free (__temp364), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SIZEOF_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->ulong_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TYPEOF_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->type_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type)
{
	ValaStruct* __temp367;
	ValaStruct* st;
	gboolean __temp368;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	__temp367 = NULL;
	st = (__temp367 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp367 == NULL ? NULL : g_object_ref (__temp367)));
	return (__temp368 = vala_struct_is_integer_type (st) || vala_struct_is_floating_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp368);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type)
{
	ValaStruct* __temp370;
	ValaStruct* st;
	gboolean __temp371;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	__temp370 = NULL;
	st = (__temp370 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp370 == NULL ? NULL : g_object_ref (__temp370)));
	return (__temp371 = vala_struct_is_integer_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp371);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_unary_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_PLUS || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_MINUS) {
		/* integer or floating point type*/
		if (!vala_semantic_analyzer_is_numeric_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
			char* __temp373;
			char* __temp372;
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp373 = NULL;
			__temp372 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp373 = g_strdup_printf ("Operator not supported for `%s'", (__temp372 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
			(__temp373 = (g_free (__temp373), NULL));
			(__temp372 = (g_free (__temp372), NULL));
			return;
		}
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	} else {
		if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_LOGICAL_NEGATION) {
			/* boolean type*/
			if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
				char* __temp375;
				char* __temp374;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp375 = NULL;
				__temp374 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp375 = g_strdup_printf ("Operator not supported for `%s'", (__temp374 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
				(__temp375 = (g_free (__temp375), NULL));
				(__temp374 = (g_free (__temp374), NULL));
				return;
			}
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		} else {
			if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT) {
				/* integer type*/
				if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
					char* __temp377;
					char* __temp376;
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					__temp377 = NULL;
					__temp376 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp377 = g_strdup_printf ("Operator not supported for `%s'", (__temp376 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
					(__temp377 = (g_free (__temp377), NULL));
					(__temp376 = (g_free (__temp376), NULL));
					return;
				}
				(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			} else {
				if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_DECREMENT) {
					ValaMemberAccess* ma;
					ValaMemberAccess* old_value;
					ValaBinaryExpression* bin;
					ValaAssignment* assignment;
					/* integer type*/
					if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
						char* __temp379;
						char* __temp378;
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						__temp379 = NULL;
						__temp378 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp379 = g_strdup_printf ("Operator not supported for `%s'", (__temp378 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
						(__temp379 = (g_free (__temp379), NULL));
						(__temp378 = (g_free (__temp378), NULL));
						return;
					}
					ma = vala_semantic_analyzer_find_member_access (self, vala_unary_expression_get_inner (expr));
					if (ma == NULL) {
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Prefix operators not supported for this expression");
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
					bin = vala_binary_expression_new ((vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT ? VALA_BINARY_OPERATOR_PLUS : VALA_BINARY_OPERATOR_MINUS), VALA_EXPRESSION (old_value), VALA_EXPRESSION (vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("1", NULL)), NULL)), NULL);
					assignment = vala_assignment_new (VALA_EXPRESSION (ma), VALA_EXPRESSION (bin), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL);
					vala_code_node_replace (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (assignment));
					vala_code_node_accept (VALA_CODE_NODE (assignment), VALA_CODE_VISITOR (self));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
					return;
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
				} else {
					if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_REF) {
						/* value type*/
						(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
					} else {
						if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_OUT) {
							/* reference type*/
							(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
						} else {
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported unary operator");
							return;
						}
					}
				}
			}
		}
	}
}


static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), NULL);
	if (VALA_IS_PARENTHESIZED_EXPRESSION (expr)) {
		ValaParenthesizedExpression* __temp380;
		ValaParenthesizedExpression* pe;
		ValaMemberAccess* __temp381;
		__temp380 = NULL;
		pe = (__temp380 = VALA_PARENTHESIZED_EXPRESSION (expr), (__temp380 == NULL ? NULL : g_object_ref (__temp380)));
		__temp381 = NULL;
		return (__temp381 = vala_semantic_analyzer_find_member_access (self, vala_parenthesized_expression_get_inner (pe)), (pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL))), __temp381);
		(pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaMemberAccess* __temp382;
		__temp382 = NULL;
		return (__temp382 = VALA_MEMBER_ACCESS (expr), (__temp382 == NULL ? NULL : g_object_ref (__temp382)));
	}
	return VALA_MEMBER_ACCESS (NULL);
}


static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CAST_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) == NULL && vala_type_reference_get_type_parameter (vala_cast_expression_get_type_reference (expr)) == NULL) {
		return;
	}
	/* if type resolving didn't succeed, skip this check 
	 FIXME: check whether cast is allowed*/
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_cast_expression_get_type_reference (expr)), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr)
{
	ValaSemanticAnalyzer * self;
	ValaPointer* __temp385;
	ValaPointer* pointer;
	ValaTypeReference* __temp386;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_POINTER_INDIRECTION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_pointer_indirection_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (!(VALA_IS_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Pointer indirection not supported for this expression");
		return;
	}
	__temp385 = NULL;
	pointer = (__temp385 = VALA_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))), (__temp385 == NULL ? NULL : g_object_ref (__temp385)));
	__temp386 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp386 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp386 == NULL ? NULL : (__temp386 = (g_object_unref (__temp386), NULL)));
	(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_pointer_get_referent_type (pointer)), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(pointer == NULL ? NULL : (pointer = (g_object_unref (pointer), NULL)));
}


static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp387;
	ValaPointer* __temp388;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ADDRESSOF_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_addressof_expression_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Address-of operator not supported for this expression");
		return;
	}
	__temp387 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp387 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp387 == NULL ? NULL : (__temp387 = (g_object_unref (__temp387), NULL)));
	__temp388 = NULL;
	(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp388 = vala_data_type_get_pointer (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(__temp388 = (g_object_unref (__temp388), NULL));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)))), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp389;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_REFERENCE_TRANSFER_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_reference_transfer_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (!(VALA_IS_MEMBER_ACCESS (vala_reference_transfer_expression_get_inner (expr)) || VALA_IS_ELEMENT_ACCESS (vala_reference_transfer_expression_get_inner (expr)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Reference transfer not supported for this expression");
		return;
	}
	if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "No reference to be transferred");
		return;
	}
	__temp389 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp389 = vala_type_reference_copy (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp389 = (g_object_unref (__temp389), NULL));
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type)
{
	ValaStruct* __temp391;
	ValaStruct* left;
	ValaStruct* __temp392;
	ValaStruct* right;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (left_type), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (right_type), NULL);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (left_type))) || !(VALA_IS_STRUCT (vala_type_reference_get_data_type (right_type)))) {
		return VALA_TYPE_REFERENCE (NULL);
	}
	/* at least one operand not struct*/
	__temp391 = NULL;
	left = (__temp391 = VALA_STRUCT (vala_type_reference_get_data_type (left_type)), (__temp391 == NULL ? NULL : g_object_ref (__temp391)));
	__temp392 = NULL;
	right = (__temp392 = VALA_STRUCT (vala_type_reference_get_data_type (right_type)), (__temp392 == NULL ? NULL : g_object_ref (__temp392)));
	if ((!vala_struct_is_floating_type (left) && !vala_struct_is_integer_type (left)) || (!vala_struct_is_floating_type (right) && !vala_struct_is_integer_type (right))) {
		gpointer __temp393;
		return VALA_TYPE_REFERENCE ((__temp393 = NULL, (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp393));
	}
	/* at least one operand not numeric*/
	if (vala_struct_is_floating_type (left) == vala_struct_is_floating_type (right)) {
		/* both operands integer or floating type*/
		if (vala_struct_get_rank (left) >= vala_struct_get_rank (right)) {
			ValaTypeReference* __temp394;
			__temp394 = NULL;
			return (__temp394 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp394);
		} else {
			ValaTypeReference* __temp395;
			__temp395 = NULL;
			return (__temp395 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp395);
		}
	} else {
		/* one integer and one floating type operand*/
		if (vala_struct_is_floating_type (left)) {
			ValaTypeReference* __temp396;
			__temp396 = NULL;
			return (__temp396 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp396);
		} else {
			ValaTypeReference* __temp397;
			__temp397 = NULL;
			return (__temp397 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp397);
		}
	}
	(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
	(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
}


static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_left (expr))) || vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_right (expr)))) {
		/* if there were any errors in inner expressions, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS) {
		ValaInvocationExpression* concat_call;
		if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->string_type)) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be strings");
			return;
		}
		/* string concatenation: convert to a.concat (b) */
		concat_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "concat", NULL)), NULL);
		vala_invocation_expression_add_argument (concat_call, vala_binary_expression_get_right (expr));
		vala_code_node_replace (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (concat_call));
		vala_code_node_accept (VALA_CODE_NODE (concat_call), VALA_CODE_VISITOR (self));
		(concat_call == NULL ? NULL : (concat_call = (g_object_unref (concat_call), NULL)));
	} else {
		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MINUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MUL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_DIV) {
			ValaTypeReference* __temp398;
			__temp398 = NULL;
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp398 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			(__temp398 == NULL ? NULL : (__temp398 = (g_object_unref (__temp398), NULL)));
			if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
				char* __temp401;
				char* __temp400;
				char* __temp399;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp401 = NULL;
				__temp400 = NULL;
				__temp399 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp401 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (__temp399 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp400 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
				(__temp401 = (g_free (__temp401), NULL));
				(__temp400 = (g_free (__temp400), NULL));
				(__temp399 = (g_free (__temp399), NULL));
				return;
			}
		} else {
			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MOD || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_LEFT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_RIGHT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_XOR) {
				ValaTypeReference* __temp402;
				__temp402 = NULL;
				(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp402 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
				(__temp402 == NULL ? NULL : (__temp402 = (g_object_unref (__temp402), NULL)));
				if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
					char* __temp405;
					char* __temp404;
					char* __temp403;
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					__temp405 = NULL;
					__temp404 = NULL;
					__temp403 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp405 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (__temp403 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp404 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
					(__temp405 = (g_free (__temp405), NULL));
					(__temp404 = (g_free (__temp404), NULL));
					(__temp403 = (g_free (__temp403), NULL));
					return;
				}
			} else {
				if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL) {
					if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
						ValaInvocationExpression* cmp_call;
						ValaLiteralExpression* __temp406;
						/* string comparison: convert to a.collate (b) OP 0 */
						cmp_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL)), NULL);
						vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
						(vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call)), vala_binary_expression_get_left (expr));
						__temp406 = NULL;
						(vala_binary_expression_set_right (expr, VALA_EXPRESSION ((__temp406 = vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("0", NULL)), NULL)))), vala_binary_expression_get_right (expr));
						(__temp406 == NULL ? NULL : (__temp406 = (g_object_unref (__temp406), NULL)));
						vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
						(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
					} else {
						ValaTypeReference* resulting_type;
						resulting_type = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)));
						if (resulting_type == NULL) {
							char* __temp409;
							char* __temp408;
							char* __temp407;
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							__temp409 = NULL;
							__temp408 = NULL;
							__temp407 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp409 = g_strdup_printf ("Relational operation not supported for types `%s' and `%s'", (__temp407 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp408 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
							(__temp409 = (g_free (__temp409), NULL));
							(__temp408 = (g_free (__temp408), NULL));
							(__temp407 = (g_free (__temp407), NULL));
							(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
							return;
						}
						(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
					}
					(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
				} else {
					if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY) {
						/* relational operation */
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_right (expr)), vala_expression_get_static_type (vala_binary_expression_get_left (expr))) && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))) {
							char* __temp412;
							char* __temp411;
							char* __temp410;
							__temp412 = NULL;
							__temp411 = NULL;
							__temp410 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp412 = g_strdup_printf ("Equality operation: `%s' and `%s' are incompatible, comparison would always evaluate to false", (__temp410 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))), (__temp411 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))))));
							(__temp412 = (g_free (__temp412), NULL));
							(__temp411 = (g_free (__temp411), NULL));
							(__temp410 = (g_free (__temp410), NULL));
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							return;
						}
						if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
							ValaInvocationExpression* cmp_call;
							ValaLiteralExpression* __temp413;
							/* string comparison: convert to a.collate (b) OP 0 */
							cmp_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL)), NULL);
							vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
							(vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call)), vala_binary_expression_get_left (expr));
							__temp413 = NULL;
							(vala_binary_expression_set_right (expr, VALA_EXPRESSION ((__temp413 = vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("0", NULL)), NULL)))), vala_binary_expression_get_right (expr));
							(__temp413 == NULL ? NULL : (__temp413 = (g_object_unref (__temp413), NULL)));
							vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
							(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
						}
						(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
					} else {
						if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_OR) {
							/* integer type or flags type*/
							(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_binary_expression_get_left (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
						} else {
							if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_OR) {
								if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) != vala_type_reference_get_data_type (self->priv->bool_type) || vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be boolean");
								}
								(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
							} else {
								g_assert_not_reached ();
							}
						}
					}
				}
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TYPE_CHECK (expr));
	if (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr)) == NULL) {
		/* if type resolving didn't succeed, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GList* types)
{
	gboolean null_found;
	gboolean class_or_iface_found;
	gboolean type_param_found;
	gboolean ref_struct_found;
	gboolean val_struct_found;
	gboolean enum_found;
	gboolean callback_found;
	ValaTypeReference* base_type;
	ValaTypeReference* last_type;
	ValaTypeReference* __temp428;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	null_found = FALSE;
	class_or_iface_found = FALSE;
	type_param_found = FALSE;
	ref_struct_found = FALSE;
	val_struct_found = FALSE;
	enum_found = FALSE;
	callback_found = FALSE;
	base_type = NULL;
	last_type = NULL;
	{
		GList* __temp424;
		GList* type_it;
		__temp424 = NULL;
		__temp424 = types;
		for (type_it = __temp424; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type;
			type = type_it->data;
			{
				ValaTypeReference* __temp415;
				ValaTypeReference* __temp414;
				__temp415 = NULL;
				__temp414 = NULL;
				last_type = (__temp415 = (__temp414 = type, (__temp414 == NULL ? NULL : g_object_ref (__temp414))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp415);
				if (vala_code_node_get_error (VALA_CODE_NODE (type))) {
					ValaTypeReference* __temp416;
					ValaTypeReference* __temp417;
					__temp416 = NULL;
					base_type = (__temp416 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp416);
					(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
					__temp417 = NULL;
					return (__temp417 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp417);
				}
				if (vala_type_reference_get_data_type (type) == NULL && vala_type_reference_get_type_parameter (type) == NULL) {
					if (!null_found) {
						null_found = TRUE;
						if (val_struct_found || enum_found) {
							(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
							break;
						}
					}
				} else {
					if (VALA_IS_CLASS (vala_type_reference_get_data_type (type)) || VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
						if (!class_or_iface_found) {
							class_or_iface_found = TRUE;
							if (type_param_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
								(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
								break;
							}
						}
					} else {
						if (vala_type_reference_get_type_parameter (type) != NULL) {
							if (!type_param_found) {
								type_param_found = TRUE;
								if (class_or_iface_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
									(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
									break;
								}
							}
						} else {
							if (VALA_IS_STRUCT (vala_type_reference_get_data_type (type))) {
								ValaStruct* __temp418;
								ValaStruct* st;
								__temp418 = NULL;
								st = (__temp418 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp418 == NULL ? NULL : g_object_ref (__temp418)));
								if (vala_data_type_is_reference_type (VALA_DATA_TYPE (st))) {
									if (!ref_struct_found) {
										ref_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || val_struct_found || enum_found || callback_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								} else {
									if (!val_struct_found) {
										val_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || enum_found || callback_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								}
								(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
							} else {
								if (VALA_IS_ENUM (vala_type_reference_get_data_type (type))) {
									if (!enum_found) {
										enum_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								} else {
									if (VALA_IS_CALLBACK (vala_type_reference_get_data_type (type))) {
										if (!callback_found) {
											callback_found = TRUE;
											if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found || enum_found) {
												(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
												break;
											}
										}
									} else {
										ValaTypeReference* __temp419;
										char* __temp421;
										char* __temp420;
										ValaTypeReference* __temp422;
										__temp419 = NULL;
										base_type = (__temp419 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp419);
										(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
										__temp421 = NULL;
										__temp420 = NULL;
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (type)), (__temp421 = g_strdup_printf ("internal error: unsupported type `%s'", (__temp420 = vala_type_reference_to_string (type)))));
										(__temp421 = (g_free (__temp421), NULL));
										(__temp420 = (g_free (__temp420), NULL));
										__temp422 = NULL;
										return (__temp422 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp422);
									}
								}
							}
						}
					}
				}
				if (base_type == NULL) {
					ValaTypeReference* __temp423;
					__temp423 = NULL;
					base_type = (__temp423 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp423);
					(vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type)), vala_type_reference_get_data_type (base_type));
					(vala_type_reference_set_type_parameter (base_type, vala_type_reference_get_type_parameter (type)), vala_type_reference_get_type_parameter (base_type));
					(vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (type)), vala_type_reference_get_non_null (base_type));
					(vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (type)), vala_type_reference_get_is_null (base_type));
					(vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (type)), vala_type_reference_get_transfers_ownership (base_type));
				} else {
					if (vala_type_reference_get_data_type (base_type) != vala_type_reference_get_data_type (type)) {
						if (vala_semantic_analyzer_is_type_compatible (self, type, base_type)) {
						} else {
							if (vala_semantic_analyzer_is_type_compatible (self, base_type, type)) {
								(vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type)), vala_type_reference_get_data_type (base_type));
							} else {
								(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
								break;
							}
						}
					}
					(vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (base_type) && vala_type_reference_get_non_null (type)), vala_type_reference_get_non_null (base_type));
					(vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (base_type) && vala_type_reference_get_is_null (type)), vala_type_reference_get_is_null (base_type));
					/* if one subexpression transfers ownership, all subexpressions must transfer ownership
					 FIXME add ref calls to subexpressions that don't transfer ownership*/
					(vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (base_type) || vala_type_reference_get_transfers_ownership (type)), vala_type_reference_get_transfers_ownership (base_type));
				}
			}
		}
	}
	if (base_type != NULL && vala_code_node_get_error (VALA_CODE_NODE (base_type))) {
		char* __temp427;
		char* __temp426;
		char* __temp425;
		__temp427 = NULL;
		__temp426 = NULL;
		__temp425 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (last_type)), (__temp427 = g_strdup_printf ("`%s' is incompatible with `%s'", (__temp425 = vala_type_reference_to_string (last_type)), (__temp426 = vala_type_reference_to_string (base_type)))));
		(__temp427 = (g_free (__temp427), NULL));
		(__temp426 = (g_free (__temp426), NULL));
		(__temp425 = (g_free (__temp425), NULL));
	}
	__temp428 = NULL;
	return (__temp428 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp428);
	(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
	(last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr)
{
	ValaSemanticAnalyzer * self;
	GList* types;
	ValaTypeReference* __temp429;
	ValaTypeReference* __temp430;
	ValaTypeReference* __temp431;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONDITIONAL_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_conditional_expression_get_condition (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_conditional_expression_get_condition (expr))), "Condition must be boolean");
		return;
	}
	/* FIXME: support memory management */
	types = NULL;
	__temp429 = NULL;
	types = g_list_append (types, (__temp429 = vala_expression_get_static_type (vala_conditional_expression_get_true_expression (expr)), (__temp429 == NULL ? NULL : g_object_ref (__temp429))));
	__temp430 = NULL;
	types = g_list_append (types, (__temp430 = vala_expression_get_static_type (vala_conditional_expression_get_false_expression (expr)), (__temp430 == NULL ? NULL : g_object_ref (__temp430))));
	__temp431 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp431 = vala_semantic_analyzer_compute_common_base_type (self, types))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp431 == NULL ? NULL : (__temp431 = (g_object_unref (__temp431), NULL)));
	(types == NULL ? NULL : (types = (g_list_foreach (types, (GFunc) g_object_unref, NULL), g_list_free (types), NULL)));
}


static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self)
{
	char* result;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	result = g_strdup_printf ("__lambda%d", self->priv->next_lambda_id);
	self->priv->next_lambda_id++;
	return result;
	(result = (g_free (result), NULL));
}


static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self)
{
	ValaSymbol* __temp433;
	ValaSymbol* sym;
	gpointer __temp438;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	__temp433 = NULL;
	sym = (__temp433 = self->priv->current_symbol, (__temp433 == NULL ? NULL : g_object_ref (__temp433)));
	while (sym != NULL) {
		ValaSymbol* __temp437;
		ValaSymbol* __temp436;
		if (VALA_IS_METHOD (vala_symbol_get_node (sym))) {
			ValaMethod* __temp434;
			ValaMethod* __temp435;
			__temp434 = NULL;
			__temp435 = NULL;
			return (__temp435 = (__temp434 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp434 == NULL ? NULL : g_object_ref (__temp434))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp435);
		}
		__temp437 = NULL;
		__temp436 = NULL;
		sym = (__temp437 = (__temp436 = vala_symbol_get_parent_symbol (sym), (__temp436 == NULL ? NULL : g_object_ref (__temp436))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp437);
	}
	return VALA_METHOD ((__temp438 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp438));
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self)
{
	ValaSymbol* __temp439;
	ValaSymbol* sym;
	gboolean __temp443;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	__temp439 = NULL;
	sym = (__temp439 = self->priv->current_symbol, (__temp439 == NULL ? NULL : g_object_ref (__temp439)));
	while (sym != NULL) {
		ValaSymbol* __temp442;
		ValaSymbol* __temp441;
		if (VALA_IS_CONSTRUCTOR (vala_symbol_get_node (sym))) {
			gboolean __temp440;
			return (__temp440 = TRUE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp440);
		}
		__temp442 = NULL;
		__temp441 = NULL;
		sym = (__temp442 = (__temp441 = vala_symbol_get_parent_symbol (sym), (__temp441 == NULL ? NULL : g_object_ref (__temp441))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp442);
	}
	return (__temp443 = FALSE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp443);
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l)
{
	ValaSemanticAnalyzer * self;
	gboolean in_instance_method;
	ValaMethod* current_method;
	ValaCallback* __temp444;
	ValaCallback* cb;
	ValaMethod* __temp445;
	ValaSymbol* __temp446;
	GList* lambda_params;
	GList* lambda_param_it;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (l));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (l)) == NULL || !(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (l)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (l), TRUE), vala_code_node_get_error (VALA_CODE_NODE (l)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression not allowed in this context");
		return;
	}
	in_instance_method = FALSE;
	current_method = vala_semantic_analyzer_find_current_method (self);
	if (current_method != NULL) {
		in_instance_method = vala_method_get_instance (current_method);
	} else {
		in_instance_method = vala_semantic_analyzer_is_in_constructor (self);
	}
	__temp444 = NULL;
	cb = (__temp444 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (l)))), (__temp444 == NULL ? NULL : g_object_ref (__temp444)));
	__temp445 = NULL;
	(vala_lambda_expression_set_method (l, (__temp445 = vala_method_new (vala_semantic_analyzer_get_lambda_name (self), vala_callback_get_return_type (cb), NULL))), vala_lambda_expression_get_method (l));
	(__temp445 == NULL ? NULL : (__temp445 = (g_object_unref (__temp445), NULL)));
	(vala_method_set_instance (vala_lambda_expression_get_method (l), vala_callback_get_instance (cb) && in_instance_method), vala_method_get_instance (vala_lambda_expression_get_method (l)));
	__temp446 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)), (__temp446 = vala_symbol_new (VALA_CODE_NODE (vala_lambda_expression_get_method (l))))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))));
	(__temp446 == NULL ? NULL : (__temp446 = (g_object_unref (__temp446), NULL)));
	(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))), self->priv->current_symbol), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))));
	lambda_params = vala_lambda_expression_get_parameters (l);
	lambda_param_it = lambda_params;
	{
		GList* __temp449;
		GList* cb_param_it;
		__temp449 = NULL;
		__temp449 = vala_callback_get_parameters (cb);
		for (cb_param_it = __temp449; cb_param_it != NULL; cb_param_it = cb_param_it->next) {
			ValaFormalParameter* cb_param;
			cb_param = cb_param_it->data;
			{
				const char* __temp447;
				char* lambda_param;
				ValaFormalParameter* param;
				ValaSymbol* __temp448;
				if (lambda_param_it == NULL) {
					break;
				}
				/* lambda expressions are allowed to have less parameters */
				__temp447 = NULL;
				lambda_param = (__temp447 = ((char*) lambda_param_it->data), (__temp447 == NULL ? NULL : g_strdup (__temp447)));
				param = vala_formal_parameter_new (lambda_param, vala_formal_parameter_get_type_reference (cb_param), NULL);
				__temp448 = NULL;
				(vala_code_node_set_symbol (VALA_CODE_NODE (param), (__temp448 = vala_symbol_new (VALA_CODE_NODE (param)))), vala_code_node_get_symbol (VALA_CODE_NODE (param)));
				(__temp448 == NULL ? NULL : (__temp448 = (g_object_unref (__temp448), NULL)));
				vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))), vala_formal_parameter_get_name (param), vala_code_node_get_symbol (VALA_CODE_NODE (param)));
				vala_method_add_parameter (vala_lambda_expression_get_method (l), param);
				lambda_param_it = lambda_param_it->next;
				(lambda_param = (g_free (lambda_param), NULL));
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(__temp449 == NULL ? NULL : (__temp449 = (g_list_free (__temp449), NULL)));
	}
	if (lambda_param_it != NULL) {
		/* lambda expressions may not expect more parameters */
		(vala_code_node_set_error (VALA_CODE_NODE (l), TRUE), vala_code_node_get_error (VALA_CODE_NODE (l)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression: too many parameters");
		(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
		(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
		(lambda_params == NULL ? NULL : (lambda_params = (g_list_free (lambda_params), NULL)));
		return;
	}
	if (vala_lambda_expression_get_expression_body (l) != NULL) {
		ValaBlock* block;
		ValaSymbol* __temp450;
		block = vala_block_new (NULL);
		__temp450 = NULL;
		(vala_code_node_set_symbol (VALA_CODE_NODE (block), (__temp450 = vala_symbol_new (VALA_CODE_NODE (block)))), vala_code_node_get_symbol (VALA_CODE_NODE (block)));
		(__temp450 == NULL ? NULL : (__temp450 = (g_object_unref (__temp450), NULL)));
		(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (block)), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (block))));
		if (vala_type_reference_get_data_type (vala_method_get_return_type (vala_lambda_expression_get_method (l))) != NULL) {
			ValaReturnStatement* __temp451;
			__temp451 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((__temp451 = vala_return_statement_new (vala_lambda_expression_get_expression_body (l), NULL))));
			(__temp451 == NULL ? NULL : (__temp451 = (g_object_unref (__temp451), NULL)));
		} else {
			ValaExpressionStatement* __temp452;
			__temp452 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((__temp452 = vala_expression_statement_new (vala_lambda_expression_get_expression_body (l), NULL))));
			(__temp452 == NULL ? NULL : (__temp452 = (g_object_unref (__temp452), NULL)));
		}
		(vala_method_set_body (vala_lambda_expression_get_method (l), block), vala_method_get_body (vala_lambda_expression_get_method (l)));
		(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
	} else {
		(vala_method_set_body (vala_lambda_expression_get_method (l), vala_lambda_expression_get_statement_body (l)), vala_method_get_body (vala_lambda_expression_get_method (l)));
		(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_method_get_body (vala_lambda_expression_get_method (l)))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_method_get_body (vala_lambda_expression_get_method (l))))));
	}
	/* lambda expressions should be usable like MemberAccess of a method */
	(vala_expression_set_symbol_reference (VALA_EXPRESSION (l), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (l)));
	(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
	(lambda_params == NULL ? NULL : (lambda_params = (g_list_free (lambda_params), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp453;
		ValaMemberAccess* ma;
		__temp453 = NULL;
		ma = (__temp453 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp453 == NULL ? NULL : g_object_ref (__temp453)));
		if (vala_code_node_get_error (VALA_CODE_NODE (ma)) || vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		/* if no symbol found, skip this check */
		if (VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
			ValaSignal* __temp454;
			ValaSignal* sig;
			ValaTypeReference* __temp455;
			ValaCallback* __temp456;
			__temp454 = NULL;
			sig = (__temp454 = VALA_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp454 == NULL ? NULL : g_object_ref (__temp454)));
			__temp455 = NULL;
			(vala_expression_set_expected_type (vala_assignment_get_right (a), (__temp455 = vala_type_reference_new ())), vala_expression_get_expected_type (vala_assignment_get_right (a)));
			(__temp455 == NULL ? NULL : (__temp455 = (g_object_unref (__temp455), NULL)));
			__temp456 = NULL;
			(vala_type_reference_set_data_type (vala_expression_get_expected_type (vala_assignment_get_right (a)), VALA_DATA_TYPE ((__temp456 = vala_signal_get_callback (sig)))), vala_type_reference_get_data_type (vala_expression_get_expected_type (vala_assignment_get_right (a))));
			(__temp456 = (g_object_unref (__temp456), NULL));
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
		} else {
			/* do nothing*/
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a))) {
			} else {
				/* do nothing*/
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "unsupported lvalue in assignment");
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (vala_code_node_get_error (VALA_CODE_NODE (a)) || vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_left (a))) || vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_right (a)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
		return;
	}
	if (vala_assignment_get_operator (a) != VALA_ASSIGNMENT_OPERATOR_SIMPLE && VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp457;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		__temp457 = NULL;
		ma = (__temp457 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp457 == NULL ? NULL : g_object_ref (__temp457)));
		if (!(VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))))) {
			ValaMemberAccess* old_value;
			ValaBinaryExpression* bin;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			bin = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, VALA_EXPRESSION (old_value), VALA_EXPRESSION (vala_parenthesized_expression_new (vala_assignment_get_right (a), vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))))), NULL);
			if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR), vala_binary_expression_get_operator (bin));
			} else {
				if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND), vala_binary_expression_get_operator (bin));
				} else {
					if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR), vala_binary_expression_get_operator (bin));
					} else {
						if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_ADD) {
							(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS), vala_binary_expression_get_operator (bin));
						} else {
							if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SUB) {
								(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS), vala_binary_expression_get_operator (bin));
							} else {
								if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_MUL) {
									(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL), vala_binary_expression_get_operator (bin));
								} else {
									if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_DIV) {
										(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV), vala_binary_expression_get_operator (bin));
									} else {
										if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD), vala_binary_expression_get_operator (bin));
										} else {
											if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT), vala_binary_expression_get_operator (bin));
											} else {
												if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT), vala_binary_expression_get_operator (bin));
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			(vala_assignment_set_right (a, VALA_EXPRESSION (bin)), vala_assignment_get_right (a));
			vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (a)), VALA_CODE_VISITOR (self));
			(vala_assignment_set_operator (a, VALA_ASSIGNMENT_OPERATOR_SIMPLE), vala_assignment_get_operator (a));
			(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
			(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp458;
		ValaMemberAccess* ma;
		__temp458 = NULL;
		ma = (__temp458 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp458 == NULL ? NULL : g_object_ref (__temp458)));
		if (VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
			ValaSignal* __temp459;
			ValaSignal* sig;
			ValaMethod* __temp460;
			ValaMethod* m;
			__temp459 = NULL;
			sig = (__temp459 = VALA_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp459 == NULL ? NULL : g_object_ref (__temp459)));
			if (vala_expression_get_symbol_reference (vala_assignment_get_right (a)) == NULL) {
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "unsupported expression for signal handler");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			__temp460 = NULL;
			m = (__temp460 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_assignment_get_right (a)))), (__temp460 == NULL ? NULL : g_object_ref (__temp460)));
			if (vala_method_get_instance (m) && m->access != VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
				/* TODO: generate wrapper function */
				(vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE), vala_code_node_get_error (VALA_CODE_NODE (ma)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "public instance methods not yet supported as signal handlers");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			if (vala_method_get_instance (m)) {
				/* instance signal handlers must have the self
				 * parameter at the end
				 * do not use G_CONNECT_SWAPPED as this would
				 * rearrange the parameters for instance
				 * methods and non-instance methods
				 */
				(vala_method_set_instance_last (m, TRUE), vala_method_get_instance_last (m));
			}
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			if (VALA_IS_PROPERTY (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
				ValaProperty* __temp461;
				ValaProperty* prop;
				__temp461 = NULL;
				prop = (__temp461 = VALA_PROPERTY (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp461 == NULL ? NULL : g_object_ref (__temp461)));
				if (vala_property_get_set_accessor (prop) == NULL) {
					char* __temp463;
					char* __temp462;
					(vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE), vala_code_node_get_error (VALA_CODE_NODE (ma)));
					__temp463 = NULL;
					__temp462 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ma)), (__temp463 = g_strdup_printf ("Property `%s' is read-only", (__temp462 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))));
					(__temp463 = (g_free (__temp463), NULL));
					(__temp462 = (g_free (__temp462), NULL));
					(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					return;
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			} else {
				if (VALA_IS_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
					ValaVariableDeclarator* __temp464;
					ValaVariableDeclarator* decl;
					ValaMemberAccess* __temp465;
					ValaMemberAccess* right_ma;
					__temp464 = NULL;
					decl = (__temp464 = VALA_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp464 == NULL ? NULL : g_object_ref (__temp464)));
					__temp465 = NULL;
					right_ma = (__temp465 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (__temp465 == NULL ? NULL : g_object_ref (__temp465)));
					if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
						ValaMethod* __temp466;
						ValaMethod* m;
						ValaCallback* __temp467;
						ValaCallback* cb;
						__temp466 = NULL;
						m = (__temp466 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))), (__temp466 == NULL ? NULL : g_object_ref (__temp466)));
						__temp467 = NULL;
						cb = (__temp467 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp467 == NULL ? NULL : g_object_ref (__temp467)));
						/* check whether method matches callback type */
						if (!vala_callback_matches_method (cb, m)) {
							char* __temp470;
							char* __temp469;
							char* __temp468;
							(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
							__temp470 = NULL;
							__temp469 = NULL;
							__temp468 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp470 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp468 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp469 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
							(__temp470 = (g_free (__temp470), NULL));
							(__temp469 = (g_free (__temp469), NULL));
							(__temp468 = (g_free (__temp468), NULL));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(vala_expression_set_static_type (vala_assignment_get_right (a), vala_variable_declarator_get_type_reference (decl)), vala_expression_get_static_type (vala_assignment_get_right (a)));
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					} else {
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
						(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
					(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
				} else {
					if (VALA_IS_FIELD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
						ValaField* __temp471;
						ValaField* f;
						ValaMemberAccess* __temp472;
						ValaMemberAccess* right_ma;
						__temp471 = NULL;
						f = (__temp471 = VALA_FIELD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp471 == NULL ? NULL : g_object_ref (__temp471)));
						__temp472 = NULL;
						right_ma = (__temp472 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (__temp472 == NULL ? NULL : g_object_ref (__temp472)));
						if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))) {
							ValaMethod* __temp473;
							ValaMethod* m;
							ValaCallback* __temp474;
							ValaCallback* cb;
							__temp473 = NULL;
							m = (__temp473 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))), (__temp473 == NULL ? NULL : g_object_ref (__temp473)));
							__temp474 = NULL;
							cb = (__temp474 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), (__temp474 == NULL ? NULL : g_object_ref (__temp474)));
							/* check whether method matches callback type */
							if (!vala_callback_matches_method (cb, m)) {
								char* __temp477;
								char* __temp476;
								char* __temp475;
								(vala_code_node_set_error (VALA_CODE_NODE (f), TRUE), vala_code_node_get_error (VALA_CODE_NODE (f)));
								__temp477 = NULL;
								__temp476 = NULL;
								__temp475 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp477 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp475 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp476 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
								(__temp477 = (g_free (__temp477), NULL));
								(__temp476 = (g_free (__temp476), NULL));
								(__temp475 = (g_free (__temp475), NULL));
								(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
								(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
								(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							(vala_expression_set_static_type (vala_assignment_get_right (a), vala_field_get_type_reference (f)), vala_expression_get_static_type (vala_assignment_get_right (a)));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						} else {
							(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
					} else {
						if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL && vala_expression_get_static_type (vala_assignment_get_right (a)) != NULL) {
							if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
								char* __temp480;
								char* __temp479;
								char* __temp478;
								/* if there was an error on either side,
								 * i.e. a.{left|right}.static_type == null, skip type check */
								(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
								__temp480 = NULL;
								__temp479 = NULL;
								__temp478 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp480 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (__temp478 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (__temp479 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
								(__temp480 = (g_free (__temp480), NULL));
								(__temp479 = (g_free (__temp479), NULL));
								(__temp478 = (g_free (__temp478), NULL));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							if (vala_semantic_analyzer_get_memory_management (self)) {
								if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
									/* rhs transfers ownership of the expression */
									if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
										/* lhs doesn't own the value */
										(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
									}
								} else {
									if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
									}
								}
							}
						}
					}
				}
			}
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		/* lhs wants to own the value
		 * rhs doesn't transfer the ownership
		 * code generator needs to add reference
		 * increment calls */
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
			ValaElementAccess* __temp481;
			ValaElementAccess* ea;
			__temp481 = NULL;
			ea = (__temp481 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (a)), (__temp481 == NULL ? NULL : g_object_ref (__temp481)));
			if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
				char* __temp484;
				char* __temp483;
				char* __temp482;
				/* if there was an error on either side,
				 * i.e. a.{left|right}.static_type == null, skip type check */
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				__temp484 = NULL;
				__temp483 = NULL;
				__temp482 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp484 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (__temp482 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (__temp483 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
				(__temp484 = (g_free (__temp484), NULL));
				(__temp483 = (g_free (__temp483), NULL));
				(__temp482 = (g_free (__temp482), NULL));
				(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
				return;
			}
			if (vala_semantic_analyzer_get_memory_management (self)) {
				if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
					GList* args;
					ValaTypeReference* __temp485;
					ValaTypeReference* element_type;
					/* rhs transfers ownership of the expression */
					args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (ea)));
					if (g_list_length (args) != 1) {
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ea)), "internal error: array reference without type arguments");
						(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					__temp485 = NULL;
					element_type = (__temp485 = VALA_TYPE_REFERENCE (args->data), (__temp485 == NULL ? NULL : g_object_ref (__temp485)));
					if (!vala_type_reference_get_takes_ownership (element_type)) {
						/* lhs doesn't own the value */
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
						(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
						(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
					(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
				} else {
					if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
					}
				}
			}
			(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
		} else {
			return;
		}
	}
	/* lhs wants to own the value
	 * rhs doesn't transfer the ownership
	 * code generator needs to add reference
	 * increment calls */
	(vala_expression_set_static_type (VALA_EXPRESSION (a), vala_expression_get_static_type (vala_assignment_get_left (a))), vala_expression_get_static_type (VALA_EXPRESSION (a)));
}


gboolean vala_semantic_analyzer_get_memory_management (ValaSemanticAnalyzer* self)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	return self->priv->_memory_management;
}


void vala_semantic_analyzer_set_memory_management (ValaSemanticAnalyzer* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	self->priv->_memory_management = value;
}


static void vala_semantic_analyzer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		g_value_set_boolean (value, vala_semantic_analyzer_get_memory_management (self));
		break;
	}
}


static void vala_semantic_analyzer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		vala_semantic_analyzer_set_memory_management (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_semantic_analyzer_class_init (ValaSemanticAnalyzerClass * klass)
{
	vala_semantic_analyzer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSemanticAnalyzerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_semantic_analyzer_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_semantic_analyzer_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_semantic_analyzer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_semantic_analyzer_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_semantic_analyzer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_semantic_analyzer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_semantic_analyzer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_semantic_analyzer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_callback = vala_semantic_analyzer_real_visit_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_semantic_analyzer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_semantic_analyzer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_semantic_analyzer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_semantic_analyzer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_semantic_analyzer_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_semantic_analyzer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_semantic_analyzer_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_semantic_analyzer_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_semantic_analyzer_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_semantic_analyzer_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_semantic_analyzer_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_block = vala_semantic_analyzer_real_visit_begin_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_block = vala_semantic_analyzer_real_visit_end_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_semantic_analyzer_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_initializer_list = vala_semantic_analyzer_real_visit_begin_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_initializer_list = vala_semantic_analyzer_real_visit_end_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_semantic_analyzer_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_semantic_analyzer_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_semantic_analyzer_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_semantic_analyzer_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_foreach_statement = vala_semantic_analyzer_real_visit_begin_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_foreach_statement = vala_semantic_analyzer_real_visit_end_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_return_statement = vala_semantic_analyzer_real_visit_end_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_semantic_analyzer_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_array_creation_expression = vala_semantic_analyzer_real_visit_begin_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_array_creation_expression = vala_semantic_analyzer_real_visit_end_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_semantic_analyzer_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_semantic_analyzer_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_semantic_analyzer_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_semantic_analyzer_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_semantic_analyzer_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_semantic_analyzer_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_literal_expression = vala_semantic_analyzer_real_visit_literal_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_semantic_analyzer_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_semantic_analyzer_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_invocation_expression = vala_semantic_analyzer_real_visit_begin_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_invocation_expression = vala_semantic_analyzer_real_visit_end_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_semantic_analyzer_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_semantic_analyzer_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_semantic_analyzer_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_object_creation_expression = vala_semantic_analyzer_real_visit_end_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_sizeof_expression = vala_semantic_analyzer_real_visit_sizeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_semantic_analyzer_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_semantic_analyzer_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_semantic_analyzer_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_semantic_analyzer_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_semantic_analyzer_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_semantic_analyzer_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_semantic_analyzer_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_semantic_analyzer_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_semantic_analyzer_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_lambda_expression = vala_semantic_analyzer_real_visit_begin_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_assignment = vala_semantic_analyzer_real_visit_begin_assignment;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_assignment = vala_semantic_analyzer_real_visit_end_assignment;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT, g_param_spec_boolean ("memory-management", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_semantic_analyzer_init (ValaSemanticAnalyzer * self)
{
	self->priv = VALA_SEMANTIC_ANALYZER_GET_PRIVATE (self);
	self->priv->next_lambda_id = 0;
}


static void vala_semantic_analyzer_dispose (GObject * obj)
{
	ValaSemanticAnalyzer * self;
	ValaSemanticAnalyzerClass * klass;
	GObjectClass * parent_class;
	self = VALA_SEMANTIC_ANALYZER (obj);
	(self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL)));
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL)));
	(self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL)));
	(self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL)));
	(self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL)));
	(self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL)));
	(self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL)));
	(self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL)));
	(self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL)));
	(self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL)));
	(self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL)));
	(self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL)));
	(self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL)));
	(self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL)));
	(self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL)));
	klass = VALA_SEMANTIC_ANALYZER_CLASS (g_type_class_peek (VALA_TYPE_SEMANTIC_ANALYZER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_semantic_analyzer_get_type ()
{
	static GType vala_semantic_analyzer_type_id = 0;
	if (G_UNLIKELY (vala_semantic_analyzer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSemanticAnalyzerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_semantic_analyzer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSemanticAnalyzer), 0, (GInstanceInitFunc) vala_semantic_analyzer_init };
		vala_semantic_analyzer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaSemanticAnalyzer", &g_define_type_info, 0);
	}
	return vala_semantic_analyzer_type_id;
}




