/* valastruct.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valastruct.h"
#include <vala/valamethod.h>
#include <vala/valasourcereference.h>
#include <vala/valatypeparameter.h>
#include <vala/valastruct.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valacodevisitor.h>
#include <vala/valanamespace.h>
#include <vala/valaattribute.h>
#include <vala/valareport.h>
#include <vala/valasymbol.h>
#include <vala/valatypereference.h>

struct _ValaStructPrivate {
	GList* type_parameters;
	GList* constants;
	GList* fields;
	GList* methods;
	GList* base_types;
	char* cname;
	char* const_cname;
	char* dup_function;
	char* free_function;
	char* type_id;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	gboolean reference_type;
	gboolean integer_type;
	gboolean floating_type;
	gint rank;
	char* marshaller_type_name;
	char* get_value_function;
	char* set_value_function;
	char* default_value;
	ValaMethod* _default_construction_method;
};
#define VALA_STRUCT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_STRUCT, ValaStructPrivate))
enum  {
	VALA_STRUCT_DUMMY_PROPERTY,
	VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD,
};
static void vala_struct_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_struct_real_get_cname (ValaDataType* base, gboolean const_type);
static void vala_struct_set_cname (ValaStruct* self, const char* cname);
static void vala_struct_set_const_cname (ValaStruct* self, const char* cname);
static char* vala_struct_real_get_lower_case_cprefix (ValaDataType* base);
static char* vala_struct_get_lower_case_csuffix (ValaStruct* self);
static void vala_struct_set_lower_case_csuffix (ValaStruct* self, const char* csuffix);
static char* vala_struct_real_get_lower_case_cname (ValaDataType* base, const char* infix);
static char* vala_struct_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static gboolean vala_struct_real_is_reference_type (ValaDataType* base);
static void vala_struct_process_ccode_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_ref_type_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_integer_type_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_floating_type_attribute (ValaStruct* self, ValaAttribute* a);
static gboolean vala_struct_real_is_reference_counting (ValaDataType* base);
static char* vala_struct_real_get_dup_function (ValaDataType* base);
static char* vala_struct_real_get_free_function (ValaDataType* base);
static void vala_struct_set_free_function (ValaStruct* self, const char* name);
static char* vala_struct_real_get_type_id (ValaDataType* base);
static char* vala_struct_real_get_marshaller_type_name (ValaDataType* base);
static void vala_struct_set_marshaller_type_name (ValaStruct* self, const char* name);
static char* vala_struct_real_get_get_value_function (ValaDataType* base);
static char* vala_struct_real_get_set_value_function (ValaDataType* base);
static void vala_struct_set_get_value_function (ValaStruct* self, const char* function);
static void vala_struct_set_set_value_function (ValaStruct* self, const char* function);
static char* vala_struct_real_get_default_value (ValaDataType* base);
static void vala_struct_set_default_value (ValaStruct* self, const char* value);
static gint vala_struct_real_get_type_parameter_index (ValaDataType* base, const char* name);
static gpointer vala_struct_parent_class = NULL;
static void vala_struct_dispose (GObject * obj);


/**
 * Creates a new struct.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created struct
 */
ValaStruct* vala_struct_new (const char* _name, ValaSourceReference* source)
{
	g_return_val_if_fail (_name != NULL, NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	GParameter * __params = g_new0 (GParameter, 2);
	GParameter * __params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _name), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	ValaStruct * self = g_object_newv (VALA_TYPE_STRUCT, __params_it - __params, __params);
	return self;
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_struct_add_type_parameter (ValaStruct* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	self->priv->type_parameters = g_list_append (self->priv->type_parameters, g_object_ref (p));
	p->type = VALA_DATA_TYPE (self);
}


/**
 * Adds the specified constant as a member to this struct.
 *
 * @param c a constant
 */
void vala_struct_add_constant (ValaStruct* self, ValaConstant* c)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_CONSTANT (c));
	self->priv->constants = g_list_append (self->priv->constants, g_object_ref (c));
}


/**
 * Adds the specified field as a member to this struct.
 *
 * @param f a field
 */
void vala_struct_add_field (ValaStruct* self, ValaField* f)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	self->priv->fields = g_list_append (self->priv->fields, g_object_ref (f));
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GList* vala_struct_get_fields (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return g_list_copy (self->priv->fields);
}


/**
 * Adds the specified method as a member to this struct.
 *
 * @param m a method
 */
void vala_struct_add_method (ValaStruct* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (m != NULL);
	self->priv->methods = g_list_append (self->priv->methods, g_object_ref (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GList* vala_struct_get_methods (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return g_list_copy (self->priv->methods);
}


static void vala_struct_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaStruct * self = VALA_STRUCT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_begin_struct (visitor, self);
	{
		GList* __temp2 = NULL;
		__temp2 = self->priv->type_parameters;
		GList* p_it;
		for (p_it = __temp2; p_it != NULL; p_it = p_it->next) {
			ValaTypeParameter* p = p_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
			}
		}
	}
	{
		GList* __temp3 = NULL;
		__temp3 = self->priv->fields;
		GList* f_it;
		for (f_it = __temp3; f_it != NULL; f_it = f_it->next) {
			ValaField* f = f_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
			}
		}
	}
	{
		GList* __temp4 = NULL;
		__temp4 = self->priv->constants;
		GList* c_it;
		for (c_it = __temp4; c_it != NULL; c_it = c_it->next) {
			ValaConstant* c = c_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
			}
		}
	}
	{
		GList* __temp5 = NULL;
		__temp5 = self->priv->methods;
		GList* m_it;
		for (m_it = __temp5; m_it != NULL; m_it = m_it->next) {
			ValaMethod* m = m_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
			}
		}
	}
	vala_code_visitor_visit_end_struct (visitor, self);
}


static char* vala_struct_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (const_type && self->priv->const_cname != NULL) {
		const char* __temp6 = NULL;
		return (__temp6 = self->priv->const_cname, (__temp6 == NULL ? NULL : g_strdup (__temp6)));
	}
	if (self->priv->cname == NULL) {
		char* __temp9 = NULL;
		char* __temp8 = NULL;
		self->priv->cname = (__temp9 = g_strdup_printf ("%s%s", (__temp8 = vala_namespace_get_cprefix (VALA_DATA_TYPE (self)->namespace)), vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp9);
		(__temp8 = (g_free (__temp8), NULL));
	}
	const char* __temp10 = NULL;
	return (__temp10 = self->priv->cname, (__temp10 == NULL ? NULL : g_strdup (__temp10)));
}


static void vala_struct_set_cname (ValaStruct* self, const char* cname)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (cname != NULL);
	char* __temp12 = NULL;
	self->priv->cname = (__temp12 = g_strdup (cname), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp12);
}


static void vala_struct_set_const_cname (ValaStruct* self, const char* cname)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (cname != NULL);
	char* __temp13 = NULL;
	self->priv->const_cname = (__temp13 = g_strdup (cname), (self->priv->const_cname = (g_free (self->priv->const_cname), NULL)), __temp13);
}


static char* vala_struct_real_get_lower_case_cprefix (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (self->priv->lower_case_cprefix == NULL) {
		char* __temp15 = NULL;
		char* __temp14 = NULL;
		self->priv->lower_case_cprefix = (__temp15 = g_strdup_printf ("%s_", (__temp14 = vala_data_type_get_lower_case_cname (VALA_DATA_TYPE (self), NULL))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), __temp15);
		(__temp14 = (g_free (__temp14), NULL));
	}
	const char* __temp16 = NULL;
	return (__temp16 = self->priv->lower_case_cprefix, (__temp16 == NULL ? NULL : g_strdup (__temp16)));
}


static char* vala_struct_get_lower_case_csuffix (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* __temp18 = NULL;
		self->priv->lower_case_csuffix = (__temp18 = vala_namespace_camel_case_to_lower_case (vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), __temp18);
	}
	const char* __temp19 = NULL;
	return (__temp19 = self->priv->lower_case_csuffix, (__temp19 == NULL ? NULL : g_strdup (__temp19)));
}


static void vala_struct_set_lower_case_csuffix (ValaStruct* self, const char* csuffix)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (csuffix != NULL);
	char* __temp21 = NULL;
	self->priv->lower_case_csuffix = (__temp21 = g_strdup (csuffix), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), __temp21);
}


static char* vala_struct_real_get_lower_case_cname (ValaDataType* base, const char* infix)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (infix == NULL) {
		infix = "";
	}
	char* __temp23 = NULL;
	char* __temp22 = NULL;
	char* __temp24 = NULL;
	return (__temp24 = g_strdup_printf ("%s%s%s", (__temp22 = vala_namespace_get_lower_case_cprefix (VALA_DATA_TYPE (self)->namespace)), infix, (__temp23 = vala_struct_get_lower_case_csuffix (self))), (__temp23 = (g_free (__temp23), NULL)), (__temp22 = (g_free (__temp22), NULL)), __temp24);
}


static char* vala_struct_real_get_upper_case_cname (ValaDataType* base, const char* infix)
{
	ValaStruct * self = VALA_STRUCT (base);
	char* __temp26 = NULL;
	char* __temp27 = NULL;
	return (__temp27 = g_utf8_strup ((__temp26 = vala_data_type_get_lower_case_cname (VALA_DATA_TYPE (self), infix)), -1), (__temp26 = (g_free (__temp26), NULL)), __temp27);
}


static gboolean vala_struct_real_is_reference_type (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	return self->priv->reference_type;
}


/**
 * Returns whether this is an integer type.
 *
 * @return true if this is an integer type, false otherwise
 */
gboolean vala_struct_is_integer_type (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), FALSE);
	return self->priv->integer_type;
}


/**
 * Returns whether this is a floating point type.
 *
 * @return true if this is a floating point type, false otherwise
 */
gboolean vala_struct_is_floating_type (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), FALSE);
	return self->priv->floating_type;
}


/**
 * Returns the rank of this integer or floating point type.
 *
 * @return the rank if this is an integer or floating point type
 */
gint vala_struct_get_rank (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), 0);
	return self->priv->rank;
}


/**
 * Sets whether this data type has value or reference type semantics.
 *
 * @param ref_type true if this data type has reference type semantics
 */
void vala_struct_set_is_reference_type (ValaStruct* self, gboolean ref_type)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	self->priv->reference_type = ref_type;
}


static void vala_struct_process_ccode_attribute (ValaStruct* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp33 = NULL;
		vala_struct_set_cname (self, (__temp33 = vala_attribute_get_string (a, "cname")));
		(__temp33 = (g_free (__temp33), NULL));
	}
	if (vala_attribute_has_argument (a, "const_cname")) {
		char* __temp34 = NULL;
		vala_struct_set_const_cname (self, (__temp34 = vala_attribute_get_string (a, "const_cname")));
		(__temp34 = (g_free (__temp34), NULL));
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* __temp35 = NULL;
		self->priv->lower_case_cprefix = (__temp35 = vala_attribute_get_string (a, "cprefix"), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), __temp35);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** __temp36 = NULL;
			__temp36 = g_strsplit (val, ",", 0);
			char** filename_it;
			for (filename_it = __temp36; *filename_it != NULL; filename_it = filename_it + 1) {
				char* filename = *filename_it;
				{
					vala_data_type_add_cheader_filename (VALA_DATA_TYPE (self), filename);
				}
			}
			(__temp36 = (g_free (__temp36), NULL));
		}
		(val = (g_free (val), NULL));
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* __temp37 = NULL;
		vala_struct_set_type_id (self, (__temp37 = vala_attribute_get_string (a, "type_id")));
		(__temp37 = (g_free (__temp37), NULL));
	}
	if (vala_attribute_has_argument (a, "marshaller_type_name")) {
		char* __temp38 = NULL;
		vala_struct_set_marshaller_type_name (self, (__temp38 = vala_attribute_get_string (a, "marshaller_type_name")));
		(__temp38 = (g_free (__temp38), NULL));
	}
	if (vala_attribute_has_argument (a, "get_value_function")) {
		char* __temp39 = NULL;
		vala_struct_set_get_value_function (self, (__temp39 = vala_attribute_get_string (a, "get_value_function")));
		(__temp39 = (g_free (__temp39), NULL));
	}
	if (vala_attribute_has_argument (a, "set_value_function")) {
		char* __temp40 = NULL;
		vala_struct_set_set_value_function (self, (__temp40 = vala_attribute_get_string (a, "set_value_function")));
		(__temp40 = (g_free (__temp40), NULL));
	}
	if (vala_attribute_has_argument (a, "default_value")) {
		char* __temp41 = NULL;
		vala_struct_set_default_value (self, (__temp41 = vala_attribute_get_string (a, "default_value")));
		(__temp41 = (g_free (__temp41), NULL));
	}
}


static void vala_struct_process_ref_type_attribute (ValaStruct* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	self->priv->reference_type = TRUE;
	if (vala_attribute_has_argument (a, "dup_function")) {
		char* __temp42 = NULL;
		vala_struct_set_dup_function (self, (__temp42 = vala_attribute_get_string (a, "dup_function")));
		(__temp42 = (g_free (__temp42), NULL));
	}
	if (vala_attribute_has_argument (a, "free_function")) {
		char* __temp43 = NULL;
		vala_struct_set_free_function (self, (__temp43 = vala_attribute_get_string (a, "free_function")));
		(__temp43 = (g_free (__temp43), NULL));
	}
}


static void vala_struct_process_integer_type_attribute (ValaStruct* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	self->priv->integer_type = TRUE;
	if (vala_attribute_has_argument (a, "rank")) {
		self->priv->rank = vala_attribute_get_integer (a, "rank");
	}
}


static void vala_struct_process_floating_type_attribute (ValaStruct* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	self->priv->floating_type = TRUE;
	if (vala_attribute_has_argument (a, "rank")) {
		self->priv->rank = vala_attribute_get_integer (a, "rank");
	}
}


/**
 * Process all associated attributes.
 */
void vala_struct_process_attributes (ValaStruct* self)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	{
		GList* __temp44 = NULL;
		__temp44 = VALA_CODE_NODE (self)->attributes;
		GList* a_it;
		for (a_it = __temp44; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_struct_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "ReferenceType") == 0) {
						vala_struct_process_ref_type_attribute (self, a);
					} else {
						if (g_utf8_collate (vala_attribute_get_name (a), "IntegerType") == 0) {
							vala_struct_process_integer_type_attribute (self, a);
						} else {
							if (g_utf8_collate (vala_attribute_get_name (a), "FloatingType") == 0) {
								vala_struct_process_floating_type_attribute (self, a);
							}
						}
					}
				}
			}
		}
	}
}


static gboolean vala_struct_real_is_reference_counting (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	return FALSE;
}


static char* vala_struct_real_get_dup_function (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (self->priv->dup_function == NULL) {
		char* __temp47 = NULL;
		char* __temp46 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (__temp47 = g_strdup_printf ("The type `%s` doesn't contain a copy function", (__temp46 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self)))))));
		(__temp47 = (g_free (__temp47), NULL));
		(__temp46 = (g_free (__temp46), NULL));
	}
	const char* __temp48 = NULL;
	return (__temp48 = self->priv->dup_function, (__temp48 == NULL ? NULL : g_strdup (__temp48)));
}


void vala_struct_set_dup_function (ValaStruct* self, const char* name)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	char* __temp50 = NULL;
	self->priv->dup_function = (__temp50 = g_strdup (name), (self->priv->dup_function = (g_free (self->priv->dup_function), NULL)), __temp50);
}


static char* vala_struct_real_get_free_function (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (self->priv->free_function == NULL) {
		char* __temp52 = NULL;
		char* __temp51 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (__temp52 = g_strdup_printf ("The type `%s` doesn't contain a free function", (__temp51 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self)))))));
		(__temp52 = (g_free (__temp52), NULL));
		(__temp51 = (g_free (__temp51), NULL));
	}
	const char* __temp53 = NULL;
	return (__temp53 = self->priv->free_function, (__temp53 == NULL ? NULL : g_strdup (__temp53)));
}


static void vala_struct_set_free_function (ValaStruct* self, const char* name)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	char* __temp55 = NULL;
	self->priv->free_function = (__temp55 = g_strdup (name), (self->priv->free_function = (g_free (self->priv->free_function), NULL)), __temp55);
}


static char* vala_struct_real_get_type_id (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (self->priv->type_id == NULL) {
		if (vala_data_type_is_reference_type (VALA_DATA_TYPE (self))) {
			char* __temp56 = NULL;
			self->priv->type_id = (__temp56 = g_strdup ("G_TYPE_POINTER"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), __temp56);
		} else {
			char* __temp58 = NULL;
			char* __temp57 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (__temp58 = g_strdup_printf ("The type `%s` doesn't declare a type id", (__temp57 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self)))))));
			(__temp58 = (g_free (__temp58), NULL));
			(__temp57 = (g_free (__temp57), NULL));
		}
	}
	const char* __temp59 = NULL;
	return (__temp59 = self->priv->type_id, (__temp59 == NULL ? NULL : g_strdup (__temp59)));
}


void vala_struct_set_type_id (ValaStruct* self, const char* name)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	char* __temp61 = NULL;
	self->priv->type_id = (__temp61 = g_strdup (name), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), __temp61);
}


static char* vala_struct_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (self->priv->marshaller_type_name == NULL) {
		if (vala_data_type_is_reference_type (VALA_DATA_TYPE (self))) {
			char* __temp62 = NULL;
			self->priv->marshaller_type_name = (__temp62 = g_strdup ("POINTER"), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), __temp62);
		} else {
			char* __temp64 = NULL;
			char* __temp63 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (__temp64 = g_strdup_printf ("The type `%s` doesn't declare a marshaller type name", (__temp63 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self)))))));
			(__temp64 = (g_free (__temp64), NULL));
			(__temp63 = (g_free (__temp63), NULL));
		}
	}
	const char* __temp65 = NULL;
	return (__temp65 = self->priv->marshaller_type_name, (__temp65 == NULL ? NULL : g_strdup (__temp65)));
}


static void vala_struct_set_marshaller_type_name (ValaStruct* self, const char* name)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	char* __temp67 = NULL;
	self->priv->marshaller_type_name = (__temp67 = g_strdup (name), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), __temp67);
}


static char* vala_struct_real_get_get_value_function (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (self->priv->get_value_function == NULL) {
		if (vala_data_type_is_reference_type (VALA_DATA_TYPE (self))) {
			return g_strdup ("g_value_get_pointer");
		} else {
			char* __temp70 = NULL;
			char* __temp69 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (__temp70 = g_strdup_printf ("The value type `%s` doesn't declare a GValue get function", (__temp69 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self)))))));
			(__temp70 = (g_free (__temp70), NULL));
			(__temp69 = (g_free (__temp69), NULL));
		}
	} else {
		const char* __temp71 = NULL;
		return (__temp71 = self->priv->get_value_function, (__temp71 == NULL ? NULL : g_strdup (__temp71)));
	}
}


static char* vala_struct_real_get_set_value_function (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	if (self->priv->set_value_function == NULL) {
		if (vala_data_type_is_reference_type (VALA_DATA_TYPE (self))) {
			return g_strdup ("g_value_set_pointer");
		} else {
			char* __temp75 = NULL;
			char* __temp74 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (__temp75 = g_strdup_printf ("The value type `%s` doesn't declare a GValue set function", (__temp74 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self)))))));
			(__temp75 = (g_free (__temp75), NULL));
			(__temp74 = (g_free (__temp74), NULL));
		}
	} else {
		const char* __temp76 = NULL;
		return (__temp76 = self->priv->set_value_function, (__temp76 == NULL ? NULL : g_strdup (__temp76)));
	}
}


static void vala_struct_set_get_value_function (ValaStruct* self, const char* function)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (function != NULL);
	char* __temp78 = NULL;
	self->priv->get_value_function = (__temp78 = g_strdup (function), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), __temp78);
}


static void vala_struct_set_set_value_function (ValaStruct* self, const char* function)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (function != NULL);
	char* __temp79 = NULL;
	self->priv->set_value_function = (__temp79 = g_strdup (function), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), __temp79);
}


static char* vala_struct_real_get_default_value (ValaDataType* base)
{
	ValaStruct * self = VALA_STRUCT (base);
	const char* __temp80 = NULL;
	return (__temp80 = self->priv->default_value, (__temp80 == NULL ? NULL : g_strdup (__temp80)));
}


static void vala_struct_set_default_value (ValaStruct* self, const char* value)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (value != NULL);
	char* __temp82 = NULL;
	self->priv->default_value = (__temp82 = g_strdup (value), (self->priv->default_value = (g_free (self->priv->default_value), NULL)), __temp82);
}


/**
 * Adds the specified struct to the list of base types of this struct.
 *
 * @param type a class or interface reference
 */
void vala_struct_add_base_type (ValaStruct* self, ValaTypeReference* type)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	self->priv->base_types = g_list_append (self->priv->base_types, g_object_ref (type));
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GList* vala_struct_get_base_types (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return g_list_copy (self->priv->base_types);
}


static gint vala_struct_real_get_type_parameter_index (ValaDataType* base, const char* name)
{
	ValaStruct * self = VALA_STRUCT (base);
	g_return_val_if_fail (name != NULL, 0);
	gint i = 0;
	{
		GList* __temp85 = NULL;
		__temp85 = self->priv->type_parameters;
		GList* p_it;
		for (p_it = __temp85; p_it != NULL; p_it = p_it->next) {
			ValaTypeParameter* p = p_it->data;
			{
				if (g_utf8_collate (vala_type_parameter_get_name (p), name) == 0) {
					return (i);
				}
				i++;
			}
		}
	}
	return -1;
}


ValaMethod* vala_struct_get_default_construction_method (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return self->priv->_default_construction_method;
}


void vala_struct_set_default_construction_method (ValaStruct* self, ValaMethod* value)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	ValaMethod* __temp89 = NULL;
	ValaMethod* __temp88 = NULL;
	self->priv->_default_construction_method = (__temp89 = (__temp88 = value, (__temp88 == NULL ? NULL : g_object_ref (__temp88))), (self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL))), __temp89);
}


static void vala_struct_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaStruct * self = VALA_STRUCT (object);
	switch (property_id) {
		case VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD:
		g_value_set_object (value, vala_struct_get_default_construction_method (self));
		break;
	}
}


static void vala_struct_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaStruct * self = VALA_STRUCT (object);
	switch (property_id) {
		case VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD:
		vala_struct_set_default_construction_method (self, g_value_get_object (value));
		break;
	}
}


static void vala_struct_class_init (ValaStructClass * klass)
{
	vala_struct_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaStructPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_struct_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_struct_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_struct_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_struct_real_accept;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_struct_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cprefix = vala_struct_real_get_lower_case_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_struct_real_get_lower_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_struct_real_get_upper_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_struct_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_counting = vala_struct_real_is_reference_counting;
	VALA_DATA_TYPE_CLASS (klass)->get_dup_function = vala_struct_real_get_dup_function;
	VALA_DATA_TYPE_CLASS (klass)->get_free_function = vala_struct_real_get_free_function;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_struct_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_struct_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_struct_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_struct_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_default_value = vala_struct_real_get_default_value;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_struct_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD, g_param_spec_object ("default-construction-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_struct_init (ValaStruct * self)
{
	self->priv = VALA_STRUCT_GET_PRIVATE (self);
	self->priv->default_value = NULL;
}


static void vala_struct_dispose (GObject * obj)
{
	ValaStruct * self = VALA_STRUCT (obj);
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_list_foreach (self->priv->type_parameters, (GFunc) g_object_unref, NULL), g_list_free (self->priv->type_parameters), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_list_foreach (self->priv->constants, (GFunc) g_object_unref, NULL), g_list_free (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_list_foreach (self->priv->fields, (GFunc) g_object_unref, NULL), g_list_free (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_list_foreach (self->priv->methods, (GFunc) g_object_unref, NULL), g_list_free (self->priv->methods), NULL)));
	(self->priv->base_types == NULL ? NULL : (self->priv->base_types = (g_list_foreach (self->priv->base_types, (GFunc) g_object_unref, NULL), g_list_free (self->priv->base_types), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->const_cname = (g_free (self->priv->const_cname), NULL));
	(self->priv->dup_function = (g_free (self->priv->dup_function), NULL));
	(self->priv->free_function = (g_free (self->priv->free_function), NULL));
	(self->priv->type_id = (g_free (self->priv->type_id), NULL));
	(self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	(self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL));
	(self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL));
	(self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL));
	(self->priv->default_value = (g_free (self->priv->default_value), NULL));
	(self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL)));
	ValaStructClass * klass;
	GObjectClass * parent_class;
	klass = VALA_STRUCT_CLASS (g_type_class_peek (VALA_TYPE_STRUCT));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_struct_get_type ()
{
	static GType vala_struct_type_id = 0;
	if (G_UNLIKELY (vala_struct_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaStructClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_struct_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaStruct), 0, (GInstanceInitFunc) vala_struct_init };
		vala_struct_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaStruct", &g_define_type_info, 0);
	}
	return vala_struct_type_id;
}




