/* valasourcefile.vala
 *
 * Copyright (C) 2006  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valasourcefile.h"
#include <vala/valasourcefilecycle.h>
#include <vala/valacodecontext.h>
#include <vala/valanamespace.h>
#include <vala/valanamespacereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasymbol.h>
#include <vala/valadatatype.h>
#include <vala/valamethod.h>
#include <vala/valafield.h>
#include <vala/valaproperty.h>
#include <vala/valaconstant.h>
#include <vala/valaformalparameter.h>
#include <vala/valatypereference.h>
#include <vala/valasourcereference.h>

struct _ValaSourceFilePrivate {
	char* _filename;
	char* _comment;
	gboolean _pkg;
	ValaSourceFileCycle* _cycle;
	gboolean _is_cycle_head;
	gint _mark;
	ValaCodeContext* _context;
	GList* using_directives;
	ValaNamespace* global_namespace;
	GList* namespaces;
	char* cheader_filename;
	char* csource_filename;
	char* cinclude_filename;
	GList* header_external_includes;
	GList* header_internal_includes;
	GList* source_external_includes;
	GList* source_internal_includes;
	GList* header_internal_full_dependencies;
	GList* header_internal_dependencies;
};
#define VALA_SOURCE_FILE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SOURCE_FILE, ValaSourceFilePrivate))
enum  {
	VALA_SOURCE_FILE_DUMMY_PROPERTY,
	VALA_SOURCE_FILE_FILENAME,
	VALA_SOURCE_FILE_COMMENT,
	VALA_SOURCE_FILE_PKG,
	VALA_SOURCE_FILE_CYCLE,
	VALA_SOURCE_FILE_IS_CYCLE_HEAD,
	VALA_SOURCE_FILE_MARK,
	VALA_SOURCE_FILE_CONTEXT,
};
static GObject * vala_source_file_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_source_file_parent_class = NULL;
static void vala_source_file_dispose (GObject * obj);


/**
 * Creates a new source file.
 *
 * @param filename source file name
 * @param pkg      true if this is a VAPI package file
 * @return         newly created source file
 */
ValaSourceFile* vala_source_file_new (ValaCodeContext* _context, const char* _filename, gboolean _pkg)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (_context), NULL);
	g_return_val_if_fail (_filename != NULL, NULL);
	GParameter * __params = g_new0 (GParameter, 3);
	GParameter * __params_it = __params;
	(__params_it->name = "context", g_value_init (&__params_it->value, VALA_TYPE_CODE_CONTEXT), g_value_set_object (&__params_it->value, _context), __params_it++);
	(__params_it->name = "filename", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _filename), __params_it++);
	(__params_it->name = "pkg", g_value_init (&__params_it->value, G_TYPE_BOOLEAN), g_value_set_boolean (&__params_it->value, _pkg), __params_it++);
	ValaSourceFile * self = g_object_newv (VALA_TYPE_SOURCE_FILE, __params_it - __params, __params);
	return self;
}


/**
 * Adds a new using directive with the specified namespace.
 *
 * @param ns reference to namespace
 */
void vala_source_file_add_using_directive (ValaSourceFile* self, ValaNamespaceReference* ns)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_NAMESPACE_REFERENCE (ns));
	self->priv->using_directives = g_list_append (self->priv->using_directives, g_object_ref (ns));
}


/**
 * Returns a copy of the list of using directives.
 *
 * @return using directive list
 */
GList* vala_source_file_get_using_directives (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return g_list_copy (self->priv->using_directives);
}


/**
 * Adds the specified namespace to this source file.
 *
 * @param ns a namespace
 */
void vala_source_file_add_namespace (ValaSourceFile* self, ValaNamespace* ns)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	self->priv->namespaces = g_list_append (self->priv->namespaces, g_object_ref (ns));
}


/**
 * Returns the implicitly declared root namespace of this source file.
 *
 * @return root namespace
 */
ValaNamespace* vala_source_file_get_global_namespace (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	ValaNamespace* __temp1 = NULL;
	return (__temp1 = self->priv->global_namespace, (__temp1 == NULL ? NULL : g_object_ref (__temp1)));
}


/**
 * Returns a copy of the list of namespaces.
 *
 * @return namespace list
 */
GList* vala_source_file_get_namespaces (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return g_list_copy (self->priv->namespaces);
}


/**
 * Visits this source file and all children with the specified
 * CodeVisitor.
 *
 * @param visitor the visitor to be called while traversing
 */
void vala_source_file_accept (ValaSourceFile* self, ValaCodeVisitor* visitor)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_begin_source_file (visitor, self);
	{
		GList* __temp4 = NULL;
		__temp4 = self->priv->using_directives;
		GList* ns_ref_it;
		for (ns_ref_it = __temp4; ns_ref_it != NULL; ns_ref_it = ns_ref_it->next) {
			ValaNamespaceReference* ns_ref = ns_ref_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (ns_ref), visitor);
			}
		}
	}
	vala_code_node_accept (VALA_CODE_NODE (self->priv->global_namespace), visitor);
	{
		GList* __temp5 = NULL;
		__temp5 = self->priv->namespaces;
		GList* ns_it;
		for (ns_it = __temp5; ns_it != NULL; ns_it = ns_it->next) {
			ValaNamespace* ns = ns_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (ns), visitor);
			}
		}
	}
	vala_code_visitor_visit_end_source_file (visitor, self);
}


/**
 * Returns the filename to use when generating C header files.
 *
 * @return generated C header filename
 */
char* vala_source_file_get_cheader_filename (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->cheader_filename == NULL) {
		char* basename = g_strndup (vala_source_file_get_filename (self), ((guint) (g_utf8_strlen (vala_source_file_get_filename (self), -1) - g_utf8_strlen (".vala", -1))));
		char* __temp6 = NULL;
		self->priv->cheader_filename = (__temp6 = g_strdup_printf ("%s.h", basename), (self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL)), __temp6);
		(basename = (g_free (basename), NULL));
	}
	const char* __temp7 = NULL;
	return (__temp7 = self->priv->cheader_filename, (__temp7 == NULL ? NULL : g_strdup (__temp7)));
}


/**
 * Returns the filename to use when generating C source files.
 *
 * @return generated C source filename
 */
char* vala_source_file_get_csource_filename (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->csource_filename == NULL) {
		char* basename = g_strndup (vala_source_file_get_filename (self), ((guint) (g_utf8_strlen (vala_source_file_get_filename (self), -1) - g_utf8_strlen (".vala", -1))));
		char* __temp9 = NULL;
		self->priv->csource_filename = (__temp9 = g_strdup_printf ("%s.c", basename), (self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL)), __temp9);
		(basename = (g_free (basename), NULL));
	}
	const char* __temp10 = NULL;
	return (__temp10 = self->priv->csource_filename, (__temp10 == NULL ? NULL : g_strdup (__temp10)));
}


/**
 * Returns the filename to use when including the generated C header
 * file.
 *
 * @return C header filename to include
 */
char* vala_source_file_get_cinclude_filename (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->cinclude_filename == NULL) {
		char* basename = g_strndup (vala_source_file_get_filename (self), ((guint) (g_utf8_strlen (vala_source_file_get_filename (self), -1) - g_utf8_strlen (".vala", -1))));
		if (vala_code_context_get_library (vala_source_file_get_context (self)) != NULL) {
			char* __temp12 = NULL;
			self->priv->cinclude_filename = (__temp12 = g_strdup_printf ("%s/%s.h", vala_code_context_get_library (vala_source_file_get_context (self)), basename), (self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL)), __temp12);
		} else {
			char* __temp13 = NULL;
			self->priv->cinclude_filename = (__temp13 = g_strdup_printf ("%s.h", basename), (self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL)), __temp13);
		}
		(basename = (g_free (basename), NULL));
	}
	const char* __temp14 = NULL;
	return (__temp14 = self->priv->cinclude_filename, (__temp14 == NULL ? NULL : g_strdup (__temp14)));
}


/**
 * Adds the specified symbol to the list of symbols code in this source
 * file depends on.
 *
 * @param sym      a symbol
 * @param dep_type type of dependency
 */
void vala_source_file_add_symbol_dependency (ValaSourceFile* self, ValaSymbol* sym, ValaSourceFileDependencyType dep_type)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_SYMBOL (sym));
	ValaDataType* t = NULL;
	if (VALA_IS_DATA_TYPE (vala_symbol_get_node (sym))) {
		ValaDataType* __temp17 = NULL;
		ValaDataType* __temp16 = NULL;
		t = (__temp17 = (__temp16 = VALA_DATA_TYPE (vala_symbol_get_node (sym)), (__temp16 == NULL ? NULL : g_object_ref (__temp16))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), __temp17);
	} else {
		if (VALA_IS_METHOD (vala_symbol_get_node (sym)) || VALA_IS_FIELD (vala_symbol_get_node (sym))) {
			if (VALA_IS_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (sym)))) {
				ValaDataType* __temp19 = NULL;
				ValaDataType* __temp18 = NULL;
				t = (__temp19 = (__temp18 = VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (sym))), (__temp18 == NULL ? NULL : g_object_ref (__temp18))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), __temp19);
			} else {
				(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
				return;
			}
		} else {
			if (VALA_IS_PROPERTY (vala_symbol_get_node (sym))) {
				ValaDataType* __temp21 = NULL;
				ValaDataType* __temp20 = NULL;
				t = (__temp21 = (__temp20 = VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (sym))), (__temp20 == NULL ? NULL : g_object_ref (__temp20))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), __temp21);
			} else {
				if (VALA_IS_CONSTANT (vala_symbol_get_node (sym))) {
					if (VALA_IS_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (sym)))) {
						ValaDataType* __temp23 = NULL;
						ValaDataType* __temp22 = NULL;
						t = (__temp23 = (__temp22 = VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (sym))), (__temp22 == NULL ? NULL : g_object_ref (__temp22))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), __temp23);
					} else {
						if (VALA_IS_NAMESPACE (vala_symbol_get_node (vala_symbol_get_parent_symbol (sym)))) {
							ValaNamespace* __temp24 = NULL;
							ValaNamespace* ns = (__temp24 = VALA_NAMESPACE (vala_symbol_get_node (vala_symbol_get_parent_symbol (sym))), (__temp24 == NULL ? NULL : g_object_ref (__temp24)));
							self->priv->source_internal_includes = g_list_concat (self->priv->source_internal_includes, vala_namespace_get_cheader_filenames (ns));
							(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
							(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
							return;
							(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
						} else {
							(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
							return;
						}
					}
				} else {
					if (VALA_IS_FORMAL_PARAMETER (vala_symbol_get_node (sym))) {
						ValaFormalParameter* __temp25 = NULL;
						ValaFormalParameter* fp = (__temp25 = VALA_FORMAL_PARAMETER (vala_symbol_get_node (sym)), (__temp25 == NULL ? NULL : g_object_ref (__temp25)));
						ValaDataType* __temp27 = NULL;
						ValaDataType* __temp26 = NULL;
						t = (__temp27 = (__temp26 = vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (fp)), (__temp26 == NULL ? NULL : g_object_ref (__temp26))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), __temp27);
						if (t == NULL) {
							(fp == NULL ? NULL : (fp = (g_object_unref (fp), NULL)));
							(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
							return;
						}
						(fp == NULL ? NULL : (fp = (g_object_unref (fp), NULL)));
					} else {
						(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
						return;
					}
				}
			}
		}
	}
	/* generic type parameter */
	if (dep_type == VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE) {
		if (vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (t))))) {
			self->priv->source_external_includes = g_list_concat (self->priv->source_external_includes, vala_data_type_get_cheader_filenames (t));
		} else {
			self->priv->source_internal_includes = g_list_concat (self->priv->source_internal_includes, vala_data_type_get_cheader_filenames (t));
		}
		(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
		return;
	}
	if (vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (t))))) {
		/* external package */
		self->priv->header_external_includes = g_list_concat (self->priv->header_external_includes, vala_data_type_get_cheader_filenames (t));
		(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
		return;
	}
	if (dep_type == VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL || !vala_data_type_is_reference_type (t)) {
		self->priv->header_internal_includes = g_list_concat (self->priv->header_internal_includes, vala_data_type_get_cheader_filenames (t));
		self->priv->header_internal_full_dependencies = g_list_append (self->priv->header_internal_full_dependencies, vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (t))));
	}
	self->priv->header_internal_dependencies = g_list_append (self->priv->header_internal_dependencies, vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (t))));
	(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
}


/**
 * Returns the list of external includes the generated C header file
 * requires.
 *
 * @return external include list for C header file
 */
GList* vala_source_file_get_header_external_includes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->header_external_includes;
}


/**
 * Adds the specified filename to the list of package-internal includes
 * the generated C header file requires.
 *
 * @param include internal include for C header file
 */
void vala_source_file_add_header_internal_include (ValaSourceFile* self, const char* include)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (include != NULL);
	self->priv->header_internal_includes = g_list_append (self->priv->header_internal_includes, g_strdup (include));
}


/**
 * Returns the list of package-internal includes the generated C header
 * file requires.
 *
 * @return internal include list for C header file
 */
GList* vala_source_file_get_header_internal_includes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->header_internal_includes;
}


/**
 * Returns the list of external includes the generated C source file
 * requires.
 *
 * @return include list for C source file
 */
GList* vala_source_file_get_source_external_includes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->source_external_includes;
}


/**
 * Returns the list of package-internal includes the generated C source
 * file requires.
 *
 * @return include list for C source file
 */
GList* vala_source_file_get_source_internal_includes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->source_internal_includes;
}


/**
 * Returns the list of source files the generated C header file requires
 * definitely.
 *
 * @return definite source file dependencies
 */
GList* vala_source_file_get_header_internal_full_dependencies (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->header_internal_full_dependencies;
}


/**
 * Returns the list of source files the generated C header file loosely
 * depends on.
 *
 * @return loose source file dependencies
 */
GList* vala_source_file_get_header_internal_dependencies (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->header_internal_dependencies;
}


char* vala_source_file_get_filename (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_filename;
}


void vala_source_file_set_filename (ValaSourceFile* self, const char* value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	char* __temp35 = NULL;
	self->priv->_filename = (__temp35 = g_strdup (value), (self->priv->_filename = (g_free (self->priv->_filename), NULL)), __temp35);
}


char* vala_source_file_get_comment (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_comment;
}


void vala_source_file_set_comment (ValaSourceFile* self, const char* value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	char* __temp38 = NULL;
	const char* __temp37 = NULL;
	self->priv->_comment = (__temp38 = (__temp37 = value, (__temp37 == NULL ? NULL : g_strdup (__temp37))), (self->priv->_comment = (g_free (self->priv->_comment), NULL)), __temp38);
}


gboolean vala_source_file_get_pkg (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), FALSE);
	return self->priv->_pkg;
}


void vala_source_file_set_pkg (ValaSourceFile* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_pkg = value;
}


ValaSourceFileCycle* vala_source_file_get_cycle (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_cycle;
}


void vala_source_file_set_cycle (ValaSourceFile* self, ValaSourceFileCycle* value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	ValaSourceFileCycle* __temp42 = NULL;
	ValaSourceFileCycle* __temp41 = NULL;
	self->priv->_cycle = (__temp42 = (__temp41 = value, (__temp41 == NULL ? NULL : g_object_ref (__temp41))), (self->priv->_cycle == NULL ? NULL : (self->priv->_cycle = (g_object_unref (self->priv->_cycle), NULL))), __temp42);
}


gboolean vala_source_file_get_is_cycle_head (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), FALSE);
	return self->priv->_is_cycle_head;
}


void vala_source_file_set_is_cycle_head (ValaSourceFile* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_is_cycle_head = value;
}


gint vala_source_file_get_mark (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), 0);
	return self->priv->_mark;
}


void vala_source_file_set_mark (ValaSourceFile* self, gint value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_mark = value;
}


ValaCodeContext* vala_source_file_get_context (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_context;
}


void vala_source_file_set_context (ValaSourceFile* self, ValaCodeContext* value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_context = value;
}


static GObject * vala_source_file_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties)
{
	GObject * obj;
	ValaSourceFileClass * klass;
	GObjectClass * parent_class;
	klass = VALA_SOURCE_FILE_CLASS (g_type_class_peek (VALA_TYPE_SOURCE_FILE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	ValaSourceFile * self = VALA_SOURCE_FILE (obj);
	{
		ValaNamespace* __temp46 = NULL;
		self->priv->global_namespace = (__temp46 = vala_namespace_new (NULL, vala_source_reference_new (self, 0, 0, 0, 0)), (self->priv->global_namespace == NULL ? NULL : (self->priv->global_namespace = (g_object_unref (self->priv->global_namespace), NULL))), __temp46);
	}
	return obj;
}


static void vala_source_file_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSourceFile * self = VALA_SOURCE_FILE (object);
	switch (property_id) {
		case VALA_SOURCE_FILE_FILENAME:
		g_value_set_string (value, vala_source_file_get_filename (self));
		break;
		case VALA_SOURCE_FILE_COMMENT:
		g_value_set_string (value, vala_source_file_get_comment (self));
		break;
		case VALA_SOURCE_FILE_PKG:
		g_value_set_boolean (value, vala_source_file_get_pkg (self));
		break;
		case VALA_SOURCE_FILE_CYCLE:
		g_value_set_object (value, vala_source_file_get_cycle (self));
		break;
		case VALA_SOURCE_FILE_IS_CYCLE_HEAD:
		g_value_set_boolean (value, vala_source_file_get_is_cycle_head (self));
		break;
		case VALA_SOURCE_FILE_MARK:
		g_value_set_int (value, vala_source_file_get_mark (self));
		break;
		case VALA_SOURCE_FILE_CONTEXT:
		g_value_set_object (value, vala_source_file_get_context (self));
		break;
	}
}


static void vala_source_file_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSourceFile * self = VALA_SOURCE_FILE (object);
	switch (property_id) {
		case VALA_SOURCE_FILE_FILENAME:
		vala_source_file_set_filename (self, g_value_get_string (value));
		break;
		case VALA_SOURCE_FILE_COMMENT:
		vala_source_file_set_comment (self, g_value_get_string (value));
		break;
		case VALA_SOURCE_FILE_PKG:
		vala_source_file_set_pkg (self, g_value_get_boolean (value));
		break;
		case VALA_SOURCE_FILE_CYCLE:
		vala_source_file_set_cycle (self, g_value_get_object (value));
		break;
		case VALA_SOURCE_FILE_IS_CYCLE_HEAD:
		vala_source_file_set_is_cycle_head (self, g_value_get_boolean (value));
		break;
		case VALA_SOURCE_FILE_MARK:
		vala_source_file_set_mark (self, g_value_get_int (value));
		break;
		case VALA_SOURCE_FILE_CONTEXT:
		vala_source_file_set_context (self, g_value_get_object (value));
		break;
	}
}


static void vala_source_file_class_init (ValaSourceFileClass * klass)
{
	vala_source_file_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSourceFilePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_source_file_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_source_file_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_source_file_constructor;
	G_OBJECT_CLASS (klass)->dispose = vala_source_file_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_FILENAME, g_param_spec_string ("filename", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_COMMENT, g_param_spec_string ("comment", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_PKG, g_param_spec_boolean ("pkg", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_CYCLE, g_param_spec_object ("cycle", "foo", "bar", VALA_TYPE_SOURCE_FILE_CYCLE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_IS_CYCLE_HEAD, g_param_spec_boolean ("is-cycle-head", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_MARK, g_param_spec_int ("mark", "foo", "bar", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_CONTEXT, g_param_spec_object ("context", "foo", "bar", VALA_TYPE_CODE_CONTEXT, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_source_file_init (ValaSourceFile * self)
{
	self->priv = VALA_SOURCE_FILE_GET_PRIVATE (self);
	self->priv->cheader_filename = NULL;
	self->priv->csource_filename = NULL;
	self->priv->cinclude_filename = NULL;
}


static void vala_source_file_dispose (GObject * obj)
{
	ValaSourceFile * self = VALA_SOURCE_FILE (obj);
	(self->priv->_filename = (g_free (self->priv->_filename), NULL));
	(self->priv->_comment = (g_free (self->priv->_comment), NULL));
	(self->priv->_cycle == NULL ? NULL : (self->priv->_cycle = (g_object_unref (self->priv->_cycle), NULL)));
	(self->priv->using_directives == NULL ? NULL : (self->priv->using_directives = (g_list_foreach (self->priv->using_directives, (GFunc) g_object_unref, NULL), g_list_free (self->priv->using_directives), NULL)));
	(self->priv->global_namespace == NULL ? NULL : (self->priv->global_namespace = (g_object_unref (self->priv->global_namespace), NULL)));
	(self->priv->namespaces == NULL ? NULL : (self->priv->namespaces = (g_list_foreach (self->priv->namespaces, (GFunc) g_object_unref, NULL), g_list_free (self->priv->namespaces), NULL)));
	(self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL));
	(self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL));
	(self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL));
	(self->priv->header_external_includes == NULL ? NULL : (self->priv->header_external_includes = (g_list_foreach (self->priv->header_external_includes, (GFunc) g_free, NULL), g_list_free (self->priv->header_external_includes), NULL)));
	(self->priv->header_internal_includes == NULL ? NULL : (self->priv->header_internal_includes = (g_list_foreach (self->priv->header_internal_includes, (GFunc) g_free, NULL), g_list_free (self->priv->header_internal_includes), NULL)));
	(self->priv->source_external_includes == NULL ? NULL : (self->priv->source_external_includes = (g_list_foreach (self->priv->source_external_includes, (GFunc) g_free, NULL), g_list_free (self->priv->source_external_includes), NULL)));
	(self->priv->source_internal_includes == NULL ? NULL : (self->priv->source_internal_includes = (g_list_foreach (self->priv->source_internal_includes, (GFunc) g_free, NULL), g_list_free (self->priv->source_internal_includes), NULL)));
	(self->priv->header_internal_full_dependencies == NULL ? NULL : (self->priv->header_internal_full_dependencies = (g_list_free (self->priv->header_internal_full_dependencies), NULL)));
	(self->priv->header_internal_dependencies == NULL ? NULL : (self->priv->header_internal_dependencies = (g_list_free (self->priv->header_internal_dependencies), NULL)));
	ValaSourceFileClass * klass;
	GObjectClass * parent_class;
	klass = VALA_SOURCE_FILE_CLASS (g_type_class_peek (VALA_TYPE_SOURCE_FILE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_source_file_get_type ()
{
	static GType vala_source_file_type_id = 0;
	if (G_UNLIKELY (vala_source_file_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSourceFileClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_source_file_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSourceFile), 0, (GInstanceInitFunc) vala_source_file_init };
		vala_source_file_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaSourceFile", &g_define_type_info, 0);
	}
	return vala_source_file_type_id;
}




