/* valacodecontext.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valacodecontext.h"
#include <vala/valamethod.h>
#include <vala/valasymbol.h>
#include <vala/valacodecontext.h>
#include <vala/valasourcefile.h>
#include <vala/valacodevisitor.h>
#include <vala/valasourcefilecycle.h>

struct _ValaCodeContextPrivate {
	char* _library;
	ValaMethod* _module_init_method;
	GList* source_files;
	ValaSymbol* root;
	GList* cycles;
};
#define VALA_CODE_CONTEXT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CODE_CONTEXT, ValaCodeContextPrivate))
enum  {
	VALA_CODE_CONTEXT_DUMMY_PROPERTY,
	VALA_CODE_CONTEXT_LIBRARY,
	VALA_CODE_CONTEXT_MODULE_INIT_METHOD,
};
static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file);
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GList* chain);
static gpointer vala_code_context_parent_class = NULL;
static void vala_code_context_dispose (GObject * obj);


/**
 * Returns the root symbol of the code tree.
 *
 * @return root symbol
 */
ValaSymbol* vala_code_context_get_root (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	ValaSymbol* __temp0 = NULL;
	return (__temp0 = self->priv->root, (__temp0 == NULL ? NULL : g_object_ref (__temp0)));
}


/**
 * Returns a copy of the list of source files.
 *
 * @return list of source files
 */
GList* vala_code_context_get_source_files (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return g_list_copy (self->priv->source_files);
}


/**
 * Adds the specified file to the list of source files.
 *
 * @param file a source file
 */
void vala_code_context_add_source_file (ValaCodeContext* self, ValaSourceFile* file)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	self->priv->source_files = g_list_append (self->priv->source_files, g_object_ref (file));
}


/**
 * Visits the complete code tree file by file.
 *
 * @param visitor the visitor to be called when traversing
 */
void vala_code_context_accept (ValaCodeContext* self, ValaCodeVisitor* visitor)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GList* __temp3 = NULL;
		__temp3 = self->priv->source_files;
		GList* file_it;
		for (file_it = __temp3; file_it != NULL; file_it = file_it->next) {
			ValaSourceFile* file = file_it->data;
			{
				vala_source_file_accept (file, visitor);
			}
		}
	}
}


/**
 * Find and resolve cycles in source file dependencies.
 */
void vala_code_context_find_header_cycles (ValaCodeContext* self)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	{
		GList* __temp4 = NULL;
		__temp4 = self->priv->source_files;
		GList* file_it;
		for (file_it = __temp4; file_it != NULL; file_it = file_it->next) {
			ValaSourceFile* file = file_it->data;
			{
				/* find cycles in dependencies between source files 
				 we're only interested in internal source files */
				if (vala_source_file_get_pkg (file)) {
					continue;
				}
				if (vala_source_file_get_mark (file) == 0) {
					vala_code_context_visit (self, file, NULL);
				}
			}
		}
	}
	{
		GList* __temp5 = NULL;
		__temp5 = self->priv->cycles;
		GList* cycle_it;
		for (cycle_it = __temp5; cycle_it != NULL; cycle_it = cycle_it->next) {
			ValaSourceFileCycle* cycle = cycle_it->data;
			{
				/* find one head for each cycle, it must not have any
				 * hard dependencies on other files in the cycle
				 */
				cycle->head = vala_code_context_find_cycle_head (self, VALA_SOURCE_FILE (cycle->files->data));
				(vala_source_file_set_is_cycle_head (cycle->head, TRUE), vala_source_file_get_is_cycle_head (cycle->head));
			}
		}
	}
	{
		GList* __temp9 = NULL;
		__temp9 = self->priv->source_files;
		GList* file2_it;
		for (file2_it = __temp9; file2_it != NULL; file2_it = file2_it->next) {
			ValaSourceFile* file2 = file2_it->data;
			{
				/* connect the source files in a non-cyclic way
				 * cycle members connect to the head of their cycle
				 
				 we're only interested in internal source files */
				if (vala_source_file_get_pkg (file2)) {
					continue;
				}
				{
					GList* __temp8 = NULL;
					__temp8 = vala_source_file_get_header_internal_dependencies (file2);
					GList* dep_it;
					for (dep_it = __temp8; dep_it != NULL; dep_it = dep_it->next) {
						ValaSourceFile* dep = dep_it->data;
						{
							if (vala_source_file_get_cycle (file2) != NULL && vala_source_file_get_cycle (dep) == vala_source_file_get_cycle (file2)) {
								/* in the same cycle */
								if (!vala_source_file_get_is_cycle_head (file2)) {
									/* include header of cycle head */
									char* __temp6 = NULL;
									vala_source_file_add_header_internal_include (file2, (__temp6 = vala_source_file_get_cinclude_filename (vala_source_file_get_cycle (file2)->head)));
									(__temp6 = (g_free (__temp6), NULL));
								}
							} else {
								/* we can just include the headers if they are not in a cycle or not in the same cycle as the current file */
								char* __temp7 = NULL;
								vala_source_file_add_header_internal_include (file2, (__temp7 = vala_source_file_get_cinclude_filename (dep)));
								(__temp7 = (g_free (__temp7), NULL));
							}
						}
					}
				}
			}
		}
	}
}


static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (file), NULL);
	{
		GList* __temp12 = NULL;
		__temp12 = vala_source_file_get_header_internal_full_dependencies (file);
		GList* dep_it;
		for (dep_it = __temp12; dep_it != NULL; dep_it = dep_it->next) {
			ValaSourceFile* dep = dep_it->data;
			{
				if (dep == file) {
					continue;
				}
				{
					GList* __temp11 = NULL;
					__temp11 = vala_source_file_get_cycle (file)->files;
					GList* cycle_file_it;
					for (cycle_file_it = __temp11; cycle_file_it != NULL; cycle_file_it = cycle_file_it->next) {
						ValaSourceFile* cycle_file = cycle_file_it->data;
						{
							/* ignore file-internal dependencies */
							if (dep == cycle_file) {
								return vala_code_context_find_cycle_head (self, dep);
							}
						}
					}
				}
			}
		}
	}
	return file;
}


/* no hard dependencies on members of the same cycle found
 * source file suitable as cycle head
 */
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GList* chain)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	/* no deep copy available yet
	 * var l = chain.copy ();
	 */
	GList* l = NULL;
	{
		GList* __temp14 = NULL;
		__temp14 = chain;
		GList* chain_file_it;
		for (chain_file_it = __temp14; chain_file_it != NULL; chain_file_it = chain_file_it->next) {
			ValaSourceFile* chain_file = chain_file_it->data;
			{
				l = g_list_append (l, chain_file);
			}
		}
	}
	l = g_list_append (l, file);
	/* end workaround 
	 mark file as currently being visited */
	(vala_source_file_set_mark (file, 1), vala_source_file_get_mark (file));
	{
		GList* __temp24 = NULL;
		__temp24 = vala_source_file_get_header_internal_dependencies (file);
		GList* dep_it;
		for (dep_it = __temp24; dep_it != NULL; dep_it = dep_it->next) {
			ValaSourceFile* dep = dep_it->data;
			{
				if (file == dep) {
					continue;
				}
				if (vala_source_file_get_mark (dep) == 1) {
					/* found cycle */
					ValaSourceFileCycle* cycle = g_object_new (VALA_TYPE_SOURCE_FILE_CYCLE, NULL);
					ValaSourceFileCycle* __temp15 = NULL;
					self->priv->cycles = g_list_append (self->priv->cycles, (__temp15 = cycle, (__temp15 == NULL ? NULL : g_object_ref (__temp15))));
					gboolean cycle_start_found = FALSE;
					{
						GList* __temp23 = NULL;
						__temp23 = l;
						GList* cycle_file_it;
						for (cycle_file_it = __temp23; cycle_file_it != NULL; cycle_file_it = cycle_file_it->next) {
							ValaSourceFile* cycle_file = cycle_file_it->data;
							{
								ValaSourceFileCycle* __temp16 = NULL;
								ValaSourceFileCycle* ref_cycle_file_cycle = (__temp16 = vala_source_file_get_cycle (cycle_file), (__temp16 == NULL ? NULL : g_object_ref (__temp16)));
								if (!cycle_start_found) {
									if (cycle_file == dep) {
										cycle_start_found = TRUE;
									}
								}
								if (!cycle_start_found) {
									continue;
								}
								if (vala_source_file_get_cycle (cycle_file) != NULL) {
									/* file already in a cycle */
									if (vala_source_file_get_cycle (cycle_file) == cycle) {
										continue;
									}
									/* file is in the same cycle, nothing to do 
									 file is in an other cycle, merge the two cycles 
									 broken memory management cycles.remove (cycle_file.cycle); */
									GList* newlist = NULL;
									{
										GList* __temp17 = NULL;
										__temp17 = self->priv->cycles;
										GList* oldcycle_it;
										for (oldcycle_it = __temp17; oldcycle_it != NULL; oldcycle_it = oldcycle_it->next) {
											ValaSourceFileCycle* oldcycle = oldcycle_it->data;
											{
												if (oldcycle != vala_source_file_get_cycle (cycle_file)) {
													newlist = g_list_append (newlist, oldcycle);
												}
											}
										}
									}
									GList* __temp18 = NULL;
									self->priv->cycles = (__temp18 = NULL, (self->priv->cycles == NULL ? NULL : (self->priv->cycles = (g_list_foreach (self->priv->cycles, (GFunc) g_object_unref, NULL), g_list_free (self->priv->cycles), NULL))), __temp18);
									{
										GList* __temp20 = NULL;
										__temp20 = newlist;
										GList* newcycle_it;
										for (newcycle_it = __temp20; newcycle_it != NULL; newcycle_it = newcycle_it->next) {
											ValaSourceFileCycle* newcycle = newcycle_it->data;
											{
												ValaSourceFileCycle* __temp19 = NULL;
												self->priv->cycles = g_list_append (self->priv->cycles, (__temp19 = newcycle, (__temp19 == NULL ? NULL : g_object_ref (__temp19))));
											}
										}
									}
									GList* __temp21 = NULL;
									newlist = (__temp21 = NULL, (newlist == NULL ? NULL : (newlist = (g_list_free (newlist), NULL))), __temp21);
									{
										GList* __temp22 = NULL;
										__temp22 = vala_source_file_get_cycle (cycle_file)->files;
										GList* inner_cycle_file_it;
										for (inner_cycle_file_it = __temp22; inner_cycle_file_it != NULL; inner_cycle_file_it = inner_cycle_file_it->next) {
											ValaSourceFile* inner_cycle_file = inner_cycle_file_it->data;
											{
												/* end workaround for broken memory management */
												if (vala_source_file_get_cycle (inner_cycle_file) != cycle) {
													/* file in inner cycle not yet added to outer cycle */
													cycle->files = g_list_append (cycle->files, inner_cycle_file);
													(vala_source_file_set_cycle (inner_cycle_file, cycle), vala_source_file_get_cycle (inner_cycle_file));
												}
											}
										}
									}
									(newlist == NULL ? NULL : (newlist = (g_list_free (newlist), NULL)));
								} else {
									cycle->files = g_list_append (cycle->files, cycle_file);
									(vala_source_file_set_cycle (cycle_file, cycle), vala_source_file_get_cycle (cycle_file));
								}
								(ref_cycle_file_cycle == NULL ? NULL : (ref_cycle_file_cycle = (g_object_unref (ref_cycle_file_cycle), NULL)));
							}
						}
					}
					(cycle == NULL ? NULL : (cycle = (g_object_unref (cycle), NULL)));
				} else {
					if (vala_source_file_get_mark (dep) == 0) {
						/* found not yet visited file */
						vala_code_context_visit (self, dep, l);
					}
				}
			}
		}
	}
	/* mark file as successfully visited */
	(vala_source_file_set_mark (file, 2), vala_source_file_get_mark (file));
	(l == NULL ? NULL : (l = (g_list_free (l), NULL)));
}


char* vala_code_context_get_library (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_library;
}


void vala_code_context_set_library (ValaCodeContext* self, const char* value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	char* __temp27 = NULL;
	const char* __temp26 = NULL;
	self->priv->_library = (__temp27 = (__temp26 = value, (__temp26 == NULL ? NULL : g_strdup (__temp26))), (self->priv->_library = (g_free (self->priv->_library), NULL)), __temp27);
}


ValaMethod* vala_code_context_get_module_init_method (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_module_init_method;
}


void vala_code_context_set_module_init_method (ValaCodeContext* self, ValaMethod* value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	ValaMethod* __temp30 = NULL;
	ValaMethod* __temp29 = NULL;
	self->priv->_module_init_method = (__temp30 = (__temp29 = value, (__temp29 == NULL ? NULL : g_object_ref (__temp29))), (self->priv->_module_init_method == NULL ? NULL : (self->priv->_module_init_method = (g_object_unref (self->priv->_module_init_method), NULL))), __temp30);
}


static void vala_code_context_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaCodeContext * self = VALA_CODE_CONTEXT (object);
	switch (property_id) {
		case VALA_CODE_CONTEXT_LIBRARY:
		g_value_set_string (value, vala_code_context_get_library (self));
		break;
		case VALA_CODE_CONTEXT_MODULE_INIT_METHOD:
		g_value_set_object (value, vala_code_context_get_module_init_method (self));
		break;
	}
}


static void vala_code_context_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaCodeContext * self = VALA_CODE_CONTEXT (object);
	switch (property_id) {
		case VALA_CODE_CONTEXT_LIBRARY:
		vala_code_context_set_library (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_MODULE_INIT_METHOD:
		vala_code_context_set_module_init_method (self, g_value_get_object (value));
		break;
	}
}


static void vala_code_context_class_init (ValaCodeContextClass * klass)
{
	vala_code_context_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCodeContextPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_code_context_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_code_context_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_code_context_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_LIBRARY, g_param_spec_string ("library", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_MODULE_INIT_METHOD, g_param_spec_object ("module-init-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_code_context_init (ValaCodeContext * self)
{
	self->priv = VALA_CODE_CONTEXT_GET_PRIVATE (self);
	self->priv->root = vala_symbol_new (NULL);
}


static void vala_code_context_dispose (GObject * obj)
{
	ValaCodeContext * self = VALA_CODE_CONTEXT (obj);
	(self->priv->_library = (g_free (self->priv->_library), NULL));
	(self->priv->_module_init_method == NULL ? NULL : (self->priv->_module_init_method = (g_object_unref (self->priv->_module_init_method), NULL)));
	(self->priv->source_files == NULL ? NULL : (self->priv->source_files = (g_list_foreach (self->priv->source_files, (GFunc) g_object_unref, NULL), g_list_free (self->priv->source_files), NULL)));
	(self->priv->root == NULL ? NULL : (self->priv->root = (g_object_unref (self->priv->root), NULL)));
	(self->priv->cycles == NULL ? NULL : (self->priv->cycles = (g_list_foreach (self->priv->cycles, (GFunc) g_object_unref, NULL), g_list_free (self->priv->cycles), NULL)));
	ValaCodeContextClass * klass;
	GObjectClass * parent_class;
	klass = VALA_CODE_CONTEXT_CLASS (g_type_class_peek (VALA_TYPE_CODE_CONTEXT));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_code_context_get_type ()
{
	static GType vala_code_context_type_id = 0;
	if (G_UNLIKELY (vala_code_context_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeContextClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_context_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeContext), 0, (GInstanceInitFunc) vala_code_context_init };
		vala_code_context_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCodeContext", &g_define_type_info, 0);
	}
	return vala_code_context_type_id;
}




