/*****************************************************************************/
/* td_moi_tree.c : Objet Gtk+
 * td_mod_tree.c : Gtk+ object
 *
 *
 * ToutDoux : Chtit gestionnaire de projet - A littl' project manager
 * Copyright (c) 2000-2001 Philippe Roy
 * Auteur - Author : Philippe Roy <ph_roy@toutdoux.org>
 *
 *
 * Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier
 * sous les termes de la licence publique gnrale GNU telle qu'elle est publie par
 * la Free Software Foundation ; soit la version 2 de la licence, ou
 * (comme vous voulez) toute version ultrieure.
 *
 * Ce programme est distribu dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; mme sans la garantie de
 * COMMERCIALIT ou d'ADQUATION A UN BUT PARTICULIER. Voir la
 * licence publique gnrale GNU pour plus de dtails.
 *
 * Vous devriez avoir reu une copie de la licence publique gnrale GNU
 * avec ce programme ; si ce n'est pas le cas, crivez  la Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/*****************************************************************************/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "commons.h"
#include "gtdk.h"
#include "database.h"
#include "icons.h"
#include "td_app.h"
#include "td_app_palette.h"
#include "td_app_bench.h"
#include "td_mod.h"
#include "td_mod_tree.h"
#include "td_mod_tree_icon.h"
#include "td_mod_bench.h"
#include "td_db_base.h"
#include "td_db_mod.h"
#include "td_db_datatable.h"
#include "td_db_connect.h"
#include "td_field.h"
#include "td_flower.h"

/*****************************************************************************/
/*** Arguments */
/*****************************************************************************/
static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_NAME,
  ARG_NAME_INTL,
  ARG_CUSTOMIZE,
  ARG_REORDERABLE,
  ARG_COLUMN_OID,
  ARG_TABLE,
  ARG_TABLE_TREE,
};

static void td_mod_tree_set_name (TdModTree *mod_tree, gchar *name)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  if (mod_tree->name != name)
    mod_tree->name = name;
}

static void td_mod_tree_set_name_intl (TdModTree *mod_tree, gchar *name_intl)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  if (mod_tree->name_intl != name_intl)
    {
      mod_tree->name_intl = name_intl;
      mod_tree->label = gtk_label_new (name_intl);
    }
}

static void td_mod_tree_set_customize (TdModTree *mod_tree, gboolean customize)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  if (mod_tree->customize != customize)
    mod_tree->customize = customize;
}

static void td_mod_tree_set_reorderable (TdModTree *mod_tree, gboolean reorderable)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  if (mod_tree->reorderable != reorderable)
    mod_tree->reorderable = reorderable;
}

static void td_mod_tree_set_column_oid (TdModTree *mod_tree, int column_oid)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  if (mod_tree->column_oid != column_oid)
    mod_tree->column_oid = column_oid;
}

static void td_mod_tree_set_table (TdModTree *mod_tree, gchar *table)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  if (mod_tree->table != table)
    mod_tree->table = table;
}

static void td_mod_tree_set_table_tree (TdModTree *mod_tree, gchar *table_tree)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  if (mod_tree->table_tree != table_tree)
    mod_tree->table_tree = table_tree;
}

static void td_mod_tree_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
  TdModTree *mod_tree;
  mod_tree = TD_MOD_TREE (object);
  switch (arg_id)
    {
    case ARG_NAME:
      td_mod_tree_set_name (mod_tree, GTK_VALUE_STRING (*arg));
      break;
    case ARG_NAME_INTL:
      td_mod_tree_set_name_intl (mod_tree, GTK_VALUE_STRING (*arg));
      break;
    case ARG_CUSTOMIZE:
      td_mod_tree_set_customize (mod_tree, GTK_VALUE_BOOL (*arg));
      break;
    case ARG_REORDERABLE:
      td_mod_tree_set_reorderable (mod_tree, GTK_VALUE_BOOL (*arg));
      break;
    case ARG_COLUMN_OID:
      td_mod_tree_set_column_oid (mod_tree, GTK_VALUE_INT (*arg));
      break;
    case ARG_TABLE:
      td_mod_tree_set_table (mod_tree, GTK_VALUE_STRING (*arg));
      break;
    case ARG_TABLE_TREE:
      td_mod_tree_set_table_tree (mod_tree, GTK_VALUE_STRING (*arg));
      break;
    default:
      break;
    }
}

static void td_mod_tree_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
  TdModTree *mod_tree;
  mod_tree = TD_MOD_TREE (object);
   switch (arg_id)
    {
    case ARG_NAME:
      GTK_VALUE_STRING (*arg) = mod_tree->name;
      break;
    case ARG_NAME_INTL:
      GTK_VALUE_STRING (*arg) = mod_tree->name_intl;
      break;
    case ARG_CUSTOMIZE:
      GTK_VALUE_BOOL (*arg) = mod_tree->customize;
      break;
    case ARG_REORDERABLE:
      GTK_VALUE_BOOL (*arg) = mod_tree->reorderable;
      break;
    case ARG_COLUMN_OID:
      GTK_VALUE_INT (*arg) = mod_tree->column_oid;
      break;
    case ARG_TABLE:
      GTK_VALUE_STRING (*arg) = mod_tree->table;
      break;
    case ARG_TABLE_TREE:
      GTK_VALUE_STRING (*arg) = mod_tree->table_tree;
      break;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

/*****************************************************************************/
/*** Initialisation */
/*****************************************************************************/
static void td_mod_tree_init (TdModTree *mod_tree)
{
  mod_tree->name = NULL;
  mod_tree->name_intl = NULL;
  mod_tree->id = -1;
  mod_tree->label = NULL;
  mod_tree->customize = FALSE;
  mod_tree->reorderable = FALSE;
  mod_tree->field = NULL;
  mod_tree->icon = NULL;
  mod_tree->column_oid = 0;
  mod_tree->table = NULL;
  mod_tree->table_tree = NULL;

  mod_tree->widget = NULL;
  mod_tree->widget_data = NULL;
  mod_tree->selected = NULL;
  mod_tree->menu_widget = NULL;

  mod_tree->list_widget = NULL;
  mod_tree->list_query = NULL;
  mod_tree->list_type = NULL;
}

static void td_mod_tree_class_init (TdModTreeClass *klass)
{
  GtkObjectClass *object_class;
  object_class = (GtkObjectClass*) klass;
  parent_class = gtk_type_class (gtk_object_get_type());
  gtk_object_add_arg_type ("TdModTree::name", GTK_TYPE_STRING, GTK_ARG_READWRITE, ARG_NAME);
  gtk_object_add_arg_type ("TdModTree::name_intl", GTK_TYPE_STRING, GTK_ARG_READWRITE, ARG_NAME_INTL);
  gtk_object_add_arg_type ("TdModTree::customize", GTK_TYPE_INT, GTK_ARG_READWRITE, ARG_CUSTOMIZE);
  gtk_object_add_arg_type ("TdModTree::reorderable", GTK_TYPE_INT, GTK_ARG_READWRITE, ARG_REORDERABLE);
  gtk_object_add_arg_type ("TdModTree::column_oid", GTK_TYPE_INT, GTK_ARG_READWRITE, ARG_COLUMN_OID);
  gtk_object_add_arg_type ("TdModTree::table", GTK_TYPE_STRING, GTK_ARG_READWRITE, ARG_TABLE);
  gtk_object_add_arg_type ("TdModTree::table_tree", GTK_TYPE_STRING, GTK_ARG_READWRITE, ARG_TABLE_TREE);
  object_class->set_arg = td_mod_tree_set_arg;
  object_class->get_arg = td_mod_tree_get_arg;
  klass->add_icon = td_mod_tree_add_icon;
}

GtkType td_mod_tree_get_type (void)
{
  static GtkType mod_tree_type = 0;
  if (!mod_tree_type)
    {
      static const GtkTypeInfo mod_tree_info =
      {
  	"TdModTree", sizeof (TdModTree), sizeof (TdModTreeClass),
  	(GtkClassInitFunc) td_mod_tree_class_init,
	(GtkObjectInitFunc) td_mod_tree_init,
	NULL, NULL, (GtkClassInitFunc) NULL,
      };
      mod_tree_type = gtk_type_unique (GTK_TYPE_OBJECT, &mod_tree_info);
    }
  return mod_tree_type;
}

/**
 * td_mod_tree_new:
 * 
 * fr: Cr un nouveau arbre
 *
 * en: Creates a new tree
 * 
 * Return value: tree
 **/

GtkObject *td_mod_tree_new (void)
{
  return GTK_OBJECT (gtk_type_new (td_mod_tree_get_type()));
}

/**
 * td_mod_tree_destroy:
 * @mod_tree: tree
 * 
 * fr: Dtruit l'arbre
 *
 * en: Destroys the tree
 **/

void td_mod_tree_destroy (TdModTree *mod_tree)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  gtk_object_destroy (GTK_OBJECT (mod_tree));
}

/**
 * td_mod_tree_show:
 * @mod_tree: tree
 * 
 * fr: Affiche l'arbre
 *
 * en: Displays the tree
 **/

void td_mod_tree_show (TdModTree *mod_tree)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  td_mod_tree_refresh (mod_tree);
}

/**
 * td_mod_tree_hide:
 * @mod_tree: tree
 * 
 * fr: Cache l'arbre
 *
 * en: Hides the tree
 **/

void td_mod_tree_hide (TdModTree *mod_tree)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
}

/*****************************************************************************/
/*** Affectations - Allocations */
/*****************************************************************************/

/**
 * td_mod_tree_add_icon:
 * @mod_tree: tree
 * @icon: icon
 * 
 * fr: Ajoute l'icne  l'arbre
 *
 * en: Adds the icon to tree
 **/

void td_mod_tree_add_icon (TdModTree *mod_tree, GtkObject *icon)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  mod_tree->icon = g_list_append (mod_tree->icon, icon);
}

/*****************************************************************************/
/*** Commandes - Commands */
/*****************************************************************************/

/**
 * td_mod_tree_create:
 * @mod_tree: tree
 * @mod: module
 * 
 * fr: Cr l'interface de l'arbre du module
 *
 * en: Creates the interface of tree of module
 **/

void td_mod_tree_create (TdModTree *mod_tree, TdMod *mod)
{
  GList *field;
  GList *icon_list;
  GList *icon_list2;
  int i, j;
  gchar *txt_tmp;
  gchar *txt_tmp2;
  gboolean without_param = FALSE;
  GtkObject *icon;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** Champs - Fields */
  field = td_mod_interface_field (mod);
  if (g_list_length (field) == 0)
    {
      without_param = TRUE;
      field = NULL;
      field = td_database_field (g_strdup_printf ("SELECT * FROM %s;", mod_tree->table));
      for (i=0; i<g_list_length (field); i++)
	if (!strcmp ((gchar*) g_list_nth_data (field, i), "td_id"))
	  {
	    gtk_object_set (GTK_OBJECT (mod_tree), "column_oid", i, NULL);
	    break;
	  }
    }
  else
    for (i=0; i<g_list_length (field); i++)
      if (!strcmp (TD_FIELD (g_list_nth_data (field, i))->name, "td_id"))
	{
	  gtk_object_set (GTK_OBJECT (mod_tree), "column_oid", i, NULL);
	  break;
	}

  /*** Bote - Box */
  mod_tree->id = mod->id;
  mod_tree->menu_widget = mod->menu_widget;
  mod_tree->widget = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (mod_tree->widget);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (mod_tree->widget), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  /*** Arbre - Tree */
  mod_tree->widget_data = gtk_ctree_new (g_list_length (field), 0);
  gtdk_clist_adjustheight (GTK_CLIST (mod_tree->widget_data));
  gtk_widget_show (mod_tree->widget_data);
  gtk_container_add (GTK_CONTAINER (mod_tree->widget), mod_tree->widget_data);
  gtk_widget_set_usize (mod_tree->widget_data, 0, -2);
  gtk_clist_column_titles_show (GTK_CLIST (mod_tree->widget_data));
  for (i=0; i<g_list_length (field); i++)
    {

      /*** Taille collone - Column size */
      txt_tmp = NULL;
      if (!without_param)
	txt_tmp = td_field_property_value (g_list_nth_data (field, i), "column size");
      if ((!txt_tmp) && (mod_tree->customize))
	txt_tmp = td_mod_custom_value (mod, mod_tree->name, "column size", g_list_nth_data (field, i));
      if (txt_tmp)
	{
	  if ((!strcmp ("automatic", txt_tmp)) || (!strcmp (_("automatic"), txt_tmp)))
	    gtk_clist_set_column_auto_resize (GTK_CLIST (mod_tree->widget_data), i, TRUE);
	  else
	    gtk_clist_set_column_width (GTK_CLIST (mod_tree->widget_data), i, atoi (txt_tmp));
	}
      else 
	gtk_clist_set_column_auto_resize (GTK_CLIST (mod_tree->widget_data), i, TRUE);

      /*** Titres - Titles */
      if (without_param)
	gtk_clist_set_column_title (GTK_CLIST (mod_tree->widget_data), i, (gchar*) (g_list_nth_data (field, i)));
      else
	gtk_clist_set_column_title (GTK_CLIST (mod_tree->widget_data), i, TD_FIELD (g_list_nth_data (field, i))->name);

      /*** Icnes - Icons */
      if (!without_param)
	{
	  icon_list = td_field_property_value_list (g_list_nth_data (field, i), "icon name");
	  if ((g_list_length (icon_list)) != 0)
	    {
	      icon_list2 = td_field_property_value_list (g_list_nth_data (field, i), "icon visible query");
	      if ((g_list_length (icon_list)) == (g_list_length (icon_list2)))
		for (j=0; j<g_list_length (icon_list); j++)
		  {
		    icon = td_mod_tree_icon_new();
		    td_mod_tree_icon_set (TD_MOD_TREE_ICON (icon), g_list_nth_data (icon_list, j));
		    gtk_object_set (GTK_OBJECT (icon), "query_visible", g_list_nth_data (icon_list2, j), NULL);
		    td_mod_tree_add_icon (mod_tree, icon);
		  }
	    }
	}

      /*** Liste - List */
      if (!without_param)
	{
	  mod_tree->list_widget = g_list_append (mod_tree->list_widget, gtk_menu_new());
	  txt_tmp = td_field_property_value (g_list_nth_data (field, i), "column list item");
	  if (txt_tmp)
	    {
	      mod_tree->list_query = g_list_append (mod_tree->list_query, txt_tmp);
	      mod_tree->list_type = g_list_append (mod_tree->list_type, "column list item");
	    }
	  else
	    {
	      txt_tmp = td_field_property_value (g_list_nth_data (field, i), "column list static query");
	      if (txt_tmp)
		{
		  mod_tree->list_query = g_list_append (mod_tree->list_query, txt_tmp);
		  mod_tree->list_type = g_list_append (mod_tree->list_type, "column list static query");
		}
	      else
		{
		  txt_tmp = td_field_property_value (g_list_nth_data (field, i), "column list query");
		  if (txt_tmp)
		    {
		      mod_tree->list_query = g_list_append (mod_tree->list_query, txt_tmp);
		      mod_tree->list_type = g_list_append (mod_tree->list_type, "column list query");
		    }
		  else
		    {
		      mod_tree->list_query = g_list_append (mod_tree->list_query, "(null)");
		      mod_tree->list_type = g_list_append (mod_tree->list_type, "(null)");
		    }
		}
	    }
	}
    }
  gtk_signal_connect (GTK_OBJECT (mod_tree->widget_data), "button_press_event", GTK_SIGNAL_FUNC (td_mod_tree_button_press), mod_tree);
  gtk_signal_connect (GTK_OBJECT (mod_tree->widget_data), "select_row", GTK_SIGNAL_FUNC (td_mod_tree_select_node), mod_tree);
  gtk_signal_connect (GTK_OBJECT (mod_tree->widget_data), "tree_expand", GTK_SIGNAL_FUNC (td_mod_tree_expand), mod_tree);
  gtk_signal_connect (GTK_OBJECT (mod_tree->widget_data), "tree_collapse", GTK_SIGNAL_FUNC (td_mod_tree_collapse), mod_tree);
  if (mod_tree->reorderable)
    {
      gtk_drag_source_set (GTK_WIDGET (mod_tree->widget_data), GDK_BUTTON1_MASK, TD_DRAG_TYPES, TD_N_DRAG_TYPES, GDK_ACTION_MOVE | GDK_ACTION_COPY);
      gtk_drag_dest_set (GTK_WIDGET (mod_tree->widget_data), GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP, TD_DROP_TYPES, TD_N_DROP_TYPES, GDK_ACTION_MOVE | GDK_ACTION_COPY);
      gtk_signal_connect (GTK_OBJECT (mod_tree->widget_data), "drag_data_received", GTK_SIGNAL_FUNC (td_mod_tree_dnd_received), mod_tree);
      gtk_signal_connect (GTK_OBJECT (mod_tree->widget_data), "drag_motion", GTK_SIGNAL_FUNC (td_mod_tree_dnd_motion), mod_tree->widget);
    }
}

/**
 * td_mod_tree_button_press:
 * @widget: ctree widget
 * @event: event button
 * @mod_tree: tree
 * 
 * fr: Evnement 'button_press' de l'arbre
 *
 * en: Event 'button_press' of tree
 **/

void td_mod_tree_button_press (GtkWidget *widget, GdkEventButton *event, TdModTree *mod_tree)
{
  GdkEventButton *bevent;
  int row, column, i;
  gchar *txt_tmp;
  TdMod *mod = NULL;
  TdModBench *bench = NULL;
  GtkWidget *menu;
  GList *menuitem = NULL;
  GList *list_tmp = NULL;
  GList *data_tmp = NULL;
  gchar *txt_tmp2;
  bevent = (GdkEventButton*) event;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  mod = TD_MOD (g_list_nth_data (TD_APP(APP)->mod_data, mod_tree->id));

  /*** Slection - Selection */
  gtk_clist_get_selection_info (GTK_CLIST (widget), bevent->x, bevent->y, &row, &column);
  row = gtdk_clist_correction (GTK_CLIST (widget), row, bevent->y);

  /*** Menu champs - Fields menu */
  if ((row >= 0) && (mod_tree->selected == gtk_ctree_node_nth (GTK_CTREE (widget), row)) && (bevent->button == 1) && 
      mod_tree->list_widget && ((int) bevent->x > GTK_CTREE_ROW (mod_tree->selected)->level * GTK_CTREE (widget)->tree_indent))
    {
      if ((strcmp ("(null)", (gchar*) g_list_nth_data (mod_tree->list_query, column))) &&
	  (strcmp ("(null)", (gchar*) g_list_nth_data (mod_tree->list_type, column))))
	{
	  txt_tmp = g_strdup_printf ("%s", (gchar*) g_list_nth_data (mod_tree->list_query, column));
	  if (!strcmp ("column list item", (gchar*) g_list_nth_data (mod_tree->list_type, column)))
	    {
	      txt_tmp2 = strtok (txt_tmp, "',");
	      while (txt_tmp2)
		{
		  list_tmp = g_list_append (list_tmp, g_strstrip (txt_tmp2));
		  txt_tmp2 = strtok (NULL, "',");
		}
	    }
	  else
	    {
	      if (!strcmp ("column list static query", (gchar*) g_list_nth_data (mod_tree->list_type, column)))
		list_tmp = td_database_column (txt_tmp);
	      else
		if (!strcmp ("column list query", (gchar*) g_list_nth_data (mod_tree->list_type, column)))
		  list_tmp = td_database_column (txt_tmp);
	    }

	  /*** Menu */
	  if ((g_list_length (gtk_container_children (GTK_CONTAINER (g_list_nth_data (mod_tree->list_widget, column))))==0) ||
	      (!strcmp ("column list query", (gchar*) g_list_nth_data (mod_tree->list_type, column))))
    	    for (i=0; i< g_list_length (list_tmp); i++)
	      {
		if (!strcmp ("(separator)", (gchar*) g_list_nth_data (list_tmp, i)))
		  menuitem = g_list_append (menuitem, gtk_menu_item_new());
		else
		  menuitem = g_list_append (menuitem, gtk_menu_item_new_with_label ((gchar*) g_list_nth_data (list_tmp, i)));
		data_tmp = g_list_append (data_tmp, mod_tree);
		data_tmp = g_list_append (data_tmp, g_list_nth_data (list_tmp, i));
		data_tmp = g_list_append (data_tmp, GINT_TO_POINTER (column));
		gtk_widget_show (GTK_WIDGET (g_list_nth_data (menuitem, i)));
		gtk_signal_connect (GTK_OBJECT (g_list_nth_data (menuitem, i)), "activate", GTK_SIGNAL_FUNC (td_mod_tree_interface_action), data_tmp);
		gtk_menu_append (GTK_MENU (g_list_nth_data (mod_tree->list_widget, column)), GTK_WIDGET (g_list_nth_data (menuitem, i)));
		data_tmp = NULL;
	      }
	  gtk_menu_popup (GTK_MENU (g_list_nth_data (mod_tree->list_widget, column)), NULL, NULL, NULL, NULL, bevent->button, -1);
	}
    }

  /*** Slection - Selection */
  td_mod_tree_select_node (GTK_CTREE (widget), row, column, NULL, mod_tree);

  /*** Tirer-lcher - Drag'n'drop */
  if ((mod_tree->reorderable) && (mod_tree->id!=-1) && (bevent->button == 1) && (row >= 0))
    gtk_signal_connect (GTK_OBJECT (widget), "drag_data_get", GTK_SIGNAL_FUNC (td_mod_tree_dnd_get), (int*) row);

  /*** Menu item */
  if (bevent->button == 3)
    gtk_menu_popup (GTK_MENU (mod_tree->menu_widget), NULL, NULL, NULL, NULL, bevent->button, bevent->time);

  /*** Menu principal - Main menu */
  if (bevent->button == 2)
    gnome_popup_menu_do_popup (TD_APP (APP)->menu_main, NULL, NULL, bevent, NULL);
}

/**
 * td_mod_tree_interface_action:
 * @menuitem: tree
 * @data: value
 * 
 * fr: Ecriture de la valeur sur la node courante
 *
 * en: Write the value to the current node
 **/

void td_mod_tree_interface_action (GtkMenuItem *menuitem, GList *data)
{
  TdModTree *mod_tree = g_list_nth_data (data, 0);
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));
  TD_FLAG_NO_REFRESH_NODE = TRUE;
  td_database_command (g_strdup_printf ("UPDATE item SET %s = %s WHERE td_id = %d;", TD_FIELD (g_list_nth_data (TD_DB_MOD (g_list_nth_data (TD_DB_BASE (TD_APP (APP)->base)->mod, mod_tree->id))->interface_field, GPOINTER_TO_INT (g_list_nth_data (data, 2))))->table_field, td_database_adaptvalue ((gchar*) g_list_nth_data (data, 1), "text"), CURRENT_ID));
  td_mod_tree_update_node (TD_MOD_TREE (mod_tree));
  td_mod_refresh_bench (TD_MOD (g_list_nth_data (TD_APP(APP)->mod_data, mod_tree->id)), GTK_OBJECT (mod_tree));
  td_mod_modified (TD_MOD (g_list_nth_data (TD_APP(APP)->mod_data, mod_tree->id))); /* FIXME: sauf pour la mme valuer : +tards */
  TD_FLAG_NO_REFRESH_NODE = FALSE;
}

/**
 * td_mod_tree_expand:
 * @widget: ctree widget
 * @node: node selected
 * @mod_tree: tree
 * 
 * fr: Evnement 'tree_expand' de l'arbre
 *
 * en: Event 'tree_expand' of tree
 **/

void td_mod_tree_expand (GtkWidget *widget, GtkCTreeNode *node, TdModTree *mod_tree)
{
  int i, j;
  GtkObject *datatable;
  GtkObject *datatable_tree;
  GList *order = NULL;
  gboolean bool_tmp;
  GtkCTreeNode *node_tmp;
  gchar *txt_tmp;
  GdkPixmap *pixmap1 = NULL;
  GdkPixmap *pixmap2 = NULL;
  GdkBitmap *mask1 = NULL;
  GdkBitmap *mask2 = NULL;
  gboolean leaf;
  gboolean expanded;
  guint8 spacing;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** Dj ouverte - Already opened */
  gtk_ctree_get_node_info (GTK_CTREE (widget), node, &txt_tmp, &spacing, &pixmap1, &pixmap2, &mask1, &mask2, &leaf, &expanded);
  if (expanded)
    return;

  /*** Go ! */
 if (GTK_CTREE_ROW (node)->children)
    node = (GTK_CTREE_ROW (node)->children);
  else
    node = NULL;
  while (node)
    {
      txt_tmp = gtk_ctree_node_get_row_data (GTK_CTREE (widget), node);
      datatable = td_database_select (g_strdup_printf ("SELECT * FROM %s WHERE td_id IN (SELECT td_id FROM %s WHERE td_id_parent = %s);", mod_tree->table, mod_tree->table_tree, txt_tmp));
      datatable_tree = td_database_select (g_strdup_printf ("SELECT * FROM %s WHERE td_id_parent = %s;", mod_tree->table_tree, txt_tmp));

      /*** Classement - Sorting */
      txt_tmp = "0";
      order = NULL;
      bool_tmp = TRUE;
      while (bool_tmp)
	{
	  bool_tmp = FALSE;
	  for (i=0; i<g_list_length (TD_DB_DATATABLE (datatable_tree)->item); i++)
	    if (!strcmp ((gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 2), txt_tmp))
	      {
		order = g_list_append (order, g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 0));
		txt_tmp =  g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 0);
		bool_tmp = TRUE;
		break;
	      }
	}

      /*** Node */
      for (i=g_list_length (order)-1; i!=-1; i--)
	for (j=0; j<g_list_length (TD_DB_DATATABLE (datatable)->item); j++)
	  if (!strcmp ((gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable)->item, j), mod_tree->column_oid), (gchar*) g_list_nth_data (order, i)))
	    {
	      node_tmp = td_mod_tree_add_node (mod_tree, node, NULL, g_list_nth_data (TD_DB_DATATABLE (datatable)->item, j));
	      break;
	    }

      /*** Suivant - Next */
      if (GTK_CTREE_ROW (node)->sibling)
	node = GTK_CTREE_ROW (node)->sibling;
      else
	node = NULL;
    }
}

/**
 * td_mod_tree_collapse:
 * @widget: ctree widget
 * @node: node selected
 * @mod_tree: tree
 * 
 * fr: Evnement 'tree_collapse' de l'arbre
 *
 * en: Event 'tree_collapse' of tree
 **/

void td_mod_tree_collapse (GtkWidget *widget, GtkCTreeNode *node, TdModTree *mod_tree)
{
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** Go ! */
  if (GTK_CTREE_ROW (node)->children)
    node = (GTK_CTREE_ROW (node)->children);
  else
    node = NULL;

  /*** Node */
  while (node)
    {
      while (GTK_CTREE_ROW (node)->children)
	gtk_ctree_remove_node (GTK_CTREE (mod_tree->widget_data), GTK_CTREE_ROW (node)->children);

      /*** Suivant - Next */
      if (GTK_CTREE_ROW (node)->sibling)
	node = GTK_CTREE_ROW (node)->sibling;
      else
	node = NULL;
    }
}

/**
 * td_mod_tree_refresh:
 * @mod_tree: tree
 * 
 * fr: Rafrachis les donnes de l'arbre
 *
 * en: Refresh the data of tree
 **/

void td_mod_tree_refresh (TdModTree *mod_tree)
{
  void (*symbol)();
  int i, j, k;
  GtkObject *datatable;
  GtkObject *datatable_tree;
  GList *order = NULL;
  gboolean bool_tmp;
  GtkObject *datatable_child = NULL;
  GtkCTreeNode *node;
  GtkCTreeNode *node_tmp;
  int id_parent;
  gchar *txt_tmp;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** Niveau 0 - Level 0 */
  gtdk_ctree_clear (GTK_CTREE (mod_tree->widget_data));
  datatable = td_database_select (g_strdup_printf ("SELECT * FROM %s WHERE td_id IN (SELECT td_id FROM %s WHERE td_id_parent = 0);", mod_tree->table, mod_tree->table_tree));
  datatable_tree = td_database_select (g_strdup_printf ("SELECT * FROM %s WHERE td_id_parent = 0;", mod_tree->table_tree));

  /*** Classement - Sorting */
  txt_tmp = "0";
  bool_tmp = TRUE;
  while (bool_tmp)
    {
      bool_tmp = FALSE;
      for (i=0; i<g_list_length (TD_DB_DATATABLE (datatable_tree)->item); i++)
	if (!strcmp ((gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 2), txt_tmp))
	  {
  	    order = g_list_append (order, g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 0));
	    txt_tmp =  g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 0);
	    bool_tmp = TRUE;
	    break;
	  }
    }

  /*** Node */
  for (i=g_list_length (order)-1; i!=-1; i--)
    for (j=0; j<g_list_length (TD_DB_DATATABLE (datatable)->item); j++)
      if (!strcmp ((gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable)->item, j), mod_tree->column_oid), (gchar*) g_list_nth_data (order, i)))
	{
	  node_tmp = td_mod_tree_add_node (mod_tree, NULL, NULL, g_list_nth_data (TD_DB_DATATABLE (datatable)->item, j));
	  break;
	}

  /*** Niveau 1 - Level 1 */
  for (i=0; i<g_list_length (TD_DB_DATATABLE (datatable)->item); i++)
    {
      id_parent = atoi (g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable)->item, i), mod_tree->column_oid));
      datatable_child = td_database_select (g_strdup_printf ("SELECT * FROM %s WHERE td_id IN (SELECT td_id FROM %s WHERE td_id_parent = %s);", mod_tree->table, mod_tree->table_tree, (gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable)->item, i), mod_tree->column_oid)));
      datatable_tree = td_database_select (g_strdup_printf ("SELECT * FROM %s WHERE td_id_parent = %s;", mod_tree->table_tree, (gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable)->item, i), mod_tree->column_oid)));
      if (datatable_child)
	{
	  node = gtk_ctree_node_nth (GTK_CTREE (mod_tree->widget_data), 0);
	  while (node)
	    {
	      if (atoi (gtk_ctree_node_get_row_data (GTK_CTREE (mod_tree->widget_data), node)) == id_parent)
		{

		  /*** Classement - Sorting */
		  txt_tmp = "0";
		  order = NULL;
		  bool_tmp = TRUE;
		  while (bool_tmp)
		    {
		      bool_tmp = FALSE;
		      for (j=0; j<g_list_length (TD_DB_DATATABLE (datatable_tree)->item); j++)
			if (!strcmp ((gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, j), 2), txt_tmp))
			  {
			    order = g_list_append (order, g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, j), 0));
			    txt_tmp =  g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, j), 0);
			    bool_tmp = TRUE;
			    break;
			  }
		    }

		  /*** Node */
		  for (j=g_list_length (order)-1; j!=-1; j--)
		    for (k=0; k<g_list_length (TD_DB_DATATABLE (datatable_child)->item); k++)
		      if (!strcmp ((gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_child)->item, k), mod_tree->column_oid), (gchar*) g_list_nth_data (order, j)))
			{
			  node_tmp = td_mod_tree_add_node (mod_tree, node, NULL, g_list_nth_data (TD_DB_DATATABLE (datatable_child)->item, k));
			  break;
			}
		}

	      /*** Suivant - Next */
	      if (GTK_CTREE_ROW (node)->sibling)
		node = GTK_CTREE_ROW (node)->sibling;
	      else
		node = NULL;
	    }
	}
    }
}

/**
 * td_mod_tree_add_node:
 * @mod_tree: tree
 * @parent: parent node
 * @sibling: sibling node
 * @data: data list
 * 
 * fr: Cration d'une nouvelle node
 *
 * en: Creates a new node
 * 
 * Return value: node
 **/

GtkCTreeNode *td_mod_tree_add_node (TdModTree *mod_tree, GtkCTreeNode *parent, GtkCTreeNode *sibling, GList *data)
{
  GtkCTreeNode *ret;
  gchar *txt[50];
  gchar *txt_tmp;
  int i, j;
  GdkPixmap *pixmap1 = NULL;
  GdkPixmap *pixmap2 = NULL;
  GdkBitmap *mask1 = NULL;
  GdkBitmap *mask2 = NULL;
  GdkPixmap *pixmap1_tmp = NULL;
  GdkPixmap *pixmap2_tmp = NULL;
  GdkBitmap *mask1_tmp = NULL;
  GdkBitmap *mask2_tmp = NULL;
  GdkColor transparent;

  /*** Valeurs - Values */
  for (i=0; i<g_list_length (data); i++)
    txt[i] = (gchar*) g_list_nth_data (data, i);
  CURRENT_ID = atoi (g_list_nth_data (data, mod_tree->column_oid));
  td_database_set_current (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));

  /*** Icones - Icons */
  for (i=0; i<g_list_length (mod_tree->icon); i++)
    if (td_database_value (TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->query_visible))
      {
	pixmap1 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask1, &transparent, (gchar**) TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->pixmap1);
	pixmap2 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask2, &transparent, (gchar**) TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->pixmap2);
	break;
      }
  if (!pixmap1)
    {
      pixmap1 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask1, &transparent, (gchar**) TD_STOCK_LIST_EMPTY);
      pixmap2 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask2, &transparent, (gchar**) TD_STOCK_LIST_EMPTY);
    }

  /*** Node */
  ret = gtk_ctree_insert_node (GTK_CTREE (mod_tree->widget_data), parent, sibling, txt, 5, pixmap1, mask1, pixmap2, mask2, FALSE, FALSE);
  gtk_ctree_node_set_row_data (GTK_CTREE (mod_tree->widget_data), ret, (gchar*) g_list_nth_data (data, mod_tree->column_oid));
  return ret;
}

/**
 * td_mod_tree_update_node:
 * @mod_tree: tree
 * 
 * fr: Mise  jour de la node courante
 *
 * en: Update the current node
 **/

void td_mod_tree_update_node (TdModTree *mod_tree)
{
  GtkCTreeNode *node;
  GList *row;
  int i;
  gchar *txt_tmp;
  gchar *value_tmp;
  GdkPixmap *pixmap1 = NULL;
  GdkPixmap *pixmap2 = NULL;
  GdkBitmap *mask1 = NULL;
  GdkBitmap *mask2 = NULL;
  GdkColor transparent;
  gboolean leaf;
  gboolean expanded;
  guint8 spacing;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** Go ! */
  row = td_database_row (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));
  node = gtk_ctree_node_nth (GTK_CTREE (mod_tree->widget_data), 0);
  while (node)
    {
      gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node, mod_tree->column_oid, &txt_tmp);
      if (atoi (txt_tmp) == CURRENT_ID)
	{

	  /*** Icones - Icons */
	  td_database_set_current (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));
	  if (g_list_length (mod_tree->icon))
	    {
	      gtk_ctree_get_node_info (GTK_CTREE (mod_tree->widget_data), node, &value_tmp, &spacing, &pixmap1, &pixmap2, &mask1, &mask2, &leaf, &expanded);
	      pixmap1 = NULL;
	      for (i=0; i<g_list_length (mod_tree->icon); i++)
		if (td_database_value (TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->query_visible))
		  {
		    pixmap1 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask1, &transparent, (gchar**) TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->pixmap1);
		    pixmap2 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask2, &transparent, (gchar**) TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->pixmap2);
		    break;
		  }
	      if (!pixmap1)
		{
		  pixmap1 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask1, &transparent, (gchar**) TD_STOCK_LIST_EMPTY);
		  pixmap2 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask2, &transparent, (gchar**) TD_STOCK_LIST_EMPTY);
		}
	      GTK_CTREE_ROW (node)->pixmap_closed = pixmap1;
	      GTK_CTREE_ROW (node)->mask_closed = mask1;
	      GTK_CTREE_ROW (node)->pixmap_opened = pixmap2;
	      GTK_CTREE_ROW (node)->mask_opened = mask2;
	      if (expanded)
		gtk_ctree_node_set_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, (gchar*) g_list_nth_data (row, 0), spacing, pixmap2, mask2);
	      else
		gtk_ctree_node_set_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, (gchar*) g_list_nth_data (row, 0), spacing, pixmap1, mask1);
	    }
	  else
	    {
	      gtk_ctree_node_get_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, &value_tmp, &spacing, &pixmap1, &mask1);
	      gtk_ctree_node_set_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, (gchar*) g_list_nth_data (row, 0), spacing, pixmap1, mask1);
	    }

	  /*** Valeur - Value */
	  for (i=1; i<g_list_length (row); i++)
	    gtk_ctree_node_set_text (GTK_CTREE (mod_tree->widget_data), node, i, (gchar*) g_list_nth_data (row, i));
	  break;
	}
      node = gtdk_ctree_next (node);
    }
}

/**
 * td_mod_tree_select_node:
 * @widget: tree widget
 * @row: event's row
 * @column: event's column
 * @event: event
 * @mod_tree: tree
 * 
 * fr: Mise  jour de la node courante
 *
 * en: Update the current node
 **/

void td_mod_tree_select_node (GtkCTree *widget, gint row, gint column, GdkEvent *event, TdModTree *mod_tree)
{
  GtkCListRow *clist_row;
  gchar *txt_tmp;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** Slection - Selection */
  TD_FLAG_NO_REFRESH_NODE = TRUE;
  if (row < 0)
    {
      CURRENT_ROW = NULL;
      CURRENT_ID = 0;
      gtk_clist_unselect_all (GTK_CLIST (widget));
      mod_tree->selected = NULL;
    }
  else
    {
      gtk_clist_unselect_all (GTK_CLIST (widget));
      if ((event) && (event->type != GDK_BUTTON_PRESS))
	gtk_ctree_select (widget, gtk_ctree_node_nth (widget, row));
      mod_tree->selected = gtk_ctree_node_nth (widget, row);
      CURRENT_ROW = td_mod_tree_read_node (mod_tree, mod_tree->selected);
      CURRENT_ID = atoi (g_list_nth_data (CURRENT_ROW, mod_tree->column_oid));
      td_database_set_current (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));
    }

  /*** Editeur des proprits et Menu - Properties editor and Menu */
  td_mod_refresh_bench (TD_MOD (g_list_nth_data (TD_APP(APP)->mod_data, mod_tree->id)), GTK_OBJECT (mod_tree));
  td_mod_refresh_etabliste (TD_MOD (g_list_nth_data (TD_APP(APP)->mod_data, mod_tree->id)), GTK_OBJECT (mod_tree));
  td_mod_refresh_menu (TD_MOD (g_list_nth_data (TD_APP(APP)->mod_data, mod_tree->id)), GTK_OBJECT (mod_tree));
  TD_FLAG_NO_REFRESH_NODE = FALSE;
}

/**
 * td_mod_tree_read_node:
 * @mod_tree: tree
 * @node: node
 * 
 * fr: Retourne la liste des valeurs de la node
 *
 * en: Return the list of values of the node
 * 
 * Return value: list
 **/

GList *td_mod_tree_read_node (TdModTree *mod_tree, GtkCTreeNode *node)
{
  GList *ret = NULL;
  gchar *txt_tmp;
  guint8 spacing_tmp;
  GdkPixmap *pixmap_tmp;
  GdkBitmap *mask_tmp;
  int i;
  gtk_ctree_node_get_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, &txt_tmp, &spacing_tmp, &pixmap_tmp, &mask_tmp);
  ret = g_list_append (ret, txt_tmp);
  for (i=1; i<TD_MOD_TREE (mod_tree)->column_oid+1; i++)
    {
      gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node, i, &txt_tmp);
      ret = g_list_append (ret, txt_tmp);
    } 
  return ret;
}

/*****************************************************************************/
/*** Tirer-lcher - Drag'n'drop */
/*****************************************************************************/

/**
 * td_mod_tree_dnd_get:
 * @widget: tree widget
 * @context: drag'n'drop context
 * @selection_data: drag'n'drop selection 
 * @info: drag'n'drop info
 * @time: drag'n'drop time
 * @row: row number of selection
 * 
 * fr: Prise de donnes au tir
 *
 * en: Catch data at drag
 **/

void td_mod_tree_dnd_get (GtkWidget *widget, GdkDragContext *context, GtkSelectionData *selection_data, guint info, guint time, int row)
{
  int length;
  gchar *txt_tmp;
  txt_tmp = "(ok)";
  length = strlen (txt_tmp);
  gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING, sizeof (txt_tmp), (guchar*) txt_tmp, length*2);
}

/**
 * td_mod_tree_dnd_received:
 * @widget: tree widget
 * @context: drag'n'drop context
 * @x: x coordinate at end of drag'n'drop
 * @y: y coordinate at end of drag'n'drop
 * @selection_data: drag'n'drop selection 
 * @info: drag'n'drop info
 * @time: drag'n'drop time
 * @mod_tree: component
 * 
 * fr: Rception des donnes au lch
 *
 * en: Receipt data at drop
 **/

void td_mod_tree_dnd_received (GtkWidget *widget, GdkDragContext *context, gint x, gint y, GtkSelectionData *selection_data, guint info, guint time, TdModTree *mod_tree)
{
  GList *data = NULL;
  int drag_position = 0;
  GtkCTreeNode *node;
  GtkCTreeNode *node_parent = NULL;
  GtkCTreeNode *node_sibling = NULL;
  int row, column;
  GtkAdjustment *tampon_adjustement;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** Fin de l'auto-expansion - End of auto-expand */
  if (FLAG_DRAG_AUTOEXPAND_ID)
    gtk_timeout_remove (FLAG_DRAG_AUTOEXPAND_ID);
  
  /*** Position */
  y -= (GTK_CONTAINER (widget)->border_width + widget->style->klass->ythickness + GTK_CLIST (widget)->column_title_area.height);
  drag_position = td_mod_tree_dnd_drag_position (widget, x, y, mod_tree->widget);
  if (!drag_position)
    {
      node = NULL;
      node_parent = NULL;
      node_sibling = NULL;
    }
  else
    {
      gtk_clist_get_selection_info (GTK_CLIST (widget), x, y, &row, &column);
      node = gtk_ctree_node_nth (GTK_CTREE (widget), row);
      switch (drag_position)
	{
	case 1: /*** Avant - Before */
	  if (GTK_CTREE_ROW (node)->parent)
	    node_parent = GTK_CTREE_ROW (node)->parent;
	  else
	    node_parent = NULL;
	  node_sibling = node;
	  break;
	case 2: /*** Dedans - In */
	  node_parent = node;
	  node_sibling = NULL;
	  break;
	case 3: /*** Aprs - After */
	  if (node)
	    {
	      if (GTK_CTREE_ROW (node)->parent)
		node_parent = GTK_CTREE_ROW (node)->parent;
	      else
		node_parent = NULL;
	      if (GTK_CTREE_ROW (node)->sibling)
		node_sibling = GTK_CTREE_ROW (node)->sibling;
	      else
		node_sibling = NULL;
	    }
	  else
	    {
	      node_parent = NULL;
	      node_sibling = NULL;
	    }
	default:
	  break;
	}
    }

  /*** Effacer ligne - Deleting line */
  tampon_adjustement = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (mod_tree->widget));
  gdk_draw_line (GTK_CLIST(widget)->clist_window, widget->style->base_gc[GTK_STATE_NORMAL],
		 0, 
		 (FLAG_DRAG_ROW_OLD * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value,
		 GTK_CLIST(widget)->clist_window_width - 1, 
		 (FLAG_DRAG_ROW_OLD * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value);
  gdk_draw_line (GTK_CLIST(widget)->clist_window, widget->style->base_gc[GTK_STATE_NORMAL],
		 0, 
		 ((FLAG_DRAG_ROW_OLD + 1) * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value,
		 GTK_CLIST(widget)->clist_window_width - 1, 
		 ((FLAG_DRAG_ROW_OLD + 1) * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value);

  /*** Base de donnes - Database */
  if ((mod_tree->selected == node) ||
      (mod_tree->selected == node_parent) ||
      (mod_tree->selected == node_sibling))
    return;
  td_mod_tree_dnd_action (mod_tree, mod_tree->selected, node, node_parent, node_sibling, context);
}

/**
 * td_mod_tree_dnd_action:
 * @mod_tree: component
 * @node_drag: node of drag
 * @node_drop: node of drop
 * @node_parent: parent of futur node
 * @node_sibling: sibling of futur node
 * @context: drag'n'drop context
 * 
 * fr: Execute une modification gnre par le drag'n'drop du module
 *
 * en: Execute a modification generated by drag'n'drop of module
 **/

void td_mod_tree_dnd_action (TdModTree *mod_tree, GtkCTreeNode *node_drag, GtkCTreeNode *node_drop, GtkCTreeNode *node_parent, GtkCTreeNode *node_sibling, GdkDragContext *context)
{
  int drag_id = 0;
  int drop_id = 0;
  int check_id;
  int id_parent;
  int id_sibling;
  int old_id_sibling;
  gchar *txt_tmp;
  GList *list_tmp;
  int i, j, k;
  GList *list_id_new = NULL;
  GList *list_id_old = NULL;
  int cur_id;
  gboolean bool_tmp;
  GtkCTreeNode *node = NULL;
  GtkObject *datatable;
  gchar *txt_field;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** Identificateur - Identifier */
  gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node_drag, mod_tree->column_oid, &txt_tmp);
  drag_id = atoi (txt_tmp);
  if (node_parent)
    {
      gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node_parent, mod_tree->column_oid, &txt_tmp);
      id_parent = atoi (txt_tmp);
    }
  else
    id_parent = 0;
  if (node_sibling)
    {
      gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node_sibling, mod_tree->column_oid, &txt_tmp);
      id_sibling = atoi (txt_tmp);
    }
  else
    id_sibling = 0;

  /*** Annulation si node_drag est parent  node_drop */
  if (node_drop)
    {
      gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node_drop, mod_tree->column_oid, &txt_tmp);
      drop_id = atoi (txt_tmp);
      check_id = atoi (td_database_value (g_strdup_printf ("SELECT td_id_parent FROM %s WHERE td_id = %d;", mod_tree->table_tree, drop_id)));
      while (check_id != 0)
	{
	  if (check_id == drag_id)
	    return;
	  check_id = atoi (td_database_value (g_strdup_printf ("SELECT td_id_parent FROM %s WHERE td_id = %d;", mod_tree->table_tree, check_id)));
	}
    }
  td_mod_modified (TD_MOD (g_list_nth_data (TD_APP(APP)->mod_data, mod_tree->id)));

  /*** Arbre dplacement - Tree move */
  if (context->actions != GDK_ACTION_COPY)
    {
      old_id_sibling = atoi (td_database_value (g_strdup_printf ("SELECT td_id_sibling FROM %s WHERE td_id = %d;", mod_tree->table_tree, drag_id)));
      /*** Changement de l'ancien frre - Change of old sibling */
      td_database_command (g_strdup_printf ("UPDATE %s SET td_id_sibling = %d WHERE td_id_sibling = %d;", mod_tree->table_tree, old_id_sibling, drag_id));
      /*** Changement du nouveau frre - Change of new sibling */
      td_database_command (g_strdup_printf ("UPDATE %s SET td_id_sibling = %d WHERE (td_id_parent = %d) AND (td_id_sibling = %d);", mod_tree->table_tree, drag_id, id_parent, id_sibling));
      /*** Changement de la node prise - Change of node taken */
      td_database_command (g_strdup_printf ("UPDATE %s SET td_id_parent = %d, td_id_sibling = %d WHERE td_id = %d;", mod_tree->table_tree, id_parent, id_sibling, drag_id));
      gtk_ctree_move (GTK_CTREE (mod_tree->widget_data), node_drag, node_parent, node_sibling);
    }

  /*** Arbre copie - Tree copy */
  else
    {

      /*** Recherche des enfants - Searching childs */
      cur_id = 0;
      list_id_old = g_list_append (list_id_old, g_strdup_printf ("%d", drag_id));
      while (g_list_nth_data (list_id_old, cur_id))
	{
	  list_tmp = td_database_column (g_strdup_printf ("SELECT td_id FROM %s WHERE td_id_parent = %s;", mod_tree->table_tree, (gchar*) g_list_nth_data (list_id_old, cur_id)));
	  for (i=0; i<g_list_length (list_tmp); i++)
	    list_id_old = g_list_append (list_id_old, g_list_nth_data (list_tmp, i));
	  cur_id++;
	}

      /*** Champs - Fields */
      list_tmp = td_database_field (g_strdup_printf ("SELECT * FROM %s;", mod_tree->table));
      txt_field = "";
      for (i=0; i<g_list_length (list_tmp)-g_list_length (TD_APP (APP)->admin_field); i++)
	{
	  if (i != 0)
	    txt_field = g_strdup_printf ("%s, ", txt_field);
	  txt_field = g_strdup_printf ("%s%s", txt_field, (gchar*) g_list_nth_data (list_tmp, i));
	}

      /*** Node */
      for (i=0; i<g_list_length (list_id_old); i++)
	{
	  td_database_insert (g_strdup_printf ("INSERT INTO %s SELECT %s FROM %s WHERE td_id = %s;", mod_tree->table, txt_field, mod_tree->table, (gchar*) g_list_nth_data (list_id_old, i)), mod_tree->table); /*** Nouvelle node dans actions - New node in actions */
	  list_id_new = g_list_append (list_id_new, g_strdup_printf ("%d", CURRENT_ID));
	  if (i == 0)
	    {
	      td_database_insert (g_strdup_printf ("INSERT INTO %s (td_id, td_id_parent, td_id_sibling) VALUES (%d, %d, %d);", mod_tree->table_tree, CURRENT_ID, id_parent, id_sibling), mod_tree->table_tree); /*** Nouvelle node dans actions_tree - New node in actions */
  	      td_database_command (g_strdup_printf ("UPDATE %s SET td_id_sibling = %d WHERE (td_id_parent = %d) AND (td_id_sibling = %d) AND (td_id <> %d);", mod_tree->table_tree, CURRENT_ID, id_parent, id_sibling, CURRENT_ID)); /*** Changement du nouveau frre - Change of new child */
	      list_tmp = td_database_row (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));
	      node = td_mod_tree_add_node (mod_tree, node_parent, node_sibling, list_tmp);
	    }
	  else
	    {
	      id_parent = atoi (td_database_value (g_strdup_printf ("SELECT td_id_parent FROM %s WHERE td_id = %s;", mod_tree->table_tree, (gchar*) g_list_nth_data (list_id_old, i))));
	      bool_tmp = FALSE;
	      for (j=0; j<g_list_length (list_id_old); j++)
		if (id_parent == atoi (g_list_nth_data (list_id_old, j)))
		  {
		    if (j >= g_list_length (list_id_new))
		      break;
		    id_parent = atoi (g_list_nth_data (list_id_new, j));
		    bool_tmp = TRUE;
		    break;
		  }
	      if (!bool_tmp)
		id_parent = 0;
	      id_sibling = atoi (td_database_value (g_strdup_printf ("SELECT td_id_sibling FROM %s WHERE td_id = %s;", mod_tree->table_tree, (gchar*) g_list_nth_data (list_id_old, i))));
	      bool_tmp = FALSE;
	      for (k=0; k<g_list_length (list_id_old); k++)
		if (id_sibling == atoi (g_list_nth_data (list_id_old, k)))
		  {
		    if (k >= g_list_length (list_id_new))
		      break;
		    id_sibling = atoi (g_list_nth_data (list_id_new, k));
		    bool_tmp = TRUE;
		    break;
		  }
	      if (!bool_tmp)
		id_sibling = 0;
	      td_database_insert (g_strdup_printf ("INSERT INTO %s (td_id, td_id_parent, td_id_sibling) VALUES (%d, %d, %d);", mod_tree->table_tree, CURRENT_ID, id_parent, id_sibling), mod_tree->table_tree); /*** Nouvelle node dans actions_tree - New node in actions_tree */
  	      td_database_command (g_strdup_printf ("UPDATE %s SET td_id_sibling = %d WHERE (td_id_parent = %d) AND (td_id_sibling = %d) AND (td_id <> %d);", mod_tree->table_tree, CURRENT_ID, id_parent, id_sibling, CURRENT_ID)); /*** Changement du nouveau frre - Change of new child */
	    }
	}

      /*** Enfants directs - Direct childs */
      datatable = td_database_select (g_strdup_printf ("SELECT * FROM %s WHERE td_id IN (SELECT td_id FROM %s WHERE td_id_parent = %s);", mod_tree->table, mod_tree->table_tree, (gchar*) g_list_nth_data (list_id_new, 0)));
      for (i=0; i<g_list_length (TD_DB_DATATABLE (datatable)->item); i++)
	td_mod_tree_add_node (mod_tree, node, NULL, g_list_nth_data (TD_DB_DATATABLE (datatable)->item, i));
    }
}

gint td_mod_tree_dnd_autoexpand (gpointer widget)
{
  GtkCTreeNode *node_selection;

  /*** Auto-expansion - Auto-expand */
  if (FLAG_DRAG_AUTOEXPAND_ROW >=0)
    {
      node_selection = gtk_ctree_node_nth (GTK_CTREE (widget), FLAG_DRAG_AUTOEXPAND_ROW);
      if (node_selection)
	if (GTK_CTREE_ROW (node_selection)->children)
	  {
	    printf ("debug : scroll pour le dnd_motion : autoexpand activ\n");
	    gtk_ctree_expand (GTK_CTREE (widget), node_selection);
	  }
    }

  /*** Destruction du timeout pour l'auto-expansion - Destroying timeout of auto-expand */
  gtk_timeout_remove (FLAG_DRAG_AUTOEXPAND_ID);
  return FLAG_DRAG_AUTOEXPAND_ID;
}

/**
 * td_mod_tree_dnd_drag_position:
 * @widget: tree widget
 * @x: x coordinate 
 * @y: y coordinate 
 * @scrolledwindow: scroll positions of tree
 * 
 * fr: Retourne le numro de ligne correspondant aux coordones (x,y)
 *
 * en: Return the row number corresponding the coordinates (x,y)
 * 
 * Return value: integer
 **/

int td_mod_tree_dnd_drag_position (GtkWidget *widget, int x, int y, GtkWidget *scrolledwindow)
{
  GtkAdjustment *tampon_adjustement;
  int row, column;

  /*** Capture de la ligne - Row capture */
  tampon_adjustement = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (scrolledwindow));
  if (((y-tampon_adjustement->value)>tampon_adjustement->page_size)||(y<0)||
      (y>((GTK_CLIST(widget)->rows)*((GTK_CLIST(widget)->row_height)+1))))
    return 0;
  gtk_clist_get_selection_info (GTK_CLIST(widget), x, y, &row, &column);
  row = gtdk_clist_correction (GTK_CLIST(widget), row, y);
  if (row < 0)
    return 0;

  /*** Avant - Before */
  if (y+tampon_adjustement->value<((row*((GTK_CLIST(widget)->row_height)+1))+((GTK_CLIST(widget)->row_height)/3)))
    return 1;

  /*** Dedans - In */
  if (y+tampon_adjustement->value<((row*((GTK_CLIST(widget)->row_height)+1))+(2*((GTK_CLIST(widget)->row_height)/3))))
    return 2;

  /*** Aprs - After */
  return 3;
}

/**
 * td_mod_tree_dnd_motion:
 * @widget: tree widget
 * @context: drag'n'drop context
 * @x: x coordinate
 * @y: y coordinate
 * @time: drag'n'drop time
 * @scrolledwindow: scroll positions of tree
 * 
 * fr: Evnement 'drag_motion' de l'arbre
 *
 * en: Event 'drag_motion' of tree
 **/

void td_mod_tree_dnd_motion (GtkWidget *widget, GdkDragContext *context, gint x, gint y, guint time, GtkWidget *scrolledwindow)
{
  int row, column, drag_position;
  float i;
  gchar *txt_tampon;
  GtkCTreeNode *node_selection = NULL; 
  GtkAdjustment *tampon_adjustement;

  /*** Fin de l'auto-expansion - End of auto-expand */
  if (FLAG_DRAG_AUTOEXPAND_ID)
    gtk_timeout_remove (FLAG_DRAG_AUTOEXPAND_ID);

  /*** Action du tirer-lcher par dfaut - Default action of drag'n'drop */
  if ((context->actions != GDK_ACTION_MOVE)&&(context->actions != GDK_ACTION_COPY))
    gdk_drag_status (context, GDK_ACTION_MOVE, 0);

  /*** Capture de la ligne - Row capture */
  y -= (GTK_CONTAINER (widget)->border_width + widget->style->klass->ythickness + GTK_CLIST(widget)->column_title_area.height);
  drag_position = td_mod_tree_dnd_drag_position (widget, x, y, scrolledwindow);
  gtk_clist_get_selection_info (GTK_CLIST (widget), x, y, &row, &column);
  tampon_adjustement = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (scrolledwindow));
  
  /*** Ascenseur - Scroll */
  /*** FIXME: bibliothques GTK+ pas claires - GTK+ libs not clear */
  if ((y<10) && (tampon_adjustement->value>tampon_adjustement->lower))
    {
      gtk_adjustment_set_value (tampon_adjustement, (tampon_adjustement->value)-((GTK_CLIST (widget)->row_height)+1));
      gtk_scrolled_window_set_vadjustment (GTK_SCROLLED_WINDOW (scrolledwindow), tampon_adjustement);
      return;
    }
  if ((y>(GTK_CLIST (widget)->clist_window_height-10)) && 
      (tampon_adjustement->value<(tampon_adjustement->upper-tampon_adjustement->page_size)))
    {
      gtk_adjustment_set_value (tampon_adjustement, (tampon_adjustement->value)+((GTK_CLIST(widget)->row_height)+1));
      gtk_scrolled_window_set_vadjustment (GTK_SCROLLED_WINDOW (scrolledwindow), tampon_adjustement);
      return;
    }

  /*** Effacer ligne - Removing line */
  gdk_draw_line (GTK_CLIST(widget)->clist_window, widget->style->base_gc[GTK_STATE_NORMAL],
		 0, 
		 (FLAG_DRAG_ROW_OLD * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value,
		 GTK_CLIST(widget)->clist_window_width - 1, 
		 (FLAG_DRAG_ROW_OLD * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value);
  gdk_draw_line (GTK_CLIST(widget)->clist_window, widget->style->base_gc[GTK_STATE_NORMAL],
		 0, 
		 ((FLAG_DRAG_ROW_OLD + 1) * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value,
		 GTK_CLIST(widget)->clist_window_width - 1, 
		 ((FLAG_DRAG_ROW_OLD + 1) * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value);
  node_selection = gtk_ctree_node_nth (GTK_CTREE(widget), row);
  if (!node_selection)
    return;
  if (!drag_position)
    return;

  /*** Dessiner ligne - Drawing line */
  switch (drag_position)
    {
    case 1: /*** Avant - Before */
      gdk_draw_line (GTK_CLIST(widget)->clist_window, widget->style->fg_gc[GTK_STATE_NORMAL],
		     GTK_CTREE(widget)->tree_indent * GTK_CTREE_ROW(node_selection)->level,
		     (row * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value,
		     GTK_CLIST(widget)->clist_window_width - 1, 
		     (row * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value);
      break;
    case 2: /*** Dedans - In */
      gdk_draw_line (GTK_CLIST(widget)->clist_window, widget->style->fg_gc[GTK_STATE_NORMAL],
		     GTK_CTREE(widget)->tree_indent * GTK_CTREE_ROW(node_selection)->level,
		     (row * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value,
		     GTK_CLIST(widget)->clist_window_width - 1, 
		     (row * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value);
      gdk_draw_line (GTK_CLIST(widget)->clist_window, widget->style->fg_gc[GTK_STATE_NORMAL],
		     GTK_CTREE(widget)->tree_indent * GTK_CTREE_ROW(node_selection)->level,
		     ((row + 1) * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value,
		     GTK_CLIST(widget)->clist_window_width - 1, 
		     ((row + 1) * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value);
      
      /*** Dpart du timeout d'auto-expansion - Start the timeout for auto-expand */
      FLAG_DRAG_AUTOEXPAND_ROW = row;
      FLAG_DRAG_AUTOEXPAND_ID = gtk_timeout_add (FLAG_DRAG_AUTOEXPAND_TIME, td_mod_tree_dnd_autoexpand, widget);
      break;
    case 3: /*** Aprs - After */
      gdk_draw_line (GTK_CLIST(widget)->clist_window, widget->style->fg_gc[GTK_STATE_NORMAL],
		     GTK_CTREE(widget)->tree_indent * GTK_CTREE_ROW(node_selection)->level,
		     ((row + 1) * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value,
		     GTK_CLIST(widget)->clist_window_width - 1, 
		     ((row + 1) * ((GTK_CLIST(widget)->row_height)+1))-tampon_adjustement->value);
      break;
    default:
      break;
    }
  FLAG_DRAG_ROW_OLD = row;
}

/*****************************************************************************/
/*** Base de donnes - Database */
/*****************************************************************************/

/**
 * td_mod_tree_action:
 * @mod_tree: component
 * @type: type of action
 * 
 * fr: Execute une modification lmentaire gnre par l'arbre
 *
 * en: Execute a elementary modification generated by the tree
 **/

void td_mod_tree_action (TdModTree *mod_tree, gchar *type)
{
  GtkCTreeNode *node;
  GtkCTreeNode *node_tmp;
  GList *row;
  int i;
  gchar *txt_tmp;
  gchar *value_tmp;
  GdkPixmap *pixmap1 = NULL;
  GdkPixmap *pixmap2 = NULL;
  GdkBitmap *mask1 = NULL;
  GdkBitmap *mask2 = NULL;
  GdkColor transparent;
  gboolean leaf;
  gboolean expanded;
  guint8 spacing;
  g_return_if_fail (mod_tree != NULL);
  g_return_if_fail (TD_IS_MOD_TREE (mod_tree));

  /*** update - update1 */
  if ((!strcmp (type, "update")) || 
      (!strcmp (type, "update1")))
    {
      row = td_database_row (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));
      node = gtk_ctree_node_nth (GTK_CTREE (mod_tree->widget_data), 0);
      while (node)
	{
	  gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node, mod_tree->column_oid, &txt_tmp);
	  if (atoi (txt_tmp) == CURRENT_ID)
	    {

	      /*** Icnes - Icons */
	      if (g_list_length (mod_tree->icon))
		{
		  td_database_set_current (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));
		  for (i=0; i<g_list_length (mod_tree->icon); i++)
		    if (td_database_value (TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->query_visible))
		      {
			gtk_ctree_get_node_info (GTK_CTREE (mod_tree->widget_data), node, &value_tmp, &spacing, &pixmap1, &pixmap2, &mask1, &mask2, &leaf, &expanded);
			pixmap1 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask1, &transparent, (gchar**) TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->pixmap1);
			pixmap2 = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window, &mask2, &transparent, (gchar**) TD_MOD_TREE_ICON (g_list_nth_data (mod_tree->icon, i))->pixmap2);
			if (expanded)
			  gtk_ctree_node_set_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, (gchar*) g_list_nth_data (row, 0), spacing, pixmap2, mask2);
			else
			  gtk_ctree_node_set_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, (gchar*) g_list_nth_data (row, 0), spacing, pixmap1, mask1);
			break;
		      }
		}
	      else
		{
		  gtk_ctree_node_get_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, &value_tmp, &spacing, &pixmap1, &mask1);
		  gtk_ctree_node_set_pixtext (GTK_CTREE (mod_tree->widget_data), node, 0, (gchar*) g_list_nth_data (row, 0), spacing, pixmap1, mask1);
		}

	      /*** Valeur - Value */
	      for (i=1; i<g_list_length (row); i++)
		gtk_ctree_node_set_text (GTK_CTREE (mod_tree->widget_data), node, i, (gchar*) g_list_nth_data (row, i));
	      if (!strcmp (type, "update1"))
		break;
	    }
	  node = gtdk_ctree_next (node);
	}
      return;
    }

  /* insert - insert1 */
  if ((!strcmp (type, "insert")) || 
      (!strcmp (type, "insert1")))
    {
      row = td_database_row (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));
      node = td_mod_tree_add_node (mod_tree, NULL, NULL, row);
      return;
    }

  /* insert_child - insert_child1 */
  if ((!strcmp (type, "insert_child")) || 
      (!strcmp (type, "insert_child1")))
    {
      row = td_database_row (g_strdup_printf ("SELECT * FROM %s WHERE td_id = %d;", mod_tree->table, CURRENT_ID));
      node = gtk_ctree_node_nth (GTK_CTREE (mod_tree->widget_data), 0);
      while (node)
	{
	  gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node, mod_tree->column_oid, &txt_tmp);
	  if (atoi (txt_tmp) == CURRENT_ID_PARENT)
	    {
	      node_tmp = td_mod_tree_add_node (mod_tree, node, NULL, row);
	      gtk_ctree_expand (GTK_CTREE (mod_tree->widget_data), node_tmp);
	      if (!strcmp (type, "insert_child1"))
		break;
	    }
	  node = gtdk_ctree_next (node);
	}
      return;
    }

  /* delette - delette1 */
  if ((!strcmp (type, "delete")) || 
      (!strcmp (type, "delete1")))
    {
      node = gtk_ctree_node_nth (GTK_CTREE (mod_tree->widget_data), 0);
      while (node)
	{
	  gtk_ctree_node_get_text (GTK_CTREE (mod_tree->widget_data), node, mod_tree->column_oid, &txt_tmp);
	  if (atoi (txt_tmp) == CURRENT_ID)
	    {
	      gtk_ctree_remove_node (GTK_CTREE (mod_tree->widget_data), node);
	      if (!strcmp (type, "delete1"))
		break;
	    }
	  node = gtdk_ctree_next (node);
	}
      CURRENT_ID = 0;
      mod_tree->selected = NULL;
      return;
    }
}
