/*****************************************************************************/
/*** commons.c : Variables et fonctions communes
 *** commons.c : Commons variables and commons functions
 * 
 * 
 * ToutDoux : Chtit gestionnaire de projet - A littl' project manager
 * Copyright (c) 2000-2001 Philippe Roy
 * Auteur - Author : Philippe Roy <ph_roy@toutdoux.org>
 *
 *
 * Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier
 * sous les termes de la licence publique gnrale GNU telle qu'elle est publie par
 * la Free Software Foundation ; soit la version 2 de la licence, ou
 * (comme vous voulez) toute version ultrieure.
 *
 * Ce programme est distribu dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; mme sans la garantie de
 * COMMERCIALIT ou d'ADQUATION A UN BUT PARTICULIER. Voir la
 * licence publique gnrale GNU pour plus de dtails.
 *
 * Vous devriez avoir reu une copie de la licence publique gnrale GNU
 * avec ce programme ; si ce n'est pas le cas, crivez  la Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/*****************************************************************************/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gmodule.h>
#include <fcntl.h>
#include <dlfcn.h>
#include <dirent.h>
#include <sys/stat.h>

#include "commons.h"
#include "gtdk.h"
#include "td_app.h"
#include "td_app_process.h"
#include "td_flower.h"

/* 
 * unzip ; strip ; touch ; finger ; mount ; fsck ; more ; yes ; umount ; sleep 
 */

/*****************************************************************************/
/*** Initialisation */
/*****************************************************************************/

/**
 * td_init:
 * 
 * fr: Initialisation de la bibliothque
 *
 * en: Library initialization
 **/

void td_init (void)
{
  /*** Rpertoires - Directory */
  TD_PACKAGE_PLUGINS_DIR = g_strdup_printf ("%s/lib/toutdoux/plugins/", TD_PACKAGE_DATA_DIR);
  TD_PACKAGE_VAR_DIR = g_strdup_printf ("%s/.toutdoux", g_get_home_dir());
  TD_PACKAGE_TMP_DIR = g_strdup_printf ("%s/.toutdoux/tmp", g_get_home_dir());

  /*** Greffons fondamentaux - Core plug-ins */
  PLUGINS_CORE_BOOKMARKS = dlopen (g_strdup_printf ("%s%s", TD_PACKAGE_PLUGINS_DIR, "libhelp_bookmarks.so"), RTLD_LAZY);

  /*** Personnalisation - Customize */
  TD_CUSTOM_GEOMETRY_QUEUE = NULL;
  TD_CUSTOM_FILE_STARTUP = NULL;

  /*** Drapeux internes - Internal flags */
  TD_FLAG_ANTI_AUTOEXPAND = FALSE;

  /*** Messages de dmarage - Starting messages */
  START_MSG = NULL;

  /*** PSTricks */
  TD_PS_HEIGHT_CANVAS = 0;
  TD_PS_HEIGHT = 0;
  TD_PS_WIDTH_CANVAS = 0;
  TD_PS_WIDTH = 0;
}

/**
 * td_exit:
 * @ret: return value
 * 
 * fr: Quitter
 *
 * en: Exit
 **/

void td_exit (int ret)
{
  gtk_exit (ret);
}

/*****************************************************************************/
/*** Ok */
/*****************************************************************************/

/**
 * td_set_ok:
 * @widget: event's widget
 * @dialog_result: dialog's flag
 * 
 * fr: Quitte la boucle Gtk+
 *
 * en: Quit the Gtk+ loop
 **/

void td_set_ok (GtkWidget *widget, gboolean *dialog_result)
{
  *dialog_result = TRUE;
  gtk_main_quit();
}

/*****************************************************************************/
/*** Fonctions - Functions */
/*****************************************************************************/

/**
 * td_func_comp:
 * @a: left-hand member
 * @b: right-hand member
 * 
 * fr: Fonction : comparaison 
 *
 * en: Function : comparaison
 * 
 * Return value: result
 **/

gint td_func_comp (gconstpointer a, gconstpointer b)
{
  return (strcmp ( (gchar*) a, (gchar*) b));
}

/*****************************************************************************/
/*** Message */
/*****************************************************************************/

/**
 * td_message_create_window:
 * @message: pop window's message
 * 
 * fr: Cr et affiche une fentre  message
 *
 * en: Creates and display a window with message
 **/

void td_message_create_window (gchar *message)
{
  GtkWidget *window;
  GtkWidget *eventbox1;
  GtkWidget *hbox1;
  GtkWidget *pixmap1;
  GtkWidget *label1;

  /*** Fentre - Window */
  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtdk_window_set_title (GTK_WINDOW (window), _("Message"), TD_WINDOW_TYPE_TOOL);
  gtk_window_set_policy (GTK_WINDOW (window), FALSE, FALSE, FALSE);
  eventbox1 = gtk_event_box_new();
  gtk_widget_show (eventbox1);
  gtk_container_add (GTK_CONTAINER (window), eventbox1);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox1);
  gtk_container_add (GTK_CONTAINER (eventbox1), hbox1);
  gtk_container_set_border_width (GTK_CONTAINER (hbox1), 5);

  /*** Image */
  pixmap1 = gtdk_pixmap_file (g_strdup_printf ("%s/share/pixmaps/%s/gnome-error.png", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR));
  gtk_widget_show (pixmap1);
  gtk_box_pack_start (GTK_BOX (hbox1), pixmap1, FALSE, FALSE, 0);

  /*** Texte - Text */
  label1 = gtk_label_new (message);
  gtk_widget_show (label1);
  gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 5);

  /*** Appels - Calls */
  gtk_widget_realize (window);
  gdk_window_set_cursor (window->window, gdk_cursor_new (GDK_ARROW));
  gtk_signal_connect_object (GTK_OBJECT (window), "delete_event", GTK_SIGNAL_FUNC (gtk_widget_destroy),GTK_OBJECT (window));
  gtk_signal_connect_object (GTK_OBJECT (eventbox1), "button_press_event", GTK_SIGNAL_FUNC (gtk_widget_destroy),GTK_OBJECT (window));
  gtk_widget_show (window);
}

/*****************************************************************************/
/*** Question */
/*****************************************************************************/

/**
 * td_question_create_window:
 * @question: question
 * @title: window's title
 * 
 * fr: Cr et affiche une fentre  question
 *
 * en: Creates and display window with question
 * 
 * Return value: 0 for yes, 1 for no and 2 for cancel
 **/

int td_question_create_window (gchar *question, gchar *title)
{
  int ret;
  gboolean oui = FALSE;
  gboolean non = FALSE;
  GtkWidget *window;
  GtkWidget *vbox1;
  GtkWidget *hbox1;
  GtkWidget *pixmap1;
  GtkWidget *label1;
  GtkWidget *hseparator1;
  GtkWidget *hbox2;
  GtkWidget *button1;
  GtkWidget *button2;
  GtkWidget *button3;

  /*** Fentre - Window */
  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtdk_window_set_title (GTK_WINDOW (window), title, TD_WINDOW_TYPE_TOOL);
  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox1);
  gtk_container_add (GTK_CONTAINER (window), vbox1);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox1);
  gtk_box_pack_start (GTK_BOX (vbox1), hbox1, TRUE, TRUE, 0);

  /*** Image */
  pixmap1 = gtdk_pixmap_file (g_strdup_printf ("%s/share/pixmaps/%s/gnome-warning.png", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR));
  gtk_widget_show (pixmap1);
  gtk_box_pack_start (GTK_BOX (hbox1), pixmap1, FALSE, FALSE, 0);

  /*** Texte - Text */
  label1 = gtk_label_new (question);
  gtk_widget_show (label1);
  gtk_box_pack_end (GTK_BOX (hbox1), label1, TRUE, TRUE, GNOME_PAD);
  hseparator1 = gtk_hseparator_new();
  gtk_widget_show (hseparator1);
  gtk_box_pack_start (GTK_BOX (vbox1), hseparator1, TRUE, TRUE, 0);

  /*** Boutons - Buttons */	
  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox2);
  gtk_box_pack_start (GTK_BOX (vbox1), hbox2, TRUE, TRUE, 0);
  button1 = gnome_stock_button (GNOME_STOCK_BUTTON_YES);
  gtk_widget_show (button1);
  gtk_box_pack_start (GTK_BOX (hbox2), button1, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);
  button2 = gnome_stock_button (GNOME_STOCK_BUTTON_NO);
  gtk_widget_show (button2);
  gtk_box_pack_start (GTK_BOX (hbox2), button2, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button2, GTK_CAN_DEFAULT);
  button3 = gnome_stock_button (GNOME_STOCK_BUTTON_CANCEL);
  gtk_widget_show (button3);
  gtk_box_pack_start (GTK_BOX (hbox2), button3, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button3, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button3);

  /*** Appels - Calls */
  gtk_signal_connect (GTK_OBJECT (button1), "clicked", GTK_SIGNAL_FUNC (td_set_ok), &oui);
  gtk_signal_connect (GTK_OBJECT (button2), "clicked", GTK_SIGNAL_FUNC (td_set_ok), &non);
  gtk_signal_connect (GTK_OBJECT (button3), "clicked", GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_signal_connect (GTK_OBJECT (window), "delete_event", GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_widget_show (GTK_WIDGET (window));
  gtk_grab_add (GTK_WIDGET (window));
  gtk_main();
  if (oui)
    /*** oui - yes */
    ret = 0;
  else
    if (non)
      /*** non - no */
      ret = 1;
    else
      /*** annuler - cancel */
      ret = 2;
	
  gtk_widget_destroy (GTK_WIDGET (window));
  return ret;
}

/*****************************************************************************/
/*** Selecteur de fichier - File selector */
/*****************************************************************************/

/**
 * td_fileselector_create_window:
 * @title: window's title
 * @file: default file
 * @extension: extension of file
 * @show: TRUE for show window, FALSE for hide window
 * 
 * fr: Cr et affiche ou cache une fentre de selection de fichier
 *
 * en: Creates and display or hide window for file selection
 * 
 * Return value: file location
 **/

gchar *td_fileselector_create_window (gchar *title, gchar *file, gchar *extension, gboolean show)
{
  gchar *txt_tmp = NULL;
  gchar *file2;
  gchar *extension2;
  gboolean ok = FALSE;

  /*** Annuler - Cancel */
  if (!show)
    {
      if (WINDOW_FILE_SELECTION)
	{
	  gtdk_window_save_conf ("file_selector", WINDOW_FILE_SELECTION, TRUE, TRUE, FALSE);
	  gtk_widget_destroy (WINDOW_FILE_SELECTION);
	}
      return NULL;
    }

  /*** Fentre - Window */
  WINDOW_FILE_SELECTION = gtk_file_selection_new (title);
  gtk_window_set_policy (GTK_WINDOW (WINDOW_FILE_SELECTION), TRUE, TRUE, FALSE);
  gtdk_window_save_defaultconf ("file_selector", 600, 400);
  gtdk_window_load_conf ("file_selector", WINDOW_FILE_SELECTION);
  if (file)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (WINDOW_FILE_SELECTION), file);
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (WINDOW_FILE_SELECTION)->ok_button), "clicked", GTK_SIGNAL_FUNC (td_set_ok), &ok);
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (WINDOW_FILE_SELECTION)->cancel_button), "clicked",GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (WINDOW_FILE_SELECTION)), "delete_event", GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_widget_show (GTK_WIDGET (WINDOW_FILE_SELECTION));
  gtk_main();

  /*** Slection - Selection */
  if (ok)
    {
      txt_tmp = g_strdup_printf ("%s", gtk_file_selection_get_filename (GTK_FILE_SELECTION (WINDOW_FILE_SELECTION)));
      if (!strcmp (txt_tmp, "(null)"))
	return NULL;

      /*** Dtection de l'extension du fichier - Detecting file extension */
      if (extension)
	{
	  file2 = g_strdup_printf ("%s", txt_tmp);
	  extension2 = strtok (file2,".");
	  extension2 = strtok (NULL,".");
	  if (!extension2)
	    txt_tmp = g_strdup_printf ("%s.%s", file2, extension);
	  else
	    txt_tmp = g_strdup_printf ("%s.%s", file2, extension2);
	}
      return (txt_tmp);
    }
  return NULL;
}


/*****************************************************************************/
/*** Extension */
/*****************************************************************************/

/**
 * td_system:
 * @command: command
 * 
 * fr: Execute une commande unix sans nouveau processus
 *
 * en: Executes an unix command without new process
 * 
 * Return value: FALSE on error
 **/

gboolean td_system (gchar *command)
{
  if (system (command) != 0)
    {
      td_app_message (_("Executing command"), g_strdup_printf (_("Executing command '%s'"), command), TD_MSG_FAILED);
      return FALSE;
    }
  td_app_message (NULL, g_strdup_printf (_("Executing command '%s'"), command), TD_MSG_COMMAND);
  return TRUE;
}

/**
 * td_exec_alias:
 * @command: command
 * 
 * fr: Retourne la commande unix aprs remplacement des alias
 *
 * en: Returns the unix command after remplacing alias
 * 
 * Return value: command
 **/

gchar *td_exec_alias (gchar *command)
{
  gchar *txt_tmp;

  /*** Rpertoire utilisateur - Home */
  while (strstr (command, "~"))
    command = td_string_replace (command, "~", g_get_home_dir());

  /*** PostScript */
  while (strstr (command, "$viewer_ps"))
    command = td_string_replace (command, "$viewer_ps", td_app_custom_value ("external", "viewer", "postscript"));

  /*** SVG */
  while (strstr (command, "$editor_svg"))
    command = td_string_replace (command, "$editor_svg", td_app_custom_value ("external", "editor", "svg"));

  /*** Editeur texte - Text editor */
  while (strstr (command, "$editor"))
    {
      txt_tmp = (td_app_custom_value ("external", "editor", "text"));
      if (!strcmp (txt_tmp, "$EDITOR"))
	txt_tmp = g_getenv ("EDITOR");
      if (!txt_tmp)
	txt_tmp = "emacs"; /*** emacs par dfaut - emacs by default */
      command = td_string_replace (command, "$editor_txt", txt_tmp);
    }

  /*** Navigateur - Browser */
  while (strstr (command, "$browser"))
    {
      txt_tmp = (td_app_custom_value ("external", "browser", "web"));
      if (!strcmp (txt_tmp, "$BROWSER"))
	txt_tmp = g_getenv ("BROWSER");
      if (!txt_tmp)
	txt_tmp = "galeon"; /*** emacs par dfaut - emacs by default */
      command = td_string_replace (command, "$browser", txt_tmp);
    }
  return command;
}

/**
 * td_exec:
 * @command: command
 * 
 * fr: Execute une commande unix avec un nouveau processus
 *
 * en: Executes an unix command with new process
 * 
 * Return value: pid or -1 on error 
 **/

int td_exec (gchar *command)
{
  int ret;
  gchar **argv;
  argv = g_strsplit (td_exec_alias (command), " ", 100);
  ret = gnome_execute_async (NULL, 100, argv);
  if (ret == -1)
    {
      td_app_message (_("Executing command"), g_strdup_printf (_("Executing command '%s'"), command), TD_MSG_FAILED);
      return -1;
    }
  td_app_message (NULL, g_strdup_printf (_("Executing command '%s'"), command), TD_MSG_COMMAND);
  return ret;
}

/**
 * td_shell:
 * @command: command
 * 
 * fr: Execute une commande shell avec un nouveau processus
 *
 * en: Executes a shell command with new process
 * 
 * Return value: pid or -1 on error 
 **/

gboolean td_shell (gchar *command)
{
  if (gnome_execute_shell (NULL, td_exec_alias (command)) == -1)
    {
      td_app_message (_("Executing command"), g_strdup_printf (_("Executing command '%s'"), command), TD_MSG_FAILED);
      return FALSE;
    }
  td_app_message (NULL, g_strdup_printf (_("Executing command '%s'"), command), TD_MSG_COMMAND);
  return TRUE;
}

/**
 * td_dlsym:
 * @module: module
 * @dir: directory of module (NULL => default directory (TD_PACKAGE_PLUGINS_DIR))
 * @file: file of module
 * @symbol: symbol called
 * @warning: warning flag
 * 
 * fr: Retourne le symbole du greffon 
 *
 * en: Returns the symbol's module
 * 
 * Return value: pointer
 **/

gpointer *td_dlsym (void *module, gchar *dir, gchar *file, gchar *symbol, gboolean warning)
{
  gpointer *ret;
  if (!dir)
    dir = TD_PACKAGE_PLUGINS_DIR;
  file = g_strconcat (dir, file, NULL);
  ret = dlsym (module, symbol);
  if (dlerror() != NULL)
    {
      if ((warning) || (TD_FLAG_DEBUG_MODE))
	td_app_message (_("Executing plug-in"), g_strdup_printf (_("Executing plug-in : file '%s' : symbol '%s' not find"), file, symbol), TD_MSG_FAILED);
      return NULL;
    }
  td_app_message (NULL, g_strdup_printf (_("Executing plug-in : file '%s' : symbol '%s'"), file, symbol), TD_MSG_PLUGINS);
  return ret;
}

/**
 * td_dataplugins_filelist:
 * 
 * fr: Retourne la liste des fichiers des greffons de donnes
 *
 * en: Returns file list of data plug-ins
 * 
 * Return value: list
 **/

GList *td_dataplugins_filelist (void)
{
  GtkObject *flower;
  GList *ret = NULL;
  int i;

  /*** Lecture du fichier - Reading file */
  flower = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower), "root", "dataplugins", "branch", "dataplugin", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower), "name");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "directory");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "file");
  td_flower_load_file (TD_FLOWER (flower), g_strdup_printf("%s/share/%s/data-plugins.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR));
  for (i=0; i<g_list_length (TD_FLOWER (flower)->items_value); i++)
    ret = g_list_append (ret, g_strdup_printf ("%s%s", TD_PACKAGE_PLUGINS_DIR, (gchar*) (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 2))));
  td_flower_destroy (TD_FLOWER (flower));
  return ret;
}

/**
 * td_dataplugins_list:
 * 
 * fr: Retourne la liste des descriptions des greffons de donnes
 *
 * en: Returns description list of data plug-ins
 * 
 * Return value: description list
 **/

GtkObject *td_dataplugins_list (void)
{
  GtkObject *ret = NULL;
  int i;

  /*** Lecture du fichier - Reading file */
  ret = td_flower_new();
  gtk_object_set (GTK_OBJECT (ret), "root", "dataplugins", "branch", "dataplugin", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (ret), "name");
  td_flower_add_branch_attribut (TD_FLOWER (ret), "directory");
  td_flower_add_branch_attribut (TD_FLOWER (ret), "file");
  td_flower_load_file (TD_FLOWER (ret), g_strdup_printf("%s/share/%s/data-plugins.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR));
  return ret;
}

/*****************************************************************************/
/*** Fichier URL - File and URL */
/*****************************************************************************/

/**
 * td_file_exist:
 * @file: file
 * 
 * fr: Retourne TRUE si le fichier existe
 *
 * en: Returns TRUE if file exist
 * 
 * Return value: boolean
 **/

gboolean td_file_exist (gchar *file)
{
  struct stat statb;
  if ((stat (file, &statb) < 0) || (S_ISDIR (statb.st_mode)))
    return FALSE;
  return TRUE;
}

/**
 * td_file_backup:
 * @file: file
 * 
 * fr: Mise en sauvegarde du fichier
 *
 * en: Create the backup of the file
 * 
 * Return value: FALSE on error
 **/

gboolean td_file_backup (gchar *file)
{
  int i;
  if (!td_file_exist (g_strdup_printf ("%s.bak", file)))
    return (td_system (g_strdup_printf ("mv %s %s.bak", file, file)));
  i=2;
  while (td_file_exist (g_strdup_printf ("%s.bak.%d", file, i)))
    i++;
  return (td_system (g_strdup_printf ("mv %s %s.bak.%d", file, file, i)));
}


/**
 * td_browser:
 * @url: url
 * 
 * fr: Ouvre l'URL avec un navigateur web
 *
 * en: Opens the URL with a web browser
 **/

void td_browser (gchar *url)
{
  gchar *txt_tmp = (td_app_custom_value ("external", "browser", "web"));
  if (((!strcmp (txt_tmp, "$BROWSER")) && (!g_getenv ("BROWSER"))) || (!strcmp (txt_tmp, "GNOME preferences")))
    gnome_help_goto (NULL, txt_tmp);
  else
    {
      if (!strcmp (txt_tmp, "$BROWSER"))
	txt_tmp = g_getenv ("BROWSER");
      if (!txt_tmp)
	txt_tmp = "galeon";
      td_exec (g_strdup_printf (("%s %s"), txt_tmp, url));
    }
}

/**
 * td_editor_txt:
 * @file: file
 * 
 * fr: Ouvre le fichier texte avec un diteur de texte
 *
 * en: Opens the text file with a text editor
 **/

void td_editor_txt (gchar *file)
{
  gchar *txt_tmp = (td_app_custom_value ("external", "editor", "text"));
  if (!strcmp (txt_tmp, "$EDITOR"))
    txt_tmp = g_getenv ("EDITOR");
  if (!txt_tmp)
    txt_tmp = "emacs";
  td_exec (g_strdup_printf (("%s %s"), txt_tmp, file));
}

/**
 * td_viewer_ps:
 * @file: file
 * 
 * fr: Ouvre le fichier postscript avec un lecteur postscript
 *
 * en: Opens the postscript file with a postscript viewer
 *
 * Return value: FALSE on error
 **/

gboolean td_viewer_ps (gchar *file)
{
  if (strstr (td_app_custom_value ("external", "viewer", "postscript"), "watch"))
    {
      if (!td_app_process_update (TD_APP(APP), file, g_strdup_printf ("%s %s", td_strsplit (td_app_custom_value ("external", "viewer", "postscript"), " ", 0), file)))
	return FALSE;
      return TRUE;
    }
  if (!td_exec (g_strdup_printf ("%s %s", td_app_custom_value ("external", "viewer", "postscript"), file)))
    return FALSE;
  return TRUE;
}

/*****************************************************************************/
/*** Date */
/*****************************************************************************/

/**
 * td_sleep:
 * @ms: delay in milliseconds
 * 
 * fr: Pause
 *
 * en: Pause
 **/

void td_sleep (int ms)
{
  struct timespec tv;
  tv.tv_sec = ms/1000;
  tv.tv_nsec = (ms%1000)*1000000;
  nanosleep(&tv,NULL);
}

/**
 * td_date_current:
 * 
 * fr: Retourne la date courante
 *
 * en: Returns the current date
 * 
 * Return value: date
 **/

GDate *td_date_current (void)
{
  time_t curtime;
  struct tm *curtime2;
  curtime = time (NULL);
  curtime2 = gmtime (&curtime);
  return g_date_new_dmy (curtime2->tm_mday, curtime2->tm_mon+1, curtime2->tm_year+1900);
}

/**
 * td_date_intl:
 * @format: date format
 * 
 * format : week day/month/year, day/month/year, week/year, month/year, month/year
 * 
 * fr: Retourne le format de date localis
 *
 * en: Returns the format data localized
 * 
 * Return value: date format
 **/

gchar *td_date_intl (gchar *format)
{
  /*** Europen - European */
  if (TD_CUSTOM_DATE_FORMAT == TD_DATE_FORMAT_EU)
    return format;

  /*** Amricain - American */
  if (!strcmp (format, "week day/month/year"))
    return "week year-month-day";
  if (!strcmp (format, "day/month/year"))
    return "year-month-day";
  if (!strcmp (format, "week/year"))
    return "year-week";
  if (!strcmp (format, "month/year"))
    return "year-month";
  return NULL;
}

/**
 * td_date_parse:
 * @date: string
 * @format: date format
 * 
 * format : day/month/year, year-month-day
 * 
 * fr: Retourne la date correspondant  la chane de caractres
 *
 * en: Returns the date corresponding the string
 * 
 * Return value: date
 **/

GDate *td_date_parse (gchar *date, gchar *format)
{
  int day, month, year;
  if (!strcmp (format , "day/month/year"))
    sscanf(date, "%d/%d/%d", &day, &month, &year);
  if (!strcmp (format , "year-month-day"))
    sscanf(date, "%d-%d-%d", &year, &month, &day);
  if (!g_date_valid_dmy (day, month, year))
    return NULL;
  return g_date_new_dmy (day, month, year);
}

/**
 * td_date_print:
 * @date: date
 * @format: date format
 * 
 * format : day, day_name, day_abbrev, day/month, day/month/year, month-day, year-month-day, 
 * week day/month/year, week year-month-day, 
 * day month_name, day month_name year, day month_abbrev, day month_abbrev year, 
 * day_name month_name, day_name month_name year, 
 * day_abbrev month_abbrev, day_abbrev month_abbrev year,
 * day_name day month_name, day_name day month_name year, day_abbrev day month_abbrev, 
 * day_abbrev day month_abbrev year, day_name month_name day, day_name month_name day year, day_abbrev month_abbrev day, day_abbrev month_abbrev day year
 * 
 * fr: Retourne la chane de caractres correspondant  la date
 *
 * en: Returns the string corresponding the date
 * 
 * Return value: string
 **/

gchar *td_date_print (GDate *date, gchar *format)
{
  return td_date_day (g_date_day (date), g_date_month (date), g_date_year (date), format);
}

/**
 * td_date_month_name:
 * @month: month number
 * @abbrev: TRUE for abbreviation
 * 
 * fr: Retourne le nom du mois
 *
 * en: Returns the month's name
 * 
 * Return value: string
 **/

gchar *td_date_month_name (int month, gboolean abbrev)
{
  if (abbrev)
    switch (month)
      {
      case 1:
	return _("jan");
      case 2:
	return _("feb");
      case 3:
	return _("mar");
      case 4:
	return _("apr");
      case 5:
	return _("may");
      case 6:
	return _("jun");
      case 7:
	return _("jul");
      case 8:
	return _("aug");
      case 9:
	return _("sep");
      case 10:
	return _("oct");
      case 11:
	return _("nov");
      case 12:
	return _("dec");
      default:
	break;
      }
  else
    switch (month)
      {
      case 1:
	return _("January");
      case 2:
	return _("February");
      case 3:
	return _("March");
      case 4:
	return _("April");
      case 5:
	return _("May");
      case 6:
	return _("June");
      case 7:
	return _("July");
      case 8:
	return _("August");
      case 9:
	return _("September");
      case 10:
	return _("October");
      case 11:
	return _("November");
      case 12:
	return _("December");
      default:
	break;
      }
  return NULL;
}

/**
 * td_date_month:
 * @month: month number
 * @year: year number
 * @format : date format
 * 
 * format : month, month_name, month_abbrev, month/year, year-month, month_name year, month_abbrev year
 * 
 * fr: Retourne la chane de caractres formate de la date
 *
 * en: Returns the formated string of the date
 * 
 * Return value: string
 **/

gchar *td_date_month (int month, int year, gchar *format)
{
  gchar *tampon;
  GDate* date_tmp;

  /*** Chiffre - Number */
  if (!strcmp (format , "month"))
    return (g_strdup_printf ("%d", month));

  /*** Nom - Name */
  if (!strcmp (format , "month_name"))
    return (td_date_month_name (month, FALSE));
  if (!strcmp (format , "month_abbrev"))
    return (td_date_month_name (month, TRUE));

  /*** Chiffres - Numbers */
  if (!strcmp (format , "month/year"))
    return (g_strdup_printf ("%02d/%d", month, year));
  if (!strcmp (format , "year-month"))
    return (g_strdup_printf ("%d-%02d", year,month));

  /*** Noms - Names */
  if (!strcmp (format , "month_name year"))
    return (g_strdup_printf ("%s %d", td_date_month_name (month, FALSE), year));
  if (!strcmp (format , "month_abbrev year"))
    return (g_strdup_printf ("%s %d", td_date_month_name (month, TRUE), year));
  return NULL;
}

/**
 * td_date_week:
 * @week: week number
 * @year: year number
 * @format : date format
 * 
 * format : week, week/year, year-week
 * 
 * fr: Retourne la chane de caractres formate de la date
 *
 * en: Returns the formated string of the date
 * 
 * Return value: string
 **/

gchar *td_date_week (int week, int year, gchar *format)
{
  if (!strcmp (format , "week"))
    return (g_strdup_printf ("%s%d", _("W"), week));
  if (!strcmp (format , "week/year"))
    return (g_strdup_printf ("%s%d/%d", _("W"), week, year));
  if (!strcmp (format , "year-week"))
    return (g_strdup_printf ("%d-%s%d", year, _("W"), week));
  return NULL;
}

/**
 * td_date_day_name:
 * @date: week number
 * @abbrev: TRUE for abbreviation
 * 
 * fr: Retourne le nom du jour
 *
 * en: Returns the day's name
 * 
 * Return value: string
 **/

gchar *td_date_day_name (GDate *date, gboolean abbrev)
{
  GDateWeekday day_name;
  day_name = g_date_weekday (date);
  if (abbrev)
    switch (day_name)
      {
      case 1:
	return _("mon");
      case 2:
	return _("tue");
      case 3:
	return _("wed");
      case 4:
	return _("thu");
      case 5:
	return _("fri");
      case 6:
	return _("sat");
      case 7:
	return _("sun");
      default:
	break;
      }
  else
    switch (day_name)
      {
      case 1:
	return _("Monday");
      case 2:
	return _("Tuesday");
      case 3:
	return _("Wednesday");
      case 4:
	return _("Thursday");
      case 5:
	return _("Friday");
      case 6:
	return _("Saturday");
      case 7:
	return _("Sunday");
      default:
	break;
      }
  return NULL;
}

/**
 * td_date_day:
 * @day: day number
 * @month: month number
 * @year: year number
 * @format : date format
 * 
 * format : day, day_name, day_abbrev, day/month, day/month/year, month-day, year-month-day, 
 * week day/month/year, week year-month-day, 
 * day month_name, day month_name year, day month_abbrev, day month_abbrev year, 
 * day_name month_name, day_name month_name year, 
 * day_abbrev month_abbrev, day_abbrev month_abbrev year,
 * day_name day month_name, day_name day month_name year, day_abbrev day month_abbrev, 
 * day_abbrev day month_abbrev year, day_name month_name day, day_name month_name day year, day_abbrev month_abbrev day, day_abbrev month_abbrev day year
 * 
 * fr: Retourne la chane de caractres formate de la date
 *
 * en: Returns the formated string of the date
 * 
 * Return value: string
 **/

gchar *td_date_day (int day, int month, int year, gchar *format)
{
  GDate *date;

  /*** Chiffre - Number */
  if (!strcmp (format, "day"))
    return (g_strdup_printf ("%02d", day));

  /*** Nom - Name */
  date = g_date_new_dmy (day, month, year);
  if (!strcmp (format, "day_name"))
    return (td_date_day_name (date, FALSE));
  if (!strcmp (format, "day_abbrev"))
    return (td_date_day_name (date, TRUE));

  /*** Chiffres - Numbers */
  if (!strcmp (format, "day/month"))
    return (g_strdup_printf ("%02d/%02d", day, month));
  if (!strcmp (format, "day/month/year"))
    return (g_strdup_printf ("%02d/%02d/%d", day, month, year));
  if (!strcmp (format, "month-day"))
    return (g_strdup_printf ("%02d-%02d", month, day));
  if (!strcmp (format, "year-month-day"))
    return (g_strdup_printf ("%d-%02d-%02d", year, month, day));
  if (!strcmp (format, "week day/month/year"))
    return (g_strdup_printf ("%s%d %02d/%02d/%d", _("W"), g_date_monday_week_of_year (date), day, month, year));
  if (!strcmp (format, "week year-month-day"))
    return (g_strdup_printf ("%s%d %02d-%02d-%d", _("W"), g_date_monday_week_of_year (date), year, month, day));

  /*** Chiffre+nom - Numbers+name */
  if (!strcmp (format, "day month_name"))
    return (g_strdup_printf ("%d %s", day, td_date_month_name (month, FALSE)));
  if (!strcmp (format, "day month_name year"))
    return (g_strdup_printf ("%d %s %d", day, td_date_month_name (month, FALSE), year));
  if (!strcmp (format, "day month_abbrev"))
    return (g_strdup_printf ("%d %s", day, td_date_month_name (month, TRUE)));
  if (!strcmp (format, "day month_abbrev year"))
    return (g_strdup_printf ("%d %s %d", day, td_date_month_name (month, TRUE), year));

  /*** Noms - Names */
  if (!strcmp (format, "day_name month_name"))
    return (g_strdup_printf ("%s %s", td_date_day_name (date, FALSE), td_date_month_name (month, FALSE)));
  if (!strcmp (format, "day_name month_name year"))
    return (g_strdup_printf ("%s %s %d", td_date_day_name (date, FALSE), td_date_month_name (month, FALSE), year));
  if (!strcmp (format, "day_abbrev month_abbrev"))
    return (g_strdup_printf ("%s %s", td_date_day_name (date, TRUE), td_date_month_name (month, TRUE)));
  if (!strcmp (format, "day_abbrev month_abbrev year"))
    return (g_strdup_printf ("%s %s %d", td_date_day_name (date, TRUE), td_date_month_name (month, TRUE), year));

  /*** Chiffre+noms EU - Numbers+names EU */
  if (!strcmp (format, "day_name day month_name"))
    return (g_strdup_printf ("%s %d %s", td_date_day_name (date, FALSE), day, td_date_month_name (month, FALSE)));
  if (!strcmp (format, "day_name day month_name year"))
    return (g_strdup_printf ("%s %d %s %d", td_date_day_name (date, FALSE), day, td_date_month_name (month, FALSE), year));
  if (!strcmp (format, "day_abbrev day month_abbrev"))
    return (g_strdup_printf ("%s %d %s", td_date_day_name (date, TRUE), day, td_date_month_name (month, TRUE)));
  if (!strcmp (format, "day_abbrev day month_abbrev year"))
    return (g_strdup_printf ("%s %d %s %d", td_date_day_name (date, TRUE), day, td_date_month_name (month, TRUE), year));

  /*** Chiffre+noms US - Number+names US */
  if (!strcmp (format, "day_name month_name day"))
    return (g_strdup_printf ("%s %s %d", td_date_day_name (date, FALSE), td_date_month_name (month, FALSE), day));
  if (!strcmp (format, "day_name month_name day year"))
    return (g_strdup_printf ("%s %s %d %d", td_date_day_name (date, FALSE), td_date_month_name (month, FALSE), day, year));
  if (!strcmp (format, "day_abbrev month_abbrev day"))
    return (g_strdup_printf ("%s %s %d", td_date_day_name (date, TRUE), td_date_month_name (month, TRUE), day));
  if (!strcmp (format, "day_abbrev month_abbrev day year"))
    return (g_strdup_printf ("%s %s %d %d", td_date_day_name (date, TRUE), td_date_month_name (month, TRUE), day, year));
  return NULL;
}

/**
 * td_date_period_day:
 * @lower: begin date
 * @upper: end date 
 * 
 * fr: Retourne la distance en jours entre les deux dates
 *
 * en: Returns the distance between dates
 * 
 * Return value: integer
 **/

int td_date_period_day (GDate *lower, GDate *upper)
{
  int day;
  int i, j;

  /*** Annes, mois et jours - Years, months and day */
  day = 0;
  for (i=g_date_year (lower); i<g_date_year (upper)+1; i++)
    {

      /*** Premire et dernire anne - First and last year */
      if ((i==g_date_year (lower)) && (i==g_date_year (upper)))
	for (j=g_date_month (lower); j<g_date_month (upper)+1; j++)
	  {
	    if ((j==g_date_month (lower)) && (j==g_date_month (upper))) /*** Premier et dernier mois - First and last month */
	      day = day + g_date_day (upper) - g_date_day (lower);
	    if ((j==g_date_month (lower)) && (j!=g_date_month (upper))) /*** Premier mois - First month */
	      day = day + g_date_days_in_month (j, i) - g_date_day (lower);
	    if ((j!=g_date_month (lower)) && (j==g_date_month (upper))) /*** Dernier mois - Last month */
	      day = day + g_date_day (upper);
	    if ((j!=g_date_month (lower)) && (j!=g_date_month (upper)))
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Premire anne - First year */
      if ((i==g_date_year (lower)) && (i!=g_date_year (upper)))
	for (j=g_date_month (lower); j<13; j++)
	  {
	    if (j==g_date_month (lower)) /*** Premier mois - First month */
	      day = day + g_date_days_in_month (j, i) - g_date_day (lower);
	    else
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Dernire anne - Last year */
      if ((i!=g_date_year (lower)) && (i==g_date_year (upper)))
	for (j=1; j<g_date_month (upper)+1; j++)
	  {
	    if (j==g_date_month (upper)) /*** Dernier mois - Last month */
	      day = day + g_date_day (upper);
	    else
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Anne intermdiaire - Intermediate year */
      if ((i!=g_date_year (lower)) && (i!=g_date_year (upper)))
	for (j=1; j<13; j++)
	  day = day + g_date_days_in_month (j, i);
    }
  return day;
}

/**
 * td_date_add_days:
 * @date: date
 * @days: days quantity for addition
 * 
 * fr: Retourne une nouvelle date resultant de l'addition
 *
 * en: Returns a new date result the addition
 * 
 * Return value: date
 **/

GDate *td_date_add_days (GDate *date, int days)
{
  GDate *ret;
  ret = g_date_new_dmy (g_date_day (date), g_date_month (date), g_date_year (date));
  g_date_add_days (ret, days);
  return ret;
}

/**
 * td_date_add_months:
 * @date: date
 * @months: months quantity for addition
 * 
 * fr: Retourne une nouvelle date resultant de l'addition
 *
 * en: Returns a new date result the addition
 * 
 * Return value: date
 **/

GDate *td_date_add_months (GDate *date, int months)
{
  GDate *ret;
  ret = g_date_new_dmy (g_date_day (date), g_date_month (date), g_date_year (date));
  g_date_add_months (ret, months);
  return ret;
}

/**
 * td_date_add_years:
 * @date: date
 * @years: months quantity for addition
 * 
 * fr: Retourne une nouvelle date resultant de l'addition
 *
 * en: Returns a new date result the addition
 * 
 * Return value: date
 **/

GDate *td_date_add_years (GDate *date, int years)
{
  GDate *ret;
  ret = g_date_new_dmy (g_date_day (date), g_date_month (date), g_date_year (date));
  g_date_add_years (ret, years);
  return ret;
}

/*****************************************************************************/
/*** Chane de caractres - String */
/*****************************************************************************/

/**
 * td_string_replace:
 * @source: source
 * @before: before
 * @after: after
 * 
 * fr: Remplace une chane de caractre
 *
 * en: Remplaces a string
 * 
 * Return value: result
 **/

gchar *td_string_replace (gchar *source, gchar *before, gchar *after)
{
  gchar *txt_tmp;
  gchar *txt_tmp2;
  int l, l1, l2, l3; 
  if ((!source) || (!before) || (!after))
    return NULL;
  txt_tmp = strstr (source, before);
  if (!txt_tmp)
    return source;
  l = strlen (source);
  l2 = strlen (txt_tmp);
  l1 = l-l2;
  l3 = strlen (before);
  txt_tmp2 = g_strdup_printf ("%s%s%s", g_strndup (source, l1), after, g_strndup (source+l1+l3, l2-l3));
  return txt_tmp2;
}

/**
 * td_string_remove:
 * @source: source
 * @string: string
 * 
 * fr: Supprime une chane de caractre
 *
 * en: Remove a string
 * 
 * Return value: result
 **/

gchar *td_string_remove (gchar *source, gchar *string)
{
  gchar *txt_tmp;
  if ((!source) || (!string))
    return NULL;
  txt_tmp = strstr (source, string);
  if (!txt_tmp)
    return source;
  return g_strstrip (g_strndup (source, strlen (source)-strlen (txt_tmp)));
}

/**
 * td_atoi:
 * @value: value
 * 
 * fr: Retourne l'entier correspondant  la chane de caractres
 *
 * en: Returns the integer corresponding the string
 * 
 * Return value: integer
 **/

int td_atoi (gchar *value)
{
  if (!value)
    return 0;
  return atoi (value);
}

/**
 * td_strsplit:
 * @string: string
 * @delimiter: delimiter
 * @index: index
 * 
 * fr: Retourne la valeur[index] du tableau
 *
 * en: Returns the value[index] of the array
 * 
 * Return value: value
 **/

gchar *td_strsplit (gchar *string, gchar *delimiter, int index)
{
  gchar **string2;
  if (!string)
    return NULL;
  if (!delimiter)
    {
      if (strstr (string, ","))
	delimiter = ",";
      else
	{
	  if (strstr (string, "."))
	    delimiter = ".";
	  else
	    {
	      if (strstr (string, "|"))
		delimiter = "|";
	      else
		{
		  if (strstr (string, "-"))
		    delimiter = "-";
		  else
		    return NULL;
		}
	    }
	}
    }
  string2 = g_strsplit (string, delimiter, index+1);
  return (g_strstrip (string2 [index]));
}

/**
 * td_strsplist:
 * @string: string
 * @delimiter: delimiter
 * 
 * fr: Retourne la liste des valeur du tableau
 *
 * en: Returns the value list of the array
 * 
 * Return value: list
 **/

GList *td_strsplist (gchar *string, gchar *delimiter)
{
  GList *ret = NULL;
  int i;
  gchar **string2;
  if (!string)
    return NULL;
  if (!delimiter)
    {
      if (strstr (string, ","))
	delimiter = ",";
      else
	{
	  if (strstr (string, "."))
	    delimiter = ".";
	  else
	    {
	      if (strstr (string, "|"))
		delimiter = "|";
	      else
		{
		  if (strstr (string, "-"))
		    delimiter = "-";
		  else
		    return NULL;
		}
	    }
	}
    }
  if (!strcmp (delimiter, "(null)"))
    {
      ret = g_list_append (ret, string);
      return ret;
    }
  string2 = g_strsplit (string, delimiter, strlen (string));
  i = -1;
  while (string2 [i++])
    ret = g_list_append (ret, string2 [i]);
  return ret;
}

/**
 * td_string_file:
 * @file: file
 * 
 * fr: Retourne le contenu du fichier
 *
 * en: Returns the file content
 * 
 * Return value: string
 **/

GString *td_string_file (gchar *file)
{
  int i;
  gint fd;
  GString *ret;
  GIOChannel *iochannel;
  char *buff;
  guint readden;
  buff = g_malloc0 (BUFFER_SIZE);
  ret = g_string_new ("");

  /* Go ! */
  fd = open (file, O_RDONLY);
  if (fd < 0)
    td_app_message (NULL, g_strdup_printf (_("Loading file '%s' unreadable"), file), TD_MSG_FAILED);
  iochannel = g_io_channel_unix_new (fd);
  g_io_channel_read (iochannel, buff, BUFFER_SIZE, &readden);
  if (readden == 0)
    return NULL;
  i=0;
  while (readden != 0)
    {
      g_string_append (ret, g_strdup_printf ("%s", buff));
      g_free (buff);
      buff = g_malloc0 (BUFFER_SIZE);
      g_io_channel_read (iochannel, buff, BUFFER_SIZE, &readden);
    }
  close (fd);
  g_free (buff);
  return ret;
}

/*****************************************************************************/
/*** Flottant - Float */
/*****************************************************************************/

/**
 * td_floor:
 * @value: value
 * @base: base
 * 
 * fr: Retourne l'arrondi par defaut
 *
 * en: Returns the round down
 * 
 * Return value: integer
 **/

int td_floor (float value, int base)
{
  return (int) value-((int) value % base);
}

/*****************************************************************************/
/*** Liste - List */
/*****************************************************************************/

/**
 * td_list_type_parse:
 * @value: value
 * 
 * fr: Retourne le type de liste correspondant  la chane de caractres
 *
 * en: Returns the list type corresponding the string
 * 
 * Return value: list type
 **/

TdListType td_list_type_parse (gchar *value)
{
  if (!strstr (value, "static"))
    return TD_LIST_TYPE_QUERY_STATIC;
  if (!strstr (value, "query"))
    return TD_LIST_TYPE_QUERY;
return TD_LIST_TYPE_ITEM;
}

/*****************************************************************************/
/*** XML */
/*****************************************************************************/

/**
 * td_xml_check_children:
 * @cur: XML node
 * @name: children name
 * 
 * fr: Retourne le petit enfant
 *
 * en: Returns the little children
 * 
 * Return value: XML node
 **/

XmlNode *td_xml_check_children (XmlNode *cur, gchar *name)
{
  if ((cur) && (!strcmp (cur->name, name)))
    {
      if (cur->xmlChildrenNode)
	return cur->xmlChildrenNode;
      else
	return NULL;
    }
  return NULL;
}

/*****************************************************************************/
/*** Filtre - Filter */
/*****************************************************************************/

/**
 * td_filter_custom_value:
 * @name: filter name
 * @category: category
 * @group: group
 * @option: option
 * 
 * fr: Retourne la valeur de l'option de personnalisation du filtre
 *
 * en: Returns the value of the option of the filter's customize
 * 
 * Return value: string
 **/

gchar *td_filter_custom_value (gchar *name, gchar *category, gchar *group, gchar *option)
{
  return (td_custom_value (name, category, group, option));
}

/*****************************************************************************/
/*** Personnalisation et paramtrage - Customizing and parameters */
/*****************************************************************************/

/**
 * td_custom_value:
 * @module: module name
 * @category: category
 * @group: group
 * @option: option
 * 
 * fr: Retourne la valeur de l'option de personnalisation du module
 *
 * en: Returns the value of the option of the module's customize
 * 
 * Return value: string
 **/

gchar *td_custom_value (gchar *module, gchar *category, gchar *group, gchar *option)
{
  gchar *value;
  gchar *file;
  gchar *file_recovery;

  /*** Fichier - File */
  file = g_strdup_printf ("%s/customize/%s.xml", TD_PACKAGE_VAR_DIR, module);
  if (!td_file_exist (file))
    {
      file = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, module, _("en"));
      if (!td_file_exist (file))
	{
	  file = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, module, "en");
	  if (!td_file_exist (file))
	    {
	      td_app_message (NULL, g_strdup_printf (_("Loading customize '%s'"), file), TD_MSG_FAILED);
	      return NULL;
	    }
	}
    }
  if ((!TD_CUSTOM_FILE_CURRENT) || (strcmp (TD_CUSTOM_FILE_CURRENT, file)))
    TD_CUSTOM_FILE_CURRENT = g_strdup_printf ("%s", file);

  /*** Valeur - Value */
  value = td_custom_value_real (category, group, option);
  if (value)
    return value;

  /*** Fichier standard - Standard file */
  if (strcmp (file, g_strdup_printf ("%s/customize/%s.xml", TD_PACKAGE_VAR_DIR, module)))
    {
      td_app_message (NULL, g_strdup_printf (_("Loading customize '%s' : standard file corrupted"), file), TD_MSG_FAILED);
      return NULL;
    }
  else
    {
      file_recovery = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, module, _("en"));
      if (!td_file_exist (file_recovery))
	{
	  file_recovery = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, module, "en");
	  if (!td_file_exist (file_recovery))
	    {
	      td_app_message (NULL, g_strdup_printf (_("Loading customize '%s' : standard file not find"), file), TD_MSG_FAILED);
	      return NULL;
	    }
	}
    }
  if (!td_file_backup (file))
    {
      td_app_message (NULL, g_strdup_printf (_("Loading customize '%s' : create backup failed"), file), TD_MSG_FAILED);
      return NULL;
    }
  TD_CUSTOM_FILE_CURRENT = g_strdup_printf ("%s", file_recovery);
  value = td_custom_value_real (category, group, option);
  if (value)
    return value;
  td_app_message (NULL, g_strdup_printf (_("Loading customize '%s' : standard file corrupted"), file), TD_MSG_FAILED);
  return NULL;
}

/**
 * td_custom_value_real:
 * @category: category
 * @group: group
 * @option: option
 * 
 * fr: Retourne la valeur de l'option de personnalisation courante
 *
 * en: Returns the value of the option of the current customize
 * 
 * Return value: string
 **/

static gchar *td_custom_value_real (gchar *category, gchar *group, gchar *option)
{
  int index_category = -1;
  int index_group = -1;
  int index_option = -1;
  int i;

  /*** Donnes - Data */
  TD_CUSTOM_CURRENT = td_flower_new();
  gtk_object_set (GTK_OBJECT (TD_CUSTOM_CURRENT),
		  "root", "customize",
		  "branch", "category",
		  "petal", "group",
		  "foliole", "option",
		  "stomate", "predef",
		  "mode_type", TRUE, NULL);
  td_flower_add_branch_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "name");
  td_flower_add_branch_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "name_intl");
  td_flower_add_petal_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "name");
  td_flower_add_petal_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "name_intl");
  td_flower_add_foliole_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "name");
  td_flower_add_foliole_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "name_intl");
  td_flower_add_foliole_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "value");
  td_flower_add_foliole_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "widget");
  td_flower_add_stomate_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "name");
  td_flower_add_stomate_attribut (TD_FLOWER (TD_CUSTOM_CURRENT), "value");
  if (!td_flower_load_file (TD_FLOWER (TD_CUSTOM_CURRENT), TD_CUSTOM_FILE_CURRENT))
    return NULL;

  /*** Index */
  for (i=0; i<g_list_length (TD_FLOWER (TD_CUSTOM_CURRENT)->items_value); i++)
    if ((GPOINTER_TO_INT (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_level, i)) == 0) &&
	(!strcmp (category, (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_value, i), 0))))
      {
	index_category = i;
	break;
      }
  if (index_category == -1)
    {
      td_app_message (NULL, g_strdup_printf (_("Loading customize : application.category 'app.%s' not found"), category), TD_MSG_FAILED);
      return NULL;
    }
  for (i=index_category+1; i<g_list_length (TD_FLOWER (TD_CUSTOM_CURRENT)->items_value); i++)
    if ((GPOINTER_TO_INT (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_level, i)) == 1) &&
	(!strcmp (group, (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_value, i), 0))))
      {
	index_group = i;
	break;
      }
    else
      if (GPOINTER_TO_INT (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_level, i)) == 0)
	break;
  if (index_group == -1)
    {
      td_app_message (NULL, g_strdup_printf (_("Loading customize : application.category.group 'app.%s.%s' not found"), category, group), TD_MSG_FAILED);
      return NULL;
    }
  for (i=index_group+1; i<g_list_length (TD_FLOWER (TD_CUSTOM_CURRENT)->items_value); i++)
    if ((GPOINTER_TO_INT (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_level, i)) == 2) &&
	(!strcmp (option, (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_value, i), 0))))
      {
	index_option = i;
	break;
      }
    else
      if (GPOINTER_TO_INT (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_level, i)) == 1)
	break;
  if (index_option == -1)
    {
      td_app_message (NULL, g_strdup_printf (_("Loading customize : application.category.group.option 'app.%s.%s.%s' not found"), category, group, option), TD_MSG_FAILED);
      return NULL;
    }

  /*** Valeur - Value */
  return (g_list_nth_data (g_list_nth_data (TD_FLOWER (TD_CUSTOM_CURRENT)->items_value, index_option), 2));
}
