// standard imports
const Lang = imports.lang;
const Signals = imports.signals;
const Tweener = imports.tweener.tweener;

// gi imports
const Tb = imports.gi.Tb;
const Clutter = imports.gi.Clutter;

// ui imports
const Background = imports.ui.background;
const Thing = imports.ui.thing;

// model imports
const PageModel = imports.model.pageModel;

const _LAYER_BACKGROUND   = 0.1;
const _LAYER_THING        = 0.2;
const _LAYER_DIMMING      = 0.3;

const _ADD_THING_TIME = 0.5;
const _ADD_THING_TRANSITION = 'easeOutCubic';

const _SET_BACKGROUND_TIME = 0.3;
const _SET_BACKGROUND_TRANSITION = 'easeOutCubic';

const _NEW_THING_TOP_MARGIN = 60;
const _NEW_THING_LEFT_MARGIN = 60;
const _NEW_THING_H_SPACING = 30;
const _NEW_THING_V_SPACING = 20;
const _NEW_THING_COLS_INC = 30;
const _NEW_THING_ROWS_INC = 10;

const _N_COLS_SLIDE_IN = 3;
const _N_ROWS_SLIDE_IN = 3;

function Page(args) {
    this._init(args);
}

Page.prototype = {
    _init : function(args) {
        args = args || {};

        if ('model' in args) {
            this._model = args.model;
        } else {
            throw new Error("Page model is required");
        }

        if ('context' in args) {
            this._context = args.context;
        } else {
            throw new Error("Page context is required");
        }

        this._loaded = false;
        this._thingsLoaded = false;
        this._things = [];
        this._activeThing = null;

        this._createMainBox();
        this._createDimBox();
        this._connectModelSignals();

        this._updateFromModel();

        if (!this._model.isNew()) {
            this._model.load();
        }
    },

    _createMainBox : function() {
        this._mainBox =
            new Tb.Box({ orientation: Tb.BoxOrientation.VERTICAL,
                         xAlign: Tb.BoxAlignment.FILL,
                         yAlign: Tb.BoxAlignment.FILL,
                         reactive: true,
                         name: "page-main-box" });

        let clickAction = new Clutter.ClickAction();

        clickAction.connect("clicked",
                            Lang.bind(this, this._onMainBoxClicked));

        this._mainBox.add_action(clickAction);
    },

    _createDimBox : function() {
        this._dimBox =
            new Tb.Box({ orientation: Tb.BoxOrientation.VERTICAL,
                         xAlign: Tb.BoxAlignment.FILL,
                         yAlign: Tb.BoxAlignment.FILL,
                         depth: _LAYER_DIMMING,
                         opacity: 0,
                         name: "page-dim-box" });

        this._mainBox.append(this._dimBox,
                             Tb.BoxPackFlags.FIXED);

        this._mainBox.set_fixed_child_align(this._dimBox,
                                            Tb.BoxAlignment.FILL,
                                            Tb.BoxAlignment.FILL);
    },

    _connectModelSignals : function() {
        this._modelStateChangedId =
            this._model.connect("state-changed",
                                Lang.bind(this, this._onModelStateChanged));
    },

    _updateFromModel : function() {
        this._updateLoaded();

        if (this._model.state == PageModel.State.LOADED) {
            this._loadModel();
        }
    },

    _updateLoaded : function() {
        let loaded = this._thingsLoaded &&
                     this._model.state == PageModel.State.LOADED;

        if (this._loaded == loaded) {
            return;
        }

        this._loaded = loaded;

        this.emit("loaded-changed");
    },

    _positionIsTaken : function(x, y, actor) {
        return (x >= actor.x && x <= actor.x + actor.width &&
                y >= actor.y && y <= actor.y + actor.height);
    },

    _areaIsTaken : function(x1, y1, x2, y2) {
        for (let i = 0; i < this._things.length; ++i) {
            let actor = this._things[i].actor;

            if (this._positionIsTaken(x1, y1, actor) ||
                this._positionIsTaken(x1, y2, actor) ||
                this._positionIsTaken(x2, y1, actor) ||
                this._positionIsTaken(x2, y2, actor)) {
                return true;
            }
        }

        return false;
    },

    _findPositionForNewThing : function(thing) {
        let [windowWidth, windowHeight] = this._context.gtkWindow.get_size();

        let totalWidth = windowWidth +_NEW_THING_LEFT_MARGIN;
        let totalHeight = windowHeight + _NEW_THING_TOP_MARGIN;

        let nCols = Math.floor(totalWidth / _NEW_THING_COLS_INC);

        let nRows = Math.floor(totalHeight / _NEW_THING_ROWS_INC);

        for (let i = 0; i < nRows; ++i) {
            for (let j = 0; j < nCols; ++j) {
                let x1 = (j * _NEW_THING_COLS_INC) +
                         (j * _NEW_THING_H_SPACING) +
                         _NEW_THING_LEFT_MARGIN;

                let y1 = (i * _NEW_THING_ROWS_INC) +
                         (i * _NEW_THING_V_SPACING) +
                         _NEW_THING_TOP_MARGIN;

                let x2 = x1 + thing.initialWidth;
                let y2 = y1 + thing.initialHeight;

                if (x2 > windowWidth || y2 > windowHeight) {
                    continue;
                }

                if (!this._areaIsTaken(x1, y1, x2, y2)) {
                    return [x1, y1];
                }
            }
        }

        return [(windowWidth - thing.initialWidth) / 2,
                (windowHeight - thing.initialHeight) / 2];
    },

    _loadThingsFromModel : function() {
        let things = this._model.things;

        for (let i = 0; i < things.length; ++i) {
            let state = things[i];
            this.addThingFromState(state, false /* add to model */);
        }
    },

    _loadBackgroundFromModel : function() {
        // If background is undefined in the model, we'll
        // just use the default one (see createBackgroundFromId)
        let backgroundId = this._model.background;
        this.setBackground(backgroundId);
    },

    _loadModel : function() {
        this._loadBackgroundFromModel();
        this._loadThingsFromModel();

        // FIXME: This should be done in a more involved way
        // by checking the loaded state of background and the
        // things in this page
        this._thingsLoaded = true;

        this._updateLoaded();
    },

    _addThing : function(thing, state) {
        thing.actor.depth = _LAYER_THING;

        let x;
        let y;

        if (state && 'x' in state && 'y' in state) {
            x = state.x;
            y = state.y;
        } else {
            [x, y] = this._findPositionForNewThing(thing);
        }

        thing.actor.x = x;
        thing.actor.y = y;

        thing._Page_activateId =
            thing.connect("activate",
                          Lang.bind(this, this._onThingActivate));

        thing._Page_deactivateId =
            thing.connect("deactivate",
                          Lang.bind(this, this._onThingDeactivate));

        thing._Page_dragBeginId =
            thing.connect("drag-begin",
                          Lang.bind(this, this._onThingDragBegin));

        thing._Page_dragEndId =
            thing.connect("drag-end",
                          Lang.bind(this, this._onThingDragEnd));

        thing._Page_removeId =
            thing.connect("remove",
                          Lang.bind(this, this._onThingRemove));

        thing._Page_saveId =
            thing.connect("save",
                          Lang.bind(this, this._onThingSave));

        this._mainBox.append(thing.actor,
                             Tb.BoxPackFlags.FIXED);

        this._things.push(thing);
    },

    _getAnimationAnchorForNewThing : function(thing) {
        let [windowWidth, windowHeight] = this._context.gtkWindow.get_size();

        let col = Math.floor(thing.actor.x / (windowWidth / _N_COLS_SLIDE_IN));
        let row = Math.floor(thing.actor.y / (windowHeight / _N_ROWS_SLIDE_IN));

        switch (col) {
        case 0:
        case 2:
            // slide in horizontally if coming from sides
            let anchorX = (col == 0 ? 1 : -1) *
                          (thing.actor.x +
                          (col == 0 ? thing.initialWidth : 0));
            return [anchorX, 0];

        case 1:
            // slide in vertically if coming from center
            let anchorY = (row == 0 || row == 2 ? 1 : -1) *
                          (thing.actor.y +
                          (row == 0 || row == 2 ? thing.initialHeight : 0));
            return [0, anchorY];
        }

        return [0, 0];
    },

    _destroyAllThings : function() {
        while (this._things.length > 0) {
            let thing = this._things[0];
            this.removeThing(thing);
        }
    },

    _animateNewThing : function(thing) {
        let [anchorX, anchorY] =
            this._getAnimationAnchorForNewThing(thing);

        thing.actor.anchorX = anchorX;
        thing.actor.anchorY = anchorY;

        Tweener.addTween(thing.actor,
                         { anchorX: 0,
                           anchorY: 0,
                           time: _ADD_THING_TIME,
                           transition: _ADD_THING_TRANSITION });
    },

    _showDimBox : function() {
        Tweener.addTween(this._dimBox,
                         { opacity: 255,
                           time: _ADD_THING_TIME,
                           transition: _ADD_THING_TRANSITION });
    },

    _hideDimBox : function() {
        Tweener.addTween(this._dimBox,
                         { opacity: 0,
                           time: _ADD_THING_TIME,
                           transition: _ADD_THING_TRANSITION });
    },

    _onMainBoxClicked : function(o, event) {
        this.setActiveThing(null);
        this.emit("clicked");
    },

    _onModelStateChanged : function() {
        this._updateFromModel();
    },

    _onThingActivate : function(thing) {
        this.setActiveThing(thing);
    },

    _onThingDeactivate : function(thing) {
        this.setActiveThing(null);
    },

    _onThingDragBegin : function(thing) {
        thing.actor.raise(null);

        if (this._activeThing && this._activeThing != thing) {
            this.setActiveThing(null);
        }
    },

    _onThingDragEnd : function(thing) {
        if (this._activeThing !== thing) {
            thing.actor.lower(this._dimBox);
        }
    },

    _onThingRemove : function(thing) {
        this.removeThing(thing);
    },

    _onThingSave : function(thing) {
        let index = this._things.indexOf(thing);
        this._model.updateThing(index, thing.onGetState());
    },

    setActiveThing : function(activeThing) {
        if (this._activeThing === activeThing) {
            return;
        }

        if (this._activeThing) {
            this._activeThing.onDeactivate();
            this._activeThing.actor.depth = _LAYER_THING;
        }

        this._activeThing = activeThing;

        if (this._activeThing) {
            this._activeThing.onActivate();

            this._dimBox.raise(null);
            this._activeThing.actor.raise(this._dimBox);

            this._showDimBox();
        } else {
            this._hideDimBox();
        }

        this.emit("active-thing-changed");
    },

    addThingFromState : function(state, addToModel) {
        // add to model by default
        if (addToModel === undefined) {
            addToModel = true;
        }

        let thingId = state.id;

        let args = { parentActor: this._mainBox,
                     context: this._context };

        let thing = Thing.createThingFromId(thingId, args);
        thing.onLoadState(state);

        this._addThing(thing, state);

        if (addToModel) {
            this._model.addThing(state);

            // animate only new things if window is visible
            if (this._context.gtkWindow.visible) {
                this.setActiveThing(thing);
                this._animateNewThing(thing);
            }
        }
    },

    removeThing : function(thing) {
        let indexToRemove = this._things.indexOf(thing);
        this._things.splice(indexToRemove, 1);

        thing.disconnect(thing._Page_activateId);
        thing.disconnect(thing._Page_deactivateId);
        thing.disconnect(thing._Page_dragBeginId);
        thing.disconnect(thing._Page_dragEndId);
        thing.disconnect(thing._Page_removeId);
        thing.disconnect(thing._Page_saveId);

        if (this._activeThing == thing) {
            this.setActiveThing(null);
        }

        thing.onRemove();

        this._model.removeThing(indexToRemove);
    },

    setBackground : function(backgroundId) {
        if (this._background && this._background.id == backgroundId) {
            return;
        }

        let oldBackground = this._background;

        this._background = Background.createBackgroundFromId(backgroundId);

        this._background.actor.depth = _LAYER_BACKGROUND;
        this._background.actor.opacity = 0;

        this._mainBox.append(this._background.actor,
                             Tb.BoxPackFlags.FIXED);

        this._mainBox.set_fixed_child_align(this._background.actor,
                                            Tb.BoxAlignment.FILL,
                                            Tb.BoxAlignment.FILL);

        this._model.setBackground(backgroundId);

        if (oldBackground) {
            this._background.actor.raise(oldBackground.actor);
        }

        Tweener.addTween(this._background.actor,
                         { opacity: 255,
                           time: oldBackground ?_SET_BACKGROUND_TIME : 0,
                           transition: _SET_BACKGROUND_TRANSITION,
                           onComplete: function() {
                               if (oldBackground) {
                                   Tweener.removeTweens(oldBackground.actor);
                                   oldBackground.destroy();
                               }
                           }});
    },

    destroy : function() {
        this._destroyAllThings();

        if (this._modelStateChangedId) {
            this._model.disconnect(this._modelStateChangedId);
            delete this._modelStateChangedId;
        }

        this._model.unload();

        if (this._mainBox) {
            this._mainBox.destroy();
            delete this._mainBox;
        }
    },

    get loaded() {
        return this._loaded;
    },

    get activeThing() {
        return this._activeThing;
    },

    get model() {
        return this._model;
    },

    get actor() {
        return this._mainBox;
    }
}

Signals.addSignalMethods(Page.prototype);
