/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * soup-apache.c: Asyncronous Callback-based SOAP Request Queue.
 *
 * Authors:
 *      Alex Graveley (alex@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_HTTPD_H

#include <httpd.h>
#include <http_config.h>
#include <http_core.h>
#include <http_log.h>
#include <http_main.h>
#include <http_protocol.h>

#include <glib.h>
#include <libsoup/soup-server.h>
#include <libsoup/soup-private.h>

static void 
soup_apache_init (server_rec *s, pool *p)
{
	soup_server_init ();
}

static int
soup_apache_add_header_from_table (gpointer    data, 
				   const char *key, 
				   const char *val)
{
	GHashTable *hash = data;
	g_hash_table_insert (hash, g_strdup (key), g_strdup (val));
	return TRUE;
}

static SoupMessage *
soup_apache_message_create (request_rec *r, gchar *read_buf, gint read_len)
{
	SoupMessage *msg;
	SoupContext *ctx;
	SoupAction   action;
	char *uri;
	
	uri = ap_construct_url (r->pool, r->unparsed_uri, r);
	ctx = soup_context_get (uri);
	if (!ctx) return NULL;

	action = (SoupAction) ap_table_get (r->headers_in, "SOAPAction");

	msg = soup_message_new (ctx, action);
	if (!msg) return NULL;

	msg->request.owner = SOUP_BUFFER_STATIC;
	msg->request.length = read_len;
	msg->request.body = read_buf;

	msg->request_headers = g_hash_table_new (soup_str_case_hash, 
						 soup_str_case_equal);
	
	ap_table_do (soup_apache_add_header_from_table,
		     (gpointer) msg->request_headers,
		     r->headers_in,
		     NULL);

	msg->response_code = HTTP_OK;
	msg->response_phrase = g_strdup ("OK");
	msg->response_headers = g_hash_table_new (soup_str_case_hash, 
						  soup_str_case_equal);

	return msg;
}

static int
soup_apache_read_request (request_rec *r, gchar **out, gint *len)
{
	int ret = OK;
	gchar *read_buf;
	gint read_len, read_left, chunk;

	ret = ap_setup_client_block (r, REQUEST_CHUNKED_ERROR);
	if (ret != OK) return ret;

	if (!ap_should_client_block (r)) return !OK;

	read_buf = ap_palloc (r->pool, r->remaining + 1);
	read_len = read_left = r->remaining;

	ap_hard_timeout ("soup_apache_read_request", r);

	for (; read_left; read_left -= chunk) {
		chunk = ap_get_client_block (r, 
					     &read_buf [read_len - read_left], 
					     read_left);

		ap_reset_timeout (r);
	
		if (chunk <= 0) return HTTP_INTERNAL_SERVER_ERROR;
	}

	ap_kill_timeout (r);

	/* Null terminate the buffer */
	read_buf[read_len] = '\0';
	
	*out = read_buf;
	*len = read_len;
			
	return OK;
}

static void
soup_apache_add_header (gchar *key, gchar *val, request_rec *r)
{
	ap_table_set (r->headers_out, key, val);
}

static int 
soup_apache_handler (request_rec *r) 
{
	SoupAction action;
	SoupServerHandler *hand;
	SoupServerAuthToken auth_token;
	SoupMessage *msg;
	int ret = OK;
	gchar *read_buf = NULL;
	const gchar *auth_type;
	gint   read_len = 0;

	if (r->method_number != M_POST) return DECLINED;

	action = (SoupAction) ap_table_get (r->headers_in, "SOAPAction");
	if (!action) {
		ap_log_error (APLOG_MARK, 
			      APLOG_INFO, 
			      r->server, 
			      "No SOAPAction header in request.");
		return DECLINED;
	}

	hand = soup_server_get_handler (action);
	if (!hand) {
		ap_log_error (APLOG_MARK, 
			      APLOG_INFO, 
			      r->server, 
			      "No handler found for methodname '%s'.", 
			      action);
		return HTTP_NOT_ACCEPTABLE;
	}

	ret = soup_apache_read_request (r, &read_buf, &read_len);
	if (ret != OK) return ret;

	msg = soup_apache_message_create (r, read_buf, read_len);
	if (!msg) return HTTP_INTERNAL_SERVER_ERROR;

	auth_type = ap_auth_type (r);

	if (auth_type && !g_strcasecmp (auth_type, "Digest")) {
		/* Digest auth currently unsupported */
	} else if (auth_type) {
		auth_token.type = SOUP_AUTH_TYPE_BASIC;
		auth_token.basic.username = r->connection->user;

		ap_get_basic_auth_pw (r, &auth_token.basic.password);

		if (!soup_server_authorize (msg, &auth_token)) {
			ap_note_basic_auth_failure (r);
			return HTTP_UNAUTHORIZED;
		}
	}

	/* Call method handler */
	(*hand->cb) (msg, hand->user_data);

	r->content_type = "text/xml; charset=utf-8";
	r->status_line = ap_psprintf (r->pool, 
				      "%d %s", 
				      msg->response_code,
				      msg->response_phrase);

	g_hash_table_foreach (msg->response_headers, 
			      (GHFunc) soup_apache_add_header,
			      r);

	ap_hard_timeout ("soup_apache_handler", r);

	ap_send_http_header (r);

	ap_rwrite (msg->response.body, msg->response.length, r);

	ap_kill_timeout (r);

	ret = msg->response_code == HTTP_OK ? OK : !OK;

	soup_message_free (msg);

	return ret;
}

static int 
soup_apache_check_user_id (request_rec *r)
{
	return DECLINED;
}

static int 
soup_apache_auth_checker (request_rec *r)
{
	return DECLINED;
}

static int 
soup_apache_access_checker (request_rec *r)
{
	return DECLINED;
}

static const handler_rec soup_apache_handlers [] =
{
    {"soup-handler", soup_apache_handler},
    {"text/xml", soup_apache_handler},
    {NULL}
};

module MODULE_VAR_EXPORT soup_module =
{
	STANDARD_MODULE_STUFF,
	soup_apache_init,               /* module initializer */
	NULL,                           /* per-directory config creator */
	NULL,                           /* dir config merger */
	NULL,                           /* server config creator */
	NULL,                           /* server config merger */
	NULL,                           /* command table */
	soup_apache_handlers,           /* [9] list of handlers */
	NULL,                           /* [2] filename-to-URI translation */
	soup_apache_check_user_id,      /* [5] check/validate user_id */
	soup_apache_auth_checker,       /* [6] check user_id is valid *here* */
	soup_apache_access_checker,     /* [4] check access by host address */
	NULL,                           /* [7] MIME type checker/setter */
	NULL,                           /* [8] fixups */
	NULL,                           /* [10] logger */
#if MODULE_MAGIC_NUMBER >= 19970103
	NULL,                           /* [3] header parser */
#endif
#if MODULE_MAGIC_NUMBER >= 19970719
	NULL,                           /* process initializer */
#endif
#if MODULE_MAGIC_NUMBER >= 19970728
	NULL,                           /* process exit/cleanup */
#endif
#if MODULE_MAGIC_NUMBER >= 19970902
	NULL                            /* [1] post read_request handling */
#endif
};

#endif /* HAVE_HTTPD_H */
