/* Copyright 2009 Yorba Foundation
 *
 * This software is licensed under the GNU LGPL (version 2.1 or later).
 * See the COPYING file in this distribution. 
 */

private class BasicProperties : Gtk.HBox {
    private Gtk.Label label = new Gtk.Label("");
    private Gtk.Label info = new Gtk.Label(""); 
    private string title;
    private time_t start_time = time_t();
    private time_t end_time = time_t();
    private Dimensions dimensions;
    private uint64 filesize;
    private int photo_count;      
    private int event_count;
    private string basic_properties_labels;
    private string basic_properties_info;
    private bool first_line; 

    public BasicProperties() {
        label.set_justify(Gtk.Justification.RIGHT);
        label.set_alignment(0, (float) 5e-1);
        info.set_alignment(0, (float) 5e-1);
        pack_start(label, false, false, 3);
        pack_start(info, true, true, 3);
        
        info.set_ellipsize(Pango.EllipsizeMode.END);
    }

    private void add_line(string label, string info) {
        if (!first_line) {
            basic_properties_labels += "\n";
            basic_properties_info += "\n";
        }
        basic_properties_labels += label;
        basic_properties_info += info;
        first_line = false;
    }
    
    private string get_prettyprint_time(Time time) {
        string timestring = time.format(_("%I:%M %p"));
        
        if (timestring[0] == '0')
            timestring = timestring.substring(1, -1);
        
        return timestring;
    }

    private string get_prettyprint_date(Time date) {
        string date_string = null;
        Time today = Time.local(time_t());
        if (date.day_of_year == today.day_of_year && date.year == today.year) {
            date_string = _("Today");
        } else if (date.day_of_year == (today.day_of_year - 1) && date.year == today.year) {
            date_string = _("Yesterday");
        } else {
            date_string = format_local_date(date);
        }

        return date_string;
    }

    private void set_text() {
        label.set_text(basic_properties_labels);
        info.set_text(basic_properties_info);
    }

    public void clear_properties() {
        title = "";
        start_time = 0;
        end_time = 0;
        dimensions = Dimensions(0,0);
        filesize = 0;
        photo_count = -1;
        event_count = -1;
        basic_properties_labels = "";
        basic_properties_info = "";
        first_line = true;
    }

    private void get_single_properties(DataView view) {
        DataSource source = view.get_source();

        title = source.get_name();
        
        if (source is PhotoSource) {
            PhotoSource photo_source = (PhotoSource) source;
            
            start_time = photo_source.get_exposure_time();
            end_time = start_time;
            
            dimensions = photo_source.get_dimensions();
            
            filesize = photo_source.get_filesize();
        } else if (source is EventSource) {
            EventSource event_source = (EventSource) source;

            start_time = event_source.get_start_time();
            end_time = event_source.get_end_time();

            filesize = event_source.get_total_filesize();
            photo_count = event_source.get_photo_count();
        }
    }

    private void get_multiple_properties(Gee.Iterable<DataView>? iter) {
        photo_count = 0;
        foreach (DataView view in iter) {
            DataSource source = view.get_source();
            
            if (source is PhotoSource) {
                PhotoSource photo_source = (PhotoSource) source;
                    
                time_t exposure_time = photo_source.get_exposure_time();

                if (exposure_time != 0) {
                    if (start_time == 0 || exposure_time < start_time)
                        start_time = exposure_time;

                    if (end_time == 0 || exposure_time > end_time)
                        end_time = exposure_time;
                }
                
                filesize += photo_source.get_filesize();
                photo_count++;
            } else if (source is EventSource) {
                EventSource event_source = (EventSource) source;
          
                if (event_count == -1)
                    event_count = 0;

                if ((start_time == 0 || event_source.get_start_time() < start_time) &&
                    event_source.get_start_time() != 0 ) {
                    start_time = event_source.get_start_time();
                }
                if ((end_time == 0 || event_source.get_end_time() > end_time) &&
                    event_source.get_end_time() != 0 ) {
                    end_time = event_source.get_end_time();
                } else if (end_time == 0 || event_source.get_start_time() > end_time) {
                    end_time = event_source.get_start_time();
                }

                filesize += event_source.get_total_filesize();
                photo_count += event_source.get_photo_count();
                event_count++;
            }
        }
    }

    private void get_properties(Page current_page) {
        ViewCollection view = current_page.get_view();
        if (view == null)
            return;

        // summarize selected items, if none selected, summarize all
        int count = view.get_selected_count();
        Gee.Iterable<DataView> iter = null;
        if (count != 0) {
            iter = view.get_selected();
        } else {
            count = view.get_count();
            iter = (Gee.Iterable<DataView>) view.get_all();
        }
        
        if (iter == null || count == 0)
            return;

        if (count == 1) {
            foreach (DataView item in iter) {
                get_single_properties(item);
                
                break;
            }
        } else {
            get_multiple_properties(iter);
        }

        if (end_time == 0)
            end_time = start_time;
        if (start_time == 0)
            start_time = end_time;
    }

    public void update_properties(Page current_page) {
        clear_properties();
        get_properties(current_page);

        if (title != "")
            add_line(_("Title:"), title);

        if (photo_count >= 0) {
            string label = _("Items:");
  
            if (event_count >= 0) {
                string event_num_string;
                if (event_count == 1)
                    event_num_string = _("%d Event");
                else
                    event_num_string = _("%d Events");

                add_line(label, event_num_string.printf(event_count));
                label = "";
            }

            string photo_num_string;
            if (photo_count == 1)
                photo_num_string = _("%d Photo");
            else
                photo_num_string = _("%d Photos");

            add_line(label, photo_num_string.printf(photo_count));
        }

        if (start_time != 0) {
            string start_date = get_prettyprint_date(Time.local(start_time));
            string start_time = get_prettyprint_time(Time.local(start_time));
            string end_date = get_prettyprint_date(Time.local(end_time));
            string end_time = get_prettyprint_time(Time.local(end_time));

            if (start_date == end_date) {
                // display only one date if start and end are the same
                add_line(_("Date:"), start_date);

                if (start_time == end_time) {
                    // display only one time if start and end are the same
                    add_line(_("Time:"), start_time);
                } else {
                    // display time range
                    add_line(_("From:"), start_time);
                    add_line(_("To:"), end_time);
                }
            } else {
                // display date range
                add_line(_("From:"), start_date);
                add_line(_("To:"), end_date);
            }
        }

        if (filesize > 0 || dimensions.has_area()) {
            string label = _("Size:");

            if (dimensions.has_area()) {
                add_line(label, "%d x %d".printf(dimensions.width, dimensions.height));
                label = "";
            }

            if (filesize > 0)
                add_line(label, format_size_for_display((int64) filesize));
        }

        set_text();
    }
}
