/*
 * Copyright (C) 2008, 2009 Nokia Corporation, all rights reserved.
 *
 * Author: Zeeshan Ali (Khattak) <zeeshanak@gnome.org>
 *                               <zeeshan.ali@nokia.com>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "rygel-http-server.h"
#include <libgupnp/gupnp.h>
#include <gee/arraylist.h>
#include <libsoup/soup.h>
#include <gee/collection.h>
#include "rygel-media-container.h"
#include "rygel-http-request.h"
#include "rygel-media-object.h"




struct _RygelHTTPServerPrivate {
	char* path_root;
	RygelMediaContainer* root_container;
	GUPnPContext* context;
	GeeArrayList* requests;
	GCancellable* cancellable;
};

#define RYGEL_HTTP_SERVER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), RYGEL_TYPE_HTTP_SERVER, RygelHTTPServerPrivate))
enum  {
	RYGEL_HTTP_SERVER_DUMMY_PROPERTY
};
#define RYGEL_HTTP_SERVER_SERVER_PATH_PREFIX "/RygelHTTPServer"
static void _rygel_http_server_server_handler_soup_server_callback (SoupServer* server, SoupMessage* msg, const char* path, GHashTable* query, SoupClientContext* client, gpointer self);
static void _rygel_http_server_on_cancelled_g_cancellable_cancelled (GCancellable* _sender, gpointer self);
static void rygel_http_server_real_run (RygelStateMachine* base, GCancellable* cancellable);
static void rygel_http_server_on_cancelled (RygelHTTPServer* self, GCancellable* cancellable);
static char* rygel_http_server_create_uri_for_path (RygelHTTPServer* self, const char* path);
static void rygel_http_server_on_request_completed (RygelHTTPServer* self, RygelHTTPRequest* request);
static void _rygel_http_server_on_request_completed_rygel_state_machine_completed (RygelHTTPRequest* _sender, gpointer self);
static void rygel_http_server_server_handler (RygelHTTPServer* self, SoupServer* server, SoupMessage* msg, const char* server_path, GHashTable* query, SoupClientContext* soup_client);
static gpointer rygel_http_server_parent_class = NULL;
static RygelStateMachineIface* rygel_http_server_rygel_state_machine_parent_iface = NULL;
static void rygel_http_server_finalize (GObject* obj);



RygelHTTPServer* rygel_http_server_construct (GType object_type, RygelContentDirectory* content_dir, const char* name) {
	RygelHTTPServer * self;
	RygelMediaContainer* _tmp1;
	RygelMediaContainer* _tmp0;
	GUPnPContext* _tmp3;
	GUPnPContext* _tmp2;
	GeeArrayList* _tmp4;
	char* _tmp5;
	g_return_val_if_fail (content_dir != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	self = g_object_newv (object_type, 0, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->root_container = (_tmp1 = (_tmp0 = content_dir->root_container, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->root_container == NULL) ? NULL : (self->priv->root_container = (g_object_unref (self->priv->root_container), NULL)), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->context = (_tmp3 = (_tmp2 = gupnp_service_info_get_context ((GUPnPServiceInfo*) content_dir), (_tmp2 == NULL) ? NULL : g_object_ref (_tmp2)), (self->priv->context == NULL) ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)), _tmp3);
	_tmp4 = NULL;
	self->priv->requests = (_tmp4 = gee_array_list_new (RYGEL_TYPE_HTTP_REQUEST, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal), (self->priv->requests == NULL) ? NULL : (self->priv->requests = (g_object_unref (self->priv->requests), NULL)), _tmp4);
	_tmp5 = NULL;
	self->priv->path_root = (_tmp5 = g_strconcat (RYGEL_HTTP_SERVER_SERVER_PATH_PREFIX "/", name, NULL), self->priv->path_root = (g_free (self->priv->path_root), NULL), _tmp5);
	return self;
}


RygelHTTPServer* rygel_http_server_new (RygelContentDirectory* content_dir, const char* name) {
	return rygel_http_server_construct (RYGEL_TYPE_HTTP_SERVER, content_dir, name);
}


static void _rygel_http_server_server_handler_soup_server_callback (SoupServer* server, SoupMessage* msg, const char* path, GHashTable* query, SoupClientContext* client, gpointer self) {
	rygel_http_server_server_handler (self, server, msg, path, query, client);
}


static void _rygel_http_server_on_cancelled_g_cancellable_cancelled (GCancellable* _sender, gpointer self) {
	rygel_http_server_on_cancelled (self, _sender);
}


static void rygel_http_server_real_run (RygelStateMachine* base, GCancellable* cancellable) {
	RygelHTTPServer * self;
	self = (RygelHTTPServer*) base;
	soup_server_add_handler (gupnp_context_get_server (self->priv->context), self->priv->path_root, _rygel_http_server_server_handler_soup_server_callback, g_object_ref (self), g_object_unref);
	if (cancellable != NULL) {
		GCancellable* _tmp1;
		GCancellable* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cancellable = (_tmp1 = (_tmp0 = cancellable, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->cancellable == NULL) ? NULL : (self->priv->cancellable = (g_object_unref (self->priv->cancellable), NULL)), _tmp1);
		g_signal_connect_object (self->priv->cancellable, "cancelled", (GCallback) _rygel_http_server_on_cancelled_g_cancellable_cancelled, self, 0);
	}
}


static void rygel_http_server_on_cancelled (RygelHTTPServer* self, GCancellable* cancellable) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cancellable != NULL);
	/* Cancel all state machines*/
	g_cancellable_cancel (self->priv->cancellable);
	soup_server_remove_handler (gupnp_context_get_server (self->priv->context), self->priv->path_root);
	g_signal_emit_by_name ((RygelStateMachine*) self, "completed");
}


static char* rygel_http_server_create_uri_for_path (RygelHTTPServer* self, const char* path) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (path != NULL, NULL);
	return g_strdup_printf ("http://%s:%u%s%s", gupnp_context_get_host_ip (self->priv->context), gupnp_context_get_port (self->priv->context), self->priv->path_root, path);
}


char* rygel_http_server_create_http_uri_for_item (RygelHTTPServer* self, RygelMediaItem* item) {
	char* escaped;
	char* query;
	char* _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (item != NULL, NULL);
	escaped = g_uri_escape_string (((RygelMediaObject*) item)->id, "", TRUE);
	query = g_strdup_printf ("?itemid=%s", escaped);
	_tmp0 = NULL;
	return (_tmp0 = rygel_http_server_create_uri_for_path (self, query), escaped = (g_free (escaped), NULL), query = (g_free (query), NULL), _tmp0);
}


static void rygel_http_server_on_request_completed (RygelHTTPServer* self, RygelHTTPRequest* request) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (request != NULL);
	/* Remove the request from our list. */
	gee_collection_remove ((GeeCollection*) self->priv->requests, request);
}


static void _rygel_http_server_on_request_completed_rygel_state_machine_completed (RygelHTTPRequest* _sender, gpointer self) {
	rygel_http_server_on_request_completed (self, _sender);
}


static void rygel_http_server_server_handler (RygelHTTPServer* self, SoupServer* server, SoupMessage* msg, const char* server_path, GHashTable* query, SoupClientContext* soup_client) {
	RygelHTTPRequest* request;
	g_return_if_fail (self != NULL);
	g_return_if_fail (server != NULL);
	g_return_if_fail (msg != NULL);
	g_return_if_fail (server_path != NULL);
	g_return_if_fail (soup_client != NULL);
	request = rygel_http_request_new (self->priv->root_container, server, msg, query);
	g_signal_connect_object ((RygelStateMachine*) request, "completed", (GCallback) _rygel_http_server_on_request_completed_rygel_state_machine_completed, self, 0);
	gee_collection_add ((GeeCollection*) self->priv->requests, request);
	rygel_state_machine_run ((RygelStateMachine*) request, self->priv->cancellable);
	(request == NULL) ? NULL : (request = (g_object_unref (request), NULL));
}


static void rygel_http_server_class_init (RygelHTTPServerClass * klass) {
	rygel_http_server_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (RygelHTTPServerPrivate));
	G_OBJECT_CLASS (klass)->finalize = rygel_http_server_finalize;
}


static void rygel_http_server_rygel_state_machine_interface_init (RygelStateMachineIface * iface) {
	rygel_http_server_rygel_state_machine_parent_iface = g_type_interface_peek_parent (iface);
	iface->run = rygel_http_server_real_run;
}


static void rygel_http_server_instance_init (RygelHTTPServer * self) {
	self->priv = RYGEL_HTTP_SERVER_GET_PRIVATE (self);
}


static void rygel_http_server_finalize (GObject* obj) {
	RygelHTTPServer * self;
	self = RYGEL_HTTP_SERVER (obj);
	self->priv->path_root = (g_free (self->priv->path_root), NULL);
	(self->priv->root_container == NULL) ? NULL : (self->priv->root_container = (g_object_unref (self->priv->root_container), NULL));
	(self->priv->context == NULL) ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL));
	(self->priv->requests == NULL) ? NULL : (self->priv->requests = (g_object_unref (self->priv->requests), NULL));
	(self->priv->cancellable == NULL) ? NULL : (self->priv->cancellable = (g_object_unref (self->priv->cancellable), NULL));
	G_OBJECT_CLASS (rygel_http_server_parent_class)->finalize (obj);
}


GType rygel_http_server_get_type (void) {
	static GType rygel_http_server_type_id = 0;
	if (rygel_http_server_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (RygelHTTPServerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) rygel_http_server_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (RygelHTTPServer), 0, (GInstanceInitFunc) rygel_http_server_instance_init, NULL };
		static const GInterfaceInfo rygel_state_machine_info = { (GInterfaceInitFunc) rygel_http_server_rygel_state_machine_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		rygel_http_server_type_id = g_type_register_static (G_TYPE_OBJECT, "RygelHTTPServer", &g_define_type_info, 0);
		g_type_add_interface_static (rygel_http_server_type_id, RYGEL_TYPE_STATE_MACHINE, &rygel_state_machine_info);
	}
	return rygel_http_server_type_id;
}




