/*
 This file is part of pybliographer
 
 Copyright (C) 1998-1999 Frederic GOBRY
 Email : gobry@idiap.ch
 	   
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2 
 of the License, or (at your option) any later version.
   
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details. 
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 
 $Id: bibtex.c,v 1.18 1999/07/30 12:34:45 gobry Exp $
*/

#include <errno.h>
#include "bibtex.h"

void bibtex_message_handler (const gchar *log_domain G_GNUC_UNUSED,
			     GLogLevelFlags log_level,
			     const gchar *message,
			     gpointer user_data G_GNUC_UNUSED)
{
    gchar * name = g_get_prgname ();

    if (name) {
	fprintf (stderr, "%s: ", name);
    }

    switch (log_level) {

    case BIB_LEVEL_ERROR:
	fprintf (stderr, "error: %s\n", message);
	break;
    case BIB_LEVEL_WARNING:
	fprintf (stderr, "warning: %s\n", message);
	break;
    case BIB_LEVEL_MESSAGE:
	printf ("%s\n", message);
	break;
	
    default:
	fprintf (stderr, "<unknown level %d>: %s\n", log_level, message);
	break;
    }
}

void 
bibtex_set_default_handler (void) {
    g_log_set_handler (G_LOG_DOMAIN, BIB_LEVEL_ERROR,   
		       bibtex_message_handler, NULL);
    g_log_set_handler (G_LOG_DOMAIN, BIB_LEVEL_WARNING, 
		       bibtex_message_handler, NULL);
    g_log_set_handler (G_LOG_DOMAIN, BIB_LEVEL_MESSAGE, 
		       bibtex_message_handler, NULL);
}


static void
free_data (gpointer key, 
	   gpointer value, 
	   gpointer user) {

    g_free (key);
    bibtex_struct_destroy ((BibtexStruct *) value, TRUE);
}

BibtexFile * 
bibtex_open (gchar * file) {
    BibtexFile * bf;

    bf = g_new (BibtexFile, 1);
    g_return_val_if_fail (bf != NULL, NULL);

    bf->file = fopen (file, "r");

    if (bf->file == NULL) {
	g_free (bf);
	bibtex_error ("%s: %s", file, g_strerror (errno));

	return NULL;
    }

    bf->table = g_hash_table_new (g_str_hash, g_str_equal);
    bf->debug = 0;
    bf->buffer  = NULL;
    bf->filename = g_strdup (file);

    /* Commencer le nouveau fichier */
    bibtex_first_entry (bf);

    return bf;
}

void 
bibtex_close (BibtexFile * bf) {
    g_return_if_fail (bf != NULL);

    g_hash_table_foreach (bf->table, free_data, NULL);
    g_hash_table_destroy (bf->table);
    g_free (bf->filename);

    /* Deallocate parser */
    bibtex_analyzer_finish (bf);

    fclose (bf->file);

    g_free (bf);
}

static void 
add_to_dico (gpointer key, gpointer value, gpointer user) {
    gchar * val;
    BibtexField * field;

    if ((val = (gchar *) g_hash_table_lookup ((GHashTable *) user,
					      value)) == NULL) {
	val = g_strdup ((char *) key);
    }
    
    field = (BibtexField *) value;

    g_hash_table_insert ((GHashTable *) user, val, field->structure);
}


BibtexEntry * 
bibtex_next_entry (BibtexFile * file) {
    gchar buf [1024];
    int pos, i;
    BibtexEntry * ent;
    char * ret;
    GHashTable * tmptable;

    int offset;

    g_return_val_if_fail (file != NULL, NULL);

    if (file->eof) return NULL;

    offset = file->offset;
    file->error = FALSE;

    do {
	ent = parse_bibtex (file);

	if (ent) {
	    /* Incrementer les numeros de ligne */
	    file->line += ent->length;
	    
	    ent->offset = offset;
	    ent->length = file->offset - offset;
	    
	    /* Rajouter les definitions au dictionnaire, si necessaire */
	    if (ent->type) {
		if (strcasecmp (ent->type, "string") == 0) {

		    g_hash_table_foreach (ent->table, add_to_dico, 
					  file->table);
		    
		    /* Return nothing, we store it as string database */
		    
		    bibtex_entry_destroy (ent, FALSE);
		    ent = NULL;
		}
		else {
		    if (strcasecmp (ent->type, "preamble") == 0) {
			bibtex_warning ("%d: skipping preamble", file->line);
			bibtex_entry_destroy (ent, TRUE);
			
			ent = NULL;
		    }
		    else {
			/* normal case; convert preamble into entry name */
			if (ent->preamble) {
			    if (ent->preamble->type == BIBTEX_STRUCT_REF) {
				ent->name = 
				    g_strdup (ent->preamble->value.ref);
			    }
			    else {
				bibtex_error ("%s:%d: entry has weird name", 
					      file->filename, file->line);
			    }
			}
			else {
			    bibtex_error ("%s:%d: entry has no identifier", 
					  file->filename,
					  file->line);
			    
			    bibtex_entry_destroy (ent, TRUE);
			    file->error = TRUE;
			    
			    return NULL;
			}
		    }
		}
	    }
	}
	else {
	    /* No ent, there has been a problem */
	    return NULL;
	}
    } 
    while (!ent);

    return ent;
}


void 
bibtex_first_entry (BibtexFile * file) {
    g_return_if_fail (file != NULL);

    rewind (file->file);
    bibtex_analyzer_finish (file);
    bibtex_analyzer_initialize (file);

    file->offset = 0;
    file->line = 1;
    file->eof = file->error = FALSE;
}

gint 
bibtex_current_offset (BibtexFile * file) {
    return file->offset;
}
    
void
bibtex_offset_entry (BibtexFile * file, gint offset) {
    g_return_if_fail (file != NULL);

    if (fseek (file->file, offset, SEEK_SET) == -1) {
	bibtex_error ("%s: can't jump to offset %d: %s", 
		      file->filename,
		      offset, g_strerror (errno));
	file->error = TRUE;
	return;
    }

    bibtex_analyzer_finish (file);
    bibtex_analyzer_initialize (file);

    file->offset = offset;
    file->eof = file->error = FALSE;
}


