# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: Fields.py,v 1.5 1999/07/30 12:32:12 gobry Exp $

import string

# Available field types
TypeText   = 0
TypeAuthor = 1
TypeTitle  = 2
TypeDate   = 3

class Description:
    """ Generic Holder for information related to Field/Entry descriptions """
    
    def __init__ (self, name):
        self.name  = name
        self.__items__ = {}
        
    def __getattr__ (self, attr):
        raise AttributeError, "no attribute `%s' in Description" % (attr)

    def __setattr__ (self, attr, value):
        self.__dict__ [attr] = value
        return
    
    def __getitem__ (self, item):
        return self.__items__ [item]

    def __setitem__ (self, item, value):
        self.__items__ [item] = value
        return
    
    def has_key (self, field):
        return self.__items__.has_key (field)

    
class FieldDescription (Description):
    """ Available informations for a given field type """
    
    def __init__ (self, name, type = TypeText):
        
        Description.__init__ (self, name)
        self.type = type
        
    
class EntryDescription (Description):
    """ Informations on a given entry type """
    
    def __init__ (self, name):
        Description.__init__ (self, name)
        
        self.__dict__ ['mandatory'] = []
        self.__dict__ ['optional']  = []
        self.__dict__ ['lcfields']  = {}
        
    def __getattr__ (self, attr):
        if attr == 'fields':
            return self.mandatory + self.optional
        
        return Description.__getattr__ (self, attr)

    def __setattr__ (self, attr, value):

        if attr == 'mandatory' or attr == 'optional':
            self.__dict__ [attr] = value
            self.__dict__ ['lcfields'] = {}
            
            for f in self.fields:
                self.__dict__ ['lcfields'] [string.lower (f.name)] = f

        else:
            Description.__setattr__ (self, attr, value)

    def __getitem__ (self, item):
        return self.__dict__ ['lcfields'] [string.lower (item)]

    def __setitem__ (self, item, value):
        self.__dict__ ['lcfields'] [string.lower (item)] = value

    def has_key (self, field):
        return self.__dict__ ['lcfields'].has_key (string.lower (field))

        
class Author:
    """ Fine description of an author """
    
    def __init__ (self, copy = None):
        """ Initialize an author from a string """

        if copy:
            self.honorific = copy [0]
            self.first = copy [1]
            self.last = copy [2]
            self.lineage = copy [3]
        else:
            self.honorific = None
            self.first = None
            self.last = None
            self.lineage = None

        self.text = None
        
    def __str__ (self):
        """ Returns textual representation """
        
        if not self.text:
            text = ""
            if self.honorific: text = text + " " + self.honorific
            if self.first:     text = text + " " + self.first
            if self.last:      text = text + " " + self.last
            if self.lineage:   text = text + " " + self.lineage
            self.text = text [1:]
            
        return self.text

    def __repr__ (self):
        return str (self)


    def match (self, regex):
        """ """
        return regex.search (str (self))


    def initials (self):
        """ Extract initials from a first name """

        total = []
        
        for atom in string.split (self.first, ' '):
            list = []
            
            for word in string.split (atom, '-'):
                list.append (word [0] + '.')
                
            total.append (string.join (list, '-'))
            
        return string.join (total, ' ')

    def __cmp__ (self, other):
        """ field comparison """
        
        r = cmp (string.lower (self.last), string.lower (other.last))
        if r != 0: return r

        r = cmp (self.first, other.first)
        if r != 0: return r

        return 0


class AuthorGroup:
    """ A group of Authors """

    def __init__ (self):
        self.authors = []

    def __getitem__ (self, pos):
        return self.authors [pos]

    def __setitem__ (self, pos, val):
        self.authors [pos] = val
        return self.authors [pos]

    def __len__ (self):
        return len (self.authors)
    
    def append (self, value):
        self.authors.append (value)
        
    def __str__ (self):
        return string.join (map (str, self.authors), ", ")

    def __repr__ (self):
        return `self.authors`

    def match (self, regex):
        return regex.search (string.join (map (str, self.authors), " "))

    def __cmp__ (self, other):
        i = 0
        s = len (self), len (other)
        m = max (s)
        
        while i < m:
            if i >= s [0]: return -1
            if i >= s [1]: return +1

            r = cmp (self [i], other [i])
            if r != 0: return r

            i = i + 1

        return 0
            
RangeError = "RangeError"

class Date:
    """ Fine description of a date """

    def __init__ (self, year = None, month = None, day = None):
        
        if year and year < 0:
            raise RangeError, "wrong year"
        self.year = year
        
        if month and (month < 1 or month > 12):
            raise RangeError, "wrong month"
        self.month = month
        
        if day and (day < 1 or day > 31):
            raise RangeError, "wrong day"
        self.day = day

        self.text = None
        
    def __cmp__ (self, other):
        # No date is no date !
        if not self.year and not other.year: return 0

        diff = (self.year or other.year) - (other.year or self.year)
        if diff: return diff

        # Same year
        if not self.month and not other.month: return 0

        diff = (self.month or other.month) - (other.month or self.month)
        if diff: return diff
        
        # Same month
        if not self.day and not other.day: return 0

        return (self.day or other.day) - (other.day or self.day)

    def __str__ (self):
        """ Returns textual representation """
        
        if self.year and self.month and self.day:
            self.text = "%d/%d/%d" % (self.day, self.month, self.year)

        if self.year and self.month:
            self.text = "%d/%d" % (self.month, self.year)
            
        self.text = str (self.year)

        return self.text


    def __repr__ (self):
        return str (self)


    def match (self, regex):
        """ """
        return regex.search (str (self))
        

class Text:
    def __init__ (self, text):
        self.text = text


    def __str__ (self):
        return str (self.text)


    def __repr (self):
        return `self.text`


    def match (self, regex):
        """   """
        return regex.search (self.text)


    def __cmp__ (self, other):
        try:
            return cmp (string.lower (self.text), string.lower (other.text))
        except AttributeError:
            print `self`
            print `other`
            raise AttributeError
