/*
 This file is part of pybliographer
 
 Copyright (C) 1998 Frederic GOBRY
 Email : gobry@idiap.ch
 	   
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2 
 of the License, or (at your option) any later version.
   
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details. 
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 
 $Id: bibtex.c,v 1.4 1998/10/28 13:29:12 gobry Exp $
*/

#include "bibtex.h"

static void free_data (gpointer key, gpointer value, gpointer user)
{
  g_free (key);
  g_free (value);
}

bibtex_file * 
bibtex_open (gchar * file)
{
  bibtex_file * bf = g_new (bibtex_file, 1);

  g_return_val_if_fail (bf != NULL, NULL);

  bf->file = fopen (file, "r");
  if (bf->file == NULL) {
    g_free (bf);
    return NULL;
  }

  bf->table = g_hash_table_new (g_str_hash, g_str_equal);
  bf->debug = 0;
  bf->buffer = NULL;
  
  /* Commencer le nouveau fichier */
  bibtex_first_entry (bf);

  return bf;
}

void 
bibtex_close (bibtex_file * bf)
{
  g_return_if_fail (bf != NULL);

  g_hash_table_foreach (bf->table, free_data, NULL);
  g_hash_table_destroy (bf->table);

  /* Deallocate parser */
  bibtex_analyzer_finish (bf);

  fclose (bf->file);
  g_free (bf);
}

static void add_to_dico (gpointer key, gpointer value, gpointer user)
{
  gchar * val = g_strdup ((char *) key);
  g_strdown (val);

  g_hash_table_insert ((GHashTable *) user, val, g_strdup ((char *) value));
}

bibtex_entry * 
bibtex_next_entry (bibtex_file * file)
{
  gchar buf [1024];
  int pos, i;
  bibtex_entry * ent;
  char * ret;

  int offset;

  g_return_val_if_fail (file != NULL, NULL);

  if (file->error || file->eof) return NULL;

  offset = file->offset;

  ent = parse_bibtex (file);

  if (ent) {
    /* Incrementer les numeros de ligne */
    file->line += ent->length;

    ent->offset = offset;
    ent->length = file->offset - offset;

    /* Lire le corps */
    pos = ftell (file->file);
    fseek (file->file, ent->offset, SEEK_SET);
    ent->body = g_new (char, ent->length + 1);
    fread (ent->body, 1, ent->length, file->file);
    ent->body [ent->length] = '\0';

    /* Se remettre a l'ancien endroit */
    fseek (file->file, pos, SEEK_SET);

    /* Rajouter les definitions au dictionnaire, si necessaire */
    if (ent->type)
      if (strcasecmp (ent->type, "string") == 0) {
	g_hash_table_foreach (ent->table, add_to_dico, file->table);
      }
      else {
	if (ent->name == NULL && strcasecmp (ent->type, "preamble") != 0) {
	  fprintf (stderr, "error: %d: entry has no identifier\n", file->line);
	  bibtex_free_entry (ent);
	  file->error = TRUE;
	  return NULL;
	}
      }
    
    
    return ent;
  }
  
  file->error = TRUE;
  return NULL;
}


void 
bibtex_first_entry (bibtex_file * file)
{
  g_return_if_fail (file != NULL);

  rewind (file->file);
  bibtex_analyzer_finish (file);
  bibtex_analyzer_initialize (file);

  file->line = file->offset = 0;
  file->eof = file->error = FALSE;
}

void
bibtex_offset_entry (bibtex_file * file, gint offset)
{
  g_return_if_fail (file != NULL);

  if (fseek (file->file, offset, SEEK_SET) == -1) {
    fprintf (stderr, "error: can't jump to offset %d\n", offset);
    file->error = 1;
    return;
  }

  bibtex_analyzer_finish (file);
  bibtex_analyzer_initialize (file);

  file->offset = offset;
  file->eof = file->error = FALSE;
}

void 
bibtex_free_entry (bibtex_entry * entry)
{
  if (entry->type)
    g_free (entry->type);

  if (entry->name)
    g_free (entry->name);

  if (entry->body) 
    g_free (entry->body);

  g_hash_table_foreach (entry->table, free_data, NULL);
    
  g_hash_table_destroy (entry->table);
  g_free (entry);
}
