///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "pagent.h"
#include "group.h"
#include <util/stringutil.h>
#include <sigc++/sigc++.h>

Error::InvalidPageNum::InvalidPageNum(int num) 
  : logic_error("Invalid page number: " + tostr(num))
{}

Pagent::Pagent(Group *_parent, const std::string& _name) :
  parent(_parent), resizable(false),
  locked(false), flow_around(false), name(_name),
  obstacle_margin(0), width(1), height(1)
{
  // a geometry change implies a properties change
  geometry_changed_signal.connect(props_changed_signal.slot());
  // a geometry change does NOT imply a redraw

  connect_to_parent();
}


Pagent::~Pagent() {}

void Pagent::set_translation(const Vector& v) {
  Matrix m = get_matrix(); m.set_tr(v);
  set_matrix(m);
}

void Pagent::set_rotation(float angle) {
  Matrix m = get_matrix(); m.set_rot(angle);
  set_matrix(m);
}

void Pagent::set_scaling(float xfactor, float yfactor) {
  Matrix m = get_matrix(); m.set_scale(xfactor, yfactor);
  set_matrix(m);
}

void Pagent::set_size(float w, float h) {
//   if(!is_resizable())
//     return;
  if(w != width || h != height) {
    width = w; height = h;
    geometry_changed_signal(this);
  }
}

void Pagent::set_lock(bool _locked) {
  if(_locked != locked) {
    locked = _locked;
    props_changed_signal(this);
    ready_to_draw_signal(this);
  }
}

void Pagent::set_flow_around(bool _flow_around) {
  if(flow_around != _flow_around) {
    flow_around = _flow_around;
    props_changed_signal(this);
  }
}

void Pagent::set_obstacle_margin(float margin) {
  if(margin != obstacle_margin) {
    obstacle_margin = margin;
    props_changed_signal(this);
  }
}

void Pagent::set_name(const std::string &_name) {
  if(_name != name) {
    name = _name;
    props_changed_signal(this);
  }
}

int Pagent::get_page_num() const {
  if(parent)
    return parent->get_page_num();
  else
    throw Error::NoParent();
}

void Pagent::set_parent(Group *_parent) {
  if(parent) {
    draw_connection.disconnect();
    geometry_connection.disconnect();
    props_connection.disconnect();
  }
  parent = _parent;
  connect_to_parent();
  props_changed_signal(this);
}

void Pagent::set_matrix(Matrix m) {
  matrix = m;
  geometry_changed_signal(this);
  ready_to_draw_signal(this);
}

Group& Pagent::get_parent() {
  if(!parent) 
    throw Error::NoParent(); 
  else
    return *parent;
}

const Group& Pagent::get_parent() const {
  if(!parent) 
    throw Error::NoParent(); 
  else
    return *parent;
}

void Pagent::connect_to_parent() {
  // Anything that happens to a pagent, happens to its parent.
  // It is simpler to put this in pagent than in group, because
  // a group would need to maintain a list of connections.
  // Note: we will get multible implicit signals.
  if(parent) {
    using SigC::slot;
      draw_connection = ready_to_draw_signal.connect
	(slot(*parent, &Group::child_ready_to_draw));

      geometry_connection = geometry_changed_signal.connect
	(slot(*parent, &Group::child_geometry_changed));

      props_connection = props_changed_signal.connect
	(slot(*parent, &Group::child_props_changed));
    }
}
