///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///

#include <gtkmm/main.h>

#include <libxml/parser.h> //xmlInitParser

#include "fonts/fontinfo.hh"
#include "util/warning.h"
#include "util/tempfile.h"
#include "widget/errordialog.h"
#include "defines.h" // VERSION

#include "window.h"
#include "config.h"
#include "icons/icon_48.h"

int main(int argc, char *argv[]) {
  ErrorDialog::init_handler();
  int win_count = 0;

  // set prefix for temp files
  Tempfile::set_prefix("pptout");

  // initialize threads
  Glib::thread_init();

#if XMLPP_MAJOR_VERSION == 0
#if XMLPP_MINOR_VERSION < 26
  //Thread-safeness for libxml
  xmlInitParser();
#endif
#endif

  // initialise gtkmm
  Gtk::Main kit(argc, argv);

  // application icon
  std::vector<Glib::RefPtr<Gdk::Pixbuf> > icon_list;
  icon_list.push_back(Gdk::Pixbuf::create_from_inline(9240, icon_48));
  Gtk::Window::set_default_icon_list(icon_list);

  // parse command line
  for(int i = 1; i < argc; i++) {
    std::string arg(argv[i]);
    if(arg == "-v") { // print version number
      warning << "Passepartout " << std::string(VERSION) << std::endl;
      return 0;
    } else if(arg == "-V") { // be verbose
      verbose.active = true;
    } else if(arg == "-d") { // be annoyingly verbose
      verbose.active = true;
      debug.active = true;
      debug << "Debug mode active" << std::endl; // Why wait?
    } else if(arg == "-h" 
	      || arg == "--help") { // display command line help
      std::cout << "Usage: pptout [options] [files...]" << std::endl;
      std::cout << "\t-d\tShow debugging messages." << std::endl;
      std::cout << "\t-h, --help\tShow this help message." << std::endl;
      std::cout << "\t-V\tBe verbose." << std::endl;
      std::cout << "\t-v\tShow version number." << std::endl;
      return 0;
    } else { // open file
      try {
	config.read(); // after the cerr aliases are set
	new FrameWindow(arg);
	win_count++;
      } catch(const std::exception& err) {
	warning << "Error opening " << arg
		<< ": " << err.what() << std::endl;
      }
    }
  }

  if(!win_count) { // open empty window
    config.read(); // after the cerr aliases are set
    new FrameWindow();
    win_count++;
  }

  // initialize font paths
  font::FontInfo::FontPaths font_paths;
  for(Config::Strings::const_iterator i = config.FontPath.values.begin();
      i != config.FontPath.values.end();
      i++) {
    font_paths.push_back(*i);
  }
  font::FontInfo::init(&font_paths);
  
  // if no windows were opened, don't run the kit 
  if(win_count) {
    gdk_rgb_init();		// Needed before rendering pixbuf to screen.
    Gtk::Main::run();
  }

  return 0;
}





