// -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*-

/*
 *  PaperBox - browser.cc
 *
 *  Copyright (C) 2007 Marko Anastasov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <algorithm>
#include <iostream>
#include <boost/bind.hpp>
#include <glibmm/fileutils.h>
#include <glibmm/main.h>
#include <glibmm-utils/log-stream-utils.h>
#include "browser.hh"

namespace paperbox {

    using std::auto_ptr;
    using std::vector;
    using boost::shared_ptr;

    auto_ptr<Browser> Browser::instance_;

    Browser*
    Browser::instance()
    {
        if (! instance_.get())
            instance_.reset(new Browser());

        return instance_.get();
    }

    Browser::Browser()
    {
        tracker_client_.reset(new TrackerPhone()); // let exception propagate

        tracker_client_->get_all_document_uris(uri_queue_);

        Glib::signal_idle().connect(
            sigc::mem_fun(*this, &Browser::on_idle_initial_document_retrieval));
    }

    Browser::~Browser()
    {
        std::cout << "browser in destruction" << std::endl;
    }

    void
    Browser::dump_document_data()
    {
        std::for_each(docs_.begin(),
                      docs_.end(),
                      boost::bind(
                          &Document::print,
                          boost::bind(&doc_map::value_type::second, _1)));
    }

    bool
    Browser::on_idle_initial_document_retrieval()
    {
        LOG_FUNCTION_SCOPE_NORMAL_DD;

        Glib::ustring uri = uri_queue_.front();

        // get metadata, emit for new documents
        try {
            shared_ptr<Document> doc;
            tracker_client_->get_document(uri, doc);

            docs_[uri.raw()] = doc;
            signal_new_document_.emit(doc);
        }
        catch (const Glib::FileError& ex) {
            LOG_EXCEPTION(uri << ": " << ex.what());
        }

        uri_queue_.pop();

        if (uri_queue_.empty())
            return false; // done
        else
            return true;
    }

    Browser::SignalNewDocument&
    Browser::signal_new_document()
    {
        return signal_new_document_;
    }

    Browser::SignalTagsChanged&
    Browser::signal_tags_changed()
    {
        return signal_tags_changed_;
    }

    // Does the validation work and forwards the add request to TrackerPhone
    void
    Browser::add_tags(const Glib::ustring& uri,
                      const std::vector<Glib::ustring>& tags)
    {
        using Glib::ustring;

        doc_map::iterator doc_iter = docs_.find(uri.raw());

        if (doc_iter == docs_.end() || tags.empty()) return;

        // only include the tags that do not intersect with the existing
        shared_ptr<Document> doc = doc_iter->second;
        vector<ustring> existing_tags = doc->get_tags();
        vector<ustring> tags_copy(tags);

        vector<ustring>::iterator it(existing_tags.begin());
        vector<ustring>::iterator end(existing_tags.end());
        for ( ; it != end; ++it) {
            tags_copy.erase(
                std::remove(tags_copy.begin(), tags_copy.end(), *it),
                tags_copy.end());
        }

        if (tags_copy.empty()) return;

        try {
            tracker_client_->add_tags(uri, tags_copy);

            vector<ustring>::iterator it(tags_copy.begin());
            vector<ustring>::iterator end(tags_copy.end());
            for ( ; it != end; ++it)
                doc->add_tag(*it);

            vector<ustring> tags_removed; // none
            signal_tags_changed_.emit(uri, tags_copy, tags_removed);
        }
        catch (const std::runtime_error& ex) {
            LOG_EXCEPTION("Could not add tags for " << uri
                          << ": " << ex.what());
        }
    }

    // Does the validation work and forwards the remove request to TrackerPhone
    void
    Browser::remove_tags(const Glib::ustring& uri,
                         const std::vector<Glib::ustring>& tags)
    {
        using Glib::ustring;

        doc_map::iterator doc_iter = docs_.find(uri.raw());

        if (doc_iter == docs_.end() || tags.empty()) return;

        // only include the tags that intersect with the existing
        shared_ptr<Document> doc = doc_iter->second;
        vector<ustring> existing_tags = doc->get_tags();
        vector<ustring> tags_copy(tags);

        vector<ustring>::iterator it(tags_copy.begin());
        vector<ustring>::iterator end(tags_copy.end());
        for ( ; it != end; ++it) {
            if (find(existing_tags.begin(), existing_tags.end(), *it) ==
                existing_tags.end()) {
                // invalid, not found
                tags_copy.erase(
                    std::remove(tags_copy.begin(), tags_copy.end(), *it),
                    tags_copy.end());
            }
        }

        if (tags_copy.empty()) return;

        try {
            tracker_client_->remove_tags(uri, tags_copy);

            vector<ustring>::iterator it(tags_copy.begin());
            vector<ustring>::iterator end(tags_copy.end());
            for ( ; it != end; ++it)
                doc->remove_tag(*it);

            vector<ustring> tags_added; // none
            signal_tags_changed_.emit(uri, tags_added, tags_copy);
        }
        catch (const std::runtime_error& ex) {
            LOG_EXCEPTION("Could not remove tags for " << uri
                          << ": " << ex.what());
        }
    }

    void
    Browser::rename_tag(const Glib::ustring& /*uri*/,
                        const Glib::ustring& /*from_tag*/,
                        const Glib::ustring& /*to_tag*/)
    {
        //TODO later as a high-level functionality, in UI it should
        // not be strictly bound to one document. It should rather be
        // done in some kind of a manage-tags dialog.
        g_debug("rename_tag not implemented yet");
    }

    void
    Browser::get_all_documents(vector<shared_ptr<Document> >& docs)
    {
        doc_map::iterator it(docs_.begin());
        doc_map::iterator end(docs_.end());

        for ( ; it != end; ++it)
            docs.push_back(it->second);
    }

    void
    Browser::get_documents_for_tag(const Glib::ustring& tag,
                                   vector<shared_ptr<Document> >& docs_ret)
    {
        doc_map::iterator it(docs_.begin());
        doc_map::iterator end(docs_.end());

        for ( ; it != end; ++it) {
            shared_ptr<Document> doc = it->second;
            if (doc->contains_tag(tag))
                docs_ret.push_back(doc);
        }
    }

} // namespace paperbox
