/*
 * Copyright (C) 2004, Red Hat, Inc.
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <pango/pango-layout.h>
#include <pango/pango-lines.h>
#include <pango/pango-glyph.h>

G_BEGIN_DECLS

#define PANGO_TYPE_RENDERER            (pango_renderer_get_type())
#define PANGO_RENDERER(object)         (G_TYPE_CHECK_INSTANCE_CAST ((object), PANGO_TYPE_RENDERER, PangoRenderer))
#define PANGO_IS_RENDERER(object)      (G_TYPE_CHECK_INSTANCE_TYPE ((object), PANGO_TYPE_RENDERER))
#define PANGO_RENDERER_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), PANGO_TYPE_RENDERER, PangoRendererClass))
#define PANGO_IS_RENDERER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), PANGO_TYPE_RENDERER))
#define PANGO_RENDERER_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), PANGO_TYPE_RENDERER, PangoRendererClass))

typedef struct _PangoRenderer        PangoRenderer;
typedef struct _PangoRendererClass   PangoRendererClass;

/**
 * PangoRenderPart:
 * @PANGO_RENDER_PART_FOREGROUND: the text itself
 * @PANGO_RENDER_PART_BACKGROUND: the area behind the text
 * @PANGO_RENDER_PART_UNDERLINE: underlines
 * @PANGO_RENDER_PART_STRIKETHROUGH: strikethrough lines
 * @PANGO_RENDER_PART_OVERLINE: overlines
 *
 * `PangoRenderPart` defines different items to render for such
 * purposes as setting colors.
 */
/* When extending, note N_RENDER_PARTS #define in pango-renderer.c */
typedef enum
{
  PANGO_RENDER_PART_FOREGROUND,
  PANGO_RENDER_PART_BACKGROUND,
  PANGO_RENDER_PART_UNDERLINE,
  PANGO_RENDER_PART_STRIKETHROUGH,
  PANGO_RENDER_PART_OVERLINE
} PangoRenderPart;

/**
 * PangoRenderer:
 *
 * `PangoRenderer` is a base class for objects that can render text
 * provided as `PangoGlyphString` or `PangoLayout`.
 *
 * By subclassing `PangoRenderer` and overriding operations such as
 * @draw_glyphs and @draw_rectangle, renderers for particular font
 * backends and destinations can be created.
 */
struct _PangoRenderer
{
  GObject parent_instance;
};

/**
 * PangoRendererClass:
 * @draw_glyphs: draws a `PangoGlyphString`
 * @draw_rectangle: draws a rectangle
 * @draw_error_underline: draws a squiggly line that approximately
 * covers the given rectangle in the style of an underline used to
 * indicate a spelling error.
 * @draw_trapezoid: draws a trapezoidal filled area
 * @draw_glyph: draws a single glyph
 * @part_changed: do renderer specific processing when rendering
 *  attributes change
 * @begin: Do renderer-specific initialization before drawing
 * @end: Do renderer-specific cleanup after drawing
 * @prepare_run: updates the renderer for a new run
 * @draw_glyph_item: draws a `PangoGlyphItem`
 *
 * Class structure for `PangoRenderer`.
 *
 * The following vfuncs take user space coordinates in Pango units
 * and have default implementations:
 * - draw_glyphs
 * - draw_rectangle
 * - draw_error_underline
 * - draw_shape
 * - draw_glyph_item
 *
 * The default draw_shape implementation draws nothing.
 *
 * The following vfuncs take device space coordinates as doubles
 * and must be implemented:
 * - draw_trapezoid
 * - draw_glyph
 */
struct _PangoRendererClass
{
  /*< private >*/
  GObjectClass parent_class;

  /*< public >*/

  void (*draw_glyphs)          (PangoRenderer    *renderer,
                                PangoFont        *font,
                                PangoGlyphString *glyphs,
                                int               x,
                                int               y);
  void (*draw_rectangle)       (PangoRenderer    *renderer,
                                PangoRenderPart   part,
                                int               x,
                                int               y,
                                int               width,
                                int               height);
  void (*draw_error_underline) (PangoRenderer    *renderer,
                                int               x,
                                int               y,
                                int               width,
                                int               height);
  void (*draw_shape)           (PangoRenderer    *renderer,
                                PangoRectangle   *ink_rect,
                                PangoRectangle   *logical_rect,
                                gpointer          data,
                                int               x,
                                int               y);
  void (*draw_trapezoid)       (PangoRenderer    *renderer,
                                PangoRenderPart   part,
                                double            y1_,
                                double            x11,
                                double            x21,
                                double            y2,
                                double            x12,
                                double            x22);
  void (*draw_glyph)           (PangoRenderer    *renderer,
                                PangoFont        *font,
                                PangoGlyph        glyph,
                                double            x,
                                double            y);

  void (*part_changed)         (PangoRenderer    *renderer,
                                PangoRenderPart   part);

  void (*begin)                (PangoRenderer    *renderer);
  void (*end)                  (PangoRenderer    *renderer);

  void (*prepare_run)          (PangoRenderer    *renderer,
                                PangoRun         *run);

  void (*draw_run)             (PangoRenderer    *renderer,
                                const char       *text,
                                PangoRun         *run,
                                int               x,
                                int               y);

  /*< private >*/

  /* Padding for future expansion */
  gpointer _pango_reserved[8];
};

PANGO_AVAILABLE_IN_ALL
GType pango_renderer_get_type            (void) G_GNUC_CONST;

PANGO_AVAILABLE_IN_ALL
void pango_renderer_draw_lines           (PangoRenderer    *renderer,
                                          PangoLines       *lines,
                                          int               x,
                                          int               y);

PANGO_AVAILABLE_IN_ALL
void pango_renderer_draw_line            (PangoRenderer    *renderer,
                                          PangoLine        *line,
                                          int               x,
                                          int               y);
PANGO_AVAILABLE_IN_ALL
void pango_renderer_draw_glyphs          (PangoRenderer    *renderer,
                                          PangoFont        *font,
                                          PangoGlyphString *glyphs,
                                          int               x,
                                          int               y);
PANGO_AVAILABLE_IN_ALL
void pango_renderer_draw_run             (PangoRenderer    *renderer,
                                          const char       *text,
                                          PangoRun         *run,
                                          int               x,
                                          int               y);
PANGO_AVAILABLE_IN_ALL
void pango_renderer_draw_rectangle       (PangoRenderer    *renderer,
                                          PangoRenderPart   part,
                                          int               x,
                                          int               y,
                                          int               width,
                                          int               height);
PANGO_AVAILABLE_IN_ALL
void pango_renderer_draw_error_underline (PangoRenderer    *renderer,
                                          int               x,
                                          int               y,
                                          int               width,
                                          int               height);
PANGO_AVAILABLE_IN_ALL
void pango_renderer_draw_trapezoid       (PangoRenderer    *renderer,
                                          PangoRenderPart   part,
                                          double            y1_,
                                          double            x11,
                                          double            x21,
                                          double            y2,
                                          double            x12,
                                          double            x22);
PANGO_AVAILABLE_IN_ALL
void pango_renderer_draw_glyph           (PangoRenderer    *renderer,
                                          PangoFont        *font,
                                          PangoGlyph        glyph,
                                          double            x,
                                          double            y);

PANGO_AVAILABLE_IN_ALL
void pango_renderer_activate             (PangoRenderer    *renderer);
PANGO_AVAILABLE_IN_ALL
void pango_renderer_deactivate           (PangoRenderer    *renderer);

PANGO_AVAILABLE_IN_ALL
void pango_renderer_part_changed         (PangoRenderer   *renderer,
                                          PangoRenderPart  part);

PANGO_AVAILABLE_IN_ALL
void        pango_renderer_set_color     (PangoRenderer    *renderer,
                                          PangoRenderPart   part,
                                          const PangoColor *color);
PANGO_AVAILABLE_IN_ALL
PangoColor *pango_renderer_get_color     (PangoRenderer    *renderer,
                                          PangoRenderPart   part);

PANGO_AVAILABLE_IN_ALL
void               pango_renderer_set_matrix      (PangoRenderer     *renderer,
                                                   const PangoMatrix *matrix);
PANGO_AVAILABLE_IN_ALL
const PangoMatrix *pango_renderer_get_matrix      (PangoRenderer     *renderer);

PANGO_AVAILABLE_IN_ALL
PangoLines *       pango_renderer_get_lines       (PangoRenderer     *renderer);

PANGO_AVAILABLE_IN_ALL
PangoLine *        pango_renderer_get_layout_line (PangoRenderer     *renderer);

PANGO_AVAILABLE_IN_ALL
PangoContext *     pango_renderer_get_context     (PangoRenderer     *renderer);

G_END_DECLS
