/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include <ctype.h>
#include <sys/time.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/file-grouplist.h>
#include <pan/base/file-headers.h>
#include <pan/base/gnksa.h>
#include <pan/base/log.h>
#include <pan/base/pan-config.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/serverlist.h>
#include <pan/base/status-item.h>

#include <pan/rules/rule-ui.h>

#include <pan/identities/identity.h>
#include <pan/identities/identity-manager.h>
#include <pan/identities/identity-ui.h>

#include <pan/article-actions.h>
#include <pan/article-find.h>
#include <pan/articlelist.h>
#include <pan/article-list-toolbar-mediator.h>
#include <pan/filter-ui.h>
#include <pan/globals.h>
#include <pan/group-action.h>
#include <pan/group-ui.h>
#include <pan/gui.h>
#include <pan/gui-notebook.h>
#include <pan/gui-paned.h>
#include <pan/grouplist.h>
#include <pan/message-send.h>
#include <pan/message-window.h>
#include <pan/newsrc-ui.h>
#include <pan/pan.h>
#include <pan/prefs.h>
#include <pan/print.h>
#include <pan/server-ui.h>
#include <pan/status-item-view.h>
#include <pan/text.h>
#include <pan/task-bodies.h>
#include <pan/task-manager.h>
#include <pan/util.h>
#include <pan/queue.h>
#include <pan/dialogs/dialogs.h>
#include <pan/xpm/caution.xpm>
#include <pan/xpm/inform.xpm>

GtkTooltips * ttips = NULL;
GdkColormap * cmap = NULL;

static unsigned int gui_refresh_timer_id = 0;
static unsigned int gui_queue_timer_id = 0;
static double KBps = 0.0;
static GtkWidget * taskbar = NULL;
static GtkWidget * error_button = NULL;
static GtkWidget * status_ok_label = NULL;
static GtkWidget * status_error_label = NULL;

#define VIEW_QTY 3
typedef struct {
	StatusItem * status_item;
	StatusItemView * view;
} ViewStruct;
static ViewStruct views[VIEW_QTY];

/**
*** private function prototypes
**/

static void exit_cb (void);
static int gui_page_get_nolock (void);
static GtkWidget* gui_page_get_pane_nolock (int page);
static gboolean gui_key_press_cb (GtkWidget*, GdkEventKey*, gpointer);
static int window_delete_event_cb (GtkWidget*, GdkEvent*, gpointer);
static int gui_refresh_timer_cb (gpointer);
static int gui_queue_timer_cb (gpointer);
static void update_menus_cb                      (gpointer, gpointer, gpointer);
static void log_entry_added_cb                   (gpointer, gpointer, gpointer);
static void status_item_active_changed_cb        (gpointer, gpointer, gpointer);
static void log_list_appended_cb                 (gpointer, gpointer, gpointer); 
static void text_fill_body_changed_cb            (gpointer, gpointer, gpointer); 
static void articles_changed_cb                  (gpointer, gpointer, gpointer);
static void turn_off_status_item_views_nolock    (void);
static void gui_set_connection_size (int size);

static void
server_import_newsrc_dialog_menu (void)
{
	Server * server = serverlist_get_active_server ();
	if (server != NULL)
		newsrc_import_dialog (server);
}
static void
server_export_newsrc_dialog_menu (void)
{
	Server * server = serverlist_get_active_server ();
	if (server != NULL)
		newsrc_export_dialog (server);
}

#if 1
#define GNOMEUIINFO_ITEM_STOCK_ACCEL(label, tooltip, callback, stock_id, accel, modifier) \
        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
		GNOME_APP_PIXMAP_STOCK, stock_id, accel, (GdkModifierType)modifier,  NULL }
#else
#define GNOMEUIINFO_ITEM_STOCK_ACCEL(label, tooltip, callback, stock_id, accel, modifier) \
        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
		GNOME_APP_PIXMAP_NONE, 0, accel, (GdkModifierType)modifier,  NULL }
#endif

/* adenopstx */
static GnomeUIInfo file_menu[] =
{
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Save _Articles"),
	                              N_("Save the selected articles and their attachments."),
	                              article_action_selected_save,
	                              GNOME_STOCK_MENU_SAVE, 'S', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Save Articles As..."),
	                              N_("Dialog to set how you want to save the selected articles"),
	                              article_action_selected_save_as,
	                              GNOME_STOCK_MENU_SAVE, 'S', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Save Articles (A_ttachments only)"),
	                              N_("Save the selected articles' attachments."),
	                              article_action_selected_save_attachments,
	                              GNOME_STOCK_MENU_SAVE, 'S', GDK_SHIFT_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Open Attachments..."),
	                              N_("Open Attachment..."),
	                              article_action_selected_open_attachments,
	                              GNOME_STOCK_MENU_SAVE, 'O', 0),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("_Print Article..."),
	                        N_("_Print Article..."),
	                        print_cb,
	                        GNOME_STOCK_MENU_PRINT),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("_New Folder..."),
	                        N_("Create a new folder."),
	                        group_new_folder_dialog,
	                        GNOME_STOCK_MENU_MAIL_NEW),
	GNOMEUIINFO_ITEM_STOCK (N_("_Delete Folder..."),
	                        N_("Delete the selected folder."),
	                        group_action_selected_destroy,
	                        GNOME_STOCK_MENU_TRASH),
	GNOMEUIINFO_ITEM_STOCK (N_("_Edit Article in Folder"),
	                        N_("Edit Selected Article in Folder"),
	                        article_action_edit_selected,
	                        GNOME_STOCK_MENU_PROP),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_EXIT_ITEM(exit_cb, NULL),
	GNOMEUIINFO_END
};

/**
***  DIALOGS
**/

static void
widget_destroyed_cb (GtkObject * o, gpointer data)
{
	*((GtkWidget**)data) = NULL;
}

static GtkWidget * identity_dialog = NULL;

static void
identity_cb (void)
{
	if (identity_dialog == NULL)
	{
		identity_dialog = identity_dialog_new ();
		gtk_signal_connect (GTK_OBJECT(identity_dialog), "destroy",
		                    GTK_SIGNAL_FUNC(widget_destroyed_cb),
		                    &identity_dialog);
		gnome_dialog_set_parent (GNOME_DIALOG(identity_dialog),
		                         GTK_WINDOW(Pan.window));
	}

	gtk_widget_show_all (identity_dialog);
}

static GtkWidget * filter_dialog = NULL;

static void
filter_cb (void)
{
	if (filter_dialog == NULL)
	{
		filter_dialog = filter_dialog_new ();
		gtk_signal_connect (GTK_OBJECT(filter_dialog), "destroy",
		                    GTK_SIGNAL_FUNC(widget_destroyed_cb),
		                    &filter_dialog);
		gnome_dialog_set_parent (GNOME_DIALOG(filter_dialog),
		                         GTK_WINDOW(Pan.window));
	}

	gtk_widget_show_all (filter_dialog);
}

static GtkWidget * rule_dialog = NULL;

static void
rules_cb (void)
{
	if (rule_dialog == NULL)
	{
		rule_dialog = rule_dialog_new ();
		gtk_signal_connect (GTK_OBJECT(rule_dialog), "destroy",
		                    GTK_SIGNAL_FUNC(widget_destroyed_cb),
		                    &rule_dialog);
		gnome_dialog_set_parent (GNOME_DIALOG(rule_dialog),
		                         GTK_WINDOW(Pan.window));
	}

	gtk_widget_show_all (rule_dialog);
}

GtkWidget* create_rules_dialog (void);

/* acdefgiloprst */
static GnomeUIInfo edit_menu[] =
{
	GNOMEUIINFO_ITEM_NONE  (N_("Select all _Groups"),
	                        N_("Select all Groups in the Group List"),
	                        grouplist_select_all),
	GNOMEUIINFO_ITEM_NONE  (N_("_Deselect all Groups"),
	                        N_("Deselect all Groups in the Group List"),
	                        grouplist_deselect_all),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_NONE  (N_("Select all Arti_cles"),
	                        N_("Select all Articles in the Article List"),
	                        articlelist_select_all_nolock),
	GNOMEUIINFO_ITEM_NONE  (N_("Add R_eplies to Selection"),
	                        N_("Add replies to the Article List selection"),
	                        articlelist_add_replies_to_selection_nolock),
	GNOMEUIINFO_ITEM_NONE  (N_("Add _Threads to Selection"),
	                        N_("Add Thread to the Article List selection"),
	                        articlelist_add_thread_to_selection_nolock),
	GNOMEUIINFO_ITEM_NONE  (N_("Deselect a_ll Articles"),
	                        N_("Deselect all Articles in the Article List"),
	                        articlelist_deselect_all_nolock),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_NONE  (N_("Select Article _Body"),
	                        N_("Select Article _Body"),
	                        text_select_all),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_FIND_ITEM(articlelist_find_text_cb, NULL),
	GNOMEUIINFO_MENU_FIND_AGAIN_ITEM(articlelist_find_next_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Filter_s..."),
	                              N_("Filter Tool."),
	                              filter_cb,
	                              GNOME_STOCK_MENU_PREF, 'F', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Rules..."),
	                              N_("Rules Tool."),
	                              rules_cb,
	                              GNOME_STOCK_MENU_PREF, 'R', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Pr_ofiles..."),
	                              N_("Profiles Tool."),
	                              identity_cb,
	                              GNOME_STOCK_MENU_PREF, 'O', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK       (N_("_Online/Offline Settings..."),
	                              N_("Online/Offline Settings..."),
	                              prefs_spawn_to_news_connections,
	                              GNOME_STOCK_MENU_PREF),
	GNOMEUIINFO_MENU_PREFERENCES_ITEM(prefs_spawn, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo post_menu [] =
{
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Post to newsgroup"),
	                              N_("Post a new message to the current group."),
	                              message_post_window,
	                              GNOME_STOCK_MENU_MAIL_NEW, 'P', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Follow-Up to newsgroup"),
	                              N_("Post a reply to the message on the news server."),
	                              message_followup_window,
	                              GNOME_STOCK_MENU_MAIL_RPL, 'F', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Reply by E-mail"),
	                              N_("Create a mail reply to the sender."),
	                              message_reply_window,
	                              GNOME_STOCK_MENU_MAIL_RPL, 'R', 0),
	GNOMEUIINFO_ITEM_STOCK       (N_("Follow-Up _and Reply"),
	                              N_("Send a reply both to the author in mail, and to the news server."),
	                              message_followup_reply_window,
	                              GNOME_STOCK_MENU_MAIL_RPL),
	GNOMEUIINFO_ITEM_NONE        (N_("For_ward article by E-mail"),
	                              N_("Forward article by E-mail"),
	                              message_forward_window),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK       (N_("Send Pending _Messages"),
	                              N_("Send Messages Queued in \"pan.sendlater\""),
	                              flush_sendlater_articles,
	                              GNOME_STOCK_MENU_MAIL_SND),
	GNOMEUIINFO_END
};

static int current_layout = -1;

static void
navigate_cached_only_cb (GtkCheckMenuItem* item, gpointer data)
{
	navigate_cached_only = item->active;
}
static void
navigate_read_on_select_cb (GtkCheckMenuItem * item, gpointer data)
{
	navigate_read_on_select = item->active;
}
static void
gui_read_next (void)
{
	articlelist_read_next (navigate_cached_only, navigate_read_on_select);
}
static void
space_reading_dummy_cb (void)
{
	/* we listen for space key keypresses directly, which seems more reliable */
}

/* adgklnoprtuv */
static GnomeUIInfo navigate_menu[] =
{
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("S_kip Uncached Articles:"),
		N_("Skip Uncached Articles:"),
		navigate_cached_only_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("_Load Articles When Selected:"),
		N_("Load Articles When Selected:"),
		navigate_read_on_select_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Space Reading"),
	                              N_("Space Reading"),
	                              space_reading_dummy_cb,
	                              GNOME_STOCK_MENU_FORWARD, GDK_space, 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Next Unread Article"),
	                              N_("Next Unread Article"),
	                              articlelist_read_next_unread,
	                              GNOME_STOCK_MENU_FORWARD, 'N', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Next _Article"),
	                              N_("Next Article"),
	                              gui_read_next,
	                              GNOME_STOCK_MENU_FORWARD, 'N', GDK_SHIFT_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Next Unread _Thread"),
	                              N_("Next Unread Thread"),
	                              articlelist_read_next_unread_thread,
	                              GNOME_STOCK_MENU_FORWARD, 'T', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Next Threa_d"),
	                              N_("Next Thread"),
	                              articlelist_read_next_thread,
	                              GNOME_STOCK_MENU_FORWARD, 'T', GDK_SHIFT_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Pre_vious Article"),
	                              N_("Previous Article"),
	                              articlelist_read_prev,
	                              GNOME_STOCK_MENU_BACK, 'V', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Previo_us Thread"),
	                              N_("Previous Thread"),
	                              articlelist_read_prev_thread,
	                              GNOME_STOCK_MENU_BACK, 'V', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Previous _read article"),	
	                              N_("Previous read article"),
	                              articlelist_read_last_read,
	                              GNOME_STOCK_MENU_BACK, 'V', GDK_SHIFT_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Parent Article"),
	                              N_("Parent Article"),
	                              articlelist_read_parent,
	                              GNOME_STOCK_MENU_UP, 'U', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("T_op of Thread"),
	                              N_("Top of Thread"),
	                              articlelist_read_top_of_thread,
	                              GNOME_STOCK_MENU_UP, 'U', GDK_CONTROL_MASK),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Next Unread _Group"),
	                              N_("Move to the Next Group with Unread Messages."),
	                              grouplist_activate_next_unread_group,
	                              GNOME_STOCK_MENU_FORWARD, 'G', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Next G_roup"),
	                              N_("Move to the Next Group."),
	                              grouplist_activate_next_group,
	                              GNOME_STOCK_MENU_FORWARD, 'G', GDK_SHIFT_MASK),
	GNOMEUIINFO_END
};

/* aein */
static GnomeUIInfo server_menu[] =
{
	/* <-- the server list gets inserted here */
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Get List of _All Groups"),
	                        N_("Refresh a list of groups from the selected servers."),
	                        grouplist_get_all,
	                        GNOME_STOCK_MENU_REFRESH),
	GNOMEUIINFO_ITEM_STOCK (N_("Get List of _New Groups"),
	                        N_("Download a list of new groups from the selected servers."),
	                        grouplist_get_new,
	                        GNOME_STOCK_MENU_REFRESH),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("_Import .newsrc..."),
	                        N_("Import .newsrc..."),
	                        server_import_newsrc_dialog_menu,
	                        GNOME_STOCK_MENU_REDO),
	GNOMEUIINFO_ITEM_STOCK (N_("_Export .newsrc..."),
	                        N_("Export .newsrc..."),
	                        server_export_newsrc_dialog_menu,
	                        GNOME_STOCK_MENU_UNDO),
	GNOMEUIINFO_END
};

static void
pan_homepage_url (void)
{
	pan_url_show ("http://pan.rebelbase.com/");
}

static void
pan_bug_url (void)
{
	pan_url_show ("http://bugzilla.gnome.org/enter_bug.cgi?product=Pan");
}

static void
pan_manual_url (void)
{
	pan_url_show ("http://pan.rebelbase.com/manual/");
}

static void
send_feedback_cb (GtkWidget * w)
{
	char * from;
	char * msg_id;
	Group * sendlater;
	Article * a;
	Identity * id;
	debug_enter ("send_feedback_cb");

	/* who is sending this? */
	from = NULL;
       	id = identity_manager_get_default (ID_MAIL_DEFAULT);
	if (id != NULL)
		from = g_strdup_printf ("\"%s\" <%s>", id->author_real, id->author_addr);

       	msg_id = gnksa_generate_message_id ("rebelbase.com");
       	sendlater = serverlist_get_named_folder (PAN_SENDLATER);
       	a = article_new (sendlater);
	a->number = 1;

	article_init_header (a, HEADER_SUBJECT, "Pan " VERSION " Feedback", DO_CHUNK_SHARE);
	article_init_header (a, PAN_MAIL_TO, "pan@rebelbase.com", DO_CHUNK_SHARE);
	article_init_header (a, HEADER_MESSAGE_ID, msg_id, DO_CHUNK);
	if (is_nonempty_string(from))
		article_init_author_from_header (a, from);
	message_edit_window (a);

	g_free (msg_id);
	g_free (from);
	debug_exit ("send_feedback_cb");
}

static void
dialog_about_cb (void)
{
	dialog_about (Pan.window);
}

/* abfhm */
GnomeUIInfo help_menu[] =
{
	GNOMEUIINFO_ITEM_STOCK (N_("Pan _Homepage..."),
	                        N_("Pan _Homepage"),
	                        pan_homepage_url,
	                        GNOME_STOCK_MENU_HOME),
	GNOMEUIINFO_ITEM_STOCK (N_("Online Users _Manual..."),
	                        N_("Online Users _Manual"),
	                        pan_manual_url,
	                        GNOME_STOCK_MENU_BOOK_YELLOW),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("_Feedback..."),
	                        N_("Mail Feedback to the Pan Authors"),
	                        send_feedback_cb,
	                        GNOME_STOCK_MENU_MAIL),
	GNOMEUIINFO_ITEM_STOCK (N_("Report a _Bug..."),
	                        N_("Submit a Pan Bug Report to the GNOME Bug Tracker"),
	                        pan_bug_url,
	                        GNOME_STOCK_MENU_TRASH_FULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_ABOUT_ITEM(dialog_about_cb, NULL),
	GNOMEUIINFO_END
};


static void
thread_articles_cb (GtkCheckMenuItem* item, gpointer data)
{
	articlelist_set_threaded (item->active);
}

static void
rot_13_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_rot13_selected_text_nolock ();
}

static void
fill_body_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_wrap (item->active);
}

static void
show_all_headers_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_show_all_headers (item->active);
}

static void
mute_quoted_text_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_mute_quoted (item->active);
}

static void
use_fixed_font_cb (GtkCheckMenuItem * item, gpointer data)
{
	text_use_fixed_font = item->active;
	text_set_font ();
}

static void
collapse_group_names_cb (GtkCheckMenuItem * item, gpointer data)
{
	if (collapse_group_names != item->active) {
		collapse_group_names = item->active;
		grouplist_refresh_nolock ();
	}
}

static void
download_bodies_too_toggle_cb (GtkCheckMenuItem * item, gpointer data)
{
	download_bodies_too = item->active;
}

static int
log_button_clicked_idle (gpointer data)
{
	GtkWidget * old_child;
	GtkWidget * new_child = status_ok_label;

	pan_lock ();
	if (((old_child = GTK_BIN(error_button)->child)) != new_child) {
		gtk_container_remove (GTK_CONTAINER(error_button), old_child);
		gtk_container_add (GTK_CONTAINER(error_button), new_child);
		gtk_widget_show (new_child);
	}
	pan_unlock ();

	return 0;
}

static int
log_button_clicked (gpointer menu, gpointer data)
{
	gui_queue_add (log_button_clicked_idle, NULL);

	dialog_log_viewer ();

	return 0;
}

static void
toggle_pane_visibility (GtkCheckMenuItem * item, gpointer data)
{
	if (current_layout == GUI_PANED)
	{
		gboolean * show_pane = NULL;
		const guint page = GPOINTER_TO_UINT(data);

		switch (page) {
			case GROUPS_PAGE: show_pane = &show_group_pane; break;
			case ARTICLELIST_PAGE: show_pane = &show_header_pane; break;
			case MESSAGE_PAGE: show_pane = &show_article_pane; break;
			default: pan_warn_if_reached ();
		}

		if (show_pane!=NULL && *show_pane!=item->active) {
			*show_pane = item->active;
			gui_set_layout (current_layout);
		}
	}
}

static void
zoom_page_cb (gpointer menu, gpointer data)
{
	const guint page = GPOINTER_TO_UINT(data);
       	gui_set_layout (GUI_NOTEBOOK);
	gui_page_set (page);
}

static void
zoom_cb (void)
{
	const int page = gui_page_get_nolock ();
       	gui_set_layout (current_layout==GUI_PANED ? GUI_NOTEBOOK : GUI_PANED);
	gui_page_set (page);
}

static GnomeUIInfo view_menu[] =
{
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Task Manager..."),
	                              N_("Open the Task Manager"),
	                              task_manager_spawn,
	                              GNOME_STOCK_MENU_PROP, 'T', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("Lo_g Viewer..."),
	                              N_("Open the Log Viewer"),
	                              log_button_clicked,
	                              GNOME_STOCK_MENU_PROP, 'L', GDK_CONTROL_MASK),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("View Pa_nes as Tabs"),
	                              N_("Zoom/Unzoom"),
	                              zoom_cb,
	                              GNOME_STOCK_MENU_JUMP_TO, 'Z', 0),
/*4*/	{
		GNOME_APP_UI_ITEM,
		N_("_View Group Tab"),
		N_("View Group Tab"),
		zoom_page_cb, GUINT_TO_POINTER(GROUPS_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO,
		'1', 0, NULL
	},      
/*5*/	{
		GNOME_APP_UI_ITEM,
		N_("View Header Ta_b"),
		N_("View Header Tab"),
		zoom_page_cb, GUINT_TO_POINTER(ARTICLELIST_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO,
		'2', 0, NULL
	},      
/*6*/	{
		GNOME_APP_UI_ITEM,
		N_("View _Article Tab"),
		N_("View Article Tab"),
		zoom_page_cb, GUINT_TO_POINTER(MESSAGE_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO,
		'3', 0, NULL
	},      
/*7*/	GNOMEUIINFO_SEPARATOR,
/*8*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Sho_w Group Pane"),
		N_("Show Group Pane"),
		toggle_pane_visibility, GUINT_TO_POINTER(GROUPS_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'1', GDK_CONTROL_MASK, NULL
	},      
/*9*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Hea_der Pane"),
		N_("Show Header Pane"),
		toggle_pane_visibility, GUINT_TO_POINTER(ARTICLELIST_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'2', GDK_CONTROL_MASK, NULL
	},      
/*10*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Art_icle Pane"),
		N_("Show Article Pane"),
		toggle_pane_visibility, GUINT_TO_POINTER(MESSAGE_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'3', GDK_CONTROL_MASK, NULL
	},      
/*11*/	GNOMEUIINFO_SEPARATOR,
/*12*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Collapse Names in Gro_up Pane"),
		N_("Collapse Names in Group Pane"),
		collapse_group_names_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
/*13*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Thr_ead/Unthread Header Pane"),
		N_("Thread/Unthread Header Pane"),
		thread_articles_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'E', 0, NULL
	},
/*14*/	{
		GNOME_APP_UI_ITEM,
		N_("_Rot13 Selected Text"),
		N_("Rot13 Selected Text"),
		rot_13_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'"', GDK_CONTROL_MASK, NULL
	},
/*15*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Fil_l/Wrap Article Body"),
		N_("Fill/Wrap Article Body"),
		fill_body_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'L', 0, NULL
	},
/*16*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Mute _Quoted Text"),
		N_("Mute Quoted Text"),
		mute_quoted_text_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'Q', 0, NULL
	},
/*17*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show All _Headers in Article Pane"),
		N_("Show All Headers in Article Pane"),
		show_all_headers_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'H', 0, NULL
	},
/*18*/	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Use Monospa_ce Font in Article Pane"),
		N_("Use Monospace Font in Article Pane"),
		use_fixed_font_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'C', 0, NULL
	},
/*19*/	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_NONE (N_("E_xpand Selected Threads"),
	                       N_("Expand Selected Threads"),
	                       articlelist_expand_selected_threads),
	GNOMEUIINFO_ITEM_NONE (N_("Expand all Thread_s"),
	                       N_("Expand all Threads"),
	                       articlelist_expand_all_threads),
	GNOMEUIINFO_ITEM_NONE (N_("C_ollapse Selected Threads"),
	                       N_("Collapse Selected Threads"),
	                       articlelist_collapse_selected_threads),
	GNOMEUIINFO_ITEM_NONE (N_("Colla_pse all Threads"),
	                       N_("Collapse all Threads"),
	                       articlelist_collapse_all_threads),
	GNOMEUIINFO_END
};

static void
mute_quoted_text_changed_cb (gpointer data, gpointer call, gpointer client)
{
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[16].widget),
	                                call!=NULL);
}

static void
show_all_headers_changed_cb (gpointer data, gpointer call, gpointer client)
{
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[17].widget),
	                                call!=NULL);
}


/* bcdflmnpstuw */
GnomeUIInfo group_menu [] =
{
	GNOMEUIINFO_ITEM_STOCK_ACCEL  (N_("_Mark Group as Read"),
	                               N_("Mark All Articles in Selected Groups as Read"),
	                               group_action_selected_mark_read,
	                               GNOME_STOCK_MENU_BLANK, 'M', GDK_CONTROL_MASK|GDK_SHIFT_MASK),
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Get Bodies along _with Articles:"),
		N_("Get Bodies along with Articles"),
		download_bodies_too_toggle_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_ITEM_STOCK_ACCEL  (N_("Get _New Articles"),
	                               N_("Get New Articles"),
	                               group_action_selected_download_new,
	                               GNOME_STOCK_MENU_BLANK, 'A', 0),
	GNOMEUIINFO_ITEM_STOCK_ACCEL  (N_("Get A_ll Articles"),
	                               N_("Get All Articles"),
	                               group_action_download_all,
	                               GNOME_STOCK_MENU_BLANK, 'A', GDK_CONTROL_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL  (N_("Get New Articles in all Su_bscribed Groups"),
	                               N_("Get New Articles in all Subscribed Groups"),
	                               group_action_subscribed_download_new,
	                               GNOME_STOCK_MENU_BOOK_RED, 'A', GDK_SHIFT_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL  (N_("_Download Articles.."),
	                               N_("Specify and Download Articles"),
	                               group_action_selected_download_dialog,
	                               GNOME_STOCK_MENU_BLANK, 'A',
	                               GDK_CONTROL_MASK|GDK_SHIFT_MASK),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Refresh Article _Counts"),
	                        N_("Refresh Article Counts"),
	                        group_action_selected_update_count_info,
	                        GNOME_STOCK_MENU_REFRESH),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Subscribe"),
	                              N_("Subscribe"),
	                              group_action_selected_subscribe,
	                              GNOME_STOCK_MENU_BOOK_RED, 'B', GDK_SHIFT_MASK),
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Unsubscribe"),
	                              N_("Unsubscribe"),
	                              group_action_selected_unsubscribe,
	                              GNOME_STOCK_MENU_BLANK, 'B', GDK_CONTROL_MASK),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK       (N_("_Properties..."),
	                              N_("Set the properties for the selected group."),
	                              group_action_selected_properties,
	                              GNOME_STOCK_MENU_PROP),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK_ACCEL  (N_("_Empty..."),
	                               N_("Remove all Articles from Selected Groups"),
	                               group_action_selected_empty,
	                               GNOME_STOCK_MENU_TRASH, 'D', GDK_CONTROL_MASK|GDK_SHIFT_MASK),
	GNOMEUIINFO_ITEM_STOCK       (N_("Dele_te"),
	                              N_("Delete the selected group(s) from Pan."),
	                              group_action_selected_destroy,
	                              GNOME_STOCK_MENU_TRASH),
	GNOMEUIINFO_END
};



extern GnomeUIInfo articlelist_menu [];

static GnomeUIInfo pan_main_menu[] = {
	GNOMEUIINFO_MENU_FILE_TREE (file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (edit_menu),
	GNOMEUIINFO_SUBTREE (N_("_View"), &view_menu),
	GNOMEUIINFO_SUBTREE (N_("_Servers"), &server_menu),
	GNOMEUIINFO_SUBTREE (N_("_Groups"), &group_menu),
	GNOMEUIINFO_SUBTREE (N_("_Articles"), &articlelist_menu),
	GNOMEUIINFO_SUBTREE (N_("_Navigate"), &navigate_menu),
	GNOMEUIINFO_SUBTREE (N_("_Post"), &post_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};


/** 
 ** global variables
 **/

static GtkWidget * appbar = NULL;
static GtkWidget * queue_qty_label = NULL;
static GtkWidget * queue_qty_button = NULL;
static GtkWidget * connection_qty_label = NULL;
static GtkWidget * connection_qty_button = NULL;

guint context_id;

GtkWidget * groups_vbox;
GtkWidget * articlelist_ctree;
GtkWidget * text_box;
GtkWidget * contents_vbox;

static void
connection_button_clicked (GtkButton *button,
                           gpointer user_data)
{
	prefs_spawn_to_news_connections ();
}

static void
queue_button_clicked (GtkButton *button,
                      gpointer user_data)
{
	task_manager_spawn ();
}


static void
gui_shutdown (void)
{
	int i;
	char * pch;
	GtkWidget * w[3];
	debug_enter ("gui_shutdown");

	/**
	***  Stop updating the UI
	**/

	pan_callback_remove (status_item_get_active_callback(),
	                     status_item_active_changed_cb, NULL);
	turn_off_status_item_views_nolock ();

	/**
	***  Stop updating the rest of the UI
	**/

	pan_callback_remove (article_get_articles_changed_callback(),
	                     articles_changed_cb, NULL);
	pan_callback_remove (log_get_entry_added_callback(),
	                     log_entry_added_cb, NULL);
	pan_callback_remove (log_get_entry_added_callback(),
	                     log_list_appended_cb, NULL);
	pan_callback_remove (text_get_fill_body_changed_callback(),
	                     text_fill_body_changed_cb, NULL);
	pan_callback_remove (text_get_show_all_headers_changed_callback(),
	                     show_all_headers_changed_cb, NULL);
	pan_callback_remove (text_get_mute_quoted_changed_callback(),
	                     mute_quoted_text_changed_cb, NULL);
	pan_callback_remove (current_article_changed,
	                     update_menus_cb, NULL);
	pan_callback_remove (grouplist_group_selection_changed,
	                     update_menus_cb, NULL);
	pan_callback_remove (grouplist_server_set_callback,
	                     update_menus_cb, NULL);
	pan_callback_remove (articlelist_get_group_changed_callback(),
	                     update_menus_cb, NULL);
	pan_callback_remove (articlelist_selection_changed,
	                     update_menus_cb, NULL);

	g_source_remove (gui_refresh_timer_id);
	gui_refresh_timer_id = 0;
	g_source_remove (gui_queue_timer_id);
	gui_queue_timer_id = 0;

	/**
	***  Save settings
	**/

	pan_config_push_prefix ("/Pan/Display/");
	pan_config_set_bool ("show_group_pane", show_group_pane);
	pan_config_set_bool ("header_pane_is_threaded", header_pane_is_threaded);
	pan_config_set_bool ("show_header_pane", show_header_pane);
	pan_config_set_bool ("show_article_pane", show_article_pane);
	pan_config_set_bool ("navigate_cached_only", navigate_cached_only);
	pan_config_set_bool ("navigate_read_on_select", navigate_read_on_select);
	pan_config_set_bool ("collapse_group_names", collapse_group_names);
	pan_config_set_bool ("do_wrap", text_get_wrap());
	pan_config_set_bool ("use_fixed_font", text_use_fixed_font);
	pan_config_set_bool ("Show_All_Headers_In_Body", text_get_show_all_headers());
	pan_config_set_bool ("Mute_Quoted_text", text_get_mute_quoted());

	pan_config_pop_prefix ();

	pan_config_set_bool ("/Pan/General/download_bodies_too",
	                       download_bodies_too);
	pan_config_set_int ("/Pan/State/viewmode", Pan.viewmode);

	grouplist_shutdown_module ();

	switch (Pan.viewmode)
	{
		case GUI_NOTEBOOK:
			pan_config_set_int ("/Pan/State/page", gui_page_get_nolock ());
			break;

		case GUI_PANED:
			pch = layout_str + 1;
			for (i=0; i<=2; ++i, ++pch)
			{
				if (tolower(*pch)=='g')
					w[i] = groups_vbox;
				else if (tolower(*pch)=='t')
					w[i] = articlelist_ctree;
				else
					w[i] = text_box;
			}
			switch(*layout_str)
			{
				case '2':
					pan_config_set_int ("/Pan/Geometry/vpaned", w[0]->allocation.height);
					pan_config_set_int ("/Pan/Geometry/hpaned", w[1]->allocation.width);
					break;
				case '4':
					pan_config_set_int ("/Pan/Geometry/vpaned", w[1]->allocation.height);
					pan_config_set_int ("/Pan/Geometry/hpaned", w[0]->allocation.width);
					break;
				case '5':
					pan_config_set_int ("/Pan/Geometry/vpaned", w[0]->allocation.height);
					pan_config_set_int ("/Pan/Geometry/hpaned", w[0]->allocation.height + w[1]->allocation.height);
					break;
				case '1':
				case '3':
				default:
					pan_config_set_int ("/Pan/Geometry/vpaned", w[0]->allocation.height);
					pan_config_set_int ("/Pan/Geometry/hpaned", w[0]->allocation.width);
					break;
			}
	}

	gui_save_column_widths (Pan.group_clist, "group");
	gui_save_column_widths (Pan.article_ctree, "thread_pane");
	gui_save_window_size (Pan.window, "main_window");
	pan_config_sync();

	/**
	***  Clear out the UI pieces
	**/

	gtk_widget_unref (groups_vbox);
	gtk_widget_unref (articlelist_ctree);
	gtk_widget_unref (text_box);

	articlelist_set_group_nolock (NULL);
	text_clear_nolock ();
	debug_exit ("gui_shutdown");
}

static void
gui_create_appbar (GtkWidget* window)
{
	int i;
	GtkWidget * w;
	GtkWidget * frame;

	pan_lock();

	appbar = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(appbar), GNOME_PAD_SMALL);

	/* connection status */
	connection_qty_label = gtk_label_new ("");
	w = connection_qty_button = gtk_button_new();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Open the Connection Manager"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", connection_button_clicked, NULL);
	gtk_container_add (GTK_CONTAINER(w), connection_qty_label);
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER(frame), w);
	gtk_box_pack_start (GTK_BOX(appbar), frame, FALSE, FALSE, 0);

	/* task status */
	queue_qty_label = gtk_label_new ("");
	w = queue_qty_button = gtk_button_new();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Open the Task Manager"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", queue_button_clicked, NULL);
	gtk_container_add (GTK_CONTAINER(w), queue_qty_label);
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER(frame), w);
	gtk_box_pack_start (GTK_BOX(appbar), frame, FALSE, FALSE, 0);

	/* status item views */
	taskbar = gtk_table_new (1, VIEW_QTY, TRUE);
	for (i=0; i<VIEW_QTY; ++i) {
		GtkWidget * w = status_item_view_new ();
                views[i].view = STATUS_ITEM_VIEW(w);
		gtk_table_attach (GTK_TABLE(taskbar), w, i, i+1, 0, 1, ~0, ~0, 0, 0);
	}
	gtk_box_pack_start (GTK_BOX(appbar), taskbar, TRUE, TRUE, 0);

	/* error button */
	status_ok_label = gnome_pixmap_new_from_xpm_d (inform_xpm);
	status_error_label = gnome_pixmap_new_from_xpm_d (caution_xpm);
	gtk_object_ref (GTK_OBJECT(status_ok_label));
	gtk_object_ref (GTK_OBJECT(status_error_label));
	error_button = w = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER(w), status_ok_label);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Open the Status Log"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", (GtkSignalFunc)log_button_clicked, NULL);
	gtk_box_pack_start (GTK_BOX(appbar), w, FALSE, FALSE, 0);

	gnome_app_set_statusbar (GNOME_APP(window), appbar);
	pan_unlock();

	gui_set_queue_size (0u, 0u);
	gui_set_connection_size (0);
}

/*---[ gui_shutdown ]-------------------------------------------------
 * save geometry states into config files
 *--------------------------------------------------------------------*/
static guint
warn_about_tasks_running (void)
{
	guint num_tasks = queue_get_running_task_count ();
	int retval = TRUE;

	if (num_tasks != 0)
	{
		guint button = 0;
		char *msg;
		GtkWidget * dialog;

		if (num_tasks == 1)
		{
			msg = g_strdup(
				_("1 task still active or queued.\n"
				  "Are you sure you want to exit Pan?"));
		}
		else
		{
			msg = g_strdup_printf (
				_("%d tasks still active or queued.\n"
				  "Are you sure you want to exit Pan?"),
				  num_tasks);
		}

		dialog = gnome_message_box_new (msg,
		                                GNOME_MESSAGE_BOX_QUESTION,
						GNOME_STOCK_BUTTON_YES,
						GNOME_STOCK_BUTTON_NO,
						NULL);

		g_free (msg);

		button = gnome_dialog_run_and_close (GNOME_DIALOG(dialog));
		if (button == 1) /* NO */
			retval = 0;
	}

	return retval;
}

static void
exit_cb (void)
{
	if (warn_about_tasks_running())
	{
		gui_shutdown ();
		gtk_object_destroy (GTK_OBJECT(Pan.window));
	}
}

static int
window_delete_event_cb (GtkWidget *widget,
			GdkEvent *event,
			gpointer  data)
{
	if (warn_about_tasks_running())
	{
		gui_shutdown ();
		return FALSE; /* causes "destroy" signal to be fired */
	}

	return TRUE;
}

static void
text_fill_body_changed_cb (gpointer call_obj,
                           gpointer call_arg,
                           gpointer user_data)
{
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[15].widget),
	                                call_arg!=NULL);
}

static int
log_list_appended_idle (gpointer data)
{
	GtkWidget * old_child;
	GtkWidget * new_child = status_error_label;

	pan_lock ();
	if (((old_child = GTK_BIN(error_button)->child)) != new_child)
	{
		gtk_container_remove (GTK_CONTAINER(error_button), old_child);
		gtk_container_add (GTK_CONTAINER(error_button), new_child);
		gtk_widget_show (new_child);
	}
	pan_unlock ();

	return 0;
}

static void
log_list_appended_cb (gpointer call_obj, gpointer call_arg, gpointer user_data)
{
	if (((LogEntry*)call_obj)->severity== LOG_ERROR)
		gui_queue_add (log_list_appended_idle, NULL);
}


/***
****
***/

static int
gui_set_title_idle (gpointer data)
{
	Server * server;
	const Group * group = GROUP(data);
	GString * s = g_string_new (NULL);
	const char * servername = NULL;
	const char * groupname = NULL;
	debug_enter ("gui_set_title_idle");

	/* look for active server & group */
	server = serverlist_get_active_server ();
	if (server != NULL)
		servername = server_get_name (server);
	if (group != NULL)
		groupname = group_get_readable_name (group);

	/* build the title */
	g_string_sprintf (s, "Pan %s ", VERSION);
	if (servername != NULL)
		g_string_sprintfa (s, " - %s", servername);
	if (groupname != NULL)
		g_string_sprintfa (s, " - %s - %d/%d",
		                   groupname,
		                   group->article_qty-group->article_read_qty,
		                   group->article_qty);

	/* set the title in the UI */
	pan_lock ();
	gtk_window_set_title (GTK_WINDOW (Pan.window), s->str);
	pan_unlock ();

	/* cleanup */
	g_string_free (s, TRUE);

	debug_exit ("gui_set_title_idle");
	return 0;
}

static void
gui_set_title (const Group * group)
{
	gui_queue_add (gui_set_title_idle, (gpointer)group);
}

static void
articles_changed_cb (gpointer call_obj, gpointer call_arg, gpointer user_data)
{
	Group * agroup = articlelist_get_group ();
	ArticleChangeEvent * e = (ArticleChangeEvent*) call_obj;
	if (agroup == e->group)
		gui_set_title (agroup);
}

/***
****
***/

static int
update_menus_idle (gpointer data)
{
	int i;
	GnomeUIInfo * menu = NULL;
	const Article * article;
	gboolean have_article;
	gboolean have_current_article;
	const Group * alist_group;
	const Group * glist_group;
	gboolean have_group;
	gboolean have_alist_folder;
	gboolean have_glist_folder;
	const Group * sendlater;
	gboolean sendlater_empty;
	gboolean posted_here;
	gboolean is_paned = current_layout == GUI_PANED;
	debug_enter ("update_menus_idle");

	article = articlelist_get_selected_article_nolock ();
	have_article = article!=NULL;
	have_current_article = get_current_article()!=NULL;
	posted_here = article!=NULL;

	alist_group = articlelist_get_group ();
	glist_group = grouplist_get_selected_group();
	have_group = alist_group!=NULL || glist_group!=NULL;
	have_alist_folder = alist_group!=NULL && group_is_folder(alist_group);
	have_glist_folder = glist_group!=NULL && group_is_folder(glist_group);

	sendlater = serverlist_get_named_folder (PAN_SENDLATER);
	sendlater_empty = sendlater!=NULL && !sendlater->article_qty;

	pan_lock ();

	/* update view menu */
	menu = view_menu;
	gtk_widget_set_sensitive (menu[8].widget, is_paned); /* hide group */
	gtk_widget_set_sensitive (menu[9].widget, is_paned); /* hide header */
	gtk_widget_set_sensitive (menu[10].widget, is_paned); /* hide article */

	/* update post menu */
	menu = post_menu;
	i = -1;
	++i; /* i is post */
	gtk_widget_set_sensitive (menu[i].widget, TRUE);
	++i; /* i is followup */
	gtk_widget_set_sensitive (menu[i].widget, have_current_article);
	++i; /* i is reply */
	gtk_widget_set_sensitive (menu[i].widget, have_current_article);
	++i; /* i is reply & followup */
	gtk_widget_set_sensitive (menu[i].widget, have_current_article);
	++i; /* i is forward by email */
	gtk_widget_set_sensitive (menu[i].widget, have_current_article);
	++i; /* separator */
	++i; /* flush pan.sendlater */
	gtk_widget_set_sensitive (menu[i].widget, !sendlater_empty);


	/* update navigate menu */
	menu = navigate_menu;
	i = -1;
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* skip uncached */
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* load when selected */
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* space reading */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* next unread */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* next */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* next unread thread */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* next thread */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* prev */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* prev thread */
	gtk_widget_set_sensitive (menu[++i].widget, have_group && articlelist_has_prev_read()); /* prev read */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* parent */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* top */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* next unrad group */
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* next group */

	/* update group menu */
	menu = group_menu;
	i = -1;
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* mark read */
	++i; /* separator */
	++i; /* get bodies too */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* get new */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* get all */
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* get all subscribed */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* download dialog */
	++i; /* separator */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* refresh article counts */
	++i; /* separator */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* subscribe */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* unsubscribe */
	++i; /* separator */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* properties */
	++i; /* separator */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* empty */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* delete */


	/* update article_menus */
	menu = articlelist_menu;
	i = -1;
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* read */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* thread read */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* unread */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* thread unread */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* download flagged */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* flag */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* flag thread */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* unflag */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* unflag thread */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* download */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* download threads */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* watch */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* ignore  */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* filter  */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* copy */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* copy thread */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* move */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* move thread */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* cancel */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* supersede */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* delete */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* delete thread  */
	gtk_widget_set_sensitive (menu[++i].widget, have_group); /* delete all  */


	/* update file menu */
	menu = file_menu;
	i = -1;
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* save article */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* save article as */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* save (attachments only) */
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* open attachments */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, have_article); /* print */
	++i;
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* new folder */
	gtk_widget_set_sensitive (menu[++i].widget, have_glist_folder && strncmp(glist_group->name, "pan.", 4)); /* delete folder */
	gtk_widget_set_sensitive (menu[++i].widget, have_alist_folder && have_article); /* edit */
	++i; /* separator */
	gtk_widget_set_sensitive (menu[++i].widget, TRUE); /* exit */


	pan_unlock ();

	/* update the title */
	gui_set_title_idle ((gpointer)alist_group);

	debug_exit ("update_menus_idle");
	return 0;
}

static void
update_menus_cb (gpointer call_object, gpointer call_arg, gpointer user_data)
{
	gui_queue_add (update_menus_idle, NULL);
}

static void
log_entry_added_cb (gpointer entry_data, gpointer unused1, gpointer unused2)
{
	LogEntry* e = (LogEntry*) entry_data;
	if (e->severity & LOG_URGENT)
	{
		if (e->severity & LOG_ERROR)
			pan_error_dialog (e->message);
		else /* LOG_INFO */
			pan_info_dialog (e->message);
	}
}



/*---[ gui_construct ]------------------------------------------------
 * @geometry:
 *
 * initialize and place the basic interface components
 *--------------------------------------------------------------------*/
void
gui_construct (const char * geometry)
{
	debug_enter ("gui_construct");

	memset (views, '\0', sizeof(ViewStruct)*VIEW_QTY);
	ttips = gtk_tooltips_new();
        cmap = gdk_colormap_get_system ();

	Pan.window = gnome_app_new ("Pan", "Pan");
	gtk_window_set_default_size (GTK_WINDOW(Pan.window), 820, 700);

	gtk_signal_connect (GTK_OBJECT(Pan.window), "delete_event", 
			    GTK_SIGNAL_FUNC (window_delete_event_cb), NULL);
	gtk_signal_connect (GTK_OBJECT(Pan.window), "destroy",
			    GTK_SIGNAL_FUNC (pan_shutdown), NULL);

	/** geometry parsing **/
	/* The following chunk of code will allow command line arguments
	 * to override the config. So if this was invoked as part of a 
	 * session we 'Do the right thing' :)
	 */
	if (geometry == NULL)
	{
		gui_restore_window_size (Pan.window, "main_window");
	}
	else
	{
		int x, y, w, h;
		if (gnome_parse_geometry (geometry, &x, &y, &w, &h) )
		{
			if (x != -1)
				gtk_widget_set_uposition (Pan.window, x, y);
			if (w != -1)
				gtk_window_set_default_size (GTK_WINDOW (Pan.window), w, h);
		}
		else
		{
			g_error (_("Unable to parse the geometry string \"%s\""), geometry);
		}
	}

/* Groups vbox */
	groups_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_set_usize (GTK_WIDGET (groups_vbox), 260, -1);
	gtk_box_pack_start (GTK_BOX (groups_vbox), GTK_WIDGET(grouplist_create()), TRUE, TRUE, 1);
        gtk_signal_connect (GTK_OBJECT (Pan.group_clist), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb),  NULL);

/* Articlelist_ctree */
	articlelist_ctree = create_articlelist_ctree ();
	article_list_toolbar_mediator_init ();

/* Text */
	text_box = text_create ();

	gui_create_appbar (Pan.window);

	gnome_app_create_menus (GNOME_APP (Pan.window), pan_main_menu);
	gnome_app_install_menu_hints (GNOME_APP (Pan.window), pan_main_menu);

	contents_vbox = gtk_vbox_new (FALSE, 0);
	gnome_app_set_contents (GNOME_APP (Pan.window), contents_vbox);

	/* connect signals */
        gtk_signal_connect (GTK_OBJECT (Pan.article_ctree), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb), NULL);

	gtk_widget_ref (groups_vbox);
	gtk_widget_ref (articlelist_ctree);
	gtk_widget_ref (text_box);
        gui_set_layout (pan_config_get_int ( "Pan/State/viewmode=1" ));

	pan_callback_add (article_get_articles_changed_callback(),
	                  articles_changed_cb, NULL);
	pan_callback_add (log_get_entry_added_callback(),
	                  log_list_appended_cb, NULL);
	pan_callback_add (status_item_get_active_callback(),
	                  status_item_active_changed_cb, NULL);
	pan_callback_add (text_get_fill_body_changed_callback(),
	                  text_fill_body_changed_cb, NULL);

	/* listen for changes to the articlelist active group
	   so that we can sensitize/desensitize menus accordingly */
	update_menus_cb (NULL, NULL, NULL);
	pan_callback_add (text_get_show_all_headers_changed_callback(),
	                  show_all_headers_changed_cb, NULL);
	pan_callback_add (text_get_mute_quoted_changed_callback(),
	                  mute_quoted_text_changed_cb, NULL);
	pan_callback_add (current_article_changed,
			  update_menus_cb, NULL);
	pan_callback_add (grouplist_group_selection_changed,
			  update_menus_cb, NULL);
	pan_callback_add (grouplist_server_set_callback,
	                  update_menus_cb, NULL);
	pan_callback_add (articlelist_get_group_changed_callback(),
			  update_menus_cb, NULL);
	pan_callback_add (articlelist_selection_changed,
			  update_menus_cb, NULL );

	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(navigate_menu[0].widget),
	                                navigate_cached_only);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(navigate_menu[1].widget),
	                                navigate_read_on_select);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[8].widget),
	                                show_group_pane);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[9].widget),
	                                show_header_pane);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[10].widget),
	                                show_article_pane);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[12].widget), 
	                                collapse_group_names);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[13].widget), 
	                                header_pane_is_threaded);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[15].widget),
	                                text_get_wrap());
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[17].widget), 
	                                text_get_show_all_headers());
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[18].widget),
	                                text_use_fixed_font);

	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(group_menu[2].widget),
	                                download_bodies_too);

	text_set_show_all_headers (pan_config_get_bool("/Pan/Display/Show_All_Headers_In_Body=false"));

	text_set_mute_quoted (pan_config_get_bool("/Pan/Display/Mute_Quoted_Text=false"));

	gui_set_title (NULL);

	gui_refresh_timer_id = pan_timeout_add (3000, gui_refresh_timer_cb, NULL); 

	gui_queue_timer_id = pan_timeout_add (160, gui_queue_timer_cb, NULL);

	server_menu_update ();

	pan_callback_add (log_get_entry_added_callback(),
	                  log_entry_added_cb, NULL);

	debug_exit ("gui_construct");
}


/****
*****   GUI LAYOUT  (NOTEBOOK, PANED)
****/

void
gui_set_layout (int new_layout)
{
	switch (new_layout) {
		case GUI_PANED: gui_paned_construct (); break;
		case GUI_NOTEBOOK: gui_notebook_construct (); break;
		default: pan_warn_if_reached (); break;
	}

	current_layout = new_layout;
	update_menus_cb (NULL, NULL, NULL);
}

static int
gui_page_set_idle (gpointer page_gpointer)
{
	int page = GPOINTER_TO_INT (page_gpointer);
	GtkWidget * focus_pane;
	debug_enter ("gui_page_set_idle");

	pan_lock ();
	focus_pane = gui_page_get_pane_nolock (page);
	switch (Pan.viewmode)
	{
		case GUI_PANED:
			/* blah, users hated this.  remove after string thaw.
			if (page==ARTICLELIST_PAGE || page==MESSAGE_PAGE)
				focus_pane = gui_page_get_pane_nolock (ARTICLELIST_PAGE); */
		    	gui_paned_page_set_nolock (page, focus_pane);
			break;

		case GUI_NOTEBOOK:
			gui_notebook_page_set_nolock (page, focus_pane);
			break;

		default:
			pan_warn_if_reached ();
			break;
	}
	pan_unlock ();

	debug_exit ("gui_page_set_idle");
	return 0;
}
void
gui_page_set (int page)
{
	gui_queue_add (gui_page_set_idle, GINT_TO_POINTER(page));
}

static int
gui_page_get_nolock (void)
{
	int retval = 0;

	switch (Pan.viewmode) {
		case GUI_PANED:    retval = gui_paned_get_current_pane_nolock();    break;
		case GUI_NOTEBOOK: retval = gui_notebook_get_current_pane_nolock(); break;
		default: pan_warn_if_reached(); break;
	}

	return retval;
}

static GtkWidget*
gui_page_get_pane_nolock (int page)
{
	GtkWidget * retval = NULL;

	switch (page) {
		case GROUPS_PAGE: retval = Pan.group_clist; break;
		case ARTICLELIST_PAGE: retval = Pan.article_ctree; break;
		case MESSAGE_PAGE: retval = Pan.text; break;
		default: pan_warn_if_reached ();
	}

	return retval;
}

static void
gui_page_activate_nolock (int page)
{
	switch (page)
	{
		case GROUPS_PAGE:
		{
			Group * g = grouplist_get_selected_group ();
			if (g != NULL)
				articlelist_set_group_nolock (g);
			break;
		}
		case ARTICLELIST_PAGE:
		{
			GPtrArray * articles;

			gui_queue_add ((GSourceFunc) articlelist_read_selected, NULL);

			/* also download any others that are selected */
			articles = articlelist_get_selected_articles_nolock ();
			if (articles->len > 1) {
				Group * g = ARTICLE(g_ptr_array_index(articles,0))->group;
				g_ptr_array_remove_index (articles, 0);
				queue_add (TASK(task_bodies_new (g, articles)));
			}
			g_ptr_array_free (articles, TRUE);

			break;
		}
		case MESSAGE_PAGE:
			break;
		default:
			pan_warn_if_reached ();
			break;
	}
}

static void
gui_page_change_nolock (int change)
{
	int page = gui_page_get_nolock () + change;

	if (0<=page && page<=2)
		gui_page_set (page);
}


/*---[ gui_key_press_cb ]---------------------------------------------
 * key press callback, when a key is pressed in the group list, article
 * list, or in the message view.  This allows better keyboard navigation
 * and the potential for new key bindings to imitate Ag*nt.
 *--------------------------------------------------------------------*/
static gboolean
gui_key_press_cb (GtkWidget      * widget,
                  GdkEventKey    * event,
                  gpointer         data)
{
	switch (event->keyval)
	{
		case GDK_Q: case GDK_q:
			gui_page_change_nolock (-1);
			break;

		case GDK_Delete:
			if (event->state & GDK_SHIFT_MASK)
				article_action_delete_selected_threads ();
			else
				article_action_delete_selected_articles ();
			break;

		case GDK_Return:
			gui_page_activate_nolock (gui_page_get_nolock ());
			gui_page_change_nolock (1);
			break;

		case GDK_space: case GDK_KP_Space:
			text_read_more ();
			break;

		default:
			break;
	}

	return TRUE;
}

/**
***
**/

void
gui_restore_column_widths (GtkWidget     * clist,
                           const char   * type)
{
	int i;
	GtkCList * list = GTK_CLIST (clist);
	const int cols = list->columns;
	int * widths = g_new (int, cols);
	char buf[1024];
	debug_enter ("gui_restore_column_widths");

	/* get width from config... */
	for (i=0; i!=cols; ++i) {
		g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_column_%d", type, i);
		widths[i] = pan_config_get_int (buf);
	}

	/* set ui.. */
	pan_lock();
	for (i=0; i!=cols; ++i)
		if (widths[i] != 0)
			gtk_clist_set_column_width (list, i, widths[i]);
	pan_unlock();

	/* cleanup */
	g_free (widths);
	debug_exit ("gui_restore_column_widths");
}


void
gui_save_column_widths (GtkWidget    * clist,
                        const char   * type)
{
	int i;
	int cols = GTK_CLIST (clist)->columns;
	GtkCList * list = GTK_CLIST (clist);
	char buf[1024];
	
	for (i=0; i<cols; i++)
	{
		g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_column_%d", type, i);
		pan_config_set_int (buf, (int)list->column[i].width);
	}


	pan_config_sync ();
}

/**
***
**/

void
gui_restore_window_size (GtkWidget     * widget,
                         const char    * key)
{
	int x, y, w, h;
	char buf[1024];
	int screen_w = gdk_screen_width ();
	int screen_h = gdk_screen_height ();
	const int fuzz = 10;

	/* sanity clause */
	g_return_if_fail (GTK_IS_WIDGET(widget));
	g_return_if_fail (!GTK_WIDGET_REALIZED(widget));
	g_return_if_fail (is_nonempty_string(key));

	/* get the dimensions */
	g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_x", key);
	x = pan_config_get_int (buf);
	g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_y", key);
	y = pan_config_get_int (buf);
	g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_width", key);
	w = pan_config_get_int (buf);
	g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_height", key);
	h = pan_config_get_int (buf);

	/* move & resize */
	if ((0<x && x<screen_w-fuzz) && (0<y && y<screen_h-fuzz))
		gtk_widget_set_uposition (widget, x, y);
	if (w>0 && h>0)
		gtk_window_set_default_size (GTK_WINDOW(widget), w, h);




}
 
void
gui_save_window_size (GtkWidget     * widget,
                      const char    * key)
{
	char buf[1024];
	int x, y, w, h;

	/* sanity clause */
	g_return_if_fail (GTK_IS_WIDGET(widget));
	g_return_if_fail (widget->window!=NULL);
	g_return_if_fail (is_nonempty_string(key));

	gdk_window_get_position (widget->window, &x, &y);
	gdk_window_get_size (widget->window, &w, &h);

	g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_x", key);
	pan_config_set_int (buf, x);
	g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_y", key);
	pan_config_set_int (buf, y);
	g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_width", key);
	pan_config_set_int (buf, w);
	g_snprintf (buf, sizeof(buf), "/Pan/Geometry/%s_height", key);
	pan_config_set_int (buf, h);

	pan_config_sync ();


}

/*****
******
*****/

static int connection_size = -1;

static gboolean _is_connecting = FALSE;

static int
gui_refresh_connection_label_idle (gpointer data)
{
	char str[128];
	char tip[128];

	/* decide what to say */
	if (connection_size != 0) {
		g_snprintf (str, sizeof(str), _("%d @ %.1f KB/s"), connection_size, KBps);
		g_snprintf (tip, sizeof(tip), _("%d connections at %.1f KB per second"),
			connection_size, KBps);
	}
	else if (_is_connecting) {
		g_snprintf (str, sizeof(str), _("Connecting"));
		g_snprintf (tip, sizeof(tip), _("Connecting to a Remote News Server to Execute a Task"));

	}
	else {
		g_snprintf (str, sizeof(str), _("Offline"));
		g_snprintf (tip, sizeof(tip), _("No Connections"));
	}

	/* update the tip */
	pan_lock ();
	if (GTK_IS_LABEL(connection_qty_label)) {
		gtk_label_set_text (GTK_LABEL(connection_qty_label), str);
	if (GTK_IS_BUTTON(connection_qty_button))
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), connection_qty_button, tip, "");
	}
	pan_unlock ();

	return 0;
}

void
gui_set_connecting_flag (gboolean is_connecting)
{
	if (_is_connecting != is_connecting)
	{
		_is_connecting = is_connecting;
		gui_queue_add (gui_refresh_connection_label_idle, NULL);
	}
}

static void
gui_set_connection_size (int size)
{
	if (size != connection_size)
	{
		connection_size = size;
		log_add_va (LOG_INFO, _("News server connection count: %d"), size);
		gui_queue_add (gui_refresh_connection_label_idle, NULL);
	}
}

void
gui_inc_connection_size (int inc)
{
	gui_set_connection_size (connection_size + inc);
}

static int
gui_refresh_timer_cb (gpointer user_data)
{
	/* update KBps */
	if (1)
	{
		static struct timeval last_time;
		static unsigned long last_KB = 0;
		static gboolean last_time_inited = FALSE;

		if (!last_time_inited)
		{
			gettimeofday (&last_time, NULL);
			last_time_inited = TRUE;
		}
		else
		{
			double time_diff;
			double KB_diff;
			struct timeval new_time;
			unsigned long new_KB;

			gettimeofday (&new_time, NULL);
			time_diff = (new_time.tv_sec - last_time.tv_sec) * 1000000.0;
			time_diff += (new_time.tv_usec - last_time.tv_usec);
			last_time = new_time;

			new_KB = pan_socket_get_total_xfer_K ();
			KB_diff = new_KB - last_KB;
			last_KB = new_KB;

			KB_diff *= 1000000.0;
			KBps = KB_diff / time_diff;
		}
	}

	gui_queue_add (gui_refresh_connection_label_idle, NULL);

	return 1;
}

/*****
******
*****/

void
gui_set_queue_size (guint running, guint size)
{
	char str[128];
	char tip[128];

	if (size == 0u) {
		g_snprintf (str, sizeof(str), _("No Tasks"));
		g_snprintf (tip, sizeof(tip), _("The Task Manager is Empty"));
	}
	else {
		g_snprintf (str, sizeof(str), _("Tasks: %u/%u"), running, size);
		g_snprintf (tip, sizeof(tip), _("%u Tasks Running, %u Tasks Total"), running, size);
	}

	pan_lock();
	gtk_label_set_text (GTK_LABEL(queue_qty_label), str);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), queue_qty_button, tip, "");
	pan_unlock();
}

static int
gui_add_status_item_idle (gpointer p)
{
	int i;
	StatusItem * status;
	debug_enter ("gui_add_status_item");

	pan_lock ();

	for (i=0; i<VIEW_QTY; ++i)
		if (views[i].status_item == NULL)
			break;

       	status = STATUS_ITEM(p);
	if (i!=VIEW_QTY) {
		pan_object_ref (PAN_OBJECT(status)); /* unref in remove_status_item_idle */
		views[i].status_item = status;
		status_item_view_set_item_nolock (STATUS_ITEM_VIEW(views[i].view), status);
	}

	pan_object_unref (PAN_OBJECT(status)); /* ref in item_active_changed_cb */
	pan_unlock ();
	debug_exit ("gui_add_status_item");
	return 0;
}

static int
gui_remove_status_item_idle (gpointer p)
{
	int i;
	StatusItem * status;
	debug_enter ("gui_remove_status_item");

	pan_lock ();

	status = STATUS_ITEM(p);
	for (i=0; i<VIEW_QTY; ++i)
		if (views[i].status_item == status)
			break;
	if (i!=VIEW_QTY) {
		StatusItemView * view = views[i].view;
		status_item_view_set_item_nolock (STATUS_ITEM_VIEW(view), NULL);
		views[i].status_item = NULL;
		pan_object_unref (PAN_OBJECT(status)); /* ref in set_item_idle */
	}

	pan_unlock ();
	pan_object_unref (PAN_OBJECT(status)); /* ref in item_active_changed_cb */
	debug_exit ("gui_remove_status_item");
	return 0;
}

static void
turn_off_status_item_views_nolock (void)
{
	int i;
	debug_enter ("turn_off_status_item_views");

	for (i=0; i<VIEW_QTY; ++i) {
		StatusItem * item = views[i].status_item;
		StatusItemView * view = views[i].view;
		if (item != NULL) {
			status_item_view_set_item_nolock (STATUS_ITEM_VIEW(view), NULL);
			views[i].status_item = NULL;
		}
	}

	debug_exit ("turn_off_status_item_views");
}

static void
status_item_active_changed_cb (gpointer call_obj, gpointer call_arg, gpointer user_data)
{
	StatusItem * item = STATUS_ITEM(call_obj);
	gboolean is_active = GPOINTER_TO_INT(call_arg) != 0;
	debug_enter ("status_item_active_changed_cb");

	/**
	***  Ref the item to make sure it's still alive when the
	***  idle func is run.  The idle func unrefs it to balance
	***  this out.
	**/

	pan_object_ref (PAN_OBJECT(item));

	/**
	***  Push the GUI work to the main thread
	**/

	gui_queue_add (is_active ? gui_add_status_item_idle : gui_remove_status_item_idle, item);


	debug_exit ("status_item_active_changed_cb");
}

/*****
******
*****/

static void
pan_widget_set_font_on_realize (GtkObject * o, gpointer user_data)
{
	char * font_name = gtk_object_get_data (o, "font_name");

	/* sanity clause */
	g_return_if_fail (GTK_IS_WIDGET(o));
	g_return_if_fail (is_nonempty_string(font_name));

	/* ...try try again */
	gtk_signal_disconnect_by_func (o, pan_widget_set_font_on_realize, user_data);
	pan_widget_set_font (GTK_WIDGET(o), font_name);
}
void
pan_widget_set_font (GtkWidget * w, const char * font_name)
{
	/* sanity clause */
	g_return_if_fail (GTK_IS_WIDGET(w));
	g_return_if_fail (is_nonempty_string(font_name));

	if (GTK_WIDGET_REALIZED(w))
	{
		GdkFont * font = use_gdk_fontset_load
			? gdk_fontset_load (font_name)
			: gdk_font_load (font_name);

		if (!font)
			log_add_va (LOG_ERROR, _("Couldn't load font \"%s\""), font_name);
		else {
			GtkStyle * style = gtk_style_copy (w->style);
			gdk_font_unref (style->font);
			style->font = font;
			gtk_widget_set_style (w, style);
		}
	}
	else
	{
		/* if at first you don't succeed... */
		gtk_object_set_data_full (GTK_OBJECT(w), "font_name", g_strdup(font_name), g_free);
		gtk_signal_connect (GTK_OBJECT(w), "realize", pan_widget_set_font_on_realize, NULL);
	}
}

/*****
******  This is for delegating small, async GUI updates to the main thread
******  to reduce the number of threads contending for a GUI lock.
*****/

static GStaticMutex gui_queue_mutex = G_STATIC_MUTEX_INIT;

static GSList * gui_queue_slist = NULL;

typedef struct
{
	GSourceFunc func;
	gpointer user_data;
}
GuiQueueItem;

static int
gui_queue_timer_cb (gpointer user_data)
{
	GSList * list;

	/* get the work list */
	g_static_mutex_lock (&gui_queue_mutex);
	list = gui_queue_slist;
	gui_queue_slist = NULL;
	g_static_mutex_unlock (&gui_queue_mutex);

	if (list != NULL)
	{
		GSList * l;

		/* gui_queue_slist is LIFO for efficiency of appending nodes;
		 * reverse the nodes to make list FIFO */
		list = g_slist_reverse (list);

		/* run the work functions */
		for (l=list; l!=NULL; l=l->next)
		{
			GuiQueueItem * queue_item = (GuiQueueItem*) l->data;
			(*queue_item->func)(queue_item->user_data);
			g_free (queue_item);
		}
		g_slist_free (list);
	}

	return 1;
}

void
gui_queue_add (GSourceFunc func, gpointer user_data)
{
	GuiQueueItem * item;

	/* create the new queue item */	
	item = g_new (GuiQueueItem, 1);
	item->func = func;
	item->user_data = user_data;

	/* add it to the queue */
	g_static_mutex_lock (&gui_queue_mutex);
	gui_queue_slist = g_slist_prepend (gui_queue_slist, item);
	g_static_mutex_unlock (&gui_queue_mutex);
}
