/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * A note on implementation
 * The queue defers all public tasks (like insert, remove, etc.)
 * to the queue thread.  This ensures that all tasks are handled
 * in sequence, rather than concurrently.  This ensures that callbacks
 * will be invoked in sync with the queue, so the queue's state won't
 * change during the callback.
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <pthread.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>

#include <glib.h>

#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/server.h>

#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/server-ui.h>

/*********************
**********************  Defines / Enumerated types
*********************/

typedef enum
{
	TODO_INSERT,
	TODO_REMOVE,
	TODO_MOVE,
	TODO_REQUEUE,
	TODO_TASK_ABORT,
	TODO_SET_MAX_SOCKET_QTY,
	TODO_SET_MAX_TRIES,
	TODO_SHUTDOWN,
	TODO_PAUSED
}
QueueTodoAction;

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	Task *task;
	int index_1;
	int index_2;
	QueueTodoAction action;
}
QueueTodo;

typedef struct
{
	PanSocket *socket;
	Server *server;
	Task *task;	/* NULL if idle */
}
SocketInfo;

/*********************
**********************  Private Function Prototypes
*********************/

static void* queue_mainloop (void*);

static void queue_run_what_we_can (void);

static void queue_set_task_status (Task* task,
                                   QueueTaskStatus status);

static void socket_cleanup (Task* task);

static void queue_new_todo (QueueTodoAction action,
                            Task* task,
                            int i1,
                            int i2);

static guint queue_get_length (void);

static void queue_do_todo (void);

/*********************
**********************  Constants
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

PanCallback * queue_task_added           = NULL;
PanCallback * queue_task_removed         = NULL;
PanCallback * queue_task_moved           = NULL;
PanCallback * queue_task_status_changed  = NULL;
PanCallback * queue_max_tries_changed    = NULL;

/***********
************  Private
***********/


/* FIXME glib 2.0: for efficiency of adding to end, use a GQueue instead of GSList */
static GSList* todo_list = NULL;
static GSList* task_list = NULL;

static pthread_mutex_t sock_lock;
static pthread_mutex_t cond_mutex;
static pthread_mutex_t todo_mutex;
static pthread_mutex_t task_lock;
static pthread_cond_t qcond = PTHREAD_COND_INITIALIZER;
static gboolean work_to_do = FALSE;

static pthread_t queue_thread = 0;

static GHashTable* task_to_status = NULL;

static GSList * sockets = NULL;

static int max_socket_qty = 4;

static int max_tries = 5;

static guint running_tasks = 0;

static gboolean paused = FALSE;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

void
queue_wakeup (void)
{
	debug_enter ("queue_wakeup");
        pthread_mutex_lock (&cond_mutex);
	work_to_do = TRUE;
        pthread_cond_signal (&qcond);
        pthread_mutex_unlock (&cond_mutex);
	debug_exit ("queue_wakeup");
}

static void
server_online_status_changed_cb (gpointer obj, gpointer arg, gpointer data)
{
	debug_enter ("server_online_status_changed_cb");
	queue_wakeup ();
	debug_exit ("server_online_status_changed_cb");
}

void
queue_init (int socket_qty, int tries)
{
	debug_enter ("queue_init");

	task_to_status = g_hash_table_new (g_direct_hash, g_direct_equal);

	queue_task_added = pan_callback_new ();
	queue_task_removed = pan_callback_new ();
	queue_task_moved = pan_callback_new ();
	queue_task_status_changed = pan_callback_new ();
	queue_max_tries_changed = pan_callback_new ();

	pthread_mutex_init (&cond_mutex, NULL);
	pthread_mutex_init (&todo_mutex, NULL);
	pthread_mutex_init (&sock_lock, NULL);

	pan_callback_add (server_get_online_status_changed_callback(),
	                  server_online_status_changed_cb, NULL);

	max_tries = tries;
	max_socket_qty = socket_qty;

	/* fire up the queue thread */
	pthread_create (&queue_thread, NULL, queue_mainloop, NULL);
	pthread_detach (queue_thread);

	debug_exit ("queue_init");
}


gboolean
queue_is_paused (void)
{
	return paused;
}
void
queue_set_paused (gboolean paused)
{
	queue_new_todo (TODO_PAUSED, NULL, paused, -1);
}
static void
real_queue_set_paused (gint p)
{
	if ((paused = p))
	{
		GSList * l;

		pthread_mutex_lock (&sock_lock);
		for (l=sockets; l!=NULL; l=l->next)
		{
			SocketInfo * i = (SocketInfo*)l->data;
			task_hint_abort (i->task);
			queue_set_task_status (i->task, QUEUE_TASK_STATUS_ABORTING);
		}
		pthread_mutex_unlock (&sock_lock);
	}
}


void
queue_shutdown (gboolean save_tasks)
{
	queue_new_todo (TODO_SHUTDOWN, NULL, save_tasks, -1);
}
static void
real_queue_shutdown (int save_tasks)
{
	gint len;
	GSList * l;
	GSList * sockets_tmp;

	/* stop listening to events */
	pan_callback_remove (server_get_online_status_changed_callback(),
	                     server_online_status_changed_cb, NULL);

	/* clear the status hashtable */
	g_hash_table_destroy (task_to_status);
	task_to_status = NULL;



	/* optionally save the tasks */
	if (save_tasks)
	{
	}

	/* unref the tasks */

	/* get the socket array */
	pthread_mutex_lock (&sock_lock);
	sockets_tmp = sockets;
	sockets = NULL;
	pthread_mutex_unlock (&sock_lock);

	/* close the sockets */
	len = g_slist_length (sockets_tmp);
	for (l=sockets_tmp; l!=NULL; l=l->next)
		pan_object_unref (PAN_OBJECT(((SocketInfo*)l->data)->socket));
	gui_inc_connection_size (-len);
	g_slist_foreach (sockets_tmp, (GFunc)g_free, NULL);
	g_slist_free (sockets_tmp);
}


void
queue_add (Task * task)
{
	g_return_if_fail (task!=NULL);
	queue_task_insert (task, task->high_priority ? 0 : -1);
}

GPtrArray*
queue_get_tasks (void)
{
	GSList * l;
	GPtrArray * a = g_ptr_array_new ();

        pthread_mutex_lock (&task_lock);
	for (l=task_list; l!=NULL; l=l->next) {
		pan_object_ref (PAN_OBJECT(l->data));
		g_ptr_array_add (a, l->data);
	}
        pthread_mutex_unlock (&task_lock);

	return a;
}

/**
***
**/

int
queue_get_max_sockets (void)
{
	return max_socket_qty;
}
void
queue_set_max_sockets (int i)
{
	queue_new_todo (TODO_SET_MAX_SOCKET_QTY, NULL, i, 0);
	queue_wakeup ();
}
static void
real_queue_set_max_sockets (int i)
{
	max_socket_qty = i;
	/* FIXME: fire some callback? */
}

guint
queue_get_running_task_count (void)
{
	gint running_count = 0;
	GSList *l;

	pthread_mutex_lock (&task_lock);
	for (l=task_list; l!=NULL; l=l->next)
	{
		QueueTaskStatus status = queue_get_task_status (l->data);

		if (status == QUEUE_TASK_STATUS_RUNNING ||
		    status == QUEUE_TASK_STATUS_QUEUED ||
		    status == QUEUE_TASK_STATUS_CONNECTING)
			++running_count;
	}
	pthread_mutex_unlock (&task_lock);

	return running_count;
}

/**
***
**/

void
queue_task_abort (Task* task)
{
	queue_new_todo (TODO_TASK_ABORT, task, -1, -1);
}
static void
real_queue_task_abort (Task* task)
{
	QueueTaskStatus status = queue_get_task_status (task);

	if (status==QUEUE_TASK_STATUS_RUNNING)
	{
		/* ask a running task to abort */
		task_hint_abort (TASK(task));
		queue_set_task_status (task, QUEUE_TASK_STATUS_ABORTING);
	}
	else if (task->hint_abort && paused)
	{
		/* the task's been stopped because the queue is paused */
		task->tries = 0;
		task->hint_abort = FALSE;
		queue_set_task_status (task, QUEUE_TASK_STATUS_QUEUED);
	}
	else
	{
		/* the task has failed */
		task->tries = 0;
		task->hint_abort = FALSE;
		queue_set_task_status (task, QUEUE_TASK_STATUS_FAILED);
	}
}

/**
***
**/

void
queue_task_insert (Task* task, int index)
{
	if (task!=NULL)
		queue_new_todo (TODO_INSERT, task, index, -1);
}

static void
real_queue_task_insert (Task* task, int index)
{
	guint task_len;
	debug2 (DEBUG_QUEUE, "queue_task_insert %p to pos %d", task, index);

	g_return_if_fail (task!=NULL);
	g_return_if_fail (index==-1 || (guint)index<=queue_get_length());

	/* add to the list */
	pthread_mutex_lock (&task_lock);
	task_list = g_slist_insert (task_list, task, index);
	g_hash_table_insert (task_to_status, task, GINT_TO_POINTER(QUEUE_TASK_STATUS_QUEUED));
	task_len = g_slist_length (task_list);
	pthread_mutex_unlock (&task_lock);

	gui_set_queue_size (running_tasks, task_len);

	/* call callbacks */
	debug1 (DEBUG_QUEUE, "queue size is now %u, calling callbacks", task_len);
	pan_callback_call (queue_task_added, task, GINT_TO_POINTER(index));

	queue_wakeup ();
	
	debug2 (DEBUG_QUEUE, "queue_task_insert %p to pos %d done", task, index);
}

/**
***
**/

void
queue_task_remove (Task* task)
{
	queue_new_todo (TODO_REMOVE, task, -1, -1);
}
static void
real_queue_task_remove (Task* task)
{
	debug1 (DEBUG_QUEUE, "queue_task_remove %p", task);

	g_return_if_fail (task!=NULL);
	if (task->is_running)
	{
		task_hint_abort (task);
		queue_set_task_status (task, QUEUE_TASK_STATUS_ABORTING);
	}
	else /* not running; simple cleanup. queue_run_thread() zeroes out is_running */
	{
		gint old_index;
		guint task_len;
		const gboolean had_socket = task->sock != NULL;
		socket_cleanup (task);

		/* remove the task */
		pthread_mutex_lock (&task_lock);
	       	old_index = g_slist_index (task_list, task);
		task_list = g_slist_remove (task_list, task);
		task_len = g_slist_length (task_list);
		pthread_mutex_unlock (&task_lock);

		gui_set_queue_size (running_tasks, task_len);

		/* fire callbacks */
		pan_callback_call (queue_task_removed, task, GINT_TO_POINTER(old_index));
		g_hash_table_remove (task_to_status, task);
		pan_object_unref (PAN_OBJECT(task));

		/* maybe another socket task can run now */
		if (had_socket)
			queue_wakeup ();
	}

	debug1 (DEBUG_QUEUE, "queue_task_remove %p done", task);
}

/**
***
**/

void
queue_task_move (Task* task, int index)
{
	queue_new_todo (TODO_MOVE, task, index, -1);
}
static void
real_queue_task_move (Task* task, int index)
{
	int old_index;
	int indices [2];

	debug2 (DEBUG_QUEUE, "queue_task_move %p to %d", task, index);

	/* sanity clause */
	g_return_if_fail (task!=NULL);
	g_return_if_fail (index==-1 || (guint)index<=queue_get_length());
	
	pthread_mutex_lock (&task_lock);
	
	old_index = g_slist_index (task_list, task);
	if (old_index != -1)
	{
		task_list = g_slist_remove (task_list, task);
		task_list = g_slist_insert (task_list, task, index);
		
		indices[0] = old_index;
		indices[1] = index;
	}

	pthread_mutex_unlock (&task_lock);

	if (old_index != -1)
		pan_callback_call (queue_task_moved, task, indices);

	debug1 (DEBUG_QUEUE, "queue_task_move %p done", task);
}

/**
***
**/

void
queue_task_requeue_failed (Task* task, int index)
{
	g_return_if_fail (task != NULL);
	g_return_if_fail (queue_get_task_status(task) == QUEUE_TASK_STATUS_FAILED);

	queue_new_todo (TODO_REQUEUE, task, index, -1);
}
static void
real_queue_task_requeue_failed (Task* task, int index)
{
	task->tries = 0;
	task->hint_abort = FALSE;
	queue_set_task_status (task, QUEUE_TASK_STATUS_QUEUED);
}

/**
***
**/

int
queue_get_max_tries (void)
{
	return max_tries;
}

void
queue_set_max_tries (int i)
{
	queue_new_todo (TODO_SET_MAX_TRIES, NULL, i, -1);
}
static void
real_queue_set_max_tries (int i)
{
	max_tries = i;
	pan_callback_call (queue_max_tries_changed, NULL, GINT_TO_POINTER(i));
}

/**
***
**/

QueueTaskStatus
queue_get_task_status (const Task* task)
{
	QueueTaskStatus status = QUEUE_TASK_STATUS_NOT_QUEUED;

	if (task_to_status != NULL) {
		gpointer p = g_hash_table_lookup (task_to_status, task);
		if (p != NULL)
			status = GPOINTER_TO_INT (p);
	}

	return status;
}

static void
queue_set_task_status (Task* task, QueueTaskStatus status)
{
	g_return_if_fail (task!=NULL);
	g_return_if_fail (g_slist_index(task_list, task) != -1);

	g_hash_table_insert (task_to_status, task,
	                     GINT_TO_POINTER(status));
	pan_callback_call (queue_task_status_changed, task,
	                   GINT_TO_POINTER(status)); 
}

/************
*************  PRIVATE ROUTINES
************/

static void
queue_new_todo (QueueTodoAction action, Task* task, int i1, int i2)
{
	QueueTodo *todo = g_new0 (QueueTodo, 1);
	todo->task = task;
	todo->index_1 = i1;
	todo->index_2 = i2;
	todo->action = action;

	pthread_mutex_lock (&todo_mutex);
	todo_list = g_slist_append (todo_list, todo);
	debug3 (DEBUG_QUEUE, "todo queue now has %d tasks (new task type: %d, int 1: %d)",
		(int)g_slist_length(todo_list), action, i1);
	pthread_mutex_unlock (&todo_mutex);

	queue_wakeup ();
}

static guint
queue_get_length (void)
{
	guint size;
	pthread_mutex_lock (&task_lock);
	size = g_slist_length (task_list);
	pthread_mutex_unlock (&task_lock);
	return size;
}

static void*
queue_run_thread (void* vp)
{
	/* preparing to run */
	gint result;
	Task* task = TASK(vp);
	gchar* desc = status_item_describe (STATUS_ITEM(task));
	gboolean aborted;
	debug2 (DEBUG_QUEUE, "queue_run_thread: task %p [%s]", task, desc);

	/* run */
	status_item_set_active (STATUS_ITEM(task), TRUE);
	if (task->sock)
		pan_socket_reset_statistics (task->sock);
	gui_set_queue_size (++running_tasks, queue_get_length());
	result = task_run (task);
	gui_set_queue_size (--running_tasks, queue_get_length());
	debug3 (DEBUG_QUEUE, "queue task[%p] (%s) returned: %d", task, desc, result);
	g_free(desc);

	/* clean up gui */
	status_item_set_active (STATUS_ITEM(task), FALSE);

	/* clean up socket */
	aborted = queue_get_task_status(task) == QUEUE_TASK_STATUS_ABORTING;
	if (aborted)
		task->sock->error = TRUE;
	socket_cleanup (task);
	/* clean up the task */
	task->is_running = FALSE;

	if (result==TASK_SUCCESS) /* success */
	{
		queue_task_remove (task);
	}
	else if (result==TASK_FAIL && task->tries<max_tries && !aborted) /* retry */
	{
		++task->tries;
		queue_set_task_status (task, QUEUE_TASK_STATUS_QUEUED);
		queue_wakeup ();
	}
	else if (aborted) /* aborted */
	{
		task->tries = 0;
		queue_task_abort (task);
	}
	else /* failed */
	{
		task->tries = 0;
		task->hint_abort = FALSE;
		queue_set_task_status (task, QUEUE_TASK_STATUS_FAILED);
		queue_wakeup ();
	}

	debug1 (DEBUG_QUEUE, "queue_run_thread: task %p done", vp);
	pthread_exit (NULL);
	return NULL;
}


static void
queue_run (Task* task)
{
	pthread_t thread;
	g_return_if_fail (task!=NULL);
	debug2 (DEBUG_QUEUE, "starting a thread to run task %p, sock %p", task, task->sock);

	queue_set_task_status (task, QUEUE_TASK_STATUS_RUNNING);

	task->is_running = TRUE;
	pthread_create (&thread, NULL, queue_run_thread, task);
	pthread_detach (thread);
}


/**
 * Allocates a socket for a task.  First it looks for an idle socket in the
 * socket pool; otherwise, if we're not maxed out, it tries to create a new
 * socket.
 *
 * Return -1 if we can't connect because all lines are busy
 * Return -2 if we can't connect because of a connection failure
 */
static int
queue_get_socket_for_task (Task *task)
{
	GSList* sl = NULL;
	int retval = -1;
	const gboolean gets_bodies = TASK(task)->gets_bodies;
	gboolean skip_first_idle = task->server->reserve_connection_for_bodies && !gets_bodies;
	gboolean reader_socket_exists = FALSE;
	Server * server = task->server;
	int socket_qty;
	debug_enter ("queue_get_socket_for_task");

	/* if we're not online, then try to go online, or not. */
	if (!server->online)
	{
		switch (going_online_preference)
		{
			case ONLINE_YES:
				server_set_online (server, TRUE);
				break;

			case ONLINE_NO:
				return -1;

			case ONLINE_PROMPT:
			default:
				server_prompt_to_go_online (server);
				return -1;
				break;
		}
	}

	pthread_mutex_lock (&sock_lock);

	/* if an idle socket for that server, use it. */
	socket_qty = 0;
	for (sl=sockets; sl!=NULL && retval!=0; sl=sl->next)
	{
		SocketInfo * si = (SocketInfo*) sl->data;

		if (si->server != server) /* ignore other servers */
			continue;

		++socket_qty;

		if (!server->online) /* server is offline */
			continue;
		if (si->task != NULL) /* socket already in use */
			continue;

		if (skip_first_idle) { /* save one article reader socket */
			reader_socket_exists = TRUE;
			skip_first_idle = FALSE;
			continue;
		}

		si->task = task;
		task->sock = si->socket;

		debug3 (DEBUG_QUEUE,
			"queue_get_socket_for_task %p "
			"reusing socket #%d (%p)",
			task, g_slist_index(sockets, si), si->socket);

		retval = 0;
	}

	/* if no idle sockets for that server... */
	if (retval && server->online && socket_qty<max_socket_qty)
	{
		const int current_connections = socket_qty;
		int slots_open = server->max_connections - current_connections;
		debug1 (DEBUG_QUEUE, "queue_get_socket_for_task %p no idle sockets", task);

		/* leave an article reader socket free */
		if (!reader_socket_exists
			&& server->reserve_connection_for_bodies
			&& !gets_bodies
			&& max_socket_qty>1
			&& server->max_connections>1)
				--slots_open;

		if (slots_open <= 0)
		{
			debug2 (DEBUG_QUEUE,
				"we've already got %d connections "
				"and %d reader connections",
				current_connections,
				(reader_socket_exists?1:0) );

			retval = -1;
		}
		else	
		{
			const char * errmsg = NULL;

			/* open a new socket.. */
			PanSocket *sock = NULL;
			debug1 (DEBUG_QUEUE, "making a new socket to %s", server->address);

			gui_set_connecting_flag (TRUE);
			sock = pan_socket_new (server->address, server->port);
			pan_socket_set_nntp_auth (sock, server->need_auth, server->username, server->password); 

			debug1 (DEBUG_QUEUE, "new socket %p", sock);
			if (sock->sockfd < 0)
				errmsg = "Unable to connect to server";
			else /* handshake */
			{
				StatusItem * status = STATUS_ITEM(task);
				gboolean posting_ok = FALSE;
				gint val = nntp_handshake (status, sock, &posting_ok);
				if (val != TASK_SUCCESS)
					errmsg = (const gchar*)(g_slist_last(status->errors)->data);
			}
			if (errmsg!=NULL)
			{
				log_add (LOG_ERROR, errmsg);
				g_warning(errmsg);
				if (sock)
					pan_object_unref (PAN_OBJECT(sock));
				sock = NULL;

				debug1 (DEBUG_QUEUE, "%p can't log in!", sock);

				retval = -2;
			}
			else
			{
				SocketInfo * si = NULL;

				debug1 (DEBUG_QUEUE, "%p logged in", sock);

				/* looks like we got a new socket okay... */
				si = g_new0 (SocketInfo, 1);
				si->socket = sock;
				si->server = server;
				si->task = task;
				task->sock = sock;
				sockets = g_slist_prepend (sockets, si);
				retval = 0;
				gui_inc_connection_size (1);
			}

			gui_set_connecting_flag (FALSE);
		}
	}

	pthread_mutex_unlock (&sock_lock);

	debug_exit ("queue_get_socket_for_item");
	return retval;
}

/**
 * Run any tasks in the task_list which can be run (ie, can get a socket,
 * if necessary) right now.
 */
static void
queue_run_what_we_can (void)
{
	GSList * l = NULL;
	debug_enter ("queue_run_what_we_can");

	/* walk through the queue and see if any tasks are waiting to be run. */
	l = task_list;
	while (l!=NULL)
	{
		Task *task = TASK(l->data);

		/* if it's not waiting to be run, leave it alone */
		if (queue_get_task_status(task) != QUEUE_TASK_STATUS_QUEUED)
		{
			debug0 (DEBUG_QUEUE, "task isn't waiting to be run");

		}

		/* if it doesn't need a socket, no problemo! */
		else if (!task->needs_socket)
		{
			debug1 (DEBUG_QUEUE,
				"task %d needs no socket; running it now",
				task);

			queue_run (task);
		}

		/* try to get a socket... */
		else if (!paused && max_socket_qty>0)
		{
			int i;
			queue_set_task_status (task, QUEUE_TASK_STATUS_CONNECTING);
			i = queue_get_socket_for_task (task);
			if (i==0)
			{
				debug1 (DEBUG_QUEUE,
					"task %d got a socket!",
					task);

				queue_run (task);
			}
			else
			{
				if (i==-2) /* connection failure */
				{



					status_item_emit_error (STATUS_ITEM(task), _("Connect Failure"));

					++task->tries;
					if (task->tries > max_tries)
						queue_task_abort (task);
				}

				queue_set_task_status (task, QUEUE_TASK_STATUS_QUEUED);
			}
		}

		if (todo_list != NULL)
		{
				queue_do_todo ();
				l = task_list;
		}
		else
		{
				l = l->next;
		}
	}

	debug_exit ("queue_run_what_we_can done");
}

/**
 * Destroy or recycle a socket when a task is done with it.
 * @param task the finished task.
 */
static void
socket_cleanup (Task* task)
{
	debug_enter ("socket_cleanup");

	if (task && task->sock)
	{
		PanSocket* socket = task->sock;
		GSList* l = NULL;
		SocketInfo * info = NULL;
		int server_socket_qty;
		int socket_qty;

		pthread_mutex_lock (&sock_lock);

		/* count the number of sockets this server has */
		socket_qty = 0;
		server_socket_qty = 0;
		for (l=sockets; l!=NULL; l=l->next) {
			SocketInfo * sockinfo = (SocketInfo*) l->data;
			if (sockinfo->socket == socket)
				info = sockinfo;
			if (sockinfo->server == task->server)
				++server_socket_qty;
			++socket_qty;
		}

		/* find this SocketInfo in our tables */
		g_assert (info!=NULL);

		/**
                 *  Recycle the socket if all are true
		 *  + there was no error
		 *  + the server is still online
		 *  + we don't have too many sockets for that server
		 *  + we don't have too many sockets overall
		 */
		if (!socket->error
		    && task->server->online
		    && socket_qty<=max_socket_qty
		    && server_socket_qty<=task->server->max_connections)
		{
		       	/* if socket's okay, reuse it */
			info->task = NULL;

			debug1 (DEBUG_QUEUE, "socket_cleanup: recycling socket %p", socket);
		}
		else
		{
			/* otherwise throw it away */
			pan_socket_putline (socket, "ccc\r\n");
			nntp_disconnect (NULL, socket);
			close (socket->sockfd);
			socket->sockfd = 0;
			pan_object_unref (PAN_OBJECT(socket));
			gui_inc_connection_size (-1);
			sockets = g_slist_remove (sockets, info);
			g_free (info);

			debug1 (DEBUG_QUEUE,
				"socket_cleanup: throwing away socket %p",
				socket);
		}
		task->sock = NULL;
		pthread_mutex_unlock (&sock_lock);
	}
	debug_exit ("socket_cleanup");
}

static gchar*
noop_describe (const StatusItem * item)
{
	return g_strdup (_("Sending 'stay connected' request"));
}

static gchar*
disconnect_describe (const StatusItem * item)
{
	return g_strdup (_("Disconnecting"));
}

/**
 * socket upkeep:  Cull out idle sockets that have been connected too long,
 * and send keepalive commands the rest of the idle sockets so that the
 * server doesn't drop us.
 */
static void
sockets_upkeep (void)
{
	GSList* l = NULL;
	GSList* cull = NULL;
	int socket_qty;
	debug_enter ("socket_upkeep");

	pthread_mutex_lock (&sock_lock);

	socket_qty = g_slist_length (sockets);

	for (l=sockets; l!=NULL; l=l->next)
	{
		gint age;
		gboolean destroy, idle, old, too_many, offline;
		SocketInfo * info;
		PanSocket * sock;
		const time_t current_time = time(0);

		info = (SocketInfo*) l->data;
		g_assert (info != NULL);
		sock = info->socket;
		g_assert (sock != NULL);

		/* calculate some information about the socket
		   to decide whether or not we want to keep it */
		age = current_time - sock->last_action_time;
		idle = info->task == NULL;
		old = age > info->server->idle_secs_before_timeout;
		too_many = socket_qty > max_socket_qty;
		offline = !info->server->online;
		destroy = idle && (old||too_many||offline);

		/* if it's idle and we're keeping it, send a keepalive msg.
		   Note that we don't let this affect the socket's
		   age, otherwise they would never grow old and close... */
		if (info->task==NULL && !destroy)
		{
			const time_t last_action_time = sock->last_action_time;
			StatusItem * status = status_item_new (noop_describe);
			status_item_set_active (status, TRUE);

			destroy = nntp_noop (STATUS_ITEM(status), sock) != TASK_SUCCESS;
			sock->last_action_time = last_action_time;

			status_item_set_active (status, FALSE);
			pan_object_unref (PAN_OBJECT(status));
		}

		/* either keep the socket or cull it from the herd */
		if (destroy) {
			--socket_qty;
			cull = g_slist_prepend (cull, info);
		}
	}

	debug1 (DEBUG_QUEUE, "sockets_upkeep: culling %u sockets",
		g_slist_length(cull));

	for (l=cull; l!=NULL; l=l->next)
	{
		const time_t current_time = time(0);
		SocketInfo * info = (SocketInfo*) l->data;
		PanSocket * sock = info->socket;
		StatusItem * status = STATUS_ITEM(status_item_new (disconnect_describe));
		const int age = current_time - sock->last_action_time;

		/* politely disconnect */
		status_item_set_active (status, TRUE);
		if (!sock->error)
			nntp_disconnect (status, sock);
		status_item_set_active (status, FALSE);
		log_add_va (LOG_INFO, _("Closing connection %p after %d seconds idle"), sock, age);

		/* destroy the socket */
		pan_object_unref (PAN_OBJECT(sock));
		gui_inc_connection_size (-1);
		sockets = g_slist_remove (sockets, info);

		/* cleanup local */
		pan_object_unref (PAN_OBJECT(status));
		g_free (info);
	}

	pan_warn_if_fail (g_slist_length(sockets) == socket_qty);

	pthread_mutex_unlock (&sock_lock);

	g_slist_free (cull);

	debug_exit ("socket_upkeep");
}

/**
 * Process the tasks that are waiting in the todo list
 */
static void
queue_do_todo (void)
{
	GSList * list = NULL;
	GSList * l = NULL;

	pthread_mutex_lock (&todo_mutex);
	list = todo_list;
	todo_list = NULL;
	pthread_mutex_unlock (&todo_mutex);

	for (l=list; l!=NULL; l=l->next)
	{
		QueueTodo* a = (QueueTodo*) l->data;
		switch (a->action)
		{
			case TODO_INSERT:
				real_queue_task_insert (a->task, a->index_1);
				break;
			case TODO_REMOVE:
				real_queue_task_remove (a->task);
				break;
			case TODO_MOVE:
				real_queue_task_move (a->task, a->index_1);
				break;
			case TODO_REQUEUE:
				real_queue_task_requeue_failed (a->task, a->index_1);
				break;
			case TODO_TASK_ABORT:
				real_queue_task_abort (a->task);
				break;
			case TODO_SET_MAX_SOCKET_QTY:
				real_queue_set_max_sockets (a->index_1);
				break;
			case TODO_SET_MAX_TRIES:
				real_queue_set_max_tries (a->index_1);
				break;
			case TODO_SHUTDOWN:
				real_queue_shutdown (a->index_1);
				l = NULL;
				break;
			case TODO_PAUSED:
				real_queue_set_paused (a->index_1);
				break;
			default:
				pan_warn_if_reached();
				break;
		}

		if (l == NULL)
			break;
	}

	/* throw the todo list away */
	g_slist_foreach (list, (GFunc)g_free, NULL);
	g_slist_free (list);
	list = NULL;
}

static void*
queue_mainloop (void* unused)
{
	const int TIMEOUT_SECS = 60;

	for (;;)
	{
		gint retcode;
		struct timeval now;
		struct timespec timeout;

		pthread_mutex_lock (&cond_mutex);
		gettimeofday (&now, NULL);
		timeout.tv_sec = now.tv_sec + TIMEOUT_SECS;
		timeout.tv_nsec = 0;
		retcode = 0;
		while (!work_to_do && retcode!=ETIMEDOUT) {
			retcode = pthread_cond_timedwait(&qcond, &cond_mutex, &timeout);
		}
		work_to_do = FALSE;
                pthread_mutex_unlock (&cond_mutex);

		if (retcode == ETIMEDOUT) {
			sockets_upkeep ();
		} else {
			queue_do_todo ();
                	queue_run_what_we_can ();
		}
	}

	return NULL;
}
