/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <gmime/gmime-utils.h>

#include <pan/base/acache.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/util-mime.h>

#include <pan/nntp.h>
#include <pan/prefs.h>
#include <pan/task.h>

gboolean online_prompt = TRUE;

/** deprecated, use nntp_command */
extern const char * sockwrite_err_msg;
extern const char * sockread_err_msg;

enum
{
	OK_GROUP				= 211,
	OK_AUTH					= 281,
	NEED_AUTHDATA				= 381,
	ERR_NOAUTH				= 480,
	ERR_AUTHREJ				= 482,
	ERR_ACCESS				= 502
};


/**
***
**/

static gint
nntp_get_response (PanSocket     * socket,
                   const gchar  ** setme_result)
{
	gint val;
	gint retval;
	const gchar * msg = NULL;
	debug_enter ("nntp_get_response");

	val = pan_socket_getline (socket, &msg);
	if (val)
	{
		if (msg == NULL)
			msg = sockread_err_msg;
		if (setme_result != NULL)
			*setme_result = msg;
		retval = val;
	}
	else
	{
		if (setme_result != NULL)
			*setme_result = msg;
		retval = atoi (msg);
	}

	debug_exit ("nntp_get_response");
	return retval;
}

gint
nntp_command (StatusItem           * status,
              PanSocket            * sock,
              const gchar         ** setme_response,
              const gchar          * command)
{
	int val;
	size_t len;
	debug_enter ("nntp_command");

	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (is_nonempty_string(command), -1);

	/**
	***  Send a command...
	**/

	len = strlen (command);
	if (len>=2 && command[len-2]=='\r' && command[len-1]=='\n')
	{
		val = pan_socket_putline (sock, command);
	}
	else /* not terminated in \r\n, so add */
	{
		gchar * tmp = g_strdup_printf ("%s\r\n", command);
		val = pan_socket_putline (sock, tmp);
		g_free (tmp);
	}
	if (val)
	{
		if (setme_response != NULL)
			*setme_response = sockwrite_err_msg;
		return val;
	}

	/**
	***  Get a response...
	**/

	val = nntp_get_response (sock, setme_response);

	if (val == ERR_NOAUTH)
	{
		val = nntp_authenticate (status, sock);
		if (val != TASK_SUCCESS)
			return -1;

		return nntp_command (status, sock, setme_response, command);
	}
	else return val;
}
 
gint
nntp_command_va (StatusItem           * status,
                 PanSocket            * sock,
                 const gchar         ** setme_result,
                 const gchar          * command_va,
                 ...)
{
	va_list args;
	char* line;
	int retval;

	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (command_va!=NULL, -1);

	va_start (args, command_va);
	line = g_strdup_vprintf (command_va, args);
	va_end (args);

	retval = nntp_command (status, sock, setme_result, line);

	g_free (line);
	return retval;
}

/**
***
**/

gint
nntp_disconnect (StatusItem   * status,
                 PanSocket    * socket)
{
	const gchar * reply;
	gint val;
	debug_enter ("nntp_disconnect");

	/* entry assertions */
	g_return_val_if_fail (socket!=NULL, TASK_FAIL_HOPELESS);

	/* try to disconnect */
	val = nntp_command (status, socket, &reply, "QUIT");
	if (val != 205) {
		status_item_emit_error_va (status, _("QUIT failed: %s"), reply);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* disconnected successfully */
	debug_exit ("nntp_disconnect");
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_handshake (StatusItem  * status,
                PanSocket   * socket,
                gboolean    * setme_posting_ok)
{
	gint val;
	const gchar * response = NULL;

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (socket!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme_posting_ok!=NULL, TASK_FAIL_HOPELESS);
	debug_enter ("nntp_handshake");

	/* get the server's handshake message */
	val = nntp_get_response (socket, &response);
	if (val==200 || val==201) { /* these values are expected */
		val = TASK_SUCCESS;
		log_add_va (LOG_INFO, _("Handshake: %s"), response);
	} else {
		status_item_emit_error_va (status, _("NNTP handshake failed: %s"), response);
		log_add_va (LOG_ERROR, _("Handshake failed: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS; 
	}

	/* can we post? */
	*setme_posting_ok = val==200;

	/* yes, we also support authentication on demand, but it appears
	   many servers are happier if we ask for authentication up-front. */
	if (socket->need_auth)
		val = nntp_authenticate (status, socket);

	/* Achim Settelmeir reports that his local INN doesn't accept GROUP
	   without MODE READER first */
	nntp_can_post (NULL, socket, setme_posting_ok);


	debug_exit ("nntp_handshake");
	return val;
}

gint
nntp_authenticate (StatusItem     * status,
                   PanSocket      * sock)
{
	gint val;
	const gchar * response = NULL;
	debug_enter ("nntp_authenticate");

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock->nntp_username!=NULL, TASK_FAIL_HOPELESS);

	/* set the username */
	val = nntp_command_va (status, sock, &response, "AUTHINFO USER %s", sock->nntp_username);
	if (val!=OK_AUTH && val!=NEED_AUTHDATA) {
		if (status != NULL)
			status_item_emit_error_va (status, _("Authentication failed: %s"), response);
		return TASK_FAIL;
	}

	/* set the password, if required */
	if (val==NEED_AUTHDATA) {
		if (!is_nonempty_string(sock->nntp_password)) {
			if (status != NULL)
				status_item_emit_error_va (status, _("Authentication failed: need a password"));
			return TASK_FAIL;
		}
		val = nntp_command_va (status, sock, &response, "AUTHINFO PASS %s", sock->nntp_password);
	}

	/* if we failed, emit an error */	
	if (val!=OK_AUTH) {
		if (status != NULL)
			status_item_emit_error_va (status, _("Authentication failed: %s"), response);
		return TASK_FAIL;
	}

	debug_exit ("nntp_authenticate");
	return TASK_SUCCESS;
}

gint
nntp_can_post (StatusItem   * status,
               PanSocket    * sock,
               gboolean     * setme_can_post)
{
	gint val;
	gint retval = 0;
	const gchar * response = NULL;
	debug_enter ("nntp_can_post");

	/* entry assertions */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme_can_post!=NULL, TASK_FAIL_HOPELESS);

	/* ask the server if we can post or not */
	val = nntp_command (status, sock, &response, "MODE READER");
	switch (val) {
		case 200:
			*setme_can_post = TRUE;
			retval = 0;
			break;
		case 201:
			*setme_can_post = FALSE;
			retval = 0;
			break;
		default:
			*setme_can_post = FALSE;
			if (status != NULL)
				log_add_va (LOG_ERROR, _("MODE READER check failed: %s"),  response);
			retval = val;
			break;
	}

	debug_exit ("nntp_can_post");
	return retval;
}

/**
***
**/

gint
nntp_set_group (StatusItem   * status,
                PanSocket    * sock,
                const char   * group_name)
{
	gint val;
	const gchar * response = NULL;
	debug_enter ("nntp_set_group");

	/* sanity checks */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(group_name), TASK_FAIL_HOPELESS);

	/* we're already there */
	if (!pan_strcmp (sock->nntp_group_name, group_name))
		return TASK_SUCCESS;

	/* change to that group */
	val = nntp_command_va (status, sock, &response, "GROUP %s", group_name);
	if (val != OK_GROUP) {
		status_item_emit_error_va (status, _("Unable to set group `%s': %s"), group_name, response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* update this socket's current group */
	replace_gstr (&sock->nntp_group_name, g_strdup(group_name));
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_post (StatusItem    * status,
           PanSocket     * sock,
           const gchar   * msg)
{
	gint val;
	const gchar * response;
	debug_enter ("nntp_post");

	/* entry assertions */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(msg), TASK_FAIL_HOPELESS);

	/* if we're in mute mode, don't post */
	if (pan_mute)
	{
		g_message ("Mute: Your Message will not actually be posted.");
		printf ("\n\n\nYour message\n%s<end of message>\n\n\n", msg);
		fflush (NULL);
	}
	else
	{
		/* tell the server we want to post */
		val = nntp_command (status, sock, &response, "POST");
		if (val != 340) {
			status_item_emit_error_va (status, _("Posting failed: %s"), response);
			return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
		}

		/* post the article */
		val = nntp_command_va (status, sock, &response, "%s\r\n.\r\n", msg);
		if (val == 240) {
			log_add_va (LOG_INFO, _("Posting complete: %s"), response);
		} else {
			status_item_emit_error_va (status, "Posting failed: %s", response);
			return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
		} 
	}

	/* if we made it this far, we succeeded! */
	debug_exit ("nntp_post");
	return TASK_SUCCESS;
}



gint
nntp_noop (StatusItem    * status,
           PanSocket     * socket)
{
	gboolean can_post = FALSE;
	debug_enter ("nntp_noop");

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (socket!=NULL, TASK_FAIL);

	/* do work */
	status_item_emit_status_va (status, _("\"stay connected\" sent to %s"), socket->host); 
	return nntp_can_post (status, socket, &can_post);
}

/**
***
**/

static gboolean
init_header_if_match (Article       * a,
                      const gchar   * header,
                      const gchar   * header_key,
                      const gchar   * article_key,
                      HeaderAction    action)
{
	gboolean match = FALSE;
	size_t len;

	/* entry assertions */
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (is_nonempty_string(header_key), FALSE);
	g_return_val_if_fail (is_nonempty_string(article_key), FALSE);
	g_return_val_if_fail (header!=NULL, FALSE);

	/* if this is the header we're looking for, keep it. */
	len = strlen (header_key);
	if (!strncmp (header, header_key, len))
	{
		const gchar * val = header + len;
		if (!pan_header_is_8bit_encoded (val))
			article_init_header (a, article_key, val, action);
		else {
			gchar * b = g_mime_utils_8bit_header_decode ((const guchar*)val);
			article_init_header (a, article_key, b, action);
			g_free (b);
		}
		match = TRUE;
	}

	return match;
}

gint
nntp_article_download (StatusItem      * status,
                       PanSocket       * sock,
                       Group           * group,
                       Article         * article,
                       const gboolean  * abort,
                       gint              verbose,
                       gboolean          checkout)
{
	const gchar * response;
	gboolean getting_headers;
	gint retval;
	gint val;
	GString * buf;
	GString * line;
	debug_enter ("nntp_article_download");

	/* sanity clause */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (group_is_valid(group), TASK_FAIL);
	g_return_val_if_fail (article_is_valid(article), TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL);

	/* let the user know what we're doing */
	if (verbose & NNTP_VERBOSE_TEXT)
		status_item_emit_status (status, article_get_subject(article));

	/* request the article from the server */
	val = nntp_command_va (status, sock, &response, "ARTICLE %lu", article->number);
	if (val != 220)
	{
		gchar * pch = g_strdup_printf (_("Getting article `%s' (Message-ID: %s) body failed: %s"),
		                               article_get_subject(article),
		                               article_get_message_id(article),
		                               response);
		status_item_emit_error (status, pch);
		g_free (pch);

		if (val==-1)
			return TASK_FAIL;
		articles_add_flag (&article, 1, STATE_ERROR);
		return TASK_FAIL_HOPELESS;
	}

	/* try to read the article... */
	getting_headers = TRUE;
	line = g_string_sized_new (80);
	buf = g_string_sized_new (80*400); /* prealloc space for ~400 lines */
	retval = TASK_SUCCESS;
	if (verbose & NNTP_VERBOSE_INIT_STEPS)
		status_item_emit_init_steps (status, article->linecount);

	for (;;)
	{
		const gchar * pch;

		/* check for end cases: user abort, socket error, and success */
		if (*abort || pan_socket_getline (sock, &response)) {
			retval = TASK_FAIL;
			break;
		}

		if (!is_nonempty_string(response))
			continue;

		if (!strncmp(response, ".\r\n", 3)) {
			retval = TASK_SUCCESS;
			break;
		}

		/* strip out the \r */
		g_string_assign (line, response);
		while ((pch = strstr (line->str, "\r\n")))
			g_string_erase (line, pch-line->str, 1);

		/* rfc 977: 2.4.1 */
		if (line->len>=2 && line->str[0]=='.' && line->str[1]=='.')
			g_string_erase (line, 0, 1);

		/* save the line */
		g_string_append (buf, line->str);
		if (verbose & NNTP_VERBOSE_NEXT_STEP)
			status_item_emit_next_step (status);

		/* save the headers that we want to save */
		if (getting_headers)
		{
			if (line->len==1 && !strcmp(line->str, "\n")) /* header/body separator */
				getting_headers = FALSE;
			else 
			{
				Article * a = article;
				gboolean matched = FALSE;

				/* trim linefeed */
				g_strstrip (line->str);
				line->len = strlen(line->str);

				/* check for headers to store in Article */
				if (!matched)
					matched = init_header_if_match (a, line->str,
					                                "Newsgroups: ",
					                                HEADER_NEWSGROUPS,
					                                DO_CHUNK_SHARE);
				if (!matched)
					matched = init_header_if_match (a, line->str,
					                                "Reply-To: ",
					                                HEADER_REPLY_TO,
					                                DO_CHUNK_SHARE);
				if (!matched)
					matched = init_header_if_match (a, line->str,
					                                "Followup-To: ",
					                                HEADER_FOLLOWUP_TO,
					                                DO_CHUNK_SHARE);
				if (!matched)
					matched = init_header_if_match (a, line->str,
					                                "Xref: ",
					                                HEADER_XREF,
					                                DO_CHUNK);
			}
		}
	}

	/* save the message */
	if (retval == TASK_SUCCESS) {
        	acache_set_message (article_get_message_id(article), buf->str, checkout);
		articles_remove_flag (&article, 1, STATE_DOWNLOAD_FLAGGED);
	}

	g_string_free (buf, TRUE);
	g_string_free (line, TRUE);
	debug_exit ("nntp_article_download");
	return retval;
}


/**
***
**/

gint
nntp_get_group_info (StatusItem       * status,
                     PanSocket        * sock,
                     const char       * group_name,
                     gulong           * article_qty,
                     gulong           * low_num,
                     gulong           * high_num,
                     const gboolean   * abort)
{
	gint val;
	const gchar * response = NULL;
	debug_enter ("nntp_get_group_info");

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (is_nonempty_string(group_name), TASK_FAIL);
	g_return_val_if_fail (article_qty!=NULL, TASK_FAIL);
	g_return_val_if_fail (low_num!=NULL, TASK_FAIL);
	g_return_val_if_fail (high_num!=NULL, TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL);

	/* change to that group */
	val = nntp_command_va (status, sock, &response, "GROUP %s", group_name);
	if (val != OK_GROUP) {
		status_item_emit_error_va (status, _("Unable to set group `%s': %s"), group_name, response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* success; parse the results */
	sscanf (response, "%*d %lu %lu %lu", article_qty, low_num, high_num);
	replace_gstr (&sock->nntp_group_name, g_strdup(group_name));
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_download_headers (StatusItem       * status,
                       PanSocket        * sock,
                       Group            * group,
                       gulong             low,
                       gulong             high,
                       const gboolean   * abort,
                       const char       * progress_fmt,
                       GPtrArray        * setme)
{
	const char * buffer;
	const char * response;
	gulong total;
	gulong first;
	gulong last;
	gboolean read_status = 0;
	gint result;
	gint val;
	GString * buf;
	debug_enter ("nntp_download_headers");

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (group_is_valid(group), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(progress_fmt), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme!=NULL, TASK_FAIL_HOPELESS);

	/**
	***  Get up-to-date information about this group
	**/

	total = first = last = 0;
	result = nntp_get_group_info (status,
	                              sock,
	                              group->name,
	                              &total,
	                              &first,
	                              &last,
	                              abort);
	if (result != TASK_SUCCESS)
		return result;


	/**
	***  If no articles to get, then we're already done
	**/

	if (total == 0) {
		const gchar * n = group_get_readable_name (group);
		status_item_emit_status_va (status, _("No articles found for group \"%s\""), n);
		return TASK_SUCCESS;
	}

	/**
	***  Tell the server that we want a series of article headers...
	**/

	val = *abort ? -1 : nntp_command_va (status, sock, &response, "XOVER %lu-%lu", low, high);
	if (val != 224) {
		status_item_emit_status_va (status, _("Getting header list failed: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/**
	***  Walk through all the articles headers that the server spits back
	**/

	buf = g_string_new (NULL);
	pan_g_ptr_array_reserve (setme, setme->len + (high-low));
	while (!*abort && !((read_status = pan_socket_getline (sock, &buffer))))
	{
		gint part = 0;
	        gint parts = 0;
		const gchar * pch;
		const gchar * s;
		Article * a;
		const gchar * march = NULL;
		const gchar * tok = NULL;
		int tok_len = 0;

		/* handle end-of-list */
		if (!strncmp(buffer, ".\r\n", 3))
			break;

		/* create the article data */
		a = article_new (group);

		/* setup for parsing */
		g_string_assign (buf, buffer);
		march = buf->str;

		/* get article number */
		a->number = get_next_token_ulong (march, '\t', &march);

		/* get subject */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((gchar*)tok)[tok_len] = '\0';
			if (!pan_header_is_8bit_encoded (tok))
				article_init_header (a, HEADER_SUBJECT, tok, DO_CHUNK_SHARE);
			else {
				gchar * dec = g_mime_utils_8bit_header_decode ((const guchar*)tok);
				article_init_header (a, HEADER_SUBJECT, dec, DO_CHUNK_SHARE);
				g_free (dec);
			}
		}

		/* get author */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((gchar*)tok)[tok_len] = '\0';
			if (is_nonempty_string(tok))
				article_init_author_from_header (a, tok);
		}

		/* get date */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((gchar*)tok)[tok_len] = '\0';
			a->date = is_nonempty_string(tok)
				? g_mime_utils_header_decode_date (tok, NULL)
				: (time_t)0;
		}

		/* get message id */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((gchar*)tok)[tok_len] = '\0';
			if (is_nonempty_string(tok) && *tok=='<')
				article_init_header (a, HEADER_MESSAGE_ID, tok, DO_CHUNK);
		}

		/* get references */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			if (tok_len>0 ) {
				((gchar*)tok)[tok_len] = '\0';
				if (is_nonempty_string(tok) && *tok=='<')
					article_init_header (a, HEADER_REFERENCES, tok, DO_CHUNK_SHARE);
			}
		}

		/* get unused field: size */
		skip_next_token (march, '\t', &march);

		/* get linecount */
		a->linecount = get_next_token_int (march, '\t', &march);

		/* get crossref */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((gchar*)tok)[tok_len] = '\0';
			if (tok_len!=0 || strncmp(tok,"Xref: ", 6))
				a->crosspost_qty = 1;
			else {
				const gchar * foo;
				a->crosspost_qty = 0;
				for (foo=tok+6; *foo; ++foo)
					if (*foo==':')
						++a->crosspost_qty;
				article_init_header (a, HEADER_XREF, tok+6, DO_CHUNK);
			}
		}

		/**
		***  Validity Checking
		**/

		if (!article_is_valid (a)) {
			status_item_emit_error_va (status, 
				_("Corrupt header skipped: %s"), buffer);
			continue;
		}


		/* Look for the (n/N) or [n/N] construct in subject lines,
		 * starting at the end of the string and working backwards */
		part = 0;
		parts = 0;
		s = a->subject;
		pch = s + strlen(s) - 1;
		while (pch != s)
		{
			/* find the ']' of [n/N] */
			--pch;
			if ((pch[1]!=')' && pch[1]!=']') || !isdigit((guchar)*pch))
				continue;

			/* find the '/' of [n/N] */
			while (s!=pch && isdigit((guchar)*pch))
				--pch;
			if (s==pch || (*pch!='/' && *pch!='|'))
				continue;

			/* N -> parts */
			parts = atoi (pch+1);
			--pch;

			/* find the '[' of [n/N] */
			while (s!=pch && isdigit((guchar)*pch))
				--pch;
			if (s==pch || (*pch!='(' && *pch!='[')) {
				parts = 0;
				continue;
			}

			/* n -> part */
			part = atoi (pch+1);
			break;
		}

		/* if not a multipart yet, AND if it's a big message, AND
		   it's either in one of the pictures/fan/sex groups or it
		   has commonly-used image names in the subject, guess it's
		   a single-part binary */
		if (!parts
			&& a->linecount>400
			&& (((pan_strstr(group->name,"binaries")
					|| pan_strstr(group->name,"fan")
				        || pan_strstr(group->name,"mag")
					|| pan_strstr(group->name,"sex")))
				|| ((pan_strstr(s,".jpg") || pan_strstr(s,".JPG")
					|| pan_strstr(s,".jpeg") || pan_strstr(s,".JPEG")
					|| pan_strstr(s,".gif") || pan_strstr(s,".GIF")
					|| pan_strstr(s,".tiff") || pan_strstr(s,".TIFF")))))
			part = parts = 1;

		/* Verify Multipart info */
		if ((parts>=1) && (part<=parts)) {
			/* yay, we found one! */
			a->parts = parts;
			a->part = part;
		}
		else {
			a->parts = 0;
			a->part = 0;
		}

		/* Add the article to the article list */
		g_ptr_array_add (setme, a);

		/* update the count & progress ui */
		status_item_emit_next_step (status);
		if (!(setme->len % 57))
			status_item_emit_status_va (status, progress_fmt, setme->len, (high-low));
	}
	g_string_free (buf, TRUE);

	if (read_status!=0)
		status_item_emit_error (status, sockread_err_msg);
	if (read_status!=0 || *abort)
		return TASK_FAIL;

	/* if we've made it this far, we succeeded! */
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_download_bodies (StatusItem       * status,
                      Group            * group,
                      PanSocket        * sock,
                      const gboolean   * abort,
                      const GPtrArray  * const articles,
                      gint             * index,
                      gboolean           checkout,
                      gboolean           abort_if_one_fails)
{
	gint result;
	gint i;
	gint lines_so_far;
	gint lines_total;
	debug_enter ("nntp_download_bodies");

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (group_is_valid(group), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (articles!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (index!=NULL, TASK_FAIL_HOPELESS);

	/* change to the group we want */
	if (*abort)
		return TASK_FAIL;
	result = nntp_set_group (status, sock, group->name);
	if (result != TASK_SUCCESS)
		return result;

	/* status item */
	lines_so_far = lines_total = 0;
	for (i=0; i<articles->len; ++i) {
		Article * article = ARTICLE(g_ptr_array_index(articles,i));
		lines_total += article->linecount;
		if (i<*index)
			lines_so_far += article->linecount;
	}
	status_item_emit_init_steps (status, lines_total);
	status_item_emit_set_step (status, lines_so_far);

	/* download the articles */
	for (; *index<articles->len && !*abort; ++(*index))
	{
		const gchar * pch;
		Article * article = ARTICLE(g_ptr_array_index(articles,*index));

		status_item_emit_status_va (status, _("Getting %d of %d: \"%s\""),
		                                    1+*index, articles->len,
		                                    article_get_subject(article));

		/* download if we have to. */
		pch = article_get_message_id (article);
		if (article_has_body (article))
		{
			result = TASK_SUCCESS;
			status_item_emit_inc_step (status, article->linecount);

			/* since it's cached locally, we can unflag */
			articles_remove_flag (&article, 1, STATE_DOWNLOAD_FLAGGED);

			if (checkout)
				acache_file_checkout (&article->message_id, 1);
		}
		else
		{
			result = nntp_article_download (status,
			                                sock,
			                                group,
			                                article,
			                                abort,
			                                NNTP_VERBOSE_NEXT_STEP,
			                                checkout);

			if (result!=TASK_SUCCESS && abort_if_one_fails)
				return result;
		}
	}

	return TASK_SUCCESS;
}


gint
nntp_cancel (StatusItem     * status,
             const Article  * article,
             PanSocket      * sock)
{
	GString * msg;
	gchar * author;
	const gchar * newsgroups;
	gint retval;
	const gchar * msg_id = article_get_message_id (article);
	debug_enter ("nntp_cancel");

	/* sanity checks */
	g_return_val_if_fail (sock != NULL, TASK_FAIL);
	g_return_val_if_fail (status != NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (article_is_valid(article), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(msg_id), TASK_FAIL_HOPELESS);
	newsgroups = article_get_header (article, HEADER_NEWSGROUPS);
	g_return_val_if_fail (is_nonempty_string(newsgroups), TASK_FAIL_HOPELESS);

	/* let the user know what we're doing */
	status_item_emit_status_va (status, _("Cancelling message \"%s\""), msg_id);

	/* build the message to post */
	msg = g_string_new (NULL);
	author = article_get_author_str(article);
        replace_gstr (&author, g_mime_utils_8bit_header_encode((const guchar*)author));
	g_string_sprintfa (msg, "From: %s\r\n", author);
	g_string_sprintfa (msg, "Newsgroups: %s\r\n", newsgroups);
	g_string_sprintfa (msg, "Subject: cancel %s\r\n", msg_id);
	g_string_sprintfa (msg, "Control: cancel %s\r\n", msg_id);
	g_string_sprintfa (msg, "\r\n");
	g_string_sprintfa (msg, "Ignore\r\nArticle canceled by Pan %s\r\n", VERSION);

	/* post the cancel message */
	retval = nntp_post (status, sock, msg->str);

	/* cleanup */
	g_free (author);
	g_string_free (msg, TRUE);

	debug_exit ("nntp_cancel");
	return retval;
}
