/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <gnome.h>
#include <gdk_imlib.h>

#include <ctype.h>

#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <pan/base/serverlist.h>

#include <pan/globals.h>
#include <pan/pan.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/util.h>

#include <pan/identities/identity.h>
#include <pan/identities/identity-manager.h>

#include <pan/dialogs/dialogs.h>
#include <pan/xpm/pan_druid.xpm>

typedef struct
{
	GtkWidget * full_name;
	GtkWidget * organization;

	GtkWidget * email;
	GtkWidget * smtp_server;
	GtkWidget * smtp_port;
	
	GtkWidget * server_address;
	GtkWidget * server_port;
	GtkWidget * auth_toggle;

	GtkWidget * server_username_label;
	GtkWidget * server_username;
	GtkWidget * server_password_label;
	GtkWidget * server_password;

	GtkWidget * profile_name;
}
Druid;

/***
****
****  HOSTNAME
****
***/


/* Returns the fully qualified domain name */
static char*
get_fqdn (void)
{
	char name[512] = { '\0' };
	char line[1025];
	char fqdn[1024] = { '\0' };
	char * cp;
	char * domain = NULL;
	FILE * inf;
	struct hostent * hp;
	struct in_addr in;

	if (gethostname(name, MAXHOSTNAMELEN))
		return NULL;

	if (isdigit((guchar)*name)) {
		in.s_addr = inet_addr (name);
		if ((hp = gethostbyaddr((char *) &in.s_addr, 4, AF_INET)))
			in.s_addr = (*hp->h_addr);
		return g_strdup((hp && strchr(hp->h_name, '.') ? hp->h_name : (char *) inet_ntoa(in)));
	}
	if ((hp = gethostbyname(name)) && !strchr(hp->h_name, '.'))
		if ((hp = gethostbyaddr(hp->h_addr, hp->h_length, hp->h_addrtype)))
			in.s_addr = (*hp->h_addr);

	sprintf(fqdn, "%s", hp ? strchr(hp->h_name, '.')
		? hp->h_name : (char *) inet_ntoa(in)
		: "");
	if (!*fqdn || (fqdn[strlen(fqdn) - 1] <= '9')) {
		*fqdn = '\0';
		inf = fopen("/etc/resolv.conf", "r");
		if (inf) {
			while (fgets(line, 1024, inf)) {
				line[1024] = '\0';
				g_strchug(line);
				g_strchomp(line);
				if (strncmp(line, "domain ", 7) == 0) {
					domain = line + 7;
					break;
				}
				if (strncmp(line, "search ", 7) == 0) {
					domain = line + 7;
					cp = strchr(domain, ' ');
					if (cp)
						*cp = '\0';
					break;
				}
			}
			if (domain)
				sprintf(fqdn, "%s.%s", name, domain);
			fclose(inf);
		}
	}
	return g_strdup (fqdn);
}

static gchar*
get_default_author_address (void)
{
	gchar * email = gnome_config_get_string ("/Pan/User/Email");

	if (email != NULL)
		g_strstrip (email);

	/* if no email in prefs, do the best we can */
	if (!is_nonempty_string(email)) {
		gchar * fqdn = get_fqdn ();
		replace_gstr (&email,
			g_strdup_printf ("%s@%s", g_get_user_name(), fqdn));
		g_free (fqdn);
	}

	return email;
}

/***
***/

static void
druid_finish (GnomeDruidPage  * page,
              GnomeDruid      * druid,
              gpointer          window)
{
	Server * server = NULL;
	GString * buf = g_string_new (NULL);
	Druid * ddata = gtk_object_get_data (GTK_OBJECT (window), "data");
	gint server_port = atoi(gtk_entry_get_text (GTK_ENTRY(ddata->server_port)));
	Identity * id = NULL;

	gint auth_toggle = GTK_TOGGLE_BUTTON (ddata->auth_toggle)->active;
	const gchar * profile_name = gtk_entry_get_text (GTK_ENTRY(ddata->profile_name));
	const gchar * server_address = gtk_entry_get_text (GTK_ENTRY(ddata->server_address));
	const gchar * server_username = gtk_entry_get_text(GTK_ENTRY(ddata->server_username));
	const gchar * server_password = gtk_entry_get_text(GTK_ENTRY(ddata->server_password));
	const gchar * full_name = gtk_entry_get_text (GTK_ENTRY(ddata->full_name));
	const gchar * email = gtk_entry_get_text (GTK_ENTRY(ddata->email));
	const gchar * organization = gtk_entry_get_text (GTK_ENTRY(ddata->organization));
	const gchar * smtp_server = gtk_entry_get_text (GTK_ENTRY(ddata->smtp_server));
	const gchar * smtp_port = gtk_entry_get_text (GTK_ENTRY(ddata->smtp_port));

	if (profile_name && server_address)
	{
		server = server_new ();
                server->name = g_strdup (profile_name);
		server->address = g_strdup (server_address);
                server->port = server_port;
                server->need_auth = auth_toggle;
                server->username = auth_toggle ? g_strdup (server_username) : NULL;
		server->password = auth_toggle ? g_strdup (server_password) : NULL;
                server->gen_msgid = TRUE;
		server->max_connections = 4;
		server->idle_secs_before_timeout = 180;
		serverlist_add_server (server);
		gnome_config_set_string ("Pan/State/Server", profile_name);
	}

	/* save user stats */
	gnome_config_set_string ("/Pan/Mail/smtp_port", smtp_port);
	gnome_config_set_string ("/Pan/Mail/smtp_address", smtp_server);
	mail_server_address = g_strdup (smtp_server);

	id = identity_new ();
	if (id != NULL)
	{
		gchar * str = g_strdup_printf ("%s <%s>", full_name, email);

		identity_set_name (id, str);
		identity_set_author_real (id, full_name);
		identity_set_author_addr (id, email);
		identity_set_organization (id, organization);

		identity_manager_add_identity (id);

		pan_object_unref (PAN_OBJECT (id));
		g_free (str);
	}

	/* start Pan proper */
	gnome_config_sync ();
	gui_construct (NULL);
	gtk_widget_show_all (Pan.window);
	serverlist_set_active_server (server);

	/* cleanup */
	g_free (ddata);
	g_string_free (buf, TRUE);
	gtk_widget_destroy (GTK_WIDGET (window));
}

static void
druid_cancel (GtkObject *object)
{
	gtk_widget_destroy (GTK_WIDGET (object));
	pan_shutdown();
}

/***
***/

static GtkWidget*
druid_name_page (GtkWidget *vbox, Druid *ddata)
{
	GtkWidget * w;
	GtkWidget * table;

	table = gtk_table_new (5, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	w = gtk_label_new (_("When you post a message to a newsgroup, your display name will appear in the From field.  Type your name in as you would like it to appear."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 0, 1, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("Display Name:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->full_name = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 1, 2, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);
	gtk_entry_set_text (GTK_ENTRY(w), g_get_real_name());

	w = gtk_label_new (_("For example: John Doe"));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 2, 3, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("If you belong to an organization and would like for your organization's name to appear with your posts, type in your organization name below.  This information is optional, you may leave it blank."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 3, 4, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("Organization:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 4, 5, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->organization = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 4, 5, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	return table;
}


/*
 * Page #2:
 *  "Internet News E-mail Address"
 */
static GtkWidget*
druid_email_page (Druid *ddata)
{
	GtkWidget * table;
	GtkWidget * w;
	gchar * email;
	gchar * smtp_server;
	gchar * smtp_port;
	gchar * fqdn;

	table = gtk_table_new (6, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	w = gtk_label_new (_("People can reply to your posts by sending you an e-mail message at the following address.") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 0, 1, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("E-mail address:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->email = gtk_entry_new ();
	email = get_default_author_address ();
	gtk_entry_set_text (GTK_ENTRY(w), email);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 1, 2, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("For example: user@host.com"));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 2, 3, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("Please enter the name of the mail server to use when sending e-mail messages."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 3, 4, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("SMTP Server:") );
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 4, 5, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->smtp_server = gtk_entry_new ();
	fqdn = get_fqdn ();
	smtp_server = g_strdup_printf ("%s", fqdn);
	gtk_entry_set_text (GTK_ENTRY(w), smtp_server);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 4, 5, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);
	
	w = gtk_label_new (_("SMTP Port:") );
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 5, 6, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->smtp_port = gtk_entry_new ();
	smtp_port = g_strdup ("25");
	gtk_entry_set_text (GTK_ENTRY(w), smtp_port);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 5, 6, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	/* cleanup */
	g_free (fqdn);
	g_free (email);
	g_free (smtp_port);
	g_free (smtp_server);
	return table;
}

static void
password_needed_cb (GtkToggleButton * tb, Druid * druid)
{
	const gboolean b = tb->active;
	gtk_widget_set_sensitive (druid->server_username, b);
	gtk_widget_set_sensitive (druid->server_password, b);
	gtk_widget_set_sensitive (druid->server_username_label, b);
	gtk_widget_set_sensitive (druid->server_password_label, b);
}

static gchar*
get_default_nntp_server (void)
{
	gchar * retval = NULL;

	if (!retval) {
		const gchar * cpch = g_getenv ("NNTPSERVER");
		if (is_nonempty_string (cpch))
			retval = g_strdup (cpch);
	}

	if (!retval) {
		gchar * fqdn = get_fqdn ();
		if (is_nonempty_string (fqdn)) {
			const gchar * dot = strchr (fqdn, '.');
			if (dot != NULL) {
				retval = g_strdup_printf ("news.%s", dot+1);
			}
		}
		g_free (fqdn);
	}

	return retval;
}

static gchar*
get_default_profile_name (void)
{
	gchar * retval = NULL;

	if (!retval) {
		gchar * pch = get_default_nntp_server ();
		if (is_nonempty_string (pch)) {
			gchar * lastdot = strrchr (pch, '.');
			if (lastdot != NULL)
				*lastdot = '\0';
			lastdot = strrchr (pch, '.');
			if (lastdot != NULL)
				retval = g_strdup (lastdot+1);
		}
		g_free (pch);
	}

	if (!retval) {
		retval = g_strdup (_("default"));
	}

	return retval;
}

/*
 * Page #3:
 *  "Internet News Server"
 */
static GtkWidget*
druid_server_page (Druid *ddata)
{
	GtkWidget * w;
	GtkWidget * table;
	gchar * default_nntp_server;

	table = gtk_table_new (7, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	w = gtk_label_new(_("Type the name of your Internet news server."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 0, 1, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("News (NNTP) server:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->server_address = gtk_entry_new();
	default_nntp_server = get_default_nntp_server ();
	if (is_nonempty_string (default_nntp_server))
		gtk_entry_set_text (GTK_ENTRY(w), default_nntp_server);
	else
		gtk_entry_set_text (GTK_ENTRY(w), "news");
	g_free (default_nntp_server);
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, 1, 2, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("Port:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 2, 3, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->server_port = gtk_entry_new();
	gtk_entry_set_text (GTK_ENTRY(w), "119");
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 2, 3, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("If you must log on to your news (NNTP) server, then select the check box below."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
 	gtk_table_attach (GTK_TABLE (table), w, 0, 2, 3, 4, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->auth_toggle = gtk_check_button_new_with_label (_("My server requires a password."));
	gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(password_needed_cb), ddata);
 	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 4, 5, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->server_username_label = w = gtk_label_new (_("Account name:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
 	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 5, 6, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->server_username = w = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, 5, 6, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->server_password_label = w = gtk_label_new (_("Password:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
 	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 6, 7, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->server_password = w = gtk_entry_new ();
	gtk_entry_set_visibility(GTK_ENTRY(w), FALSE);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 6, 7, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	password_needed_cb (GTK_TOGGLE_BUTTON(ddata->auth_toggle), ddata);

	return table;
}


/*
 * Page #4:
 *  "Save Profile"
 */
static GtkWidget *
druid_profile_page (Druid *ddata)
{
	gchar * pch;
	GtkWidget * w = NULL;
	GtkWidget * table = NULL;

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	w = gtk_label_new (_("Type in the name you would like Pan to show when referring to this profile you have just setup.  You can name the profile anything you want to."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
 	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 0, 1, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("Profile Name:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = ddata->profile_name = gtk_entry_new();
	pch = get_default_profile_name ();
	gtk_entry_set_text (GTK_ENTRY(w), pch);
	g_free (pch);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 1, 2, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	w = gtk_label_new (_("For example: UUNET Server"));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 2, 3, GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	return table;
}


/*---[ druid_spawn ]--------------------------------------------------
 * spawn the first-run wizard
 *--------------------------------------------------------------------*/
void
dialog_newuser (void)
{
	GtkWidget * window;
	GtkWidget * druid;
	GtkWidget * page_start;
	GtkWidget * page_a;
	GtkWidget * page_b;
	GtkWidget * page_c;
	GtkWidget * page_d;
	GtkWidget * page_finish;
	GdkImlibImage * logo;
	GdkImlibImage * watermark = NULL;
	Druid * ddata;
	
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER_ALWAYS);
	gtk_window_set_title (GTK_WINDOW (window), _("Pan Setup Wizard"));

	druid = gnome_druid_new ();
	gnome_druid_set_show_finish (GNOME_DRUID(druid), TRUE);
	ddata = g_new (Druid, 1);

	/* create the logo iamge */
	logo = NULL;
	if (1) {
		gchar * fname = gnome_pixmap_file ("gnome-logo-icon.png");
		if (fname != NULL) {
			logo = gdk_imlib_load_image (fname);
			g_free (fname);
		}
	}

	watermark = gdk_imlib_create_image_from_xpm_data (pan_druid_xpm);

	gtk_object_set_data (GTK_OBJECT(window), "data", ddata);

	/* Initialize the druid pages */
	page_start = gnome_druid_page_start_new_with_vals (
		_("Welcome to Pan"),
		_("Thank you for trying Pan!\n \nIf you are new to Usenet,\nhttp://pan.rebelbase.com/intro.php\nis a starting point to see what Usenet is,\nwhere to find a free news server,\nand how to use Pan.\n \nSince this is your first time running Pan,\n we'll need to gather some information.\n"),
		logo,
		watermark);
	page_a = gnome_druid_page_standard_new_with_vals (_("Your Name"), logo);
	page_b = gnome_druid_page_standard_new_with_vals (_("E-mail"), logo);
	page_c = gnome_druid_page_standard_new_with_vals (_("News Server"), logo);
	page_d = gnome_druid_page_standard_new_with_vals (_("Save Profile"), logo);
	page_finish = gnome_druid_page_finish_new_with_vals (
		_("Congratulations"),
		_("You have successfully entered all of\nthe information required to set up Pan.\n\nTo save these settings, click Finish."),
		logo,
		watermark);

	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_a)->vbox),
	                    druid_name_page (GNOME_DRUID_PAGE_STANDARD(page_a)->vbox, ddata), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_b)->vbox),
	                    druid_email_page (ddata), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_c)->vbox),
	                    druid_server_page (ddata), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_d)->vbox),
	                    druid_profile_page (ddata), TRUE, TRUE, 0);

	gtk_signal_connect (GTK_OBJECT (page_finish), "finish",
	                    GTK_SIGNAL_FUNC (druid_finish), window);

	gtk_signal_connect_object (GTK_OBJECT (druid), "cancel",
	                           GTK_SIGNAL_FUNC (druid_cancel),
	                           GTK_OBJECT (window));

	/* tie it all together */
	gtk_container_add (GTK_CONTAINER (window), druid);		
	gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (page_start));
	gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (page_a));
	gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (page_b));
	gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (page_c));
	gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (page_d));
	gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (page_finish));
	gnome_druid_set_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (page_start));

	gtk_widget_set_usize (window, 600, 400);
	gtk_widget_show_all (window);
}
