/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __PAN_GLIB_EXTENSIONS_H__
#define __PAN_GLIB_EXTENSIONS_H__

#include <stddef.h>
#include <glib.h>

/**
***  Strings
**/

#define is_nonempty_string(A) ((A) && *(A)!='\0')

gint pan_strcmp (const gchar* a, const gchar  * b);

gchar* pan_stristr (const gchar * string, const gchar * pattern);

gboolean string_ends_with (const gchar* str, const gchar* end);

void replace_gstr (gchar ** target_free_old, gchar * assign_from_me);

gchar* pan_substitute (const gchar  * original,
                       const gchar  * search,
                       const gchar  * replace);

/**
***  Glib
**/

gchar* pan_strstr (const gchar * s1,
                   const gchar * s2);

void pan_g_ptr_array_append (GPtrArray   * target,
                             gpointer    * source_ptr,
			     guint         source_qty);

void pan_g_ptr_array_assign  (GPtrArray  * target,
                              gpointer   * source_ptr,
                              guint        source_qty);

GPtrArray * pan_g_ptr_array_dup     (const GPtrArray  * source);


void pan_hash_to_ptr_array  (GHashTable * hash,
                             GPtrArray  * fillme);

void pan_g_ptr_array_reserve (GPtrArray  * target,
                              int          n);

void pan_g_ptr_array_insert (GPtrArray   * target,
                             gpointer      ptr,
                             int           index);

void pan_g_ptr_array_foreach (GPtrArray  * target,
                              GFunc        func,
                              gpointer     user_data);

void pan_g_string_replace (GString    * str,
                           const char * search,
                           const char * replace);

void pan_g_string_strstrip (GString    * str);

void pan_g_string_append_len (GString       * str,
                              const gchar   * appendme,
                              guint           len);

/**
***  Tokens
**/


void skip_next_token (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token);

int get_next_token_int (const char   * pch,
                        const char     delimiter,
                        const char  ** setme_next_token);

glong get_next_token_long (const char   * pch,
                           const char     delimiter,
                           const char  ** setme_next_token);

gulong get_next_token_ulong (const char   * pch,
                             const char     delimiter,
                             const char  ** setme_next_token);

char* get_next_token_str (const char   * pch,
                          const char     delimiter,
                          const char  ** setme_next_token);

gboolean get_next_token_g_str (const char   * pch,
                               const char     delimiter,
                               const char  ** setme_next_token,
                               GString      * setme);

const gchar*  get_next_token_run (const gchar   * pch,
                                  const gchar     delimiter,
                                  const gchar  ** setme_next_token,
                                  const gchar  ** setme_start,
                                  gint          * setme_len);


gchar* pan_str_escape   (const gchar * encode_for_xml);

gchar* pan_str_unescape (const gchar * decode_xml);


/**
 * Formats a number with commas; ie, 10321 becomes "10,321"
 * @param num the number to format
 * @param setme the string where the resulting string is written
 */
void commatize_ulong (gulong num,
                      char* setme);


/**
 * Works like msort, except it's an in-place sort
 */
void msort (void *b,
            size_t n,
            size_t s,
            int(*cmp)(const void *, const void*));

/**
 * Works like bsearch, except it returns the lower bound index
 * in the array.  This is useful for finding a sorted insertion
 * point or for finding the nearest match.
 *
 * @param key the item to search for
 * @param base the beginning of the sorted array to search
 * @param n number of items in the array
 * @param size the size of each item in the array
 * @param compare standard c compare func
 * @param exact_match (optional): if non-NULL, this is set to TRUE or FALSE if
 *        the lower bound is an exact match of key
 *
 * @return the index of the lower bound of the search
 * @prerequisite base must be sorted with respect to compare
 */
int    lower_bound   (const void   * key,
                      const void   * base,
                      size_t         n,
                      size_t         size,
                      int            (*compare)(const void *, const void *),
                      gboolean     * exact_match );



typedef enum
{
	RUN_SUCCESS = 0,
	RUN_FAIL = -1,
	RUN_FAIL_HOPELESS = -2
}
RunStatus;

#ifdef __GNUC__
#define PRETTY_FUNCTION __PRETTY_FUNCTION__
#else
#define PRETTY_FUNCTION ""
#endif

#define pan_warn_if_reached()           G_STMT_START{           \
     g_log (G_LOG_DOMAIN,                                       \
            G_LOG_LEVEL_WARNING,                                \
            "file %s: line %d func %s: should not be reached, please report bug to pan@rebelbase.com",     \
            __FILE__,                                           \
            __LINE__,                                           \
            PRETTY_FUNCTION);       }G_STMT_END

#define pan_warn_if_fail(expr)          G_STMT_START{                   \
     if (!(expr))                                                       \
       {                                                                \
         g_log (G_LOG_DOMAIN,                                           \
                G_LOG_LEVEL_CRITICAL,                                   \
                "file %s: line %d func %s: assertion `%s' failed, please report bug to pan@rebelbase.com.",        \
                __FILE__,                                               \
                __LINE__,                                               \
                PRETTY_FUNCTION,                                        \
                #expr);                                                 \
       };                               }G_STMT_END


#endif /* __PAN_GLIB_EXTENSIONS_H__ */
