/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>


static GStaticMutex _log_mutex = G_STATIC_MUTEX_INIT;
static GStringChunk * _strings = NULL;
static GMemChunk * _entries = NULL;
static GPtrArray * _log_entries = NULL;
static gboolean _inited = FALSE;


static void
log_rebuild_entry_memory (void)
{
	if (_log_entries != NULL)
		g_ptr_array_free (_log_entries, TRUE);
	_log_entries = g_ptr_array_new ();

	if (_entries != NULL)
		g_mem_chunk_destroy (_entries);
	_entries = g_mem_chunk_create (LogEntry, 64, G_ALLOC_ONLY);

	if (_strings != NULL)
		g_string_chunk_free (_strings);
	_strings = g_string_chunk_new (2048);

	_inited = TRUE;
}

static void
log_init (void)
{
	log_rebuild_entry_memory ();
}

void
log_clear (void)
{
	g_static_mutex_lock (&_log_mutex);
	log_rebuild_entry_memory ();
	pan_callback_call (log_get_entry_list_changed_callback(), NULL, NULL);
	g_static_mutex_unlock (&_log_mutex);
}

void
log_shutdown_module (void)
{
	g_static_mutex_lock (&_log_mutex);
	if (_inited)
	{
		_inited = FALSE;

		g_ptr_array_free (_log_entries, TRUE);
		g_mem_chunk_destroy (_entries);
		g_string_chunk_free (_strings);

		_log_entries = NULL;
		_entries = NULL;
		_strings = NULL;
	}
	g_static_mutex_unlock (&_log_mutex);
}

PanCallback *
log_get_entry_added_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb == NULL) cb = pan_callback_new ();
	return cb;
}

PanCallback *
log_get_entry_list_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb == NULL) cb = pan_callback_new ();
	return cb;
}

GPtrArray *
log_get_entries (void)
{
	return _log_entries;
}

static void
log_add_entry (LogEntry * entry)
{
	g_ptr_array_add (log_get_entries(), entry);

	if (entry->severity == LOG_ERROR)
		g_warning ("%s", entry->message);
	/*else
		g_message ("%s", entry->message);*/
}

static LogEntry*
log_entry_new (LogSeverity severity, const gchar * msg)
{
	LogEntry * entry;

	if (!_inited)
		log_init ();
       
	entry = g_chunk_new (LogEntry, _entries);
	entry->message = g_strchomp (g_string_chunk_insert (_strings, msg));
	entry->date = time(NULL);
	entry->severity = severity;
	return entry;
}

void
log_add (LogSeverity severity, const gchar * message)
{
	LogEntry * entry;
	g_static_mutex_lock (&_log_mutex);
	log_add_entry (entry = log_entry_new (severity, message));
	g_static_mutex_unlock (&_log_mutex);

	pan_callback_call (log_get_entry_added_callback(), entry, NULL);
}

void
log_add_va (LogSeverity severity, const gchar * format, ...)
{
	va_list args;
	gchar * message;
	LogEntry * entry;
	g_return_if_fail (format != NULL);

	g_static_mutex_lock (&_log_mutex);
	va_start (args, format);
	message = g_strdup_vprintf (format, args);
	va_end (args);
	log_add_entry (entry = log_entry_new (severity, message));
	g_free (message);
	g_static_mutex_unlock (&_log_mutex);

	pan_callback_call (log_get_entry_added_callback(), entry, NULL);
}
