/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@rebelbase.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __GROUPS_H__
#define __GROUPS_H__

#include <glib.h>

#include <pan/base/newsrc.h>
#include <pan/base/server.h>
#include <pan/base/status-item.h>

typedef enum
{
	FILTER_SHOW_MATCHES                     = 1,
	FILTER_SHOW_MATCHES_AND_REPLIES	        = 2,
	FILTER_SHOW_MATCHES_AND_REFERENCES      = 3,
	FILTER_SHOW_THREAD                      = 4
}
FilterShow;

enum
{
	GROUP_ALL				= (1<<0),
	GROUP_SUBSCRIBED			= (1<<1), 
	GROUP_NEW				= (1<<2),
	GROUP_FOLDERS				= (1<<3)
};

enum
{
        STATE_FILTER_UNREAD                     = (1<<0),
        STATE_FILTER_READ                       = (1<<1),
        STATE_FILTER_WATCHED                    = (1<<2),
        STATE_FILTER_IGNORED                    = (1<<3),
        STATE_FILTER_NORMAL_RANK                = (1<<4),
        STATE_FILTER_COMPLETE_BINARIES          = (1<<5),
        STATE_FILTER_INCOMPLETE_BINARIES        = (1<<6),
        STATE_FILTER_NONBINARIES                = (1<<7),
        STATE_FILTER_SAVED                      = (1<<8),
        STATE_FILTER_QUEUED                     = (1<<9),
        STATE_FILTER_IDLE                       = (1<<10),
	STATE_FILTER_NEW			= (1<<11),
	STATE_FILTER_CACHED			= (1<<12),
	STATE_FILTER_NOT_CACHED			= (1<<13),
	STATE_FILTER_OLD                        = (1<<14),
	STATE_FILTER_MINE                       = (1<<15),
	STATE_FILTER_NOT_MINE                   = (1<<16)
};

#define GROUP(x) ((Group*)(x))

/**
 * For the time being, groups are 'owned' by their server.
 * When we go group-centric, this will change.
 *
 * Having Group a handle to its server lets us pass around
 * groups w/o a server argument for context, which should make
 * refitting for groupcentric easier...
 */
typedef struct _Group
{
	GMutex       * article_mutex;

        guint          articles_dirty           : 1;     /* do the articles need saving? */
	guint          articles_threading_dirty : 1;
	gchar          permission;                       /* (read only) 'y'=posting, 'n'=nopost, 'm'=moderated */
	gint8          flags;                            /* (read only) new, subscribed, etc */
	gint8          sort_style;                       /* (read only) hint for sorting in UI */
	gulong         filter_show;                      /* (read only) for filtering articles */
	guint32        filter_bits;                      /* (read only) for filtering articles */
	gint32         article_qty;                      /* (read only) # of articles in group */
	gint32         article_read_qty;                 /* (read only) # of read articles */
	gulong         article_low;                      /* (read only) low article number */
	gulong         article_high;                     /* (read only) high article number */
	gulong         article_high_old;                 /* (read only) previous high article */

	Server       * server;
	gchar        * name;                             /* alt.what.is.my.name */
	gchar        * filter_name;                      /* article toolbar, articlelist */
	gchar        * identity_name;                    /* last identity used for this group */
	gchar        * description;                      /* maybe null */
	gchar        * readable_name;                    /* maybe null, user alias for group */
	gchar        * download_dir;                     /* where to save files, or NULL */
	GStringChunk * chunk;                            /* string chunk used by articles */

	/* PRIVATE */
	gint           _articles_refcount;
	GMemChunk    * _article_chunk;                   /* Article objects */
	Newsrc       * _newsrc;                          /* tracks read/unread */
	Newsrc       * _purged;                          /* tracks deleted articles */
	GHashTable   * _articles;                        /* articles */
	gchar        * _one_big_chunk;                   /* used by loading local headers */
}
Group;


/**
***  PUBLIC
**/

Group*    group_new                      (Server        * server,
                                          const char    * name);

void      group_inc_article_read_qty     (Group         * group,
                                          gint            article_read_qty);

void      group_set_article_qty          (Group         * group,
                                          gint            article_qty);

void      group_set_article_read_qty     (Group         * group,
                                          gint            article_read_qty);

void      group_set_article_range        (Group         * group,
                                          gulong          article_low,
                                          gulong          article_high);

void      group_get_article_range        (const Group   * group,
                                          gulong        * article_low,
                                          gulong        * article_high);

void      group_mark_new_article_number  (Group         * group,
                                          gulong          everything_above_this_is_new);

void      group_mark_all_read            (Group         * group,
                                          gboolean        read);

void      group_set_sort_style           (Group         * group,
                                          gint            sort_style);

void      group_set_filter               (Group         * group,
                                          guint           filter_bits,
                                          gulong          filter_show,
                                          const gchar   * filter_name);

void      group_set_identity             (Group         * group,
                                          const gchar   * identity_name);

void      group_set_flags                (Group         * group,
                                          guint           flags);

void      group_set_download_dir         (Group         * group,
                                          const char    * download_dir);

gboolean  group_is_folder                (const Group   * group);

gboolean  group_is_group                 (const Group   * group);

gboolean  group_is_valid                 (const Group   * group);

gboolean  group_is_moderated             (const Group   * group);

gboolean  group_is_read_only             (const Group   * group);

void      group_set_is_folder            (Group         * group,
                                          gboolean        is_folder);

void      group_set_readable_name        (Group         * group,
                                          const gchar   * readable_name);

const gchar*    group_get_readable_name        (const Group   * group);


void      groups_set_subscribed          (Group        ** group,
                                          gint            qty,
                                          gboolean        subscribed);

gboolean  group_is_subscribed            (const Group   * group);

void      group_set_new                  (Group         * group,
                                          gboolean        is_new);

gboolean  group_is_new                   (const Group   * group);

void      group_set_dirty                (Group         * group);

void      group_empty                    (Group         * group,
                                          gboolean        clear_counts);

/**
 * ARTICLES
 *
 * the add and remove article functions all update the group's
 * statistics and the way it appears in the Pan user interface.
 * If you're adding or removing more than one groups, it's much
 * better to use the plural functions once instead of calling
 * the singular versions many times.
 */

struct _Article;

void          group_add_article                   (Group            * group,
                                                   struct _Article  * article);

void          group_add_articles                  (Group            * group,
                                                   const GPtrArray  * articles,
                                                   StatusItem       * status_or_null,
					           GPtrArray        * used,
					           GPtrArray        * ignored);

void          group_add_articles_remove_unused    (Group          * group,
                                                   GPtrArray        * articles,
                                                   StatusItem       * status_or_null);

void          group_remove_article                (Group            * group,
                                                   struct _Article  * article);

void          group_remove_articles               (Group            * group,
                                                   const GPtrArray  * articles);

void          group_remove_crossposts             (Group            * g,
		                                   gulong           * numbers,
					           gint               number_qty);

void          group_ref_articles                  (Group            * group,
                                                   StatusItem       * item);

void          group_unref_articles                (Group            * group,
                                                   StatusItem       * item);


void          group_expire_articles_not_in_range  (Group            * group,
                                                   gulong             low,
                                                   gulong             high);

void          group_thread_if_needed              (Group            * group);

const gchar*  group_chunk_string                  (Group            * group,
                                                   const gchar      * chunkme,
                                                   gboolean           share);

void          group_set_articles_dirty            (Group            * group);

Newsrc*       group_get_newsrc                    (Group            * group);

gboolean      group_has_newsrc                    (const Group      * group);

Newsrc*       group_get_purged                    (Group            * group);

GHashTable*   group_get_articles                  (Group            * group);

GPtrArray*    group_get_article_array             (Group            * group);

gpointer      group_get_article_by_message_id     (Group            * group,
                                                   const gchar      * message_id);

struct _Article;

struct _Article*  group_alloc_new_article        (Group         * group);


/**
***  PROTECTED
**/

void group_constructor    (Group                        * group,
			   const gchar                  * name);

void group_destructor     (Group                        * group);


extern const gchar * PAN_SENDLATER;
extern const gchar * PAN_SENT;


/**
***  Events
**/

/**
 * @call_obj: Group*
 * @call_arg: GPtrArray of Article* sorted by message_id
 */
PanCallback*  group_get_articles_added_callback (void);

/**
 * @call_obj: Group*
 * @call_arg: GPtrArray of Article* sorted by message_id
 */
PanCallback*  group_get_articles_removed_callback (void);

/**
 * @call_obj: Group**
 * @call_arg: GINT_TO_POINTER(qty)
 */
PanCallback*  group_get_groups_changed_callback (void);

/**
 * @call_obj: Group*
 * @call_arg: non-NULL if all marked read, NULL if marked unread
 */
PanCallback* group_get_group_marked_read_callback (void);

/**
 * @call_obj: Group*
 * @call_arg: NULL
 */
PanCallback* group_get_group_emptied_callback (void);




#endif /* __GROUPS_H__ */
