/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <glib.h>

#include <pan/base/debug.h>
#include <pan/base/gnksa.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/*****
******
*****/

static gboolean
is_unquoted_char (int ch)
{
	/* ASCII printable character except !()<>@,;:\".[] */
	return isgraph(ch) && strchr("!()<>@,;:\\\".[]", ch)==NULL;
}

static gboolean
is_quoted_char (int ch)
{
	/* ASCII printable character except "()<>\ */
	return isgraph(ch) && strchr("\"()<>\\", ch)==NULL;
}

static gboolean
is_tag_char (int ch)
{
	/* ascii printable character except !()<>@,;:\"[]/?=> */
	return isgraph(ch) && strchr("!()<>@,;\\\"[]/?=", ch)==NULL;
}

static gboolean
is_code_char (int ch)
{
	/* ascii printable character except ? */
	return isgraph(ch) && ch!='?';
}

static gboolean
is_paren_char (int ch)
{
	/* ascii printable character except ()<>\ */
	return isgraph(ch) && strchr("()<>\\",ch)==NULL;
}

/**
***
**/

static gboolean
read_space (const char * start, const char ** end)
{
	/* 1*( <HT (ASCII 9)> / <blank (ASCII 32)> */
	const gchar * pch = start;
	while (*pch=='\t' || *pch==' ') ++pch;
	if (pch != start) {
		*end = pch;
		return TRUE;
	}
	return FALSE;
	
}

static gboolean
read_codes (const char * pch, const char ** end)
{
	if (!is_code_char(*pch)) return FALSE;
	while (is_code_char(*pch)) ++pch;
	*end = pch;
	return TRUE;
}

static gboolean
read_encoding (const char * pch, const char ** end)
{
	if (!is_tag_char(*pch)) return FALSE;
	while (is_tag_char(*pch)) ++pch;
	*end = pch;
	return TRUE;
}

static gboolean
read_charset (const char * pch, const char ** end)
{
	return read_encoding (pch, end);
}

static gboolean
read_encoded_word (const char * pch, const char ** end)
{
	/* "=?" charset "?" encoding "?" codes "?=" */
	if (pch[0]!='=' || pch[1]!='?') return FALSE;
	pch += 2;
	if (!read_charset (pch, &pch)) return FALSE;
	if (*pch != '?') return FALSE;
	++pch;
	if (!read_encoding (pch, &pch)) return FALSE;
	if (*pch != '?') return FALSE;
	++pch;
	if (!read_codes (pch, &pch)) return FALSE;
	if (pch[0]!='?' || pch[1]!='=') return FALSE;
	*end = pch + 2;
	return TRUE;
}

static gboolean
read_unquoted_word (const char * pch, const char ** end)
{
	/* 1*unquoted-char */
	if (!is_unquoted_char(*pch)) return FALSE;
	while (is_unquoted_char((int)*pch)) ++pch;
	*end = pch;
	return TRUE;
}

static gboolean
read_quoted_word (const char * pch, const char ** end)
{
	/* quote 1*(quoted-char / space) quote */
	if (*pch!='"') return FALSE;
	++pch;
	for (;;) {
		if (read_space(pch,&pch))
			continue;
		if (is_quoted_char ((int)*pch)) {
			++pch;
			continue;
		}
		break;
	}
	if (*pch!='"') return FALSE;
	++pch;
	*end = pch;
	return TRUE;
}

static gboolean
read_plain_word (const char * pch, const char ** end)
{
	/* unquoted-word / quoted-word / encoded-word */
	if (read_unquoted_word(pch,end)) return TRUE;
	if (read_quoted_word(pch,end)) return TRUE;
	if (read_encoded_word(pch,end)) return TRUE;
	return FALSE;
}

static gboolean
read_plain_phrase (const char * pch, const char ** end)
{
	/* plain-word *(space plain-word) */
	const char * tmp = NULL;
	if (!read_plain_word(pch,&pch))
		return FALSE;
	for (;;) {
		tmp = pch;
		if (!read_space(pch,&pch)) break;
		if (!read_plain_word(pch,&pch)) break;
	}
	*end = tmp;
	return TRUE;
}

static gboolean
read_paren_char (const char * pch, const char ** end)
{
	if (!is_paren_char((int)*pch))
		return FALSE;
	*end = pch + 1;
	return TRUE;
}


static gboolean
read_paren_phrase (const char * pch, const char ** end)
{
	/* 1* (paren-char / space / encoded-word */

	if (!read_paren_char(pch,&pch)
		&& !read_space(pch,&pch)
		&& !read_encoded_word(pch,&pch))
			return FALSE;

	for (;;)
	{
		if (!read_paren_char(pch,&pch)
			&& !read_space(pch,&pch)
			&& !read_encoded_word(pch,&pch))
				break;
	}

	*end = pch;
	return TRUE;
}

/*****
******
*****/

/**
 *  "[244.14.124.12]"
 *  "244.12.14.12"
 */
static int
gnksa_check_domain_literal (const char* domain)
{
	int i;
	int x[4];
	gboolean need_closing_brace;
	const char * pch;

	/* parse domain literal into ip number */

	pch = domain;
	need_closing_brace = *pch == '[';
	if (need_closing_brace) ++pch; /* move past open brace */

	/* %u.%u.%u.%u */
	for (i=0; i<4; ++i) {
		char * end = NULL;
		x[i] = strtoul (pch, &end, 10);
		if (end == pch)
			return GNKSA_BAD_DOMAIN_LITERAL;
		if (x[i]<0 || x[i]>255)
			return GNKSA_BAD_DOMAIN_LITERAL;
		if (i!=3) {
			if (*end != '.')
				return GNKSA_BAD_DOMAIN_LITERAL;
			++end;
		}
		pch = end;
	}

	if (need_closing_brace && *pch!=']')
		return GNKSA_BAD_DOMAIN_LITERAL;

	return GNKSA_OK;
}

/*****
******
*****/

static int
gnksa_check_domain (const char * domain)
{
	int i;
	int label_qty;
	const char * top;
	const char * pch;
	const gchar * label = NULL;
	gint label_len = 0;

	debug1 (DEBUG_GNKSA, "gnksa_check_domain: [%s]", domain);

	/* check for domain literal */
	if (*domain == '[')
		return gnksa_check_domain_literal (domain);

	/* check for empty */
	if (!is_nonempty_string (domain)
		|| *domain=='.'
		|| domain[strlen(domain)-1]=='.'
		|| pan_strstr(domain,"..")!=NULL)
		return GNKSA_ZERO_LENGTH_LABEL;

	/* count the labels */
	label_qty = 0;
	pch = domain;
	while (get_next_token_run (pch, '.', &pch, &label, &label_len))
		++label_qty;

	/* make sure we have more than one label in the domain */
	if (label_qty < 2)
		return GNKSA_SINGLE_DOMAIN;

	/* check for illegal labels */
	pch = domain;
	for (i=0; i<label_qty-1; ++i)
	{
		get_next_token_run (pch, '.', &pch, &label, &label_len);
		if (label_len > 63)
			return GNKSA_ILLEGAL_LABEL_LENGTH;
		if (label[0]=='-' || label[label_len-1]=='-')
			return GNKSA_ILLEGAL_LABEL_HYPHEN;
	}

	/* last label -- toplevel domain */
	top = get_next_token_run (pch, '.', &pch, &label, &label_len);
	debug1 (DEBUG_GNKSA, "top level domain: [%s]", top);
	switch ((int)strlen(top))
	{
		case 1:
			if (isdigit((guchar)top[0]))
				return gnksa_check_domain_literal (domain);

			/* single-letter TLDs dont exist */
			return GNKSA_ILLEGAL_DOMAIN;
			break;

		case 2:
			if (isdigit((guchar)top[0]) || isdigit((guchar)top[1]))
				return gnksa_check_domain_literal (domain);
			break;
		case 3:
			if (isdigit((guchar)top[0]) || isdigit((guchar)top[2]) || isdigit((guchar)top[3]))
				return gnksa_check_domain_literal (domain);
			break;
		default:
			break;
	}

	return GNKSA_OK;
}

/*****
******
*****/

static int
gnksa_check_localpart (const char * localpart)
{
	gint word_len = 0;
	const gchar * pch;
	const gchar * word = NULL;

	/* make sure it's not empty... */
	if (!is_nonempty_string(localpart))
		return GNKSA_LOCALPART_MISSING;

	/* break localpart up into its unquoted words */
	pch = localpart;
	while (get_next_token_run (pch, '.', &pch, &word, &word_len))
	{
		gint i;

		if (word_len < 1)
			return GNKSA_ZERO_LENGTH_LOCAL_WORD;

		for (i=0; i<word_len; ++i)
			if (!is_unquoted_char(word[i]))
				return GNKSA_INVALID_LOCALPART;
	}

	return GNKSA_OK;
}

/*****
******
*****/

static gint
gnksa_check_address (const gchar * address)
{
	gint retval = GNKSA_OK;
	gchar * tmp;
	gchar * begin;

	/* get rid of vacuous case */
	if (!is_nonempty_string(address))
		return GNKSA_LOCALPART_MISSING;

	/* check the address */
	tmp = g_strdup (address);
	begin = strrchr (address, '@');
	if (begin==NULL)
	{
		if (retval == GNKSA_OK)
			retval = GNKSA_INVALID_DOMAIN;
	}
	else
	{
		/* temporarily split string between username & fqdn */
		*begin++ = '\0';
		g_strdown (begin);

		/* check the domain */
		if (retval == GNKSA_OK)
			retval = gnksa_check_domain (begin);

		/* check the localpart */
		if (retval == GNKSA_OK)
			retval = gnksa_check_localpart (address);

		/* restore separator */
		*--begin = '@';
	}

	g_free (tmp);
	return retval;
}

/*****
******
*****/

static int
gnksa_split_from (const char   * from,
                  char        ** alloc_and_setme_address,
		  char        ** alloc_and_setme_realname,
		  int          * addrtype,
		  gboolean       strict)
{
	char * begin;
	char * end;
	char * work;
	char * lparen;

	*alloc_and_setme_address = NULL;
	*alloc_and_setme_realname = NULL;

	work = g_strdup (from);

	if (is_nonempty_string(work))
		g_strstrip (work);

	/* empty string */
	if (!*work)
	{
		*addrtype = GNKSA_ADDRTYPE_OLDSTYLE;
		g_free (work);
		return GNKSA_LPAREN_MISSING;
	}

	end = &work [ strlen(work) - 1 ];
	debug2 (DEBUG_GNKSA, "work: [%s], end: [%s]", work, end);
	if (*end == '>') /* Charles Kerr <charles@rebelbase.com> */
	{
		*addrtype = GNKSA_ADDRTYPE_ROUTE;

		/* get address part */
		begin = strrchr (work, '<');
		if (begin == NULL) {
			g_free (work);
			return GNKSA_LANGLE_MISSING;
		}

		/* copy route address from inside the <> brackets */
		*alloc_and_setme_address = g_strdup (begin+1);
		*strchr(*alloc_and_setme_address, '>') = '\0';

		/* From: [plain-phrase space] "<" address ">" */
		*begin = '\0';
		if (strict) {
			const char * tmp = work;
			if ((*tmp) && (!read_plain_phrase(tmp,&tmp)||!read_space(tmp,&tmp))) {
				g_free (work);
				return GNKSA_ILLEGAL_PLAIN_PHRASE;
			}
		}

		/* get realname part */
		g_strstrip (work);
		*alloc_and_setme_realname = g_strdup (work);
	}
	else if ((lparen = strchr (work,'(')) != NULL) /* charles@rebelbase.com (Charles Kerr) */
	{
		size_t len;
		gchar * address;
		gchar * real;
		const gchar * end;

		*addrtype = GNKSA_ADDRTYPE_OLDSTYLE;

		/* address part */
		address = lparen==work ? NULL : g_strndup (work, lparen-work);
		if (is_nonempty_string (address))
			g_strstrip (address);
		*alloc_and_setme_address = address;
		if (strict) {
			gint val = gnksa_check_address (address);
			if (val) {
				g_free (work);
				return val;
			}
		}

		/* real name part */
		real = lparen + 1;
		g_strstrip (real);
		len = strlen (real);
		if (real[len-1]!=')') {
			g_free (work);
			return GNKSA_RPAREN_MISSING;
		}
		real[len-1] = '\0';
		end = NULL;
		if (strict && (!read_paren_phrase(lparen+1,&end) || end==NULL)) {
			g_free (work);
			return GNKSA_ILLEGAL_PAREN_PHRASE;
		}
		*alloc_and_setme_realname = g_strdup (real);
	}
	else if (strchr(work,'@') != NULL) /* charles@rebelbase.com */
	{
		*alloc_and_setme_address = work;
		return strict ? gnksa_check_address(work) : GNKSA_OK;
	}
	else /* who knows what this thing is... */
	{
		*alloc_and_setme_realname = work;
		return GNKSA_LPAREN_MISSING;
	}

	g_free (work);
	return GNKSA_OK;
}


/*****
******
*****/

/************
*************  PUBLIC
************/

/*****
******
*****/

GPtrArray*
gnksa_split_addresses (const char * from)
{
	GPtrArray * a = g_ptr_array_new ();
	gboolean quoted = FALSE;
	const gchar * pch = from;
	const gchar * start = from;

	for (pch=start=from; is_nonempty_string(pch); )
	{
		while (*pch && !strchr("\",;:",*pch))
			++pch;

		if (*pch=='"') /* toggle quoted */
			quoted = !quoted;
		else if (!quoted) /* delimiter */
		{
			gchar * s = g_strndup (start, pch-start);
			g_strstrip (s);
			if (is_nonempty_string(s))
				g_ptr_array_add (a, s);
			else /* too small; throw it back */
				g_free (s);
			if (*pch)
				start = pch + 1;
		}

		if (*pch)
			++pch;
	}

	return a;
}

/*****
******
*****/


/*****
******
*****/

int
gnksa_do_check_from (const char   * from,
                     char        ** alloc_and_setme_address,
                     char        ** alloc_and_setme_realname,
		     gboolean       strict)
{
	int addrtype = 0;
	int retval;
	char * address;
	char * realname;
	debug1 (DEBUG_GNKSA, "From: [%s]", from);

	/* split from */
	retval = gnksa_split_from (from,
	                           alloc_and_setme_address,
		         	   alloc_and_setme_realname, 
			           &addrtype,
				   strict);

	address = *alloc_and_setme_address;
	realname = *alloc_and_setme_realname;
	debug1 (DEBUG_GNKSA, "Name: [%s]", (realname?realname:"NULL"));
	debug1 (DEBUG_GNKSA, "Address: [%s]", (address?address:"NULL"));

	/* check address */
	if (address != NULL) {
		if (retval == GNKSA_OK)
			retval = gnksa_check_address (address);
	}

	debug1 (DEBUG_GNKSA, "GNKSA [%d]", retval);
	return retval;
}

/*****
******
*****/

void
gnksa_strip_realname (gchar * realname)
{
	gchar * pch;

	g_return_if_fail (is_nonempty_string(realname));

	g_strstrip (realname);
	if (*realname=='(') {
		size_t len = strlen (realname+1);
		g_memmove (realname, realname+1, len+1);
		pch = realname + len - 1;
		if (*pch == ')')
			*pch = '\0';
	}

	while ((pch = strchr(realname,'"')) != NULL)
		g_memmove (pch, pch+1, strlen(pch+1)+1);
}

/*****
******
*****/

int
gnksa_check_from (const char* from, gboolean strict)
{
	int retval;
	char * address;
	char * realname;

	address = realname = NULL;
	retval = gnksa_do_check_from (from, &address, &realname, strict);
	g_free (address);
	g_free (realname);

	return retval;
}

/*****
******
*****/

/* see son-of-1036 5.3 */

static int
gnksa_check_message_id (const char* message_id)
{
	int val;
	gchar * tmp;
	gchar * pch;

	/* make a temporary copy that we can poke */
       	tmp = g_strdup (message_id);
	if (is_nonempty_string(tmp))
		g_strstrip (tmp);

	/* make sure it's <= 250 octets (son of gnksa 1036 5.3) */
	if (strlen(tmp) > 250) {
		g_free (tmp);
		return GNKSA_ILLEGAL_LABEL_LENGTH;
	}

	/* make sure the message-id is wrapped in < > */
	if (*tmp!='<') {
		g_free (tmp);
		return GNKSA_LANGLE_MISSING;
	}
	if (tmp[strlen(tmp)-1]!='>') {
		g_free (tmp);
		return GNKSA_RANGLE_MISSING;
	}

	/* find the '@' separator */
	pch = strrchr (tmp, '@');
	if (pch == NULL) {
		g_free (tmp);
		return GNKSA_ATSIGN_MISSING;
	}

	/* check the domain name */
	++pch;
	*strrchr(pch,'>') = '\0';
	val = gnksa_check_localpart (pch); /* yes, I mean to do this */
	if (val != GNKSA_OK) {
		g_free (tmp);
		return val;
	}

	/* check the local-part */
	--pch;
	*pch = '\0';
	if (!g_strcasecmp (tmp+1, "postmaster")) { /* son-of-1036 */
		g_free (tmp);
		return GNKSA_INVALID_LOCALPART;
	}
	val = gnksa_check_localpart (tmp+1);
	if (val != GNKSA_OK) {
		g_free (tmp);
		return val;
	}

	g_free (tmp);
	return GNKSA_OK;
}

/*****
******
*****/

static char*
remove_broken_message_ids_from_references (const char * references)
{
	gchar * pch;
	gchar * tmp;
	GString * fixed;

	/* sanity checks */
	g_return_val_if_fail (is_nonempty_string(references), NULL);

	/* remove broken message-ids
	   FIXME: this tokenizing works, but it's hard to read. */
       	fixed = g_string_sized_new (1024);
       	tmp = g_strdup (references);
	pch = tmp;
	for (;;)
	{
		gchar tmp_end_val;
		gchar * begin;
		gchar * end;

		/* skip to the beginning of a message id */
		while (*pch!='<')
			++pch;
		begin = pch;
		++pch;
		end = strpbrk (pch, "<> ");
		if (end==NULL)
			end = pch + strlen(pch);
		else if (*end == '>')
			++end;

		/* zero-terminate the message-id to check it for validity */
		tmp_end_val = *end;
		*end = '\0';
		if (gnksa_check_message_id(begin)==GNKSA_OK) {
			g_string_append (fixed, begin);
			g_string_append_c (fixed, ' ');
		}
		*end = tmp_end_val;
		pch = end;

		/* if that was the last token, then we're done */
		if (tmp_end_val == '\0')
			break;

	}
	pan_g_string_strstrip (fixed); /* remove trailing space  */

	/* cleanup */
	pch = fixed->str;
	g_free (tmp);
	g_string_free (fixed, FALSE);

	return pch;
}

/**
 * Try to trim references down to an acceptable length for the NNTP server,
 * @param untrimmed "references" string
 * @return newly-allocated trimmed "references" string
 */
gchar*
gnksa_trim_references_to_len (const gchar* refs, int cutoff)
{
	char * fixed = remove_broken_message_ids_from_references (refs);
	char * retval;
	const int fixed_len = strlen (fixed);

	if (fixed_len < cutoff) /* this is too easy. :) */
	{
		retval = g_strdup(fixed);
	}
	else /* plow through them */
	{
		int leader_len;
		int len_left;
		const char * str_end = fixed + fixed_len;
		const char * pch;

		/* isolate first id, which we must always keep */
	       	pch = fixed;
		while (*pch!='<') ++pch; /* find beginning of first id */
		while (*pch!='>') ++pch; /* find end of first id */
		pch += 2; /* keep the "> " */
		leader_len = pch - fixed;


		/* trim out 2..n until we fit.  This implicitly lets later
		   ids make the final cut, which is in line with gnksa's
		   requirement that the final three ids not be trimmed out. */
		len_left = cutoff - leader_len;
		while (str_end-pch > len_left) {
			++pch; /* skip until we fit */
		}
		while (*pch && *pch!='<' ) ++pch; /* can't include half an id */

		/* generate return string */
		retval = g_malloc (leader_len + (str_end-pch) + 1);
		g_memmove (retval, fixed, leader_len);
		g_memmove (retval+leader_len, pch, (str_end-pch)+1); /*+1 for end*/
	}

	/* return */
	g_free (fixed);
	pan_warn_if_fail (strlen(retval) <= cutoff);
	return retval;
}

static const gchar* default_domain = "nospam.com";

/**
 * thus spake son-of-1036: "the most popular method of generating local parts
 * is to use the date and time, plus some way of distinguishing between
 * simultaneous postings on the same host (e.g., a process number), and encode
 * them in a suitably-reduced alphabet.
 */
gchar*
gnksa_generate_message_id (const gchar * domain_name)
{
	char * tmp;
	GString * msgid = g_string_sized_new (256);

	/* start with '<' */
	g_string_append_c (msgid, '<');

	/* add unique local part to message-id */
	{
		gulong pid;
		struct timeval now;
		struct tm local_now;
		char buf[64];

		pid = (gulong) getpid ();
		gettimeofday (&now, NULL);
		localtime_r (&now.tv_sec, &local_now);
		strftime (buf, sizeof(buf), "%Y.%m.%d.%H.%M.%S", &local_now);

		g_string_append (msgid, "pan.");
		g_string_append (msgid, buf);
		g_string_sprintfa (msgid, ".%ld.%lu", now.tv_usec, pid);
	}

	/* delimit */
	g_string_append_c (msgid, '@');

	/* add domain name to message-id */
	if (!is_nonempty_string(domain_name))
		domain_name = default_domain;
	g_string_append (msgid, domain_name);

	/* end with '>' */
	g_string_append_c (msgid, '>');

	/* return */
	tmp = msgid->str;
	g_string_free (msgid, FALSE);
	return tmp;
}

gchar*
gnksa_generate_message_id_from_email_addr (const gchar * addr)
{
	const gchar * pch;
	gchar * domain;
	gchar * retval;

	/* find the domain in the email address */
	pch = NULL;
	if (is_nonempty_string(addr)) {
		pch = strchr (addr, '@');
		if (pch == NULL)
			pch = addr;
		else
			++pch;
	}
	if (!is_nonempty_string(pch)) {
		g_warning (_("No email address provided; generating message-id with domain \"%s\""), default_domain);
		pch = default_domain;
	}

	/* make a copy of the rhs of the email address and trim out the '>' */
	domain = g_strdup (pch);
	pch = strchr (domain, '>');
	if (pch != NULL)
		*((gchar*)pch) = '\0';
	g_strstrip (domain);

	/* generate the domain name */
	retval = gnksa_generate_message_id (domain);

	/* cleanup */
	g_free (domain);
	return retval;
}


static gchar*
gnksa_trim_references (const gchar* refs)
{
	const int GNKSA_CUTOFF = 998; /* gnksa rule 7 */
	return gnksa_trim_references_to_len (refs, GNKSA_CUTOFF);
}

gchar*
gnksa_generate_references (const gchar * references,
                           const gchar * message_id)
{
	gchar * retval = NULL;

	if (is_nonempty_string (message_id))
	{
		/* build a full references string */
		GString * str = g_string_sized_new (1024);
		if (is_nonempty_string(references)) {
			g_string_append (str, references);
			g_string_append_c (str, ' ');
		}
		g_string_append (str, message_id);

		/* make sure they fit gnksa guidelines */
		retval = gnksa_trim_references (str->str);
		g_string_free (str, TRUE);
	}

	return retval;
}

/***
****
****  DATE
****
***/

/* converts struct tm to time_t, but does not take the local
   timezone into account unless ``local'' is nonzero.
   This is lifted from mutt 1.2.4, which is GPL and
   copyrighted by Michael R. Elkins <me@cs.hmc.edu> */
static time_t
mutt_mktime (struct tm *t)
{
	time_t g;

	static int AccumDaysPerMonth[12] = {
		0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
	};

	/* Compute the number of days since January 1 in the same year */
	g = AccumDaysPerMonth [t->tm_mon % 12];

	/* The leap years are 1972 and every 4. year until 2096,
	 * but this algoritm will fail after year 2099 */
	g += t->tm_mday;
	if ((t->tm_year % 4) || t->tm_mon < 2)
		g--;
	t->tm_yday = g;

	/* Compute the number of days since January 1, 1970 */
	g += (t->tm_year - 70) * 365;
	g += (t->tm_year - 69) / 4;

	/* Compute the number of hours */
	g *= 24;
	g += t->tm_hour;

	/* Compute the number of minutes */
	g *= 60;
	g += t->tm_min;

	/* Compute the number of seconds */
	g *= 60;
	g += t->tm_sec;

	return g;
}

gchar*
rfc822_date_generate (time_t secs)
{
	static const gchar *const months[] = {
		"Jan","Feb","Mar","Apr","May","Jun",
		"Jul","Aug","Sep","Oct","Nov","Dec"
	};
	static const gchar *const days[] = {
		"Sun","Mon","Tue","Wed","Thu","Fri","Sat"
	};

	int diff;
	int tz_diff_hours;
	int tz_diff_minutes;
	gboolean neg;
	gchar fmt[64] = { '\0' };
	gchar buf[64] = { '\0' };
	struct tm ut;
	struct tm local;

	/* if 0 passed in, use current time */
	if (secs == 0)
		secs = time (NULL);

	/* figure out the difference between this timezone and UT */
	localtime_r (&secs, &local);
	gmtime_r (&secs, &ut);
	diff = (int) difftime (mutt_mktime(&local), mutt_mktime(&ut));
	neg = diff<0;
	diff = abs(diff);
	tz_diff_hours = diff / 3600;
	diff %= 3600;
	tz_diff_minutes = diff / 60;

	/* decide on the time string format.  see son of 1036 sec 5.1 for
	 * the rationale of the offset-from-ut notation.
         *
	 * Update: INN 1.7.2-4 (and others) don't accept the <TZ> string at
	 * the end of son-of-1036's date string.  Thanks to Valentini Jove
	 * for testing to see what worked & what didn't.
	 *
	 * Update 2: Big Thanks to Tov Jacobsen for investigating what
	 * format other programs use, for suggesting code, and for noting
	 * that we need to roll our own day-of-week and month abbreviations
	 * because son-of-1036 requires English text but strftime() localizes
	 * (Okt instead of Oct or Man instead of Mon).
	 *
	 * Update 3: Gaelyne Gasson spotted a bug that the minute tz offset
	 * should have been divided by 60 to give minutes instead of seconds.
	 * This was causing problems in that tz, which is +0930.  I'm feeling
	 * pretty dumb here... :)
	 */
	sprintf (fmt, "%s, %%d %s %%Y %%H:%%M:%%S %c%02d%02d",
		days[local.tm_wday],
		months[local.tm_mon],
		(neg?'-':'+'),
		tz_diff_hours,
		tz_diff_minutes);

	/* generate the time */
	strftime (buf, sizeof(buf), fmt, &local);
	return g_strdup (buf);
}

/***
****
***/

#ifndef HAVE_LOCALTIME_R
struct tm*
localtime_r(const time_t *const timep, struct tm *p_tm)
{
	struct tm * retval;
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;

	g_static_mutex_lock (&mutex);
	retval = localtime (timep);
	if (retval != NULL) {
		memcpy (p_tm, retval, sizeof(struct tm));
		retval = p_tm;
	}
	g_static_mutex_unlock (&mutex);

	return retval;
}
#endif

#ifndef HAVE_GMTIME_R
struct tm*
gmtime_r (const time_t * const timep, struct tm * p_tm)
{
	struct tm * retval;
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;

	g_static_mutex_lock (&mutex);
	retval = gmtime (timep);
	if (retval != NULL) {
		memcpy (p_tm, retval, sizeof(struct tm));
		retval = p_tm;
	}
	g_static_mutex_unlock (&mutex);

	return retval;
}
#endif
