/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>

#include <gmime/gmime.h>
#include <gmime/gmime-stream-file.h>

#include <pan/base/acache.h>
#include <pan/base/debug.h>
#include <pan/base/decode.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>
#include <pan/base/util-mime.h>

/**
*** Private Routines
**/

static gchar*
create_filename (const gchar       * path,
                 const gchar       * subject,
                 const gchar       * default_filename,
                 const GMimePart   * part)
{
	gint i;
	gchar * retval;
	GString * filename;
	debug_enter ("create_filename");

	g_return_val_if_fail (is_nonempty_string(path), NULL);
	g_return_val_if_fail (part!=NULL, NULL);

       	filename = g_string_new (NULL);

	/* first try the filename specified by the user */
	if (!filename->len && is_nonempty_string(default_filename))
		g_string_assign (filename, default_filename);

	/* otherwise try the filename specified by the article */
	if (!filename->len) {
		const gchar * pch = g_mime_part_get_filename (part);
		if (is_nonempty_string(pch)) {
			/* if some bozo included the windows pathname in the filename, strip that out */
			if (isalpha((guchar)pch[0]) && pch[1]==':' && pch[2]=='\\')
				pch = strrchr (pch, '\\') + 1;
			g_string_assign (filename, pch);
		}
	}

	/* otherwise try the article's subject */
	if (!filename->len && is_nonempty_string(subject))
		g_string_assign (filename, subject);

	/* otherwise punt */
	if (!filename->len)
		g_string_assign (filename, _("UNKNOWN"));

	/* filter out directory characters */
	if (1) {
		const gchar * in;
		gchar * buf = g_malloc (filename->len*2);
		gchar * out = buf;
		for (in=filename->str; *in; ++in) {
			if (*in==G_DIR_SEPARATOR) {
				*out++ = '_';
			}
			else if (*in=='\\') {
				*out++ = '\\',
				*out++ = '\\';
			}
			else {
				*out++ = *in;
			}
		}
		*out = '\0';
		g_string_assign (filename, buf);
		g_free (buf);
	}

	/* add the directory & look for uniqueness */
	for (i=1;; ++i)
	{
		const char * front = filename->str;
		const gchar * lastdot = strrchr (front, '.');
		gchar * unique;
		gchar * lead;
		gchar * tail;
		if (lastdot == NULL) {
			lead = g_strdup (front);
			tail = g_strdup ("");
		} else {
			lead = g_strndup (front, lastdot-front);
			tail = g_strdup (lastdot);
		}

		if (i==1 && is_nonempty_string(path))
		{
			unique = g_strdup_printf ("%s%c%s%s",
			                          path, G_DIR_SEPARATOR,
			                          lead, tail);
		}
		else if (i==1)
		{
			unique = g_strdup_printf ("%s%s", lead, tail);
		}
		else if (is_nonempty_string(path))
		{
			unique = g_strdup_printf ("%s%c%s_%d%s",
			                          path, G_DIR_SEPARATOR,
			                          lead, i, tail);
		}
		else
		{
			unique = g_strdup_printf ("%s_%d%s", lead, i, tail);
		}

		/* cleanup */
		g_free (lead);
		g_free (tail);

		if (!file_exists(unique)) {
			g_string_assign (filename, unique);
			g_free (unique);
			break;
		}

		g_free (unique);
	}

	retval = filename->str;
	g_string_free (filename, FALSE);
	pan_normalize_filename_inplace (retval);
	debug_exit ("create_filename");
	return retval;
}


/*****
******
*****/

static void
write_big_article (GPtrArray * articles, FILE * fp_out)
{
	gint i;
	gboolean is_uu = FALSE;
	gboolean ignoring_uu_noise = FALSE;
	const gchar * cut_begin = "BEGIN -- Cut Here -- cut here";
	const size_t cut_begin_len = strlen (cut_begin);
	const gchar * cut_end = "END -- Cut Here -- cut here";
	const size_t cut_end_len = strlen (cut_end);
	debug_enter ("write_big_article");

	/* sanity clause */
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (articles->len);

	/* get the articles & find out how big the buffer must be */
	for (i=0; i!=articles->len; ++i)
	{
		const Article * a;
		gchar * message = NULL;
		gchar * body;

		/* get the article message */
		a = ARTICLE(g_ptr_array_index(articles,i));
		if (a != NULL)
			message = acache_get_message (article_get_message_id(a));
		if (message == NULL) {
			log_add_va (LOG_ERROR,
				_("Couldn't find body for message `%s' - decoding may be incomplete or incorrect"),
				article_get_subject(a));
			continue;
		}

		/* find the beginning of the body */
		body = pan_strstr (message, "\n\n");
		if (body != NULL)
			body += 2;

		/* keep the headers for the first item */
		if (i==0 && body!=NULL)
			fwrite (message, sizeof(char), body-message, fp_out);

		/* if we couldn't find the headers, assume they're missing.
		   AFAIK this won't happen; this is just CYA. */
		if (body == NULL)
			body = message;

		if (1)
		{
			gint line_len = 0;
			const gchar * march = body;
			const gchar * line_begin = NULL;

			while (get_next_token_run (march, '\n', &march, &line_begin, &line_len))
			{
				gboolean skip_line = ignoring_uu_noise;

				if (is_uu) {
					if (cut_end_len==line_len && !strncmp (line_begin, cut_end, cut_end_len))
						skip_line = ignoring_uu_noise = TRUE;
					else if (cut_begin_len==line_len && !strncmp (line_begin, cut_begin, cut_begin_len))
						ignoring_uu_noise = FALSE;
					else if (uu_is_ending_line(line_begin))
						is_uu = FALSE;
					else if (!is_uu_line (line_begin, line_len))
						skip_line = TRUE;
				} else {
					if (uu_is_beginning_line(line_begin))
						is_uu = TRUE;
				}

				if (!skip_line)
					fwrite (line_begin, sizeof(char), line_len+1, fp_out);
			}
		}

		/* cleanup */
		g_free (message);
	}

	debug_exit ("write_big_article");
}

static void
get_array_of_decodable_parts_func (GMimePart * part, gpointer data)
{
	GPtrArray * a = (GPtrArray*) data;
	const GMimeContentType * type;
       
	type = g_mime_part_get_content_type (part);
	if ((is_nonempty_string(g_mime_part_get_filename (part))) ||
		(!g_mime_content_type_is_type (type, "text", "*") && !g_mime_content_type_is_type (type, "multipart", "*")))
		g_ptr_array_add (a, part);
}

static void
get_decodable_parts (GPtrArray       * articles,
                     GMimeMessage   ** setme_msg,
                     GPtrArray      * fillme_array)
{
	FILE * fp;
	gchar * tmpfilename;
	debug_enter ("get_decodable_parts");

	/* sanity check */
	g_return_if_fail (articles!=NULL);

	/* get the message */
	fp = NULL;
	tmpfilename = pan_make_temp (&fp);
	if (fp != NULL)
	{
		/* write the articles to one big file */
		write_big_article (articles, fp);
		fclose (fp);
		fp = NULL;

		/* parse that file */
		fp = fopen (tmpfilename, "r");
		*setme_msg = pan_g_mime_parser_construct_message_from_file (fp);

		/* build the array of decodable parts */
		g_mime_message_foreach_part (*setme_msg, get_array_of_decodable_parts_func, fillme_array);
	}

	remove (tmpfilename);
	g_free (tmpfilename);
	debug_exit ("get_decodable_parts");
}

/**
*** Public Routines
**/

gboolean
decode_article (const decode_data * dd)
{
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
	GPtrArray * articles;
	GMimeMessage * mm;
	GPtrArray * attachments;
	GSList * p;
	gint i = 0;
	gboolean use_mutex;
	gboolean success = TRUE;
	Article * first;
	GString * filenames;
	debug_enter ("decode_article");

	/* sanity clause */
	g_return_val_if_fail (dd!=NULL, FALSE);
	g_return_val_if_fail (dd->server!=NULL, FALSE);
	g_return_val_if_fail (dd->articles!=NULL, FALSE);
	for (p=dd->articles; p!=NULL; p=p->next) {
		Article * a = ARTICLE(p->data);	
		g_return_val_if_fail (article_is_valid(a), FALSE);
		g_return_val_if_fail (acache_has_message(article_get_message_id(a)), FALSE);
	}

	first = ARTICLE(dd->articles->data);

	/* This is sad, really. individual GMimeMessages are safe enough;
	   the reason we use mutexes here is that decoding a big mp3 takes
	   up so _much_ memory that running two at once makes my system
	   swap to its knees. */
	use_mutex = g_slist_length (dd->articles) > 3;
	if (use_mutex) {
		status_item_emit_status_va (dd->item, _("Waiting to decode \"%s\""),
			article_get_subject(first));
		g_static_mutex_lock (&mutex);
	}

	/* let the client know what we're doing */
	status_item_emit_status_va (dd->item, _("Processing \"%s\""), article_get_subject(first));

	/* make a GPtrArray of the articles */
	articles = g_ptr_array_new ();
	for (p=dd->articles; p!=NULL; p=p->next)
		g_ptr_array_add (articles, p->data);

	/* get the decodable parts */
	status_item_emit_status_va (dd->item, _("Decoding \"%s\""), article_get_subject(first));
	mm = NULL;
	attachments = g_ptr_array_new ();
	get_decodable_parts (articles, &mm, attachments);

	/* decode & save the parts */
	success = attachments->len != 0;
	filenames = g_string_new (NULL);
	for (i=0; success && i!=attachments->len; ++i)
	{
		FILE * fp = NULL;
		const GMimePart * part = (const GMimePart*) g_ptr_array_index (attachments, i);
		const GMimeDataWrapper * content;
		gchar * filename = create_filename (dd->path, article_get_subject(first), dd->filename, part);

		/* remember this filename */
		g_string_sprintfa (filenames, "%s\n", filename);

		/* access the path */
		if (success) {
			gchar * path = g_dirname (filename);
			gboolean path_ok = directory_check (path);
			if (!path_ok) {
				log_add_va (LOG_ERROR, _("Decode can't access path \"%s\""), path);
				success = FALSE;
			}
			g_free (path);
		}

		/* open the output file */
		if (success) {
			errno = 0;
			fp = fopen (filename, "w+");
			if (fp == NULL) {
				log_add_va (LOG_ERROR, _("Decode unable to create file \"%s\" %s"),
						filename,
						errno==0 ? "" : g_strerror(errno));
				success = FALSE;
			}
		}


		/* make sure there's content to write */
		content = g_mime_part_get_content_object (part);
		if (!content)
			success = FALSE;

		/* write the content */
		if (success) {
			GMimeStream *stream = g_mime_stream_file_new (fp);
			g_mime_data_wrapper_write_to_stream ((GMimeDataWrapper*)content, (stream));
			g_mime_stream_unref (stream);
			fp = NULL;
			status_item_emit_status_va (dd->item, _("Saved \"%s\""), filename);
		}

		/* log the success */
		if (success) {
			gchar * msg = g_strdup_printf (
				_("Decoded \"%s\" from group \"%s\", \"%s\" part #%d"),
				filename,
				(dd->group ? dd->group->name : "Unknown"),
				article_get_subject(first),
				i+1);
			log_add (LOG_INFO, msg);
			debug0 (DEBUG_DECODE, msg);
			g_free (msg);
		}

		/* cleanup this iteration */
		g_free (filename);
	}

	/* remember the filename */
	if (filenames->len && !article_get_header (first, PAN_HEADER_FILENAME))
		article_set_header (first, PAN_HEADER_FILENAME, filenames->str, DO_CHUNK);

	/* update the node */
	articles_remove_flag (&first, 1, STATE_SAVE_QUEUED);
	articles_add_flag (&first, 1, (success?STATE_DECODED:STATE_DECODE_FAILED));

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	g_mime_message_destroy (mm);
	g_ptr_array_free (attachments, TRUE);
	g_string_free (filenames, TRUE);
	if (use_mutex)
		g_static_mutex_unlock (&mutex);
	debug_exit ("decode_article");

	return success;
}
