/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/task-body.h>
#include <pan/text.h>

/*********************
**********************  Private Function Prototypes
*********************/

static int task_body_run (Task * task);

static gchar * task_body_describe (const StatusItem * status);

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static void
task_body_destructor (PanObject * o)
{
	Group * group;
	TaskBody * task;
	debug_enter ("task_body_destructor");

	task = TASK_BODY(o);

	/* checkin the body */
	acache_file_checkin (&task->article->message_id, 1);

	/* unref the group + article */
	group = task->article->group;
	task->article = NULL;
	group_unref_articles (group, STATUS_ITEM(task));

	task_destructor (o);
	debug_exit ("task_body_destructor");
}

/************
*************  PUBLIC ROUTINES
************/


PanObject*
task_body_new (Article * a)
{
	TaskBody * body = NULL;
	gboolean body_exists;
	gboolean needs_socket;
	gboolean is_folder;
	const gchar * message_id;
	debug_enter ("task_body_new");

 	is_folder = group_is_folder (a->group);
	message_id = article_get_message_id (a);

	/* sanity clause */
	g_return_val_if_fail (article_is_valid(a), NULL);
	g_return_val_if_fail (group_is_valid(a->group), NULL);
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);

	/* ref the group + article */
	group_ref_articles (a->group, NULL);

	/* do we already have the body for this message? */
	body_exists = article_has_body (a);

	/* do we need to pull this body from the network? */
	needs_socket = TRUE;
	if (is_folder || body_exists) {
		acache_file_checkout (&message_id, 1);
		needs_socket = FALSE;
	}

	/* create the task object */
       	body = g_new0 (TaskBody, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_body_new: %p", body);

	/* initialize the parent class */
	task_constructor (TASK(body),
	                  task_body_destructor,
	                  task_body_describe,
	                  task_body_run,
	                  a->group->server, TRUE, needs_socket);
	TASK(body)->is_article_reader = TRUE;
	TASK(body)->gets_bodies = TRUE;

	/* initialize the task-body */
	debug1 (DEBUG_PAN_OBJECT, "task_body: %p", a);
	body->article = a;

	debug_exit ("task_body_new");
	return PAN_OBJECT(body);
}

/*****
******
*****/

static gchar*
task_body_describe (const StatusItem * status)
{
	const gchar * message_id;
	const gchar * subject;
	gchar * retval;
	Group * group;
	debug_enter ("task_body_describe");

	/* sanity clause */
	g_return_val_if_fail (status!=NULL, NULL);
	g_return_val_if_fail (article_is_valid(TASK_BODY(status)->article), NULL);
	g_return_val_if_fail (group_is_valid(TASK_BODY(status)->article->group), NULL);

	message_id = article_get_message_id (TASK_BODY(status)->article);
	subject = article_get_subject (TASK_BODY(status)->article);
	group = TASK_BODY(status)->article->group;

	retval = NULL;
	if (TASK(status)->needs_socket)
	{
		retval =  g_strdup_printf (_("Downloading body of \"%s\" (%s) from group %s"),
		                           message_id,
		                           subject,
		                           group_get_readable_name(group));
	}
	else
	{
		retval =  g_strdup_printf (_("Cached article \"%s\" (%s) from group %s"),
		                           message_id,
		                           subject,
		                           group_get_readable_name(group));
	}

	debug_exit ("task_body_describe");
	return retval;
}

/*****
******
*****/

static int
task_body_run (Task * task)
{
	StatusItem * status;
	TaskBody * body;
	Server * server;
	Group * group;
	Article * article;
	const gboolean * abort;
	const gchar * message_id;
	gboolean is_folder;
	debug_enter ("task_body_run");

	/* initialize variables */
	status = STATUS_ITEM(task);
	body = TASK_BODY(task);
	server = task->server;
	group = body->article->group;
	article = body->article;
	abort = &task->hint_abort;
	message_id = article_get_message_id (article);
	
	/* sanity clause */
	g_return_val_if_fail (server!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (group_is_valid(group), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(group->name), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (article_is_valid(article), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(message_id), TASK_FAIL_HOPELESS);

	/* let the user know what's happening */
	if (!task->needs_socket)
		status_item_emit_status_va (status,
			_("Cached Article \"%s\""), message_id );

	/* Check the cache before we download */
	is_folder = group_is_folder (article->group);
	if (!is_folder && !article_has_body(article))
	{
		gint result = 0;

		if (*abort) {
			debug_exit ("task_body_run");
			return TASK_FAIL;
		}

		/* set the group */
	        result = nntp_set_group (status, task->sock, group->name);
	        if (result != TASK_SUCCESS) {
			status_item_emit_error_va (status,
				_("Couldn't select group \"%s\""),
				group_get_readable_name(group));
			debug_exit ("task_body_run");
			return result;
		}

		/* download the article */
		if (*abort) {
			debug_exit ("task_body_run");
			return TASK_FAIL;
		}

		result = nntp_article_download (status,
		                                task->sock,
		                                group,
		                                article,
		                                &task->hint_abort,
		                                NNTP_VERBOSE,
		                                TRUE);
		if (result != TASK_SUCCESS)
		{
			status_item_emit_status_va (status,
				_("Downloading article \"%s\" failed."),
				article_get_subject(article));

			/* if user's expecting to see something but we've failed,
			   show them the error messages instead */
			if (status->errors!=NULL)
			{
				GString * tmp = g_string_sized_new (1024);
				GSList * l = status->errors;
				while (l != NULL) {
					g_string_append (tmp, (gchar*)l->data);
					g_string_append_c (tmp, '\n');
					l = l->next;
				}
				text_set_raw (tmp->str);
				g_string_free (tmp, TRUE);
			}

			debug_exit ("task_body_run");
			return result;
		}
	}

	/* read the message */
	text_set_from_cached_article (article);
	articles_set_read (&article, 1, TRUE);

	debug_exit ("task_body_run");
	return TASK_SUCCESS;
}
