/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <errno.h>
#include <string.h>

#include <pan/base/group.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/log.h>
#include <pan/base/newsrc.h>
#include <pan/base/newsrc-port.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

void
newsrc_import (Server          * server,
               const gchar     * filename_in,
               gboolean          subscribed_only)
{
	GArray * body;
	gchar * filename;

	/* sanity clause */
	g_return_if_fail (server!=NULL);
	g_return_if_fail (is_nonempty_string(filename_in));

	/* make sure the file exists */
	filename = pan_substitute (filename_in, "~", g_get_home_dir());
	g_return_if_fail (file_exists(filename));

	/* read the file into 'body'*/
	body = read_file (filename);
	if (body==NULL || !body->len)
	{
		log_add_va (LOG_ERROR, _("Error reading file `%s'"), filename);
	}
	else /* march through each line of newsrc... */
	{
		const gchar * pch = body->data;
		gint groups_imported_qty = 0;
		GString * line = g_string_new (NULL);
		GPtrArray * sub_y = g_ptr_array_new ();
		GPtrArray * sub_n = g_ptr_array_new ();
		GPtrArray * groups_to_add = g_ptr_array_new ();

		while (get_next_token_g_str (pch, '\n', &pch, line))
		{
			gchar * pch;

			/* get the group name */
			pch = strpbrk (line->str, "!:");
			if (pch!=NULL)
			{
				Newsrc * n;
				gboolean subscribed;
				gchar * name;
				Group * group;

				/* maybe skip non-subscribed groups */
				if (subscribed_only && *pch!=':')
					continue;

				/* get the group which this line is for. */
				name = g_strndup (line->str, pch-line->str);
				group = server_get_named_group (server, name);

				/* if we didn't have this group, add it */
				if (group == NULL) {
					group = group_new (server, name);
					g_ptr_array_add (groups_to_add, group);
				}

				/* import the newsrc line */
				n = group_get_newsrc (group);
				subscribed = FALSE;
				newsrc_import_line (n, &subscribed, line->str);
				g_ptr_array_add (subscribed?sub_y:sub_n, group);
				group_set_article_range (group, n->group_low, n->group_high);

				/* track how many groups imported */
				++groups_imported_qty;

				/* iteration cleanup */
				g_free (name);
			}
		}

		if (sub_y->len)
			groups_set_subscribed ((Group**)sub_y->pdata, sub_y->len, TRUE);
		if (sub_n->len)
			groups_set_subscribed ((Group**)sub_n->pdata, sub_n->len, FALSE);


		/* add any groups we didn't already have */
		if (groups_to_add->len != 0)
			server_add_groups (server,
					   (Group**)groups_to_add->pdata,
					   groups_to_add->len,
					   NULL,
					   NULL);

		/* log our progress */
		log_add_va (LOG_INFO, _("Imported %d groups (%d new) from `%s'"),
				      groups_imported_qty,
				      (int)groups_to_add->len,
				      filename);

		g_string_free (line, TRUE);
		g_ptr_array_free (sub_y, TRUE);
		g_ptr_array_free (sub_n, TRUE);
		g_ptr_array_free (groups_to_add, TRUE);
	}

	/* cleanup */
	g_free (filename);
	g_array_free (body, TRUE);
}

/*****
******
*****/

void
newsrc_export (Server          * server,
               const gchar     * filename_in,
               gboolean          subscribed_only,
               gboolean          backup_first)
{
	gint i;
	gint groups_exported_qty;
	gint subbed_exported_qty;
	ServerGroupsType type;
	FILE * fp;
	gchar * tmpfile;
	gchar * filename;
	GString * s;
	GPtrArray * a;

	/* sanity clause */
	g_return_if_fail (server!=NULL);
	g_return_if_fail (is_nonempty_string(filename_in));
	filename = pan_substitute (filename_in, "~", g_get_home_dir());
	filename_in = NULL;

	/* open a temp file to write to */
	fp = NULL;
	tmpfile = pan_make_temp (&fp);
	if (fp == NULL)
	{
		log_add_va (LOG_ERROR,
			_("Couldn't create file `%s' to export newsrc file %s"),
			filename,
			g_strerror(errno));
		g_free (filename);
		return;
	}

	/* get the groups */
	type = subscribed_only ? SERVER_GROUPS_SUBSCRIBED : SERVER_GROUPS_ALL;
	a = server_get_groups (server, type);

	/* walk through the groups */
	s = g_string_sized_new (4096);
	groups_exported_qty = 0;
	subbed_exported_qty = 0;
	for (i=0; i<a->len; ++i)
	{
		Group * g;
		Newsrc * n;

		/* get the group's newsrc */
		g = GROUP(g_ptr_array_index(a, i));
		n = group_has_newsrc(g) ? group_get_newsrc(g) : NULL;

		/* write its .newsrc line */
		g_string_truncate (s, 0);
		newsrc_export_line (n, g->name,
		                    group_is_subscribed(g), s);
		fprintf (fp, "%s\n", s->str);

		/* keep track of how many we've exported */
		++groups_exported_qty;
		if (group_is_subscribed(g))
			++subbed_exported_qty;
	}
	fclose (fp);

	/* backup? */
	if (backup_first) {
		gchar * backup = g_strdup_printf ("%s.bak", filename);
		pan_rename (filename, backup);
		g_free (backup);

	}

	/* move the temp file to newsrc */
	pan_rename (tmpfile, filename);

	/* log */
	log_add_va (LOG_INFO, _("Exported %d groups (%d subscribed) to `%s'"),
	                      groups_exported_qty,
	                      subbed_exported_qty,
	                      filename);

	/* cleanup */
	g_free (filename);
	g_free (tmpfile);
	g_ptr_array_free (a, TRUE);
	g_string_free (s, TRUE);
}
