/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/filters/filter-aggregate.h>
#include <pan/filters/filter-binary.h>
#include <pan/filters/filter-cached.h>
#include <pan/filters/filter-edit-ui.h>
#include <pan/filters/filter-manager.h>
#include <pan/filters/filter-new.h>
#include <pan/filters/filter-phrase.h>
#include <pan/filters/filter-read.h>
#include <pan/filters/filter-saved.h>
#include <pan/filters/filter-thread.h>
#include <pan/filters/filter-top.h>
#include <pan/filters/filter-ui.h>
#include <pan/filters/filter-manager.h>

#include <pan/articlelist.h>
#include <pan/article-list-toolbar-mediator.h>
#include <pan/article-toolbar.h>

static void
article_toolbar_changed_cb (gpointer call_obj, gpointer call_arg, gpointer data)
{
	Group * group;
	ArticleToolbarCallbackStruct * cbs = (ArticleToolbarCallbackStruct *) call_obj;
	debug_enter ("article_toolbar_changed_cb");

	/* update the articlelist */
	articlelist_set_filter (cbs->filter, cbs->filter_show);

	/* update the group's filter stats for posterity */
	group = articlelist_get_group ();
	if (group != NULL) {
		const gchar * filter_name = NULL;
		if (cbs->filter!=NULL && IS_FILTER_TOP(cbs->filter))
			filter_name = FILTER_TOP(cbs->filter)->name;
		group_set_filter (group, cbs->filter_bits, cbs->filter_show, filter_name);
	}

	debug_exit ("article_toolbar_changed_cb");
}

static void
articlelist_group_changed_cb (gpointer call_obj, gpointer call_arg, gpointer data)
{
	const Group * group;
	guint bits = ~0;
	gulong show = FILTER_SHOW_MATCHES_AND_REFERENCES;
	const gchar * name = NULL;
	debug_enter ("articlelist_group_changed_cb");

	/* determine which name, bits, & show to use */
	group = GROUP(call_arg);
	if (group != NULL) {
		bits = group->filter_bits;
		show = group->filter_show;
		name = group->filter_name;
	}

	/* notify the toolbar to update */
	pan_callback_remove (article_toolbar_get_filter_changed_callback(), article_toolbar_changed_cb, NULL);
	article_toolbar_set_filter (name, bits, show);
	pan_callback_add (article_toolbar_get_filter_changed_callback(), article_toolbar_changed_cb, NULL);

	debug_exit ("articlelist_group_changed_cb");
}

void
article_list_toolbar_mediator_init (void)
{
	pan_callback_add (article_toolbar_get_filter_changed_callback(),
	                  article_toolbar_changed_cb,
	                  NULL);
	pan_callback_add (articlelist_get_group_changed_callback(),
	                  articlelist_group_changed_cb,
	                  NULL);

}

/***
****
****  Filter Utilities
****
***/

Filter*
filter_create_from_bits_with_phrase (gulong               l,
                                     FilterPhraseState    phrase_state,
                                     const gchar        * phrase_str)
{
	Filter * retval;
	FilterAggregate * top;
	const gboolean show_new = (l & STATE_FILTER_NEW) != 0;
	const gboolean show_old = (l & STATE_FILTER_OLD) != 0;
	const gboolean show_unread = (l & STATE_FILTER_UNREAD) != 0;
	const gboolean show_read = (l & STATE_FILTER_READ) != 0;
	const gboolean show_bin = (l & STATE_FILTER_COMPLETE_BINARIES) != 0;
	const gboolean show_inc_bin = (l & STATE_FILTER_INCOMPLETE_BINARIES) != 0;
	const gboolean show_nonbinary = (l & STATE_FILTER_NONBINARIES) != 0;
	const gboolean show_watched = (l & STATE_FILTER_WATCHED) != 0;
	const gboolean show_ignored = (l & STATE_FILTER_IGNORED) != 0;
	const gboolean show_normal = (l & STATE_FILTER_NORMAL_RANK) != 0;
	const gboolean show_saved = (l & STATE_FILTER_SAVED) != 0;
	const gboolean show_idle = (l & STATE_FILTER_IDLE) != 0;
	const gboolean show_queued = (l & STATE_FILTER_QUEUED) != 0;
	const gboolean show_cached = (l & STATE_FILTER_CACHED) != 0;
	const gboolean show_non_cached = (l & STATE_FILTER_NOT_CACHED) != 0;
	gint n;
	debug_enter ("filter_create_from_bits_with_phrase");

	/* top */
	top = FILTER_AGGREGATE(filter_aggregate_new ());
	filter_aggregate_set_type (top, AGGREGATE_TYPE_AND);

	/* new/old */
	n = 0;
	if (show_new) ++n;
	if (show_old) ++n;
	if (n == 1) {
		Filter * filter = filter_new_article_new ();
		if (show_old)
			filter_negate (filter);
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* read/unread */
	n = 0;
	if (show_read) ++n;
	if (show_unread) ++n;
	if (n == 1) {
		Filter * filter = filter_read_new ();
		if (show_unread)
			filter_negate (filter);
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* attachments */
	n = 0;
	if (show_bin) ++n;
	if (show_inc_bin) ++n;
	if (show_nonbinary) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_binary_new ();
		if (n==1) {
			if (show_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_COMPLETE);
			else if (show_inc_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_INCOMPLETE);
			else if (show_nonbinary)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_NONBINARY);
		} else {
			if (!show_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_COMPLETE);
			else if (!show_inc_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_INCOMPLETE);
			else if (!show_nonbinary)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_NONBINARY);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}


	/* thread state */
	n = 0;
	if (show_watched) ++n;
	if (show_ignored) ++n;
	if (show_normal) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_thread_new ();
		if (n==1) {
			if (show_watched)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_WATCHED);
			else if (show_ignored)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_IGNORED);
			else if (show_normal)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_NOTHING);
		} else {
			if (!show_watched)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_WATCHED);
			else if (!show_ignored)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_IGNORED);
			else if (!show_normal)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_NOTHING);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* saved/queued/idle */
	n = 0;
	if (show_saved) ++n;
	if (show_queued) ++n;
	if (show_idle) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_saved_new ();
		if (n==1) {
			if (show_saved)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_SAVED);
			else if (show_queued)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_QUEUED);
			else if (show_idle)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_NOT_SAVED);
		} else {
			if (!show_saved)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_SAVED);
			else if (!show_queued)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_QUEUED);
			else if (!show_idle)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_NOT_SAVED);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* cached */
	n = 0;
	if (show_cached) ++n;
	if (show_non_cached) ++n;
	if (n == 1) {
		Filter * filter = filter_cached_new ();
		if (show_non_cached)
			filter_negate (filter);
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* phrase */
	if (is_nonempty_string(phrase_str)) {
		Filter * filter = filter_phrase_new ();
		filter_phrase_set (FILTER_PHRASE(filter), phrase_state, phrase_str, 0);
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* retval */
	if (filter_aggregate_child_size(top) != 1)
		retval = FILTER(top);
	else {
		retval = filter_aggregate_get_child_at (top, 0);
		pan_object_ref (PAN_OBJECT(retval));
		pan_object_unref (PAN_OBJECT(top)); /* remove old filter */
	}

	debug_exit ("filter_create_from_bits_with_phrase");
	return retval;
}

Filter*
filter_create_from_bits (const gchar   *    filter_name,
                         gulong             filter_bits)
{
	Filter * retval = NULL;
	
	if (is_nonempty_string (filter_name))
		retval = filter_manager_get_named_filter (filter_name);
	if (retval == NULL)
		retval = filter_create_from_bits_with_phrase (filter_bits, FILTER_PHRASE_SUBJECT, NULL);

	return retval;
}
