# -*- coding: utf-8 -*-

# Copyright (C) 2004-2007 Johan Svedberg <johan@svedberg.com>

# This file is part of ontv.

# ontv is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# ontv is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with ontv; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

import os.path
import sys
from gettext import gettext as _

import gtk
import gnome
import gobject
import vte

import GUI
import Utils

GRABBERS = {"Belgium": "tv_grab_be",
            "Brazil": "tv_grab_br",
            "Brazil (NET cable service)": "tv_grab_br_net",
            "Germany": "tv_grab_de_tvtoday",
            "Denmark": "tv_grab_dk",
            "Estonia": "tv_grab_ee",
            "Spain": "tv_grab_es",
            "Spain (laguiatv)": "tv_grab_es_laguiatv",
            "Finland": "tv_grab_fi",
            "France": "tv_grab_fr",
            "Hungary": "tv_grab_huro",
            "Romania": "tv_grab_huro",
            "Iceland": "tv_grab_is",
            "Italy": "tv_grab_it",
            "Japan": "tv_grab_jp",
            "North America": "tv_grab_na_dd",
            "Netherlands": "tv_grab_nl",
            "Netherlands (wolf)": "tv_grab_nl_wolf",
            "Norway": "tv_grab_no",
            "Reunion Island (France)": "tv_grab_re",
            "Sweden": "tv_grab_se_swedb",
            "United Kingdom (bleb.org)": "tv_grab_uk_bleb",
            "United Kingdom (rt)": "tv_grab_uk_rt",
            "South Africa": "tv_grab_za"}

class XMLTVAssistant(gtk.Assistant):

    __gsignals__ = {"apply": "override", "cancel": "override",
                    "close": "override", "prepare": "override"}

    def __init__(self, config, xmltvfile, country=None):
        gtk.Assistant.__init__(self)

        self.config = config
        self.xmltvfile = xmltvfile
        self.country = country

        self.props.title = _("OnTV XMLTV Configuration")

        self.__size_group = gtk.SizeGroup(gtk.SIZE_GROUP_HORIZONTAL)

        self.__previous_page = -1

        self.__create_page_1()
        self.__create_page_2()
        self.__create_page_3()
        self.__create_page_4()
        self.__create_page_5()

    def __create_page_1(self):
        page = gobject.new(gtk.Alignment, xalign=0.5, yalign=0.5, xscale=0,
                           yscale=0, border_width=6)

        vbox = gtk.VBox(spacing=6)

        page.add(vbox)

        label = gobject.new(gtk.Label, label=_("Welcome! This assistant will help you setup the program OnTV will use to download TV listings for your country."),
                            justify=gtk.JUSTIFY_FILL, use_markup=True,
                            wrap=True, xalign=0)
        vbox.pack_start(label, expand=True)

        page.show_all()

        self.append_page(page)
        self.set_page_header_image(page, GUI.load_icon("ontv"))
        self.set_page_title(page, _("OnTV XMLTV Configuration"))
        self.set_page_complete(page, True)
        self.set_page_type(page, gtk.ASSISTANT_PAGE_INTRO)

    def __create_page_2(self):
        page = gobject.new(gtk.Alignment, xalign=0.5, yalign=0.5, xscale=0,
                           yscale=0, border_width=6)

        vbox = gtk.VBox(spacing=6)

        page.add(vbox)

        label = gobject.new(gtk.Label, label=_("Please select which country you want to download TV listings for:"),
                            justify=gtk.JUSTIFY_FILL, use_markup=True,
                            wrap=True, xalign=0)
        vbox.pack_start(label, expand=True)

        self.country_combobox = gtk.combo_box_new_text()
        self.country_combobox.connect("changed",
                                      self.__country_combobox_changed, page)
        for country in GRABBERS.keys():
            if Utils.is_in_path(GRABBERS[country]):
                self.country_combobox.append_text(country)
        vbox.pack_start(self.country_combobox, expand=True)

        page.show_all()

        self.append_page(page)
        self.set_page_header_image(page, GUI.load_icon("ontv"))
        self.set_page_title(page, _("Select Country"))
        self.set_page_type(page, gtk.ASSISTANT_PAGE_CONTENT)

    def __country_combobox_changed(self, country_combobox, page):
        country_combobox_active = country_combobox.get_active()
        if country_combobox_active >= 0:
            country_combobox_model = country_combobox.get_model()
            self.selected_country = country_combobox_model[country_combobox_active][0]
            grabber = GRABBERS[self.selected_country]
            output = os.path.expanduser("~/.xmltv/%s.xmltv" % grabber)
            self.grabber_command = "%s %s %s" % (grabber, "--output", output)
            self.set_page_complete(page, True)
        else:
            self.set_page_complete(page, False)

    def __create_page_3(self):
        page = gobject.new(gtk.Alignment, xalign=0.5, yalign=0.5, xscale=0,
                           yscale=0, border_width=6)

        vbox = gtk.VBox(spacing=6)

        label = gobject.new(gtk.Label, label=_("Configure the grabber by answering the questions in the terminal below."),
                            justify=gtk.JUSTIFY_FILL, use_markup=True,
                            wrap=True, xalign=0)
        vbox.pack_start(label, expand=True)

        self.hbox = gtk.HBox()
        self.terminal = vte.Terminal()
        self.terminal.set_emulation("xterm")
        self.terminal.set_scrollback_lines(100)
        self.terminal.set_size(80, 25)
        self.hbox.pack_start(self.terminal)

        scrollbar = gtk.VScrollbar()
        scrollbar.set_adjustment(self.terminal.get_adjustment())
        self.hbox.pack_start(scrollbar)

        vbox.pack_start(self.hbox, expand=True)

        page.add(vbox)

        page.show_all()

        self.append_page(page)
        self.set_page_header_image(page, GUI.load_icon("ontv"))
        self.set_page_title(page, _("Configure grabber"))
        self.set_page_type(page, gtk.ASSISTANT_PAGE_CONTENT)


    def __create_page_4(self):
        page = gobject.new(gtk.Alignment, xalign=0.5, yalign=0.5, xscale=0,
                           yscale=0, border_width=6)

        vbox = gtk.VBox(spacing=6)

        label = gobject.new(gtk.Label, label=_("Please wait while the TV listings are being downloaded and sorted. This may take several minutes depending on the speed of your Internet connection."),
                            justify=gtk.JUSTIFY_FILL, use_markup=True,
                            wrap=True, xalign=0)
        vbox.pack_start(label, expand=True)


        self.progressbar = gtk.ProgressBar()
        vbox.pack_start(self.progressbar, expand=True)

        page.add(vbox)
        page.show_all()

        self.append_page(page)
        self.set_page_header_image(page, GUI.load_icon("ontv"))
        self.set_page_title(page, _("Downloading and sorting TV listings"))
        self.set_page_type(page, gtk.ASSISTANT_PAGE_PROGRESS)

    def __create_page_5(self):
        page = gobject.new(gtk.Alignment, xalign=0.5, yalign=0.5, xscale=0,
                           yscale=0, border_width=6)

        vbox = gtk.VBox(spacing=6)

        page.add(vbox)

        label = gobject.new(gtk.Label, label=_("Congratulations! OnTV is now ready for use. You can select which channels to monitor in the preferences dialog."),
                            justify=gtk.JUSTIFY_FILL, use_markup=True,
                            wrap=True, xalign=0)
        vbox.pack_start(label, expand=True)

        page.show_all()

        self.append_page(page)
        self.set_page_header_image(page, GUI.load_icon("ontv"))
        self.set_page_title(page, _("All done!"))
        self.set_page_complete(page, True)
        self.set_page_type(page, gtk.ASSISTANT_PAGE_SUMMARY)

    def do_apply(self):
        pass

    def do_cancel(self):
        self.destroy()

    def do_close(self):
        self.destroy()

    def do_prepare(self, page):
        page_number = self.get_current_page()
        page = self.get_nth_page(page_number)
        if page_number == 1 and self.country:
            GUI.set_active_from_string(self.country_combobox, self.country)
        if page_number == 2 and self.__previous_page == 1:
            pid = self.terminal.fork_command()
            self.terminal.connect("child-exited", self.__terminal_exited, page)
            self.terminal.feed_child("clear\n")
            grabber = self.grabber_command.split()[0]
            arguments = "--configure --quiet"
            configure_command = "%s %s" % (grabber, arguments)
            self.terminal.feed_child(configure_command + " && exit\n")
            font = self.terminal.get_font()
            font.set_size(10)
            self.terminal.grab_focus()
        elif page_number == 3 and self.__previous_page == 2:
            id = gobject.timeout_add(250, self.__pulse_progressbar,
                                     self.progressbar)
            self.xmltvfile.connect("downloading-done", self.__downloading_done,
                                   (self.progressbar, id, page))
            self.xmltvfile.download(self.grabber_command)
            self.progressbar.set_text(_("Downloading TV listings..."))

        self.__previous_page = page_number

    def __terminal_exited(self, terminal, page):
        self.set_page_complete(page, True)

    def __pulse_progressbar(self, progressbar):
        progressbar.pulse()
        return True

    def __downloading_done(self, xmltvfile, pid, condition, (progressbar, id,
                                                             page)):
        if Utils.is_in_path("tv_sort"):
            self.xmltvfile.connect("sorting-done", self.__sorting_done,
                                   (self.progressbar, id, page))
            arguments = self.grabber_command.split()
            if "--output" in arguments:
                index = arguments.index("--output") + 1
                output = arguments[index]
                self.xmltvfile.sort(output)
                progressbar.set_text(_("Sorting TV listings..."))
        else:
            self.__set_page_4_done(progressbar, id, page)

    def __sorting_done(self, xmltvfile, pid, condition, (progressbar, id,
                                                         page)):
        self.__set_page_4_done(progressbar, id, page)

    def __set_page_4_done(self, progressbar, id, page):
        gobject.source_remove(id)
        progressbar.set_fraction(1.0)
        progressbar.set_text(_("Done!"))
        self.config.grabber_command = self.grabber_command
        configured_grabbers = self.config.configured_grabbers
        if not self.selected_country in configured_grabbers:
            configured_grabbers.append(self.selected_country)
            self.config.configured_grabbers = configured_grabbers
        self.set_page_complete(page, True)
        self.xmltvfile.load()

# vim: set sw=4 et sts=4 tw=79 fo+=l:
