#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <scsi/scsi.h>
#include <scsi/sg.h>
#include <glib.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "nautilus-cd-burner.h"

static char *last_error = NULL;

static char **
read_lines (char *filename)
{
	char *contents;
	gsize len;
	char *p, *n;
	GPtrArray *array;
	
	if (g_file_get_contents (filename,
				 &contents,
				 &len, NULL)) {
		
		array = g_ptr_array_new ();
		
		p = contents;
		while ((n = memchr (p, '\n', len - (p - contents))) != NULL) {
			*n = 0;
			g_ptr_array_add (array, g_strdup (p));
			p = n + 1;
		}
		if ((gsize)(p - contents) < len) {
			g_ptr_array_add (array, g_strndup (p, len - (p - contents)));
		}

		g_ptr_array_add (array, NULL);
		
		g_free (contents);
		return (char **)g_ptr_array_free (array, FALSE);
	}
	return NULL;
}

struct scsi_unit {
	gboolean exist;
	char *vendor;
	char *model;
	char *rev;
	int bus;
	int id;
	int lun;
	int type;
};

struct cdrom_unit {
	char *device;
	gboolean can_write_cdr;
	gboolean can_write_cdrw;
	gboolean can_write_dvdr;
	gboolean can_write_dvdram;
};

static void
parse_sg_line (char *device_str, char *devices, struct scsi_unit *unit)
{
	char vendor[9], model[17], rev[5];
	int host_no, access_count, queue_depth, device_busy, online, channel;
	
	unit->exist = FALSE;
	
	if (strcmp (device_str, "<no active device>") == 0) {
		unit->exist = FALSE;
		return;
	}
	if (sscanf (device_str, "%8c\t%16c\t%4c", vendor, model, rev) != 3) {
		g_warning ("Couldn't match line in /proc/scsi/sg/device_strs\n");
		return;
	}
	vendor[8] = 0; model[16] = 0; rev[4] = 0;

	unit->vendor = g_strdup (g_strstrip (vendor));
	unit->model = g_strdup (g_strstrip (model));
	unit->rev = g_strdup (g_strstrip (rev));

	if (sscanf (devices, "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d",
		    &host_no,
		    &channel, &unit->id, &unit->lun, &unit->type,
		    &access_count, &queue_depth, &device_busy,
		    &online) != 9) {
		unit->bus = channel;
		g_warning ("Couldn't match line in /proc/scsi/sg/devices\n");
		return;
	}
	unit->exist = TRUE;
}

static int
count_strings (char *p)
{
	int n_strings;

	n_strings = 0;
	while (*p != 0) {
		n_strings++;
		while (*p != '\t' && *p != 0) {
			p++;
		}
		if (*p == '\t') {
			p++;
		}
	}
	return n_strings;
}

static int
get_cd_scsi_id (char dev, int *bus, int *id, int *lun)
{
	char *devname;
	int fd;
	struct {
		long mux4;
		long hostUniqueId;
	} m_idlun;
	
	
	devname = g_strdup_printf ("/dev/scd%c", dev);

	fd = open(devname, O_RDONLY | O_NONBLOCK);
	if (fd < 0) {
		g_free (devname);
		devname = g_strdup_printf ("/dev/sr%c", dev);
		fd = open(devname, O_RDONLY | O_NONBLOCK);
	}
	if (fd < 0) {
		g_warning ("Failed to open cd device\n");
		return 0;
	}
	g_free (devname);
    
	if (ioctl (fd, SCSI_IOCTL_GET_BUS_NUMBER, bus) < 0) {
		g_warning ("Failed to get scsi bus nr\n");
		close (fd);
		return 0;
	}
	if (ioctl (fd, SCSI_IOCTL_GET_IDLUN, &m_idlun) < 0) {
		g_warning ("Failed to get scsi id and lun\n");
		close(fd);
		return 0;
	}
	*id = m_idlun.mux4 & 0xFF;
	*lun = (m_idlun.mux4 >> 8)  & 0xFF;
	
	close(fd);
	return 1;
}

static struct scsi_unit *
lookup_scsi_unit (int bus, int id, int lun,
		  struct scsi_unit *units, int n_units)
{
	int i;

	for (i = 0; i < n_units; i++) {
		if (units[i].bus == bus &&
		    units[i].id == id &&
		    units[i].lun == lun) {
			return &units[i];
		}
	}
	return NULL;
}

static int
get_device_max_speed (char *id)
{
	int max_speed, i;
	const char *argv[20]; /* Shouldn't need more than 20 arguments */
	char *dev_str, *stdout_data, *speed;

	max_speed = 1;

	i = 0;
	argv[i++] = "cdrecord";
	argv[i++] = "-prcap";
	dev_str = g_strdup_printf ("dev=%s", id);
	argv[i++] = dev_str;
	argv[i++] = NULL;

	if (g_spawn_sync (NULL,
			  (char **)argv,
			  NULL,
			  G_SPAWN_SEARCH_PATH | G_SPAWN_STDERR_TO_DEV_NULL,
			  NULL, NULL,
			  &stdout_data,
			  NULL,
			  NULL,
			  NULL)) {
		speed = strstr (stdout_data, "Maximum write speed in kB/s:");
		if (speed != NULL) {
			speed += strlen ("Maximum write speed in kB/s:");
			max_speed = (int)floor (atol (speed) / 176.0 + 0.5);
		}
	}

	g_free (dev_str);
	return max_speed;
	
}

static void
add_linux_cd_recorder (struct cdrom_unit *cdrom,
		       struct scsi_unit *units,
		       int n_units)
{
	int bus, id, lun;
	struct scsi_unit *unit;
	CDRecorder *recorder;
	
	if (cdrom->device[0] == 's' &&
	    cdrom->device[1] == 'r') {
		get_cd_scsi_id (cdrom->device[2], &bus, &id, &lun);

		unit = lookup_scsi_unit (bus, id, lun, units, n_units);
		if (unit) {
			recorder = g_new0 (CDRecorder, 1);
			recorder->id = g_strdup_printf ("%d,%d,%d",
							bus, id, lun);
			recorder->name = g_strdup_printf ("%s - %s",
							  unit->vendor,
							  unit->model);
			recorder->max_speed = get_device_max_speed (recorder->id);
			if (cdrom->can_write_dvdr || cdrom->can_write_dvdram) {
				recorder->type = RECORDER_TYPE_DVD;
			} else {
				recorder->type = RECORDER_TYPE_CD;
			}
			
			recorders = g_list_append (recorders, recorder);
		}
	}
		
	
}

static void
linux_scan (void)
{
	char **device_str, **devices;
	char **cdrom_info;
	struct scsi_unit *units;
	struct cdrom_unit *cdroms;
	char *p, *t;
	int n_units, n_cdroms, i, j;
	int fd;

	/* Open /dev/sg0 to force loading of the sg module if not loaded yet */
	fd = open ("/dev/sg0", O_RDONLY);
	if (fd != -1) {
		close (fd);
	}
	
	devices = read_lines ("/proc/scsi/sg/devices");
	if (devices == NULL) {
		return;
	}
	device_str = read_lines ("/proc/scsi/sg/device_strs");
	if (device_str == NULL) {
		g_strfreev (devices);
		return;
	}

	for (n_units = 0; device_str[n_units] != NULL && devices[n_units] != NULL; n_units++) {
	}

	units = g_new0 (struct scsi_unit, n_units);
	for (i = 0; i < n_units; i++) {
		parse_sg_line (device_str[i], devices[i], &units[i]);
	}
	
	g_strfreev (device_str);
	g_strfreev (devices);
	    
	cdrom_info = read_lines ("/proc/sys/dev/cdrom/info");
	if (cdrom_info == NULL || cdrom_info[0] == NULL || cdrom_info[1] == NULL) {
		g_free (units);
		return;
	}

	if (!g_str_has_prefix (cdrom_info[2], "drive name:\t")) {
		g_free (units);
		return;
	}
	p = cdrom_info[2] + strlen ("drive name:\t");
	while (*p == '\t') {
		p++;
	}
	n_cdroms = count_strings (p);
	cdroms = g_new0 (struct cdrom_unit, n_cdroms);
	
	for (j = 0; j < n_cdroms; j++) {
		t = strchr (p, '\t');
		if (t != NULL) {
			*t = 0;
		}
		cdroms[j].device = g_strdup (p);
		if (t != NULL) {
			p = t + 1;
		}
	}

	for (i = 3; cdrom_info[i] != NULL; i++) {
		if (g_str_has_prefix (cdrom_info[i], "Can write CD-R:")) {
			p = cdrom_info[i] + strlen ("Can write CD-R:");
			while (*p == '\t') {
				p++;
			}
			for (j = 0; j < n_cdroms; j++) {
				cdroms[j].can_write_cdr = *p++ == '1';
				/* Skip tab */
				p++;
			}
		}
		if (g_str_has_prefix (cdrom_info[i], "Can write CD-RW:")) {
			p = cdrom_info[i] + strlen ("Can write CD-RW:");
			while (*p == '\t') {
				p++;
			}
			for (j = 0; j < n_cdroms; j++) {
				cdroms[j].can_write_cdrw = *p++ == '1';
				/* Skip tab */
				p++;
			}
		}
		if (g_str_has_prefix (cdrom_info[i], "Can write DVD-R:")) {
			p = cdrom_info[i] + strlen ("Can write DVD-R:");
			while (*p == '\t') {
				p++;
			}
			for (j = 0; j < n_cdroms; j++) {
				cdroms[j].can_write_dvdr = *p++ == '1';
				/* Skip tab */
				p++;
			}
		}
		if (g_str_has_prefix (cdrom_info[i], "Can write DVD-RAM:")) {
			p = cdrom_info[i] + strlen ("Can write DVD-RAM:");
			while (*p == '\t') {
				p++;
			}
			for (j = 0; j < n_cdroms; j++) {
				cdroms[j].can_write_dvdram = *p++ == '1';
				/* Skip tab */
				p++;
			}
		}
	}
	g_strfreev (cdrom_info);

	for (i = 0; i < n_cdroms; i++) {
		if (cdroms[i].can_write_cdr ||
		    cdroms[i].can_write_cdrw ||
		    cdroms[i].can_write_dvdr ||
		    cdroms[i].can_write_dvdram) {
			add_linux_cd_recorder (&cdroms[i], units, n_units);
		}
		    
	}
	
	
	g_free (units);
	g_free (cdroms);
}

GList *recorders = NULL;

void
scan_for_recorders (gboolean add_image)
{
	CDRecorder *recorder;

	linux_scan ();
	
	if (add_image) {
		/* File */
		recorder = g_new0 (CDRecorder, 1);
		recorder->name = _("File image");
		recorder->max_speed = 0;
		recorder->type = RECORDER_TYPE_FILE;
		
		recorders = g_list_append (recorders, recorder);
	}
	
}

CDRecorder *
get_recorder (int nr)
{
	return g_list_nth (recorders, nr)->data;
}

struct cdrecord_output {
	GMainLoop *loop;
	int result;
	int pid;
	int stdin;
	GString *line;
	GString *stderr;
	gboolean changed_text;
	gboolean send_return;
	gboolean expect_cdrecord_to_die;
};

static void
cancel_cdrecord (gpointer data)
{
	struct cdrecord_output *cdrecord_output = data;

	kill (cdrecord_output->pid, SIGINT);

	cdrecord_output->result = RESULT_CANCEL;
	g_main_loop_quit (cdrecord_output->loop);
}

static void
reload_dialog_response (GtkDialog *dialog, gint response_id, gpointer data)
{
	struct cdrecord_output *cdrecord_output = data;

	gtk_widget_destroy (GTK_WIDGET (dialog));

	if (cdrecord_output->send_return) {
		write (cdrecord_output->stdin, "\n", 1);
	} else {
		kill (cdrecord_output->pid, SIGUSR1);
	}
}

static void
insert_cd_response (GtkDialog *dialog, gint response_id, gpointer data)
{
	struct cdrecord_output *cdrecord_output = data;

	gtk_widget_destroy (GTK_WIDGET (dialog));

	cdrecord_output->result = RESULT_RETRY;
	g_main_loop_quit (cdrecord_output->loop);
}


static gboolean  
cdrecord_stdout_read (GIOChannel   *source,
		      GIOCondition  condition,
		      gpointer      data)
{
	struct cdrecord_output *cdrecord_output = data;
	char *line;
	char buf[1];
	unsigned int track, mb_written, mb_total;
	GIOStatus status;
	GtkWidget *reload_dialog;
	
	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);
	
	if (status == G_IO_STATUS_NORMAL) {
		if (cdrecord_output->line) {
			g_string_append (cdrecord_output->line, line);
			g_free (line);
			line = g_string_free (cdrecord_output->line, FALSE);
			cdrecord_output->line = NULL;
		}
		
		if (sscanf (line, "Track %2u: %d of %d MB written",
			    &track, &mb_written, &mb_total) == 3) {
			if (!cdrecord_output->changed_text) {
				cd_progress_set_text (_("Writing CD"));
			}

			cd_progress_set_fraction ((mb_total > 0) ? ((double)mb_written/mb_total) * 0.98 : 0.0);
		} else if (g_str_has_prefix (line, "Re-load disk and hit <CR>") ||
			   g_str_has_prefix (line, "send SIGUSR1 to continue")) {
			reload_dialog = gtk_message_dialog_new (cd_progress_get_window (),
								GTK_DIALOG_DESTROY_WITH_PARENT,
								GTK_MESSAGE_INFO,
								GTK_BUTTONS_OK,
								_("Please reload the disc in the CD writer."));
			gtk_window_set_title (GTK_WINDOW (reload_dialog),
					      _("Reload CD"));
			cdrecord_output->send_return = (*line == 'R');
			g_signal_connect (reload_dialog, "response", (GCallback)reload_dialog_response, cdrecord_output);
			gtk_widget_show (reload_dialog);
		} else if (g_str_has_prefix (line, "Fixating...")) {
			cd_progress_set_text (_("Fixating CD"));
		} else if (g_str_has_prefix (line, "Fixating time:")) {
			cd_progress_set_fraction (1.0);
			cdrecord_output->result = RESULT_FINISHED;
		} else if (g_str_has_prefix (line, "Last chance to quit, ")) {
			cd_progress_set_cancel_func (cancel_cdrecord, TRUE, &cdrecord_output);
		}

		g_free (line);
		
	} else if (status == G_IO_STATUS_AGAIN) {
		/* A non-terminated line was read, read the data into the buffer. */
		status = g_io_channel_read_chars (source, buf, 1, NULL, NULL);
		if (status == G_IO_STATUS_NORMAL) {
			if (cdrecord_output->line == NULL) {
				cdrecord_output->line = g_string_new (NULL);
			}
			g_string_append_c (cdrecord_output->line, buf[0]);
		}
	} else if (status == G_IO_STATUS_EOF) {
		return FALSE;
	}

	return TRUE;
}

static gboolean  
cdrecord_stderr_read (GIOChannel   *source,
		     GIOCondition  condition,
		     gpointer      data)
{
	struct cdrecord_output *cdrecord_output = data;
	char *line;
	GIOStatus status;
	GtkWidget *insert_cd_dialog;
	
	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);

	/* TODO: Handle errors */
	if (status == G_IO_STATUS_NORMAL) {
		g_string_prepend (cdrecord_output->stderr, line);
		if (strstr (line, "No disk / Wrong disk!") != NULL) {
			insert_cd_dialog = gtk_message_dialog_new (cd_progress_get_window (),
								   GTK_DIALOG_DESTROY_WITH_PARENT,
								   GTK_MESSAGE_INFO,
								   GTK_BUTTONS_OK,
								   _("Please insert a blank CD in the CD writer."));
			gtk_window_set_title (GTK_WINDOW (insert_cd_dialog),
					      _("Insert blank CD"));
			g_signal_connect (insert_cd_dialog, "response", (GCallback)insert_cd_response, cdrecord_output);
			gtk_widget_show (insert_cd_dialog);
			cdrecord_output->expect_cdrecord_to_die = TRUE;
		} else if (strstr (line, "Data may not fit on current disk") != NULL) {
			last_error = g_strdup (_("The files selected did not fit on the disc"));
		}
		g_free (line);
	} else if (status == G_IO_STATUS_EOF) {
		if (!cdrecord_output->expect_cdrecord_to_die) {
			g_main_loop_quit (cdrecord_output->loop);
		}
		return FALSE;
	} else {
		g_print ("cdrecord stderr read failed, status: %d\n", status);
	}

	return TRUE;
}

static void
details_clicked (GtkButton *button, gpointer data)
{
	struct cdrecord_output *cdrecord_output = data;
	GtkWidget *dialog, *label, *text, *scrolled_window;
	GtkTextBuffer *buffer;

	
	dialog = gtk_dialog_new_with_buttons (_("CD writing error details"),
					      cd_progress_get_window (),
					      GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
					      GTK_STOCK_CLOSE,
					      NULL);

	label = gtk_label_new (_("Detailed error output from cdrecord:"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
			    label,
			    FALSE, FALSE, 0);

	buffer = gtk_text_buffer_new (NULL);
	gtk_text_buffer_set_text (buffer, cdrecord_output->stderr->str, -1);
	
	text = gtk_text_view_new_with_buffer (buffer);
	g_object_unref (buffer);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (text), FALSE);
	gtk_widget_show (text);

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_ALWAYS);

	gtk_container_add (GTK_CONTAINER (scrolled_window), text);
	gtk_widget_show (scrolled_window);
	
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
			    scrolled_window,
			    TRUE, TRUE, 0);


	g_signal_connect (dialog, "response", (GCallback)gtk_widget_destroy, NULL);

	gtk_widget_show (dialog);
}

int
write_iso (CDRecorder *recorder,
	   const char *filename,
	   int speed,
	   gboolean eject,
	   gboolean dummy_write)
{
	struct cdrecord_output cdrecord_output;
	const char *argv[20]; /* Shouldn't need more than 20 arguments */
	char *speed_str, *dev_str;
	int stdout_pipe, stderr_pipe;
	guint stdout_tag, stderr_tag;
	GIOChannel *channel;
	GtkWidget *button, *dialog;
	GError *error;
	int i;

	i = 0;
	argv[i++] = "cdrecord";
	speed_str = g_strdup_printf ("speed=%d", speed);
	if (speed != 0) {
		argv[i++] = speed_str;
	}
	dev_str = g_strdup_printf ("dev=%s", recorder->id);
	argv[i++] = dev_str;
	if (dummy_write) {
		argv[i++] = "-dummy";
	}
	if (eject) {
		argv[i++] = "-eject";
	}
	argv[i++] = "-v";
	argv[i++] = "-data";
	argv[i++] = filename;
	argv[i++] = NULL;

	cdrecord_output.stderr = NULL;
 retry:
	cdrecord_output.result = RESULT_ERROR;
	cdrecord_output.expect_cdrecord_to_die = FALSE;
	cdrecord_output.line = NULL;
	if (cdrecord_output.stderr != NULL) {
		g_string_truncate (cdrecord_output.stderr, 0);
	} else {
		cdrecord_output.stderr = g_string_new (NULL);
	}
	
	cd_progress_set_text (_("Preparing to write CD"));
	cdrecord_output.changed_text = FALSE;
	cd_progress_set_fraction (0.0);
	cd_progress_set_cancel_func (cancel_cdrecord, FALSE, &cdrecord_output);

	error = NULL;
	if (!g_spawn_async_with_pipes  (NULL,
					(char **)argv,
					NULL,
					G_SPAWN_SEARCH_PATH,
					NULL, NULL,
					&cdrecord_output.pid,
					&cdrecord_output.stdin,
					&stdout_pipe,
					&stderr_pipe,
					&error)) {
		g_warning ("cdrecord command failed: %s\n", error->message);
		g_error_free (error);
		/* TODO: Better error handling */
	} else {
		/* Make sure we don't block on a read. */
		fcntl (stdout_pipe, F_SETFL, O_NONBLOCK);
		fcntl (stdout_pipe, F_SETFL, O_NONBLOCK);

		cdrecord_output.loop = g_main_loop_new (NULL, FALSE);
	
		channel = g_io_channel_unix_new (stdout_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stdout_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     cdrecord_stdout_read,
					     &cdrecord_output);
		g_io_channel_unref (channel);
		channel = g_io_channel_unix_new (stderr_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stderr_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     cdrecord_stderr_read,
					     &cdrecord_output);
		g_io_channel_unref (channel);
		
		cd_progress_set_cancel_func (cancel_cdrecord, FALSE, &cdrecord_output);
		
		g_main_loop_run (cdrecord_output.loop);
		g_main_loop_unref (cdrecord_output.loop);
		
		g_source_remove (stdout_tag);
		g_source_remove (stderr_tag);

		if (cdrecord_output.result == RESULT_RETRY) {
			goto retry;
		}
	}

	g_free (speed_str);
	g_free (dev_str);
	
	
	if (cdrecord_output.result == RESULT_ERROR) {
		dialog = gtk_message_dialog_new (cd_progress_get_window (),
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 last_error ? _("There was an error writing to the CD:\n%s") : _("There was an error writing to the CD"),
						 last_error);
		gtk_window_set_title (GTK_WINDOW (dialog),
				      _("Error writing to CD"));
		button = gtk_button_new_with_mnemonic (_("_Details"));
		gtk_widget_show (button);
		gtk_box_pack_end (GTK_BOX (GTK_DIALOG (dialog)->action_area),
				  button,
				  FALSE, TRUE, 0);
		g_signal_connect (button, "clicked", (GCallback)details_clicked, &cdrecord_output);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
	}

	g_string_free (cdrecord_output.stderr, TRUE);

	return cdrecord_output.result;
}
