/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*This file is part of GNU MlView
 *
 *GNU MlView is free software; 
 *you can redistribute it and/or modify it under the terms of 
 *the GNU General Public License as 
 *published by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope 
 *that it will be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include "mlview-validation-output.h"

static void xml_node_cut_cb (MlViewXMLDocument *a_xml_doc,
                             xmlNode *a_parent_node,
                             xmlNode *a_cut_node,
                             MlViewValidationOutput *a_this)
{
        guint i = 0;
        MlViewValidationMessage *msg = NULL;
        
        g_return_if_fail (a_xml_doc && MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_this && a_this->messages);
        
        for (i = 0 ; i < a_this->messages->len ; i++) {
                msg = g_array_index (a_this->messages, MlViewValidationMessage*, i);
                
                if (msg->node && xmlNodeIsChildOf (msg->node, a_cut_node))
                        msg->node = NULL;
        }
}

static void
xml_document_closed_cb (MlViewXMLDocument *a_xml_doc, MlViewValidationOutput *a_this)
{
        guint i = 0;
        MlViewValidationMessage *msg = NULL;
        
        g_return_if_fail (a_xml_doc && MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_this && a_this->messages);
        
        for (i = 0 ; i < a_this->messages->len ; i++) {
                msg = g_array_index (a_this->messages, MlViewValidationMessage*, i);
                
                msg->node = NULL;
        }
}

static void
xml_node_name_changed_cb (MlViewXMLDocument *a_xml_doc, 
                          xmlNode *a_node,
                          MlViewValidationOutput *a_this)
{
        guint i = 0;
        MlViewValidationMessage *msg = NULL;
        
        g_return_if_fail (a_xml_doc && MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_this && a_this->messages);
        
        for (i = 0 ; i < a_this->messages->len ; i++) {
                msg = g_array_index (a_this->messages, MlViewValidationMessage*, i);
                
                if (msg->node && msg->node == a_node)
                        msg->node = NULL;
        }
}

static void
connect_to_doc (MlViewValidationOutput *a_this, MlViewXMLDocument *a_doc)
{
        g_return_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc));
        g_return_if_fail (a_this);
        
        a_this->document = a_doc;

        g_signal_connect (G_OBJECT (a_doc), "node-cut",
                          G_CALLBACK (xml_node_cut_cb), a_this);
        g_signal_connect (G_OBJECT (a_doc), "document-closed",
                          G_CALLBACK (xml_document_closed_cb), a_this);
        g_signal_connect (G_OBJECT (a_doc), "name-changed", 
                          G_CALLBACK (xml_node_name_changed_cb), a_this);
}

static void
disconnect_from_doc (MlViewValidationOutput *a_this)
{
        g_return_if_fail (a_this && a_this->document && 
                          MLVIEW_IS_XML_DOCUMENT (a_this->document));    

        g_signal_handlers_disconnect_by_func (G_OBJECT (a_this->document),
                                              G_CALLBACK (xml_node_cut_cb),
                                              a_this);
        g_signal_handlers_disconnect_by_func (G_OBJECT (a_this->document),
                                              G_CALLBACK (xml_document_closed_cb),
                                              a_this);
        g_signal_handlers_disconnect_by_func (G_OBJECT (a_this->document),
                                              G_CALLBACK (xml_node_name_changed_cb),
                                              a_this);
}

MlViewValidationOutput *
mlview_validation_output_new (MlViewXMLDocument *a_doc) 
{
        MlViewValidationOutput *out = NULL;
        
        g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc), NULL);
        
        out = g_try_malloc (sizeof (MlViewValidationOutput));
        
        if (!out)
                goto cleanup;
        
        out->messages = g_array_new (FALSE, TRUE, sizeof (MlViewValidationMessage*));
        
        if (!out->messages)
                goto cleanup;
        
        connect_to_doc (out, a_doc);

        g_object_ref (G_OBJECT (a_doc));
        
        return out;
        
 cleanup:
        mlview_utils_trace_debug ("object creation problem !");

        if (out->messages) {
                g_array_free (out->messages, TRUE);
                out->messages = NULL;
        }

        if (out) {
                g_free (out);
                out = NULL;
        }

        return NULL;
}

void
mlview_validation_output_free (MlViewValidationOutput *a_this)
{
        guint i = 0;
        MlViewValidationMessage *msg = NULL;
        
        g_return_if_fail (a_this && a_this->messages);
        
        if (a_this->document)
                disconnect_from_doc (a_this);

        g_object_unref (G_OBJECT (a_this->document));

        for (i = 0 ; i < a_this->messages->len ; i++) {
                msg = g_array_index (a_this->messages, MlViewValidationMessage*, i);
                
                if (msg) {
                        g_free (msg);
                        msg = NULL;
                }
        }
        
        g_array_free (a_this->messages, TRUE);
        a_this->messages = NULL;

        g_free (a_this);
        a_this = NULL;
}
