/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <glade/glade.h>
#include <libgnomeui/libgnomeui.h>
#include "mlview-view-adapter.h"

/**
 *@file
 *The definition of the #MlViewViewAdapter class.
 */

struct _MlViewViewAdapterPrivate {
        guchar *name ;
        GtkWidget *set_name_dialog ;
        /*The xml document the view is connected to */
        MlViewXMLDocument *mlview_xml_document;

        /*The name of this view */
        gchar *view_name;

        /*the interactif name edition widget */
        GtkDialog *name_edition_dialog;
        gboolean dispose_has_run ;
};


/*signals*/
enum {
        /*must be the last one! */
        NUMBER_OF_SIGNALS
};

static GtkVBoxClass *parent_class = NULL;


static void mlview_view_adapter_class_init (MlViewViewAdapterClass * a_klass);

static void mlview_view_adapter_init (MlViewViewAdapter * a_mlview_xml_doc);

static enum MLVIEW_VIEW_ADAPTER_STATUS mlview_view_adapter_connect_to_document (MlViewViewAdapter * a_tree_view,
                                                                                MlViewXMLDocument * a_xml_doc);

static enum MLVIEW_VIEW_ADAPTER_STATUS mlview_view_adapter_disconnect_from_document (MlViewViewAdapter * a_this,
                                                                                               MlViewXMLDocument * a_xml_doc);


static GtkWidget *build_name_edition_dialog (MlViewAppContext *
                                             a_app_context);

static void  set_name_edition_widget_value (GtkWidget * a_edition_widget,
                                            gchar * a_value);
static const guchar *get_name_edition_widget_value (GtkWidget *a_edition_widget);

static void mlview_view_adapter_dispose (GObject *a_view) ;
static void mlview_view_adapter_finalize (GObject *a_view) ;

static enum MlViewStatus mlview_view_adapter_set_document (MlViewIView *a_this,
                                                           MlViewXMLDocument *a_mlview_xml_doc) ;

static enum MlViewStatus mlview_view_adapter_get_document (MlViewIView *a_this,
                                                           MlViewXMLDocument **a_doc) ;

static enum MlViewStatus mlview_view_adapter_get_name (MlViewIView *a_this,
                                                       guchar **a_name) ;

static enum MlViewStatus mlview_view_adapter_set_name_interactive (MlViewIView * a_this) ;

static enum MlViewStatus mlview_view_adapter_set_name (MlViewIView *a_this,
                                                       guchar * a_name) ;

static enum MlViewStatus mlview_view_adapter_connect_to_doc (MlViewIView *a_this,
                                                             MlViewXMLDocument *a_doc) ;

static enum MlViewStatus mlview_view_adapter_disconnect_from_doc (MlViewIView *a_this,
                                                                  MlViewXMLDocument *a_doc) ;


static enum MlViewStatus mlview_view_adapter_get_impl (MlViewIView *a_this, 
                                                       GtkWidget **a_impl)  ;

static const char *NAME_EDITION_ENTRY_FIELD_KEY = "name-edition-entry-field";

#define PRIVATE(object) ((object)->priv)


/****************************************************
 *Private (static) methods of this class.
 ****************************************************/

/*
 *mlview_view_adapter_class_init:
 *@a_klass: the current class
 *
 *Private initializer of the class. Part of the Gtk typing system framework.
 */
static void
mlview_view_adapter_class_init (MlViewViewAdapterClass *a_klass)
{
        GObjectClass *gobject_class = NULL ;

        g_return_if_fail (a_klass != NULL);
        parent_class = g_type_class_peek_parent (a_klass);
        g_return_if_fail (parent_class) ;
        gobject_class = G_OBJECT_CLASS (a_klass) ;

        gobject_class->dispose = mlview_view_adapter_dispose ;
        gobject_class->finalize = mlview_view_adapter_finalize ;
}


static void
mlview_view_adapter_iview_init (MlViewIView *a_iface)
{
        g_return_if_fail (a_iface) ;

        /******************************************
         *init the implementation of the 
         *MlViewIview interface
         ***************************************/
        a_iface->connect_to_doc = mlview_view_adapter_connect_to_doc ;
        a_iface->disconnect_from_doc = mlview_view_adapter_disconnect_from_doc ;
        a_iface->set_name = mlview_view_adapter_set_name ;
        a_iface->set_name_interactive = 
                mlview_view_adapter_set_name_interactive ;
        a_iface->get_name = mlview_view_adapter_get_name ;
        a_iface->set_document = mlview_view_adapter_set_document ;
        a_iface->get_document = mlview_view_adapter_get_document ;
        a_iface->get_impl = mlview_view_adapter_get_impl ;
        
}

/**
 *Connects the view adapter to the relevant signals
 *of the document object model.
 *@param a_this the current instance of #MlViewIView
 *@param a_doc the document object model to connect to
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
static enum MlViewStatus
mlview_view_adapter_connect_to_doc (MlViewIView *a_this,
                                    MlViewXMLDocument *a_doc)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        mlview_utils_trace_info 
                ("Ooops, you must implement the MlViewIView::connect_to_doc interface") ;
        return MLVIEW_ERROR ;
}

/**
 *Disconnect from the signal emited by the document
 *object model. That is, unregister of the callbacks registered
 *by mlview_view_adapter_connect_to_doc().
 *@param a_this the current instance of #MlViewIView
 *@param a_doc the document object model to disconnect from.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
static enum MlViewStatus
mlview_view_adapter_disconnect_from_doc (MlViewIView *a_this,
                                         MlViewXMLDocument *a_doc)
{
        g_return_val_if_fail (a_this
                              && MLVIEW_IS_VIEW_ADAPTER (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        mlview_utils_trace_info 
                ("Ooops, you must implement "
                 "the MlViewIView::disconnect_from_doc interface") ;
        return MLVIEW_ERROR ;
}

/**
 *Sets the document name to the new one. 
 *The new name is copied so that
 *the caller can free a_name if she wish. 
 *This function makes a_this
 *emit the "name-changed" signal. 
 *
 *@param a_this the document view.
 *@param a_name the new document name.
 *@return the
 */
static enum MlViewStatus
mlview_view_adapter_set_name (MlViewIView *a_this,
                              guchar * a_name)
{
        MlViewViewAdapter *adapter = NULL ;

        g_return_val_if_fail (a_this != NULL
                              && MLVIEW_IS_VIEW_ADAPTER (a_this),
                              MLVIEW_BAD_PARAM_ERROR);
        adapter = MLVIEW_VIEW_ADAPTER (a_this) ;
        g_return_val_if_fail (adapter, MLVIEW_BAD_PARAM_ERROR) ;
        if (PRIVATE (adapter)->view_name) {
                g_free (PRIVATE (adapter)->view_name);
                PRIVATE (adapter)->view_name = NULL;
        }
        if (a_name)
                PRIVATE (adapter)->view_name =
                        g_strdup (a_name);
        return MLVIEW_OK ;
}

static enum MlViewStatus
mlview_view_adapter_set_name_interactive (MlViewIView * a_this)
{
        MlViewViewAdapter *adapter = NULL ;
        gint button = 0;
        guchar *view_name = NULL;
        MlViewAppContext *app_context = NULL;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this != NULL
                              && MLVIEW_IS_VIEW_ADAPTER (a_this),
                              MLVIEW_BAD_PARAM_ERROR);

        adapter = MLVIEW_VIEW_ADAPTER (a_this) ;
        g_return_val_if_fail (PRIVATE (adapter) 
                              && PRIVATE 
                              (adapter)->mlview_xml_document,
                              MLVIEW_BAD_PARAM_ERROR) ;
        app_context =
                mlview_xml_document_get_app_context
                (PRIVATE (adapter)->mlview_xml_document);
        g_return_val_if_fail (app_context != NULL, MLVIEW_ERROR);
        if (PRIVATE (adapter)->name_edition_dialog == NULL) {
                PRIVATE (adapter)->name_edition_dialog =
                        GTK_DIALOG (build_name_edition_dialog
                                    (app_context));
        }
        g_return_val_if_fail 
                (PRIVATE (adapter)->name_edition_dialog, 
                 MLVIEW_ERROR) ;
        status = mlview_iview_get_name (MLVIEW_IVIEW (adapter), 
                                        &view_name) ;
        g_return_val_if_fail (status == MLVIEW_OK, MLVIEW_ERROR) ;
        set_name_edition_widget_value
                (GTK_WIDGET
                 (PRIVATE (adapter)->name_edition_dialog),
                 view_name);
        button = gtk_dialog_run (PRIVATE (adapter)->name_edition_dialog);
        switch (button) {
        case GTK_RESPONSE_ACCEPT: /*user clicked ok */
                view_name =
                        (guchar *) get_name_edition_widget_value
                        (GTK_WIDGET
                         (PRIVATE (adapter)->name_edition_dialog));

                if (view_name != NULL)
                        mlview_iview_set_name
                                (MLVIEW_IVIEW (adapter), view_name);
                break;
        default:
                break;
        }
        gtk_widget_hide 
                (GTK_WIDGET (PRIVATE (adapter)->name_edition_dialog));
        return MLVIEW_OK ;
}


/**
 *@param a_this the current instance of MlViewViewAdapter.
 *@return a pointer to the name of the current xml document. 
 *Note the caller should neither 
 */
static enum MlViewStatus
mlview_view_adapter_get_name (MlViewIView *a_this,
                              guchar **a_name)
{
        MlViewViewAdapter *adapter = NULL ;

        g_return_val_if_fail (a_this != NULL
                              && MLVIEW_IS_VIEW_ADAPTER (a_this)
                              && a_name, 
                              MLVIEW_BAD_PARAM_ERROR);

        adapter = MLVIEW_VIEW_ADAPTER (a_this) ;
        g_return_val_if_fail (PRIVATE (adapter), 
                              MLVIEW_BAD_PARAM_ERROR) ;
        *a_name = PRIVATE (adapter)->view_name;
        return MLVIEW_OK ;
}

/**
 *Associates an mlview xml document to the 
 *current instance of MlViewViewAdapter. 
 *
 *@param a_mlview_xml_doc the new mlview xml document 
 *to associate to the current instanceof MlViewViewAdapter. 
 *@param a_xml_doc_view the current instance of MlViewViewAdapter.
 *
 */
static enum MlViewStatus
mlview_view_adapter_set_document (MlViewIView *a_this,
                                  MlViewXMLDocument *a_mlview_xml_doc)
{
        MlViewViewAdapter *adapter = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_VIEW_ADAPTER (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        adapter = MLVIEW_VIEW_ADAPTER (a_this) ;
        g_return_val_if_fail (PRIVATE (adapter), 
                              MLVIEW_BAD_PARAM_ERROR);
        PRIVATE (adapter)->mlview_xml_document =
                a_mlview_xml_doc;
        return MLVIEW_OK ;
}


/**
 *@param a_this the current instance of MlViewViewAdapter.
 */
static enum MlViewStatus
mlview_view_adapter_get_document (MlViewIView *a_this,
                                  MlViewXMLDocument **a_doc)
{
        MlViewViewAdapter *adapter = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_VIEW_ADAPTER (a_this)
                              && a_doc,
                              MLVIEW_BAD_PARAM_ERROR);
        adapter = MLVIEW_VIEW_ADAPTER (a_this) ;
        g_return_val_if_fail (adapter && PRIVATE (adapter),
                              MLVIEW_BAD_PARAM_ERROR) ;
        *a_doc = PRIVATE (adapter)->mlview_xml_document;
        return MLVIEW_OK ;
}

static enum MlViewStatus
mlview_view_adapter_get_impl (MlViewIView *a_this, 
                              GtkWidget **a_impl) 
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_IVIEW (a_this)
                              && MLVIEW_IS_VIEW_ADAPTER (a_this)
                              && a_impl, MLVIEW_BAD_PARAM_ERROR) ;

        *a_impl = GTK_WIDGET (a_this) ;
        return MLVIEW_OK ;
}

/*
 *mlview_view_adapter_init:
 *@a_this: the current instance of MlViewViewAdapter.
 *
 *The initialyzer of the MlViewViewAdapter object. 
 *Part of the gtk typing system
 *framework.
 */
static void
mlview_view_adapter_init (MlViewViewAdapter *a_this)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_VIEW_ADAPTER
                          (a_this));

        g_return_if_fail (PRIVATE (a_this) == NULL);

        PRIVATE (a_this) =
                g_malloc0 (sizeof (MlViewViewAdapterPrivate));
}

/**
 *Connects the current view to the signals 
 *emitted by the document.
 *
 *After successfull completion of this function, 
 *each signal emitted by the document to notify
 *a change of its state will trigger a visual change
 *in the current view.
 *
 *The signals fired by the document are declared in the
 *global structure gv_signals_fired_by_document .
 *
 *Note that this function references (increases the ref count) of
 *the document a_xml_doc (calls mlview_xml_document_ref() ), so please, 
 *make sure the destructor of the current instance of #MlViewViewAdapter
 *calls mlview_xml_document_unref().
 *
 *@param a_this the "this pointer".
 *@param a_xml_doc the document which signals are to be connected to.
 *@return the status.
 */
static  enum MLVIEW_VIEW_ADAPTER_STATUS
mlview_view_adapter_connect_to_document (MlViewViewAdapter * a_this,
                                         MlViewXMLDocument * a_xml_doc) 
{

        g_return_val_if_fail (a_this != NULL
                              && MLVIEW_IS_VIEW_ADAPTER (a_this)
                              && a_xml_doc
                              && MLVIEW_IS_XML_DOCUMENT (a_xml_doc),
                              MLVIEW_BAD_PARAM_ERROR);

        /*
         *TODO:
         *do the connection here. That is,
         *call the document signal connection
         *code of each subwidgets here.
         */
        mlview_xml_document_ref (a_xml_doc);
        return MLVIEW_OK;
}


static enum MLVIEW_VIEW_ADAPTER_STATUS
mlview_view_adapter_disconnect_from_document (MlViewViewAdapter * a_this,
                                              MlViewXMLDocument * a_xml_doc)
{

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_VIEW_ADAPTER (a_this)
                              && a_xml_doc
                              && MLVIEW_IS_XML_DOCUMENT (a_xml_doc),
                              MLVIEW_VIEW_ADAPTER_BAD_PARAMETER) ;
        /*
         *TODO: discronnect from document here.
         *That is, call the disconnection code
         *of each subwidget here.
         */
        return MLVIEW_VIEW_ADAPTER_OK;
}


/**
 *Helper function. Builds the view name entry widget.
 *This widget embedds a GtkEntry widget wich actually holds the view name.
 *to get/set the that GtkEntry widget, you just have to invoke 
 *the gtk_object_get_data () or
 *gtk_object_set_data () using the key NAME_EDITION_ENTRY_FIELD_KEY. 
 *
 *@return the newly built name edition widget. 
 */
static GtkWidget *
build_name_edition_dialog (MlViewAppContext * a_app_context)
{
        GtkWidget *result = NULL;
        GtkWidget *name_entry_field = NULL;

        g_return_val_if_fail (a_app_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), NULL);

        result = gtk_dialog_new_with_buttons
                 (_("Type the name of the current view"), NULL,
                  GTK_DIALOG_MODAL, 
                  GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
                  GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,                  
                  NULL);
        gtk_dialog_set_default_response (GTK_DIALOG (result),
                                         GTK_RESPONSE_ACCEPT) ;

        name_entry_field = gtk_entry_new ();

        gtk_object_set_data (GTK_OBJECT (result),
                             NAME_EDITION_ENTRY_FIELD_KEY,
                             name_entry_field);
        gtk_box_pack_start_defaults (GTK_BOX 
                                     (GTK_DIALOG (result)->vbox),
                                     name_entry_field) ;

        gtk_widget_show_all (result);

        return result;
}


/**
 *
 */
static void
set_name_edition_widget_value (GtkWidget * a_edition_widget,
                               gchar * a_value)
{
        GtkEntry *name_entry = NULL;

        g_return_if_fail (a_edition_widget != NULL);
        g_return_if_fail (GTK_IS_DIALOG (a_edition_widget));

        name_entry =
                gtk_object_get_data (GTK_OBJECT
                                     (a_edition_widget),
                                     NAME_EDITION_ENTRY_FIELD_KEY);

        g_return_if_fail (name_entry != NULL);
        g_return_if_fail (GTK_IS_ENTRY (name_entry));
        if (a_value)
                gtk_entry_set_text (name_entry, a_value);
        else
                gtk_entry_set_text (name_entry, "");
}


static const guchar *
get_name_edition_widget_value (GtkWidget * a_edition_widget)
{
        GtkEntry *name_entry = NULL;

        g_return_val_if_fail (a_edition_widget != NULL, NULL);
        g_return_val_if_fail (GTK_IS_DIALOG (a_edition_widget),
                              NULL);

        name_entry =
                gtk_object_get_data (GTK_OBJECT
                                     (a_edition_widget),
                                     NAME_EDITION_ENTRY_FIELD_KEY);

        g_return_val_if_fail (name_entry != NULL, NULL);
        g_return_val_if_fail (GTK_IS_ENTRY (name_entry), NULL);

        return gtk_entry_get_text (name_entry);
}


static void
mlview_view_adapter_dispose (GObject *a_view)
{
        MlViewViewAdapter *view = NULL;
        MlViewXMLDocument *doc = NULL;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_if_fail (a_view
                          && MLVIEW_IS_VIEW_ADAPTER (a_view));

        view = MLVIEW_VIEW_ADAPTER (a_view);
        g_return_if_fail (PRIVATE (view));
        if (PRIVATE (view)->dispose_has_run == TRUE)
                return ;
        status = mlview_iview_get_document (MLVIEW_IVIEW (view), 
                                            &doc);
        
        if (doc)
                mlview_view_adapter_disconnect_from_document
                        (view, doc);
        if (PRIVATE (view)->view_name) {
                g_free (PRIVATE (view)->view_name);
                PRIVATE (view)->view_name = NULL;
        }
        if (doc) {
                mlview_xml_document_unref (doc);
                doc = NULL ;
        }
        if (parent_class && G_OBJECT_CLASS (parent_class)->dispose)
                G_OBJECT_CLASS (parent_class)->dispose (a_view) ;
        PRIVATE (view)->dispose_has_run = TRUE ;
}

static void
mlview_view_adapter_finalize (GObject *a_view)
{
        MlViewViewAdapter *view = NULL ;

        g_return_if_fail (a_view 
                          && MLVIEW_IS_VIEW_ADAPTER (a_view)) ;
        
        view = MLVIEW_VIEW_ADAPTER (a_view) ;
        g_return_if_fail (PRIVATE (view)) ;
        
        if (PRIVATE (view)) {
                g_free (PRIVATE (view)) ;
                PRIVATE (view) = NULL ;
        }
}

/*
   NOT USED ATM
 *
static enum MlViewStatus
mlview_view_adapter_set_name_interactive2 (MlViewIView *a_this)
{
        GladeXML *glade_xml = NULL ;
        MlViewViewAdapter *adapter = NULL ;
        GtkWidget *view_name_entry = NULL ;
        guchar *view_name ;
        gint button = 0 ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_VIEW_ADAPTER (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        adapter = MLVIEW_VIEW_ADAPTER (a_this) ;
        g_return_val_if_fail (adapter 
                              && PRIVATE (adapter), 
                              MLVIEW_ERROR) ;

        if (!PRIVATE (adapter)->set_name_dialog) {
                guchar *glade_file_path = NULL ;
                glade_file_path = gnome_program_locate_file 
                        (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                         PACKAGE "/ixml-doc-view-set-name-dialog.glade",
                         TRUE, NULL) ;
                g_return_val_if_fail (glade_file_path, 
                                      MLVIEW_ERROR) ;
                glade_xml = glade_xml_new 
                        (glade_file_path,
                         "MlViewDocViewNameDialog",
                         NULL) ;
                if (!glade_xml) {
                        mlview_utils_trace_info 
                                ("glade xml file loading failed") ;
                        return MLVIEW_ERROR ;
                }
                PRIVATE (adapter)->set_name_dialog =
                        glade_xml_get_widget 
                        (glade_xml, "MlViewDocViewNameDialog") ;
                if (!PRIVATE (adapter)->set_name_dialog) {
                        mlview_utils_trace_info
                                ("getting widget from glade failed") ;
                        goto cleanup ;
                }
                view_name_entry = glade_xml_get_widget
                        (glade_xml, "ViewNameEntry") ;
                if (!view_name_entry) {
                        mlview_utils_trace_info
                                ("getting widget from glade failed") ;
                        goto cleanup ;
                }
        }
        button = gtk_dialog_run
                (GTK_DIALOG (PRIVATE (adapter)->set_name_dialog)) ;
        switch (button) {
        case GTK_RESPONSE_OK:
                view_name = (guchar *)gtk_entry_get_text
                        (GTK_ENTRY (view_name_entry)) ;
                break ;
        case GTK_RESPONSE_CANCEL:
                goto cleanup ;
                break ;
        default:
                break ;
        }
        gtk_widget_hide (PRIVATE (adapter)->set_name_dialog) ;
        if (view_name) {
                status = mlview_iview_set_name 
                        (MLVIEW_IVIEW (a_this), view_name) ;
        }
        
 cleanup:
        if (glade_xml) {
                g_object_unref (glade_xml) ;
                glade_xml = NULL ;
        }
        return status ;
}
*/
/*********************************************
 *Public methods of this class.
 *********************************************/


/**
 *Instance intializer of the #MlViewViewAdapter
 *class.
 *Should be called by the constructor.
 *@param a_this the current instance of #MlViewViewAdapter
 *@param a_mlview_xml_doc the document object model this view
 *is supposed to work on.
 *@return MLVIEW_OK upon succesful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_view_adapter_construct (MlViewViewAdapter * a_this,
                               MlViewXMLDocument * a_mlview_xml_doc)
{
        g_return_val_if_fail (a_this != NULL
                              && MLVIEW_IS_VIEW_ADAPTER (a_this)
                              && a_mlview_xml_doc
                              && PRIVATE (a_this),
                              MLVIEW_VIEW_ADAPTER_BAD_PARAMETER);

        PRIVATE (a_this)->mlview_xml_document =
                a_mlview_xml_doc;

        return mlview_view_adapter_connect_to_document
                (a_this, a_mlview_xml_doc);
}

/**
 *The constructor of the #MlViewViewAdapter class
 *@param a_mlview_doc the document object model the
 *view is supposed to work on.
 */
MlViewViewAdapter *
mlview_view_adapter_new (MlViewXMLDocument * a_mlview_doc)
{
        MlViewViewAdapter *result = NULL;

        g_return_val_if_fail (a_mlview_doc != NULL, NULL);

        result = g_object_new (MLVIEW_TYPE_VIEW_ADAPTER,
                               NULL);

        if (mlview_view_adapter_construct
            (result, a_mlview_doc)
            == MLVIEW_VIEW_ADAPTER_OK) {
                return result;
        } else {
                return NULL;
        }
}


/**
 *References the current instance of #MlViewViewAdapter.
 *That is, increments the reference count associated to
 *the current instance of #MlViewXMLDocument.
 *
 *@param a_this the "this pointer".
 */
void
mlview_view_adapter_ref (MlViewViewAdapter * a_this)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_VIEW_ADAPTER (a_this)
                          && PRIVATE (a_this));

        gtk_widget_ref (GTK_WIDGET (a_this));
}


/**
 *Unreferences the current instance of #MlViewXMLDocumentVIew.
 *That is, decrements the reference count associated to
 *to the current instance of #MlViewXMLDocument.
 *
 *@param a_this in out parameter. A pointer to the this pointer.
 *If the ref count reaches zero and if the current instance has been
 *deleted, *a_this is set to NULL.
 */
void
mlview_view_adapter_unref (MlViewViewAdapter * a_this)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_VIEW_ADAPTER (a_this)
                          && PRIVATE (a_this));

        gtk_widget_unref (GTK_WIDGET (a_this));
}


/**
 *The type builder of this class. This method is part of the Gtk typing
 *system framework. 
 *@return the type identifier of #MlViewViewAdapter
 */
gint
mlview_view_adapter_get_type (void)
{
        static gint type_id = 0;

        if (type_id == 0) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewViewAdapterClass),
                        NULL, NULL,
                        (GClassInitFunc)
                        mlview_view_adapter_class_init,
                        NULL, NULL,
                        sizeof (MlViewViewAdapter),
                        0,
                        (GInstanceInitFunc)
                        mlview_view_adapter_init
                };
                static const GInterfaceInfo iview_info = {
                        (GInterfaceInitFunc) mlview_view_adapter_iview_init,
                        NULL,
                        NULL
                } ;                
                type_id = g_type_register_static (GTK_TYPE_VBOX,
                                                  "MlViewViewAdapter",
                                                  &type_info, 0);
                g_type_add_interface_static (type_id,
                                             MLVIEW_TYPE_IVIEW,
                                             &iview_info) ;
        }
        return type_id;
}


/**
 *Getter of the file descriptor associated the the document object
 *model being edited by this view
 *@param a_this the current instance of MlViewViewAdapter.
 */
MlViewFileDescriptor* 
mlview_view_adapter_get_file_descriptor (MlViewViewAdapter * a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_VIEW_ADAPTER
                              (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        return mlview_xml_document_get_file_descriptor
                (PRIVATE (a_this)->mlview_xml_document);
}

