/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it 
 *will be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the GNU 
 *General Public License along with MlView; 
 *see the file COPYING. If not, write to 
 *the Free Software Foundation, Inc., 
 *59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <string.h>
#include <glade/glade.h>
#include "mlview-tree-view.h"
#include "mlview-tree-editor2.h"
#include "mlview-icon-tree.h"

/**
 *@file
 *The definition of the #MlViewTreeView class.
 */
struct _MlViewTreeViewPrivate {
        guchar *name ;
        GtkWidget *set_name_dialog ;

        GtkNotebook *trees; /* all tree editors */
        xmlNode *current_node; /* current node */
        MlViewTreeEditor2 *tree_editor; /* current tree editor*/

        MlViewNodeEditor *node_editor;
        MlViewAppContext *app_context;
        MlViewXMLDocument *xml_doc ;
        GtkDialog *expand_tree_dialog;
        /*The main paned that divides the view in two main parts */
        GtkPaned *main_paned;
        GtkPaned *upper_paned1;
        GtkCList *feasible_children;
        GtkCList *feasible_prev_siblings;
        GtkCList *feasible_next_siblings;
        GtkCList *feasible_attributes ;        
        guint main_paned_percentage;
        gboolean dispose_has_run;
} ;

enum {
        DOCUMENT_CHANGED,
        SIGNAL_NUM
};

#define PRIVATE(tree_view) (tree_view->priv)

static void
visual_node_selected_cb (MlViewTreeEditor2 * a_editor,
                         GtkTreeRowReference *a_ref,
                         gpointer a_user_data);

static void
tree_selected_cb (GtkNotebook *notebook,
                  GtkNotebookPage *page, 
                  guint page_num,
                  gpointer data);

static void xml_node_changed_cb (MlViewNodeEditor * a_editor,
                                 gpointer a_data);

static void toggle_expand_to_leaves_cb (GtkToggleButton * a_toggle_button,
                                        gpointer * a_depth_entry);

static void update_feasible_attributes_list_cb (MlViewTreeEditor2 * a_editor,
                                                GtkTreeRowReference *a_ref,
                                                gpointer a_user_data) ;

static void update_feasible_next_siblings_list_cb (MlViewTreeEditor2 * a_editor,
                                                   GtkTreeRowReference * a_ref,
                                                   gpointer a_user_data);

static void update_feasible_prev_siblings_list_cb (MlViewTreeEditor2 * a_editor,
                                                   GtkTreeRowReference * a_ref,
                                                   gpointer a_user_data);

static void update_feasible_children_list_cb (MlViewTreeEditor2 * a_editor,
                                              GtkTreeRowReference *a_ref,
                                              gpointer a_user_data);

static void selected_a_possible_child_cb (GtkCList * a_possible_children,
                                          gint a_row, gint a_column,
                                          GdkEventButton * event,
                                          gpointer a_user_data);

static void selected_a_possible_next_sibling_cb (GtkCList * a_possible_next_siblings,
                                                 gint a_row, gint a_column,
                                                 GdkEventButton * event,
                                                 gpointer a_user_data);

static void selected_a_possible_prev_sibling_cb (GtkCList * a_possible_prev_siblings,
                                                 gint a_row, gint a_column,
                                                 GdkEventButton * event,
                                                 gpointer a_user_data);

static void doc_path_changed_cb (MlViewXMLDocument * a_xml_doc,
                                 gpointer a_xml_doc_tree_view);

static GtkDialog *get_expand_tree_dialog (MlViewTreeView * a_this);

static void mlview_tree_view_init (MlViewTreeView * a_tree_view);

static void mlview_tree_view_class_init (MlViewTreeViewClass * a_class);

static enum MlViewStatus mlview_tree_view_execute_action (MlViewIView *a_this,
                                                          MlViewAction *a_action) ;

static enum MlViewStatus mlview_tree_view_connect_to_doc (MlViewIView *a_this,
                                                          MlViewXMLDocument *a_doc)  ;

static enum MlViewStatus mlview_tree_view_disconnect_from_doc (MlViewIView *a_this,
                                                               MlViewXMLDocument *a_doc) ;

static void mlview_tree_view_dispose (GObject *a_this);

static void mlview_tree_view_finalise (GObject *a_this) ;

static MlViewViewAdapterClass *parent_class = NULL;
static guint p_signals[SIGNAL_NUM] = { 0 };

/*========================================
 *private (static) gtk framework functions
 *========================================*/

static void
mlview_tree_view_init (MlViewTreeView *a_this)
{
        MlViewIView *view_iface = NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_VIEW (a_this)
                          && MLVIEW_IS_IVIEW (a_this));

        if (PRIVATE (a_this) == NULL) {
                PRIVATE (a_this) =
                        g_try_malloc (sizeof (MlViewTreeViewPrivate));
                if (!PRIVATE (a_this)) {
                        mlview_utils_trace_info ("g_try_malloc failed") ;
                        return ;
                }
                memset (PRIVATE (a_this), 0, sizeof (MlViewTreeViewPrivate)) ;
        }

        view_iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
        g_return_if_fail (view_iface) ;
        view_iface->execute_action = mlview_tree_view_execute_action ;
        view_iface->connect_to_doc = mlview_tree_view_connect_to_doc ;
        view_iface->disconnect_from_doc = mlview_tree_view_disconnect_from_doc ;
}


/**
 *Class initialyzer.
 *Defines the signal "document-changed" and sets it default handler to NULL. 
 *
 */
static void
mlview_tree_view_class_init (MlViewTreeViewClass *a_class)
{
        GObjectClass *gobject_class = NULL ;

        g_return_if_fail (a_class != NULL);

        parent_class = g_type_class_peek_parent (a_class);
        gobject_class = G_OBJECT_CLASS (a_class) ;
        g_return_if_fail (gobject_class) ;

        gobject_class->dispose = mlview_tree_view_dispose ;
        gobject_class->finalize = mlview_tree_view_finalise ;

        p_signals[DOCUMENT_CHANGED] =
                g_signal_new ("document-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewTreeViewClass,
                               document_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);
        a_class->document_changed = NULL;
}


/**********************************************************
 *private helper function and other private methods.
 *********************************************************/
static void
doc_path_changed_cb (MlViewXMLDocument * a_xml_doc,
                     gpointer a_xml_doc_tree_view)
{
        MlViewFileDescriptor *file_desc = NULL;
        MlViewTreeView *tree_view = NULL;
        gchar *path = NULL;

        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_xml_doc_tree_view != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_xml_doc_tree_view));

        tree_view =
                MLVIEW_TREE_VIEW (a_xml_doc_tree_view);

        file_desc =
                mlview_xml_document_get_file_descriptor
                (a_xml_doc);

        g_return_if_fail (file_desc != NULL);

        path = mlview_file_descriptor_get_file_path (file_desc);

        g_return_if_fail (path != NULL);

        mlview_tree_view_set_xml_document_path
                (tree_view, path);

}

static void
mlview_tree_view_dispose (GObject *a_this)
{
        MlViewTreeView *thiz = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)) ;

        thiz = MLVIEW_TREE_VIEW (a_this) ;
        if (PRIVATE (thiz)->dispose_has_run == TRUE)
                return ;
        
        if (PRIVATE (thiz)->expand_tree_dialog) {
                gtk_widget_destroy
                        (GTK_WIDGET
                         (PRIVATE (thiz)->expand_tree_dialog));
                PRIVATE (thiz)->expand_tree_dialog = NULL;
        }
        if (PRIVATE (thiz)->xml_doc) {
                mlview_iview_disconnect_from_doc 
                        (MLVIEW_IVIEW (thiz), 
                         PRIVATE (thiz)->xml_doc) ;
                mlview_xml_document_unref
                        (PRIVATE (thiz)->xml_doc) ;
                PRIVATE (thiz)->xml_doc = NULL ;
        }
        /*call the destroy method of the parent */
        if (G_OBJECT_CLASS (parent_class)->dispose)
                G_OBJECT_CLASS (parent_class)->dispose
                        (a_this) ;
        PRIVATE (thiz)->dispose_has_run = TRUE ;
}

static void
mlview_tree_view_finalise (GObject *a_this)
{
        MlViewTreeView *thiz = NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_VIEW (a_this)) ;
        thiz = MLVIEW_TREE_VIEW (a_this) ;
        g_return_if_fail (thiz) ;
        if (PRIVATE (thiz)) {
                g_free (PRIVATE (thiz)) ;
                PRIVATE (thiz) = NULL ;
        }
}


/*========================================
 *private (static) helper functions
 *========================================*/

/**
  *This callback is called when the user
  *selects a tree in the trees' notebook.
  *
  */

static void
tree_selected_cb (GtkNotebook *notebook,
                  GtkNotebookPage *page, 
                  guint page_num,
                  gpointer data)
{
        MlViewTreeEditor2 *tree;
        MlViewTreeView *thiz = data;

        g_return_if_fail (thiz && MLVIEW_IS_TREE_VIEW (thiz));

        tree = MLVIEW_TREE_EDITOR2 (gtk_notebook_get_nth_page (notebook, page_num));
        
        g_return_if_fail (tree && MLVIEW_IS_TREE_EDITOR2 (tree));
        
        PRIVATE (thiz)->tree_editor = tree;
        /*if (PRIVATE (thiz)->current_node)
          mlview_tree_editor2_select_node (tree, PRIVATE (thiz)->current_node);*/
}

/**
 *This callback is called when an xml 
 *node is selected by the user. It calls the
 *mlview_node_editor_edit_xml_node() method of the MlViewNodeEditor object. 
 *
 */
static void
visual_node_selected_cb (MlViewTreeEditor2 * a_editor,
                         GtkTreeRowReference * a_ref,
                         gpointer a_user_data)
{
        MlViewTreeView *view = NULL;
        xmlNode *xml_node = NULL;
        MlViewXMLDocument *mlview_xml_doc = NULL;
        GtkTreeView *tree_view = NULL ;

        g_return_if_fail (a_editor
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)
                          && PRIVATE (a_editor)
                          && a_ref
                          && a_user_data) ;        
        tree_view = mlview_tree_editor2_get_tree_view (a_editor) ;
        g_return_if_fail (tree_view && GTK_IS_TREE_VIEW (tree_view)) ;
        view = a_user_data ;
        g_return_if_fail (MLVIEW_IS_TREE_VIEW (view)
                          && PRIVATE (view) 
                          && PRIVATE (view)->node_editor
                          && MLVIEW_IS_NODE_EDITOR 
                          (PRIVATE (view)->node_editor)) ;

        xml_node =
                mlview_tree_editor2_get_xml_node2 (a_editor, a_ref) ;
        g_return_if_fail (xml_node) ;

        PRIVATE (view)->current_node = xml_node;

        mlview_iview_get_document
                (MLVIEW_IVIEW (view), &mlview_xml_doc);
        g_return_if_fail (mlview_xml_doc);

        mlview_node_editor_edit_xml_node (PRIVATE (view)->node_editor,
                                          mlview_xml_doc, xml_node);
        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (view)->node_editor));
}

/**
 *The callback of the "element-changed signal" 
 *emited by the element editor of MlViewEditor 
 */
static void
xml_node_changed_cb (MlViewNodeEditor * a_node_editor,
                     gpointer a_data)
{
        GtkTreeView *visual_tree=NULL;
        xmlNodePtr xml_node=NULL;
        MlViewTreeView *view=NULL;
        gint i, nb_pages;
        MlViewTreeEditor2 *tree;

        g_return_if_fail (a_node_editor
                          && MLVIEW_IS_NODE_EDITOR (a_node_editor)
                          && a_data
                          && MLVIEW_IS_TREE_VIEW (a_data)) ;
        xml_node =
                mlview_node_editor_get_current_xml_node 
                (a_node_editor);
        g_return_if_fail (xml_node != NULL);

        /*get the visual tree */
        view = MLVIEW_TREE_VIEW (a_data);
        
        g_return_if_fail (view && PRIVATE (view)
                          && PRIVATE (view)->trees);
        nb_pages = gtk_notebook_get_n_pages (PRIVATE (view)->trees);
        for (i = 0; i < nb_pages; i++) {
                tree = MLVIEW_TREE_EDITOR2 (gtk_notebook_get_nth_page 
                                            (PRIVATE (view)->trees, i));
                g_return_if_fail (tree && MLVIEW_IS_TREE_EDITOR2 (tree));
                visual_tree = mlview_tree_editor2_get_tree_view (tree);
                g_return_if_fail (visual_tree && GTK_IS_TREE_VIEW (visual_tree));
                mlview_tree_editor2_update_visual_node2 (tree, xml_node); }
}

/**
 *
 */
static void
toggle_expand_to_leaves_cb (GtkToggleButton * a_toggle_button,
                            gpointer * a_depth_entry)
{
        g_return_if_fail (a_toggle_button != NULL);
        g_return_if_fail (GTK_IS_TOGGLE_BUTTON
                          (a_toggle_button));
        g_return_if_fail (a_depth_entry != NULL);
        g_return_if_fail (GTK_IS_WIDGET (a_depth_entry));

        if (gtk_toggle_button_get_active (a_toggle_button) ==
            TRUE)
                gtk_widget_set_sensitive (GTK_WIDGET
                                          (a_depth_entry),
                                          FALSE);
        else
                gtk_widget_set_sensitive (GTK_WIDGET
                                          (a_depth_entry), TRUE);
}

/**
 *This callback is invoked when the user selects a visual xml node. 
 *It's a callback function for the "node-selected" signal of
 *#MlViewTreeEditor2.
 *
 */
void
update_feasible_children_list_cb (MlViewTreeEditor2 * a_editor,
                                  GtkTreeRowReference * a_ref,
                                  gpointer a_user_data)
{
        MlViewTreeView *view = NULL;
        xmlNode *xml_node = NULL;
        GList *children_name_list = NULL;
        gint nb_of_names = 0;
        GtkTreeView *tree_view=NULL ;

        g_return_if_fail (a_editor 
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)
                          && a_ref && a_user_data
                          && MLVIEW_IS_TREE_VIEW 
                          (a_user_data));
        tree_view = mlview_tree_editor2_get_tree_view (a_editor) ;
        g_return_if_fail (tree_view 
                          && GTK_IS_TREE_VIEW (tree_view)) ;
        view = MLVIEW_TREE_VIEW (a_user_data);
        g_return_if_fail (PRIVATE (view) != NULL);
        gtk_clist_clear (PRIVATE (view)->feasible_children);
        xml_node = mlview_tree_editor2_get_xml_node2 (a_editor,
                                                      a_ref) ;
        g_return_if_fail (xml_node) ;
        if (xml_node->type == XML_ELEMENT_NODE) {

                nb_of_names =
                        mlview_parsing_utils_build_element_name_completion_list
                        (PRIVATE (view)->app_context,
                         ADD_CHILD,
                         xml_node, &children_name_list);

        }
        if (nb_of_names > 0) {
                gchar *clist_text[1] = { NULL };
                gchar *cur_name =
                        (gchar *) children_name_list->data;
                GList *cur = children_name_list;

                while (cur_name) {
                        clist_text[0] = cur_name;
                        gtk_clist_append (PRIVATE (view)->
                                          feasible_children,
                                          clist_text);

                        cur = (cur->next) ? cur->next : NULL;
                        cur_name =
                                (cur) ? (gchar *) cur->
                                data : NULL;
                }
        }
}

/**
 *This callback is invoked when the user selects a visual xml node. 
 *It is a callback of the "node-selected" signal.
 */
static void
update_feasible_prev_siblings_list_cb (MlViewTreeEditor2 *a_editor,
                                       GtkTreeRowReference *a_ref,
                                       gpointer a_user_data)
{
        MlViewTreeView *view = NULL;
        xmlNode *xml_node = NULL;
        GList *children_name_list = NULL;
        gint nb_of_names = 0;
        GtkTreeView *tree_view=NULL ;

        g_return_if_fail (a_editor
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)
                          && PRIVATE (a_editor)
                          && a_ref
                          && a_user_data) ;
        tree_view = mlview_tree_editor2_get_tree_view (a_editor) ;
        g_return_if_fail (tree_view) ;
        view = MLVIEW_TREE_VIEW (a_user_data);
        g_return_if_fail (PRIVATE (view) != NULL);

        gtk_clist_clear (PRIVATE (view)->feasible_prev_siblings);
        xml_node =
                mlview_tree_editor2_get_xml_node2 (a_editor,
                                                   a_ref) ;
        g_return_if_fail (xml_node);
        if (xml_node->type == XML_ELEMENT_NODE) {
                nb_of_names =
                        mlview_parsing_utils_build_element_name_completion_list
                        (PRIVATE (view)->app_context,
                         INSERT_BEFORE,
                         xml_node, &children_name_list);
        }
        if (nb_of_names > 0) {
                gchar *clist_text[1] = { NULL };
                gchar *cur_name =
                        (gchar *) children_name_list->data;
                GList *cur = children_name_list;

                while (cur_name) {
                        clist_text[0] = cur_name;
                        gtk_clist_append
                                (PRIVATE (view)->
                                 feasible_prev_siblings,
                                 clist_text);
                        cur = (cur->next) ? cur->next : NULL;
                        cur_name =
                                (cur) ? (gchar *) cur->
                                data : NULL;
                }
        }
}


/**
 *This callback is invoked when the user selects a visual xml node. 
 *This is a callback for the "node-selected" signal.
 */
static void
update_feasible_next_siblings_list_cb (MlViewTreeEditor2 *a_editor,
                                       GtkTreeRowReference *a_ref,
                                       gpointer a_user_data)
{
        MlViewTreeView *view = NULL;
        xmlNode *xml_node = NULL;
        GList *children_name_list = NULL;
        gint nb_of_names = 0;
        GtkTreeView *tree_view=NULL ;

        g_return_if_fail (a_editor
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)
                          && PRIVATE (a_editor)
                          && a_ref
                          && a_user_data) ;
        tree_view = mlview_tree_editor2_get_tree_view (a_editor) ;
        g_return_if_fail (tree_view) ;
        view = MLVIEW_TREE_VIEW (a_user_data);
        g_return_if_fail (PRIVATE (view) != NULL);
        gtk_clist_clear (PRIVATE (view)->feasible_next_siblings);
        xml_node = 
                mlview_tree_editor2_get_xml_node2 (a_editor, a_ref) ;
        g_return_if_fail (xml_node != NULL);
        if (xml_node->type == XML_ELEMENT_NODE) {
                nb_of_names =
                        mlview_parsing_utils_build_element_name_completion_list
                        (PRIVATE (view)->app_context,
                         INSERT_AFTER, xml_node,
                         &children_name_list);
        }
        if (nb_of_names > 0) {
                gchar *clist_text[1] = { NULL };
                gchar *cur_name =
                        (gchar *) children_name_list->data;
                GList *cur = children_name_list;

                while (cur_name) {
                        clist_text[0] = cur_name;
                        gtk_clist_append
                                (PRIVATE (view)->
                                 feasible_next_siblings,
                                 clist_text);
                        cur = (cur->next) ? cur->next : NULL;
                        cur_name =
                                (cur) ? (gchar *) cur->
                                data : NULL;
                }
        }
}


/**
 *This callback is invoked when the user selects a visual xml node. 
 *
 */

static void
update_feasible_attributes_list_cb (MlViewTreeEditor2 * a_editor,
                                    GtkTreeRowReference *a_ref,
                                    gpointer a_user_data)
{
        MlViewTreeView *view = NULL;
        xmlNode *xml_node = NULL;
        GList *children_name_list = NULL;
        gint nb_of_names = 0;
        GtkTreeView *tree_view=NULL ;

        g_return_if_fail (a_editor 
                          && MLVIEW_IS_TREE_EDITOR2 (a_editor)
                          && PRIVATE (a_editor)
                          && a_ref
                          && a_user_data) ;
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_user_data));

        tree_view = mlview_tree_editor2_get_tree_view (a_editor) ;
        g_return_if_fail (tree_view) ;
        view = MLVIEW_TREE_VIEW (a_user_data);
        g_return_if_fail (PRIVATE (view) != NULL);
        gtk_clist_clear (PRIVATE (view)->feasible_attributes);
        xml_node = mlview_tree_editor2_get_xml_node2 (a_editor,
                                                      a_ref) ;
        g_return_if_fail (xml_node != NULL);
        if (xml_node->type == XML_ELEMENT_NODE) {
                nb_of_names =
                        mlview_parsing_utils_build_attribute_name_completion_list
                        (PRIVATE (view)->app_context, xml_node,
                         &children_name_list, FALSE);
        }
        if (nb_of_names > 0) {
                gchar *clist_text[1] = { NULL };
                gchar *cur_name =
                        (gchar *) children_name_list->data;
                GList *cur = children_name_list;

                while (cur_name) {
                        clist_text[0] = cur_name;
                        gtk_clist_append (PRIVATE (view)->
                                          feasible_attributes,
                                          clist_text);
                        cur = (cur->next) ? cur->next : NULL;
                        cur_name =
                                (cur) ? (gchar *) cur->
                                data : NULL;
                }
        }
}


static void
selected_a_possible_child_cb (GtkCList * a_possible_children,
                              gint a_row, gint a_column,
                              GdkEventButton * event,
                              gpointer a_user_data)
{
        MlViewTreeView *doc_tree_view = NULL;
        GtkTreeIter cur_sel_start={0} ;
        gchar *element_name = NULL;
        gint ret_code = 0;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_if_fail (a_possible_children != NULL);
        g_return_if_fail (GTK_IS_CLIST (a_possible_children));
        g_return_if_fail (a_user_data != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_user_data));

        doc_tree_view = MLVIEW_TREE_VIEW (a_user_data);
        g_return_if_fail (PRIVATE (doc_tree_view) != NULL);
        g_return_if_fail (PRIVATE (doc_tree_view)->tree_editor !=
                          NULL);
        status = mlview_tree_editor2_get_cur_sel_start_iter
                (PRIVATE (doc_tree_view)->tree_editor,
                 &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        /*do not free element_name!! */
        ret_code = gtk_clist_get_text (a_possible_children,
                                       a_row, a_column,
                                       &element_name);
        if (ret_code == 1) {
                /*text retrieved ! */
                xmlNode *new_node = NULL;

                if (strcmp (element_name, "#PCDATA") == 0) {
                        new_node = xmlNewNode (NULL, "text");
                        new_node->type = XML_TEXT_NODE;
                } else {
                        new_node =
                                xmlNewNode (NULL, element_name);
                }
                mlview_tree_editor2_add_child_node 
                        (PRIVATE (doc_tree_view)->tree_editor,
                         &cur_sel_start, new_node) ;
        }
}

static void
selected_a_possible_prev_sibling_cb (GtkCList * a_possible_prev_siblings,
                                     gint a_row, gint a_column,
                                     GdkEventButton * event,
                                     gpointer a_user_data)
{
        MlViewTreeView *doc_tree_view = NULL;
        enum MlViewStatus stat = MLVIEW_OK ;
        GtkTreeIter cur_sel_start = {0} ;
        gchar *element_name = NULL;
        gint ret_code = 0;

        g_return_if_fail (a_possible_prev_siblings != NULL);
        g_return_if_fail (GTK_IS_CLIST
                          (a_possible_prev_siblings));
        g_return_if_fail (a_user_data != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_user_data));
        doc_tree_view = MLVIEW_TREE_VIEW (a_user_data);
        g_return_if_fail (PRIVATE (doc_tree_view) != NULL);
        g_return_if_fail (PRIVATE (doc_tree_view)->tree_editor !=
                          NULL);
        stat = mlview_tree_editor2_get_cur_sel_start_iter 
                (PRIVATE (doc_tree_view)->tree_editor,
                 &cur_sel_start) ;
        g_return_if_fail (stat == MLVIEW_OK) ;
        /*do not free element_name!! */
        ret_code = gtk_clist_get_text (a_possible_prev_siblings,
                                     a_row, a_column,
                                     &element_name);
        if (ret_code == 1) {
                /*text retrieved ! */
                xmlNode *new_node = NULL;
                if (strcmp (element_name, "#PCDATA") == 0) {
                        new_node = xmlNewNode (NULL, "text");
                        new_node->type = XML_TEXT_NODE;
                } else {
                        new_node =
                                xmlNewNode (NULL, element_name);
                }
                stat = mlview_tree_editor2_insert_sibling_node
                        (PRIVATE (doc_tree_view)->tree_editor,
                         &cur_sel_start, new_node, TRUE) ;
                g_return_if_fail (stat == MLVIEW_OK);
        }
}


static void
selected_a_possible_next_sibling_cb (GtkCList * a_possible_next_siblings,
                                     gint a_row, gint a_column,
                                     GdkEventButton * event,
                                     gpointer a_user_data)
{
        MlViewTreeView *doc_tree_view = NULL;
        GtkTreeIter cur_sel_start = {0} ;
        enum MlViewStatus stat = MLVIEW_OK ;
        gchar *element_name = NULL;
        gint ret_code = 0;

        g_return_if_fail (a_possible_next_siblings != NULL);
        g_return_if_fail (GTK_IS_CLIST
                          (a_possible_next_siblings));
        g_return_if_fail (a_user_data != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_user_data));
        doc_tree_view = MLVIEW_TREE_VIEW (a_user_data);
        g_return_if_fail (PRIVATE (doc_tree_view) != NULL);
        g_return_if_fail (PRIVATE (doc_tree_view)->tree_editor !=
                          NULL);
        stat = mlview_tree_editor2_get_cur_sel_start_iter
                (PRIVATE (doc_tree_view)->tree_editor,
                 &cur_sel_start) ;
        g_return_if_fail (stat == MLVIEW_OK) ;
        /*do not free element_name!! */
        ret_code = gtk_clist_get_text (a_possible_next_siblings,
                                       a_row, a_column,
                                       &element_name);
        if (ret_code == 1) {
                /*text retrieved ! */
                xmlNode *new_node = NULL;
                if (strcmp (element_name, "#PCDATA") == 0) {
                        new_node = xmlNewNode (NULL, "text");
                        new_node->type = XML_TEXT_NODE;
                } else {
                        new_node =
                                xmlNewNode (NULL, element_name);
                }
                stat = mlview_tree_editor2_insert_sibling_node
                        (PRIVATE (doc_tree_view)->tree_editor,
                         &cur_sel_start, new_node, FALSE) ;
        }
}


/**
 *
 */
static void
selected_a_possible_attribute_cb (GtkCList * a_possible_attributes,
                                  gint a_row, gint a_column,
                                  GdkEventButton * a_event,
                                  gpointer a_user_data)
{
        MlViewTreeView *doc_tree_view = NULL;
        MlViewXMLDocument *mlview_xml_doc = NULL;        
        enum MlViewStatus stat = MLVIEW_OK ;
        GtkTreeIter cur_sel_start = {0} ;
        GtkTreeView *visual_tree = NULL;
        gchar *attribute_name = NULL;
        gint ret_code = 0;

        g_return_if_fail (a_possible_attributes != NULL);
        g_return_if_fail (GTK_IS_CLIST (a_possible_attributes));
        g_return_if_fail (a_user_data != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_user_data));

        doc_tree_view = MLVIEW_TREE_VIEW (a_user_data);

        g_return_if_fail (PRIVATE (doc_tree_view) != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (doc_tree_view));
        g_return_if_fail (PRIVATE (doc_tree_view)->tree_editor !=
                          NULL);
        g_return_if_fail (PRIVATE (doc_tree_view)->node_editor !=
                          NULL);
        mlview_iview_get_document
                (MLVIEW_IVIEW (doc_tree_view), &mlview_xml_doc);
        g_return_if_fail (mlview_xml_doc != NULL);
        stat = mlview_tree_editor2_get_cur_sel_start_iter 
                (PRIVATE (doc_tree_view)->tree_editor,
                 &cur_sel_start) ;
        g_return_if_fail (stat == MLVIEW_OK) ;

        visual_tree = mlview_tree_editor2_get_tree_view 
                (PRIVATE (doc_tree_view)->tree_editor) ;
        g_return_if_fail (visual_tree != NULL);
        ret_code = gtk_clist_get_text 
                (a_possible_attributes, a_row, 
                 a_column, &attribute_name); /*do not free attribute_name!! */
        if (ret_code == 1) {
                /*text retrieved ! */
                xmlNode *xml_node = NULL;
                xmlAttr *attr = NULL;
                xml_node = mlview_tree_editor2_get_xml_node
                        (PRIVATE (doc_tree_view)->tree_editor,
                         &cur_sel_start) ;
                g_return_if_fail (xml_node != NULL);
                attr = xmlNewProp (xml_node,
                                   attribute_name,
                                   "value");
                stat = mlview_tree_editor2_update_visual_node 
                        (PRIVATE (doc_tree_view)->tree_editor,
                         &cur_sel_start) ;
                g_return_if_fail (stat == MLVIEW_OK) ;
                mlview_node_editor_edit_xml_node
                        (PRIVATE (doc_tree_view)->node_editor,
                         mlview_xml_doc, xml_node) ;                
        }
}

/**
 *This function builds a GtkDialog that will ask the user
 *to entrer the depth to which she wants the tree to be expanded.
 *The expansion depth is entered in a GtkEntry widget 
 *associated with the returned
 *dialog (using the function gtk_object_set_data ()) via the key "depth-entry".
 *The expand-to-leaves info is intered in a 
 *GtkCheckButton widget associated with the
 *returned dialg (using the function gtk_object_set_data ()) 
 *via the key "expand-to-leave".
 *The info intered by the user can then be recovered by using the function
 *gtk_object_get_data () using the key "depth-entry" 
 *the get the GtkEntry widget that
 *captured the depth entry if any or the 
 *key "expand-to-leaves" to get the GtkCheckButton widget
 *that captured the "expand-to-leaves" info. 
 *
 */
static GtkDialog *
get_expand_tree_dialog (MlViewTreeView * a_this) 
{
        GtkWidget *table = NULL,
                *label = NULL,
                *depth_entry = NULL;
        GtkWidget *expand_to_leaves = NULL;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_TREE_VIEW
                              (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        if (PRIVATE (a_this)->expand_tree_dialog == NULL) {
                PRIVATE (a_this)->expand_tree_dialog =
                        GTK_DIALOG
                        (gtk_dialog_new_with_buttons
                         (_("Choose the depth of the tree expansion"),
                          NULL, GTK_DIALOG_MODAL,
                          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                          GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,
                          NULL));
                g_return_val_if_fail 
                        (PRIVATE (a_this)->expand_tree_dialog, NULL) ;
                gtk_dialog_set_default_response 
                        (GTK_DIALOG 
                         (PRIVATE (a_this)->expand_tree_dialog),
                          GTK_RESPONSE_ACCEPT) ;
                expand_to_leaves =
                        gtk_check_button_new_with_label 
                        (_("expand to leaves"));

                gtk_box_pack_start_defaults
                        (GTK_BOX
                         (PRIVATE (a_this)->expand_tree_dialog->
                          vbox), expand_to_leaves);

                label = gtk_label_new (_
                                       ("absolute expansion depth:"));
                depth_entry = gtk_entry_new ();
                gtk_entry_set_text (GTK_ENTRY (depth_entry),
                                    "2");

                table = gtk_table_new (1, 2, FALSE);
                gtk_table_attach_defaults (GTK_TABLE (table),
                                           label, 0, 1, 0, 1);
                gtk_table_attach_defaults (GTK_TABLE (table),
                                           depth_entry, 1, 2, 0,
                                           1);

                gtk_box_pack_start_defaults
                        (GTK_BOX
                         (PRIVATE (a_this)->expand_tree_dialog->
                          vbox), table);

                g_signal_connect (G_OBJECT (expand_to_leaves),
                                  "toggled",
                                  G_CALLBACK
                                  (toggle_expand_to_leaves_cb),
                                  depth_entry);

                gtk_widget_show_all
                        (PRIVATE (a_this)->expand_tree_dialog->
                         vbox);

                gtk_object_set_data
                        (GTK_OBJECT
                         (PRIVATE (a_this)->expand_tree_dialog),
                         "expand-to-leaves", expand_to_leaves);

                gtk_object_set_data
                        (GTK_OBJECT
                         (PRIVATE (a_this)->expand_tree_dialog),
                         "depth-entry", depth_entry);
        }

        return PRIVATE (a_this)->expand_tree_dialog;
}


static enum MlViewStatus
mlview_tree_view_execute_action (MlViewIView *a_this,
                                 MlViewAction *a_action)
{
        MlViewTreeView *view = NULL ;
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_IVIEW (a_this)
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && a_action,
                              MLVIEW_BAD_PARAM_ERROR) ;

        view = MLVIEW_TREE_VIEW (a_this) ;
        g_return_val_if_fail (PRIVATE (view), MLVIEW_BAD_PARAM_ERROR) ;

        if (!strcmp
            (a_action->name,
             "add-child-node-interactive")) {
                mlview_tree_view_add_child_node_interactive 
                        (view) ;
        } else if (!strcmp 
                   (a_action->name, 
                    "insert-prev-sibling-node-interactive") ) {
                mlview_tree_view_insert_prev_sibling_node_interactive 
                        (view) ;
        } else if (!strcmp 
                   (a_action->name,
                    "insert-next-sibling-node-interactive") ) {
                mlview_tree_view_insert_next_sibling_node_interactive
                        (view);
        } else if (!strcmp (a_action->name,
                            "cut-node") ) {
                mlview_tree_view_cut_node (view) ;
        } else if (!strcmp (a_action->name, 
                            "copy-node") ) {
                mlview_tree_view_copy_node (view) ;
        } else if (!strcmp (a_action->name, 
                            "paste-node-as-child") ) {
                mlview_tree_view_paste_node_as_child 
                        (view) ;
        } else if (!strcmp (a_action->name, 
                            "paste-node-as-prev-sibling") ) {
                mlview_tree_view_paste_node_as_prev_sibling 
                        (view) ;
        } else if (!strcmp (a_action->name, 
                            "paste-node-as-next-sibling") ) {
                mlview_tree_view_paste_node_as_next_sibling 
                        (view) ;
        } else if (!strcmp (a_action->name, 
                            "expand-tree-to-depth-interactive") ) {
                mlview_tree_view_expand_tree_to_depth_interactive 
                        (view) ;
        } else if (!strcmp (a_action->name, 
                            "find-node-that-contains-str-interactive") ) {
                mlview_tree_view_find_xml_node_that_contains_str_interactive
                        (view) ;
        } else  {
                guchar *err_msg = NULL ;
                g_strconcat ("Unknown edition action: ",
                             a_action->name,
                             NULL) ;
                mlview_utils_trace_info (err_msg) ;
                if (err_msg) {
                        g_free (err_msg) ;
                        err_msg = NULL ;
                }
        }
        return MLVIEW_OK ;
}

static enum MlViewStatus
mlview_tree_view_connect_to_doc (MlViewIView *a_this,
                                 MlViewXMLDocument *a_doc) 
{
        MlViewTreeView *tree_view = NULL ;
        MlViewTreeEditor2 *tree;
        gint i, nb_pages;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_IVIEW (a_this)
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && a_doc
                              && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;

        tree_view = MLVIEW_TREE_VIEW (a_this) ;
        g_return_val_if_fail (tree_view 
                              && PRIVATE (tree_view)->trees,
                              MLVIEW_BAD_PARAM_ERROR) ;

        nb_pages = gtk_notebook_get_n_pages (PRIVATE (tree_view)->trees);
        for (i = 0; i < nb_pages; i++) {
                tree = MLVIEW_TREE_EDITOR2 
                        (gtk_notebook_get_nth_page (PRIVATE (tree_view)->trees, i));
                g_return_val_if_fail (tree && MLVIEW_IS_TREE_EDITOR2 (tree),
                                      MLVIEW_BAD_PARAM_ERROR);
                mlview_tree_editor2_connect_to_doc (tree, a_doc); }
      
        mlview_node_editor_connect_to_doc (PRIVATE
                                           (tree_view)->node_editor,
                                           a_doc) ;
        return MLVIEW_OK ;
}

static enum MlViewStatus
mlview_tree_view_disconnect_from_doc (MlViewIView *a_this,
                                      MlViewXMLDocument *a_doc)
{
        MlViewTreeView *tree_view = NULL ;
        MlViewTreeEditor2 *tree;
        gint i, nb_pages;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_IVIEW (a_this)
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && a_doc
                              && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;

        tree_view = MLVIEW_TREE_VIEW (a_this) ;
        g_return_val_if_fail (tree_view 
                              && PRIVATE (tree_view)->trees,
                              MLVIEW_BAD_PARAM_ERROR) ;

        nb_pages = gtk_notebook_get_n_pages (PRIVATE (tree_view)->trees);
        for (i = 0; i < nb_pages; i++) {
                tree = MLVIEW_TREE_EDITOR2 
                        (gtk_notebook_get_nth_page (PRIVATE (tree_view)->trees, i));
                g_return_val_if_fail (tree && MLVIEW_IS_TREE_EDITOR2 (tree),
                                      MLVIEW_BAD_PARAM_ERROR);
                mlview_tree_editor2_disconnect_from_doc (tree, a_doc); }
      
        mlview_node_editor_disconnect_from_doc (PRIVATE
                                                (tree_view)->node_editor,
                                                a_doc) ;
        return MLVIEW_OK ;
}

/*========================================
 *public gtk framework functions
 *========================================*/
guint
mlview_tree_view_get_type (void)
{
        static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewTreeViewClass),
                        NULL, /*base_init*/
                        NULL, /*base_finalize*/
                        (GClassInitFunc)
                        mlview_tree_view_class_init,
                        NULL, /*class_finalize*/
                        NULL, /*class data*/
                        sizeof (MlViewTreeView),
                        0, /*n_prealloc*/
                        (GInstanceInitFunc)
                        mlview_tree_view_init
                };                
                type = g_type_register_static
                        (MLVIEW_TYPE_VIEW_ADAPTER,
                         "MlViewTreeView", &type_info, 0);
        }
        return type;
}

/**
 *Getter of the current tree editor used by this tree view and
 *that is currently selected.
 *@param a_this the current instance of #MlViewTreeView
 *@return the tree editor.
 */
MlViewTreeEditor2 *
mlview_tree_view_get_tree_editor (MlViewTreeView *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        return PRIVATE (a_this)->tree_editor;
}

/**
 *Returns the notebook that contains the tree editors used
 *by the tree view.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@return the notebook that contains the tree editors, or
 *NULL in case of error.
 */
GtkNotebook *
mlview_tree_view_get_trees (MlViewTreeView *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);
        
        return PRIVATE (a_this)->trees;
}

/**
 *Gets the node editor used by the this tree view.
 *@param a_this the current instance of #MlViewTreeView.
 *@return the instance of #MlViewNodeEditor used by the tree view, or
 *NULL in case of error.
 */
MlViewNodeEditor *
mlview_tree_view_get_node_editor (MlViewTreeView *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        return PRIVATE (a_this)->node_editor;
}


/**
 *A constructor of the #MlViewTreeView class.
 *@param a_mlview_xml_doc the xml document object model 
 *the view must work on.
 *@param a_name the name of the view
 *@param a_app_context the mlview application context.
 *@return the newly created tree view, or NULL in case of an error.
 */
GtkWidget *
mlview_tree_view_new (MlViewXMLDocument * a_mlview_xml_doc, 
                      gchar * a_name,
                      MlViewAppContext * a_app_context)
{
        MlViewTreeView *tree_view = NULL;
        xmlDoc *xml_doc = NULL ;

        g_return_val_if_fail (a_mlview_xml_doc != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_mlview_xml_doc) != NULL,
                              NULL);

        tree_view =
                g_object_new (MLVIEW_TYPE_TREE_VIEW,
                              NULL);
        mlview_tree_view_construct (tree_view,
                                    a_mlview_xml_doc,
                                    a_name,
                                    a_app_context) ;
        mlview_iview_connect_to_doc (MLVIEW_IVIEW (tree_view),
                                     a_mlview_xml_doc) ;
        
        xml_doc = mlview_xml_document_get_xml_document 
                (a_mlview_xml_doc) ;

        if (xml_doc) {
                mlview_xml_document_select_node (a_mlview_xml_doc,
                                                 (xmlNode*)xml_doc) ;
        }
        return GTK_WIDGET (tree_view);
}

/**
 *Setter of the mlview application context.
 *@param a_this the current instance of #MlViewTreeView
 *@param a_app_context the new mlview application context.
 */
void
mlview_tree_view_set_app_context (MlViewTreeView *a_this,
                                  MlViewAppContext *a_app_context)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->app_context = a_app_context;

        if (PRIVATE (a_this)->tree_editor) {
                mlview_tree_editor2_set_application_context 
                        (PRIVATE (a_this)->tree_editor,
                         a_app_context) ;
        }

        if (PRIVATE (a_this)->node_editor) {
                mlview_node_editor_set_application_context
                        (PRIVATE (a_this)->node_editor,
                         a_app_context);
        }
}

/**
 *Instance initialyzer of the #MlViewTreeView class.
 *Should be called by the constructor.
 *@param a_this the current instance of #MlViewTreeView
 *@param a_mlview_xml_doc the xml document object model the
 *view must "work" on.
 *@param a_name the name of the view
 *@param a_app_context the mlview application context to
 *be associated with the tree view.
 */
void
mlview_tree_view_construct (MlViewTreeView *a_this,
                            MlViewXMLDocument *a_mlview_xml_doc,
                            gchar * a_name,
                            MlViewAppContext *a_app_context)
{
        GtkWidget *table = NULL,
                *scrolled = NULL;
        MlViewTreeEditor2 *raw_xml = NULL;
        MlViewTreeEditor2 *elements = NULL;
        gint i, nb_pages;
        MlViewTreeEditor2 *tree_editor;
    
        enum MLVIEW_VIEW_ADAPTER_STATUS status = NOK;
        const gchar *feasible_children_titles[1] =
                { _("Possible Children") };

        const gchar *feasible_prev_siblings_titles[1] =
                { _("Possible previous siblings") };

        const gchar *feasible_next_siblings_titles[1] =
                { _("Possible next siblings") };

        const gchar *feasible_attributes_titles[1] =
                { _("Possible attributes") };

        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_this));
        g_return_if_fail (a_mlview_xml_doc != NULL) ;
        PRIVATE (a_this)->xml_doc = a_mlview_xml_doc ;
        mlview_xml_document_ref (a_mlview_xml_doc) ;

        status = mlview_view_adapter_construct
                (MLVIEW_VIEW_ADAPTER (a_this),
                 a_mlview_xml_doc);
        g_return_if_fail (status == MLVIEW_VIEW_ADAPTER_OK);

        /*The main vpaned of the view */
        PRIVATE (a_this)->main_paned =
                GTK_PANED (gtk_vpaned_new ());

        gtk_box_pack_start (GTK_BOX (a_this),
                            GTK_WIDGET (PRIVATE (a_this)->
                                        main_paned), TRUE, TRUE,
                            0);
        gtk_widget_show (GTK_WIDGET
                         (PRIVATE (a_this)->main_paned));

        /*The trees and the trees' notebook */

        /* Notebook */ 
        PRIVATE (a_this)->upper_paned1 =
                GTK_PANED (gtk_hpaned_new ());
        gtk_paned_add1 (GTK_PANED
                        (PRIVATE (a_this)->main_paned),
                        GTK_WIDGET (PRIVATE (a_this)->
                                    upper_paned1));
        PRIVATE (a_this)->trees = GTK_NOTEBOOK (gtk_notebook_new ());
        gtk_notebook_set_tab_pos (PRIVATE (a_this)->trees, GTK_POS_BOTTOM);
        gtk_paned_add1 (GTK_PANED (PRIVATE (a_this)->upper_paned1),
                        GTK_WIDGET (PRIVATE (a_this)->trees));
        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->trees),
                          "switch-page",
                          G_CALLBACK (tree_selected_cb),
                          a_this);
        
        /* Trees */
        raw_xml = MLVIEW_TREE_EDITOR2 (mlview_tree_editor2_new (a_app_context));
        elements = MLVIEW_TREE_EDITOR2 (mlview_icon_tree_new (a_app_context));

        PRIVATE (a_this)->tree_editor = raw_xml;

        gtk_notebook_append_page (PRIVATE (a_this)->trees,
                                  GTK_WIDGET (elements),
                                  gtk_label_new (_("Elements"))); 
        gtk_notebook_append_page (PRIVATE (a_this)->trees, 
                                  GTK_WIDGET (raw_xml),
                                  gtk_label_new (_("Raw XML")));

        /*the table with the feasible children, siblings and attributes */
        PRIVATE (a_this)->feasible_children =
                GTK_CLIST (gtk_clist_new_with_titles
                           (1, (gchar**)feasible_children_titles));

        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->
                           feasible_children), "select-row",
                          G_CALLBACK
                          (selected_a_possible_child_cb),
                          a_this);

        PRIVATE (a_this)->feasible_prev_siblings =
                GTK_CLIST (gtk_clist_new_with_titles
                           (1, (gchar**)feasible_prev_siblings_titles));

        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->
                           feasible_prev_siblings), "select-row",
                          G_CALLBACK
                          (selected_a_possible_prev_sibling_cb),
                          a_this);

        PRIVATE (a_this)->feasible_next_siblings =
                GTK_CLIST (gtk_clist_new_with_titles
                           (1, (gchar**) feasible_next_siblings_titles));

        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->
                           feasible_next_siblings), "select-row",
                          G_CALLBACK
                          (selected_a_possible_next_sibling_cb),
                          a_this);

        PRIVATE (a_this)->feasible_attributes =
                GTK_CLIST
                (gtk_clist_new_with_titles
                 (1, (gchar**)feasible_attributes_titles));

        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->
                           feasible_attributes), "select-row",
                          GTK_SIGNAL_FUNC
                          (selected_a_possible_attribute_cb),
                          a_this);

        table = gtk_table_new (2, 2, TRUE);

        scrolled = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add
                (GTK_CONTAINER (scrolled),
                 GTK_WIDGET (PRIVATE (a_this)->
                             feasible_children));

        gtk_table_attach_defaults (GTK_TABLE (table), scrolled,
                                   0, 1, 0, 1);

        scrolled = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add (GTK_CONTAINER (scrolled),
                           GTK_WIDGET
                           (PRIVATE (a_this)->
                            feasible_prev_siblings));

        gtk_table_attach_defaults (GTK_TABLE (table), scrolled,
                                   1, 2, 0, 1);

        scrolled = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add (GTK_CONTAINER (scrolled),
                           GTK_WIDGET
                           (PRIVATE (a_this)->
                            feasible_next_siblings));

        gtk_table_attach_defaults (GTK_TABLE (table), scrolled,
                                   1, 2, 1, 2);

        scrolled = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add (GTK_CONTAINER (scrolled),
                           GTK_WIDGET
                           (PRIVATE (a_this)->
                            feasible_attributes));

        gtk_table_attach_defaults (GTK_TABLE (table), scrolled,
                                   0, 1, 1, 2);

        gtk_paned_pack2 (GTK_PANED
                         (PRIVATE (a_this)->upper_paned1),
                         table, FALSE, TRUE);

        /*The node editor */
        PRIVATE (a_this)->node_editor =
                MLVIEW_NODE_EDITOR 
                (mlview_node_editor_new
                 (a_app_context, a_mlview_xml_doc));

        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->node_editor),
                          "element-changed",
                          G_CALLBACK (xml_node_changed_cb),
                          a_this);

        gtk_paned_pack2 (GTK_PANED
                         (PRIVATE (a_this)->main_paned),
                         GTK_WIDGET (PRIVATE (a_this)->node_editor),
                         FALSE, TRUE);

        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (a_this)->
                              node_editor));

        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (a_this)->
                              main_paned));

        PRIVATE (a_this)->app_context = a_app_context ;

        /* Build trees */
        nb_pages = gtk_notebook_get_n_pages (PRIVATE (a_this)->trees);
        for (i = 0; i < nb_pages; i++) {
                tree_editor = MLVIEW_TREE_EDITOR2
                        (gtk_notebook_get_nth_page (PRIVATE (a_this)->trees, i));
                mlview_tree_editor2_edit_xml_doc
                        (tree_editor, a_mlview_xml_doc, NULL);
/*
                g_signal_connect (G_OBJECT
                                  (tree),
                                  "node-selected",
                                  G_CALLBACK (visual_node_selected_cb),
                                  a_this);
*/
                g_signal_connect (G_OBJECT
                                  (tree_editor),
                                  "node-selected",
                                  G_CALLBACK
                                  (update_feasible_children_list_cb),
                                  a_this);

                g_signal_connect (G_OBJECT
                                  (tree_editor),
                                  "node-selected",
                                  G_CALLBACK
                                  (update_feasible_prev_siblings_list_cb),
                                  a_this);
                g_signal_connect (G_OBJECT
                                  (tree_editor),
                                  "node-selected",
                                  G_CALLBACK
                                  (update_feasible_next_siblings_list_cb),
                                  a_this);
                g_signal_connect (GTK_OBJECT
                                  (tree_editor),
                                  "node-selected",
                                  G_CALLBACK
                                  (update_feasible_attributes_list_cb),
                                  a_this);
                
        }
        g_signal_connect (G_OBJECT (a_mlview_xml_doc),
                          "file-path-changed",
                          G_CALLBACK
                          (doc_path_changed_cb),
                          a_this) ;
}

/**
 *Interactively add a child node to the currently selected xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_add_child_node_interactive (MlViewTreeView * a_this)
{
        MlViewTreeEditor2 *tree_editor = NULL;
        tree_editor =
                mlview_tree_view_get_tree_editor
                (a_this);
        if (tree_editor)
                mlview_tree_editor2_add_child_node_interactive
                        (tree_editor);
}


/**
 *Interactively adds a previous sibling node to the currently selected
 *xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_insert_prev_sibling_node_interactive (MlViewTreeView * a_this)
{
        MlViewTreeEditor2 *tree_editor = NULL;
        tree_editor =
                mlview_tree_view_get_tree_editor
                (a_this);

        if (tree_editor) {
                mlview_tree_editor2_insert_prev_sibling_node_interactive
                        (tree_editor);
        }
}


/**
 *Interactively insert a next sibling node to the currently selected
 *xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_insert_next_sibling_node_interactive (MlViewTreeView * a_this) 
{
        MlViewTreeEditor2 *tree_editor = NULL;
        tree_editor =
                mlview_tree_view_get_tree_editor
                (a_this);

        if (tree_editor) {
                mlview_tree_editor2_insert_next_sibling_node_interactive
                        (tree_editor);
        }
}


/**
 *Cuts the currently selected xml node.
 *That is, unlink it from the tree and put it in
 *the xml node clipboard.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_cut_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor2 *tree_editor = NULL;
        GtkTreeIter cur_sel_start = {0};
        enum MlViewStatus status = MLVIEW_OK ;

        tree_editor =
                mlview_tree_view_get_tree_editor
                (a_this);

        if (tree_editor == NULL)
                return;
        status = mlview_tree_editor2_get_cur_sel_start_iter 
                (tree_editor, &cur_sel_start);
        g_return_if_fail (status == MLVIEW_OK) ;
        mlview_tree_editor2_cut_node (tree_editor,
                                      &cur_sel_start);
}


/**
 *Copy the currently selected xml node into the clipboard.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_copy_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor2 *tree_editor = NULL;
        GtkTreeIter cur_sel_start = {0} ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_if_fail (a_this != NULL);

        tree_editor =
                mlview_tree_view_get_tree_editor
                (a_this);
        if (tree_editor == NULL)
                return;
        status = mlview_tree_editor2_get_cur_sel_start_iter
                (tree_editor, &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        mlview_tree_editor2_copy_node (tree_editor,
                                       &cur_sel_start) ;
}


/**
 *Paste the last node found in the clipboard into the xml document
 *model.
 *The pasted node is pasted as a child node of the currently selected
 *xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_paste_node_as_child (MlViewTreeView * a_this) 
{
        GtkTreeIter cur_sel_start={0} ;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_if_fail (a_this
                          && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->tree_editor);

        status = mlview_tree_editor2_get_cur_sel_start_iter 
                (PRIVATE (a_this)->tree_editor,
                 &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        mlview_tree_editor2_paste_node_as_child
                (PRIVATE (a_this)->tree_editor,
                 &cur_sel_start);
}

/**
 *Paste the last node found in the clipboard into the xml document
 *model.
 *The pasted node is pasted as a previous sibling node of the 
 *currently selected xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_paste_node_as_prev_sibling (MlViewTreeView * a_this) 
{
        GtkTreeIter cur_sel_start={0} ;
        enum MlViewStatus status=MLVIEW_OK ;

        g_return_if_fail (a_this != NULL
                          && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this));

        status = mlview_tree_editor2_get_cur_sel_start_iter
                (PRIVATE (a_this)->tree_editor, 
                 &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        status = mlview_tree_editor2_paste_node_as_sibling
                (PRIVATE (a_this)->tree_editor, 
                 &cur_sel_start, TRUE) ;
}


/**
 *Pastes the last node found in the clipboard at
 *the place of the the currently selected xml node.
 *@param a_this a pointer to the current instance
 *of MlViewTreeView.
 */
void 
mlview_tree_view_paste_node_as_next_sibling (MlViewTreeView * a_this) 
{
        GtkTreeIter cur_sel_start={0} ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        status = mlview_tree_editor2_get_cur_sel_start_iter
                (PRIVATE (a_this)->tree_editor,
                 &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        mlview_tree_editor2_paste_node_as_sibling
                (PRIVATE (a_this)->tree_editor,
                 &cur_sel_start, FALSE) ;
}


/**
 *Sets the proportions of the the upper paned of the tree view.
 *@param a_this the current instance of #MlViewTreeView
 *@param a_percentage of width taken by the leftmost part of
 *the upper paned1
 */
void
mlview_tree_view_set_upper_paned1_proportions (MlViewTreeView * a_this,
                                               const guint a_percentage) 
{
        gint separator_position = 0;
        GtkWidget *top_level_widget = NULL;
        
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        top_level_widget =
                gtk_widget_get_toplevel (GTK_WIDGET
                                         (a_this));
        g_return_if_fail (top_level_widget != NULL);

        separator_position =
                top_level_widget->allocation.width *
                a_percentage / 100;

        gtk_paned_set_position (PRIVATE (a_this)->
                                upper_paned1,
                                separator_position);

        gtk_widget_show_all (GTK_WIDGET (a_this));
}


/**
 *Sets the GtkPaned top/bottom proportions of the main 
 *paned to a_percentage. This method gets the bounds of the top level widget
 *in order to determine the total size of the GtkVPaned 
 *(which actually has the same boundaries as the top level widget) and do
 *the calculation of the proportions of the top part vs bottom 
 *part of the GtkVPaned.
 *Therefore, this method must be called 
 *only *AFTER* the document has been opened and 
 *the top level of the document is shown. 
 *
 */
void
mlview_tree_view_set_main_paned_proportions (MlViewTreeView * a_this,
                                             const guint a_percentage) 
{
        guint separator_position = 0;
        GtkWidget *top_level_widget = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);
        top_level_widget =
                gtk_widget_get_toplevel (GTK_WIDGET
                                         (a_this));
        g_return_if_fail (top_level_widget != NULL);

        PRIVATE (a_this)->main_paned_percentage =
                a_percentage;

        separator_position =
                top_level_widget->allocation.height *
                a_percentage / 100;

        gtk_paned_set_position (PRIVATE (a_this)->
                                main_paned, separator_position);

        gtk_widget_show_all (GTK_WIDGET (a_this));
}


/**
 *Sets percentage taken by the upper part of the
 *view (tree editor and completion list) vs 
 *the lower part (node editor) , and, the percentage taken
 *by the left hand part (tree editor) vs right hand part 
 *(completion list)
 *@param a_this the current instance of #MlViewTreeView
 *@param a_top_down_percentage the percentage of height taken
 *by the upper part of the main paned
 *@param a_left_right_percentage the percentage of width taken
 *by the left part of the upper paned.
 */
void
mlview_tree_view_set_all_paned_proportions (MlViewTreeView * a_this,
                                            const guint a_top_down_percentage,
                                            const guint a_left_right_percentage) 
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->node_editor !=
                          NULL);

        mlview_tree_view_set_upper_paned1_proportions
                (a_this, a_left_right_percentage);
        mlview_tree_view_set_main_paned_proportions
                (a_this, a_top_down_percentage);
}


/**
 *Sets the title of the xml DOM to @a_file_path.
 *Updates this information in the tree editor and in the
 *XML DOM. 
 *@param a_this the current instance of #MlViewTreeView.
 *@param a_file_path the new file path.
 */
void
mlview_tree_view_set_xml_document_path (MlViewTreeView * a_this, 
                                        gchar * a_file_path) 
{
        guchar *base_name = NULL;

        g_return_if_fail (a_file_path != NULL);
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (PRIVATE (a_this)->tree_editor) {
                mlview_tree_editor2_set_xml_document_path
                        (PRIVATE (a_this)->tree_editor,
                         a_file_path) ;
        }
        base_name = (guchar *) g_basename (a_file_path);

        mlview_iview_set_name
                (MLVIEW_IVIEW (a_this), base_name);
}


/**
 *Opens a dialog box to ask the user the depth to which the tree
 *should be expanded and expands the tree to that depths. 
 *@param a_this the current instance of #MlViewTreeView
 */
void 
mlview_tree_view_expand_tree_to_depth_interactive (MlViewTreeView * a_this) 
{
        GtkDialog *dialog = NULL;
        GtkWidget *expand_to_leaves = NULL,
                *depth_entry = NULL;
        gint button = 0,
                depth = 0;
        gchar *depth_str = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        dialog = get_expand_tree_dialog (a_this);
        g_return_if_fail (dialog != NULL);

        button = gtk_dialog_run (dialog);

        expand_to_leaves =
                gtk_object_get_data (GTK_OBJECT (dialog),
                                     "expand-to-leaves");
        g_return_if_fail (expand_to_leaves != NULL);

        depth_entry = gtk_object_get_data (GTK_OBJECT (dialog),
                                           "depth-entry");
        g_return_if_fail (depth_entry != NULL);

        switch (button) {
        case GTK_RESPONSE_ACCEPT: /*user clicked OK */
                if (gtk_toggle_button_get_active
                    (GTK_TOGGLE_BUTTON (expand_to_leaves)) ==
                    TRUE) {
                        mlview_tree_editor2_expand_tree_to_depth
                                (PRIVATE (a_this)->tree_editor,
                                 -1);
                } else {
                        depth_str = (guchar *)
                                gtk_entry_get_text (GTK_ENTRY
                                                    (depth_entry));

                        if (depth_str) {
                                depth = atoi (depth_str);
                                mlview_tree_editor2_expand_tree_to_depth
                                        (PRIVATE
                                         (a_this)->tree_editor, 
                                         depth);
                        }
                }
                break;
        case GTK_RESPONSE_REJECT: /*user clicked CANCEL */
        case GTK_RESPONSE_CLOSE: /*closed the dialog box */
        default:               /*the sky falls on our heads */
                break;
        }

        gtk_widget_hide (GTK_WIDGET (dialog));
}

/**
 *Finds the graphical node that contains the given string.
 *@param a_this a pointer to the current instance of
 *#MlViewTreeView.
 *@return the graphical xml node that contains the given string.
 */
void
mlview_tree_view_find_xml_node_that_contains_str_interactive (MlViewTreeView * a_this) 
{
        MlViewTreeEditor2 *tree_editor = NULL;

        g_return_if_fail (a_this != NULL);

        tree_editor =
                mlview_tree_view_get_tree_editor
                (a_this);
        g_return_if_fail (tree_editor != NULL);
        mlview_tree_editor2_search_interactive (tree_editor) ;
}

/**
 *Updates the contextual menu
 *@param a_this the current instance of #MlViewTreeView
 *@param a_menu_ptr out paramater the contextual menu.
 */
void
mlview_tree_view_update_contextual_menu (MlViewTreeView * a_this, 
                                         GtkMenu ** a_menu_ptr) 
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (a_menu_ptr != NULL);
        g_return_if_fail (*a_menu_ptr != NULL);
        g_return_if_fail (GTK_IS_MENU (*a_menu_ptr));

        if (PRIVATE (a_this)->tree_editor) {
                
        }
}
