/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by 
 *the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that 
 *it will be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. If not, write to the 
 *Free Software Foundation, Inc., 
 *59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 Dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

#include <libxml/hash.h>

#include "mlview-attributes-list.h"
#include "mlview-name-value-pair.h"
#include "mlview-attribute-picker.h"
#include "mlview-utils.h"


/**
 *@file
 *The definition of the class #MlViewAttributesList.
 */
enum {
        EDIT_STATE_CHANGED,
        ATTRIBUTE_CHANGED,
        NUMBER_OF_SIGNALS
};

struct _MlViewAttributesListPrivate {
        /**
         *visual list of attributes
         */
        GtkCList *attributes_view;

        /**
         *The name edition entry of the widget
         */
        GtkEntry *name_edit_entry;

        /**
         *The value edition entry of the widget
         */
        GtkEntry *value_edit_entry;

        /**
         *The "add attribute" button of the widget
         */
        GtkButton *add_attribute_button;

        /**
         *The "remove attribute" button of the widget
         */
        GtkButton *remove_attribute_button;

        /**
         *The title of the attributes names column
         */
        gchar *names_title;

        /**
         *The title of the attributes values column
         */
        gchar *values_title;

        /**
         *the row that is currently selected bu the user.
         */
        gint current_selected_row;

        /**
	 *says if the current instance is editable or not. 
	 *ONLY use the setter to change it !!!
	 */
        gboolean editable;

        /**
	 *The id returned by gtk_signal_connect() when
	 *we connect to the "changed" signal of 
	 *MlViewAttributesListPrivate#name_edit_entry .
	 *This id is used to disconnect the handler from that signal.
	 */
        guint name_edit_entry_changed_handler_id;

        /**
	 *The id returned by gtk_signal_connect() when
	 *we connect to the "changed" signal of 
	 *MlViewAttributesListPrivate#value_edit_entry .
	 *This id is used to disconnect the handler from that signal.
	 */
        guint value_edit_entry_changed_handler_id;

        /**
	 *The current xmlNode that holds the list 
	 *of attributes being edited.
	 */
        xmlNodePtr current_xml_node;

        /**
	 *The current application context.
	 */
        MlViewAppContext *app_context;

        /**
	 *The attribute picker used to 
	 *populate the list.
	 */
        MlViewAttributePicker *attribute_picker;
};


#define PRIVATE(list) ((list)->priv)

/*compulsory functions to comply with the Gtk type system framework*/
static void
mlview_attributes_list_class_init (MlViewAttributesListClass * a_klass);

static void
mlview_attributes_list_init (MlViewAttributesList * a_attributes);

static void
mlview_attributes_list_destroy (GtkObject * a_object);

static GtkVBoxClass *parent_class = NULL;

/*signals array*/
static guint mlview_attributes_list_signals[NUMBER_OF_SIGNALS] =
        { 0 };

/*private helper functions*/
static xmlAttr *
mlview_attributes_list_add_attribute_to_node_interactive (MlViewAttributesList * a_list, 
                                                          xmlNode * a_node);

/*signal callbacks*/

static void
row_selected_cb (GtkCList * a_clist,
                  gint a_row,
                  gint a_column,
                  GdkEventButton * a_event,
                  MlViewAttributesList * a_attributes);

/*called when a name edit entry changed*/
static void
name_edit_entry_changed_cb (GtkEntry * a_entry,
                            gpointer * a_attributes);

/*called when a value edit entry changed*/
static void
value_edit_entry_changed_cb (GtkEntry * a_entry,
                             MlViewAttributesList *
                              a_attributes);

/*called when a value edit entry changed*/
static void
add_attribute_button_clicked_cb (GtkButton * a_add_attribute_button,
                                 MlViewAttributesList *
                                 a_attributes_list);

static void
remove_attribute_button_clicked_cb (GtkButton * a_remove_attribute_button,
                                    MlViewAttributesList *a_attributes_list);

/*default signal handlers*/
static void
mlview_attributes_list_edit_state_changed_default_handler (MlViewAttributesList * a_attributes, 
                                                           gpointer data);

static void
mlview_attributes_list_attribute_changed_default_handler (MlViewAttributesList * a_attributes, 
                                                          gpointer data);


/*
 *Mandatory private functions to comply with the gtk object
 *framework. go to http://www.gtk.org/tutorial/sec-theanatomyofawidget.html
 *to learn about what the object oriented widget framework looks like.
 */


/**
 *class initialyzer. 
 *@param a_klass the vtable.
 */
static void
mlview_attributes_list_class_init (MlViewAttributesListClass *a_klass)
{
        GtkObjectClass *object_class;

        g_return_if_fail (a_klass != NULL);

        parent_class = gtk_type_class (GTK_TYPE_VBOX);

        object_class = (GtkObjectClass *) a_klass;

        /*overload the destroy method of GtkObjectClass */
        object_class->destroy = mlview_attributes_list_destroy;

        /*signal definitions */
        mlview_attributes_list_signals[EDIT_STATE_CHANGED] =
                g_signal_new ("edit-state-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewAttributesListClass,
                               edit_state_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        mlview_attributes_list_signals[ATTRIBUTE_CHANGED] =
                g_signal_new ("attribute-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewAttributesListClass,
                               attribute_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        /*gtk_object_class_add_signals
           (object_class, 
           mlview_attributes_list_signals,
           NUMBER_OF_SIGNALS) ; */

        a_klass->edit_state_changed =
                mlview_attributes_list_edit_state_changed_default_handler;

        a_klass->attribute_changed =
                mlview_attributes_list_attribute_changed_default_handler;
}

/**
 *The allocator of the instance MlViewAttributesList.
 *@param a_attributes the current instance of MlViewAttributesList to
 *initialyzes
 */
static void
mlview_attributes_list_init (MlViewAttributesList * a_attributes)
{
        const gchar *titles[2] = { "names", "values" };
        GtkWidget *nv_edition_table = NULL,
                *scrolled_window = NULL;

        g_return_if_fail (a_attributes != NULL);

        PRIVATE (a_attributes) =
                g_malloc0 (sizeof (MlViewAttributesListPrivate));

        PRIVATE (a_attributes)->current_selected_row = -1;
        gtk_box_set_spacing (GTK_BOX (a_attributes), 0);

        /*initialyze the attributes viewer */
        PRIVATE (a_attributes)->attributes_view =
                GTK_CLIST (gtk_clist_new_with_titles
                           (2, (gchar **) titles));

        gtk_signal_connect (GTK_OBJECT
                            (PRIVATE (a_attributes)->
                             attributes_view), "select_row",
                            GTK_SIGNAL_FUNC (row_selected_cb),
                            a_attributes);
        scrolled_window = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add
                (GTK_CONTAINER (scrolled_window),
                 GTK_WIDGET (PRIVATE (a_attributes)->
                             attributes_view));

        gtk_box_pack_start (GTK_BOX (a_attributes),
                            scrolled_window, TRUE, TRUE, 0);

        gtk_widget_show (GTK_WIDGET
                         (PRIVATE (a_attributes)->
                          attributes_view));

        /*init the value/name edition entries */
        PRIVATE (a_attributes)->name_edit_entry =
                GTK_ENTRY (gtk_entry_new ());
        gtk_entry_set_editable (PRIVATE (a_attributes)->
                                name_edit_entry,
                                PRIVATE (a_attributes)->
                                editable);

        PRIVATE (a_attributes)->
                name_edit_entry_changed_handler_id =
                gtk_signal_connect (GTK_OBJECT
                                    (PRIVATE (a_attributes)->
                                     name_edit_entry), "changed",
                                    GTK_SIGNAL_FUNC
                                    (name_edit_entry_changed_cb),
                                    a_attributes);

        PRIVATE (a_attributes)->value_edit_entry =
                GTK_ENTRY (gtk_entry_new ());
        gtk_entry_set_editable (PRIVATE (a_attributes)->
                                value_edit_entry,
                                PRIVATE (a_attributes)->
                                editable);

        PRIVATE (a_attributes)->
                value_edit_entry_changed_handler_id =
                gtk_signal_connect (GTK_OBJECT
                                    (PRIVATE (a_attributes)->
                                     value_edit_entry),
                                    "changed",
                                    GTK_SIGNAL_FUNC
                                    (value_edit_entry_changed_cb),
                                    a_attributes);

        /*add/remove attribute buttons */
        PRIVATE (a_attributes)->add_attribute_button =
                GTK_BUTTON (gtk_button_new_with_label
                            (_("Add an attribute")));

        gtk_signal_connect
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->add_attribute_button),
                 "clicked",
                 GTK_SIGNAL_FUNC
                 (add_attribute_button_clicked_cb),
                 a_attributes);

        PRIVATE (a_attributes)->remove_attribute_button =
                GTK_BUTTON
                (gtk_button_new_with_label
                 (_("Remove attributes")));

        gtk_signal_connect
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->
                  remove_attribute_button), "clicked",
                 GTK_SIGNAL_FUNC
                 (remove_attribute_button_clicked_cb),
                 a_attributes);

        nv_edition_table = gtk_table_new (2, 2, TRUE);

        gtk_table_attach_defaults
                (GTK_TABLE (nv_edition_table),
                 GTK_WIDGET (PRIVATE (a_attributes)->
                             name_edit_entry), 0, 1, 0, 1);

        gtk_widget_show (GTK_WIDGET
                         (PRIVATE (a_attributes)->
                          name_edit_entry));

        gtk_table_attach_defaults
                (GTK_TABLE (nv_edition_table),
                 GTK_WIDGET (PRIVATE (a_attributes)->
                             value_edit_entry), 1, 2, 0, 1);

        gtk_widget_show
                (GTK_WIDGET
                 (PRIVATE (a_attributes)->value_edit_entry));

        gtk_table_attach_defaults
                (GTK_TABLE (nv_edition_table),
                 GTK_WIDGET (PRIVATE (a_attributes)->
                             add_attribute_button), 0, 1, 1, 2);

        gtk_widget_show
                (GTK_WIDGET
                 (PRIVATE (a_attributes)->add_attribute_button));

        gtk_table_attach_defaults
                (GTK_TABLE (nv_edition_table),
                 GTK_WIDGET (PRIVATE (a_attributes)->
                             remove_attribute_button), 1, 2, 1,
                 2);

        gtk_widget_show
                (GTK_WIDGET
                 (PRIVATE (a_attributes)->
                  remove_attribute_button));

        gtk_box_pack_start (GTK_BOX (a_attributes),
                            GTK_WIDGET (nv_edition_table),
                            FALSE, TRUE, 0);

        gtk_widget_show (GTK_WIDGET (nv_edition_table));

        mlview_attributes_list_set_editable (a_attributes, TRUE);

        PRIVATE (a_attributes)->names_title = NULL;

        PRIVATE (a_attributes)->values_title = NULL;
}

/**
 *The destroy method.
 *overloads the destroy method of GtkObject. 
 *See the method mlview_attributes_list_class_init.
 *frees all the dynamically allocated 
 *members of the current instance and 
 *calls the destroy method of the parent class. 
 *
 *@param a_object the instance of MlViewAttributesLists to destroy.
 */
static void
mlview_attributes_list_destroy (GtkObject * a_object)
{
        MlViewAttributesList *attributes;

        g_return_if_fail (a_object != NULL);
        g_return_if_fail (MLVIEW_IS_ATTRIBUTES_LIST (a_object));


        attributes = MLVIEW_ATTRIBUTES_LIST (a_object);

        if (PRIVATE (attributes) == NULL)
                return;

        /*destroy non ref'd allocated widgets and other objects */
        if (PRIVATE (attributes)->names_title) {
                g_free (PRIVATE (attributes)->names_title);
                PRIVATE (attributes)->names_title = NULL;
        }

        if (PRIVATE (attributes)->values_title) {
                g_free (PRIVATE (attributes)->values_title);
                PRIVATE (attributes)->values_title = NULL;
        }

        if (PRIVATE (attributes)->attribute_picker) {
                gtk_widget_destroy
                        (GTK_WIDGET
                         (PRIVATE (attributes)->
                          attribute_picker));
                PRIVATE (attributes)->attribute_picker = NULL;
        }

        if (PRIVATE (attributes)) {
                g_free (PRIVATE (attributes));
                PRIVATE (attributes) = NULL;
        }

        /*call the destroy method of the object class */
        if (GTK_OBJECT_CLASS (parent_class)->destroy)
                (*GTK_OBJECT_CLASS (parent_class)->
                 destroy) (a_object);

}

/*===================================
 *private helper functions
 *==================================*/

/**
 *Interactively adds an attribute to the list.
 *@param a_list the current instance of MlViewAttributesList.
 *@param a_node the xmlNode to add the attributes to.
 */
static xmlAttr
        *
        mlview_attributes_list_add_attribute_to_node_interactive
        (MlViewAttributesList * a_list, xmlNode * a_node) {
        gint button;
        gchar *name_str = NULL,
                *value_str = NULL;
        gboolean loop = TRUE;
        xmlAttr *result = NULL;

        g_return_val_if_fail (a_list != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_ATTRIBUTES_LIST (a_list),
                              NULL);

        if (PRIVATE (a_list)->attribute_picker == NULL) {
                PRIVATE (a_list)->attribute_picker =
                        MLVIEW_ATTRIBUTE_PICKER
                        (mlview_attribute_picker_new
                         ((guchar*)
                          _("Enter attribute name and value"),
                          PRIVATE (a_list)->app_context));
        }

        mlview_attribute_picker_grab_focus_to_name_entry
                (PRIVATE (a_list)->attribute_picker);

        gtk_window_set_modal (GTK_WINDOW
                              (PRIVATE (a_list)->
                               attribute_picker), TRUE);

        name_str = mlview_attribute_picker_get_attribute_name
                (PRIVATE (a_list)->attribute_picker);

        if (!mlview_utils_is_white_string (name_str)) {
                mlview_attribute_picker_select_attribute_name
                        (PRIVATE (a_list)->attribute_picker);
        }

        mlview_attribute_picker_set_current_xml_node
                (PRIVATE (a_list)->attribute_picker, a_node);

        mlview_attribute_picker_build_attribute_name_choice_list
                (PRIVATE (a_list)->attribute_picker, a_node);

        while (loop) {
                xmlAttributeType attr_type;

                gtk_widget_realize
                        (GTK_WIDGET
                         (PRIVATE (a_list)->attribute_picker));

                mlview_app_context_set_window_icon
                        (PRIVATE (a_list)->app_context,
                         GTK_WINDOW (PRIVATE (a_list)->
                                     attribute_picker));

                button = gtk_dialog_run
                        (GTK_DIALOG
                         (PRIVATE (a_list)->attribute_picker));

                switch (button) {
                case GTK_RESPONSE_ACCEPT:
                        name_str =
                                mlview_attribute_picker_get_attribute_name
                                (PRIVATE (a_list)->
                                 attribute_picker);
                        value_str =
                                mlview_attribute_picker_get_attribute_value
                                (PRIVATE (a_list)->
                                 attribute_picker);
                        attr_type =
                                mlview_attribute_picker_get_attribute_type
                                (PRIVATE (a_list)->
                                 attribute_picker);

                        if (!mlview_utils_is_white_string
                            (value_str)
                            &&
                            !mlview_utils_is_white_string
                            (name_str)) {

                                guchar *local_name = NULL;
                                xmlNs *ns = NULL;

                                mlview_utils_parse_full_name
                                        (a_node, name_str, &ns,
                                         &local_name);
                                result = xmlNewNsProp (a_node,
                                                       ns,
                                                       local_name,
                                                       value_str);

                                /*FIXME: do this only when validation is on */
                                if (attr_type == XML_ATTRIBUTE_ID
                                    && a_node->doc
                                    && a_node->doc->ids) {

                                        xmlID *id = NULL;

                                        result->atype =
                                                XML_ATTRIBUTE_ID;
                                        id = xmlMalloc (sizeof
                                                        (xmlID));
                                        g_assert (id != NULL);
                                        id->value =
                                                g_strdup
                                                (value_str);
                                        id->attr = result;
                                        xmlHashAddEntry
                                                (result->doc->
                                                 ids,
                                                 (const xmlChar
                                                  *) value_str,
                                                 id);
                                }

                                if (local_name) {
                                        g_free (local_name);
                                        local_name = NULL;
                                }
                                loop = FALSE;
                        }
                        break;
                case GTK_RESPONSE_REJECT:
                        loop = FALSE;
                        break;
                case GTK_RESPONSE_CLOSE:
                        loop = FALSE;
                        break;
                default:
                        break;
                }
        }

        gtk_widget_hide (GTK_WIDGET
                         (PRIVATE (a_list)->attribute_picker));

        return result;
}

/*signal handlers and callbacks*/


/**
 *default callback of the signal "edit_state_changed":
 *sets the two GtkEntry MlViewAttributesListPrivate#name_edit_entry 
 *and MlViewAttributesListPrivate#value_edit_entry to an editable state 
 *@param a_attributes the current instance of #MlViewAttributesList .
 *@param a_data custom used data. Not used for the moment.
 */
static void
 mlview_attributes_list_edit_state_changed_default_handler
        (MlViewAttributesList * a_attributes, gpointer a_data) {
        g_return_if_fail (a_attributes != NULL);
        gtk_entry_set_editable (PRIVATE (a_attributes)->
                                name_edit_entry,
                                PRIVATE (a_attributes)->
                                editable);
        gtk_entry_set_editable (PRIVATE (a_attributes)->
                                value_edit_entry,
                                PRIVATE (a_attributes)->
                                editable);
}


static void
 mlview_attributes_list_attribute_changed_default_handler
        (MlViewAttributesList * a_attributes, gpointer data) {

}


/**
 *callback of the signal "row_selected" of GtkCList.
 *called when a propery is selected.
 *saves the row num of the attribute selected. 
 *
 *@param a_clist the GtkCList instance that emited the signal.
 *@param a_column the column of the GtkCList that was selected.
 *@param a_row the row of the GtkCList that was selected
 *@param a_event the event that caused the selection.
 *@a_attributes the current instance of 
 *#MlViewAttributesList on which the selection is performed 
 */
static void
row_selected_cb (GtkCList * a_clist,
                 gint a_row,
                 gint a_column,
                 GdkEventButton * a_event,
                 MlViewAttributesList * a_attributes)
{
        gchar *prop_name = NULL,
                *prop_value = NULL;
        xmlAttr *xml_attr;

        g_return_if_fail (a_attributes != NULL);
        g_return_if_fail (PRIVATE (a_attributes)->
                          current_xml_node != NULL);
        g_return_if_fail (PRIVATE (a_attributes)->
                          name_edit_entry != NULL);
        g_return_if_fail (PRIVATE (a_attributes)->
                          value_edit_entry != NULL);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->name_edit_entry),
                 PRIVATE (a_attributes)->
                 name_edit_entry_changed_handler_id);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->value_edit_entry),
                 PRIVATE (a_attributes)->
                 value_edit_entry_changed_handler_id);

        gtk_entry_set_text (PRIVATE (a_attributes)->
                            name_edit_entry, "");
        gtk_entry_set_text (PRIVATE (a_attributes)->
                            value_edit_entry, "");

        gtk_clist_get_text (a_clist, a_row,
                            ATTRIBUTE_NAME_OFFSET, &prop_name);

        gtk_clist_get_text (a_clist, a_row,
                            ATTRIBUTE_VALUE_OFFSET, &prop_value);

        mlview_attributes_list_get_attribute (a_attributes,
                                              a_row, &xml_attr);

        gtk_entry_set_text (PRIVATE (a_attributes)->
                            name_edit_entry, prop_name);

        gtk_entry_set_text (PRIVATE (a_attributes)->
                            value_edit_entry, prop_value);

        gtk_signal_handler_unblock
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->name_edit_entry),
                 PRIVATE (a_attributes)->
                 name_edit_entry_changed_handler_id);

        gtk_signal_handler_unblock
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->value_edit_entry),
                 PRIVATE (a_attributes)->
                 value_edit_entry_changed_handler_id);

        PRIVATE (a_attributes)->current_selected_row = a_row;
}


/**
 *callback of the signal "changed" of GtkEntry ;
 *Is called when the name edit entry of 
 *the current instance of #MlViewAttributesList 
 *changes (the user is editing it).
 *Copies the new value of the name edit entry 
 *to the corresponding attribute name in the GtkCList.
 *After doing all this, emits the signals 
 *"attribute-changed" on the current instance of #MlViewAttributesList. 
 *
 *@param a_attributes the current instance of #MlViewAttributesList
 *@param a_entry the instance of GtkEntry that emited the signal
 */
static void
name_edit_entry_changed_cb (GtkEntry * a_entry,
                            gpointer * a_attributes)
{
        MlViewAttributesList *attributes = NULL;
        xmlAttrPtr xml_attr = NULL;
        gchar *name = NULL;

        g_return_if_fail (a_attributes != NULL);
        attributes = MLVIEW_ATTRIBUTES_LIST (a_attributes);

        g_return_if_fail (PRIVATE (attributes)->
                          current_selected_row >= 0);
        g_return_if_fail (PRIVATE (attributes)->
                          current_xml_node != NULL);

        mlview_attributes_list_get_attribute
                (attributes,
                 PRIVATE (attributes)->current_selected_row,
                 &xml_attr);

        if (xml_attr != NULL) {
                guchar *local_name = NULL;
                xmlNs *ns = NULL;

                name = (guchar *)
                        gtk_entry_get_text (GTK_ENTRY (a_entry));
                mlview_utils_parse_full_name
                        (PRIVATE (attributes)->current_xml_node,
                         name, &ns, &local_name);

                xmlSetNs ((xmlNodePtr) xml_attr, ns);

                xmlNodeSetName ((xmlNodePtr) xml_attr,
                                local_name);

                if (ns && ns->prefix)
                        name = g_strconcat (ns->prefix, ":",
                                            local_name, NULL);
                else
                        name = g_strdup (local_name);

                gtk_clist_set_text
                        (GTK_CLIST
                         (PRIVATE (attributes)->attributes_view),
                         PRIVATE (attributes)->
                         current_selected_row,
                         ATTRIBUTE_NAME_OFFSET, name);

                if (local_name) {
                        g_free (local_name);
                }

                if (name) {
                        g_free (name);
                }
        }

        gtk_signal_emit (GTK_OBJECT (attributes),
                         mlview_attributes_list_signals
                         [ATTRIBUTE_CHANGED]);
}


/**
 *callback of the signal "changed" of GtkEntry.
 *Is called when the value edit entry 
 *of the current instance of #MlViewAttributesList 
 *changes (the user is editing it).
 *Copies the new value of the value edit 
 *entry to the corresponding attribute name in the GtkCList.
 *Emits the signals "attribute-changed" on the current 
 *instance of #MlViewAttributesList 
 *
 *@param a_entry the instance of GtkEntry that emited the signal.
 *@param a_attributes the current instance of #MlViewAttributesList.
 */
static void
value_edit_entry_changed_cb (GtkEntry * a_entry,
                             MlViewAttributesList * a_attributes)
{
        guchar *value = NULL;
        xmlAttrPtr xml_attr = NULL;

        g_return_if_fail (a_attributes != NULL
                          &&
                          MLVIEW_IS_ATTRIBUTES_LIST
                          (a_attributes));

        g_return_if_fail (PRIVATE (a_attributes)->
                          current_selected_row >= 0);

        value = (guchar *)
                gtk_entry_get_text (GTK_ENTRY (a_entry));
        mlview_attributes_list_get_attribute (a_attributes,
                                              PRIVATE
                                              (a_attributes)->
                                              current_selected_row,
                                              &xml_attr);

        if (xml_attr != NULL) {
                xmlSetProp (xml_attr->parent,
                            xml_attr->name, g_strdup (value));

                gtk_clist_set_text
                        (GTK_CLIST
                         (PRIVATE (a_attributes)->
                          attributes_view),
                         PRIVATE (a_attributes)->
                         current_selected_row,
                         ATTRIBUTE_VALUE_OFFSET, value);
        }

        gtk_signal_emit (GTK_OBJECT (a_attributes),
                         mlview_attributes_list_signals
                         [ATTRIBUTE_CHANGED]);
}

/**
 *@param a_add_attribute_button the instance 
 *of GtkButton that emitted the "clicked" signal.
 *@param a_attributes_list the current instance 
 *of #MlViewAttributesList.
 */
static void
add_attribute_button_clicked_cb (GtkButton *
                                 a_add_attribute_button,
                                 MlViewAttributesList *
                                 a_attributes_list)
{
        g_return_if_fail (a_add_attribute_button != NULL);
        g_return_if_fail (a_attributes_list != NULL);

        mlview_attributes_list_create_attribute
                (a_attributes_list);
}


/**
 *@param a_remove_attribute_button the instance 
 *of GtkButtonClass that emitted the "clicked" signal.
 *@param a_attributes_list 
 *the current instance of #MlViewAttributesList.
 *
 */
static void
remove_attribute_button_clicked_cb (GtkButton * a_remove_attribute_button,
                                    MlViewAttributesList *a_attributes_list)
{
        g_return_if_fail (a_remove_attribute_button != NULL);
        g_return_if_fail (a_attributes_list != NULL);
        g_return_if_fail 
                (PRIVATE (a_attributes_list)->current_selected_row > -1);

        mlview_attributes_list_remove_attribute
                (a_attributes_list,
                 PRIVATE (a_attributes_list)->current_selected_row);
}

/*===================================================================
 *public methods
 *==================================================================
 */

/**
 *type builder. 
 *@return the id of the #MlViewAttributesList type.
 */
guint
mlview_attributes_list_get_type (void)
{
        static guint mlview_attributes_list_type = 0;

        if (!mlview_attributes_list_type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewAttributesListClass),
                        NULL,   /* base_init */
                        NULL,   /* base_finalize */
                        (GClassInitFunc)
                                mlview_attributes_list_class_init,
                        NULL,   /* class_finalize */
                        NULL,   /* class_data */
                        sizeof (MlViewAttributesList),
                        0,
                        (GInstanceInitFunc)
                        mlview_attributes_list_init
                };

                mlview_attributes_list_type =
                        g_type_register_static (GTK_TYPE_VBOX,
                                                "MlViewAttributesList",
                                                &type_info, 0);

        }

        return mlview_attributes_list_type;
}


/**
 *Creates a new #MlViewAttributesList.
 *
 *@param a_names_titles the title of the names column
 *@param a_value_title the title of the values column
 *@return the newly created instance of #MlViewAttributesList.
 */
GtkWidget *
mlview_attributes_list_new (gchar * a_names_title,
                            gchar * a_values_title,
                            MlViewAppContext * a_app_context)
{
        MlViewAttributesList *attributes;

        attributes = gtk_type_new (MLVIEW_TYPE_ATTRIBUTES_LIST);

        mlview_attributes_list_set_app_context (attributes,
                                                a_app_context);

        if (a_names_title) {
                gtk_clist_set_column_title
                        (GTK_CLIST
                         (PRIVATE (attributes)->attributes_view),
                         ATTRIBUTE_NAME_OFFSET, a_names_title);
        }

        if (a_values_title) {
                gtk_clist_set_column_title
                        (GTK_CLIST
                         (PRIVATE (attributes)->attributes_view),
                         ATTRIBUTE_VALUE_OFFSET, a_values_title);
        }

        return GTK_WIDGET (attributes);
}


/**
 *Sets a new application context to the current instance of 
 *#MlViewAttributeLists. 
 *
 *@param a_attributes the current instance of attribute list.
 *@param a_app_context the new app context to set.
 */
void
mlview_attributes_list_set_app_context (MlViewAttributesList *
                                        a_attributes,
                                        MlViewAppContext *
                                        a_app_context)
{
        g_return_if_fail (a_attributes);
        g_return_if_fail (MLVIEW_IS_ATTRIBUTES_LIST
                          (a_attributes));
        g_return_if_fail (PRIVATE (a_attributes) != NULL);

        PRIVATE (a_attributes)->app_context = a_app_context;
}


/**
 *Sets the editable boolean to a_editable 
 *and emits the signal "edit-state-changed". 
 *(if and only if a_editable != the previous editable state)
 *After setting the editable boolean, 
 *emits the signal "edit-state-changed" 
 *on the current instance of #MlViewAttributesList. 
 *
 *@param a_attributes the current intance of MlViewAttributesList.
 *@param a_editable the new editable state 
 *(TRUE=> the attributes list become editable FALSE=> it becomes read/only)
 */
void
mlview_attributes_list_set_editable (MlViewAttributesList *
                                     a_attributes,
                                     gboolean a_editable)
{
        g_return_if_fail (a_attributes != NULL);
        g_return_if_fail (MLVIEW_IS_ATTRIBUTES_LIST
                          (a_attributes));
        g_return_if_fail (PRIVATE (a_attributes) != NULL);

        PRIVATE (a_attributes)->editable = a_editable;
        gtk_signal_emit (GTK_OBJECT (a_attributes),
                         mlview_attributes_list_signals
                         [EDIT_STATE_CHANGED]);
}

/**
 *Sets the titles of the names and/or values of the attributes list. 
 *
 *@param a_attributes the current instance of #MlViewAttributesList
 *@param a_names_title the new names 
 *column title. If NULL or equal to "", 
 *the names column title is not changed.
 *@param a_values_titles the new values title. 
 *If NULL or equal to "", the values column title is not changed.
 */
void
mlview_attributes_list_set_titles (MlViewAttributesList *
                                   a_attributes,
                                   gchar * a_names_title,
                                   gchar * a_values_title)
{

        g_return_if_fail (a_attributes != NULL);
        g_return_if_fail (a_names_title != NULL);
        g_return_if_fail (a_values_title != NULL);

        if (strcmp (a_names_title, "")) {
                if (PRIVATE (a_attributes)->names_title)
                        g_free (PRIVATE (a_attributes)->
                                names_title);
                PRIVATE (a_attributes)->names_title =
                        g_strdup (a_names_title);
        }

        if (strcmp (a_values_title, "")) {
                if (PRIVATE (a_attributes)->values_title)
                        g_free (PRIVATE (a_attributes)->
                                values_title);
                PRIVATE (a_attributes)->values_title =
                        g_strdup (a_values_title);
        }
}

/**
 *sets a attribute name and value. 
 *
 *@param a_offset the attribute number (starting from 0)
 *@param a_attributes the current instance of #MlViewAttributesList.
 *@param a_value the new attribute value . 
 *If NULL returns, a debug message is printed.
 *@param a_name the new attribute name. 
 *If NULL, returns, a debug message is printed.
 */
void
mlview_attributes_list_set_attribute (MlViewAttributesList *
                                      a_attributes,
                                      guint a_offset,
                                      const xmlAttrPtr
                                      a_xml_attr)
{
        g_return_if_fail (a_attributes != NULL
                          &&
                          MLVIEW_IS_ATTRIBUTES_LIST
                          (a_attributes));

        g_return_if_fail (PRIVATE (a_attributes)->
                          attributes_view != NULL);
        g_return_if_fail (a_xml_attr != NULL);
        g_return_if_fail (PRIVATE (a_attributes)->
                          name_edit_entry != NULL);
        g_return_if_fail (PRIVATE (a_attributes)->
                          value_edit_entry != NULL);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->name_edit_entry),
                 PRIVATE (a_attributes)->
                 name_edit_entry_changed_handler_id);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->value_edit_entry),
                 PRIVATE (a_attributes)->
                 value_edit_entry_changed_handler_id);

        gtk_clist_set_text (PRIVATE (a_attributes)->
                            attributes_view, a_offset,
                            ATTRIBUTE_NAME_OFFSET,
                            a_xml_attr->name);
        gtk_clist_set_text (PRIVATE (a_attributes)->
                            attributes_view, a_offset,
                            ATTRIBUTE_VALUE_OFFSET,
                            xmlNodeListGetString (a_xml_attr->
                                                  doc,
                                                  a_xml_attr->
                                                  children, 1));

        gtk_clist_set_row_data (PRIVATE (a_attributes)->
                                attributes_view, a_offset,
                                a_xml_attr);

        gtk_entry_set_text (PRIVATE (a_attributes)->
                            name_edit_entry, "");
        gtk_entry_set_text (PRIVATE (a_attributes)->
                            value_edit_entry, "");

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->name_edit_entry),
                 PRIVATE (a_attributes)->
                 name_edit_entry_changed_handler_id);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->value_edit_entry),
                 PRIVATE (a_attributes)->
                 value_edit_entry_changed_handler_id);
}

/**
 *Edits the xml attributes hold by a given xml node.
 *
 *@param a_attributes_list the current instance of #MlViewAttributesList
 *@param a_xml_node the xml node that holds the attributes list to edit.
 */
void
mlview_attributes_list_edit_xml_attributes (MlViewAttributesList
                                            * a_attributes_list,
                                            const xmlNodePtr
                                            a_xml_node)
{
        xmlAttrPtr xml_attr = NULL;

        g_return_if_fail (a_attributes_list != NULL);
        g_return_if_fail (MLVIEW_IS_ATTRIBUTES_LIST
                          (a_attributes_list));
        g_return_if_fail (PRIVATE (a_attributes_list) != NULL);
        g_return_if_fail (a_xml_node);

        /*
         *even if there are no attributes, do not 
         *forget to store the pointer to the xml_node so that
         *it will be possible to add new attributes to that xml_node.
         */
        PRIVATE (a_attributes_list)->current_xml_node =
                a_xml_node;

        if (a_xml_node->properties == NULL)
                return;

        for (xml_attr = a_xml_node->properties;
             xml_attr != NULL; xml_attr = xml_attr->next) {

                if (xml_attr->name) {
                        mlview_attributes_list_add_attribute_to_view
                                (a_attributes_list, xml_attr);
                }
        }

        gtk_widget_show_all (GTK_WIDGET (a_attributes_list));
}

/**
 *adds a new attribute to the attributes list .
 *
 *@param a_attributes the current instance of #MlViewAttributesList.
 *@param a_value the value of the new attribute.
 *@param a_name the name of the new attribute.
 */
gint mlview_attributes_list_add_attribute_to_view
        (MlViewAttributesList * a_attributes,
         const xmlAttrPtr a_xml_attr) {
        gchar *attributes[2],
        *name = NULL,
                *value = NULL;
        gint row_offset = 0;

        g_return_val_if_fail (a_attributes != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_attributes)->
                              attributes_view != NULL, -1);
        g_return_val_if_fail (a_xml_attr != NULL, -1);

        if (a_xml_attr->ns != NULL && a_xml_attr->ns->prefix)
                name = g_strconcat (a_xml_attr->ns->prefix,
                                    ":", a_xml_attr->name, NULL);
        else
                name = g_strdup (a_xml_attr->name);

        value = xmlNodeListGetString (a_xml_attr->doc,
                                      a_xml_attr->children, 1);

        attributes[ATTRIBUTE_NAME_OFFSET] = name;
        attributes[ATTRIBUTE_VALUE_OFFSET] = value;

        row_offset =
                gtk_clist_append (PRIVATE (a_attributes)->
                                  attributes_view, attributes);
        gtk_clist_set_row_data (PRIVATE (a_attributes)->
                                attributes_view, row_offset,
                                a_xml_attr);
        if (name) {
                g_free (name);
                name = NULL;
        }

        if (value) {
                g_free (value);
                value = NULL;
        }

        return row_offset;
}


/**
 *Gets the attribute that is at a given offset 
 *in the current instance of #MlViewAttributesList.
 *
 *@param a_attributes the current instance of 
 *#MlViewAttributesList.
 *@param a_offset the offset of the attribute.
 *@param a_xml_attr_ptr out parameter. A pointer 
 *to the xmlAttrPtr (the libxml2 data structure that holds an attribute)
 */
void
mlview_attributes_list_get_attribute (MlViewAttributesList *
                                      a_attributes,
                                      const guint a_offset,
                                      xmlAttrPtr *
                                      a_xml_attr_ptr)
{
        g_return_if_fail (a_attributes != NULL);
        *a_xml_attr_ptr =
                gtk_clist_get_row_data
                (PRIVATE (a_attributes)->attributes_view,
                 a_offset);
}


/**
 *clears all the attributes edited by the current 
 *instance of #MlViewAttributesList.
 *
 *@param a_attributes the current instance of MlViewAttributesList.
 */
void
mlview_attributes_list_clear (MlViewAttributesList *
                              a_attributes)
{
        g_return_if_fail (a_attributes != NULL
                          &&
                          MLVIEW_IS_ATTRIBUTES_LIST
                          (a_attributes));
        g_return_if_fail (PRIVATE (a_attributes)->
                          attributes_view);
        g_return_if_fail (PRIVATE (a_attributes)->
                          name_edit_entry);
        g_return_if_fail (PRIVATE (a_attributes)->
                          value_edit_entry);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->name_edit_entry),
                 PRIVATE (a_attributes)->
                 name_edit_entry_changed_handler_id);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->value_edit_entry),
                 PRIVATE (a_attributes)->
                 value_edit_entry_changed_handler_id);

        gtk_clist_clear (PRIVATE (a_attributes)->
                         attributes_view);
        gtk_entry_set_text (PRIVATE (a_attributes)->
                            name_edit_entry, "");
        gtk_entry_set_text (PRIVATE (a_attributes)->
                            value_edit_entry, "");

        PRIVATE (a_attributes)->current_xml_node = NULL;
        PRIVATE (a_attributes)->current_selected_row = -1;

        gtk_signal_handler_unblock
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->name_edit_entry),
                 PRIVATE (a_attributes)->
                 name_edit_entry_changed_handler_id);

        gtk_signal_handler_unblock
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->value_edit_entry),
                 PRIVATE (a_attributes)->
                 value_edit_entry_changed_handler_id);
}

/**
 *Interactively creates a new attribute and inserts it at the end of the attribute list.
 *The name and value of the attribute are the one entered by the user.
 *Emits a "attribute-changed" signal on #MlViewAttributesList. 
 *
 *@param a_attributes the current instance of #MlViewAttributesList
 *
 */
gint
mlview_attributes_list_create_attribute (MlViewAttributesList *
                                         a_attributes)
{
        xmlAttrPtr xml_attr = NULL;
        gint result;

        g_return_val_if_fail (a_attributes != NULL,
                              MLVIEW_ATTRIBUTES_LIST_NULL_PARAMETER);

        g_return_val_if_fail (PRIVATE (a_attributes)->
                              current_xml_node != NULL,
                              MLVIEW_ATTRIBUTES_LIST_NULL_XML_NODE);

        if (PRIVATE (a_attributes)->editable == FALSE)
                return MLVIEW_ATTRIBUTES_LIST_READ_ONLY;

        xml_attr =
                mlview_attributes_list_add_attribute_to_node_interactive
                (a_attributes,
                 PRIVATE (a_attributes)->current_xml_node);

        if (xml_attr == NULL)
                return -1;

        result = mlview_attributes_list_add_attribute_to_view
                (a_attributes, xml_attr);
        gtk_signal_emit (GTK_OBJECT (a_attributes),
                         mlview_attributes_list_signals
                         [ATTRIBUTE_CHANGED]);
        return result;
}


/**
 *Removes the attribute located at 
 *offset a_attr_offset in the attribute list.
 *If the removal succeeds, emits a 
 *"attribute-changed" signal on #MlViewAttributesList. 
 *
 *@param a_attributes the current instance of #MlViewAttributesList.
 *@param a_attr_offset the offset to consider.
 */
void
mlview_attributes_list_remove_attribute (MlViewAttributesList *a_attributes,
                                         const guint a_attr_offset)
{
        xmlAttrPtr xml_attr = NULL;
        gint result;

        g_return_if_fail (a_attributes != NULL);
        g_return_if_fail (PRIVATE (a_attributes)->
                          current_xml_node != NULL);
        g_return_if_fail (PRIVATE (a_attributes)->
                          attributes_view);

        xml_attr =
                (xmlAttrPtr) gtk_clist_get_row_data
                (PRIVATE (a_attributes)->attributes_view,
                 a_attr_offset);

        g_return_if_fail (xml_attr != NULL);

        result = xmlRemoveProp (xml_attr);

        g_return_if_fail (result == 0);

        gtk_clist_remove (PRIVATE (a_attributes)->
                          attributes_view,
                          PRIVATE (a_attributes)->
                          current_selected_row);

        gtk_signal_handler_block
                (GTK_OBJECT (PRIVATE
                             (a_attributes)->name_edit_entry),
                 PRIVATE (a_attributes)->
                 name_edit_entry_changed_handler_id);

        gtk_signal_handler_block
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->value_edit_entry),
                 PRIVATE (a_attributes)->
                 value_edit_entry_changed_handler_id);

        gtk_entry_set_text (PRIVATE (a_attributes)->
                            name_edit_entry, "");
        gtk_entry_set_text (PRIVATE (a_attributes)->
                            value_edit_entry, "");

        gtk_signal_handler_unblock
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->name_edit_entry),
                 PRIVATE (a_attributes)->
                 name_edit_entry_changed_handler_id);

        gtk_signal_handler_unblock
                (GTK_OBJECT
                 (PRIVATE (a_attributes)->value_edit_entry),
                 PRIVATE (a_attributes)->
                 value_edit_entry_changed_handler_id);

        gtk_signal_emit (GTK_OBJECT (a_attributes),
                         mlview_attributes_list_signals
                         [ATTRIBUTE_CHANGED]);
}


/**
 *Gets the underlying GtkCList used by the current 
 *instance of #MlViewAttributesList.
 *@param a_attributes the current instance 
 *of MlViewAttributesList.
 *@return the underlying GtkCList 
 *used by the current instance of #MlViewAttributesList.
 */
GtkCList *
mlview_attributes_list_get_internal_clist (MlViewAttributesList *
                                           a_attributes)
{
        g_return_val_if_fail (a_attributes != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_ATTRIBUTES_LIST
                              (a_attributes), NULL);

        return PRIVATE (a_attributes)->attributes_view;
}
