/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

#include "mlview-settings-builder.h"
#include "mlview-global-settings.h"

typedef struct _XMLDocumentSettingsWidgetStruct
 XMLDocumentSettingsWidgetStruct;

struct _XMLDocumentSettingsWidgetStruct {
        /*the default expansion depth of the xml tree. */
        GtkEntry *tree_exp_depth;
        GtkCheckButton *validation_on;
};

struct _MlViewSettingsBuilderPrivate {
        MlViewAppContext *app_context;
        MlViewSettingsManager *settings_manager;

        /*MlViewXMLDocument settings ... */
        XMLDocumentSettingsWidgetStruct doc_widgets;
        MlViewSettingsEntry *xml_doc_stgs_entry;
        GtkWidget *xml_document_settings_widget;

        /*Settings of Other MlView parts come down here ... */
};


#define PRIVATE(dialog) ((dialog)->private)

static GtkObjectClass *parent_class = NULL;


/*private function of this class*/
static void mlview_settings_builder_init
        (MlViewSettingsBuilder * a_settings_builder);

static void mlview_settings_builder_init_class
        (MlViewSettingsBuilderClass * a_klass);

static void
 mlview_xml_doc_load_settings
        (MlViewSettingsEntry * a_settings_entry,
         xmlDoc * a_xml_doc);

static void
 mlview_xml_doc_settings_changed_handler
        (MlViewSettingsEntry * a_settings, void *a_user_data);

static void
 mlview_xml_doc_get_settings_from_settings_widget_handler
        (MlViewSettingsEntry * a_settings_entry);

static void
 mlview_xml_doc_set_settings_to_settings_widget_handler
        (MlViewSettingsEntry * a_settings_entry);

static void
 mlview_settings_builder_construct
        (MlViewSettingsBuilder * a_settings_builder,
         MlViewAppContext * a_app_context);

/**
 *Gtk framework class initializer. 
 *
 */
static void
mlview_settings_builder_init_class (MlViewSettingsBuilderClass *
                                    a_klass)
{
        GtkObjectClass *object_class =
                GTK_OBJECT_CLASS (a_klass);
        parent_class = gtk_type_class (GTK_TYPE_OBJECT);

        /*overload the destroy method of GtkObject */
        object_class->destroy = mlview_settings_builder_destroy;
}

static void
mlview_settings_builder_construct (MlViewSettingsBuilder *
                                   a_settings_builder,
                                   MlViewAppContext *
                                   a_app_context)
{
        g_return_if_fail (a_settings_builder != NULL);
        g_return_if_fail (MLVIEW_IS_SETTINGS_BUILDER
                          (a_settings_builder));
        g_return_if_fail (PRIVATE (a_settings_builder) != NULL);
        g_return_if_fail (a_app_context != NULL);

        PRIVATE (a_settings_builder)->settings_manager =
                mlview_settings_manager_new (a_app_context);
        PRIVATE (a_settings_builder)->app_context =
                a_app_context;

        g_return_if_fail (PRIVATE (a_settings_builder)->
                          settings_manager != NULL);
}


/**
 *
 */
static void
mlview_settings_builder_init (MlViewSettingsBuilder *
                              a_settings_builder)
{
        g_return_if_fail (a_settings_builder != NULL);
        g_return_if_fail (MLVIEW_IS_SETTINGS_BUILDER
                          (a_settings_builder));

        /*build the xml document related settings widget part */
        PRIVATE (a_settings_builder) =
                g_malloc0 (sizeof
                           (MlViewSettingsBuilderPrivate));
}


/**
 *
 */
static void
 mlview_settings_builder_build_xml_document_settings_widget
        (MlViewSettingsBuilder * a_settings_builder) {
        GtkWidget *frame = NULL,
                *label = NULL,
                *table = NULL,
                *separator = NULL;

        g_return_if_fail (a_settings_builder != NULL);
        g_return_if_fail (MLVIEW_IS_SETTINGS_BUILDER
                          (a_settings_builder));
        g_return_if_fail (PRIVATE (a_settings_builder) != NULL);

        PRIVATE (a_settings_builder)->
                xml_document_settings_widget =
                gtk_vbox_new (FALSE, 0);

        /*default node expansion level setting frame */

        frame = gtk_frame_new (_
                               ("When opening an xml document, use:"));

        table = gtk_table_new (1, 2, TRUE);

        PRIVATE (a_settings_builder)->doc_widgets.tree_exp_depth
                = GTK_ENTRY (gtk_entry_new ());

        label = gtk_label_new (_("Node expansion level:"));

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults
                (GTK_TABLE (table),
                 GTK_WIDGET
                 (PRIVATE
                  (a_settings_builder)->doc_widgets.
                  tree_exp_depth), 1, 2, 0, 1);

        gtk_container_add (GTK_CONTAINER (frame), table);

        gtk_box_pack_start
                (GTK_BOX
                 (PRIVATE
                  (a_settings_builder)->
                  xml_document_settings_widget), frame, FALSE,
                 TRUE, 0);

        separator = gtk_hseparator_new ();
        gtk_box_pack_start
                (GTK_BOX
                 (PRIVATE
                  (a_settings_builder)->
                  xml_document_settings_widget), separator,
                 FALSE, TRUE, 0);

        /*Validation setting frame */
        frame = gtk_frame_new (_("Validation"));

        PRIVATE (a_settings_builder)->doc_widgets.validation_on
                =
                GTK_CHECK_BUTTON (gtk_check_button_new_with_label
                                  (_("Validation on")));

        gtk_container_add
                (GTK_CONTAINER (frame),
                 GTK_WIDGET
                 (PRIVATE
                  (a_settings_builder)->doc_widgets.
                  validation_on));

        gtk_box_pack_start
                (GTK_BOX
                 (PRIVATE
                  (a_settings_builder)->
                  xml_document_settings_widget), frame, FALSE,
                 TRUE, 0);

        gtk_widget_show_all
                (PRIVATE
                 (a_settings_builder)->
                 xml_document_settings_widget);
}

static void
 mlview_settings_builder_destroy_xml_doc_stgs_entry
        (MlViewSettingsBuilder * a_settings_builder) {
        g_return_if_fail (a_settings_builder != NULL);
        g_return_if_fail (MLVIEW_IS_SETTINGS_BUILDER
                          (a_settings_builder));
        g_return_if_fail (PRIVATE (a_settings_builder) != NULL);

        if (PRIVATE (a_settings_builder)->xml_doc_stgs_entry) {
                mlview_settings_entry_destroy
                        (PRIVATE (a_settings_builder)->
                         xml_doc_stgs_entry);

                PRIVATE (a_settings_builder)->
                        xml_doc_stgs_entry = NULL;
        }
}


/**
 *
 */
static void
 mlview_settings_builder_build_xml_doc_stgs_entry
        (MlViewSettingsBuilder * a_settings_builder) {
        GHashTable *settings = NULL;
        MlViewSettingsEntryHandlers *handlers;

        g_return_if_fail (a_settings_builder != NULL);
        g_return_if_fail (MLVIEW_IS_SETTINGS_BUILDER
                          (a_settings_builder));
        g_return_if_fail (PRIVATE (a_settings_builder) != NULL);

        settings =
                mlview_app_context_get_settings_hash_table
                (PRIVATE (a_settings_builder)->app_context);

        if (settings) {
                g_hash_table_insert (settings, (gpointer)
                                     MLVIEW_STG_K_IS_VALIDATION_ON,
                                     (gpointer)
                                     MLVIEW_STG_V_YES);
        }

        if (PRIVATE (a_settings_builder)->xml_doc_stgs_entry ==
            NULL) {
                PRIVATE (a_settings_builder)->
                        xml_doc_stgs_entry =
                        mlview_settings_entry_new (_
                                                   ("XML Documents"),
                                                   settings,
                                                   PRIVATE
                                                   (a_settings_builder)->
                                                   xml_document_settings_widget,
                                                   &PRIVATE
                                                   (a_settings_builder)->
                                                   doc_widgets);
                /*
                 *set the handlers.
                 */
                handlers =
                        mlview_settings_entry_get_handlers
                        (PRIVATE (a_settings_builder)->
                         xml_doc_stgs_entry);

                handlers->notify_settings_changed =
                        mlview_xml_doc_settings_changed_handler;

                handlers->
                        load_settings_from_xml_doc_into_settings_entry
                        = mlview_xml_doc_load_settings;

                handlers->get_settings_from_settings_widget =
                        mlview_xml_doc_get_settings_from_settings_widget_handler;

                handlers->set_settings_to_settings_widget =
                        mlview_xml_doc_set_settings_to_settings_widget_handler;
        }
}


static void
 mlview_xml_doc_settings_changed_handler
        (MlViewSettingsEntry * a_settings, void *a_user_data) {
        //do not have anything to do yet.
}


static void
 mlview_xml_doc_get_settings_from_settings_widget_handler
        (MlViewSettingsEntry * a_settings_entry) {
        GHashTable *settings;
        XMLDocumentSettingsWidgetStruct *settings_widgets_struct;
        gchar *depth_str;

        g_return_if_fail (a_settings_entry != NULL);

        settings =
                mlview_settings_entry_get_settings
                (a_settings_entry);

        g_return_if_fail (settings != NULL);
        settings_widgets_struct =
                mlview_settings_entry_get_settings_widgets_struct
                (a_settings_entry);
        g_return_if_fail (settings_widgets_struct);

        /*get "tree view depth expansion" setting */
        depth_str = (guchar *)
                gtk_entry_get_text (settings_widgets_struct->
                                    tree_exp_depth);

        if (depth_str != NULL && strcmp (depth_str, ""))
                g_hash_table_insert (settings, (gpointer)
                                     MLVIEW_STG_K_DEFAULT_TREE_EXPANSION_LEVEL,
                                     depth_str);

        /*
         *get the "validation on" setting
         */
        if (gtk_toggle_button_get_active
            (GTK_TOGGLE_BUTTON
             (settings_widgets_struct->validation_on))) {
                g_hash_table_insert (settings, (gpointer)
                                     MLVIEW_STG_K_IS_VALIDATION_ON,
                                     (gpointer)
                                     MLVIEW_STG_V_YES);
        } else {
                g_hash_table_insert (settings, (gpointer)
                                     MLVIEW_STG_K_IS_VALIDATION_ON,
                                     (gpointer) MLVIEW_STG_V_NO);
        }
}

static void
 mlview_xml_doc_set_settings_to_settings_widget_handler
        (MlViewSettingsEntry * a_settings_entry) {
        GHashTable *settings = NULL;
        XMLDocumentSettingsWidgetStruct *settings_widgets_struct
                = NULL;

        gchar *tmp_str = NULL;

        g_return_if_fail (a_settings_entry != NULL);

        settings =
                mlview_settings_entry_get_settings
                (a_settings_entry);

        g_return_if_fail (settings != NULL);

        settings_widgets_struct =
                mlview_settings_entry_get_settings_widgets_struct
                (a_settings_entry);

        g_return_if_fail (settings_widgets_struct);

        /*set expansion depth */
        tmp_str = g_hash_table_lookup (settings, (gpointer)
                                       MLVIEW_STG_K_DEFAULT_TREE_EXPANSION_LEVEL);

        /*do not free this content of tmp_str !! */
        gtk_entry_set_text (settings_widgets_struct->
                            tree_exp_depth, tmp_str);

        /*set validation_on */
        tmp_str = g_hash_table_lookup
                (settings,
                 (gpointer) MLVIEW_STG_K_IS_VALIDATION_ON);

        if (!strcmp (tmp_str, MLVIEW_STG_V_YES)) {
                gtk_toggle_button_set_active
                        (GTK_TOGGLE_BUTTON
                         (settings_widgets_struct->
                          validation_on), TRUE);
        } else {
                gtk_toggle_button_set_active
                        (GTK_TOGGLE_BUTTON
                         (settings_widgets_struct->
                          validation_on), FALSE);
        }
}


static void
mlview_xml_doc_load_settings (MlViewSettingsEntry *
                              a_settings_entry,
                              xmlDoc * a_xml_doc)
{
#define NODE_SET_IS_NOT_EMPTY(xpath_object) ((xpath_object) \
                                             &&(xpath_object)->nodesetval \
                                             && (xpath_object)->nodesetval->nodeNr)

#define NODE_SET_GET_NODE(xpath_object,indice) ((xpath_object)->nodesetval->nodeTab[indice])

        xmlXPathObject *xpath_object = NULL;
        gchar *expansion_level_str = NULL,
                *validation_on_str = NULL;
        GHashTable *settings = NULL;

        g_return_if_fail (a_settings_entry != NULL);
        g_return_if_fail (a_xml_doc != NULL);

        if (!mlview_settings_management_get_settings_xml_nodes
            (&xpath_object, "/gMlView:Settings/gMlView:TreeView",
             a_xml_doc)) {
                if (NODE_SET_IS_NOT_EMPTY (xpath_object)) {
                        xmlNode *node;

                        node = NODE_SET_GET_NODE (xpath_object,
                                                  0);
                        if (node) {
                                expansion_level_str =
                                        xmlGetProp (node,
                                                    "nodeExpansionLevel");
                        }
                        xmlXPathFreeObject (xpath_object);
                        xpath_object = NULL;
                }
        }

        if (!mlview_settings_management_get_settings_xml_nodes
            (&xpath_object,
             "/gMlView:Settings/gMlView:Validation", a_xml_doc))
        {
                if (NODE_SET_IS_NOT_EMPTY (xpath_object)) {
                        xmlNode *node;

                        node = NODE_SET_GET_NODE (xpath_object,
                                                  0);
                        if (node) {
                                validation_on_str =
                                        xmlGetProp (node,
                                                    "turnOn");
                        }
                        xmlXPathFreeObject (xpath_object);
                        xpath_object = NULL;
                }
        }

        /*set the settings */
        settings =
                mlview_settings_entry_get_settings
                (a_settings_entry);
        g_return_if_fail (settings != NULL);

        if (!strcmp (validation_on_str, "true")) {
                g_hash_table_insert (settings, (gpointer)
                                     MLVIEW_STG_K_IS_VALIDATION_ON,
                                     (gpointer)
                                     MLVIEW_STG_V_YES);
        } else {
                g_hash_table_insert (settings, (gpointer)
                                     MLVIEW_STG_K_IS_VALIDATION_ON,
                                     (gpointer) MLVIEW_STG_V_NO);
        }

        /*
         *FIXME: this causes a memleak because
         *expansion_level_str must be freed.
         *Hmmh, I must think about a way to handle
         *all this elegantly.
         */
        g_hash_table_insert (settings, (gpointer)
                             MLVIEW_STG_K_DEFAULT_TREE_EXPANSION_LEVEL,
                             expansion_level_str);

        if (validation_on_str) {
                xmlFree (validation_on_str);
                validation_on_str = NULL;
        }

}


/**
 *
 */
void
mlview_settings_builder_destroy (GtkObject * a_object)
{
        MlViewSettingsBuilder *settings_builder = NULL;

        g_return_if_fail (a_object != NULL);
        g_return_if_fail (MLVIEW_IS_SETTINGS_BUILDER (a_object));

        settings_builder = MLVIEW_SETTINGS_BUILDER (a_object);

        mlview_settings_builder_destroy_xml_doc_stgs_entry
                (settings_builder);

        if (PRIVATE (settings_builder)->settings_manager) {
                mlview_settings_manager_destroy
                        (PRIVATE (settings_builder)->
                         settings_manager);
                PRIVATE (settings_builder)->settings_manager =
                        NULL;
        }
        if (PRIVATE (settings_builder)) {
                g_free (PRIVATE (settings_builder));
                PRIVATE (settings_builder) = NULL;
        }

        /*call the destroy method of the parent */
        if (GTK_OBJECT_CLASS (parent_class)->destroy)
                (*GTK_OBJECT_CLASS (parent_class)->
                 destroy) (a_object);
}


/*=====================================================
 *Public Methods ...
 *
 *====================================================*/

MlViewSettingsBuilder *
mlview_settings_builder_new (MlViewAppContext * a_app_context)
{
        MlViewSettingsBuilder *settings_builder;

        settings_builder =
                gtk_type_new (MLVIEW_TYPE_SETTINGS_BUILDER);
        mlview_settings_builder_construct (settings_builder,
                                           a_app_context);

        return settings_builder;
}


gint
mlview_settings_builder_get_type (void)
{
        static guint settings_builder_type = 0;

        if (!settings_builder_type) {
                static const GTypeInfo settings_builder_type_info
                        = {
                        sizeof (MlViewSettingsBuilderClass),
                        NULL,   /* base_init */
                        NULL,   /* base_finalize */
                        (GClassInitFunc)
                                mlview_settings_builder_init_class,
                        NULL,   /* class_finalize */
                        NULL,   /* class_data */
                        sizeof (MlViewSettingsBuilder),
                        0,
                        (GInstanceInitFunc)
                        mlview_settings_builder_init
                };

                settings_builder_type =
                        g_type_register_static (GTK_TYPE_OBJECT,
                                                "MlViewSettingsBuilder",
                                                &settings_builder_type_info,
                                                0);
        }

        return settings_builder_type;
}

void
 mlview_settings_builder_build_settings
        (MlViewSettingsBuilder * a_settings_builder) {
        xmlDoc *settings_xml_doc = NULL;

        g_return_if_fail (a_settings_builder != NULL);
        g_return_if_fail (MLVIEW_IS_SETTINGS_BUILDER
                          (a_settings_builder));
        g_return_if_fail (PRIVATE (a_settings_builder) != NULL);

        /*build all the settings entries */
        mlview_settings_builder_build_xml_document_settings_widget
                (a_settings_builder);

        /*install all the settings entries */
        mlview_settings_builder_build_xml_doc_stgs_entry
                (a_settings_builder);

        if (PRIVATE (a_settings_builder)->xml_doc_stgs_entry !=
            NULL) {
                mlview_settings_manager_install_settings (PRIVATE
                                                          (a_settings_builder)->
                                                          settings_manager,
                                                          PRIVATE
                                                          (a_settings_builder)->
                                                          xml_doc_stgs_entry);
        }

        /*if user mlview settings dir/files does not exist, create them */
        if (!mlview_settings_manager_personal_settings_file_exists ()) {
                gint settings_file_creation_status;

                settings_file_creation_status =
                        mlview_settings_manager_create_personal_settings_file
                        (&settings_xml_doc);

                if (settings_file_creation_status != 0
                    || settings_xml_doc == NULL) {
                        mlview_app_context_error
                                (PRIVATE (a_settings_builder)->
                                 app_context,
                                 _
                                 ("Could not create the user personal settings.\nPlease, Please, make sure you have write acces to your home directory"));
                        exit (-1);
                }

        }

        if (!settings_xml_doc) {
                settings_xml_doc =
                        mlview_settings_manager_load_settings_from_disk
                        (PRIVATE (a_settings_builder)->
                         settings_manager);
        }

        mlview_settings_manager_post_settings_install_init
                (PRIVATE (a_settings_builder)->settings_manager,
                 settings_xml_doc);

        if (settings_xml_doc) {
                xmlFreeDoc (settings_xml_doc);
                settings_xml_doc = NULL;
        }
}

MlViewSettingsManager
        * mlview_settings_builder_get_settings_manager
        (MlViewSettingsBuilder * a_settings_builder) {
        g_return_val_if_fail (a_settings_builder != NULL, NULL);

        g_return_val_if_fail (MLVIEW_IS_SETTINGS_BUILDER
                              (a_settings_builder), NULL);
        g_return_val_if_fail (PRIVATE (a_settings_builder) !=
                              NULL, NULL);

        return PRIVATE (a_settings_builder)->settings_manager;
}
