/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute it 
 *and/or modify it under the terms of 
 *the GNU General Public License as published
 * by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will be useful, 
 *but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of MERCHANTABILITY or 
 *FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the GNU General Public License 
 *along with MlView; 
 *see the file COPYING. If not, write to the 
 *Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *Boston, MA 02111-1307, USA.
 *
 *Copyright 2001-2002 Dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

/**
 *@file
 *The definition of the #MlViewAppContext class.
 *
 *The context is the way by which the application container 
 *and the applications communicate (here, the application is MlViewEditor).
 *Both applications and the application container can store information in it.
 *The application container insures the consistency of the 
 *application context during the life time of the application. 
 *
 */
#include "mlview-app-context.h"
#include "mlview-global-settings.h"
#include "mlview-file-descriptor.h"
#include "mlview-utils.h"

#include <stdarg.h>

/**
 *The private data members of MlViewAppContext.
 */
struct _MlVAppContextPrivate {

 /**The place to store some elements of the context.*/
        GHashTable *context_elements;

 /**The place to store key/values pairs (the values are strings)*/
        GHashTable *global_settings;

 /**A cache for the xpms. 
	 *We have choosen this method
	 *to enable pixmap designers to
	 *change the pixmaps and have
	 *mlview load the new pixmap dynamically.
	 */
        GHashTable *pixmaps_cache;

        /** 
	 *mlview application bitmaps cache.
	 *one the bitmap have been loaded from the disk, they
	 *are cached in this hash table.
	 */
        GHashTable *bitmaps_cache;

 /**
	 *The place where the error messages are
	 *put before being displayed.
	 */
        gchar *error_msg_buffer;

 /**
	 *The title of the error dialog box
	 */
        gchar *error_dialog_title;

 /**
	 *A cache for the instance of MlViewFileSelection
	 *used by instance of MlViewAppContext.
	 */
        MlViewFileSelection *file_sel;

 /**
	 *The xml catalog loaded by mlview
	 */
        xmlCatalog *xml_catalog;

        /*
         *the last attribute id number 
         *part generated by mlview
         */
        gint last_id;

        gboolean dispose_has_run;
};


/**
 *the macro to access the private part of MlViewAppContext object
 */
#define PRIVATE(mlv_app_context) (mlv_app_context->private)

static GObjectClass *p_parent_class = NULL;

static void
 mlview_app_context_init_class (MlViewAppContextClass * a_klass);

static void
 mlview_app_context_init (MlViewAppContext * a_ctxt);

static void
 mlview_app_context_finalize (GObject * a_this);

static void
 mlview_app_context_dispose (GObject * a_object);

static GObject *mlview_app_context_new (void);

static void
 pixmaps_cache_foreach_func (gchar * a_key,
                             GdkPixmap * a_value,
                             gchar * user_data);

static void
 bitmaps_cache_foreach_func (gchar * a_key,
                             GdkBitmap * a_value,
                             gchar * user_data);

/*=====================================================
 *Private methods imposed by the gtk object framework.
 *====================================================*/

/**
 *The standard type builder. 
 *@return the id of the type of
 *#MlViewAppContext.
 */
guint
mlview_app_context_get_type (void)
{
        static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewAppContextClass),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_app_context_init_class,
                        NULL, NULL,
                        sizeof (MlViewAppContext),
                        0,
                        (GInstanceInitFunc)
                        mlview_app_context_init
                };
                type = g_type_register_static (G_TYPE_OBJECT,
                                               "MlViewAppContext",
                                               &type_info, 0);
        }
        return type;
}

/**
 *The standard class initialyzer. 
 *@param a_klass the vtable structure of MlViewAppContext.
 */
static void
mlview_app_context_init_class (MlViewAppContextClass * a_klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (a_klass);

        p_parent_class = g_type_class_peek_parent (a_klass);
        g_return_if_fail (G_IS_OBJECT_CLASS (p_parent_class));
        object_class->dispose = mlview_app_context_dispose;
        object_class->finalize = mlview_app_context_finalize;
}


/**
 *The MlViewAppContext instance initialyzer.
 *This methods allocates the MlViewAppContext object structure.
 *
 *@param a_ctxt the current instance of MlViewAppContext.
 */
static void
mlview_app_context_init (MlViewAppContext * a_ctxt)
{
        g_return_if_fail (a_ctxt != NULL);

        if (PRIVATE (a_ctxt) == NULL)
                PRIVATE (a_ctxt) =
                        g_malloc0 (sizeof
                                   (MlVAppContextPrivate));

        if (PRIVATE (a_ctxt)->global_settings == NULL)
                PRIVATE (a_ctxt)->global_settings =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        if (PRIVATE (a_ctxt)->context_elements == NULL)
                PRIVATE (a_ctxt)->context_elements =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        if (PRIVATE (a_ctxt)->pixmaps_cache == NULL)
                PRIVATE (a_ctxt)->pixmaps_cache =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        if (PRIVATE (a_ctxt)->bitmaps_cache == NULL)
                PRIVATE (a_ctxt)->bitmaps_cache =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        /*
         *initialise the list of available encodings
         *supported by mlview and ref count it.
         */
        mlview_utils_init_available_encodings_list ();

        mlview_utils_ref_available_encodings ();

        PRIVATE (a_ctxt)->dispose_has_run = FALSE;
}


/**
 *The default constructor of MlViewAppContext.
 *
 *@return the newly created instance of MlViewAppContext.
 */
static GObject *
mlview_app_context_new (void)
{
        MlViewAppContext *result = NULL;

        result = g_object_new (MLVIEW_TYPE_APP_CONTEXT, NULL);

        return G_OBJECT (result);
}


/**
 *The instance getter of the MlViewAppContext class.
 *Actually, if you want to use a 'singleton pattern', use this method.
 *All the invocations to this method will allways return the same object ;
 *
 *@return the singleton object. The firts time this method is called, creates
 *a new object and returns it. subsequent calls will always 
 *returns the same object.
 */
MlViewAppContext *
mlview_app_context_get_instance (void)
{
        static MlViewAppContext *app_context = NULL;

        if (!app_context) {
                app_context =
                        MLVIEW_APP_CONTEXT
                        (mlview_app_context_new ());
        }
        return app_context;
}


/**
 *a getter of the last attribute id automatically generated.
 *This method is used in the automatic generation of attributes ids.
 *
 *@param a_app_context: the current instance of MlViewAppContext.
 *
 *
 *@returns a pointer to the last id stored in the object a_app_context using
 *the method mlview_app_context_set_last_id() ; 
 *You should manipulate this pointer
 *with great care or else you can break the attribute id storage.
 */
gint *
mlview_app_context_get_last_id_ptr (MlViewAppContext *
                                    a_app_context)
{
        g_return_val_if_fail (a_app_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), NULL);
        g_return_val_if_fail (PRIVATE (a_app_context) != NULL,
                              NULL);

        return &PRIVATE (a_app_context)->last_id;
}


/**
 *The getter of the last generated attribute id.
 *This method is used in the automatic generation of attributes ids.
 *
 *@param a_app_context the current instance of MlViewAppContext.
 *@return the last generated attribute id.
 */
gint
mlview_app_context_get_last_id (MlViewAppContext * a_app_context)
{
        g_return_val_if_fail (a_app_context != NULL, 0);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), 0);
        g_return_val_if_fail (PRIVATE (a_app_context) != NULL,
                              0);

        return PRIVATE (a_app_context)->last_id;
}


/**
 *A setter of the last generated attribut id.
 *This method is used in the automatic attribute id generation process.
 *
 *@param a_app_context the current instance of MlViewAppContext.
 *@param a_new_id the new id.
 */
void
mlview_app_context_set_last_id (MlViewAppContext * a_app_context,
                                gint a_new_id)
{
        g_return_if_fail (a_app_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_app_context));
        g_return_if_fail (PRIVATE (a_app_context) != NULL);

        PRIVATE (a_app_context)->last_id = a_new_id;
}


/**
 *Stores a key/value pair in the context.
 *This method is used tho stores objects in the context.
 *If you want to store strings, (for settings for ex) 
 *use mlview_app_context_set_settings_value() method instead.
 *
 *@param a_context the current instance of MlViewAppContext class.
 *@param a_element_name the key of the object to store. 
 *Note that this string is not copied so this
 *pointer must remain valid during all the lifetime of a_context. 
 *When a_context is destroyed, 
 *a_element_name is _not_ destroyed. 
 *So the caller must manage the life time of this string.
 *@param a_context_element the element to store in the context. 
 *This pointer is not destroyed upon
 *destruction of a_context. 
 *It is up to the caller to manage the life time of this pointer. If
 *it is NULL, a NULL value is stored.
 */
void
mlview_app_context_set_element (MlViewAppContext * a_context,
                                const gchar * a_element_name,
                                gpointer a_context_element)
{
        g_return_if_fail (a_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_context));
        g_return_if_fail (PRIVATE (a_context) != NULL);
        g_return_if_fail (a_element_name != NULL);

        if (PRIVATE (a_context)->context_elements == NULL)
                PRIVATE (a_context)->context_elements =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        g_hash_table_insert (PRIVATE (a_context)->
                             context_elements,
                             (guchar *) a_element_name,
                             a_context_element);
}


/**
 *Gets an object stored in the context using the 
 *mlview_app_context_set_element() method.
 *@param a_context the current instance of MlViewAppContext.
 *@param a_element_name the key of the object stored.
 *@return the element stored or NULL if nothing has been stored. 
 *If a NULL value has been stored
 *using this key, (using the mlview_app_context_set_element() method) 
 *a NULL value is also returned.
 */
gpointer
mlview_app_context_get_element (MlViewAppContext * a_context,
                                const gchar * a_element_name)
{
        g_return_val_if_fail (a_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT (a_context),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_context) != NULL, NULL);

        return g_hash_table_lookup (PRIVATE (a_context)->
                                    context_elements,
                                    a_element_name);
}


/**
 *Pushes a message on the app bar. 
 *@param a_context the current instance of MlViewAppContext.
 *@param a_msg_format the format of the message (same parameter as printf)
 */
void
mlview_app_context_sbar_push_message (MlViewAppContext *
                                      a_context,
                                      const gchar * a_msg_format,
                                      ...)
{
        GnomeAppBar *app_bar = NULL;
        GnomeApp *app = NULL;
        va_list params;
        gchar *msg = NULL;


        if (a_context && PRIVATE (a_context)) {
                app = g_hash_table_lookup
                        (PRIVATE (a_context)->context_elements,
                         "GnomeApp");

                app_bar = g_hash_table_lookup
                        (PRIVATE (a_context)->context_elements,
                         "GnomeAppBar");
        }

        va_start (params, a_msg_format);
        msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        if (msg && app && app_bar) {
                gnome_appbar_push (app_bar, msg);
        } else if (msg && app) {
                gnome_app_message (app, msg);
        } else if (msg) {
                g_printerr (msg);
        }

        if (msg) {
                g_free (msg);
                msg = NULL;
        }
}


/**
 *Pops the last message pushed on the message stack.
 *The effect is that the message pushed before the 
 *last message has been pushed, will be
 *displayed. 
 *
 *@param a_context the current instance of MlViewAppContext.
 *
 */
void
mlview_app_context_sbar_pop_message (MlViewAppContext *
                                     a_context)
{
        GnomeAppBar *app_bar = NULL;

        if (a_context && PRIVATE (a_context)) {
                app_bar = g_hash_table_lookup
                        (PRIVATE (a_context)->context_elements,
                         "GnomeAppBar");
        }

        if (app_bar)
                gnome_appbar_pop (app_bar);
}


/**
 *Sets the message that is displayed when no message is pushed on the
 *status bar message stack. 
 *
 *@param a_context the current instance of MlViewAppContext.
 *@param a_msg_format the format of the message 
 *(as in the standard printf function)
 */
void
mlview_app_context_sbar_set_default_message (MlViewAppContext *
                                             a_context,
                                             const gchar *
                                             a_msg_format, ...)
{
        GnomeAppBar *app_bar = NULL;
        va_list params;
        gchar *msg = NULL;

        if (a_context && PRIVATE (a_context)) {
                app_bar = g_hash_table_lookup
                        (PRIVATE (a_context)->context_elements,
                         "GnomeAppBar");
        }

        va_start (params, a_msg_format);
        msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        if (app_bar)
                gnome_appbar_set_default (app_bar, msg);

        if (msg) {
                g_free (msg);
                msg = NULL;
        }
}


/**
 *Displays a message to the user. 
 *It opens a Modal dialog with the message on it.
 *The user just has to click OK. 
 *
 *@param a_context the current instance of MlViewAppContext.
 *@param a_msg_format the format of the message (as in the printf funtion)
 */
void
mlview_app_context_message (MlViewAppContext * a_context,
                            const gchar * a_msg_format, ...)
{
        GnomeApp *app = NULL;
        va_list params;
        gchar *msg = NULL;

        if (a_context && PRIVATE (a_context))
                app = g_hash_table_lookup
                        (PRIVATE (a_context)->context_elements,
                         "GnomeApp");

        va_start (params, a_msg_format);
        msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        if (msg && app) {
                gnome_app_message (app, msg);
        } else if (msg) {
                g_printerr (msg);
                return;
        }

        if (msg) {
                g_free (msg);
                msg = NULL;
        }
}

/**
 *Displays a (fatal) error to the user. It actually 
 *shows a modal dialog box to the user.
 *The user just has to click OK. 
 *
 *@param a_context the current instance of MlViewAppContext.
 *@param a_msg_format the format of the 
 *message (as in the standard printf funtion)
 */
void
mlview_app_context_error (MlViewAppContext * a_context,
                          const gchar * a_msg_format, ...)
{
        GnomeApp *app = NULL;
        va_list params;
        gchar *err_msg = NULL;

        if (a_context && PRIVATE (a_context))
                app = g_hash_table_lookup
                        (PRIVATE (a_context)->context_elements,
                         "GnomeApp");

        va_start (params, a_msg_format);
        err_msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        /*Add a title to the error message if needed */
        if (err_msg && PRIVATE (a_context)->error_dialog_title) {
                gchar *tmp_str = err_msg;

                err_msg = g_strconcat
                        (PRIVATE (a_context)->error_dialog_title,
                         "\n", tmp_str, NULL);
                g_free (tmp_str);

                /*delete the error dialog title */
                g_free (PRIVATE (a_context)->error_dialog_title);
                PRIVATE (a_context)->error_dialog_title = NULL;
        }

        if (app && err_msg) {
                /*gnome_app_error (app, err_msg) ; */
                mlview_utils_display_error_dialog (a_context,
                                                   err_msg);
        } else if (err_msg) {
                g_printerr (err_msg);
                return;
        }

        if (err_msg)
                g_free (err_msg);
}


/**
 *Sets the title of the next error message that will be displayed by a call
 *to the function mlview_app_context_error() or 
 *mlview_app_context_display_buffered_error().
 *After a call to one of these two functions, 
 *all subsequent calls will display an error dialog
 *with no title. 
 *
 *@param a_title the new title of the error 
 *message dialog thatwill be used to display the 
 *content of the error message buffer.
 *look at definition of mlview_app_context_bufferize_error(). 
 *@param a_context the current application context.
 *
 *
 */
void
mlview_app_context_set_error_dialog_title (MlViewAppContext *
                                           a_context,
                                           const gchar * a_title)
{
        if (!a_context || !PRIVATE (a_context))
                return;

        if (PRIVATE (a_context)->error_dialog_title) {
                g_free (PRIVATE (a_context)->error_dialog_title);
                PRIVATE (a_context)->error_dialog_title = NULL;
        }

        if (a_title)
                PRIVATE (a_context)->error_dialog_title =
                        g_strdup (a_title);
        else
                PRIVATE (a_context)->error_dialog_title = NULL;
}


/**
 *Getter of error message dialog title.
 *@param a_context the current application context.
 *
 *@return a pointer to the title of the next 
 *error message that will be displayed. 
 */
gchar *
mlview_app_context_get_error_dialog_title (MlViewAppContext *
                                           a_context)
{
        if (!a_context || !PRIVATE (a_context))
                return NULL;

        return PRIVATE (a_context)->error_dialog_title;
}


/**
 *Bufferizes the error message given by a_msg_format and the other parameters.
 *The buffered error message can be displayed to the user by calling 
 *the function
 *@mlview_app_context_display_buffered_error() 
 *with the same a_context in parameter.
 *Note that if this function is called with a NULL a_context param, the
 *formated error message is sent to stderr and is not buffered. Subsquent call
 *to mlview_app_context_display_buffered_error() will not display anything. 
 *
 *@param a_msg_format the format string. 
 *Plays the same role asthe format string used in the function fprintf.
 *@param ... the parameters that may be used in the format string. 
 *@param a_context the current application context.
 *
 *
 */
void
mlview_app_context_bufferize_error (MlViewAppContext * a_context,
                                    const gchar * a_msg_format,
                                    ...)
{
        GnomeApp *app = NULL;
        va_list params;
        gchar *temp_err_msg = NULL;

        if (a_context && PRIVATE (a_context))
                app = g_hash_table_lookup (PRIVATE (a_context)->
                                           context_elements,
                                           "GnomeApp");

        va_start (params, a_msg_format);
        temp_err_msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        if (app && temp_err_msg) {

                if (!PRIVATE (a_context)->error_msg_buffer)

                        PRIVATE (a_context)->error_msg_buffer =
                                g_strdup (temp_err_msg);
                else {
                        gchar *old_str =
                                PRIVATE (a_context)->
                                error_msg_buffer;

                        PRIVATE (a_context)->error_msg_buffer =
                                g_strconcat
                                (PRIVATE (a_context)->
                                 error_msg_buffer, temp_err_msg,
                                 NULL);
                        g_free (old_str);
                }
        } else if (temp_err_msg) {
                g_printerr (temp_err_msg);
        }
        if (temp_err_msg)
                g_free (temp_err_msg);
}


/**
 *Displays the error message that has been buffered by calling the function
 *mlview_app_context_bufferize_error() with a_context in parameter.
 *If a_context is set to NULL,  this function does nothing. 
 *
 *@param a_context the current application context.
 */
void
mlview_app_context_display_buffered_error (MlViewAppContext *
                                           a_context)
{

        if (!a_context || !PRIVATE (a_context)
            || !PRIVATE (a_context)->error_msg_buffer)
                return;

        mlview_app_context_error (a_context,
                                  PRIVATE (a_context)->
                                  error_msg_buffer);

        g_free (PRIVATE (a_context)->error_msg_buffer);

        PRIVATE (a_context)->error_msg_buffer = NULL;
}

/**
 *Tells wether if there are error messages to be displayed or not.
 *Note the messages that may have been stored in the 
 *message buffer have been stored
 *using the function mlview_app_context_bufferize_error() .
 *
 *@param a_context the current instance of MlViewAppContext.
 *@return TRUE if the error buffer is empty and false if not. 
 */
gboolean
mlview_app_context_error_buffer_is_empty (MlViewAppContext *
                                          a_context)
{
        g_return_val_if_fail (a_context != NULL, TRUE);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT (a_context),
                              TRUE);
        g_return_val_if_fail (PRIVATE (a_context) != NULL, TRUE);

        return (PRIVATE (a_context)->error_msg_buffer == NULL
                || !strcmp (PRIVATE (a_context)->
                            error_msg_buffer, ""));
}


/**
 *Displays a warning message. It actually displays a modal 
 *dialog showing the message.
 *The user just has to click OK. 
 *If a_context is set to NULL, the message is sent to stderr. 
 *
 *@param a_msg_format the format of the warning message to display.
 *@param a_context the current application context. 
 *If NULL, stderr is used to display.
 */
void
mlview_app_context_warning (MlViewAppContext * a_context,
                            const gchar * a_msg_format, ...)
{
        GnomeApp *app = NULL;
        va_list params;
        gchar *warning_msg = NULL;

        if (a_context && PRIVATE (a_context))
                app = g_hash_table_lookup
                        (PRIVATE (a_context)->context_elements,
                         "GnomeApp");

        va_start (params, a_msg_format);
        warning_msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        if (app && warning_msg) {
                gnome_app_warning (app, warning_msg);
        } else if (warning_msg) {
                g_printerr (warning_msg);
        }

        if (warning_msg)
                g_free (warning_msg);
}


/**
 *Getter of the application wide settings hash table.
 *This hash table can be accessed by all the components that have received
 *the current instance of MlViewAppContext in parameter.
 *Note that this hash table is set by default during the construction 
 *of the instance
 *of MlViewAppContext. It can be also set by a call to the function 
 *link mlview_app_context_set_settings_hash_table()  ;
 *
 *@param a_context the current instance of MlViewAppContext.
 *
 *@return the application wide settings hashtable. 
 */
GHashTable *
mlview_app_context_get_settings_hash_table (MlViewAppContext *
                                            a_context)
{
        g_return_val_if_fail (a_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT (a_context),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_context) != NULL, NULL);

        return PRIVATE (a_context)->global_settings;
}

/**
 *Sets a new application wide settings hash table.
 *The data set must be a string. If you want to set something
 *else than a string, use the method mlview_app_context_set_element()
 *instead.
 *Note that Subsequent calls to mlview_app_context_settings_exist() ,
 *, mlview_app_context_set_settings_value() and
 *mlview_app_context_get_settings_value() will use the new hash table
 *provided.
 *
 *@param a_settings the new settings hash table to set.
 *@param a_context the current instance of MlViewAppContext.
 *
 */
void
mlview_app_context_set_settings_hash_table (MlViewAppContext *
                                            a_context,
                                            GHashTable *
                                            a_settings)
{
        g_return_if_fail (a_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_context));
        g_return_if_fail (PRIVATE (a_context) != NULL);

        PRIVATE (a_context)->global_settings = a_settings;
}


/**
 *Tests wether if the application wide settings
 * hash table is present (non null) or not.
 *
 *@param a_app_context the current instance of MlViewAppContext.
 *@return TRUE if the application wide settings 
 *hash table is present and FALSE if not.
 */
gboolean
mlview_app_context_settings_exist (MlViewAppContext *
                                   a_app_context)
{
        g_return_val_if_fail (a_app_context != NULL, FALSE);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), FALSE);
        g_return_val_if_fail (PRIVATE (a_app_context) != NULL,
                              FALSE);

        if (PRIVATE (a_app_context)->global_settings)
                return TRUE;

        return FALSE;
}

/**
 *Lookups the application wide settings hash 
 *table to search a given setting value.
 *Do *NOT* free the returned pointer. 
 *
 *@param a_app_context the current instance of 
 *MlViewAppContext.Looks up in the settings 
 *hash table (if the settings hash table is present) for
 *the value that matches the key a_key. If the settings 
 *hash table is not present or 
 *@param a_key the key of the settings to get.
 *
 *@return the settings value or NULL if it does not exist.
 */
gchar *
mlview_app_context_get_settings_value (MlViewAppContext *
                                       a_app_context,
                                       const gchar * a_key)
{
        g_return_val_if_fail (a_app_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), NULL);
        g_return_val_if_fail (PRIVATE (a_app_context) != NULL,
                              NULL);

        if (PRIVATE (a_app_context)->global_settings)
                return g_hash_table_lookup
                        (PRIVATE (a_app_context)->
                         global_settings, a_key);

        return NULL;
}


/**
 *Inserts the pair (a_key, a_value) into the settings hash table. 
 *(if the hash table exists of course.)
 *If the hash table does not exists, does nothing.
 *If the a_key already exists in the settings hash table, overrides it. 
 *
 *@param a_value the value of the setting key/value pair. 
 *Should be a static storage class object so that we do not have to free it.
 *@param a_key the key of the setting key/value pair. 
 *Should be a static storage class array of char.
 *@param a_app_context the current instance of MlViewAppContext.
 *
 */
void
mlview_app_context_set_settings_value (MlViewAppContext *
                                       a_app_context,
                                       const gchar * a_key,
                                       const gchar * a_value)
{
        g_return_if_fail (a_app_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_app_context));
        g_return_if_fail (PRIVATE (a_app_context) != NULL);

        if (PRIVATE (a_app_context)->global_settings) {
                g_hash_table_insert (PRIVATE (a_app_context)->
                                     global_settings,
                                     (gpointer) a_key,
                                     (gpointer) a_value);
        }
}


/**
 *Gets the file selection dialog box associated to the mlview application.
 *The first call to this method creates the file selection. Subsequent
 *calls to this method returns a pointer to the same object, making these calls
 *very fast.
 *
 *@param a_app_context: the current xml document considered.
 *@param a_title the title of the mlview file selector.
 *
 *@returns the file selection associated to this mlview app context.
 */
MlViewFileSelection *
mlview_app_context_get_file_selector (MlViewAppContext *
                                      a_app_context,
                                      const gchar * a_title)
{

        g_return_val_if_fail (a_app_context != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_app_context) != NULL,
                              NULL);

        if (!PRIVATE (a_app_context)->file_sel) {
                PRIVATE (a_app_context)->file_sel =
                        MLVIEW_FILE_SELECTION
                        (mlview_file_selection_new ());

                mlview_utils_set_window_deco
                        (GTK_WINDOW
                         (PRIVATE (a_app_context)->file_sel),
                         a_app_context, "file-selection-dialog");
        }

        if (a_title && strcmp (a_title, ""))
                gtk_window_set_title
                        (GTK_WINDOW
                         (PRIVATE (a_app_context)->file_sel),
                         a_title);

        return PRIVATE (a_app_context)->file_sel;
}


/**
 *Sets the icon of the window of the widget. All the windows of MlView
 *have the same icon. This function fetches that right icon
 *and set it to the window. 
 *
 *@param a_widget the widget for which the icon must be set.
 *@param a_app_context the current instance of application context.
 *
 */
void
mlview_app_context_set_window_icon (MlViewAppContext *
                                    a_app_context,
                                    GtkWindow * a_widget)
{
        gchar *icon_path = NULL;
        GdkPixbuf *pixbuf = NULL;

        g_return_if_fail (a_app_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_app_context));
        g_return_if_fail (a_widget != NULL);
        g_return_if_fail (GTK_IS_WINDOW (a_widget));

        /*load application icon */
        icon_path =
                mlview_app_context_get_settings_value
                (a_app_context, MLVIEW_STG_K_ICON_PATH);

        if (icon_path) {
                pixbuf = gdk_pixbuf_new_from_file (icon_path,
                                                   NULL);
                if (pixbuf)
                        gtk_window_set_icon (a_widget, pixbuf);
        }
}


/**
 *Load a pixmap by it name.
 *
 *@param a_pixmap out parameter. A pointer to the returned pixmap.
 *@param a_bitmap out parameter. A pointer to the returned bitmap.
 *@param a_xpm_name the name of the xpm file to get. 
 *Please, give the name withoutthe extension. 
 *@param a_app_context the current instance of MlViewAppContext.
 *
 *@return 0 if ok, an error code if not.
 */
gint
mlview_app_context_get_xpm (MlViewAppContext * a_app_context,
                            const gchar * a_xpm_name,
                            GdkPixmap ** a_pixmap,
                            GdkBitmap ** a_bitmap)
{
        enum ReturnStatus {
                OK,
                BAD_PARAM,
                FILE_NOT_READABLE,
                COULD_NOT_LOAD_PIXMAP
        };

        gchar *pixmap_dir = NULL;

        gchar *real_xpm_name = NULL;

        gchar *pixmap_file = NULL;

        MlViewFileDescriptor *file_desc = NULL;

        gboolean is_readable = FALSE;

        GtkWidget *app = NULL;

        g_return_val_if_fail (a_app_context != NULL, BAD_PARAM);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), BAD_PARAM);
        g_return_val_if_fail (PRIVATE (a_app_context) != NULL,
                              BAD_PARAM);
        g_return_val_if_fail (PRIVATE (a_app_context)->
                              pixmaps_cache, BAD_PARAM);
        g_return_val_if_fail (PRIVATE (a_app_context)->
                              bitmaps_cache, BAD_PARAM);
        g_return_val_if_fail (a_pixmap != NULL, BAD_PARAM);
        g_return_val_if_fail (a_bitmap != NULL, BAD_PARAM);

        *a_pixmap = NULL;
        *a_bitmap = NULL;

        real_xpm_name =
                mlview_app_context_get_settings_value
                (a_app_context, a_xpm_name);

        g_return_val_if_fail (real_xpm_name != NULL, BAD_PARAM);

        /*first look in the cache */
        *a_pixmap =
                g_hash_table_lookup (PRIVATE (a_app_context)->
                                     pixmaps_cache,
                                     real_xpm_name);
        *a_bitmap =
                g_hash_table_lookup (PRIVATE (a_app_context)->
                                     bitmaps_cache,
                                     real_xpm_name);

        if (*a_pixmap == NULL) {
                app = mlview_app_context_get_element
                        (a_app_context, "GnomeApp");

                g_return_val_if_fail (app != NULL, BAD_PARAM);
                g_return_val_if_fail (app->window != NULL,
                                      BAD_PARAM);

                pixmap_dir =
                        mlview_app_context_get_settings_value
                        (a_app_context,
                         MLVIEW_STG_K_PIXMAPS_DIR);

                pixmap_file =
                        g_strconcat (pixmap_dir, "/",
                                     real_xpm_name, ".xpm",
                                     NULL);

                file_desc =
                        mlview_file_descriptor_new (pixmap_file);

                g_return_val_if_fail (file_desc != NULL,
                                      BAD_PARAM);

                if (!mlview_file_descriptor_is_readable
                    (file_desc, &is_readable)
                    && is_readable == TRUE) {

                        *a_pixmap =
                                gdk_pixmap_create_from_xpm (app->
                                                            window,
                                                            a_bitmap,
                                                            NULL,
                                                            pixmap_file);

                } else {

                        mlview_file_descriptor_destroy
                                (file_desc);
                        return FILE_NOT_READABLE;
                }

                g_free (pixmap_file);
                pixmap_file = NULL;

                mlview_file_descriptor_destroy (file_desc);
                file_desc = NULL;

                if (!*a_pixmap)
                        return COULD_NOT_LOAD_PIXMAP;

                g_hash_table_insert (PRIVATE (a_app_context)->
                                     pixmaps_cache,
                                     real_xpm_name, *a_pixmap);
                g_hash_table_insert (PRIVATE (a_app_context)->
                                     bitmaps_cache,
                                     real_xpm_name, *a_bitmap);
        }

        return OK;
}


/**
 *Makes a_window always stay on top of the mlview application root window. 
 *
 *@param a_window the window that must be transient for 
 *the mlview app root window.
 *@param a_app_context the current application context.
 */
void
mlview_app_context_set_window_transient_for_app (MlViewAppContext
                                                 * a_app_context,
                                                 GtkWindow *
                                                 a_window)
{
        GnomeApp *app = NULL;

        g_return_if_fail (a_app_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_app_context));
        g_return_if_fail (a_window != NULL);
        g_return_if_fail (GTK_IS_WINDOW (a_window));

        app = mlview_app_context_get_element (a_app_context,
                                              "GnomeApp");
        g_return_if_fail (app != NULL);
        g_return_if_fail (GNOME_IS_APP (app));

        gtk_window_set_transient_for (a_window,
                                      GTK_WINDOW (app));
}

/**
 *Associates an xml catalog to the current instance of MlViewAppContext.
 *When a_app_context is destroyed, 
 *the instance of xml catalog is _NOT_ destroyed.
 *
 *@param a_xml_catalog the new xml catalog
 *@param a_app_context the current instance of MlViewAppContext.
 *
 */
void
mlview_app_context_set_xml_catalog (MlViewAppContext *
                                    a_app_context,
                                    xmlCatalog * a_xml_catalog)
{
        g_return_if_fail (a_app_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_app_context));
        g_return_if_fail (PRIVATE (a_app_context) != NULL);

        PRIVATE (a_app_context)->xml_catalog = a_xml_catalog;
}


/**
 *Getter of the xml catalog associated to the current instance of
 *MlViewAppContext.
 *
 *@param the current instance of MlViewAppContext .
 *@return the instance of xmlCatalog associated to the current instance of
 *MlViewAppContext.
 */
xmlCatalog *
mlview_app_context_get_xml_catalog (MlViewAppContext *
                                    a_app_context)
{
        g_return_val_if_fail (a_app_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), NULL);
        g_return_val_if_fail (PRIVATE (a_app_context) != NULL,
                              NULL);

        return (PRIVATE (a_app_context)->xml_catalog);
}

static void
pixmaps_cache_foreach_func (gchar * a_key,
                            GdkPixmap * a_value,
                            gchar * user_data)
{
        gdk_pixmap_unref (a_value);
}

static void
bitmaps_cache_foreach_func (gchar * a_key,
                            GdkBitmap * a_value,
                            gchar * user_data)
{
        gdk_bitmap_unref (a_value);
}


/*
 *The destroy method associated to this class. 
 *
 *@param a_object the instance of MlViewAppContext to be destroyed.
 *
 */
static void
mlview_app_context_dispose (GObject * a_object)
{
        MlViewAppContext *ctxt = NULL;

        g_return_if_fail (a_object);

        ctxt = MLVIEW_APP_CONTEXT (a_object);

        g_return_if_fail (PRIVATE (ctxt));
        g_return_if_fail (PRIVATE (ctxt)->dispose_has_run ==
                          FALSE);

        if (PRIVATE (ctxt)->context_elements) {
                g_hash_table_destroy (PRIVATE (ctxt)->
                                      context_elements);
        }

        if (PRIVATE (ctxt)->global_settings) {
                g_hash_table_destroy (PRIVATE (ctxt)->
                                      global_settings);
                PRIVATE (ctxt)->global_settings = NULL;
        }

        if (PRIVATE (ctxt)->pixmaps_cache) {
                g_hash_table_foreach (PRIVATE (ctxt)->
                                      pixmaps_cache, (GHFunc)
                                      pixmaps_cache_foreach_func,
                                      NULL);
                PRIVATE (ctxt)->pixmaps_cache = NULL;
        }

        if (PRIVATE (ctxt)->file_sel) {
                gtk_widget_destroy (GTK_WIDGET
                                    (PRIVATE (ctxt)->file_sel));
                PRIVATE (ctxt)->file_sel = NULL;
        }

        if (PRIVATE (ctxt)->bitmaps_cache) {
                g_hash_table_foreach (PRIVATE (ctxt)->
                                      bitmaps_cache, (GHFunc)
                                      bitmaps_cache_foreach_func,
                                      NULL);
                PRIVATE (ctxt)->bitmaps_cache = NULL;
        }

        if (PRIVATE (ctxt)->xml_catalog) {
                xmlFreeCatalog (PRIVATE (ctxt)->xml_catalog);
                PRIVATE (ctxt)->xml_catalog = NULL;
        }

        mlview_utils_unref_available_encodings ();

        PRIVATE (ctxt)->dispose_has_run = TRUE;
}


static void
mlview_app_context_finalize (GObject * a_this)
{
        MlViewAppContext *ctxt = MLVIEW_APP_CONTEXT (a_this);

        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (ctxt));

        g_return_if_fail (PRIVATE (ctxt)
                          && PRIVATE (ctxt)->dispose_has_run ==
                          TRUE);

        g_free (PRIVATE (ctxt));
        PRIVATE (ctxt) = NULL;

}
