/* query-join.c
 *
 * Copyright (C) 2002 Vivien Malerba
 * Copyright (C) 2002 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <string.h>
#include "query.h"
#include "marshal.h"

#define T_OR_F(bool) bool ? "t" : "f"

#define bT_OR_F(str) (str[0] == 't') ? TRUE : FALSE

static void query_join_class_init (QueryJoinClass * class);
static void query_join_init (QueryJoin * qj);
static void query_join_dispose (GObject   * object);
static void query_join_finalize (GObject   * object);

GObject   *query_join_new_from_xml (ConfManager *conf, xmlNodePtr node);
xmlNodePtr query_join_save_to_xml (QueryJoin *qj);

struct _QueryJoinPrivate {
	Query         *query;
	QueryJoinType  join_type;
	gboolean       activated;
		
	/* tables or queries for the join (the object is either a DbTable or a Query) */
	QueryView      *ant_view;
	QueryView      *suc_view;

	/* condition in the join */
	QueryCond      *condition;

	/* cardinality of the join */
	QueryJoinCard   card;

	/* referential integrity */
	gboolean        ref_integrity;
	gboolean        cascade_update;
	gboolean        cascade_delete;

	gboolean        skip_view;    /* deal with duplicate suc_views (appear in loops) */	
};


/*
 * static variables 
 */

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass   *parent_class = NULL;

/* get a pointer on the class of QueryJoin for static data access */
static QueryJoinClass *query_join_class = NULL;

enum
{
	NULLIFIED, 
	TYPE_CHANGED,
	CARD_CHANGED,
	COND_CHANGED,
	LAST_SIGNAL
};

static gint query_join_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };


guint
query_join_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryJoinClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_join_class_init,
			NULL,
			NULL,
			sizeof (QueryJoin),
			0,
			(GInstanceInitFunc) query_join_init
		};		

		type = g_type_register_static (G_TYPE_OBJECT, "QueryJoin", &info, 0);
	}
	return type;
}

static void
query_join_class_init (QueryJoinClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);
	query_join_class = class;

	query_join_signals[NULLIFIED] =
		g_signal_new ("nullified",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryJoinClass, nullified),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	query_join_signals[TYPE_CHANGED] =
		g_signal_new ("type_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryJoinClass, type_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	query_join_signals[CARD_CHANGED] =
		g_signal_new ("card_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryJoinClass, card_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	query_join_signals[COND_CHANGED] =
		g_signal_new ("cond_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryJoinClass, cond_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	class->nullified = NULL;
	class->type_changed = NULL;
	class->card_changed = NULL;
	class->cond_changed = NULL;

	object_class->dispose = query_join_dispose;
	object_class->finalize = query_join_finalize;
	/* FIXME: support change of integrity and of condition fields */
}


static void
query_join_init (QueryJoin * qj)
{
	qj->priv = g_new0 (QueryJoinPrivate, 1);
	qj->priv->query = NULL;
	qj->priv->activated = FALSE;
	qj->priv->join_type = QUERY_JOIN_INNER;
	qj->priv->ant_view = NULL;
	qj->priv->suc_view = NULL;
	qj->priv->condition = NULL;
	qj->priv->card = QUERY_JOIN_UNDEFINED;
	qj->priv->skip_view = FALSE;
}


static void view_obj_status_changed_cb (QueryView *qv, QueryJoin *qj);
static void query_view_content_removed_cb (QueryView *qv, GObject *content, QueryJoin *qj);
GObject   *
query_join_new (Query *q,  QueryView *ant_view, QueryView *suc_view)
{
	GObject   *obj;
	QueryJoin *qj;

	/* lots of tests */
	g_return_val_if_fail (q && IS_QUERY (q), NULL);
	g_return_val_if_fail (ant_view && IS_QUERY_VIEW (ant_view), NULL);
	g_return_val_if_fail (suc_view && IS_QUERY_VIEW (suc_view), NULL);
	g_return_val_if_fail (ant_view != suc_view, NULL);

	obj = g_object_new (QUERY_JOIN_TYPE, NULL);
	qj = QUERY_JOIN (obj);

	qj->priv->query = q;
	qj->priv->ant_view = ant_view;
	qj->priv->suc_view = suc_view;

	/* views destruction signals */
	g_signal_connect (G_OBJECT (ant_view), "status_changed",
			  G_CALLBACK (view_obj_status_changed_cb), qj);
	g_signal_connect (G_OBJECT (suc_view), "status_changed",
			  G_CALLBACK (view_obj_status_changed_cb), qj);
	
	g_signal_connect (G_OBJECT (ant_view), "content_removed",
			  G_CALLBACK (query_view_content_removed_cb), qj);
	g_signal_connect (G_OBJECT (suc_view), "content_removed",
			  G_CALLBACK (query_view_content_removed_cb), qj);
	
	qj->priv->activated = TRUE;

	return obj;
}

static void query_join_deactivate (QueryJoin *qj);
static void 
view_obj_status_changed_cb (QueryView *qv, QueryJoin *qj)
{
	if (! query_view_get_is_activated (qv)) {
		query_join_deactivate (qj);
	}
}

static void
query_view_content_removed_cb (QueryView *qv, GObject *content, QueryJoin *qj)
{
	query_join_deactivate (qj);
}

static void cond_nullified_cb (QueryCond *cond, QueryJoin *qj);
static void cond_changed_cb (QueryCond *cond, QueryJoin *qj);

static void 
query_join_deactivate (QueryJoin *qj)
{
	if (! qj->priv->activated)
		return;

	/* condition */
	if (qj->priv->condition) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (qj->priv->condition),
						      G_CALLBACK (cond_nullified_cb), qj);
		g_signal_handlers_disconnect_by_func (G_OBJECT (qj->priv->condition),
						      G_CALLBACK (cond_changed_cb), qj);
		g_object_unref (G_OBJECT (qj->priv->condition));
		qj->priv->condition = NULL;
	}
	
	/* signal handlers */
	g_signal_handlers_disconnect_by_func (G_OBJECT (qj->priv->ant_view),
					      G_CALLBACK (view_obj_status_changed_cb), qj);
	g_signal_handlers_disconnect_by_func (G_OBJECT (qj->priv->suc_view),
					      G_CALLBACK (view_obj_status_changed_cb), qj);
	
	g_signal_handlers_disconnect_by_func (G_OBJECT (qj->priv->ant_view),
					      G_CALLBACK (query_view_content_removed_cb), qj);
	g_signal_handlers_disconnect_by_func (G_OBJECT (qj->priv->suc_view),
					      G_CALLBACK (query_view_content_removed_cb), qj);
	
	qj->priv->ant_view = NULL;
	qj->priv->suc_view = NULL;
	
	qj->priv->query = NULL;
	qj->priv->join_type = QUERY_JOIN_INNER;
	qj->priv->card = QUERY_JOIN_UNDEFINED;
	
	qj->priv->activated = FALSE;

#ifdef debug_signal
	g_print (">> 'NULLIFIED' from query_join_deactivate\n");
#endif
	g_signal_emit (G_OBJECT (qj), query_join_signals[NULLIFIED], 0);
#ifdef debug_signal
	g_print ("<< 'NULLIFIED' from query_join_deactivate\n");
#endif	
	
}

static void 
query_join_dispose (GObject   * object)
{
	QueryJoin *qj;

	g_print ("query_join_dispose (%p)\n", object); /* PRINT */
	g_return_if_fail (object != NULL);
        g_return_if_fail (IS_QUERY_JOIN (object));

	qj = QUERY_JOIN (object);

	if (qj->priv) 
		query_join_deactivate (qj);

	/* parent class */
	parent_class->dispose (object);
}


static void 
query_join_finalize (GObject   * object)
{
	QueryJoin *qj;

	g_return_if_fail (object != NULL);
        g_return_if_fail (IS_QUERY_JOIN (object));

	qj = QUERY_JOIN (object);

	if (qj->priv) {
		g_free (qj->priv);
		qj->priv = NULL;
	}

	parent_class->finalize (object);
}


gchar *
query_join_render_as_sql   (QueryJoin *qj)
{
	return NULL;
}


xmlNodePtr 
query_join_render_as_xml   (QueryJoin *qj)
{
	return NULL;
}
					  

static gchar *char_get_join_type (QueryJoinType jt);
static gchar *char_get_join_card (QueryJoinCard card);

xmlNodePtr 
query_join_save_to_xml     (QueryJoin *qj)
{
	xmlNodePtr node, cond;
	gchar *str;

	node = xmlNewNode (NULL, "QueryJoin");

	str = query_view_get_xml_id (qj->priv->ant_view);
	xmlSetProp (node, "ant_view", str);
	g_free (str);

	str = query_view_get_xml_id (qj->priv->suc_view);
	xmlSetProp (node, "suc_view", str);
	g_free (str);

	str = char_get_join_type (qj->priv->join_type);
	xmlSetProp (node, "join_type", str);

	str = char_get_join_card (qj->priv->card);
	xmlSetProp (node, "join_card", str);

	if (qj->priv->condition) {
		cond = query_cond_save_to_xml (qj->priv->condition);
		xmlAddChild (node, cond);
	}

	xmlSetProp (node, "ref_integrity", T_OR_F(qj->priv->ref_integrity));
	xmlSetProp (node, "cascade_update", T_OR_F(qj->priv->cascade_update));
	xmlSetProp (node, "cascade_delete", T_OR_F(qj->priv->cascade_delete));
	xmlSetProp (node, "skip_view", T_OR_F(qj->priv->skip_view));

	return node;
}

static gchar *
char_get_join_type (QueryJoinType jt)
{
	gchar *str;

	switch (jt) {
	case QUERY_JOIN_INNER:
		str = "INNER_JOIN";
		break;
	case QUERY_JOIN_LEFT_OUTER:
		str = "LEFT_JOIN";
		break;
	case QUERY_JOIN_RIGHT_OUTER:
		str = "RIGHT_JOIN";
		break;
	case QUERY_JOIN_FULL_OUTER:
		str = "FULL_JOIN";
		break;
	case QUERY_JOIN_CROSS:
		str = "CROSS_JOIN";
		break;
	default:
		str = "???";
		break;
	}

	return str;
}

static gchar *
char_get_join_card (QueryJoinCard card)
{
	gchar *str;

	switch (card) {
	case QUERY_JOIN_1_1:
		str = "1_1";
		break;
	case QUERY_JOIN_1_N:
		str = "1_N";
		break;
	case QUERY_JOIN_N_1:
		str = "N_1";
		break;
	case QUERY_JOIN_UNDEFINED:
		str = "UNDEF";
		break;
	default:
		str = "???";
		break;
	}

	return str;
}


static QueryJoinType enum_get_join_type (gchar *str);
static QueryJoinCard enum_get_join_card (gchar *str);
GObject   *
query_join_new_from_xml   (ConfManager *conf, xmlNodePtr node)
{
	QueryJoin *qj = NULL;
	QueryView *ant_view = NULL;
	QueryView *suc_view = NULL;
	gchar *str;

	str = xmlGetProp (node, "ant_view");
	if (str) {
		ant_view = query_view_find_from_xml_name (conf, NULL, str);
		g_free (str);
	}

	str = xmlGetProp (node, "suc_view");
	if (str) {
		suc_view = query_view_find_from_xml_name (conf, NULL, str);
		g_free (str);
	}

	if (ant_view && suc_view) {
		xmlNodePtr tree;
		qj = QUERY_JOIN (query_join_new (query_view_get_query (ant_view), ant_view, suc_view));
		str = xmlGetProp (node, "join_type");
		if (str) {
			query_join_set_join_type (qj, enum_get_join_type (str));
			g_free (str);
		}
		
		tree = node->xmlChildrenNode;

		while (tree) {
			if (!strcmp (tree->name, "QueryCond")) {
				GObject *obj;

				obj = query_cond_new_from_xml (conf, query_view_get_query (ant_view), tree);
				if (obj) {
					QueryCond *cond = QUERY_COND (obj);
					/* FIXME: check that ALL the QueryField objects in this QueryCond
					   refer to either the ant_view or suc_view if they are fields */
					query_join_add_cond (qj, cond);
					g_object_unref (cond);
				}
				else
					g_warning ("Can't load QueryCond!");
			}
			tree = tree->next;
		}

		/* set the cardinality after adding the pairs because adding pairs sets default values
		   which need to be overridden by the XML file */
		str = xmlGetProp (node, "join_card");
		if (str) {
			query_join_set_card (qj, enum_get_join_card (str));
			g_free (str);
		}
		str = xmlGetProp (node, "ref_integrity");
		if (str) {
			qj->priv->ref_integrity = bT_OR_F(str);
			g_free (str);
		}
		str = xmlGetProp (node, "cascade_update");
		if (str) {
			qj->priv->cascade_update = bT_OR_F(str);
			g_free (str);
		}
		str = xmlGetProp (node, "cascade_delete");
		if (str) {
			qj->priv->cascade_delete = bT_OR_F(str);
			g_free (str);
		}
		str = xmlGetProp (node, "skip_view");
		if (str) {
			qj->priv->skip_view = bT_OR_F(str);
			g_free (str);
		}
	}

	if (qj)
		return G_OBJECT (qj);
	else
		return NULL;
}


static 
QueryJoinType enum_get_join_type (gchar *str)
{
	QueryJoinType jt;

	switch (*str) {
	case 'L':
		jt = QUERY_JOIN_LEFT_OUTER;
		break;
	case 'R':
		jt = QUERY_JOIN_RIGHT_OUTER;
		break;
	case 'F':
		jt = QUERY_JOIN_FULL_OUTER;
		break;
	case 'C':
		jt = QUERY_JOIN_CROSS;
		break;
	case 'I':
	default:
		jt = QUERY_JOIN_INNER;
		break;
	}
	
	return jt;
}

static QueryJoinCard 
enum_get_join_card (gchar *str)
{
	QueryJoinCard card;

	switch (*str) {
	case '1':
		if (*(str+2) == '1')
			card = QUERY_JOIN_1_1;
		else
			card = QUERY_JOIN_1_N;
		break;
	case 'N':
		card = QUERY_JOIN_N_1;
		break;
	default:
		card = QUERY_JOIN_UNDEFINED;
		break;
	}
	
	return card;
}


gchar	*
query_join_get_join_type_sql (QueryJoin *qj)
{
	gchar *str;

	switch (qj->priv->join_type) {
	case QUERY_JOIN_INNER:
		str = "INNER JOIN";
		break;
	case QUERY_JOIN_LEFT_OUTER:
		str = "LEFT JOIN";
		break;
	case QUERY_JOIN_RIGHT_OUTER:
		str = "RIGHT JOIN";
		break;
	case QUERY_JOIN_FULL_OUTER:
		str = "FULL JOIN";
		break;
	case QUERY_JOIN_CROSS:
		str = "CROSS JOIN";
		break;
	default:
		str = "???";
		break;
	}

	return str;
}


void
query_join_set_join_type (QueryJoin *qj, QueryJoinType jt)
{
	if (jt != qj->priv->join_type) {
		qj->priv->join_type = jt;

#ifdef debug_signal
		g_print (">> 'TYPE_CHANGED' from query_join_set_type\n");
#endif
		g_signal_emit (G_OBJECT (qj), query_join_signals[TYPE_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'TYPE_CHANGED' from query_join_set_type\n");
#endif	
	}
}

void        
query_join_set_card (QueryJoin *qj, QueryJoinCard card) 
{
	if (card != qj->priv->card) {
		qj->priv->card = card;

#ifdef debug_signal
		g_print (">> 'CARD_CHANGED' from query_join_set_card\n");
#endif
		g_signal_emit (G_OBJECT (qj), query_join_signals[CARD_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'CARD_CHANGED' from query_join_set_card\n");
#endif	
	}
}


/* Try to set the default to the best query join cardinality as possible */
static QueryJoinCard
compute_best_appropriate_card (QueryJoin *qj)
{
	QueryJoinCard card = QUERY_JOIN_UNDEFINED;
	GSList *list, *fields;
	gboolean ant_key = TRUE;
	gboolean suc_key = TRUE;


	/* if no condition then we don't try anything */
	if (! qj->priv->condition)
		return card;

	/* see if we have a key on the ant_field and suc_field */
	list = query_cond_get_query_fields (qj->priv->condition);
	fields = list;
	while (list) {
		QueryField *qf = QUERY_FIELD (list->data);
		if ((query_field_get_ftype (qf) == QUERY_FIELD_FIELD) &&
		    query_field_field_is_db_field (qf)) {
			DbField *field = query_field_field_get_db_field (qf);
			
			if (! field->is_key) {
				if (query_field_field_get_queryview (qf) == qj->priv->ant_view)
					ant_key = FALSE;
				if (query_field_field_get_queryview (qf) == qj->priv->suc_view)
					suc_key = FALSE;
			}
		}
		
		list = g_slist_next (list);
	}
	g_slist_free (fields);

	if (ant_key) {
		if (suc_key)
			card = QUERY_JOIN_1_1;
		else
			card = QUERY_JOIN_1_N;
	}
	else {
		if (suc_key)
			card = QUERY_JOIN_N_1;
		else
			card = QUERY_JOIN_UNDEFINED;
	}

	g_print ("STEP1: card=%d\n", card);

	/* if we have undefined, see if we can have something better with
	   not NULL values instead of keys (being less exigent) */
	if (card == QUERY_JOIN_UNDEFINED) {
		ant_key = TRUE;
		suc_key = TRUE;
		
		/* see if we have a non NULL value on the ant_field and suc_field */
		list = query_cond_get_query_fields (qj->priv->condition);
		fields = list;
		while (list) {
			QueryField *qf = QUERY_FIELD (list->data);
			if ((query_field_get_ftype (qf) == QUERY_FIELD_FIELD) &&
			    query_field_field_is_db_field (qf)) {
				DbField *field = query_field_field_get_db_field (qf);
				
				if (field->null_allowed) {
					if (query_field_field_get_queryview (qf) == qj->priv->ant_view)
						ant_key = FALSE;
					if (query_field_field_get_queryview (qf) == qj->priv->suc_view)
						suc_key = FALSE;
				}
			}
			
			list = g_slist_next (list);
		}
		g_slist_free (fields);
		
		if (ant_key) {
			if (suc_key)
				card = QUERY_JOIN_1_1;
			else
				card = QUERY_JOIN_1_N;
		}
		else {
			if (suc_key)
				card = QUERY_JOIN_N_1;
			else
				card = QUERY_JOIN_UNDEFINED;
		}
	}

	return card;
}

void
query_join_add_cond_fields (QueryJoin *qj, QueryField *field1, QueryField *field2)
{
	QueryCond *cond;
	gboolean error = FALSE;

	g_return_if_fail (qj && IS_QUERY_JOIN (qj));
	g_return_if_fail (field1 && IS_QUERY_FIELD (field1));
	g_return_if_fail (field2 && IS_QUERY_FIELD (field2));

	/* first checks and return on error */
	if (query_field_get_ftype (field1) == QUERY_FIELD_FIELD) {
		QueryView *qv = query_field_field_get_queryview (field1);
		if ((qv != qj->priv->ant_view) && (qv != qj->priv->suc_view)) {
			error = TRUE;
			g_warning ("query_join_add_cond_fields: QueryView for field1 is not in join's views\n");
		}
	}

	if (query_field_get_ftype (field2) == QUERY_FIELD_FIELD) {
		QueryView *qv = query_field_field_get_queryview (field2);
		if ((qv != qj->priv->ant_view) && (qv != qj->priv->suc_view)) {
			error = TRUE;
			g_warning ("query_join_add_cond_fields: QueryView for field2 is not in join's views\n");
		}
	}

	if (error)
		return;

	/* create new QueryCond */
	cond = QUERY_COND (query_cond_new_as_leaf (qj->priv->query, QUERY_COND_OP_EQUAL, field1, field2, NULL));
	query_join_add_cond (qj, cond);
	g_object_unref (G_OBJECT (cond));
}

void
query_join_add_cond (QueryJoin *qj, QueryCond *cond)
{
	gboolean error = FALSE;
	GSList *list, *fields;

	g_return_if_fail (qj && IS_QUERY_JOIN (qj));
	g_return_if_fail (cond && IS_QUERY_COND (cond));

	/* checks on the QueryCond */
	list = query_cond_get_query_fields (cond);
	fields = list;
	while (list && !error) {
		QueryField *qf = QUERY_FIELD (list->data);
		if (query_field_get_ftype (qf) == QUERY_FIELD_FIELD) {
			QueryView *qv = query_field_field_get_queryview (qf);
			if ((qv != qj->priv->ant_view) && (qv != qj->priv->suc_view)) {
				error = TRUE;
				g_warning ("query_join_add_cond: QueryView for QF in QCond is not in join's views\n");
			}
		}
		list = g_slist_next (list);
	}
	g_slist_free (fields);

	if (error)
		return;
	else { /* QueryCond is OK, continue */
		if (qj->priv->condition) {
			if (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_AND) {
				/* we already have a AND node here */
				query_cond_add_to_node (qj->priv->condition, cond, -1);
			}
			else {
				QueryCond *oldhead = qj->priv->condition;
				QueryCond *and = QUERY_COND (query_cond_new_as_node (qj->priv->query, QUERY_COND_AND));

				g_signal_handlers_disconnect_by_func (G_OBJECT (oldhead),
								      G_CALLBACK (cond_nullified_cb), qj);
				g_signal_handlers_disconnect_by_func (G_OBJECT (oldhead),
								      G_CALLBACK (cond_changed_cb), qj);

				qj->priv->condition = and;
				g_object_ref (G_OBJECT (and));
				g_signal_connect (G_OBJECT (and), "nullified",
						  G_CALLBACK (cond_nullified_cb), qj);
				g_signal_connect (G_OBJECT (and), "changed",
						  G_CALLBACK (cond_changed_cb), qj);				

				query_cond_add_to_node (and, oldhead, -1);
				g_object_unref (G_OBJECT (oldhead));
				query_cond_add_to_node (and, cond, -1);
			}
		}
		else {
			qj->priv->condition = cond;

			g_object_ref (G_OBJECT (cond));
			g_signal_connect (G_OBJECT (cond), "nullified",
					  G_CALLBACK (cond_nullified_cb), qj);
			g_signal_connect (G_OBJECT (cond), "changed",
					  G_CALLBACK (cond_changed_cb), qj);
#ifdef debug_signal
			g_print (">> 'COND_CHANGED' from query_join_add_cond\n");
#endif
			g_signal_emit (G_OBJECT (qj), query_join_signals[COND_CHANGED], 0);
#ifdef debug_signal
			g_print (">> 'COND_CHANGED' from query_join_add_cond\n");
#endif	
		}

		/* try to guess the best cardinality as possible */
		query_join_set_card (qj, compute_best_appropriate_card (qj));
	}
}

static void 
cond_nullified_cb (QueryCond *cond, QueryJoin *qj)
{
	query_join_del_cond (qj, cond);
}

static void 
cond_changed_cb (QueryCond *cond, QueryJoin *qj)
{
#ifdef debug_signal
	g_print (">> 'COND_CHANGED' from cond_changed_cb\n");
#endif
	g_signal_emit (G_OBJECT (qj), query_join_signals[COND_CHANGED], 0);
#ifdef debug_signal
	g_print (">> 'COND_CHANGED' from cond_changed_cb\n");
#endif	
}

void 
query_join_del_cond (QueryJoin *qj, QueryCond *cond)
{
	QueryCond *parent;

	g_return_if_fail (IS_QUERY_COND (cond));

	if (cond == qj->priv->condition) {
		/* disconnect signal handler */
		g_signal_handlers_disconnect_by_func (G_OBJECT (cond),
						      G_CALLBACK (cond_nullified_cb), qj);
		g_signal_handlers_disconnect_by_func (G_OBJECT (qj->priv->condition),
						      G_CALLBACK (cond_changed_cb), qj);

		g_object_unref (G_OBJECT (qj->priv->condition));

		qj->priv->condition = NULL;
#ifdef debug_signal
		g_print (">> 'COND_CHANGED' from query_join_add_cond\n");
#endif
		g_signal_emit (G_OBJECT (qj), query_join_signals[COND_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'COND_CHANGED' from query_join_add_cond\n");
#endif
	}
	else {
		parent = query_cond_get_parent (cond);
		g_return_if_fail (parent);
		query_cond_del_from_node (parent, cond);

		/* Clean the qj->priv->condition if necessary */
		if ((query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_AND) ||
		    (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_OR)) {
			if (g_slist_length (query_cond_get_children (qj->priv->condition)) == 0) 
				query_join_del_cond (qj, qj->priv->condition);
			else {
				if (g_slist_length (query_cond_get_children (qj->priv->condition)) == 1) {
					QueryCond *newhead = QUERY_COND (query_cond_get_children (qj->priv->condition)->data);
					g_object_ref (G_OBJECT (newhead));
					query_cond_del_from_node (qj->priv->condition, newhead);
					query_join_del_cond (qj, qj->priv->condition);

					qj->priv->condition = newhead;
					g_signal_connect (G_OBJECT (newhead), "nullified",
							  G_CALLBACK (cond_nullified_cb), qj);
					g_signal_connect (G_OBJECT (newhead), "changed",
							  G_CALLBACK (cond_changed_cb), qj);				

#ifdef debug_signal
					g_print (">> 'COND_CHANGED' from query_join_add_cond\n");
#endif
					g_signal_emit (G_OBJECT (qj), query_join_signals[COND_CHANGED], 0);
#ifdef debug_signal
					g_print ("<< 'COND_CHANGED' from query_join_add_cond\n");
#endif
				}
			}
		}
	}
}


QueryJoin *query_join_copy (QueryJoin *qj)
{
	/* FER FIXME - deep copy: needed to copy global relationships to
	 * a query when a new table is added */

	g_warning ("query_join_copy () not yet implemented, will return NULL");
	return NULL;
}


/* query_join_swap_views 
 *
 * Swaps the left view with the right view (it also swaps the field pairs
 * and updates the cardinality accordingly (1-n changes to n-1 and vice-versa)
 *
 * See also the dfinition of a Join list in query_add_join()
 * 
 * It should be used carefully to avoid inadvertent introduction of
 * inconsistencies. Like the following example where ij means INNER JOIN
 * and the conditions over the fiels is omited for the sake of simplicity:
 * 
 * The query FROM (A ij B) ij C is represented as (A,B) (B,C). A single
 * swap of (B,C) to (C,B) would give the sequence (A,B) (C,B) which will
 * not be properly translated to the intended, original, query. We would
 * obtain FROM (A ij B) ij B
 *
 */

void query_join_swap_views (QueryJoin *qj)
{
	QueryView *tmp;

	/* swap views */
	tmp = qj->priv->suc_view;
	qj->priv->suc_view = qj->priv->ant_view;
	qj->priv->ant_view = tmp;
	
	/* reverse relation cardinality */
	if (qj->priv->card == QUERY_JOIN_1_N)
		qj->priv->card = QUERY_JOIN_N_1;
	else {
		if (qj->priv->card == QUERY_JOIN_N_1)
			qj->priv->card = QUERY_JOIN_1_N;
	}
}


QueryCond *
query_join_get_condition (QueryJoin *qj)
{
	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), NULL);

	return qj->priv->condition;
}

#ifdef debug
void query_join_print (QueryJoin *qj)
{
	if (qj)
		g_return_if_fail (IS_QUERY_JOIN (qj));

	if (!qj) 
		g_print("QJ NULL\n");
	else {
		if (qj->priv->condition) {
			gchar *str;
			str = query_cond_render_as_sql (qj->priv->condition, NULL);
			g_print("(%s, %s) ON (%s)", query_view_get_name(qj->priv->ant_view),
				query_view_get_name(qj->priv->suc_view), str);
			g_free (str);
		}
		else {
			g_print("(%s, %s) ON (NULL)", query_view_get_name(qj->priv->ant_view),
				query_view_get_name(qj->priv->suc_view));
		}
	}
}
#endif

void
query_join_set_skip_view (QueryJoin *qj)
{
	g_return_if_fail (qj && IS_QUERY_JOIN (qj));

	qj->priv->skip_view = TRUE;
}

void
query_join_unset_skip_view (QueryJoin *qj)
{
	g_return_if_fail (qj && IS_QUERY_JOIN (qj));

	qj->priv->skip_view = TRUE;
}

gboolean
query_join_skip_view (QueryJoin *qj)
{
	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), FALSE);

	return qj->priv->skip_view;
}

QueryView *
query_join_get_ant_view (QueryJoin *qj)
{
	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), NULL);

	return qj->priv->ant_view;
}

QueryView *
query_join_get_suc_view (QueryJoin *qj)
{
	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), NULL);

	return qj->priv->suc_view;
}

Query *
query_join_get_query (QueryJoin *qj)
{
	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), NULL);

	return qj->priv->query;
}

QueryJoinType 
query_join_get_join_type (QueryJoin *qj)
{
	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), QUERY_JOIN_INNER);

	return qj->priv->join_type;
}

QueryJoinCard
query_join_get_card (QueryJoin *qj)
{
	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), QUERY_JOIN_UNDEFINED);

	return qj->priv->card;
}

guint 
query_join_cond_get_nb_conditions (QueryJoin *qj)
{
	guint nb_cond = 0;

	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), 0);	

	if (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_COND) 
		nb_cond = 1;
	else {
		if (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_AND) 
			nb_cond = g_slist_length (query_cond_get_children (qj->priv->condition));
		else
			nb_cond = 1;
	}

	return nb_cond;	
}

static GObject *cond_get_object_ref (QueryCond *cond, gboolean left_op);
guint
query_join_cond_get_nb_equi_pairs (QueryJoin *qj)
{
	guint nb_equi = 0;

	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), 0);	

	if (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_COND) {
		GObject *left_op = NULL, *right_op = NULL;
		
		/* getting left and right operators */
		left_op = cond_get_object_ref (qj->priv->condition, TRUE);
		right_op = cond_get_object_ref (qj->priv->condition, FALSE);

		if (left_op && right_op)
			nb_equi = 1;
	}
	else {
		if (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_AND) {
			GSList *list;

			list = query_cond_get_children (qj->priv->condition);
			
			/* we want to accept only EQUI conditions here */
			while (list) {
				if (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_COND) {
					GObject *left_op, *right_op;
					
					left_op = cond_get_object_ref (qj->priv->condition, TRUE);
					right_op = cond_get_object_ref (qj->priv->condition, FALSE);
					if (left_op && right_op)
						nb_equi++;
				}

				list = g_slist_next (list);
			}
		}
	}

	return nb_equi;
}

gboolean
query_join_cond_is_equi_only (QueryJoin *qj)
{
	gboolean equi_only = FALSE;

	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), FALSE);	

	if (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_COND) {
		GObject *left_op = NULL, *right_op = NULL;
		
		/* getting left and right operators */
		left_op = cond_get_object_ref (qj->priv->condition, TRUE);
		right_op = cond_get_object_ref (qj->priv->condition, FALSE);

		if (left_op && right_op)
			equi_only = TRUE;
	}
	else {
		if (query_cond_get_cond_type (qj->priv->condition) == QUERY_COND_AND) {
			GSList *list;
			gboolean equi_joins_only = TRUE;

			list = query_cond_get_children (qj->priv->condition);
			
			/* we want to accept only EQUI conditions here */
			while (list && equi_joins_only) {
				if (query_cond_get_cond_type (qj->priv->condition) != QUERY_COND_COND)
					equi_joins_only = FALSE;
				else {
					GObject *left_op, *right_op;
					
					left_op = cond_get_object_ref (qj->priv->condition, TRUE);
					right_op = cond_get_object_ref (qj->priv->condition, FALSE);
					if (!left_op || !right_op)
						equi_joins_only = FALSE;
				}

				list = g_slist_next (list);
			}

			if (equi_joins_only)
				equi_only = TRUE;;
		}
	}

	return equi_only;
}

/* returns the DbField or QueryField from a QueryCond if it is and EQUAL condition */
static GObject *
cond_get_object_ref (QueryCond *cond, gboolean left_op)
{
	GObject *obj = NULL;

	if ((query_cond_get_cond_type (cond) == QUERY_COND_COND) && 
	    (query_cond_get_op_type (cond) == QUERY_COND_OP_EQUAL)) {
		QueryField *qf;
		
		if (left_op)
			qf = query_cond_get_op_left (cond);
		else
			qf = query_cond_get_op_right (cond);
		
		if (query_field_get_ftype (qf) == QUERY_FIELD_FIELD) {
			if (query_field_field_is_db_field (qf))
				obj = G_OBJECT (query_field_field_get_db_field (qf));
			else
				obj = G_OBJECT (query_field_field_get_query_field (qf));
		}
	}

	return obj;
}


GSList *real_query_join_cond_get_equi_pairs (QueryJoin *qj, QueryCond *cond, GSList *head);
/* 
 * To get the pairs of fields (DbField or QueryField) which are part of an equi condition
 * 
 * Returns a list of QueryJoinPair, to free after usage (the list + the QueryJoinPairs) 
 */
GSList *
query_join_cond_get_equi_pairs (QueryJoin *qj)
{
	g_return_val_if_fail (qj && IS_QUERY_JOIN (qj), NULL);

	return real_query_join_cond_get_equi_pairs (qj, qj->priv->condition, NULL);
}

GSList *
real_query_join_cond_get_equi_pairs (QueryJoin *qj, QueryCond *cond, GSList *head)
{
	GSList *retlist = head;

	if (cond) {
		if ((query_cond_get_cond_type (cond) == QUERY_COND_COND) && 
		    (query_cond_get_op_type (cond) == QUERY_COND_OP_EQUAL)) {
			GObject *field1, *field2;

			field1 = cond_get_object_ref (cond, TRUE);
			field2 = cond_get_object_ref (cond, FALSE);
			
			if (field1 && field2) {
				QueryJoinPair *pair = g_new0 (QueryJoinPair, 1);

				pair->field1 = field1;
				pair->field2 = field2;

				retlist = g_slist_append (retlist, pair);
			}
		}
		
		if (query_cond_get_cond_type (cond) != QUERY_COND_COND) {
			GSList *list;

			list = query_cond_get_children (cond);
			
			while (list) {
				retlist = real_query_join_cond_get_equi_pairs (qj, QUERY_COND (list->data),
									       retlist);
				list = g_slist_next (list);
			}
		}
	}

	return retlist;
}
