/* value.c
 *
 * Copyright (C) 2001 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * This module represents QueryField of type QUERY_FIELD_FIELD:
 * a field of a QueryView (DbTable (table or view) or Query).
 */

#include <config.h>
#include "../query.h"
#include "../database.h"
#include "../query-field-private.h"
#include "../server-access.h"

static void        q_init            (QueryField *qf);
static void        q_finalize        (QueryField *qf);
static void        q_deactivate      (QueryField *qf);
static void        q_activate        (QueryField *qf);
static GtkWidget * q_get_edit_widget (QueryField *qf);
static GtkWidget * q_get_sel_widget  (QueryField *qf, GCallback callback, gpointer data);
static gchar     * q_render_as_sql   (QueryField *qf, GSList * missing_values);
static xmlNodePtr  q_render_as_xml   (QueryField *qf, GSList * missing_values);
static gchar     * q_render_as_string(QueryField *qf, GSList * missing_values);
static void        q_save_to_xml     (QueryField *qf, xmlNodePtr node);
static void        q_load_from_xml   (QueryField *qf, xmlNodePtr node);
static void        q_copy_other_field(QueryField *qf, QueryField *other);
static gboolean    q_is_equal_to     (QueryField *qf, QueryField *other);
static GSList    * q_get_monitored_objects (QueryField *qf);
static void        q_replace_comp    (QueryField *qf, gint ref, GObject   *old, GObject   *new);

/* Weak ref from the refering object */
static void        q_obj_destroyed_cb (QueryField *qf, GObject *obj);

typedef struct {
	/* the type of value */
	ServerDataType *value_type;
	gboolean        is_dt_destroyed; /* TRUE if the weak ref on the data type is no more */

	/* default value if any */
	GdaValue       *value;

	/* still ask for a value */
	gboolean        ask_value;

	/* XML field name if we did not find it in the first place */
	gchar          *type_name;
} private_data;

#define QF_PRIVATE_DATA(qf) ((private_data *) qf->private_data)

QueryFieldIface * 
query_field_value_get_iface()
{
	QueryFieldIface *iface;

	iface = g_new0 (QueryFieldIface, 1);
	iface->field_type = QUERY_FIELD_FIELD;
	iface->name = "value";
	iface->pretty_name = _("Constant value (may be modified at run time)");
	iface->init = q_init;
	iface->destroy = q_finalize;
	iface->deactivate = q_deactivate;
	iface->activate = q_activate;
	iface->get_edit_widget = q_get_edit_widget;
	iface->get_sel_widget = q_get_sel_widget;
	iface->render_as_sql = q_render_as_sql;
	iface->render_as_xml = q_render_as_xml;
	iface->render_as_string = q_render_as_string;
	iface->save_to_xml = q_save_to_xml;
	iface->load_from_xml = q_load_from_xml;
	iface->copy_other_field = q_copy_other_field;
	iface->is_equal_to = q_is_equal_to;
	iface->get_monitored_objects = q_get_monitored_objects;
	iface->replace_comp = q_replace_comp;

	return iface;
}

static void        
q_init            (QueryField *qf)
{
	private_data *data;
	data = g_new0 (private_data, 1);
	qf->private_data = (gpointer) data;
	QF_PRIVATE_DATA(qf)->value_type = NULL;
	QF_PRIVATE_DATA(qf)->type_name = NULL;
	QF_PRIVATE_DATA(qf)->value = NULL;
	QF_PRIVATE_DATA(qf)->is_dt_destroyed = FALSE;
	QF_PRIVATE_DATA(qf)->ask_value = FALSE;
}

static void        
q_finalize         (QueryField *qf)
{
	query_field_deactivate (qf);
	if (qf->private_data) {
		if (QF_PRIVATE_DATA(qf)->type_name)
			g_free (QF_PRIVATE_DATA(qf)->type_name);
		g_free (qf->private_data);
		qf->private_data = NULL;
	}
}


static gchar * qf_get_xml_id (QueryField *qf);
static void
q_deactivate      (QueryField *qf)
{
	if (! qf->activated)
		return;

	/* This function disconnects any event handler from any object
	   this QueryField wants to receive events from.
	   Here we disconnect from the data type if we are connected */

	if (! QF_PRIVATE_DATA(qf)->is_dt_destroyed) {
		g_object_weak_unref (G_OBJECT (QF_PRIVATE_DATA(qf)->value_type),
				     (GWeakNotify) q_obj_destroyed_cb, qf);
	}

	if (QF_PRIVATE_DATA(qf)->type_name) {
		g_free (QF_PRIVATE_DATA(qf)->type_name);
		QF_PRIVATE_DATA(qf)->type_name = NULL;
	}

	QF_PRIVATE_DATA(qf)->type_name = qf_get_xml_id (qf);
	QF_PRIVATE_DATA(qf)->value_type = NULL;
	
	query_field_set_activated (qf, FALSE);
}

static gchar *
qf_get_xml_id (QueryField *qf)
{
	gchar *str = NULL;

	if (QF_PRIVATE_DATA(qf)->value_type) 
		str = server_data_type_get_xml_id (QF_PRIVATE_DATA(qf)->value_type);

	return str;
}

static void
q_activate        (QueryField *qf)
{
	/* this function gets references to any object this QueryField wants to 
	   receive events from. */

	ServerDataType *value_type;
	
	if (qf->activated)
		return;

	value_type = QF_PRIVATE_DATA(qf)->value_type;
	
	if (!value_type && QF_PRIVATE_DATA(qf)->type_name) 
		value_type = server_data_type_get_from_xml_id (qf->query->conf->srv, 
							       QF_PRIVATE_DATA(qf)->type_name);


	if (value_type) {
		QF_PRIVATE_DATA(qf)->value_type = value_type;

		g_object_weak_ref (G_OBJECT (QF_PRIVATE_DATA(qf)->value_type),
				   (GWeakNotify) q_obj_destroyed_cb, qf);

		/* now it's activated */ 
		query_field_set_activated (qf, TRUE);
	}
}

static void        
q_obj_destroyed_cb (QueryField *qf, GObject *obj)
{
	QF_PRIVATE_DATA(qf)->is_dt_destroyed = TRUE;

	q_deactivate (qf);
	/* if the data type disappear, then do nothing, just remain non activated */
	/*g_object_unref (G_OBJECT (qf));*/
}


static GtkWidget * 
q_get_edit_widget (QueryField *qf)
{
	/* FIXME */
	GtkWidget *wid, *frame, *vb;

	frame = gtk_frame_new (_("Constant value"));
	gtk_container_set_border_width (GTK_CONTAINER (frame), GNOME_PAD/2.);

	vb = gtk_vbox_new (FALSE, GNOME_PAD/2.);
	gtk_container_add (GTK_CONTAINER (frame), vb);
	gtk_container_set_border_width (GTK_CONTAINER (vb), GNOME_PAD/2.);

	wid = gtk_label_new ("FIXME");
	gtk_widget_set_usize (GTK_WIDGET (wid), 200, 100);

	gtk_box_pack_start (GTK_BOX (vb), wid, TRUE, TRUE, GNOME_PAD/2.);
	gtk_widget_show_all (vb);

	return frame;
}

static GtkWidget * 
q_get_sel_widget  (QueryField *qf, GCallback callback, gpointer data)
{
	GtkWidget *button;
	gchar *str;

	if (qf->activated) {
		/* FIXME */
		str = g_strdup (_("Constant"));
	}
	else
		str = g_strdup (_("Constant"));

	button = gtk_button_new_with_label (str);	
	g_free (str);
	g_signal_connect (G_OBJECT (button), "clicked", callback, data);
	g_object_set_data (G_OBJECT (button), "qf", qf);

	/* Set the "QF_obj_emit_sig" attribute so that we can attach attributes to that button
	   which will be transmitted when the user clicks on it */
	g_object_set_data (G_OBJECT (button), "QF_obj_emit_sig", button);

	return button;
}



static gchar * 
q_render_as_sql   (QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;


	return str;
}

static xmlNodePtr  
q_render_as_xml   (QueryField *qf, GSList * missing_values)
{
	return NULL;
}

static gchar * 
q_render_as_string(QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;
	
	str = q_render_as_sql (qf, missing_values);
	if (!str) 
		str = g_strdup ("A constant");

	return str;
}

static void  
q_save_to_xml     (QueryField *qf, xmlNodePtr node)
{
	if (qf->activated) {
		gchar *str;
	
		/* node object ref */
		str = server_data_type_get_xml_id (QF_PRIVATE_DATA(qf)->value_type);
		xmlSetProp (node, "object", str);
		g_free (str);
		
		/* FIXME: save the default value if any, using a CDATA, we don't want any strings here
		 because that can be a problem with some binary data */
		if (QF_PRIVATE_DATA(qf)->value) {
			xmlSetProp (node, "value", "TBD");
		}

		xmlSetProp (node, "type", "value");
	}
	else
		g_warning ("QueryField not activated; can't save\n");
}

static void        
q_load_from_xml   (QueryField *qf, xmlNodePtr node)
{
	query_field_deactivate (qf);
	
	/* check we have a QueryField */
	if (!strcmp (node->name, "QueryField")) {
		gchar *str;

		str = xmlGetProp (node, "type");
		if (!str || (str && strcmp (str, "value"))) {
			if (str) g_free (str);
			return;
		}

		str = xmlGetProp (node, "object");
		if (str) {
			if (QF_PRIVATE_DATA(qf)->type_name)
				g_free (QF_PRIVATE_DATA(qf)->type_name);
			QF_PRIVATE_DATA(qf)->type_name = str;
		}
		query_field_activate (qf);
	}
}

static void        
q_copy_other_field(QueryField *qf, QueryField *other)
{
	/* we can't call q_destroy(qf) because we don't know what the type
	   of QueryField it was before. This is normally done by the
	   QueryField object before the copy */

	if (QF_PRIVATE_DATA(other)->value_type) {
		QF_PRIVATE_DATA(qf)->value_type = QF_PRIVATE_DATA(other)->value_type;
		query_field_activate (qf);
	}
	else {
		if (QF_PRIVATE_DATA(other)->type_name) {
			QF_PRIVATE_DATA(qf)->type_name = g_strdup (QF_PRIVATE_DATA(other)->type_name);
			query_field_activate (qf);
		}
	}
}

static gboolean
q_is_equal_to (QueryField *qf, QueryField *other)
{
	gboolean retval = FALSE;

	if (qf->activated && other->activated) {
		if (QF_PRIVATE_DATA(qf)->value_type == QF_PRIVATE_DATA(other)->value_type)
			retval = TRUE;
	}

	return retval;
}


static GSList *
q_get_monitored_objects (QueryField *qf)
{
	GSList *list = NULL;

	if (qf->activated) 
		list = g_slist_prepend (NULL, QF_PRIVATE_DATA(qf)->value_type);

	return list;
}

static void
q_replace_comp (QueryField *qf, gint ref, GObject   *old, GObject   *new)
{
	/* no reference to other QueryFields */
	return;
}


/* 
 * 
 * QueryField object's different implementations
 * 
 *
 */


ServerDataType  * 
query_field_value_get_data_type (QueryField *qf)
{
	g_assert (qf && IS_QUERY_FIELD (qf));
	g_assert (qf->field_type == QUERY_FIELD_VALUE);

	if (! qf->activated)
		return NULL;
	else 
		return QF_PRIVATE_DATA(qf)->value_type;
}

gboolean 
query_field_value_input_required (QueryField *qf)
{
	return QF_PRIVATE_DATA(qf)->ask_value;
}

const GdaValue *
query_field_value_get_default_value (QueryField *qf)
{
	return QF_PRIVATE_DATA(qf)->value;
}
