/* query-view.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include "query.h"
#include "marshal.h"

static void query_cond_class_init (QueryCondClass * class);
static void query_cond_init       (QueryCond * qc);
static void query_cond_dispose    (GObject   * object);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass   *parent_class = NULL;

enum
{
	CHANGED,
	LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = { 0 };


guint
query_cond_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryCondClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_cond_class_init,
			NULL,
			NULL,
			sizeof (QueryCond),
			0,
			(GInstanceInitFunc) query_cond_init
		};		

		type = g_type_register_static (G_TYPE_OBJECT, "QueryCond", &info, 0);
	}
	return type;
}

static void
query_cond_class_init (QueryCondClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	signals[CHANGED] =
		g_signal_new ("changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryCondClass, changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	class->changed = NULL;
	object_class->dispose = query_cond_dispose;
}


static void
query_cond_init (QueryCond * qc)
{
	qc->query = NULL;
	qc->cond_type = QUERY_COND_AND;
	qc->children = NULL;
	qc->op_type = QUERY_COND_OP_EQUAL;
	qc->left_op = NULL;
	qc->right_op = NULL;
	qc->right_op2 = NULL;
}



GObject *
query_cond_new_as_node (Query *q, QueryCondType type) 
{
	GObject   *obj;
	QueryCond *qc;

	g_return_val_if_fail (q && IS_QUERY (q), NULL);
	g_return_val_if_fail (type == QUERY_COND_COND, NULL);

	obj = g_object_new (QUERY_COND_TYPE, NULL);
	qc = QUERY_COND (obj);

	qc->query = q;
	qc->cond_type = type;

	return obj;
}


static void query_field_destroy_notify (QueryCond *qc, QueryField *qf);
GObject *
query_cond_new_as_leaf      (Query *q, QueryCondOpType type, 
			     QueryField *left_op, QueryField *right_op, QueryField *right_op2)
{
GObject   *obj;
	QueryCond *qc;

	g_return_val_if_fail (q && IS_QUERY (q), NULL);

	obj = g_object_new (QUERY_COND_TYPE, NULL);
	qc = QUERY_COND (obj);

	qc->query = q;
	qc->cond_type = QUERY_COND_COND;
	qc->op_type = type;

	if (IS_QUERY_FIELD (left_op) && (QUERY_FIELD (left_op)->query == q)) {
		qc->left_op = left_op;
		g_object_weak_ref (G_OBJECT (left_op), (GWeakNotify) query_field_destroy_notify, qc);
	}
	if (IS_QUERY_FIELD (right_op) && (QUERY_FIELD (right_op)->query == q)) {
		qc->right_op = right_op;
		g_object_weak_ref (G_OBJECT (right_op), (GWeakNotify) query_field_destroy_notify, qc);
	}

	switch (type) {
	case QUERY_COND_OP_IN:
	case QUERY_COND_OP_BETWEEN:
		if (IS_QUERY_FIELD (right_op2) && (QUERY_FIELD (right_op2)->query == q)) {
			qc->right_op2 = right_op2;
			g_object_weak_ref (G_OBJECT (right_op2), (GWeakNotify) query_field_destroy_notify, qc);
		}
		break;
	default:
		break;
	}

	return obj;	
}


static void 
query_field_destroy_notify (QueryCond *qc, QueryField *qf)
{
	if (qc->left_op == qf)
		qc->left_op = NULL;
	if (qc->right_op == qf)
		qc->right_op = NULL;
	if (qc->right_op2 == qf)
		qc->right_op2 = NULL;
	query_cond_free (qc);
}

GObject   *
query_cond_new_copy (QueryCond *qc)
{
	QueryCond *nqc;

	g_return_val_if_fail (qc && IS_QUERY_COND (qc), NULL);
	if (qc->cond_type == QUERY_COND_COND) 
		nqc = QUERY_COND (query_cond_new_as_leaf (qc->query, qc->op_type, 
							  qc->left_op, qc->right_op, qc->right_op2));
	else {
		GSList *list;

		nqc = QUERY_COND (query_cond_new_as_node (qc->query, qc->cond_type));
		list = qc->children;
		while (list) {
			QueryCond *nchild;

			nchild = QUERY_COND (query_cond_new_copy (QUERY_COND (list->data)));
			query_cond_add_to_node (nqc, nchild, -1);
			list = g_slist_next (list);
		}
	}
	
	return G_OBJECT (nqc);
}

void
query_cond_free (QueryCond *qc)
{
	g_return_if_fail (qc && IS_QUERY_COND (qc));

	g_object_unref (G_OBJECT (qc));
}


static void query_cond_reinit (GObject   * object);

static void
query_cond_dispose (GObject   * object)
{
	g_return_if_fail (object && IS_QUERY_COND (object));
	
	query_cond_reinit (object);
	/* for the parent class */
	parent_class->dispose (object);
}

static void child_changed_cb (QueryCond *child, QueryCond *qc);
static void child_node_destroy_notify (QueryCond *qc, QueryCond *child); /* GWeakNotify */
static void 
query_cond_reinit (GObject   * object)
{
	QueryCond *qc;
	GSList *list;

	g_return_if_fail (object && IS_QUERY_COND (object));


	qc = QUERY_COND (object);

	/* operators */
	if (qc->left_op) {
		g_object_weak_unref (G_OBJECT (qc->left_op), (GWeakNotify) query_field_destroy_notify, qc);
		qc->left_op = NULL;
	}

	if (qc->right_op) {
		g_object_weak_unref (G_OBJECT (qc->right_op), (GWeakNotify) query_field_destroy_notify, qc);
		qc->right_op = NULL;
	}

	if (qc->right_op2) {
		g_object_weak_unref (G_OBJECT (qc->right_op2), (GWeakNotify) query_field_destroy_notify, qc);
		qc->right_op2 = NULL;
	}				     

	/* children if any */
	list = qc->children;
	while (list) {
		g_object_weak_unref (G_OBJECT (list->data), (GWeakNotify) child_node_destroy_notify, qc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), G_CALLBACK (child_changed_cb), qc);
		query_cond_free (QUERY_COND (list->data));

		list = g_slist_next (list);
	}

	if (qc->children) {
		g_slist_free (qc->children);
		qc->children = NULL;
	}

	query_cond_init (qc);
}

/* If qc node is a in fact a leaf type (=> QUERY_COND_COND) then we substitute the leaf
   with a new AND node and we add the old leaf to that new ADD node 
   pos == -1 => the new QueryCond must be appended to the other ones */
void
query_cond_add_to_node (QueryCond *qc, QueryCond *child, gint pos)
{
	g_return_if_fail (qc && IS_QUERY_COND (qc));
	g_return_if_fail (child && IS_QUERY_COND (child));

	if (qc->cond_type == QUERY_COND_COND) {
		/* we first need to convert to an AND node */
		Query *q = qc->query;
		QueryCond *qc2 = QUERY_COND (query_cond_new_copy (qc));
		
		query_cond_reinit (G_OBJECT (qc));
		qc->query = q;
		qc->cond_type = QUERY_COND_AND;
		query_cond_add_to_node (qc, qc2, -1);
	}

	/* add the child */
	qc->children = g_slist_insert (qc->children, child, pos);
	g_object_weak_ref (G_OBJECT (child), (GWeakNotify) child_node_destroy_notify, qc);
	g_signal_connect (G_OBJECT (child), "changed",
			  G_CALLBACK (child_changed_cb), qc);

	/* emit signal */
#ifdef debug_signal
	g_print (">> 'CHANGED' from query_cond_add_to_node\n");
#endif
	g_signal_emit (G_OBJECT (qc), signals[CHANGED], 0);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from query_cond_add_to_node\n");
#endif
}

static void
child_changed_cb (QueryCond *child, QueryCond *qc)
{
#ifdef debug_signal
	g_print (">> 'CHANGED' from QueryCond::child_changed_cb\n");
#endif
	g_signal_emit (G_OBJECT (qc), signals[CHANGED], 0);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from QueryCond::child_changed_cb\n");
#endif
}

static void
child_node_destroy_notify (QueryCond *qc, QueryCond *child) /* GWeakNotify */
{
	qc->children = g_slist_remove (qc->children, child);
	g_signal_handlers_disconnect_by_func (G_OBJECT (child), G_CALLBACK (child_changed_cb), qc);

	/* emit signal */
#ifdef debug_signal
	g_print (">> 'CHANGED' from QueryCond::child_node_destroy_notify\n");
#endif
	g_signal_emit (G_OBJECT (qc), signals[CHANGED], 0);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from QueryCond::child_node_destroy_notify\n");
#endif
}

void
query_cond_del_from_node (QueryCond *qc, QueryCond *child, gboolean destroy_child)
{
	g_return_if_fail (qc && IS_QUERY_COND (qc));
	g_return_if_fail (child && IS_QUERY_COND (child));
	g_return_if_fail (child->query != qc->query);
	g_return_if_fail (g_slist_find (qc->children, child));
	
	g_object_weak_unref (G_OBJECT (child), (GWeakNotify) child_node_destroy_notify, qc);
	qc->children = g_slist_remove (qc->children, child);
	g_signal_handlers_disconnect_by_func (G_OBJECT (child), G_CALLBACK (child_changed_cb), qc);

	/* emit signal */
#ifdef debug_signal
	g_print (">> 'CHANGED' from query_cond_del_from_node\n");
#endif
	g_signal_emit (G_OBJECT (qc), signals[CHANGED], 0);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from query_cond_del_from_node\n");
#endif

	if (destroy_child) 
		query_cond_free (child);
}

void
query_cond_set_leaf_type (QueryCond *qc, QueryCondOpType type)
{
	g_return_if_fail (qc && IS_QUERY_COND (qc));

	if (qc->op_type != type) {
		qc->op_type = type;

		/* emit signal */
#ifdef debug_signal
		g_print (">> 'CHANGED' from query_cond_set_leaf_type\n");
#endif
		g_signal_emit (G_OBJECT (qc), signals[CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'CHANGED' from query_cond_set_leaf_type\n");
#endif

	}
}


/*
 * XML Related stuff 
 */
static const gchar *query_cond_get_op_str   (QueryCondOpType op_type);
static const gchar *query_cond_get_op_sql   (QueryCondOpType op_type);
static const gchar *query_cond_get_type_str (QueryCondType cond_type);
static QueryCondType query_cond_get_type_enum (const gchar *type);
static QueryCondOpType query_cond_get_op_enum (const gchar *optype);

xmlNodePtr 
query_cond_save_to_xml (QueryCond *qc)
{
	xmlNodePtr node;
	gchar *str;
	const gchar *cstr;

	node = xmlNewNode (NULL, "QueryCond");

	/* Condition */
	cstr = query_cond_get_type_str (qc->cond_type);
	xmlSetProp (node, "type", cstr);

	if (qc->cond_type == QUERY_COND_COND) {
		cstr = query_cond_get_op_str (qc->op_type);
		xmlSetProp (node, "op", cstr);

		/* operators */
		if (qc->left_op) {
			str = query_field_get_xml_id (qc->left_op);
			xmlSetProp (node, "left_op", str);
			g_free (str);
		}
		
		if (qc->right_op) {
			str = query_field_get_xml_id (qc->right_op);
			xmlSetProp (node, "right_op", str);
			g_free (str);
		}
		
		if (qc->right_op2) {
			str = query_field_get_xml_id (qc->right_op2);
			xmlSetProp (node, "right_op2", str);
			g_free (str);
		}
	}
	else {
		/* Children nodes? */
		if (qc->cond_type != QUERY_COND_COND) {
			GSList *list = qc->children;
			
			while (list) {
				xmlNodePtr nde;
				nde = query_cond_save_to_xml (QUERY_COND (list->data));
				xmlAddChild (node, nde);
				list = g_slist_next (list);
			}
		}
	}
	
	return node;
}

GObject *
query_cond_new_from_xml (ConfManager *conf, Query *q,  xmlNodePtr node)
{
	GObject *obj = NULL;
	gchar *str;
	QueryCond *qc;
		
	g_return_val_if_fail (conf && IS_CONF_MANAGER (conf), NULL);
	g_return_val_if_fail (q && IS_QUERY (q), NULL);
	g_return_val_if_fail (node, NULL);

	obj = g_object_new (QUERY_COND_TYPE, NULL);
	qc = QUERY_COND (obj);

	qc->query = q;
		
	str = xmlGetProp (node, "type");
	if (str) {
		qc->cond_type = query_cond_get_type_enum (str);
		g_free (str);
	}
		
	if (qc->cond_type == QUERY_COND_COND) {
		QueryField *qf;

		str = xmlGetProp (node, "op");
		if (str) {
			qc->op_type = query_cond_get_op_enum (str);
			g_free (str);
		}
			
		/* operators */
		str = xmlGetProp (node, "left_op");
		if (str) {
			qf = query_get_field_by_xmlid (q, str);
			if (qf)
				qc->left_op = qf;
			g_free (str);
		}

		str = xmlGetProp (node, "right_op");
		if (str) {
			qf = query_get_field_by_xmlid (q, str);
			if (qf)
				qc->right_op = qf;
			g_free (str);
		}

		str = xmlGetProp (node, "right_op2");
		if (str) {
			qf = query_get_field_by_xmlid (q, str);
			if (qf)
				qc->right_op2 = qf;
			g_free (str);
		}
	}
	else { /* children QueryCond */
		xmlNodePtr subnode;

		subnode = node->children;
		while (subnode) {
			GObject *scond;

			scond = query_cond_new_from_xml (conf, q, subnode);
			query_cond_add_to_node (qc, QUERY_COND (scond), -1);
			subnode = subnode->next;
		}
	}

	return obj;
}

static QueryCondType
query_cond_get_type_enum (const gchar *type)
{
	QueryCondType ret;

	switch (*type) {
	case 'C':
		ret = QUERY_COND_COND;
		break;
	case 'A':
		ret = QUERY_COND_AND;
		break;
	case 'O':
		ret = QUERY_COND_OR;
		break;
	case 'N':
		ret = QUERY_COND_NOT;
		break;
	default:
		ret = LAST_QUERY_COND_TYPE;
		break;
	}

	return ret;
}

static const gchar *
query_cond_get_type_str (QueryCondType cond_type)
{
	gchar *str;

	switch (cond_type) {
	case QUERY_COND_COND:
		str = "COND";
		break;
	case QUERY_COND_AND:
		str = "AND";
		break;
	case QUERY_COND_OR:
		str = "OR";
		break;
	case QUERY_COND_NOT:
		str = "NOT";
		break;
	default:
		str = "unknown";
		break;
	}

	return str;
}



static const gchar *
query_cond_get_op_str (QueryCondOpType op_type)
{
	gchar *str;
 
	switch (op_type) {	
	case QUERY_COND_OP_EQUAL:
		str = "EQ";
		break;
	case QUERY_COND_OP_DIFF:
		str = "NE";
		break;
	case QUERY_COND_OP_SUP:
		str = "SUP";
		break;
	case QUERY_COND_OP_SUPEQUAL:
		str = "ESUP";
		break;
	case QUERY_COND_OP_INF:
		str = "IF";
		break;
	case QUERY_COND_OP_INFEQUAL:
		str = "EINF";
		break;
	case QUERY_COND_OP_LIKE:
		str = "LIKE";
		break;
	case QUERY_COND_OP_REGEX:
		str = "REG";
		break;
	case QUERY_COND_OP_IN:
		str = "IN";
		break;
	case QUERY_COND_OP_BETWEEN:
		str = "BTW";
		break;
	default:
		str = "unknown";
		break;
	}

	return str;
}

static const gchar *
query_cond_get_op_sql (QueryCondOpType op_type)
{
	gchar *str;
 
	switch (op_type) {	
	case QUERY_COND_OP_EQUAL:
		str = "=";
		break;
	case QUERY_COND_OP_DIFF:
		str = "!=";
		break;
	case QUERY_COND_OP_SUP:
		str = ">";
		break;
	case QUERY_COND_OP_SUPEQUAL:
		str = ">=";
		break;
	case QUERY_COND_OP_INF:
		str = "<";
		break;
	case QUERY_COND_OP_INFEQUAL:
		str = "<=";
		break;
	case QUERY_COND_OP_LIKE:
		str = "LIKE";
		break;
	case QUERY_COND_OP_REGEX:
		str = "~";
		break;
	case QUERY_COND_OP_IN:
		str = "IN";
		break;
	case QUERY_COND_OP_BETWEEN:
		str = "BETWEEN";
		break;
	default:
		str = "unknown";
		break;
	}

	return str;
}


static QueryCondOpType
query_cond_get_op_enum (const gchar *optype)
{
	QueryCondOpType op;

	switch (*optype) {
	case 'E':
		switch (*(optype+1)) {
		case 'S':
			op = QUERY_COND_OP_SUPEQUAL;
			break;
		case 'I':
			op = QUERY_COND_OP_INFEQUAL;
			break;
		case 'Q':
		default:
			op = QUERY_COND_OP_EQUAL;
			break;
		}
		break;
	case 'N':
		op = QUERY_COND_OP_DIFF;
		break;
	case 'S':
		op = QUERY_COND_OP_SUP;
		break;
	case 'I':
		switch (*(optype+1)) {
		case 'N':
			op = QUERY_COND_OP_IN;
			break;
		case 'F':
		default:
			op = QUERY_COND_OP_INF;
			break;
		}
		break;
	case 'L':
		op = QUERY_COND_OP_LIKE;
		break;
	case 'R':
		op = QUERY_COND_OP_REGEX;
		break;
	case 'B':
		op = QUERY_COND_OP_BETWEEN;
		break;
	default:
		op = LAST_QUERY_COND_OP;
		break;
	}

	return op;
}


/* Rendering */
gchar *
query_cond_render_as_sql    (QueryCond *qc, GSList * missing_values)
{
	GString *string;
	gchar *str, *str2, *str3;
	GSList *list;
	g_return_val_if_fail (qc && IS_QUERY_COND (qc), NULL);
	
	string = g_string_new ("");

	switch (qc->cond_type) {
	case QUERY_COND_COND:
		str = query_field_render_as_sql (qc->left_op, missing_values);
		str2 = query_field_render_as_sql (qc->right_op, missing_values);
		if (qc->right_op2)
			str3 = query_field_render_as_sql (qc->right_op2, missing_values);
		else
			str3 = NULL;

		switch (qc->op_type) {
		case QUERY_COND_OP_BETWEEN:
			g_string_append_printf (string, "%s BETWEEN %s AND %s", str, str2, str3);
			break;
		default:
			g_string_append_printf (string, "%s %s %s", str, query_cond_get_op_sql (qc->op_type) ,str2);
			break;
		}
		g_free (str);
		g_free (str2);
		if (str3)
			g_free (str3);
		break;
	case QUERY_COND_NOT:
		if (qc->children)
			str = query_cond_render_as_sql (qc->children->data, missing_values);
		else
			str = g_strdup ("???");

		g_string_append_printf (string, "(NOT %s)", str);

		g_free (str);
		break;
	default:
		g_string_append (string, "(");
		list = qc->children;
		while (list) {
			str = query_cond_render_as_sql (qc->children->data, missing_values);

			if (list == qc->children)
				g_string_append_printf (string, "%s", str);
			else
				g_string_append_printf (string, "%s %s", query_cond_get_type_str (qc->cond_type), str);

			g_free (str);
			list = g_slist_next (list);
		}
		g_string_append (string, ")");
		break;
	}

	str = string->str;
	g_string_free (string, FALSE);
	return str;
}

xmlNodePtr
query_cond_render_as_xml (QueryCond *qc, GSList * missing_values)
{
	xmlNodePtr node = NULL;
	
	/* FIXME */

	return node;
}






/* "Tree" interrogations */

gboolean
query_cond_is_ancestor (QueryCond *qc, QueryCond *ancestor)
{
	gboolean found = FALSE;

	g_return_val_if_fail (qc && IS_QUERY_COND (qc), FALSE);
	g_return_val_if_fail (ancestor && IS_QUERY_COND (ancestor), FALSE);
	g_return_val_if_fail (ancestor->query == qc->query, FALSE);

	if (qc == ancestor)
		return TRUE;

	if (ancestor->children) {
		if (g_slist_find (ancestor->children, qc))
			found = TRUE;
		else {
			GSList *list = ancestor->children;
			while (list && !found) {
				found = query_cond_is_ancestor (qc, QUERY_COND (list->data));
				list = g_slist_next (list);
			}
		}
	}

	return found;
}

QueryCond *
query_cond_get_parent (QueryCond *qc, QueryCond *top_cond)
{
	QueryCond *retval = NULL;

	g_return_val_if_fail (qc && IS_QUERY_COND (qc), NULL);
	g_return_val_if_fail (top_cond && IS_QUERY_COND (top_cond), NULL);
	g_return_val_if_fail (top_cond->query == qc->query, NULL);

	if (top_cond->children) {
		if (g_slist_find (top_cond->children, qc))
			retval = top_cond;
		else {
			GSList *list = top_cond->children;
			while (list && !qc) {
				retval = query_cond_get_parent (qc, QUERY_COND (list->data));
				list = g_slist_next (list);
			}
		}
	}

	return retval;
}


#ifdef debug
/* structure debug function */
void
query_cond_dump_contents (QueryCond *qc, guint offset)
{
	gchar *str, *sql;
	guint i;

	str = g_new0 (gchar, offset+1);
	for (i=0; i<offset; i++)
		str[i] = ' ';
	str[offset] = 0;
	
	g_print ("%s  *" D_COL_H2 "%s" D_COL_NOR "\n", str,  query_cond_get_type_str (qc->cond_type));

	sql = query_cond_render_as_sql (qc, NULL);
	g_print ("%s  * renders as: %s\n", str, sql);

	g_free (sql);
	g_free (str);
}
#endif
