/* data-form.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "data-form.h"
#include "marshal.h"
#include "data-entry.h"

static void data_form_class_init (DataFormClass * class);
static void data_form_init (DataForm * form);
static void data_form_finalize (GObject   * object);

enum
{
	ACTION_REFRESH,
	ACTION_INSERT,
	ACTION_VIEWALL,
	ACTION_CHOOSE,
	CONTENTS_MODIFIED,
	LAST_SIGNAL
};


typedef enum   _DataFormMode   DataFormMode;
typedef struct _ValueRef       ValueRef;
typedef struct _EntryEntity    EntryEntity;

enum _DataFormMode {
	/* edition of data from a query, for a given DbTable (for which there is only one 
	   corresponding QueryView object. The only data entry widgets are the ones related
	   to the DbTable */
        DATA_FORM_EDIT_MODE,

	/* same as EDIT_MODE but all entries are empty (or imposed by the Query) to start with */
        DATA_FORM_INSERT_MODE,

	/* data selection mode: selects data from a given Query and an given QueryView in this
	   Query: displays choice combo entries. It creates its own Query and also managed its own
	   ServerResultset object. */
        DATA_FORM_SELECT_MODE, 

	/* just showing the data from the ServerResultset of the QueryExec object, no modification
	   allowed (if modifications are required, then create another DataForm in EDIT or INSERT 
	   mode*/
        DATA_FORM_BROWSE_MODE  /* same as EDIT but no modification allowed */
};

struct _DataFormPriv
{
	DataFormMode     mode;

	QueryExec       *qx;       /* not NULL */
	QueryView       *qv;

        GSList          *value_refs;
        GSList          *entries;

	GSList          *children;      /* sub forms */

	/* SELECT mode only */
	DataForm        *master;
	Query           *internal_query;
	ServerResultset *internal_rs;

	/* EDIT and BROWSE modes only */
        gulong           numrow;

	/* EDIT, INSERT and BROWSE modes only */
	GtkWidget       *actions_area;
	guint            actions; /* possible user actions */
};

struct _ValueRef {
        gpointer        ref;       /* DbField or QueryParamValue */
        EntryEntity    *entry;
	GdaValue       *orig_value;
};
#define VALUE_REF(vr) ((ValueRef*)(vr))


struct _EntryEntity {
        ValueRef       *value_ref;
        QueryField     *qf;

        guint           numcol; /* if SELECT, refers to the internal_rs and not the RS of QueryExec */

	/* if DataEntry only */
        DataEntry      *entry;

	/* if other DataForm only */
	DataForm       *form;
	QueryField     *join_to;
};
#define ENTRY_ENTITY(ee) ((EntryEntity*)(ee))


/* Returns the value of a QueryField from a Form:
   the QF needs to be a field and reference a QueryView which is represented 
   by the DataForm df.
   returns a new GdaValue, or NULL if there is a problem (along with a warning)
*/
static GdaValue *data_form_get_gdavalue (DataForm *df, QueryField *qf);

/* Tries to find a DataForm for a given QueryView. */
static DataForm *data_form_find_from_query_view (DataForm *master, QueryView *qv);

static ValueRef *data_form_find_value_ref (DataForm *df, gpointer ref);

/* Builds the default builtin Form from the internal structure of the DataForm */
static GtkWidget *build_complete_noglade_form_widget (DataForm * form);

static void     data_form_user_action_cb (GtkWidget *wid, DataForm * form);

static gint data_form_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
data_form_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (DataFormClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) data_form_class_init,
			NULL,
			NULL,
			sizeof (DataForm),
			0,
			(GInstanceInitFunc) data_form_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "DataForm", &info, 0);
	}

	return type;
}

static void
data_form_class_init (DataFormClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	data_form_signals[ACTION_REFRESH] =
		g_signal_new ("action_refresh",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (DataFormClass, action_refresh),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	data_form_signals[ACTION_INSERT] =
		g_signal_new ("action_insert",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (DataFormClass, action_insert),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	data_form_signals[ACTION_VIEWALL] =
		g_signal_new ("action_viewall",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (DataFormClass, action_viewall),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	data_form_signals[ACTION_CHOOSE] =
		g_signal_new ("action_choose",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (DataFormClass, action_choose),
			      NULL, NULL,
			      marshal_VOID__ULONG, G_TYPE_NONE, 1,
			      G_TYPE_ULONG);
	data_form_signals[CONTENTS_MODIFIED] =
		g_signal_new ("contents_modified",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (DataFormClass, contents_modified),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	class->action_refresh = NULL;
	class->action_insert = NULL;
	class->action_viewall = NULL;
	class->action_choose = NULL;
	class->contents_modified = NULL;
	object_class->finalize = data_form_finalize;
}

static void
data_form_init (DataForm * form)
{
	form->priv = g_new0 (DataFormPriv, 1);
	form->priv->qx = NULL;
	form->priv->qv = NULL;
	form->priv->value_refs = NULL;
	form->priv->entries = NULL;
	form->priv->master = NULL;
	form->priv->children = NULL;
	form->priv->internal_query = NULL;
	form->priv->internal_rs = NULL;
	form->priv->numrow = 0;
	form->priv->actions = 0;
	form->priv->actions_area = NULL;
}


static void data_form_edit_initialize (DataForm * form);
GtkWidget *
data_form_new_edit (QueryExec *qx, gulong numrow)
{
	GObject   *obj;
	DataForm *form;
	GSList *list;
	gint i;

	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), NULL);

	obj = g_object_new (DATA_FORM_TYPE, NULL);
	form = DATA_FORM (obj);

	form->priv->qx = qx;
	form->priv->mode = DATA_FORM_EDIT_MODE;
	form->priv->numrow = numrow;
	form->priv->master = form; /* this form is the master for the other sub forms */
	form->priv->actions = query_exec_get_user_actions (qx);

	/* Find the right QueryView */
	list = query_exec_get_query_ref(qx)->views;
	i = 0;
	while (list) {
		QueryView *qv = QUERY_VIEW (list->data);
		if (IS_DB_TABLE (qv->obj) && (DB_TABLE (qv->obj) == query_exec_get_modif_table (qx))) {
			form->priv->qv = qv;
			i++;
		}
		list = g_slist_next (list);
	}
	if (i>1)
		g_warning ("There is more than 1 QueryView for the edited table!");

	data_form_edit_initialize (form);

	return GTK_WIDGET (obj);
}

static void data_form_insert_initialize (DataForm * form);
GtkWidget *
data_form_new_insert (QueryExec *qx)
{
	GObject   *obj;
	DataForm *form;
	GSList *list;
	gint i;

	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), NULL);

	obj = g_object_new (DATA_FORM_TYPE, NULL);
	form = DATA_FORM (obj);

	form->priv->qx = qx;
	form->priv->mode = DATA_FORM_INSERT_MODE;
	form->priv->master = form; /* this form is the master for the other sub forms */
	form->priv->actions = QUERY_ACTION_COMMIT;

	/* Find the right QueryView */
	list = query_exec_get_query_ref(qx)->views;
	i = 0;
	while (list) {
		QueryView *qv = QUERY_VIEW (list->data);
		if (IS_DB_TABLE (qv->obj) && (DB_TABLE (qv->obj) == query_exec_get_modif_table (qx))) {
			form->priv->qv = qv;
			i++;
		}
		list = g_slist_next (list);
	}
	if (i>1)
		g_warning ("There is more than 1 QueryView for the edited table!");

	data_form_insert_initialize (form);

	return GTK_WIDGET (obj);
}


static void data_form_browse_initialize (DataForm * form);
GtkWidget *
data_form_new_browse (QueryExec *qx, gulong numrow)
{
	GObject   *obj;
	DataForm *form;

	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), NULL);

	/* FIXME: how to set the parameters for the GtkVBox? */
	obj = g_object_new (DATA_FORM_TYPE, NULL);
	form = DATA_FORM (obj);

	form->priv->qx = qx;
	form->priv->mode = DATA_FORM_BROWSE_MODE;
	form->priv->numrow = numrow;

	data_form_browse_initialize (form);

	return GTK_WIDGET (obj);
}


static void data_form_select_initialize (DataForm * form);
GtkWidget *
data_form_new_select (QueryExec *qx, QueryView *qv, gulong numrow)
{
	GObject   *obj;
	DataForm *form;

	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), NULL);
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);
	g_return_val_if_fail (QUERY_VIEW (qv)->query != query_exec_get_query_ref (qx), NULL);

	/* FIXME: how to set the parameters for the GtkVBox? */
	obj = g_object_new (DATA_FORM_TYPE, NULL);
	form = DATA_FORM (obj);

	form->priv->qx = qx;
	form->priv->mode = DATA_FORM_SELECT_MODE;
	form->priv->numrow = numrow;

	data_form_select_initialize (form);

	return GTK_WIDGET (obj);
}

static void
data_form_finalize (GObject   * object)
{
	DataForm *df;
	GSList *list;

	g_return_if_fail (object && IS_DATA_FORM (object));

	df = DATA_FORM (object);

	if (df->priv) {
		/* ValueRef */
		if (df->priv->value_refs) {
			list = df->priv->value_refs;
			while (list) {
				if (VALUE_REF (list->data)->orig_value)
					gda_value_free (VALUE_REF (list->data)->orig_value);
				g_free (list->data);
				list = g_slist_next (list);
			}
			g_slist_free (df->priv->value_refs);
			df->priv->value_refs = NULL;
		}
		
		/* Entries */
		if (df->priv->entries) {
			list = df->priv->entries;
			while (list) {
				g_free (list->data);
				list = g_slist_next (list);
			}
			g_slist_free (df->priv->entries);
			df->priv->entries = NULL;
		}
		
		/* Children */
		if (df->priv->children) {
			g_slist_free (df->priv->children);
			df->priv->children = NULL;
		}
		
		/* Internal query */
		if (df->priv->internal_query) {
			g_object_unref (G_OBJECT (df->priv->internal_query));
			df->priv->internal_query = NULL;
		}
		
		/* Internal ServerResultset */
		if (df->priv->internal_rs) {
			g_object_unref (G_OBJECT (df->priv->internal_rs));
			df->priv->internal_rs = NULL;
		}
		g_free (df->priv);
		df->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}












/*
 *
 * EDIT MODE
 *
 */
static GtkWidget *data_form_add_action_buttons (DataForm *form, GCallback cb, gpointer data);
static void build_edit_insert_form_structure (DataForm * form);
static void 
data_form_edit_initialize (DataForm * form)
{
	GtkWidget *topwid = NULL;

	if (! query_exec_get_data_set (form->priv->qx) || 
	    (server_resultset_get_nbtuples (query_exec_get_data_set (form->priv->qx)) == 0))
		topwid = gtk_label_new (_("No Data!"));
	else {
		build_edit_insert_form_structure (form);

		if (0) {
			/* FIXME: update when glade forms are introduced */
			/* we have a custom glade form here */
			topwid = gtk_label_new ("Glade imported form to create");
		}
		else {
			/* default builtin form */
			topwid = build_complete_noglade_form_widget (form);
		}
	}

	if (topwid) {
		GtkWidget *buttons;
		gtk_widget_show_all (topwid);
		gtk_box_pack_start_defaults (GTK_BOX (form), topwid);
		
		buttons = data_form_add_action_buttons (form, G_CALLBACK (data_form_user_action_cb), form);
		gtk_widget_show_all (buttons);
		gtk_box_pack_start (GTK_BOX (form), buttons, FALSE, FALSE, GNOME_PAD/2.);
	}
}

static void 
build_edit_insert_form_structure (DataForm * form)
{
	QueryField *qf;
	GSList *list;
	
	list = query_exec_get_query_ref (form->priv->qx)->fields;
	while (list) {
		qf = QUERY_FIELD (list->data);

		if (qf->field_type == QUERY_FIELD_FIELD) {
			QueryView *qv;

			qv = query_field_field_get_queryview (qf);
			if (qv && IS_DB_TABLE (qv->obj) && 
			    (DB_TABLE (qv->obj) == query_exec_get_modif_table (form->priv->qx))) {
				GSList *joins;
				QueryJoin *qj = NULL;
				
				/* Find a join if it exists */
				joins = query_find_equi_join_with_end (query_exec_get_query_ref (form->priv->qx), qf);
				if (joins) {
					if (g_slist_length (joins) > 1)
						g_warning ("More than one Foreign Key on the %s field, ignored",
							   query_field_field_get_db_field (qf)->name);
					else
						qj = QUERY_JOIN (joins->data);
					g_slist_free (joins);
				}
				
				if (qj) { /* We create a sub form a QueryJoin */
					QueryView *oqv;
					QueryField *oqf = NULL;
					DbField *ofield = NULL;

					DataForm *df;
					ValueRef *vr;
					EntryEntity *ee;
					const GdaValue *value;
					gint pos;

					/* set the other QueryView */
					if (qj->ant_view == qv)
						oqv = qj->suc_view;
					else
						oqv = qj->ant_view;
					
					/* set the other DbField */
					if (QUERY_JOIN_PAIR (qj->pairs->data)->ant_field == G_OBJECT (qf))
						ofield = DB_FIELD (QUERY_JOIN_PAIR (qj->pairs->data)->suc_field);
					else
						ofield = DB_FIELD (QUERY_JOIN_PAIR (qj->pairs->data)->ant_field);

					/* set the other QueryField */
					oqf = query_get_field_by_dbfield (query_exec_get_query_ref (form->priv->qx), 
									  oqv, ofield);
					g_assert (oqv);
					g_assert (oqf);
					g_assert (ofield);
					
					if (! (df = data_form_find_from_query_view (form->priv->master, oqv))) {
						/* No data form found for oqv, build a new one */
						df = DATA_FORM (data_form_new_select (form->priv->qx, oqv, 
										      form->priv->numrow));
						df->priv->master = form->priv->master;
						form->priv->master->priv->children = g_slist_append (form->priv->master->priv->children,
												     df);
					}
				
					pos = query_get_pos_by_field (query_exec_get_query_ref (form->priv->qx), qf);
					/* EntryEntity */
					ee = g_new0 (EntryEntity, 1);
					ee->qf = qf;
					ee->numcol = pos;
					ee->entry = NULL;
					ee->form = df;
					ee->join_to = oqf;
					form->priv->entries = g_slist_append (form->priv->entries, ee);

					/* ValueRef */
					vr = data_form_find_value_ref (form, ofield);
					if (!vr) {
						vr = g_new0 (ValueRef, 1);
						vr->ref = ofield;
						vr->entry = ee;
						if (form->priv->mode == DATA_FORM_EDIT_MODE) {
							value = server_resultset_get_gdavalue (query_exec_get_data_set (form->priv->qx),
											       pos,
											       form->priv->numrow);

							vr->orig_value = gda_value_copy (value);
						}
						else
							vr->orig_value = gda_value_new_null ();

						vr->entry = ee;
							
						form->priv->value_refs = g_slist_append (form->priv->value_refs,
											 vr);
					}
					ee->value_ref = vr;
				}
				else { /* No QueryJoin for that QueryField */
					if (qf->is_printed && !qf->is_hidden) {
						DataEntry *de;
						DataDisplayFns *fns;
						const GdaValue *value;
						DbField *field = NULL;
						EntryEntity *ee;
						ValueRef *vr;
						gint pos;
						ServerAccess *srv;
							
						pos = query_get_pos_by_field (query_exec_get_query_ref (form->priv->qx),
									      qf);
						if (form->priv->mode == DATA_FORM_EDIT_MODE)
							value = server_resultset_get_gdavalue (query_exec_get_data_set (form->priv->qx),
											       form->priv->numrow, pos);
						else
							value = NULL;

						srv = query_exec_get_query_ref (form->priv->qx)->conf->srv;
						field = query_field_field_get_db_field (qf);
						if (server_access_is_object_bound (srv, G_OBJECT (qf)))
							fns = server_access_get_object_display_fns (srv, 
												    G_OBJECT (qf));
						else
							fns = server_access_get_object_display_fns (srv, 
												    G_OBJECT (field));
						if (value) 
							de = DATA_ENTRY ((fns->widget_from_value) (value));
						else {
							GdaValue *myvalue;
							myvalue = gda_value_new_null ();
							de = DATA_ENTRY ((fns->widget_from_value) (myvalue));
							gda_value_free (myvalue);
						}
						de->fns = fns;
						de->query_field = qf;
						
						if (form->priv->mode == DATA_FORM_INSERT_MODE) {
							if (field->default_val) {
								de->value_is_defaultval = TRUE;
								de->is_default_possible = TRUE;
							}
							else {
								de->value_is_defaultval = FALSE;
								de->is_default_possible = FALSE;
							}
						}
						if (! field->null_allowed)
							de->value_required = TRUE;

						data_entry_refresh_display (de);
						
						/* EntryEntity */
						ee = g_new0 (EntryEntity, 1);
						ee->qf = qf;
						ee->numcol = pos;
						ee->entry = de;
						ee->form = NULL;
						ee->join_to = NULL;
						form->priv->entries = g_slist_append (form->priv->entries, ee);

						/* ValueRef */
						vr = data_form_find_value_ref (form, field);
						if (!vr) {
							vr = g_new0 (ValueRef, 1);
							vr->ref = field;
							vr->entry = ee;
							pos = query_get_pos_by_field (query_exec_get_query_ref (form->priv->qx), qf);
							if (form->priv->mode == DATA_FORM_EDIT_MODE) {
								value = server_resultset_get_gdavalue (query_exec_get_data_set (form->priv->qx),
												       form->priv->numrow, pos);
								vr->orig_value = gda_value_copy (value);
							}
							else 
								vr->orig_value = gda_value_new_null ();

							vr->entry = ee;
								
							form->priv->value_refs = g_slist_append (form->priv->value_refs,
												 vr);
						}
						ee->value_ref = vr;
					}
				}
			}
		}
		list = g_slist_next (list);
	}
}








/*
 *
 * INSERT MODE
 *
 */

static void data_form_insert_initialize (DataForm * form)
{
	GtkWidget *topwid = NULL;


	build_edit_insert_form_structure (form);
	
	if (0) {
		/* FIXME: update when glade forms are introduced */
		/* we have a custom glade form here */
		topwid = gtk_label_new ("Glade imported form to create");
	}
	else {
		/* default builtin form */
		topwid = build_complete_noglade_form_widget (form);
	}


	if (topwid) {
		GtkWidget *buttons;
		gtk_widget_show_all (topwid);
		gtk_box_pack_start_defaults (GTK_BOX (form), topwid);
		
		buttons = data_form_add_action_buttons (form, G_CALLBACK (data_form_user_action_cb), form);
		gtk_widget_show_all (buttons);
		gtk_box_pack_start (GTK_BOX (form), buttons, FALSE, FALSE, GNOME_PAD/2.);
	}
}









/*
 *
 * BROWSE MODE
 *
 */

static void data_form_browse_initialize (DataForm * form)
{

}







/*
 *
 * SELECT MODE
 *
 */

static void build_select_form_structure (DataForm * form);
static void data_form_select_initialize (DataForm * form)
{
	GtkWidget *topwid = NULL;

	if (!query_exec_get_data_set (form->priv->qx)) 
		topwid = gtk_label_new (_("No Data!"));
	else {
		build_select_form_structure (form);

		if (0) {
			/* we have a custom glade form here */
			topwid = gtk_label_new ("Glade imported form to create");
		}
		else {
			/* default builtin form */
			topwid = build_complete_noglade_form_widget (form);
		}
	}

	if (topwid) {
		gtk_widget_show_all (topwid);
		gtk_box_pack_start_defaults (GTK_BOX (form), topwid);
	}
}

static void 
build_select_form_structure (DataForm * form)
{

}











/*
 *
 * FUNCTIONS FOR more than ONE MODES
 *
 */
static void data_entry_contents_modified_cb (DataEntry *de, DataForm *form);
static GtkWidget *
build_complete_noglade_form_widget (DataForm * form)
{
	GSList *list;
	guint i = 0;
	GtkWidget *topwid;
	
	topwid = gtk_table_new (g_slist_length (form->priv->entries), 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (topwid), 0);
	
	list = form->priv->entries;
	while (list) {
		GtkWidget *wid=NULL, *label;
		
		if (ENTRY_ENTITY (list->data)->entry) {
			wid = GTK_WIDGET (ENTRY_ENTITY (list->data)->entry);

			/* fetch the "contents_modified" signal */
			g_signal_connect (wid, "contents_modified",
					  G_CALLBACK (data_entry_contents_modified_cb), form);
		}
		if (ENTRY_ENTITY (list->data)->form)
			wid = GTK_WIDGET (ENTRY_ENTITY (list->data)->form);
		gtk_table_attach_defaults (GTK_TABLE (topwid), wid,
					   1, 2, i, i+1);
		
		if (ENTRY_ENTITY (list->data)->qf->name)
			label = gtk_label_new (ENTRY_ENTITY (list->data)->qf->name);
		else
			label = gtk_label_new (DB_FIELD (ENTRY_ENTITY (list->data)->value_ref->ref)->name);
		
		gtk_table_attach (GTK_TABLE (topwid), label,
				  0, 1, i, i+1, 
				  0, 0, GNOME_PAD/2., 0);
		
		
		list = g_slist_next (list);
		i++;
	}

	return topwid;
}




static void data_entry_refresh_actions_area (DataForm *form);
static void 
data_entry_contents_modified_cb (DataEntry *de, DataForm *form)
{
	data_entry_refresh_actions_area (form);
}


static gboolean data_form_is_on_first_entry (DataForm *form);
static gboolean data_form_is_on_last_entry (DataForm *form);
static gboolean data_form_is_diff_than_orig (DataForm *form);
static void 
data_entry_refresh_actions_area (DataForm *form)
{
	gchar *str;
	GtkWidget *wid, *collector;
	gboolean is_first, is_last, is_modified, to_commit;

	collector = form->priv->actions_area;
	if (!collector)
		return;

	is_first = data_form_is_on_first_entry (form);
	is_last = data_form_is_on_last_entry (form);
	is_modified = data_form_is_diff_than_orig (form);

	/* make sure there is no missing value in the form */
	to_commit = is_modified;
	if (to_commit) {
		GSList *list = form->priv->entries;

		while (list && to_commit) {
			if (ENTRY_ENTITY (list->data)->entry) {
				DataEntry *de = DATA_ENTRY (ENTRY_ENTITY (list->data)->entry);
				if (de->value_required && data_entry_is_value_null (de))
					to_commit = FALSE;
			}

			list = g_slist_next (list);
		}
	}

	/* QUERY_ACTION_FIRST */
	str = g_strdup_printf ("w%d", QUERY_ACTION_FIRST);
	wid = g_object_get_data (G_OBJECT (collector), str);
	g_free (str);
	if (wid)
		gtk_widget_set_sensitive (wid, !is_first);

	/* QUERY_ACTION_PREV */
	str = g_strdup_printf ("w%d", QUERY_ACTION_PREV);
	wid = g_object_get_data (G_OBJECT (collector), str);
	g_free (str);
	if (wid)
		gtk_widget_set_sensitive (wid, !is_first);

	/* QUERY_ACTION_LAST */
	str = g_strdup_printf ("w%d", QUERY_ACTION_LAST);
	wid = g_object_get_data (G_OBJECT (collector), str);
	g_free (str);
	if (wid)
		gtk_widget_set_sensitive (wid, !is_last);

	/* QUERY_ACTION_NEXT */
	str = g_strdup_printf ("w%d", QUERY_ACTION_NEXT);
	wid = g_object_get_data (G_OBJECT (collector), str);
	g_free (str);
	if (wid)
		gtk_widget_set_sensitive (wid, !is_last);


	/* QUERY_ACTION_COMMIT */
	str = g_strdup_printf ("w%d", QUERY_ACTION_COMMIT);
	wid = g_object_get_data (G_OBJECT (collector), str);
	g_free (str);
	if (wid)
		gtk_widget_set_sensitive (wid, to_commit);

	/* QUERY_ACTION_DELETE */
	str = g_strdup_printf ("w%d", QUERY_ACTION_DELETE);
	wid = g_object_get_data (G_OBJECT (collector), str);
	g_free (str);
	if (wid)
		gtk_widget_set_sensitive (wid, !is_modified);
}

static gboolean
data_form_is_on_first_entry (DataForm *form)
{
	gboolean retval;
	g_return_val_if_fail (form && IS_DATA_FORM (form), FALSE);

	if (! query_exec_get_data_set (form->priv->qx) || 
	    (server_resultset_get_nbtuples (query_exec_get_data_set (form->priv->qx)) == 0))
		retval = TRUE;
	else {
		if (form->priv->numrow == 0)
			retval = TRUE;
		else
			retval = FALSE;
	}
	

	return retval;
}

static gboolean
data_form_is_on_last_entry  (DataForm *form)
{
	gboolean retval;
	g_return_val_if_fail (form && IS_DATA_FORM (form), FALSE);

	if (! query_exec_get_data_set (form->priv->qx) || 
	    (server_resultset_get_nbtuples (query_exec_get_data_set (form->priv->qx)) == 0))
		retval = TRUE;
	else {
		if (form->priv->numrow == server_resultset_get_nbtuples (query_exec_get_data_set (form->priv->qx))-1)
			retval = TRUE;
		else
			retval = FALSE;
	}

	return retval;
}

static gboolean 
data_form_is_diff_than_orig (DataForm *form)
{
	gboolean retval = FALSE;
	GSList *list;

	g_return_val_if_fail (form && IS_DATA_FORM (form), FALSE);

	list = form->priv->entries;
	while (list) {
		EntryEntity *ee;

		ee = ENTRY_ENTITY (list->data);

		if (ee->entry) {
			if (DATA_ENTRY (ee->entry)->value_is_modified)
				retval = TRUE;
		}
		
		if (ee->form) { /* refresh the sub form */
			if (data_form_is_diff_than_orig (ee->form))
				retval = TRUE;
		}
		
		list = g_slist_next (list);
	}

	return retval;
}


void
data_form_refresh (DataForm *form)
{
	GSList *list;

	g_return_if_fail (form && IS_DATA_FORM (form));

	list = form->priv->entries;
	while (list) {
		EntryEntity *ee;

		ee = ENTRY_ENTITY (list->data);

		if (ee->entry) {
			gint pos;
			const GdaValue *value;

			pos = query_get_pos_by_field (query_exec_get_query_ref (form->priv->qx), ee->qf);
			value = server_resultset_get_gdavalue (query_exec_get_data_set (form->priv->qx),
							       form->priv->numrow, pos);
			(*ee->entry->fns->widget_update)(GTK_WIDGET (ee->entry), value, TRUE);
		}
		
		if (ee->form) { /* refresh the sub form */
			data_form_refresh (ee->form);
		}
		
		list = g_slist_next (list);
	}
}

static GdaValue *
data_form_get_gdavalue (DataForm *df, QueryField *qf)
{
	/* FIXME */

	return NULL;
}

static DataForm *
data_form_find_from_query_view (DataForm *start, QueryView *qv)
{
	DataForm *retval = NULL;
	GSList *list;

	g_return_val_if_fail (start && IS_DATA_FORM (start), NULL);
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);

	list = start->priv->children;
	while (list && !retval) {
		if (DATA_FORM (list->data)->priv->qv == qv)
			retval = DATA_FORM (list->data);
		list = g_slist_next (list);
	}

	return retval;
}

static ValueRef *
data_form_find_value_ref (DataForm *df, gpointer ref)
{
	ValueRef *vr = NULL;
	GSList *list;

	g_return_val_if_fail (df && IS_DATA_FORM (df), NULL);

	list = df->priv->value_refs;
	while (list && !vr) {
		if (VALUE_REF (list->data)->ref == ref)
			vr = VALUE_REF (list->data);
		list = g_slist_next (list);
	}

	return vr;
}


static void form_do_commit (DataForm *form);
static void 
data_form_user_action_cb (GtkWidget *wid, DataForm * form)
{
	QueryEnvUserActions action;

	action = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (wid), "action_id"));
	switch (action) {

	case QUERY_ACTION_INSERT:
		g_signal_emit (G_OBJECT (form), data_form_signals [ACTION_INSERT], 0);
		break;
	case QUERY_ACTION_REFRESH:
		g_signal_emit (G_OBJECT (form), data_form_signals [ACTION_REFRESH], 0);
		break;
	case QUERY_ACTION_VIEWALL:
		g_signal_emit (G_OBJECT (form), data_form_signals [ACTION_VIEWALL], 0);
		break;
	case QUERY_ACTION_FIRST:
		form->priv->numrow = 0;
		data_form_refresh (form);
		break;
	case QUERY_ACTION_LAST:
		form->priv->numrow = server_resultset_get_nbtuples (query_exec_get_data_set (form->priv->qx))-1;
		data_form_refresh (form);
		break;
	case QUERY_ACTION_PREV:
		if (form->priv->numrow >= 1)
			form->priv->numrow--;
		data_form_refresh (form);
		break;
	case QUERY_ACTION_NEXT:
		if (form->priv->numrow+1 < server_resultset_get_nbtuples (query_exec_get_data_set (form->priv->qx)))
			form->priv->numrow++;
		data_form_refresh (form);
		break;
	case QUERY_ACTION_COMMIT:
		form_do_commit (form);
		break;
	case QUERY_ACTION_DELETE:
		g_print ("To be implemented in %s line %d\n", __FILE__, __LINE__);
		break;
	case QUERY_ACTION_EDIT:
		g_print ("To be implemented in %s line %d\n", __FILE__, __LINE__);
		break;
	default :
		break;
	}
}

static void 
form_do_commit (DataForm *form)
{
	GSList *list, *values = NULL;

	list = form->priv->entries;
	while (list) {
		QueryParamValue *value;
		
		if (ENTRY_ENTITY (list->data)->entry) {
			DataEntry *de = ENTRY_ENTITY (list->data)->entry;
			
			value = g_new0 (QueryParamValue, 1);
			value->field = ENTRY_ENTITY (list->data)->qf;
			value->value = data_entry_get_gdavalue (de);
		}
		else {
			DataForm *subform = ENTRY_ENTITY (list->data)->form;
			value = g_new0 (QueryParamValue, 1);
			value->field = ENTRY_ENTITY (list->data)->qf;
			value->value = data_form_get_gdavalue (subform, value->field);
		}
		values = g_slist_append (values, value);

		list = g_slist_next (list);
	}


	/* FIXME: about default value and NULL values */
	g_print ("Started implementation in %s line %d\n", __FILE__, __LINE__);
}


/*
 * Actions buttons
 */
static void add_stock_pixmap (GtkWidget * button_box, gchar * action,
			      gchar * tip, QueryEnvUserActions action_val,
			      GCallback cb, gpointer data,
			      GObject *collector);

static GtkWidget *
data_form_add_action_buttons (DataForm *form, GCallback cb, gpointer data)
{
	GtkWidget *vbox, *bbox;
	gint actions = form->priv->actions;

	vbox = gtk_vbox_new (FALSE, GNOME_PAD / 4.);
	/* first row */
	bbox = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_START);
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), GNOME_PAD / 4.);

	if (actions & QUERY_ACTION_FIRST)
		add_stock_pixmap (bbox, GTK_STOCK_GOTO_FIRST,
				  _("Move to the first record"),
				  QUERY_ACTION_FIRST, cb, data, G_OBJECT (vbox));

	if (actions & QUERY_ACTION_PREV)
		add_stock_pixmap (bbox, GTK_STOCK_GO_BACK,
				  _("Move to the previous record"),
				  QUERY_ACTION_PREV, cb, data, G_OBJECT (vbox));

	if (actions & QUERY_ACTION_NEXT)
		add_stock_pixmap (bbox, GTK_STOCK_GO_FORWARD,
				  _("Move to the next record"),
				  QUERY_ACTION_NEXT, cb, data, G_OBJECT (vbox));

	if (actions & QUERY_ACTION_LAST)
		add_stock_pixmap (bbox, GTK_STOCK_GOTO_LAST,
				  _("Move to the last record"),
				  QUERY_ACTION_LAST, cb, data, G_OBJECT (vbox));


	/* second row */
	bbox = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_START);
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), GNOME_PAD / 4.);

	if (actions & QUERY_ACTION_EDIT)
		add_stock_pixmap (bbox, GTK_STOCK_INDEX,
				  _("Edit data"),
				  QUERY_ACTION_EDIT, cb, data, G_OBJECT (vbox));

	if (actions & QUERY_ACTION_VIEWALL)
		add_stock_pixmap (bbox, GTK_STOCK_INDEX,
				  _("View all as grid"),
				  QUERY_ACTION_VIEWALL, cb, data, G_OBJECT (vbox));

	if (actions & QUERY_ACTION_INSERT)
		add_stock_pixmap (bbox, GTK_STOCK_NEW,
				  _("Insert new data"),
				  QUERY_ACTION_INSERT, cb, data, G_OBJECT (vbox));

	if (actions & QUERY_ACTION_COMMIT)
		add_stock_pixmap (bbox, GTK_STOCK_SAVE,
				  _("Save changes"),
				  QUERY_ACTION_COMMIT, cb, data, G_OBJECT (vbox));

	if (actions & QUERY_ACTION_DELETE)
		add_stock_pixmap (bbox, GTK_STOCK_REMOVE,
				  _("Delete selected data"),
				  QUERY_ACTION_DELETE, cb, data, G_OBJECT (vbox));

	if (actions & QUERY_ACTION_REFRESH)
		add_stock_pixmap (bbox, GTK_STOCK_REFRESH,
				  _("Refresh display"),
				  QUERY_ACTION_REFRESH, cb, data, G_OBJECT (vbox));


	/* Necessary settings to be set */
	form->priv->actions_area = vbox;
	data_entry_refresh_actions_area (form);

	return vbox;
}

static void
add_stock_pixmap (GtkWidget * button_box, gchar * action,
		  gchar * tip, QueryEnvUserActions action_val,
		  GCallback cb, gpointer data,
		  GObject *collector)
{
	GtkWidget *button, *pix;
	static GtkTooltips *tips = NULL;
	gchar *str;

	button = gtk_button_new ();
	pix = gtk_image_new_from_stock (action, GTK_ICON_SIZE_MENU);
	gtk_container_add (GTK_CONTAINER (button), pix);
	gtk_container_add (GTK_CONTAINER (button_box), button);

	if (!tips)
		tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), button, tip, NULL);
	gtk_tooltips_enable (GTK_TOOLTIPS (tips));

	str = g_strdup_printf ("w%d", action_val);
	g_object_set_data (G_OBJECT (collector), str, button);
	g_free (str);

	g_object_set_data (G_OBJECT (button), "action_id", GUINT_TO_POINTER (action_val));
	g_object_set_data (G_OBJECT (button), "collector", collector);
	
	g_signal_connect (G_OBJECT (button), "clicked", cb, data);
}
