/* data-entry.c
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "data-entry.h"
#include "marshal.h"

static void data_entry_class_init (DataEntryClass * class);
static void data_entry_init (DataEntry * wid);
static void data_entry_finalize (GObject   * object);

static void data_entry_refresh_status_display (DataEntry *de);
static void m_modified (DataEntry * wid);

/* signals */
enum
{
	CONTENTS_MODIF_SIGNAL,
	LAST_SIGNAL
};

static gint data_entry_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
data_entry_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (DataEntryClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) data_entry_class_init,
			NULL,
			NULL,
			sizeof (DataEntry),
			0,
			(GInstanceInitFunc) data_entry_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "DataEntry", &info, 0);
	}
	return type;
}


static void
data_entry_class_init (DataEntryClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	data_entry_signals[CONTENTS_MODIF_SIGNAL] =
		g_signal_new ("contents_modified",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (DataEntryClass, contents_modified),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE, 0);

	class->contents_modified = m_modified;
	object_class->finalize = data_entry_finalize;
}

static void 
m_modified (DataEntry * wid)
{
	data_entry_refresh_display (wid);
}

void       
data_entry_refresh_display (DataEntry *wid)
{
	g_return_if_fail (wid && IS_DATA_ENTRY (wid));
	
	if (wid->fns) {
		/* determine the is_null and is_modified values */
		GdaValue *now;

		now = (wid->fns->value_from_widget) (GTK_WIDGET (wid));
		if (now) {
			wid->value_is_null = gda_value_is_null (now) ? TRUE : FALSE;

			if (wid->orig_value) {
				if (gda_value_is_null (wid->orig_value)) {
					if (gda_value_is_null (now))
						wid->value_is_modified = FALSE;
					else
						wid->value_is_modified = TRUE;
				}
				else {
					if (gda_value_is_null (now))
						wid->value_is_modified = TRUE;
					else 
						wid->value_is_modified = 
							gda_value_compare (now, wid->orig_value) ? TRUE: FALSE;
				}
			}
			else
				wid->value_is_modified = TRUE;
			gda_value_free (now);
		}
		else
			wid->value_is_null = TRUE;

		/* the default value flag comes down if we have modified the entry */
		if (wid->value_is_modified)
			wid->value_is_defaultval = FALSE;
		else {
			if (wid->value_is_defaultval)
				wid->value_is_modified = TRUE;
		}

		data_entry_refresh_status_display (wid);
	}

	g_print ("null: %d, modif: %d\n", wid->value_is_null, wid->value_is_modified);
}

static void
data_entry_init (DataEntry * wid)
{
	wid->top_box = NULL;
	wid->event_box = NULL;
	wid->children = NULL;

	wid->query_field = NULL;
	wid->fns = NULL;
	wid->is_default_possible = FALSE;
	wid->value_required = FALSE;

	wid->orig_type = GDA_VALUE_TYPE_UNKNOWN;
	wid->orig_value = NULL;
	wid->value_is_null = FALSE;
	wid->value_is_defaultval = FALSE;
	wid->value_is_modified = FALSE;
}

static gint event_cb (GtkWidget *widget, GdkEvent *event, DataEntry *de);
GtkWidget *
data_entry_new ()
{
	GObject   *obj;
	DataEntry *de;
	GtkWidget *event, *hbox, *vbox;

	obj = g_object_new (DATA_ENTRY_TYPE, NULL);
	gtk_container_set_border_width (GTK_CONTAINER (obj), 0);

	de = DATA_ENTRY (obj);
	de->orig_style = gtk_style_copy (gtk_widget_get_style (GTK_WIDGET (obj)));

	/* eventbox */
	event = gtk_event_box_new ();
	de->event_box = event;
	gtk_container_set_border_width (GTK_CONTAINER (event), 0);	
	gtk_box_pack_start (GTK_BOX (de), event, TRUE, TRUE, 0); 
	g_signal_connect (G_OBJECT (event), "event",
			  G_CALLBACK (event_cb), de);

	/* hbox */
	hbox = gtk_hbox_new (FALSE, 0); 
	gtk_container_add (GTK_CONTAINER (event), hbox);

	/* vbox */
	vbox = gtk_vbox_new (FALSE, 0); 
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 3); 
	de->top_box = vbox;

	return GTK_WIDGET (obj);
}


static void mitem_to_null_activated_cb (GtkWidget *mitem, DataEntry *de);
static void mitem_to_defval_activated_cb (GtkWidget *mitem, DataEntry *de);
static gint
event_cb (GtkWidget *widget, GdkEvent *event, DataEntry *de)
{
	gboolean done = FALSE;

	if (event->type == GDK_BUTTON_PRESS) {
		GdkEventButton *bevent = (GdkEventButton *) event; 
		if (bevent->button == 3) {
			GtkWidget *menu, *mitem;
			gchar *str, *name = NULL;
			gboolean nullval = FALSE;
			gboolean defval = de->is_default_possible;
			gint nb_opts = 0;

			/* which menu do we display? */
			if (!de->value_required) {
				if ((de->query_field && !de->value_is_null) ||
				    (de->value_is_null && de->value_is_defaultval)) {
					name = de->query_field->name;
					
					if (de->query_field->field_type == QUERY_FIELD_FIELD) {
						DbField *field;
						field = query_field_field_get_db_field (de->query_field);
						if (field) 
							nullval = field->null_allowed;
					}
				}
			}

			if (defval && de->value_is_defaultval)
				defval = FALSE;

			/* build the menu */
			menu = gtk_menu_new ();
			if (nullval) {
				if (name)
					str = g_strdup_printf (_("Set '%s' to NULL"), name);
				else
					str = g_strdup (_("Set to NULL"));

				mitem = gtk_menu_item_new_with_label (str);
				gtk_widget_show (mitem);
				g_signal_connect (G_OBJECT (mitem), "activate",
						  G_CALLBACK (mitem_to_null_activated_cb), de);
				gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
				g_free (str);
				nb_opts++;
			}

			if (defval) {
				if (name)
					str = g_strdup_printf (_("Set '%s' to default value"), name);
				else
					str = g_strdup (_("Set to default value"));

				mitem = gtk_menu_item_new_with_label (str);
				gtk_widget_show (mitem);
				g_signal_connect (G_OBJECT (mitem), "activate",
						  G_CALLBACK (mitem_to_defval_activated_cb), de);
				gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
				g_free (str);
				nb_opts++;
			}

			if (!nb_opts) {
				if (name)
					str = g_strdup_printf (_("No possible operation for '%s'"), name);
				else
					str = g_strdup (_("No possible operation"));
				mitem = gtk_menu_item_new_with_label (str);
				g_free (str);
				gtk_widget_show (mitem);
				gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
			}
			
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
					bevent->button, bevent->time);
			done = TRUE;
		}
	}
	
	return done;
}

static void 
mitem_to_null_activated_cb (GtkWidget *mitem, DataEntry *de)
{
	if (de->fns) {
		GdaValue *value;

		de->value_is_defaultval = FALSE;
		value = gda_value_new_null ();
		(*de->fns->widget_update)(GTK_WIDGET (de), value, FALSE);
		gda_value_free (value);
	}
}

static void
mitem_to_defval_activated_cb (GtkWidget *mitem, DataEntry *de)
{
	de->value_is_defaultval = TRUE;
	data_entry_refresh_status_display (de);
}



void
data_entry_set_orig_value (DataEntry *de, const GdaValue *value)
{
	g_return_if_fail (de && IS_DATA_ENTRY (de));

	if (de->orig_value) {
		gda_value_free (de->orig_value);
		de->orig_value = NULL;
		de->orig_type = GDA_VALUE_TYPE_UNKNOWN;
	}
	
	if (value) {
		de->orig_value = gda_value_copy (value);
		de->value_is_modified = FALSE;
		if (gda_value_is_null (value))
			de->value_is_null = TRUE;
		else
			de->value_is_null = FALSE;
		de->orig_type = gda_value_get_type (value);
	}
	else {
		/* failsafe defaults */
		de->value_is_null = TRUE;
		de->value_is_modified = FALSE;
		de->orig_type = GDA_VALUE_TYPE_NULL;
	}
}

static void
data_entry_finalize (GObject   * object)
{
	DataEntry *de;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_DATA_ENTRY (object));

	de = DATA_ENTRY (object);

	if (de->orig_value) {
		gda_value_free (de->orig_value);
		de->orig_value = NULL;
	}

	if (de->children) {
		g_slist_free (de->children);
		de->children = NULL;
	}

	if (de->orig_style) {
		g_free (de->orig_style);
		de->orig_style = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}


static void 
data_entry_refresh_status_display (DataEntry *de)
{
	GdkColor *color, *color2 = NULL;

	g_return_if_fail (de && IS_DATA_ENTRY (de));

	color = & (de->orig_style->bg[GTK_STATE_NORMAL]);

	if (de->value_is_null) {
		color2 = g_new0 (GdkColor, 1);
		gdk_color_parse ("#00cd66", color2); /* kind of green color */
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color2, FALSE, TRUE)) {
			g_free (color2);
			color2 = NULL;
			color = & (de->orig_style->bg[GTK_STATE_NORMAL]);
		}
		else
			color = color2;
	}

	if (de->value_is_defaultval) {
		if (color2)
			g_free (color2);
		color2 = g_new0 (GdkColor, 1);
		gdk_color_parse ("#6495ed", color2); /* kind of blue color */
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color2, FALSE, TRUE)) {
			g_free (color2);
			color2 = NULL;
			color = & (de->orig_style->bg[GTK_STATE_NORMAL]);
		}
		else
			color = color2;
	}

	if (de->value_required && data_entry_is_value_null (de)) { /* WARNING, we need a value */
		if (color2)
			g_free (color2);
		color2 = g_new0 (GdkColor, 1);
		gdk_color_parse ("#ff6a6a", color2); /* kind of red color */
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color2, FALSE, TRUE)) {
			g_free (color2);
			color2 = NULL;
			color = & (de->orig_style->bg[GTK_STATE_NORMAL]);
		}
		else
			color = color2;
	}

	gtk_widget_modify_bg (de->event_box, GTK_STATE_NORMAL, color);
	if (color2)
		g_free (color2);
}

void
data_entry_pack_default (DataEntry * de, GtkWidget * wid)
{
	gtk_box_pack_start (GTK_BOX (de->top_box), wid, TRUE, TRUE, 7);
}

GdaValue *
data_entry_get_gdavalue (DataEntry * de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), NULL);
	g_return_val_if_fail (de->fns, NULL);

	return (de->fns->value_from_widget) (GTK_WIDGET (de));
}

gboolean
data_entry_is_value_null (DataEntry * de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), FALSE);

	return de->value_is_null;
}
