/* mg-work-widget.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "mg-base.h"
#include "mg-conf.h"
#include "mg-context.h"
#include "mg-qfield.h"
#include "mg-entity.h"
#include "mg-parameter.h"
#include "mg-work-widget.h"


/* signals */
enum
{
	LAST_SIGNAL
};

static gint mg_work_widget_signals[LAST_SIGNAL] = { };

static void mg_work_widget_iface_init (gpointer g_class);

GType
mg_work_widget_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgWorkWidgetIface),
			(GBaseInitFunc) mg_work_widget_iface_init,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) NULL,
			NULL,
			NULL,
			0,
			0,
			(GInstanceInitFunc) NULL
		};
		
		type = g_type_register_static (G_TYPE_INTERFACE, "MgWorkWidget", &info, 0);
		g_type_interface_add_prerequisite (type, G_TYPE_OBJECT);
	}
	return type;
}


static void
mg_work_widget_iface_init (gpointer g_class)
{
	static gboolean initialized = FALSE;

	if (! initialized) {
		initialized = TRUE;
	}
}


/**
 * mg_work_widget_run
 * @iface: an object which implements the #MgWorkWidget interface
 * @mode: an OR'ed collection of flags from #MgEnumsMode
 * 
 * Makes the widget run the queries it has to run in order to display
 * usefull data. Havin a separate step to "run" the widget allows to
 * have the execution context related to other events.
 *
 * If mode==0, then the @mode argument is ignored and the previous mode is unchanged.
 */
void
mg_work_widget_run (MgWorkWidget *iface, guint mode)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (mode & MG_ACTION_MODIF_COMMIT_IMMEDIATE)
		mode = mode | MG_ACTION_MODIF_AUTO_COMMIT;

	if (MG_WORK_WIDGET_GET_IFACE (iface)->run)
		(MG_WORK_WIDGET_GET_IFACE (iface)->run) (iface, mode);
}

/**
 * mg_work_widget_set_mode
 * @iface: an object which implements the #MgWorkWidget interface
 * @mode: an OR'ed collection of flags from #MgEnumsMode
 * 
 * Changes the mode that was set when mg_work_widget_run() was called.
 */
void
mg_work_widget_set_mode (MgWorkWidget *iface, guint mode)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (mode & MG_ACTION_MODIF_COMMIT_IMMEDIATE)
		mode = mode | MG_ACTION_MODIF_AUTO_COMMIT;

	if (MG_WORK_WIDGET_GET_IFACE (iface)->set_mode)
		(MG_WORK_WIDGET_GET_IFACE (iface)->set_mode) (iface, mode);
}

/**
 * mg_work_widget_entry_set_editable
 * @iface: an object which implements the #MgWorkWidget interface
 * @field: a #MgQfield object which belongs to the #MgQuery which @iface uses
 * @editable:
 *
 * Sets if the data entry in the @iface widget corresponding to @field can be edited or not.
 */
void 
mg_work_widget_entry_set_editable (MgWorkWidget *iface, MgQfield *field, gboolean editable)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (MG_WORK_WIDGET_GET_IFACE (iface)->set_entry_editable)
		(MG_WORK_WIDGET_GET_IFACE (iface)->set_entry_editable) (iface, field, editable);	
}

/**
 * mg_work_widget_entry_show_actions
 * @iface: an object which implements the #MgWorkWidget interface
 * @field: a #MgQfield object which belongs to the #MgQuery which @iface uses
 * @show_actions:
 * 
 * Controls if #MgDataEntry widget used to manipulate the data corresponding to the @field field
 * must show its actions or not. If @field is %NULL, then the @show_actions flag applies for
 * all the fields.
 */
void
mg_work_widget_entry_show_actions (MgWorkWidget *iface, MgQfield *field, gboolean show_actions)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (MG_WORK_WIDGET_GET_IFACE (iface)->show_entry_actions)
		(MG_WORK_WIDGET_GET_IFACE (iface)->show_entry_actions) (iface, field, show_actions);
}


/**
 * mg_work_widget_alldata_show_actions
 * @iface: an object which implements the #MgWorkWidget interface
 * @show_actions:
 * 
 * Controls if the action buttons provided by the @iface widget (for instance SAVE, RESET, MOVE TO NEXT, etc)
 * are displayed or not.
 */
void
mg_work_widget_alldata_show_actions (MgWorkWidget *iface, gboolean show_actions)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (MG_WORK_WIDGET_GET_IFACE (iface)->show_global_actions)
		(MG_WORK_WIDGET_GET_IFACE (iface)->show_global_actions) (iface, show_actions);
}

/**
 * mg_work_widget_get_param_for_field_exec
 * @iface: an object which implements the #MgWorkWidget interface
 * @field: a #MgQfield object which belongs to the #MgQuery which @iface uses
 * 
 * Get the #MgParameter object corresponding to @field, in the case @field represents
 * a parameter for the @iface widget (that is a modification of the value of a returned value
 * will provoque a data refresh in the @iface widget).
 * 
 * Returns:
 */
MgParameter *
mg_work_widget_get_param_for_field_exec (MgWorkWidget *iface, MgQfield *field)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), NULL);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field) (iface, field, NULL, TRUE);
	return NULL;
}

/**
 * mg_work_widget_get_param_for_field_sql_exec
 * @iface: an object which implements the #MgWorkWidget interface
 * @field_name: the name of a #MgQfield object which belongs to the #MgQuery which @iface uses
 * 
 * Get the #MgParameter object corresponding to @field_name, in the case @field_name represents
 * a parameter for the @iface widget (that is a modification of the value of a returned value
 * will provoque a data refresh in the @iface widget).
 * 
 * Returns:
 */
MgParameter *
mg_work_widget_get_param_for_field_sql_exec (MgWorkWidget *iface, const gchar *field_name)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), NULL);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field) (iface, NULL, field_name, TRUE);
	return NULL;
}

/**
 * mg_work_widget_get_param_for_field_data
 * @iface: an object which implements the #MgWorkWidget interface
 * @field: a #MgQfield object which belongs to the #MgQuery which @iface uses
 * 
 * Get the #MgParameter object corresponding to @field, in the case @field
 * is a data which is manipulated by the widget, and which can be selected 
 * by the user.
 *
 * Returns: the requested #MgParameter, or %NULL
 */
MgParameter *
mg_work_widget_get_param_for_field_data (MgWorkWidget *iface, MgQfield *field)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), NULL);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field) (iface, field, NULL, FALSE);
	return NULL;
}

/**
 * mg_work_widget_get_param_for_field_sql_data
 * @iface: an object which implements the #MgWorkWidget interface
 * @field_name: the name of a #MgQfield object which belongs to the #MgQuery which @iface uses
 * 
 * Get the #MgParameter object corresponding to @field_name, in the case @field
 * is a data which is manipulated by the widget, and which can be selected 
 * by the user.
 *
 * Returns: the requested #MgParameter, or %NULL
 */
MgParameter *
mg_work_widget_get_param_for_field_sql_data (MgWorkWidget *iface, const gchar *field_name)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), NULL);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field) (iface, NULL, field_name, FALSE);
	return NULL;
}

/**
 * mg_work_widget_has_been_changed
 * @iface: an object which implements the #MgWorkWidget interface
 * 
 * Tells if the user has changed any of the data displayed by @iface since
 * the last refresh.
 *
 * Returns: TRUE if @iface has been changed
 */
gboolean
mg_work_widget_has_been_changed (MgWorkWidget *iface)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), FALSE);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->has_been_changed)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->has_been_changed) (iface);
	return FALSE;
}

/**
 * mg_work_widget_is_exec_context_valid
 * @iface: an object which implements the #MgWorkWidget interface
 * 
 * Checks if the @iface widget can be run, that is if no parameter is
 * missing in the queries that it needs to run.
 *
 * Returns: TRUE if the exec context is valid
 */
gboolean
mg_work_widget_is_exec_context_valid (MgWorkWidget *iface)
{
	MgContext *context = NULL;
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), TRUE);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->get_exec_context)
		context = (MG_WORK_WIDGET_GET_IFACE (iface)->get_exec_context) (iface);

	if (context)
		return mg_context_is_valid (context);
	else
		return TRUE;
}

/**
 * mg_work_widget_get_actions_group
 * @iface: an object which implements the #MgWorkWidget interface
 *
 * Each widget imlplementing the #MgWorkWidget interface provides actions. Actions can be triggered
 * using the mg_work_widget_perform_action() method, but using this method allows for the creation of
 * toolbars, menus, etc calling these actions.
 *
 * The actions are among: 
 * <itemizedlist><listitem><para>Data edition actions: "WorkWidgetNew", "MgWorkWidgetCommit", 
 *    "WorkWidgetDelete, "WorkWidgetUndelete, "WorkWidgetReset", </para></listitem>
 * <listitem><para>Record by record moving: "WorkWidgetFirstRecord", "WorkWidgetPrevRecord", 
 *    "WorkWidgetNextRecord", "WorkWidgetLastRecord",</para></listitem>
 * <listitem><para>Chuncks of records moving: "WorkWidgetFirstChunck", "WorkWidgetPrevChunck", 
 *     "WorkWidgetNextChunck", "WorkWidgetLastChunck".</para></listitem></itemizedlist>
 * 
 * Returns: the #GtkActionGroup with all the possible actions on the widget.
 */
GtkActionGroup *
mg_work_widget_get_actions_group (MgWorkWidget *iface)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), NULL);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->get_actions_group)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->get_actions_group) (iface);
	return NULL;
}

/**
 * mg_work_widget_perform_action
 * @iface: an object which implements the #MgWorkWidget interface
 * @action: a #MgAction action
 *
 * Forces the widget to perform the selected @action, as if the user
 * had pressed on the corresponding action button in the @iface widget,
 * if the corresponding action is possible and if the @iface widget
 * supports the action.
 */
void
mg_work_widget_perform_action (MgWorkWidget *iface, MgAction action)
{
	gchar *action_name = NULL;
	GtkActionGroup *group;
	GtkAction *gtkaction;

	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));
	group = mg_work_widget_get_actions_group (iface);
	if (!group) {
		g_warning ("Object class %s does not support the mg_work_widget_get_actions_group() method",
			   G_OBJECT_TYPE_NAME (iface));
		return;
	}
	
	switch (action) {
	case MG_ACTION_NEW_DATA:
		action_name = "WorkWidgetNew";
		break;
	case MG_ACTION_WRITE_MODIFIED_DATA:
		action_name = "WorkWidgetCommit";
		break;
        case MG_ACTION_DELETE_SELECTED_DATA:
		action_name = "WorkWidgetDelete";
		break;
        case MG_ACTION_UNDELETE_SELECTED_DATA:
		action_name = "WorkWidgetUndelete";
		break;
        case MG_ACTION_RESET_DATA:
		action_name = "WorkWidgetReset";
		break;
        case MG_ACTION_MOVE_FIRST_RECORD:
		action_name = "WorkWidgetFirstRecord";
		break;
        case MG_ACTION_MOVE_PREV_RECORD:
		action_name = "WorkWidgetPrevRecord";
		break;
        case MG_ACTION_MOVE_NEXT_RECORD:
		action_name = "WorkWidgetNextRecord";
		break;
        case MG_ACTION_MOVE_LAST_RECORD:
		action_name = "WorkWidgetLastRecord";
		break;
        case MG_ACTION_MOVE_FIRST_CHUNCK:
		action_name = "WorkWidgetFirstChunck";
		break;
        case MG_ACTION_MOVE_PREV_CHUNCK:
		action_name = "WorkWidgetPrevChunck";
		break;
        case MG_ACTION_MOVE_NEXT_CHUNCK:
		action_name = "WorkWidgetNextChunck";
		break;
        case MG_ACTION_MOVE_LAST_CHUNCK:
		action_name = "WorkWidgetLastChunck";
		break;
	default:
		g_assert_not_reached ();
	}

	gtkaction = gtk_action_group_get_action (group, action_name);
	if (gtkaction)
		gtk_action_activate (gtkaction);
}


/**
 * mg_work_widget_bind_to_work_widget_xml
 * @dest_iface: an object which implements the #MgWorkWidget interface
 * @dest_field_xml_id:
 * @source_iface: an object which implements the #MgWorkWidget interface
 * @source_field_xml_id:
 *
 * Retreives the #MgParameter objects for the @dest_field_xml_id and @source_field_xml_id
 * and bind them together: when the 'source' parameter changes, the 'dest' one takes
 * the value of the 'source' one.
 */
void
mg_work_widget_bind_to_work_widget_xml (MgWorkWidget *dest_iface, const gchar *dest_field_xml_id, 
				    MgWorkWidget *source_iface, const gchar *source_field_xml_id)
{
	gchar *str, *ptr, *tok;
	MgQuery  *query_dest, *query_src;
	MgParameter *param_dest, *param_src;
	MgContext *context = NULL;
	MgConf *conf;
	MgQfield *field;

	g_return_if_fail (dest_iface && IS_MG_WORK_WIDGET (dest_iface));
	g_return_if_fail (source_iface && IS_MG_WORK_WIDGET (source_iface));
	g_return_if_fail (dest_field_xml_id && *dest_field_xml_id);
	g_return_if_fail (source_field_xml_id && *source_field_xml_id);
	
	/* find the destination parameter:
	 * it MUST be in the Exec context of the SELECT query used by the widget, 
	 * otherwise it's stupid and dangerous.
	 */
	if (MG_WORK_WIDGET_GET_IFACE (dest_iface)->get_exec_context)
		context = (MG_WORK_WIDGET_GET_IFACE (dest_iface)->get_exec_context) (dest_iface);
	else {
		g_warning ("The object of class %s does not support any execution context!", 
			   G_OBJECT_TYPE_NAME (dest_iface));
		return;
	}
	conf = mg_base_get_conf (MG_BASE (context));
	str = g_strdup (dest_field_xml_id);
	ptr = strtok_r (str, ":", &tok);
	g_return_if_fail (ptr);
	query_dest = mg_conf_get_query_by_xml_id (conf, ptr);
	g_free (str);
	g_return_if_fail (query_dest);
	field = MG_QFIELD (mg_entity_get_field_by_xml_id (MG_ENTITY (query_dest), dest_field_xml_id));
	g_return_if_fail (field);
	param_dest = mg_work_widget_get_param_for_field_exec (dest_iface, field);
	g_return_if_fail (param_dest);
	g_return_if_fail (g_slist_find (context->parameters, param_dest));

	/* find the source parameter */
	if (MG_WORK_WIDGET_GET_IFACE (source_iface)->get_exec_context)
		context = (MG_WORK_WIDGET_GET_IFACE (source_iface)->get_exec_context) (source_iface);
	else {
		g_warning ("The object of class %s does not support any execution context!", 
			   G_OBJECT_TYPE_NAME (source_iface));
		return;
	}
	conf = mg_base_get_conf (MG_BASE (context));
	str = g_strdup (source_field_xml_id);
	ptr = strtok_r (str, ":", &tok);
	g_return_if_fail (ptr);
	query_src = mg_conf_get_query_by_xml_id (conf, ptr);
	g_free (str);
	g_return_if_fail (query_src);
	field = MG_QFIELD (mg_entity_get_field_by_xml_id (MG_ENTITY (query_src), source_field_xml_id));
	g_return_if_fail (field);
	param_src = mg_work_widget_get_param_for_field_data (source_iface, field);
	g_return_if_fail (param_src);

	/* bind the two parameters */
	mg_parameter_bind_to_param (param_dest, param_src);
}

/**
 * mg_work_widget_bind_to_work_widget_sql
 * @dest_iface: an object which implements the #MgWorkWidget interface
 * @dest_field_name:
 * @source_iface: an object which implements the #MgWorkWidget interface
 * @source_field_name:
 *
 * Retreives the #MgParameter objects for the @dest_field_name and @source_field_name
 * and bind them together: when the 'source' parameter changes, the 'dest' one takes
 * the value of the 'source' one.
 *
 * Returns: FALSE if an error occured
 */
gboolean
mg_work_widget_bind_to_work_widget_sql (MgWorkWidget *dest_iface, const gchar *dest_field_name, 
					MgWorkWidget *source_iface, const gchar *source_field_name)
{
	MgParameter *param_dest, *param_src;
	MgContext *context = NULL;

	g_return_val_if_fail (dest_iface && IS_MG_WORK_WIDGET (dest_iface), FALSE);
	g_return_val_if_fail (source_iface && IS_MG_WORK_WIDGET (source_iface), FALSE);
	g_return_val_if_fail (dest_field_name && *dest_field_name, FALSE);
	g_return_val_if_fail (source_field_name && *source_field_name, FALSE);
	
	/* find the destination parameter:
	 * it MUST be in the Exec context of the SELECT query used by the widget, 
	 * otherwise it's stupid and dangerous.
	 */
	if (MG_WORK_WIDGET_GET_IFACE (dest_iface)->get_exec_context)
		context = (MG_WORK_WIDGET_GET_IFACE (dest_iface)->get_exec_context) (dest_iface);
	else {
		g_warning ("The object of class %s does not support any execution context!", 
			   G_OBJECT_TYPE_NAME (dest_iface));
		return FALSE;
	}

	param_dest = mg_work_widget_get_param_for_field_sql_exec (dest_iface, dest_field_name);
	g_return_val_if_fail (param_dest, FALSE);
	g_return_val_if_fail (g_slist_find (context->parameters, param_dest), FALSE);

	/* find the source parameter */
	if (MG_WORK_WIDGET_GET_IFACE (source_iface)->get_exec_context)
		context = (MG_WORK_WIDGET_GET_IFACE (source_iface)->get_exec_context) (source_iface);
	else {
		g_warning ("The object of class %s does not support any execution context!", 
			   G_OBJECT_TYPE_NAME (source_iface));
		return FALSE;
	}
	param_src = mg_work_widget_get_param_for_field_sql_data (source_iface, source_field_name);
	g_return_val_if_fail (param_src, FALSE);

	/* bind the two parameters */
	mg_parameter_bind_to_param (param_dest, param_src);
}


/**
 * mg_work_widget_bind_to_context
 * @dest_iface: an object which implements the #MgWorkWidget interface
 * @dest_field_xml_id:
 * @source_context: a #MgContext object
 * @source_field_xml_id:
 *
 * Retreives the #MgParameter objects for the @dest_field_xml_id and @source_field_xml_id
 * and bind them together: when the 'source' parameter changes, the 'dest' one takes
 * the value of the 'source' one.
 */
void
mg_work_widget_bind_to_context (MgWorkWidget *dest_iface, const gchar *dest_field_xml_id,
				MgContext *source_context, const gchar *source_field_xml_id)
{
	gchar *str, *ptr, *tok;
	MgQuery  *query_dest, *query_src;
	MgParameter *param_dest, *param_src;
	MgContext *context;
	MgConf *conf;
	MgQfield *field;

	g_return_if_fail (dest_iface && IS_MG_WORK_WIDGET (dest_iface));
	g_return_if_fail (source_context && IS_MG_CONTEXT (source_context));
	g_return_if_fail (dest_field_xml_id && *dest_field_xml_id);
	g_return_if_fail (source_field_xml_id && *source_field_xml_id);
	
	/* find the destination parameter:
	 * it MUST be in the Exec context of the SELECT query used by the widget, 
	 * otherwise it's stupid and dangerous.
	 */
	if (MG_WORK_WIDGET_GET_IFACE (dest_iface)->get_exec_context)
		context = (MG_WORK_WIDGET_GET_IFACE (dest_iface)->get_exec_context) (dest_iface);
	else {
		g_warning ("The object of class %s does not support any execution context!", 
			   G_OBJECT_TYPE_NAME (dest_iface));
		return;
	}
	conf = mg_base_get_conf (MG_BASE (context));
	str = g_strdup (dest_field_xml_id);
	ptr = strtok_r (str, ":", &tok);
	g_return_if_fail (ptr);
	query_dest = mg_conf_get_query_by_xml_id (conf, ptr);
	g_free (str);
	g_return_if_fail (query_dest);
	field = MG_QFIELD (mg_entity_get_field_by_xml_id (MG_ENTITY (query_dest), dest_field_xml_id));
	g_return_if_fail (field);
	param_dest = mg_work_widget_get_param_for_field_exec (dest_iface, field);
	g_return_if_fail (param_dest);
	g_return_if_fail (g_slist_find (context->parameters, param_dest));

	/* find the source parameter */
	conf = mg_base_get_conf (MG_BASE (source_context));
	str = g_strdup (source_field_xml_id);
	ptr = strtok_r (str, ":", &tok);
	g_return_if_fail (ptr);
	query_src = mg_conf_get_query_by_xml_id (conf, ptr);
	g_free (str);
	g_return_if_fail (query_src);
	field = MG_QFIELD (mg_entity_get_field_by_xml_id (MG_ENTITY (query_src), source_field_xml_id));
	g_return_if_fail (field);
	param_src = mg_context_find_parameter_for_field (source_context, field);
	g_return_if_fail (param_src);

	/* bind the two parameters */
	mg_parameter_bind_to_param (param_dest, param_src);
}

/**
 * mg_work_widget_bind_to_context_all
 * @dest_iface: an object which implements the #MgWorkWidget interface
 * @source_context: a #MgContext object
 *
 * For each #MgParameter of @source_context, retreives the corresponding #MgParameter 
 * in @dest_iface's exec. context, and bind the two together:
 * when the 'source' parameter changes, the 'dest' one takes
 * the value of the 'source' one.
 */
void
mg_work_widget_bind_to_context_all (MgWorkWidget *dest_iface, MgContext *source_context)
{
	GSList *list, *dest_fields;
	MgParameter *param_dest, *param_src;

	g_return_if_fail (dest_iface && IS_MG_WORK_WIDGET (dest_iface));
	g_return_if_fail (source_context && IS_MG_CONTEXT (source_context));

	list = source_context->parameters;
	while (list) {
		param_src = MG_PARAMETER (list->data);
		dest_fields = mg_parameter_get_dest_fields (param_src);
		param_dest = NULL;
		while (dest_fields && !param_dest) {
			param_dest = mg_work_widget_get_param_for_field_exec (dest_iface, MG_QFIELD (dest_fields->data));
			dest_fields = g_slist_next (dest_fields);
		}

		if (param_dest) 
			/* bind the two parameters */
			mg_parameter_bind_to_param (param_dest, param_src);
		
		list = g_slist_next (list);
	}
}
