/* mainpageenv.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <libgda/gda-data-model-array.h>
#include <libgnomedb/gnome-db-grid.h>
#include "mainpageenv.h"
#include "query-env.h"
#include "query-env-editor.h"

static void main_page_env_class_init (MainPageEnvClass * class);
static void main_page_env_init (MainPageEnv * wid);
static void main_page_env_finalize (GObject   * object);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

typedef struct
{
	ConfManager *conf;
	QueryEnv    *env;
}
Row_Data;

/*
 * static functions 
 */
static void selection_made (GnomeDbGrid * wid, gint row, MainPageEnv *mpe);
static void selection_unmade (GnomeDbGrid * wid, MainPageEnv *mpe);
static void remove_env_cb (GObject   * obj, MainPageEnv *mpe);
static void create_env_cb (GObject   * obj, MainPageEnv *mpe);
static void properties_env_cb (GObject   * obj, MainPageEnv *mpe);

static void main_page_env_add_cb (GObject   * obj, QueryEnv * env, MainPageEnv *mpe);
static void main_page_env_drop_cb (GObject   * obj, QueryEnv * env, MainPageEnv *mpe);

guint
main_page_env_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MainPageEnvClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) main_page_env_class_init,
			NULL,
			NULL,
			sizeof (MainPageEnv),
			0,
			(GInstanceInitFunc) main_page_env_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "MainPageEnv", &info, 0);
	}

	return type;
}

static void
main_page_env_class_init (MainPageEnvClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->finalize = main_page_env_finalize;
}



static void
main_page_env_init (MainPageEnv * wid)
{
	GtkWidget *bb;
	gint i;
	GdaDataModel *model;

	/* Toolbar */
	bb = gnome_db_new_toolbar_widget (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_box_pack_start (GTK_BOX (wid), bb, FALSE, TRUE, GNOME_PAD / 2);
	gtk_widget_show (bb);

	wid->new_env = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_NEW,
						 _("Create new"), NULL,
						 GTK_SIGNAL_FUNC (create_env_cb), wid, -1); 
	wid->properties = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_PROPERTIES,
						    _("View properties"), NULL,
						    GTK_SIGNAL_FUNC (properties_env_cb), wid, -1); 
	wid->remove_env = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_DELETE,
						    _("Delete selected"), NULL,
						    GTK_SIGNAL_FUNC (remove_env_cb), wid, -1);

	/* list */
	model = gda_data_model_array_new (3);
	gda_data_model_set_column_title (model, 0, _("Form/Grid Name"));
	gda_data_model_set_column_title (model, 1, _("Form/Grid Description"));
	gda_data_model_set_column_title (model, 2, _("Associated query"));

	wid->list = gnome_db_grid_new_with_model (model);
	gtk_box_pack_start (GTK_BOX (wid), wid->list, TRUE, TRUE, GNOME_PAD);
	gtk_widget_show (wid->list);
	g_signal_connect (G_OBJECT (wid->list), "row_selected",
			    G_CALLBACK (selection_made), wid);
	g_signal_connect (G_OBJECT (wid->list), "selection_cleared",
			    G_CALLBACK (selection_unmade), wid);

	wid->sel_row = -1;
}


GtkWidget *
main_page_env_new (ConfManager * conf)
{
	GObject   *obj;
	MainPageEnv *wid;

	obj = g_object_new (MAIN_PAGE_ENV_TYPE, NULL);
	wid = MAIN_PAGE_ENV (obj);
	wid->conf = conf;
	wid->queries = NULL;

	g_signal_connect (G_OBJECT (conf), "env_added",
			  G_CALLBACK (main_page_env_add_cb), wid);

	g_signal_connect (G_OBJECT (conf), "env_removed",
			  G_CALLBACK (main_page_env_drop_cb), wid);


	gtk_widget_set_sensitive (wid->remove_env, FALSE);
	gtk_widget_set_sensitive (wid->new_env, FALSE);
	gtk_widget_set_sensitive (wid->properties, FALSE);
	conf_manager_register_sensitive_on_connect (wid->conf, GTK_WIDGET (wid->new_env));

	return GTK_WIDGET (obj);
}

static void main_page_env_finalize (GObject   * object)
{
	parent_class->finalize (object);
}

static void
selection_made (GnomeDbGrid * wid, gint row, MainPageEnv *mpe)
{
	mpe->sel_row = row;
	gtk_widget_set_sensitive (mpe->remove_env, TRUE);
	gtk_widget_set_sensitive (mpe->properties, TRUE);
}


static void
selection_unmade (GnomeDbGrid * wid, MainPageEnv *mpe)
{
	mpe->sel_row = -1;
	gtk_widget_set_sensitive (mpe->remove_env, FALSE);
	gtk_widget_set_sensitive (mpe->properties, FALSE);
}

static void remove_env_answer_cb (gint reply, GObject   *obj);
static void
remove_env_cb (GObject   * obj, MainPageEnv *mpe)
{
	Row_Data *rdata = NULL;
	gchar *txt;

	if (mpe->sel_row >= 0)
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpe->list), mpe->sel_row);
	if (rdata) {
		if (rdata->env->name)
			txt = g_strdup_printf (_("Do you really want to remove\n"
						 "the grid/form '%s' for query '%s'?"),
					       rdata->env->name, rdata->env->q->name);
		else
			txt = g_strdup_printf (_("Do you really want to remove\n"
						 "the grid/form (unnamed) for query '%s'?"),
					       rdata->env->q->name);
		g_object_set_data (obj, "env", rdata->env);
		gnome_question_dialog (txt, (GnomeReplyCallback)
				       remove_env_answer_cb, obj);
		g_free (txt);
	}
}

static void
remove_env_answer_cb (gint reply, GObject   * obj)
{
	QueryEnv *env;

	env = QUERY_ENV (g_object_get_data (obj, "env"));

	if (reply == 0) 
		query_del_env (env->q, G_OBJECT (env));
}


static void 
create_env_cb (GObject   * obj, MainPageEnv *mpe)
{
	g_print("\n\nCreation NOT YET DONE\n\n");
}

static void properties_env_cb (GObject   * obj, MainPageEnv *mpe)
{
	Row_Data *rdata = NULL;

	if (mpe->sel_row >= 0)
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpe->list), mpe->sel_row);
	if (rdata) {
		/* FIXME */
		GtkWidget *dlg;	
		dlg = query_env_editor_get_in_dialog (rdata->env);
		gtk_widget_show (dlg);
	}
}



static void names_changed_cb (GObject   *obj, MainPageEnv *mpe);
static void
main_page_env_add_cb (GObject   * obj, QueryEnv * env, MainPageEnv *mpe)
{
	gint i = 0;
	Row_Data *rdata;
	GList *value_list = NULL;
	GdaDataModel *model;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (mpe->list));

	value_list = g_list_append (value_list, gda_value_new_string (""));
	value_list = g_list_append (value_list, gda_value_new_string (""));
	value_list = g_list_append (value_list, gda_value_new_string (""));

	rdata = g_new0 (Row_Data, 1);
	rdata->conf = mpe->conf;
	rdata->env = env;
	gnome_db_grid_set_row_data (GNOME_DB_GRID (mpe->list), i, rdata);
	names_changed_cb (G_OBJECT (env->q), mpe);
	

	/* Signal from QueryEnv */
	g_signal_connect (G_OBJECT (env), "name_changed",
			  G_CALLBACK (names_changed_cb), mpe);

	/* Signal from Query */
	if (! g_slist_find (mpe->queries, env->q)) {
		mpe->queries = g_slist_append (mpe->queries, env->q);

		g_signal_connect (G_OBJECT (env->q), "name_changed",
				  G_CALLBACK (names_changed_cb), mpe);
	}
}

static void
main_page_env_drop_cb (GObject   * obj, QueryEnv * env, MainPageEnv *mpe)
{
	gint i = 0, rows;
	gboolean found = FALSE;
	Row_Data *rdata;
	GdaDataModel *model;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (mpe->list));
	rows = gda_data_model_get_n_rows (model);

	while ((i < rows) && !found) {
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpe->list), i);
		if (rdata->env == env) {
			gint j = 0, nbq = 0;
			Row_Data *rdata2;
			
			/* How many times does the Query represented by QueryEnv appear in the list ? */
			while (j < rows) {
				rdata2 = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpe->list), j);
				if ((rdata != rdata2) && (rdata2->env->q == rdata->env->q))
					nbq++;
				j++;
			}
			if (! nbq) {
				mpe->queries = g_slist_remove (mpe->queries, rdata->env->q);
				g_signal_handlers_disconnect_by_func (G_OBJECT (rdata->env->q), 
								      G_CALLBACK (names_changed_cb), mpe);
			}

			/* removing the list entry */
			found = TRUE;
			g_free (rdata);
			gda_data_model_remove_row (model, gda_data_model_get_row (model, i));
			if (mpe->sel_row == i) {
				mpe->sel_row = -1;
				gtk_widget_set_sensitive (mpe->remove_env, FALSE);
			}
		}
		i++;
	}
}

static void 
names_changed_cb (GObject   *obj, MainPageEnv *mpe)
{
	gint i = 0, rows;
	Row_Data *rdata;
	Query *q;
	GdaDataModel *model;

	if (IS_QUERY_ENV (obj))
		q = QUERY_ENV (obj)->q;
	else
		q = QUERY (obj);

	g_assert (q);
	model = gnome_db_grid_get_model (GNOME_DB_GRID (mpe->list));
	rows = gda_data_model_get_n_rows (model);
	
	while (i < rows) {
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mpe->list), i);
		if (rdata->env->q == q) {
			gchar *col[3];

			if (rdata->env->name && *(rdata->env->name))
				gda_value_set_string (gda_data_model_get_value_at (model, 0, i), rdata->env->name);
			else
				gda_value_set_string (gda_data_model_get_value_at (model, 0, i), _("<No name>"));
			gda_value_set_string (gda_data_model_get_value_at (model, 0, i), rdata->env->descr);
			gda_value_set_string (gda_data_model_get_value_at (model, 0, i), q->name);

			gda_data_model_row_updated (model, i);
		}
		i++;
	}
	
}
