/* canvas-query-cond.c
 *
 * Copyright (C) 2002 - 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "canvas-query-cond.h"

static void canvas_query_cond_class_init (CanvasQueryCondClass * class);
static void canvas_query_cond_init (CanvasQueryCond * item);
static void canvas_query_cond_dispose (GObject   * object);

static void canvas_query_cond_bounds (GnomeCanvasItem *item, 
				      double *x1, double *y1, double *x2, double *y2);
static void canvas_query_cond_set_property (GObject              *object,
					    guint                 param_id,
					    const GValue         *value,
					    GParamSpec           *pspec);
static void canvas_query_cond_get_property (GObject              *object,
					    guint                 param_id,
					    GValue               *value,
					    GParamSpec           *pspec);

struct _CanvasQueryCondPrivate
{
	/* objects being represented */
	QueryCond          *cond;
	GSList             *items;

	/* real dimensions of the item */
	double              xsize;
	double              ysize;
};

enum
{
	PROP_0,
	PROP_QUERY_COND
};

/* for the parent object */
static GObjectClass *parent_class = NULL;

guint
canvas_query_cond_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (CanvasQueryCondClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) canvas_query_cond_class_init,
			NULL,
			NULL,
			sizeof (CanvasQueryCond),
			0,
			(GInstanceInitFunc) canvas_query_cond_init
		};	       
		
		type = g_type_register_static (CANVAS_BASE_TYPE, "CanvasQueryCond", &info, 0);
	}

	return type;
}

static void
canvas_query_cond_class_init (CanvasQueryCondClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	GnomeCanvasItemClass *item_class = (GnomeCanvasItemClass *) class;

	parent_class = g_type_class_peek_parent (class);
	object_class->dispose = canvas_query_cond_dispose;
	item_class->bounds = canvas_query_cond_bounds;

	/* Properties */
	object_class->set_property = canvas_query_cond_set_property;
	object_class->get_property = canvas_query_cond_get_property;

	g_object_class_install_property
                (object_class, PROP_QUERY_COND,
                 g_param_spec_pointer ("query_cond", NULL, NULL,
				       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}


static void
canvas_query_cond_init (CanvasQueryCond * cqc)
{
	cqc->priv = g_new0 (CanvasQueryCondPrivate, 1);

	cqc->priv->cond = NULL;
	cqc->priv->items = NULL;

	cqc->priv->xsize = 0.;
	cqc->priv->ysize = 0.;
}

static void cond_destroyed_cb (CanvasQueryCond *cqc, GObject *obj);
static void cond_changed_cb  (GObject *obj, CanvasQueryCond *cqc);
static void
canvas_query_cond_dispose (GObject   * object)
{
	CanvasQueryCond *cqc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_CANVAS_QUERY_COND (object));

	cqc = CANVAS_QUERY_COND (object);

	g_print ("Disposing CQC %p...\n", cqc);
	if (cqc->priv) {
		if (cqc->priv->cond) {
			if (cqc->priv->cond) 
				g_object_weak_unref (G_OBJECT (cqc->priv->cond), 
						     (GWeakNotify) cond_destroyed_cb, cqc);
			
			cqc->priv->cond = NULL;
		}
		g_free (cqc->priv);
		cqc->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
	g_print ("Disposing CQC %p Done.\n", cqc);
}

void
canvas_query_cond_free (CanvasQueryCond *cqc)
{
	g_return_if_fail (cqc && IS_CANVAS_QUERY_COND (cqc));
	gtk_object_destroy (GTK_OBJECT (cqc));
}

static void post_init (CanvasQueryCond * cqc);
static void
canvas_query_cond_set_property (GObject              *object,
				guint                 param_id,
				const GValue         *value,
				GParamSpec           *pspec)
{
	CanvasQueryCond *cqc;
	gpointer ptr;

	cqc = CANVAS_QUERY_COND (object);

	switch (param_id) {
	case PROP_QUERY_COND:
		ptr = g_value_get_pointer (value);
		if (ptr) {
			g_assert (IS_QUERY_COND (ptr));
			cqc->priv->cond = QUERY_COND (ptr);
		}
		break;
	}

	post_init (cqc);
}

static void 
canvas_query_cond_get_property (GObject              *object,
				guint                 param_id,
				GValue               *value,
				GParamSpec           *pspec)
{
	g_print ("Get Property\n");
}


static void redraw_cond_contents (CanvasQueryCond *cqc);
static void 
post_init (CanvasQueryCond * cqc)
{
	if (cqc->priv->cond)
		g_object_weak_ref (G_OBJECT (cqc->priv->cond), 
				   (GWeakNotify) cond_destroyed_cb, cqc);
	redraw_cond_contents (cqc);
}

static void 
cond_changed_cb  (GObject *obj, CanvasQueryCond *cqc)
{
	redraw_cond_contents (cqc);
}

static void
cond_destroyed_cb (CanvasQueryCond *cqc, GObject   *obj)
{
	cqc->priv->cond = NULL;
	canvas_query_cond_free (CANVAS_QUERY_COND (cqc));
}

#define ROOT_COLOR "violet"
#define AND_COLOR  "green"
#define OR_COLOR   "gold"
#define NOT_COLOR  "red"
#define OUTLINE_COLOR "black"

static int node_item_event(GnomeCanvasItem *ci, GdkEvent *event, CanvasQueryCond *cqc);
static int qf_item_event(GnomeCanvasItem *ci, GdkEvent *event, CanvasQueryCond *cqc);
static int op_item_event(GnomeCanvasItem *ci, GdkEvent *event, CanvasQueryCond *cqc);
static void 
redraw_cond_contents (CanvasQueryCond *cqc)
{
	GnomeCanvasItem *item = NULL;
	GSList *list;
	gfloat xspan = 0., yspan = 0.;
	/* separation between cond displays */
	gfloat xspace = 10.;
	gfloat yspace = 10.;
	double x1, y1, x2, y2;
	/* size of the "root" of the expression */
	gfloat rootxsize = 30.;
	gfloat rootysize = 15.;
	/* colors */
	gchar *color = OUTLINE_COLOR;
	
	/* destroy any existing GnomeCanvasItem for the displayed cond */
	list = cqc->priv->items;
	while (list) {
                gtk_object_destroy (GTK_OBJECT (list->data));
                list = g_slist_next (list);
        }
	g_slist_free (cqc->priv->items);
        cqc->priv->items = NULL;

	/* creating the new items */
	if (cqc->priv->cond) {
		if (query_cond_get_cond_type (cqc->priv->cond) == QUERY_COND_COND) {
			/* draw that condition */
			gchar *opstr;
			QueryField *qf;
			gchar *undefinedqf = "???";

			/* left operator */
			qf = query_cond_get_op_left (cqc->priv->cond);
			opstr = query_field_render_as_string (qf, NULL);
			if (!opstr)
				opstr = g_strdup (undefinedqf);

			xspan = xspace/2.;
			item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
						      GNOME_TYPE_CANVAS_TEXT,
						      "text", opstr,
						      "x", xspan,
						      "y", 0.,
						      "fill_color", OUTLINE_COLOR,
						      "anchor", GTK_ANCHOR_WEST,
						      "x_offset", 0.,
						      "y_offset", 0.,
						      NULL);
			cqc->priv->items = g_slist_append (cqc->priv->items, item);
			g_free (opstr);

			gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), 
						      &x1, &y1, &x2, &y2);
			
			xspan += (x2-x1) + xspace/2.;
			if (y2-y1 > yspan)
				yspan = (y2-y1);

			/* operator */
			opstr = query_cond_render_op_sql (cqc->priv->cond);
			item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
						      GNOME_TYPE_CANVAS_TEXT,
						      "text", opstr,
						      "x", xspan,
						      "y", 0.,
						      "fill_color", OUTLINE_COLOR,
						      "anchor", GTK_ANCHOR_WEST,
						      "x_offset", 0.,
						      "y_offset", 0.,
						      NULL);
			cqc->priv->items = g_slist_append (cqc->priv->items, item);

			gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), 
						      &x1, &y1, &x2, &y2);
			
			xspan += (x2-x1) + xspace/2.;
			if (y2-y1 > yspan)
				yspan = (y2-y1);

			/* rigth operator */
			qf = query_cond_get_op_right (cqc->priv->cond);
			opstr = query_field_render_as_string (qf, NULL);
			if (!opstr)
				opstr = g_strdup (undefinedqf);

			item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
						      GNOME_TYPE_CANVAS_TEXT,
						      "text", opstr,
						      "x", xspan,
						      "y", 0.,
						      "fill_color", OUTLINE_COLOR,
						      "anchor", GTK_ANCHOR_WEST,
						      "x_offset", 0.,
						      "y_offset", 0.,
						      NULL);
			cqc->priv->items = g_slist_append (cqc->priv->items, item);
			g_free (opstr);

			gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), 
						      &x1, &y1, &x2, &y2);
			
			xspan += (x2-x1) + xspace/2.;
			if (y2-y1 > yspan)
				yspan = (y2-y1);

			qf = query_cond_get_op_right2 (cqc->priv->cond);
			if (qf) {
				opstr = "AND";
				item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
							      GNOME_TYPE_CANVAS_TEXT,
							      "text", opstr,
							      "x", xspan,
							      "y", 0.,
							      "fill_color", OUTLINE_COLOR,
							      "anchor", GTK_ANCHOR_WEST,
							      "x_offset", 0.,
							      "y_offset", 0.,
							      NULL);
				cqc->priv->items = g_slist_append (cqc->priv->items, item);
				
				gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), 
							      &x1, &y1, &x2, &y2);
				
				xspan += (x2-x1) + xspace/2.;
				if (y2-y1 > yspan)
					yspan = (y2-y1);

				opstr = query_field_render_as_string (qf, NULL);
				if (!opstr)
					opstr = g_strdup (undefinedqf);
				
				item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
							      GNOME_TYPE_CANVAS_TEXT,
							      "text", opstr,
							      "x", xspan,
							      "y", 0.,
							      "fill_color", OUTLINE_COLOR,
							      "anchor", GTK_ANCHOR_WEST,
							      "x_offset", 0.,
							      "y_offset", 0.,
							      NULL);
				cqc->priv->items = g_slist_append (cqc->priv->items, item);
				g_free (opstr);
				
				gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), 
							      &x1, &y1, &x2, &y2);
				
				xspan += (x2-x1) + xspace/2.;
				if (y2-y1 > yspan)
					yspan = (y2-y1);
			}
			
			/* shift everything down yspan/2 so the y=0 is right in the top left */
			list = cqc->priv->items;
			while (list) {
				gnome_canvas_item_move (GNOME_CANVAS_ITEM (list->data),
							0., yspan/2.);
				list = g_slist_next (list);
			}

			/* draw a yellow background */
			item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
						      GNOME_TYPE_CANVAS_RECT,
						      "x1", 0.,
						      "y1", 0., 
						      "x2", xspan,
						      "y2", yspan,
						      "fill_color", "yellow",
						      "outline_color", "yellow",
						      "width_units", 1.0,
						      NULL);
			cqc->priv->items = g_slist_append (cqc->priv->items, item);
			gnome_canvas_item_lower_to_bottom (item);
		}
		else {
			/* draw the sub conditions */
			GSList *list;
			gint interval = 1;
			GnomeCanvasPoints *vline, *hline;
			gint pointno = 0;
			gchar *nodestr = "";

			switch (query_cond_get_cond_type (cqc->priv->cond)) {
			case QUERY_COND_AND:
				color = AND_COLOR;
				nodestr = _("AND");
				break;
			case QUERY_COND_OR:
				color = OR_COLOR;
				nodestr = _("OR");
				break;
			case QUERY_COND_NOT:
				color = NOT_COLOR;
				nodestr = _("NOT");
				break;
			default:
				break;
			}

			vline = gnome_canvas_points_new (2);

			list = query_cond_get_children (cqc->priv->cond);
			while (list) {				
				/* item for the sub condition */
				item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
							      CANVAS_QUERY_COND_TYPE,
							      "x", xspace + rootxsize,
							      "y", yspan,
							      "query_cond", list->data,
							      "allow_move", FALSE,
							      NULL);
				
				cqc->priv->items = g_slist_append (cqc->priv->items, item);
				gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), 
							      &x1, &y1, &x2, &y2);
				

				/* move the previous item by half of their height
				   because they do zero on their middle */
				gnome_canvas_item_move (item, 0., (y2-y1)/2.);
				
				/* horizontal & vertical lines */
				vline->coords[pointno] = rootxsize / 2.;
				vline->coords[pointno+1] = yspan + (y2-y1)/2.;
				if (pointno == 0)
					pointno = 2;

				hline = gnome_canvas_points_new (2);
				hline->coords[0] = rootxsize / 2.;
				hline->coords[1] = yspan + (y2-y1)/2.;
				hline->coords[2] = xspace + rootxsize;
				hline->coords[3] = yspan + (y2-y1)/2.;
				item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
							      GNOME_TYPE_CANVAS_LINE,
							      "points", hline,
							      "fill_color", color,
                                                              "width_units", 2.,
                                                              /*"cap_style", GDK_CAP_ROUND,*/
                                                              "line_style", GDK_LINE_SOLID,
                                                              "smooth", TRUE,
							      NULL);
				gnome_canvas_points_free (hline);
				cqc->priv->items = g_slist_append (cqc->priv->items, item);
				
				/* compute the coordinates for the potential next item */
				if (!g_slist_next (list))
					interval = 0;

				yspan += (y2 - y1) + interval * yspace;
				if (x2-x1+xspace+rootxsize > xspan)
					xspan = x2-x1+xspace+rootxsize;
				
				list = g_slist_next (list);
			}

			if (g_slist_length (query_cond_get_children (cqc->priv->cond)) > 1) {
				item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
							      GNOME_TYPE_CANVAS_LINE,
							      "points", vline,
							      "fill_color", color,
							      "width_units", 2.,
							      /*"cap_style", GDK_CAP_ROUND,*/
							      "line_style", GDK_LINE_SOLID,
							      "smooth", TRUE,
							      NULL);
				cqc->priv->items = g_slist_append (cqc->priv->items, item);
			}
			gnome_canvas_points_free (vline);


			/* draw text and a rectangle for the node itself */
			item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
						      GNOME_TYPE_CANVAS_TEXT,
						      "text", nodestr,
						      "x", xspace/3.,
						      "y", yspan/2.,
						      "fill_color", OUTLINE_COLOR,
						      "anchor", GTK_ANCHOR_WEST,
						      "x_offset", 0.,
						      "y_offset", 0.,
						      NULL);
			cqc->priv->items = g_slist_append (cqc->priv->items, item);

			gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), 
						      &x1, &y1, &x2, &y2);
			rootxsize = xspace *2./3. + (x2-x1);
			rootysize = yspace/4. + (y2-y1);

			item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
						      GNOME_TYPE_CANVAS_RECT,
						      "x1", 0.,
						      "y1", yspan/2. - rootysize / 2.,
						      "x2", rootxsize,
						      "y2", yspan/2. + rootysize / 2.,
						      "fill_color", color,
						      "outline_color", color,
						      "width_units", 1.0,
						      NULL);
			
			cqc->priv->items = g_slist_append (cqc->priv->items, item);
			xspan += rootxsize;
			gnome_canvas_item_lower_to_bottom (item);

			g_signal_connect(G_OBJECT (item), "event",
					 G_CALLBACK (node_item_event), cqc);
		}
	}

	/* if the cond has no parent then we draw a "root" node */
	if ((cqc->priv->cond && !query_cond_get_parent (cqc->priv->cond)) ||
	    !cqc->priv->cond) {
		rootxsize = 10.;
		rootysize = 10.;

		/* shift on the right all the previous items */
		list = cqc->priv->items;
		while (list) {
			gnome_canvas_item_move (GNOME_CANVAS_ITEM (list->data),
						rootxsize, 0.);
			list = g_slist_next (list);
		}

		/* draw a rectangle as the "root" of the expression */	
		item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cqc),
					      GNOME_TYPE_CANVAS_ELLIPSE,
					      "x1", 0.,
					      "y1", yspan/2. - rootysize/2.,
					      "x2", rootxsize,
					      "y2", yspan/2. + rootysize/2.,
					      "fill_color", ROOT_COLOR,
					      "outline_color", ROOT_COLOR,
					      "width_units", 1.0,
					      NULL);
		
		cqc->priv->items = g_slist_append (cqc->priv->items, item);
		xspan += rootxsize + xspace;

		if (yspan < rootysize)
			yspan = rootysize;
	}

	/* shift everything up yspan/2 so the y=0 is right in the middle */
	list = cqc->priv->items;
	while (list) {
		gnome_canvas_item_move (GNOME_CANVAS_ITEM (list->data),
					0., - yspan/2.);
		list = g_slist_next (list);
	}

	
	/* store the real size values */
	cqc->priv->xsize = xspan;
	cqc->priv->ysize = yspan;
}

static void 
canvas_query_cond_bounds (GnomeCanvasItem *item, 
			  double *x1, double *y1, double *x2, double *y2)
{
	CanvasQueryCond *cqc;

	cqc = CANVAS_QUERY_COND (item);

	*x1 = 0.;
	*y1 = 0.;
	*x2 = cqc->priv->xsize;
	*y2 = cqc->priv->ysize;
}


static void create_alias_cb(GtkWidget * button, CanvasQueryCond *cqc);
static void delete_view_cb(GtkWidget * button, CanvasQueryCond *cqc);
static int 
node_item_event(GnomeCanvasItem *ci, GdkEvent *event, CanvasQueryCond *cqc)
{
	gboolean done = TRUE;
	GtkWidget *menu, *entry;

	switch (event->type) {
	case GDK_ENTER_NOTIFY:
		break;
	case GDK_LEAVE_NOTIFY:
		break;
	case GDK_BUTTON_PRESS:
		menu = gtk_menu_new ();
		entry = gtk_menu_item_new_with_label (_("Create alias"));
		g_signal_connect (G_OBJECT (entry), "activate",
				  G_CALLBACK (create_alias_cb), cqc);
		gtk_menu_append (GTK_MENU (menu), entry);
		gtk_widget_show (entry);
		entry = gtk_menu_item_new_with_label (_("Delete"));
		g_signal_connect (G_OBJECT (entry), "activate",
				  G_CALLBACK (delete_view_cb), cqc);
		gtk_menu_append (GTK_MENU (menu), entry);
		gtk_widget_show (entry);
		gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
				NULL, NULL, ((GdkEventButton *)event)->button,
				((GdkEventButton *)event)->time);
		break;
	default:
		done = FALSE;
		break;
	}

	return done;	
}

static void 
create_alias_cb(GtkWidget * button, CanvasQueryCond *cqc)
{

}

static void 
delete_view_cb(GtkWidget * button, CanvasQueryCond *cqc)
{

}

