/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 * 
 *  medusa.h: general header for programs that link with medusa
 *
 *  Copyright (C) 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors: Maciej Stachowiak <mjs@eazel.com> 
 *  
 */

#include <libmedusa/medusa-search-service.h>
#include <libmedusa/medusa-search-service-private.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>



struct _MedusaSearchServiceConnection {
        int socket_fd;
        guint cookie;

        gboolean busy;
        char *buffer;
};

static int
initialize_socket ()
{
        int search_request_port;
        int connect_result;
        struct sockaddr_un daemon_address;    
        
        search_request_port = socket (AF_LOCAL, SOCK_STREAM, 0);
        g_return_val_if_fail (search_request_port != -1, -1);
        
        daemon_address.sun_family = AF_LOCAL;

#define POSIX_GUARANTEED_UNIX_SOCKET_PATH_SIZE 100

        strncpy (daemon_address.sun_path, SEARCH_SOCKET_PATH, POSIX_GUARANTEED_UNIX_SOCKET_PATH_SIZE - 1);

#undef POSIX_GUARANTEED_UNIX_SOCKET_PATH_SIZE
        
        connect_result = connect (search_request_port, (struct sockaddr *) &daemon_address,
                                  SUN_LEN (&daemon_address));
        if (connect_result == -1) {
                return -1;
        }
        
        return search_request_port;
}



static void
authenticate_connection (MedusaSearchServiceConnection *connection) {
	char *request;
        char *file_name;
        int cookie_fd, key;

	/* Send request for cookie */
	request = g_strdup_printf ("%s\t%d\t%d\n", COOKIE_REQUEST, getuid (), getpid());

        /* FIXME: check return value */
	write (connection->socket_fd, request, strlen (request));

        g_free (request);
        
        
        /* Go look for cookie */
        file_name = g_strdup_printf ("%s/%d_%d", COOKIE_PATH, getuid (), getpid ());
        
        cookie_fd = open (file_name, O_RDONLY);
        
        /* FIXME: maybe have search daemon send response once cookie
           file is created instead of this looop */
        
        /* Keep looking if cookie file isn't created yet */
        while (cookie_fd == -1) {
                cookie_fd = open (file_name, O_RDONLY);
        }
        
        g_free (file_name);
        
        read (cookie_fd, &key, sizeof (int));
        close (cookie_fd);
  
        connection->cookie = key;
}


GnomeVFSResult
medusa_search_service_connection_is_available ()
{
        GnomeVFSResult result;
        MedusaSearchServiceConnection *connection;
        /* FIXME: This is inefficient. */
        connection = medusa_search_service_connection_new (&result);
        if (connection != NULL) {
	        medusa_search_service_connection_destroy (connection);
        }

        return result;
        
}


MedusaSearchServiceConnection *
medusa_search_service_connection_new (GnomeVFSResult *result)
{
        MedusaSearchServiceConnection *connection;

        connection = g_new0 (MedusaSearchServiceConnection, 1);
        
        connection->socket_fd = initialize_socket ();

        if (connection->socket_fd == -1) {
                *result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
                return NULL;
        }

	*result = GNOME_VFS_OK;
        authenticate_connection (connection);

        return connection;
}


static void
send_search_request (MedusaSearchServiceConnection *connection, 
                     const char *search_uri)
{
        char *request;

        request = g_strdup_printf ("%d %d %d\t%s", getuid(), getpid(), connection->cookie, search_uri);

#ifdef DEBUG_SEARCH_API
        printf ("about to send %s\n", request);
#endif

        /* FIXME: check error code */
        write (connection->socket_fd, request, strlen(request));

        g_free (request);

}



GnomeVFSResult           
medusa_search_service_connection_start_search (MedusaSearchServiceConnection *connection,
                                               const char *search_uri)
{
        if (connection->busy) {
                return GNOME_VFS_ERROR_IN_PROGRESS;
        } else {
                send_search_request (connection, search_uri);
                connection->busy = TRUE;
                return GNOME_VFS_OK;
        }
}


GnomeVFSResult medusa_search_service_connection_read_search_result (MedusaSearchServiceConnection *connection,
                                                                    char **result)
{
        char tmpbuf[513];
        const char *newline_pos;
        char *result_uri;
        int len;
        char *buffer;

        if (!connection->busy) {
                *result = NULL;
                return GNOME_VFS_ERROR_EOF;
        }
        if (connection->buffer == NULL) {
                connection->buffer = g_strdup ("");
        }

        while (strchr (connection->buffer, '\n') == NULL) {
                len = read (connection->socket_fd, &tmpbuf, 512 * sizeof (char)); 

#ifdef DEBUG_SEARCH_API
                puts ("read from socket");
#endif

                tmpbuf[len] = '\0';
                buffer = g_strconcat (connection->buffer, tmpbuf, NULL);
                g_free (connection->buffer);
                connection->buffer = buffer;
        }



        /* grab a result */
        newline_pos = strchr (connection->buffer, '\n');
        result_uri = g_strndup (connection->buffer, newline_pos - connection->buffer);
        g_return_val_if_fail (result_uri != NULL, GNOME_VFS_ERROR_NOT_FOUND);

        /* move buffer forward */
        buffer = g_strdup (newline_pos + 1);
        g_free (connection->buffer);
        connection->buffer = buffer;


        if (strcmp (result_uri, SEARCH_END_TRANSMISSION) == 0) {
                g_free (result_uri);
                *result = NULL;
                connection->busy = FALSE;
                return GNOME_VFS_ERROR_EOF;
        } else {
                *result = result_uri;
                return GNOME_VFS_OK;
        }
}



void medusa_search_service_connection_destroy (MedusaSearchServiceConnection *connection)
{
        close (connection->socket_fd);
        g_free (connection->buffer);
        g_free (connection);
}


